/*
 *  Copyright 1994-2016 Olivier Girondel
 *
 *  This file is part of lebiniou.
 *
 *  lebiniou is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  lebiniou is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with lebiniou. If not, see <http://www.gnu.org/licenses/>.
 */

#include "globals.h"
#include "context.h"
#include "brandom.h"
#include "pictures.h"
#include "colormaps.h"
#include "sequences.h"


#define DELAY_MIN 15
#define DELAY_MAX 30


#ifdef WITH_WEBCAM
#include "webcam.h"

extern int webcams;
extern char *video_base;


static webcam_t *cams[MAX_CAMS];
static pthread_t thr[MAX_CAMS];

#define MAX_TRIES 5


static void
Context_open_webcam(Context_t *ctx)
{
  int i;
  uint8_t try = 0;

  parse_options();

  for (i = 0; i < MAX_CAMS; i++)
    pthread_mutex_init(&ctx->cam_mtx[i], NULL);

  for (i = 0; (i < webcams) && (try < MAX_TRIES); i++) {
  retry:
    cams[i] = xmalloc(sizeof(webcam_t));

    cams[i]->io = IO_METHOD_MMAP;
    cams[i]->fd = -1;
    cams[i]->ctx = ctx;
    cams[i]->cam_no = i;

    if (-1 != open_device(cams[i], try)) {
      if (-1 != init_device(cams[i])) {
	enumerate_cids(cams[i]);
	list_inputs(cams[i]);
	start_capturing(cams[i]);

	pthread_create(&thr[i], NULL, loop, (void *)cams[i]);
      } else {
	fprintf(stderr, "[i] Webcam: failed to initialize device #%d\n", i);
	close_device(cams[i]);
	xfree(cams[i]);
	try++;
	goto retry;
      }
    } else
      xfree(cams[i]);
  }
}
#endif


Context_t *
Context_new()
{
  int i;

  Context_t *ctx = xcalloc(1, sizeof(Context_t));

  ctx->running = 1;

  if (libbiniou_verbose)
    printf("[+] Creating buffers... ");
  for (i = 0; i < NSCREENS; i++) {
    if (libbiniou_verbose)
      printf("%d ", i);
    ctx->buffers[i] = Buffer8_new();
  }
#ifdef WITH_WEBCAM
  {
    int k;
    for (k = 0; k < webcams; k++) {
      for (i = 0; i < CAM_SAVE; i++)
	ctx->cam_save[k][i] = Buffer8_new();
      ctx->cam_ref[k] = Buffer8_new();
      ctx->rgba_cam_buffers[k] = BufferRGBA_new();
    }
  }
#endif
  ctx->rgba_buffers[ACTIVE_BUFFER] = BufferRGBA_new();
  if (libbiniou_verbose)
    printf("\n");

#if WITH_GL
  glGenTextures(NSCREENS, ctx->textures); // TODO: delete on exit
  glGenTextures(MAX_CAMS, ctx->cam_textures); // TODO: delete on exit
#endif

  if (pictures != NULL) {
    if (libbiniou_verbose)
      printf("[+] Creating pictures fader (%i pictures)\n", pictures->size);
    ctx->pf = PictFader_new(pictures->size);

    if (libbiniou_verbose)
      printf("[+] Creating pictures timer\n");
    ctx->a_picts = Alarm_new(DELAY_MIN, DELAY_MAX);
  }

  if (colormaps != NULL) {
    if (libbiniou_verbose)
      printf("[+] Creating colormaps fader (%i colormaps)\n", colormaps->size);
    ctx->cf = CmapFader_new(colormaps->size);

    if (libbiniou_verbose)
      printf("[+] Creating colormaps timer\n");
    ctx->a_cmaps = Alarm_new(DELAY_MIN, DELAY_MAX);
  }

  ctx->a_random = Alarm_new(DELAY_MIN, DELAY_MAX);
  ctx->random_mode = BR_NONE;

  if (libbiniou_verbose)
    printf("[+] Initializing sequence manager\n");
  ctx->sm = SequenceManager_new();

  Context_load_banks(ctx);

  if (libbiniou_verbose)
    printf("[+] Initializing 3D engine\n");
  Params3d_init(&ctx->params3d);
	
  ctx->events = NULL;
  ctx->frames = ctx->nb_events = 0;
  for (i = 0; i < NFPS; i++)
    ctx->fps[i] = 0;

  ctx->timer = b_timer_new();
  ctx->fps_timer = b_timer_new();

  ctx->osd_mode = OSD_NONE;
  ctx->display_colormap = 0;

  ctx->outputs = NULL;

#ifdef FEAT_TARGET
  ctx->target = 1; /* maybe someday we want a switch to disable,
		      or worse, a global variable :) */
  ctx->target_pic = Picture8_new();
  /* FIXME temp */
  {
    extern char *data_dir;
    gchar *tmp;
    int res;

    tmp = g_strdup_printf("%s/%s", data_dir, "/images/zebulon/z-biniou-tv-1.png");
    VERBOSE(printf("[+] Loading '%s'\n", tmp));
    res = Picture8_load_PNG(ctx->target_pic, tmp);
    assert(!res);
    if (res == -1)
      xerror("Picture8_load_PNG(%s)\n", tmp);
    g_free(tmp);
  }
#endif

#ifdef WITH_WEBCAM
  VERBOSE(printf("[i] Initialazing %d webcams base: %s\n", webcams, video_base));

  ctx->cam = 0;
  Context_open_webcam(ctx);

  for (i = 0; i < webcams; i++)
    Buffer8_copy(ctx->target_pic->buff, ctx->cam_save[i][0]);
#endif

  return ctx;
}


#ifdef WITH_WEBCAM
static void
Context_close_webcam(const u_char cam_no)
{
  if (NULL != cams[cam_no]) {
    pthread_join(thr[cam_no], NULL);
    stop_capturing(cams[cam_no]);
    uninit_device(cams[cam_no]);
    close_device(cams[cam_no]);
  }
}
#endif


void
Context_delete(Context_t *ctx)
{
  int i;
  GSList *outputs = ctx->outputs;

#ifdef WITH_WEBCAM
  if (webcams) {
    int i;

    for (i = 0; i < webcams; i++)
      Context_close_webcam(i);
  }
  xfree(video_base);
#endif

  if (libbiniou_verbose)
    printf("[i] %lu frames, %lu events\n", ctx->frames, ctx->nb_events);

  if (ctx->input_plugin != NULL)
    Plugin_delete(ctx->input_plugin);

  for ( ; outputs != NULL; outputs = g_slist_next(outputs)) {
    Plugin_t *output = (Plugin_t *)outputs->data;

    Plugin_delete(output);
  }
  g_slist_free(outputs);
  
  if (libbiniou_verbose)
    printf("[+] Freeing buffers... ");
  for (i = 0; i < NSCREENS; i++) {
    if (libbiniou_verbose)
      printf("%d ", i);
    Buffer8_delete(ctx->buffers[i]);
  }
#ifdef WITH_WEBCAM
  {
    int k;
    for (k = 0; k < webcams; k++) {
      for (i = 0; i < CAM_SAVE; i++)
	Buffer8_delete(ctx->cam_save[k][i]);
      Buffer8_delete(ctx->cam_ref[k]);
      BufferRGBA_delete(ctx->rgba_cam_buffers[k]);
    }
  }
#endif
  BufferRGBA_delete(ctx->rgba_buffers[ACTIVE_BUFFER]);
  if (libbiniou_verbose)
    printf("\n");

  if (ctx->pf != NULL) {
    if (libbiniou_verbose)
      printf("[+] Freeing pictures fader\n");
    PictFader_delete(ctx->pf);

    if (libbiniou_verbose)
      printf("[+] Freeing pictures timer\n");
    Alarm_delete(ctx->a_picts);
  }

  if (ctx->cf != NULL) {
    if (libbiniou_verbose)
      printf("[+] Freeing colormaps fader\n");
    CmapFader_delete(ctx->cf);

    if (libbiniou_verbose)
      printf("[+] Freeing colormaps timer\n");
    Alarm_delete(ctx->a_cmaps);
  }

  Alarm_delete(ctx->a_random);
  SequenceManager_delete(ctx->sm);
  b_timer_delete(ctx->timer);
  b_timer_delete(ctx->fps_timer);

#ifdef FEAT_TARGET
  if (NULL != ctx->target_pic)
    Picture8_delete(ctx->target_pic);
#endif

  xfree(ctx);
}


void
Context_set_colormap(Context_t *ctx)
{
  /* find the cmap, if any. default cmap otherwise */
  if (ctx->cf != NULL) {
    ctx->cf->fader->target
      = (ctx->sm->next->cmap_id) ?
      Colormaps_index(ctx->sm->next->cmap_id) : 0;
    CmapFader_set(ctx->cf);
  }
}


void
Context_set_picture(Context_t *ctx)
{
  /* find the picture, if any. default picture otherwise */
  if (ctx->pf != NULL) {
    ctx->pf->fader->target
      = (ctx->sm->next->picture_id) ?
      Pictures_index(ctx->sm->next->picture_id) : 0;
    PictFader_set(ctx->pf);
  }
}

void
Context_set(Context_t *ctx)
{
  GList *tmp;

#ifdef XDEBUG
  printf("*** Context_set: cur sequence= %li\n", ctx->sm->cur->id);
  printf("*** Context_set: next sequence= %li\n", ctx->sm->next->id);
#endif

  tmp = g_list_first(ctx->sm->cur->layers);
	
  /* call on_switch_off() on old plugins */
  while (tmp != NULL) {
    Layer_t *layer = tmp->data;
    Plugin_t *p = layer->plugin;

    assert(p != NULL);
    if (p->on_switch_off != NULL)
      if (!(*p->options & BEQ_DISABLED))
	p->on_switch_off(ctx);
    tmp = g_list_next(tmp);
  }		

  Context_set_colormap(ctx);
  Context_set_picture(ctx);

  /* call on_switch_on() on new plugins */
  tmp = g_list_first(ctx->sm->next->layers);
  while (tmp != NULL) {
    Layer_t *layer = (Layer_t *)tmp->data;
    Plugin_t *p = layer->plugin;
	  
    assert(p != NULL);
    if (p->on_switch_on != NULL)
      if (!(*p->options & BEQ_DISABLED))
	p->on_switch_on(ctx);
    tmp = g_list_next(tmp);
  }	
	
  Sequence_copy(ctx->sm->next, ctx->sm->cur);

  Context_update_auto(ctx);
	
  Sequence_display(ctx->sm->cur);
  okdone("Context_set");
}


void
Context_update_auto(Context_t *ctx)
{
  /* set auto stuff */
  if (ctx->pf != NULL) {
    ctx->pf->on = ctx->sm->cur->auto_pictures;
    if (ctx->pf->on && (pictures != NULL) && (pictures->size > 1)) {
      /* select random picture and reinitialize timer */
      PictFader_random(ctx->pf);
      Alarm_init(ctx->a_picts);
    }
  }
  
  if (ctx->cf != NULL) {
    ctx->cf->on = ctx->sm->cur->auto_colormaps;
    if (ctx->cf->on && (colormaps->size > 1)) {
      /* select random colormap and reinitialize timer */
      CmapFader_random(ctx->cf);
      Alarm_init(ctx->a_cmaps);
    }
  }
}


int
Context_add_rand(Sequence_t *seq,
		 const enum PluginOptions options, const int not_lens)
{
  Plugin_t *p;

  do {
    p = Plugins_get_random(options);
    if (p == NULL)
      return -1;
  } while (Sequence_find(seq, p) != NULL);

  if (*p->options & BEQ_FLUSH)
    Sequence_clear(seq, 0);

  Sequence_insert(seq, p);

  if ((*p->options & BE_LENS) && !not_lens && (seq->lens == NULL))
    seq->lens = p;
	
  return 0;
}


void
Context_randomize(Context_t *ctx)
{
  Sequence_t *new = ctx->sm->next;
  int rand;

  /* Use random picture */
  if (pictures != NULL) {
    if (pictures->size > 1) {
      rand = b_rand_int_range(0, pictures->size-1);
      new->auto_pictures = b_rand_boolean();
    } else
      rand = new->auto_pictures = 0;
    new->picture_id = pictures->pics[rand]->id;
  }
  
  /* Use random colormap */
  assert(colormaps != NULL);
  if (colormaps->size > 1) {
    rand = b_rand_int_range(0, colormaps->size-1);
    new->auto_colormaps = b_rand_boolean();
  } else
    rand =new->auto_colormaps = 0;
  new->cmap_id = colormaps->cmaps[rand]->id;

  /* Set 3D rotations */
  if (b_rand_boolean())
    Params3d_randomize(&ctx->params3d);
  else
    ctx->params3d.do_auto_rotate = 0;
}


void
Context_insert_plugin(Context_t *ctx, Plugin_t *p)
{
  /* switch the plugin on */
  if (p->on_switch_on != NULL) {
    if (libbiniou_verbose)
      printf("[i] on_switch_on(%s)\n", p->name);
    p->on_switch_on(ctx);
  }

  Sequence_insert(ctx->sm->cur, p);
}


void
Context_remove_plugin(Context_t *ctx, Plugin_t *p)
{
  /* switch the plugin off */
  if (p->on_switch_off != NULL) {
    if (libbiniou_verbose)
      printf("[i] on_switch_off(%s)\n", p->name);
    p->on_switch_off(ctx);
  }

  Sequence_remove(ctx->sm->cur, p);
}


void
Context_set_max_fps(Context_t *ctx, const u_short max_fps)
{
  ctx->sync_fps = 1;
  assert(max_fps);
  ctx->max_fps = max_fps;
  ctx->i_max_fps = 1.0 / ctx->max_fps;
}


void
Context_set_random_mode(Context_t *ctx, const enum RandomMode r)
{
  ctx->random_mode = r;
}


void
Context_set_osd_mode(Context_t *ctx, const enum OSDMode m)
{
  ctx->osd_mode = m;
}


float
Context_fps(const Context_t *ctx)
{
  float mfps = 0.0;
  int i;

  for (i = 0; i < NFPS; i++)
    mfps += ctx->fps[i];
  return (mfps / (float)NFPS);
}


void
Context_previous_sequence(Context_t *ctx)
{
  Sequence_t *s;

  if (ctx->sm->curseq == NULL)
    return;

  if (ctx->sm->curseq->prev != NULL)
    ctx->sm->curseq = ctx->sm->curseq->prev;
  else
    ctx->sm->curseq = g_list_last(sequences->seqs);

  s = (Sequence_t *)ctx->sm->curseq->data;
  Sequence_copy(s, ctx->sm->next);

  Context_set(ctx);
}


void
Context_next_sequence(Context_t *ctx)
{
  Sequence_t *s;

  if (ctx->sm->curseq == NULL)
    return;

  if (ctx->sm->curseq->next != NULL)
    ctx->sm->curseq = ctx->sm->curseq->next;
  else
    ctx->sm->curseq = sequences->seqs;

  s = (Sequence_t *)ctx->sm->curseq->data;
  Sequence_copy(s, ctx->sm->next);

  Context_set(ctx);
}


void
Context_latest_sequence(Context_t *ctx)
{
  Sequence_t *s;

  if (ctx->sm->curseq == NULL)
    return;

  ctx->sm->curseq = sequences->seqs;

  s = (Sequence_t *)ctx->sm->curseq->data;
  Sequence_copy(s, ctx->sm->next);

  Context_set(ctx);
}


void
Context_random_sequence(Context_t *ctx)
{
  u_short rand;
  GList *tmp;

  rand = Shuffler_get(sequences->shuffler);

  tmp = g_list_nth(sequences->seqs, rand);
  assert(tmp != NULL);
  
  if (libbiniou_verbose)
    printf("[s] Random sequence: %d\n", rand);
  
  ctx->sm->curseq = tmp;
  Sequence_copy(tmp->data, ctx->sm->next);

  Context_set(ctx);
}


void
Context_set_sequence(Context_t *ctx, const uint32_t id)
{
  Sequence_t *seq = Sequences_find(id);

  if (NULL == seq)
    if (ctx->sm->transient->id == id)
      seq = ctx->sm->transient;

  assert(NULL != seq);
  ctx->sm->curseq = seq->layers;
  Sequence_copy(seq, ctx->sm->next);

  Context_set(ctx);
}


void
Context_use_sequence_bank(Context_t *ctx, const u_char bank)
{
  u_long id;

  id = ctx->banks[SEQUENCES][ctx->bankset[SEQUENCES]][bank];
  if (id) {
    Sequence_t *seq;

    printf("[i] Using sequence in bank #%d\n", (bank+1));

    if (libbiniou_verbose)
      printf("[s] Set sequence: %li\n", id);

    seq = Sequences_find(id);
    if (NULL == seq)
      if (ctx->sm->transient->id == id)
	seq = ctx->sm->transient;
    assert(NULL != seq);
    ctx->sm->curseq = seq->layers;
    Sequence_copy(seq, ctx->sm->next);

    ctx->bank[SEQUENCES] = bank;

    Context_set(ctx);
  } else
    printf("[i] Bank %d/%d is empty\n", ctx->bankset[SEQUENCES]+1, bank+1);
}


void
Context_clear_bank(Context_t *ctx, const u_char bank)
{
  ctx->banks[ctx->bank_mode][ctx->bankset[ctx->bank_mode]][bank] = 0;
}


Buffer8_t *
active_buffer(const Context_t *ctx)
{
  return ctx->buffers[ACTIVE_BUFFER];
}


Buffer8_t *
passive_buffer(const Context_t *ctx)
{
  return ctx->buffers[PASSIVE_BUFFER];
}


#ifdef WITH_WEBCAM
void
Context_push_webcam(Context_t *ctx, Buffer8_t *buff, const int cam)
{
  int i;

  Buffer8_delete(ctx->cam_save[cam][CAM_SAVE-1]);
  for (i = CAM_SAVE-1; i >= 1; i--)
    ctx->cam_save[cam][i] = ctx->cam_save[cam][i-1];
  ctx->cam_save[cam][0] = buff;
}
#endif
