/*
 * Copyright (C) 2025
 *      Mark Muth <MarkMuth@gmx.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

import QtQuick 2.4
import Lomiri.Components 1.3
import MediaScanner 0.1
import "../components"
import "../components/Delegates"
import "../components/Flickables"
import "../components/HeadState"
import "../components/ListItemActions"


MusicPage {
    id: filesPage
    objectName: "filesPage"
    title: i18n.tr("Files")
    searchable: true
    searchResultsCount: filesModelFilter.count
    state: "default"
    states: [
        SearchableHeadState {
            thisPage: filesPage
            searchEnabled: filesModelFilter.count > 0
        },
        MultiSelectHeadState {
            listview: filelist
            thisPage: filesPage
        },
        SearchHeadState {
            id: searchHeader
            thisPage: filesPage
        }
    ]

    // see: https://bugs.launchpad.net/autopilot-qt/+bug/1341671
    property bool bug1341671workaround: true

    MultiSelectListView {
        id: filelist
        anchors {
            bottomMargin: units.gu(2)
            fill: parent
            topMargin: units.gu(2)
        }
        objectName: "filestab-listview"
        model: SortFilterModel {
            id: filesModelFilter
            property alias rowCount: filesModel.rowCount
            model: SongsModel {
                id: filesModel
                store: musicStore
            }
            sort.property: "filename"
            sort.order: Qt.AscendingOrder
            sortCaseSensitivity: Qt.CaseInsensitive
            filter.property: "title"
            filter.pattern: new RegExp(searchHeader.query, "i")
            filterCaseSensitivity: Qt.CaseInsensitive
        }

        onStateChanged: {
            if (state === "multiselectable") {
                filesPage.state = "selection"
            } else {
                searchHeader.query = ""  // force query back to default
                filesPage.state = "default"
            }
        }

        delegate: MusicListItem {
            id: file
            objectName: "filesPageListItem" + index
            imageSource: {"art": model.art}
            multiselectable: true
            subtitle {
                text: model.filename.replace(XDG_MUSIC_DIR + "/", "")
            }
            title {
                objectName: "tracktitle"
                text: model.title
            }
            trailingActions: AddToQueueAndPlaylist {
            }

            onItemClicked: {
                if (filesPage.state === "search") {  // only play single track when searching
                    player.mediaPlayer.playlist.clearWrapper();
                    player.mediaPlayer.playlist.addItem(Qt.resolvedUrl(filesModelFilter.get(index).filename));
                    trackQueueClick(0)
                } else {
                    trackClicked(filesModelFilter, index)  // play track
                }
            }
        }
    }
}
