// license:BSD-3-Clause
// copyright-holders:David Haywood
/*

    Scorpion 4 Hardware Platform (c)1996 Bell Fruit Manufacturing

    preliminary driver

    -----------------

    note: default Jackpot keys should be set to whatever value the game
          mentions it should be using with none present, many games accept
          multiple keys.  A number of Mazooma games will attempt to use
          invalid settings by default with no key.

    Reel configs have been done to allow games to boot past the initial
    spin up tests where possible*, some reels probably still aren't correct
    and as a result several games will end up with reels erroring or
    spinning endlessly ingame, once the correct reel types are identified
    the details in the config structures should be replaced.

    * a handful of games still don't

    ----------------

    General Info

    Whitbread licensed sets were designed for use in Whitbread owned
    establishments, they tend to have much 'flatter' profiles than the
    regular sets meaning they pay out small prizes with a regular frequency
    rather than saving up for larger (multi) jackpot payments etc.

*/

#include "emu.h"
#include "bfm_sc4.h"
#include "bfm_sc45_helper.h"

#include "awpvid.h" //DMD01

#include "cpu/m68000/m68000.h"
#include "cpu/m6809/m6809.h"
#include "machine/68307.h"
#include "machine/68340.h"

#include "speaker.h"

#include "bfm_sc4.lh"
#include "sc4_dmd.lh"

#include "sc4dnd.lh"
#include "sc4dndtp.lh"
#include "sc4dnddw.lh"

// MFME2MAME layouts:
#include "sc4acesh.lh"
#include "sc4bantm.lh"
#include "sc4bedcl.lh"
#include "sc4blast.lh"
#include "sc4blokq.lh"
#include "sc4bobcl.lh"
#include "sc4brksp.lh"
#include "sc4brollb.lh"
#include "sc4cabin.lh"
#include "sc4canca.lh"
#include "sc4captn.lh"
#include "sc4cariqd.lh"
#include "sc4carrya.lh"
#include "sc4cashm.lh"
#include "sc4cclasd.lh"
#include "sc4cclim.lh"
#include "sc4cfcas.lh"
#include "sc4chainb.lh"
#include "sc4chavy.lh"
#include "sc4chick.lh"
#include "sc4chuba.lh"
#include "sc4clashd.lh"
#include "sc4clue.lh"
#include "sc4cmania.lh"
#include "sc4cmous.lh"
#include "sc4cnfr.lh"
#include "sc4coro.lh"
#include "sc4crc.lh"
#include "sc4crcla.lh"
#include "sc4crzgna.lh"
#include "sc4crzkya.lh"
#include "sc4cvanif.lh"
#include "sc4cvclb.lh"
#include "sc4monot.lh"
#include "sc4mspid.lh"

#include "sc4dayltf.lh"
#include "sc4deepi.lh"
#include "sc4derby.lh"
#include "sc4dmineb.lh"
#include "sc4drubyb.lh"
#include "sc4duckqa.lh"
#include "sc4dyna.lh"
#include "sc4eascsc.lh"
#include "sc4emmerc.lh"
#include "sc4fastfb.lh"
#include "sc4fd7tha.lh"
#include "sc4fever.lh"
#include "sc4fire.lh"
#include "sc4fpitcb.lh"
#include "sc4ftopib.lh"
#include "sc4fullt.lh"
#include "sc4fwpcs.lh"
#include "sc4gaga.lh"
#include "sc4gball.lh"
#include "sc4gd.lh"
#include "sc4gdclb.lh"
#include "sc4ggdlxf.lh"
#include "sc4gldcl.lh"
#include "sc4gslam.lh"
#include "sc4gx.lh"
#include "sc4hdd.lh"
#include "sc4hi5.lh"
#include "sc4hilo.lh"
#include "sc4himi.lh"
#include "sc4hissa.lh"
#include "sc4holyw.lh"
#include "sc4hotpr.lh"
#include "sc4hotwd.lh"
#include "sc4hyde.lh"
#include "sc4sprng.lh"

#include "sc4jack.lh"
#include "sc4kalei.lh"
#include "sc4knokb.lh"
#include "sc4lirb.lh"
#include "sc4lined.lh"
#include "sc4lockb.lh"
#include "sc4lkbcl.lh"
#include "sc4mmmg.lh"
#include "sc4mgr.lh"
#include "sc4mtbb.lh"
#include "sc4mtbclb.lh"
#include "sc4monoba.lh"
#include "sc4mondxe.lh"
#include "sc4mr2rc.lh"
#include "sc4motor.lh"
#include "sc4mou.lh"
#include "sc4nmtj.lh"
#include "sc4nudita.lh"
#include "sc4pacqpa.lh"
#include "sc4paccl.lh"
#include "sc4pstata.lh"
#include "sc4pgold.lh"
#include "sc4po8.lh"
#include "sc4ppclb.lh"
#include "sc4pipe.lh"
#include "sc4plumb.lh"
#include "sc4polic.lh"
#include "sc4pony.lh"
#include "sc4potsha.lh"
#include "sc4pwrbq.lh"
#include "sc4pwrsg.lh"
#include "sc4pen1c.lh"
#include "sc4qmodo.lh"
#include "sc4quidrb.lh"
#include "sc4rhxcsa.lh"
#include "sc4rhxcl.lh"
#include "sc4redsq.lh"
#include "sc4rotc.lh"
#include "sc4revod.lh"
#include "sc4richl.lh"
#include "sc4rttt.lh"
#include "sc4roksc.lh"
#include "sc4rogds.lh"
#include "sc4rmo.lh"
#include "sc4rtc.lh"
#include "sc4srrca.lh"
#include "sc4smk7.lh"
#include "sc4slci.lh"
#include "sc4solgl.lh"
#include "sc4slihd.lh"
#include "sc4sparkd.lh"
#include "sc4sbust.lh"
#include "sc4starpk.lh"
#include "sc4strk.lh"
#include "sc4sahed.lh"
#include "sc4susk.lh"
#include "sc4suscl.lh"
#include "sc4s6cd.lh"
#include "sc4taekw.lh"
#include "sc4takcl.lh"
#include "sc4tempt.lh"
#include "sc4lions.lh"
#include "sc4tic2.lh"
#include "sc4tbana.lh"
#include "sc4tload.lh"
#include "sc4tpshta.lh"
#include "sc4trail.lh"
#include "sc4ttomb.lh"
#include "sc4tridn.lh"
#include "sc4tfclb.lh"
#include "sc4vrgclb.lh"
#include "sc4wldjkc.lh"
#include "sc4xmarka.lh"



/*

    Scorpion 4 Hardware Platform (c)1996 Bell Fruit Manufacturing

    preliminary driver

    -----------------

    Scorpion 4:::

    Main CPU is a MC68307FG16, present on Motherboard

    Configuration is SC4 motherboard + game card

    The game card contains the program roms, sound rom and YMZ280B with a gal and optional RTC

    The GAL is only dumped for highroller, not sure if other boards have a different part.

    Adder 4 video board adds an additional card with a MC68340PV25E (25.175Mhz)

    -------------------------------

    The hopper(s) are not currently emulated, many of the games can
    be operated in 'Door Open' mode granting you free credits.

    Most games will show a RAM error on first boot due, after that they
    will initialize their NVRAM.

    If 'Read Meters' is shown press the 'Cancel' button (this moves around
    per game, so where it maps might not be obvious)  Doing this will allow
    the games to run in Door Open mode, pressing 'Start' (also moves around)
    will allow you to test the game.  Not all games have this feature.

    Pressing the service key ('Green Button') often allows test mode to be
    entered, some games have more comprehensive tests than others.

    Various (poorly programmed) sets require specific Jackpot 'keys' etc. to
    boot and won't even warn you if they're invalid, others allow you to
    set options if keys are not present. (again the buttons to do so move
    between games)

    Many games have missing sound roms, incorrect sound roms, or badly
    dumped sound roms.  We also have several dumps where only sound roms
    are present.

    Many of the titles here were also released on the SC5 platform.

*/





uint8_t sc4_state::read_input_matrix(int row)
{
	uint8_t value;

	if (row<4)
		value = (m_io_ports[row].read_safe(0x00) & 0x1f) + ((m_io_ports[row+8].read_safe(0x00) & 0x07) << 5);
	else
		value = (m_io_ports[row].read_safe(0x00) & 0x1f) + ((m_io_ports[row+4].read_safe(0x00) & 0x18) << 2);

	return value;
}

uint16_t sc4_state::sc4_cs1_r(offs_t offset, uint16_t mem_mask)
{
	int pc = m_maincpu->pc();

	if (offset<0x100000/2)
	{
		// allow some sets to boot, should probably return this data on Mbus once we figure out what it is
		if ((pc == m_chk41addr) && (offset == m_chk41addr>>1))
		{
			uint32_t r_A0 = m_maincpu->state_int(M68K_A0);
			uint32_t r_A1 = m_maincpu->state_int(M68K_A1);
			uint32_t r_D1 = m_maincpu->state_int(M68K_D1);

			if (r_D1 == 0x7)
			{
				auto &mspace = m_maincpu->space(AS_PROGRAM);
				bool valid = true;
				for (int i=0;i<8;i++)
				{
					uint8_t code = mspace.read_byte(r_A0+i);
					if (code != 0xff) // assume our mbus code just returns 0xff for now..
						valid = false;
				}

				if (valid && m_dochk41)
				{
					m_dochk41 = false;
					// the value is actually random.. probably based on other reads
					// making this a comparison?
					printf("Ident code? ");
					for (int i=0;i<8;i++)
					{
						uint8_t code = mspace.read_byte(r_A1+i);
						printf("%02x",code);
						mspace.write_byte(r_A0+i, code);
					}
					printf("\n");
				}
			}
		}


		return m_cpuregion->as_u16(offset);
	}
	else
		logerror("%08x maincpu read access offset %08x mem_mask %04x cs %d\n", pc, offset*2, mem_mask, 1);

	return 0x0000;
}

uint16_t sc4_state::sc4_mem_r(offs_t offset, uint16_t mem_mask)
{
	int pc = m_maincpu->pc();
	int cs = m_maincpu->get_cs(offset * 2);
	int base = 0, end = 0, base2 = 0, end2 = 0;
//  if (!(debugger_access())) printf("cs is %d\n", cs);
	uint16_t retvalue;


	switch ( cs )
	{
		case 1:
			return sc4_cs1_r(offset);

		case 2:
			base = 0x800000/2;
			end = base + 0x10000 / 2;

			base2 = 0x810000/2;
			end2 = base2 + 0x10000 / 2;


			if ((offset >= base) && (offset < end))
			{
				offset -= base;
				return m_mainram[offset];
			}
			else if ((offset >= base2) && (offset < end2))
			{
				offset -= base2;
				logerror("%08x maincpu read access offset %08x mem_mask %04x cs %d\n", pc, offset*2, mem_mask, cs);
				int addr = (offset<<1);

				if (addr < 0x0080)
				{
					uint16_t retvalue = 0x0000;
					if (ACCESSING_BITS_8_15)
					{
						logerror("mem_mask&0xff00 unhandled\n");
					}

					if (ACCESSING_BITS_0_7)
					{
						retvalue = read_input_matrix((addr & 0x00f0)>>4);
					}
					return retvalue;
				}
				else
				{
					switch (addr)
					{
						case 0x0240:
							retvalue = 0x00ff;

							if (ACCESSING_BITS_8_15)
							{
								retvalue |= (m_sec->data_r() << 14);
								retvalue |= m_in_coin->read() << 8; // coin?
								//printf("%08x maincpu read access offset %08x mem_mask %04x cs %d (LAMPS etc.)\n", pc, offset*2, mem_mask, cs);
							}
							return retvalue;

						case 0x02e0:
							return 0x0080; // status of something?

						case 0x1000:
							return m_in[16]->read();

						case 0x1010:
							return m_in[17]->read();

						case 0x1020:
							return m_in[18]->read();

						case 0x1030:
							return m_in[19]->read();

						case 0x1040: // door switch, test switch etc.
							return m_in[20]->read();

						case 0x1244:
							return m_ymz->read(0);

						case 0x1246:
							return m_ymz->read(1);

						default:
							logerror("%08x maincpu read access offset %08x mem_mask %04x cs %d (LAMPS etc.)\n", pc, offset*2, mem_mask, cs);
					}
				}
			}
			else
			{
				logerror("%08x maincpu read access offset %08x mem_mask %04x cs %d\n", pc, offset*2, mem_mask, cs);
			}
			break;

		case 3:
			base = 0xc00000/2;
			end = base + 0x20 / 2;

			if ((offset>=base) && (offset<end))
			{
				offset-=base;
				return m_duart->read(offset);
			}
			else
			{
				logerror("%08x maincpu read access offset %08x mem_mask %04x cs %d\n", pc, offset*2, mem_mask, cs);
				return 0x0000;
			}

		case 4:
			logerror("%08x maincpu read access offset %08x mem_mask %04x cs %d\n", pc, offset*2, mem_mask, cs);
			return 0x0000;//0xffff;

		default:
			logerror("%08x maincpu read access offset %08x mem_mask %04x cs %d (invalid?)\n", pc, offset*2, mem_mask, cs);

	}

	return 0x0000;
}

void bfm_sc45_state::machine_start()
{
	m_lamps.resolve();
	m_matrix.resolve();
	m_digits.resolve();
}

void bfm_sc45_state::mux_output_w(offs_t offset, uint8_t data)
{
	int const off = offset<<3;

	for (int i=0; i<8; i++)
		m_lamps[off+i] = BIT(data, i);
}

void bfm_sc45_state::mux_output2_w(offs_t offset, uint8_t data)
{
	int const off = offset<<3;

	// some games use this as a matrix port (luckb etc.)
	for (int i=0; i<8; i++)
		m_matrix[off+i] = BIT(data, i);

	// others drive 7-segs with it..  so rendering it there as well in our debug layouts

	if (m_segment_34_encoding)
	{
		m_segment_34_cache[offset] = data;

		for (int digit = 0; digit < 32; digit += 2)
		{
			uint16_t const short_data = (m_segment_34_cache[digit + 1] << 8) | m_segment_34_cache[digit];
			uint8_t byte_data_first_segment = (short_data >> 1) & 15;
			uint8_t const byte_data_second_segment = (short_data >> 6) & 15;
			m_digits[digit] = SEGMENT_34_ENCODING_LOOKUP[byte_data_first_segment];
			m_digits[digit + 1] = SEGMENT_34_ENCODING_LOOKUP[byte_data_second_segment];

			if (digit == 0 || digit == 2)
			{
				byte_data_first_segment = (short_data >> 11) & 15;
				m_digits[(digit / 2) + 32] = SEGMENT_34_ENCODING_LOOKUP[byte_data_first_segment];
			}
		}
	}
	else
	{
		m_digits[offset] = bitswap<8>(data,0,7,6,5,4,3,2,1);
	}
}

void sc4_state::sc4_mem_w(offs_t offset, uint16_t data, uint16_t mem_mask)
{
	int pc = m_maincpu->pc();
	int cs = m_maincpu->get_cs(offset * 2);
	int base = 0, end = 0, base2 = 0, end2 = 0;

	switch ( cs )
	{
		case 1:
			if (offset<0x100000/2)
				logerror("%08x maincpu write access offset %08x data %04x mem_mask %04x cs %d (ROM WRITE?!)\n", pc, offset*2, data, mem_mask, cs);
			else
				logerror("%08x maincpu write access offset %08x data %04x mem_mask %04x cs %d\n", pc, offset*2, data, mem_mask, cs);

			break;

		case 2:
			base = 0x800000/2;
			end = base + 0x10000 / 2;
			base2 = 0x810000/2;
			end2 = base2 + 0x10000 / 2;

			if ((offset>=base) && (offset<end))
			{
				offset-=base;
				COMBINE_DATA(&m_mainram[offset]);
			}
			else if ((offset>=base2) && (offset<end2))
			{
				offset-=base2;
				int addr = (offset<<1);

				if (addr < 0x0200)
				{
					if (ACCESSING_BITS_8_15)
					{
						logerror("lamp write mem_mask&0xff00 unhandled\n");
					}

					if (ACCESSING_BITS_0_7)
					{   // lamps
						mux_output_w((addr & 0x01f0)>>4, data);
					}

				}
				else if ((addr >= 0x1000) && (addr < 0x1200))
				{
					if (ACCESSING_BITS_8_15)
					{
						logerror("lamp write mem_mask&0xff00 unhandled\n");
					}

					if (ACCESSING_BITS_0_7)
					{   // lamps
						mux_output2_w((addr & 0x01f0)>>4, data);
					}
				}
				else
				{
					switch (addr)
					{
						case 0x0330:
							logerror("%08x meter write %04x\n",pc, data);
							//m_meterstatus = (m_meterstatus&0xc0) | (data & 0x3f);
							m_sec->clk_w(!BIT(data, 5));
							break;

						case 0x1248:
							m_ymz->write(0, data & 0xff);
							break;

						case 0x124a:
							m_ymz->write(1, data & 0xff);
							break;

						case 0x1330:
							bfm_sc4_reel4_w(data&0xf);
							//m_meterstatus = (m_meterstatus&0x3f) | ((data & 0x30) << 2);
							m_sec->data_w(!BIT(data, 4));
							m_sec->cs_w(!BIT(data, 5));
							break;

						default:
							logerror("%08x maincpu write access offset %08x data %04x mem_mask %04x cs %d (LAMPS etc.)\n", pc, offset*2, data, mem_mask, cs);
					}
				}
			}
			else
			{
				logerror("%08x maincpu write access offset %08x data %04x mem_mask %04x cs %d\n", pc, offset*2, data, mem_mask, cs);
			}
			break;

		case 3:
			base = 0xc00000/2;
			end = base + 0x20 / 2;

			if ((offset>=base) && (offset<end))
			{
				offset-=base;
				m_duart->write(offset,data&0x00ff);
			}
			else
			{
				logerror("%08x maincpu write access offset %08x data %04x mem_mask %04x cs %d\n", pc, offset*2, data, mem_mask, cs);
			}

			break;

		case 4:
			logerror("%08x maincpu write access offset %08x data %04x mem_mask %04x cs %d\n", pc, offset*2, data, mem_mask, cs);
			break;

		default:
			logerror("%08x maincpu write access offset %08x data %04x mem_mask %04x cs %d (invalid?)\n", pc, offset*2, data, mem_mask, cs);
	}
}

void sc4_state::sc4_map(address_map &map)
{
	map(0x0000000, 0xffffff).rw(FUNC(sc4_state::sc4_mem_r), FUNC(sc4_state::sc4_mem_w));
	map(0x0000000, 0x0fffff).r(FUNC(sc4_state::sc4_cs1_r)); // technically we should be going through the cs handler, but this is always set to ROM, and assuming that is a lot faster
}




uint16_t sc4_adder4_state::adder4_mem_r(offs_t offset, uint16_t mem_mask)
{
	int pc = m_adder4cpu->pc();
	int cs = m_adder4cpu->get_cs(offset * 2);

	switch ( cs )
	{
		case 1:
			return m_adder4cpuregion[offset];

		case 2:
			offset &=0x7fff;
			return m_adder4ram[offset];

		default:
			//logerror("%08x adder4cpu read access offset %08x mem_mask %08x cs %d\n", pc, offset*4, mem_mask, cs);
			(void)pc;
			break;
	}

	return 0x0000;
}

void sc4_adder4_state::adder4_mem_w(offs_t offset, uint16_t data, uint16_t mem_mask)
{
	int pc = m_adder4cpu->pc();
	int cs = m_adder4cpu->get_cs(offset * 2);

	switch ( cs )
	{
		default:
			//logerror("%08x adder4cpu write access offset %08x data %08x mem_mask %08x cs %d\n", pc, offset*4, data, mem_mask, cs);
			(void)pc;
			break;

		case 2:
			offset &=0x7fff;
			COMBINE_DATA(&m_adder4ram[offset]);
			break;


	}

}

void sc4_adder4_state::sc4_adder4_map(address_map &map)
{
	map(0x00000000, 0xffffffff).rw(FUNC(sc4_adder4_state::adder4_mem_r), FUNC(sc4_adder4_state::adder4_mem_w));
}








void bfm_sc45_state::bfm_sc4_reset_serial_vfd()
{
	if (m_vfd0)
	{
		m_vfd0->reset();
	}
	else
	{
		m_vfd1->reset();
	}
	vfd_old_clock = false;
}

void bfm_sc45_state::bfm_sc45_write_serial_vfd(bool cs, bool clock, bool data)
{
	// if we're turned on
	if ( cs )
	{
		if ( !vfd_enabled )
		{
			bfm_sc4_reset_serial_vfd();
			vfd_old_clock = clock;
			vfd_enabled = true;
		}
		else
		{
			// if the clock line changes
			if ( clock != vfd_old_clock )
			{
				if ( !clock )
				{
				//Should move to the internal serial process when DM01 is device-ified
//                  m_vfd0->shift_data(!data);
					vfd_ser_value <<= 1;
					if (data) vfd_ser_value |= 1;

					vfd_ser_count++;
					if ( vfd_ser_count == 8 )
					{
						vfd_ser_count = 0;
						if (m_dm01)
						{
							m_dm01->writedata(vfd_ser_value);
						}
						else if (m_vfd0)
						{
							m_vfd0->write_char(vfd_ser_value);
						}
						else
						{
							m_vfd1->write_char(vfd_ser_value);
						}
					}
				}
				vfd_old_clock = clock;
			}
		}
	}
	else
	{
		vfd_enabled = false;
	}
}


void sc4_state::bfm_sc4_68307_porta_w(address_space &space, bool dedicated, uint8_t data, uint8_t line_mask)
{
	m_reel12_latch = data;

	if(m_reel[0])
	{
		m_reel[0]->update( data    &0x0f);
		awp_draw_reel(machine(),"reel1", *m_reel[0]);
	}

	if (m_reel[1])
	{
		m_reel[1]->update((data>>4)&0x0f);
		awp_draw_reel(machine(),"reel2", *m_reel[1]);
	}
}

void sc4_state::bfm_sc4_reel3_w(uint8_t data)
{
	m_reel3_latch = data;

	if(m_reel[2])
	{
		m_reel[2]->update( data    &0x0f);
		awp_draw_reel(machine(),"reel3", *m_reel[2]);
	}
}

void sc4_state::bfm_sc4_reel4_w(uint8_t data)
{
	m_reel4_latch = data;

	if(m_reel[3])
	{
		m_reel[3]->update( data    &0x0f);
		awp_draw_reel(machine(),"reel4", *m_reel[3]);
	}
}

void sc4_state::bfm_sc4_68307_portb_w(address_space &space, bool dedicated, uint16_t data, uint16_t line_mask)
{
//  if (dedicated == false)
	{
		int pc = m_maincpu->pc();
		// serial output to the VFD at least..
		logerror("%08x bfm_sc4_68307_portb_w %04x %04x\n", pc, data, line_mask);

		bfm_sc45_write_serial_vfd((data & 0x4000)?1:0, (data & 0x1000)?1:0, !(data & 0x2000)?1:0);

		bfm_sc4_reel3_w((data&0x0f00)>>8);
	}

}
uint8_t sc4_state::bfm_sc4_68307_porta_r(address_space &space, bool dedicated, uint8_t line_mask)
{
	int pc = m_maincpu->pc();
	logerror("%08x bfm_sc4_68307_porta_r\n", pc);
	return 0xbb;// machine().rand();
}

uint16_t sc4_state::bfm_sc4_68307_portb_r(address_space &space, bool dedicated, uint16_t line_mask)
{
	if (dedicated==false)
	{
		return 0x0000;
	}
	else
	{
		// generating certain interrupts expects the bit 0x8000 to be set here
		// but it's set to dedicated i/o, not general purpose, source?
		return 0x8040;
	}
}

void sc4_state::machine_reset()
{
	bfm_sc45_state::machine_reset();

	m_dochk41 = true;
}


void sc4_state::machine_start()
{
	bfm_sc45_state::machine_start();

	m_nvram->set_base(m_mainram, sizeof(m_mainram));

	m_maincpu->set_port_callbacks(
			m68307_cpu_device::porta_read_delegate(*this, FUNC(sc4_state::bfm_sc4_68307_porta_r)),
			m68307_cpu_device::porta_write_delegate(*this, FUNC(sc4_state::bfm_sc4_68307_porta_w)),
			m68307_cpu_device::portb_read_delegate(*this, FUNC(sc4_state::bfm_sc4_68307_portb_r)),
			m68307_cpu_device::portb_write_delegate(*this, FUNC(sc4_state::bfm_sc4_68307_portb_w)));
}


void sc4_state::bfm_sc4_duart_irq_handler(int state)
{
	// triggers after reel tests on luckb, at the start on dnd...
	// not sure this is right, causes some games to crash
	logerror("bfm_sc4_duart_irq_handler\n");
	if (state == ASSERT_LINE)
	{
		m_maincpu->licr2_interrupt();
	}
}

void sc4_state::bfm_sc4_duart_txa(int state)
{
	logerror("bfm_sc4_duart_tx\n");
}



uint8_t sc4_state::bfm_sc4_duart_input_r()
{
	//  printf("bfm_sc4_duart_input_r\n");
	return m_optic_pattern;
}

void sc4_state::bfm_sc4_duart_output_w(uint8_t data)
{
//  logerror("bfm_sc4_duart_output_w\n");
	m_reel56_latch = data;

	if(m_reel[4])
	{
		m_reel[4]->update( data    &0x0f);
		awp_draw_reel(machine(),"reel5", *m_reel[4]);
	}

	if (m_reel[5])
	{
		m_reel[5]->update((data>>4)&0x0f);
		awp_draw_reel(machine(),"reel6", *m_reel[5]);
	}
}


void sc4_state::m68307_duart_txa(int state)
{
	logerror("m68307_duart_tx %02x\n", state);
}

uint8_t sc4_state::m68307_duart_input_r()
{
	logerror("m68307_duart_input_r\n");
	return 0x00;
}

void sc4_state::m68307_duart_output_w(uint8_t data)
{
	logerror("m68307_duart_output_w %02x\n", data);
}

/* default dmd */
void sc4_state::bfmdm01_busy(int state)
{
	// Must tie back to inputs somehow!
}

void sc4_state::sc4_common(machine_config &config)
{
	M68307(config, m_maincpu, 16000000);    // 68307! (EC000 core)
	m_maincpu->set_addrmap(AS_PROGRAM, &sc4_state::sc4_map);
	m_maincpu->serial_a_tx_callback().set(FUNC(sc4_state::m68307_duart_txa));
	m_maincpu->serial_inport_callback().set(FUNC(sc4_state::m68307_duart_input_r));
	m_maincpu->serial_outport_callback().set(FUNC(sc4_state::m68307_duart_output_w));

	/* sound hardware */
	SPEAKER(config, "mono").front_center();

	NVRAM(config, "nvram", nvram_device::DEFAULT_ALL_1);

	MC68681(config, m_duart, 16000000/4); // ?? Mhz
	m_duart->set_clocks(XTAL(16'000'000)/2/8, XTAL(16'000'000)/2/16, XTAL(16'000'000)/2/16, XTAL(16'000'000)/2/8);
	m_duart->irq_cb().set(FUNC(sc4_state::bfm_sc4_duart_irq_handler));
	m_duart->a_tx_cb().set(FUNC(sc4_state::bfm_sc4_duart_txa));
	m_duart->inport_cb().set(FUNC(sc4_state::bfm_sc4_duart_input_r));
	m_duart->outport_cb().set(FUNC(sc4_state::bfm_sc4_duart_output_w));

	BFM_BDA(config, m_vfd0, 60, 0);

//  config.set_default_layout(layout_bfm_sc4);

	YMZ280B(config, m_ymz, 16000000); // ?? Mhz
	m_ymz->add_route(ALL_OUTPUTS, "mono", 1.0);

	SEC(config, m_sec);
}

//Standard 6 reels all connected
void sc4_state::sc4(machine_config &config)
{
	sc4_common(config);

	REEL(config, m_reel[0], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[0]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<0>));
	REEL(config, m_reel[1], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[1]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<1>));
	REEL(config, m_reel[2], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[2]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<2>));
	REEL(config, m_reel[3], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[3]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<3>));
	REEL(config, m_reel[4], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[4]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<4>));
	REEL(config, m_reel[5], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[5]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<5>));
}

//Standard 3 reels
void sc4_state::sc4_3reel(machine_config &config)
{
	sc4_common(config);

	REEL(config, m_reel[0], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[0]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<0>));
	REEL(config, m_reel[1], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[1]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<1>));
	REEL(config, m_reel[2], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[2]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<2>));

}

//Standard 4 reels
void sc4_state::sc4_4reel(machine_config &config)
{
	sc4_common(config);

	REEL(config, m_reel[0], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[0]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<0>));
	REEL(config, m_reel[1], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[1]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<1>));
	REEL(config, m_reel[2], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[2]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<2>));
	REEL(config, m_reel[3], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[3]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<3>));
}

//4 reels, with the last connected to RL4 not RL3
void sc4_state::sc4_4reel_alt(machine_config &config)
{

	sc4_common(config);

	REEL(config, m_reel[0], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[0]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<0>));
	REEL(config, m_reel[1], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[1]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<1>));
	REEL(config, m_reel[2], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[2]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<2>));

	REEL(config, m_reel[4], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[4]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<4>));
}


//Standard 5 reels
void sc4_state::sc4_5reel(machine_config &config)
{
	sc4_common(config);

	REEL(config, m_reel[0], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[0]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<0>));
	REEL(config, m_reel[1], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[1]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<1>));
	REEL(config, m_reel[2], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[2]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<2>));
	REEL(config, m_reel[3], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[3]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<3>));
	REEL(config, m_reel[4], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[4]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<4>));
}

//5 reels, with RL4 skipped
void sc4_state::sc4_5reel_alt(machine_config &config)
{
	sc4_common(config);

	REEL(config, m_reel[0], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[0]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<0>));
	REEL(config, m_reel[1], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[1]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<1>));
	REEL(config, m_reel[2], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[2]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<2>));

	REEL(config, m_reel[4], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[4]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<4>));
	REEL(config, m_reel[5], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[5]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<5>));

}

//6 reels, last 200 steps
void sc4_state::sc4_200_std(machine_config &config)
{

	sc4_common(config);

	REEL(config, m_reel[0], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[0]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<0>));
	REEL(config, m_reel[1], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[1]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<1>));
	REEL(config, m_reel[2], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[2]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<2>));
	REEL(config, m_reel[3], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[3]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<3>));
	REEL(config, m_reel[4], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[4]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<4>));
	REEL(config, m_reel[5], STARPOINT_200STEP_REEL, 12, 24, 0x09, 7, 200*2);
	m_reel[5]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<5>));
}

//6 reels, last 200 steps
void sc4_state::sc4_200_alt(machine_config &config)
{
	sc4_common(config);

	REEL(config, m_reel[0], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[0]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<0>));
	REEL(config, m_reel[1], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[1]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<1>));
	REEL(config, m_reel[2], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[2]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<2>));
	REEL(config, m_reel[3], STARPOINT_200STEP_REEL, 12, 24, 0x09, 7, 200*2);
	m_reel[3]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<3>));
	REEL(config, m_reel[4], STARPOINT_200STEP_REEL, 12, 24, 0x09, 7, 200*2);
	m_reel[4]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<4>));
	REEL(config, m_reel[5], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[5]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<5>));
}

//6 reels, RL4 200 steps
void sc4_state::sc4_200_alta(machine_config &config)
{
	sc4_common(config);

	REEL(config, m_reel[0], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[0]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<0>));
	REEL(config, m_reel[1], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[1]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<1>));
	REEL(config, m_reel[2], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[2]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<2>));
	REEL(config, m_reel[3], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[3]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<3>));
	REEL(config, m_reel[4], STARPOINT_200STEP_REEL, 12, 24, 0x09, 7, 200*2);
	m_reel[4]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<4>));
	REEL(config, m_reel[5], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[5]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<5>));
}

//6 reels, 3 48 step, 3 200 step
void sc4_state::sc4_200_altb(machine_config &config)
{
	sc4_common(config);

	REEL(config, m_reel[0], STARPOINT_200STEP_REEL, 12, 24, 0x09, 7, 200*2);
	m_reel[0]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<0>));
	REEL(config, m_reel[1], STARPOINT_200STEP_REEL, 12, 24, 0x09, 7, 200*2);
	m_reel[1]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<1>));
	REEL(config, m_reel[2], STARPOINT_200STEP_REEL, 12, 24, 0x09, 7, 200*2);
	m_reel[2]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<2>));
	REEL(config, m_reel[3], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[3]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<3>));
	REEL(config, m_reel[4], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[4]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<4>));
	REEL(config, m_reel[5], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[5]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<5>));
}

//5 reels, last one 200 steps
void sc4_state::sc4_200_5r(machine_config &config)
{
	sc4_common(config);

	REEL(config, m_reel[0], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[0]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<0>));
	REEL(config, m_reel[1], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[1]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<1>));
	REEL(config, m_reel[2], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[2]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<2>));
	REEL(config, m_reel[3], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[3]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<3>));
	REEL(config, m_reel[4], STARPOINT_200STEP_REEL, 12, 24, 0x09, 7, 200*2);
	m_reel[4]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<4>));
}



//5 reels, last one 200 steps, RL4 skipped
void sc4_state::sc4_200_5ra(machine_config &config)
{
	sc4_common(config);

	REEL(config, m_reel[0], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[0]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<0>));
	REEL(config, m_reel[1], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[1]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<1>));
	REEL(config, m_reel[2], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[2]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<2>));

	REEL(config, m_reel[4], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[4]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<4>));
	REEL(config, m_reel[5], STARPOINT_200STEP_REEL, 12, 24, 0x09, 7, 200*2);
	m_reel[5]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<5>));
}

//5 reels, last one 200 steps, RL5 skipped
void sc4_state::sc4_200_5rb(machine_config &config)
{
	sc4_common(config);

	REEL(config, m_reel[0], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[0]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<0>));
	REEL(config, m_reel[1], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[1]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<1>));
	REEL(config, m_reel[2], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[2]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<2>));
	REEL(config, m_reel[3], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[3]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<3>));

	REEL(config, m_reel[5], STARPOINT_200STEP_REEL, 12, 24, 0x09, 7, 200*2);
	m_reel[5]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<5>));
}

//5 reels, RL5 200 steps, RL4 skipped
void sc4_state::sc4_200_5rc(machine_config &config)
{
	sc4_common(config);

	REEL(config, m_reel[0], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[0]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<0>));
	REEL(config, m_reel[1], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[1]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<1>));
	REEL(config, m_reel[2], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[2]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<2>));

	REEL(config, m_reel[4], STARPOINT_200STEP_REEL, 12, 24, 0x09, 7, 200*2);
	m_reel[4]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<4>));
	REEL(config, m_reel[5], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[5]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<5>));
}

//4 reels, last one 200 steps
void sc4_state::sc4_200_4r(machine_config &config)
{
	sc4_common(config);

	REEL(config, m_reel[0], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[0]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<0>));
	REEL(config, m_reel[1], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[1]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<1>));
	REEL(config, m_reel[2], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[2]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<2>));
	REEL(config, m_reel[3], STARPOINT_200STEP_REEL, 12, 24, 0x09, 7, 200*2);
	m_reel[3]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<3>));
}

//4 reels, last one 200 steps, RL4 skipped
void sc4_state::sc4_200_4ra(machine_config &config)
{
	sc4_common(config);

	REEL(config, m_reel[0], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[0]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<0>));
	REEL(config, m_reel[1], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[1]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<1>));
	REEL(config, m_reel[2], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[2]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<2>));

	REEL(config, m_reel[4], STARPOINT_200STEP_REEL, 12, 24, 0x09, 7, 200*2);
	m_reel[4]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<4>));
}


//4 reels, last one 200 steps, RL4,5 skipped
void sc4_state::sc4_200_4rb(machine_config &config)
{
	sc4_common(config);

	REEL(config, m_reel[0], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[0]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<0>));
	REEL(config, m_reel[1], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[1]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<1>));
	REEL(config, m_reel[2], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[2]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<2>));

	REEL(config, m_reel[5], STARPOINT_200STEP_REEL, 12, 24, 0x09, 7, 200*2);
	m_reel[5]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<5>));
}

void sc4_state::sc4_4reel_200(machine_config &config)
{
	sc4_common(config);

	REEL(config, m_reel[0], STARPOINT_200STEP_REEL, 12, 24, 0x09, 7, 200*2);
	m_reel[0]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<0>));
	REEL(config, m_reel[1], STARPOINT_200STEP_REEL, 12, 24, 0x09, 7, 200*2);
	m_reel[1]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<1>));
	REEL(config, m_reel[2], STARPOINT_200STEP_REEL, 12, 24, 0x09, 7, 200*2);
	m_reel[2]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<2>));
	REEL(config, m_reel[3], STARPOINT_200STEP_REEL, 12, 24, 0x09, 7, 200*2);
	m_reel[3]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<3>));
}

void sc4_state::sc4_3reel_200(machine_config &config)
{
	sc4_common(config);

	REEL(config, m_reel[0], STARPOINT_200STEP_REEL, 12, 24, 0x09, 7, 200*2);
	m_reel[0]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<0>));
	REEL(config, m_reel[1], STARPOINT_200STEP_REEL, 12, 24, 0x09, 7, 200*2);
	m_reel[1]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<1>));
	REEL(config, m_reel[2], STARPOINT_200STEP_REEL, 12, 24, 0x09, 7, 200*2);
	m_reel[2]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<2>));
}

void sc4_state::sc4_3reel_200_48(machine_config &config)
{

	sc4_common(config);

	REEL(config, m_reel[0], STARPOINT_200STEP_REEL, 12, 24, 0x09, 7, 200*2);
	m_reel[0]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<0>));
	REEL(config, m_reel[1], STARPOINT_200STEP_REEL, 12, 24, 0x09, 7, 200*2);
	m_reel[1]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<1>));
	REEL(config, m_reel[2], STARPOINT_200STEP_REEL, 12, 24, 0x09, 7, 200*2);
	m_reel[2]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<2>));
	REEL(config, m_reel[3], STARPOINT_48STEP_REEL, 1, 3, 0x09, 4);
	m_reel[3]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<3>));
}

void sc4_state::sc4_no_reels(machine_config &config)
{
	sc4_common(config);
}

void sc4_adder4_state::machine_start()
{
	sc4_state::machine_start();

	m_adder4ram = make_unique_clear<uint16_t[]>(0x8000);
}

void sc4_adder4_state::sc4_adder4(machine_config &config)
{
	sc4_common(config);

	M68340(config, m_adder4cpu, 25175000);     // 68340 (CPU32 core)
	m_adder4cpu->set_addrmap(AS_PROGRAM, &sc4_adder4_state::sc4_adder4_map);
}

void sc4_state::sc4dmd(machine_config &config)
{
	sc4_common(config);
	/* video hardware */

	//config.set_default_layout(layout_sc4_dmd);
	BFM_DM01(config, m_dm01, 0);
	m_dm01->busy_callback().set(FUNC(sc4_state::bfmdm01_busy));

	REEL(config, m_reel[0], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[0]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<0>));
	REEL(config, m_reel[1], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[1]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<1>));
	REEL(config, m_reel[2], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[2]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<2>));
	REEL(config, m_reel[3], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[3]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<3>));
	REEL(config, m_reel[4], STARPOINT_48STEP_REEL, 16, 24, 0x09, 7);
	m_reel[4]->optic_handler().set(FUNC(sc4_state::reel_optic_cb<4>));
}

INPUT_PORTS_START( sc4_raw ) // completley unmapped, but named inputs for all the ports, used for testing.
	PORT_START("IN-0")
	PORT_DIPNAME( 0x01, 0x00, "IN 0-0 (STRB 0 Data 0)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x01, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x00, "IN 0-1 (STRB 0 Data 1)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x02, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x00, "IN 0-2 (STRB 0 Data 2)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x04, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x00, "IN 0-3 (STRB 0 Data 3)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x08, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x00, "IN 0-4 (STRB 0 Data 4)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x10, DEF_STR( On ) )
	PORT_BIT(           0xffe0, IP_ACTIVE_HIGH, IPT_UNUSED )

	PORT_START("IN-1")
	PORT_DIPNAME( 0x01, 0x00, "IN 1-0 (STRB 1 Data 0)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x01, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x00, "IN 1-1 (STRB 1 Data 1)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x02, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x00, "IN 1-2 (STRB 1 Data 2)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x04, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x00, "IN 1-3 (STRB 1 Data 3)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x08, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x00, "IN 1-4 (STRB 1 Data 4)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x10, DEF_STR( On ) )
	PORT_BIT(           0xffe0, IP_ACTIVE_HIGH, IPT_UNUSED )

	PORT_START("IN-2")
	PORT_DIPNAME( 0x01, 0x00, "IN 2-0 (STRB 2 Data 0)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x01, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x00, "IN 2-1 (STRB 2 Data 1)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x02, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x00, "IN 2-2 (STRB 2 Data 2)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x04, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x00, "IN 2-3 (STRB 2 Data 3)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x08, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x00, "IN 2-4 (STRB 2 Data 4)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x10, DEF_STR( On ) )
	PORT_BIT(           0xffe0, IP_ACTIVE_HIGH, IPT_UNUSED )

	PORT_START("IN-3")
	PORT_DIPNAME( 0x01, 0x00, "IN 3-0 (STRB 3 Data 0)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x01, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x00, "IN 3-1 (STRB 3 Data 1)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x02, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x00, "IN 3-2 (STRB 3 Data 2)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x04, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x00, "IN 3-3 (STRB 3 Data 3)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x08, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x00, "IN 3-4 (STRB 3 Data 4)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x10, DEF_STR( On ) )
	PORT_BIT(           0xffe0, IP_ACTIVE_HIGH, IPT_UNUSED )

	PORT_START("IN-4")
	PORT_DIPNAME( 0x01, 0x00, "IN 4-0 (STRB 4 Data 0)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x01, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x00, "IN 4-1 (STRB 4 Data 1)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x02, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x00, "IN 4-2 (STRB 4 Data 2)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x04, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x00, "IN 4-3 (STRB 4 Data 3)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x08, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x00, "IN 4-4 (STRB 4 Data 4)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x10, DEF_STR( On ) )
	PORT_BIT(           0xffe0, IP_ACTIVE_HIGH, IPT_UNUSED )


	PORT_START("IN-5")
	PORT_DIPNAME( 0x01, 0x00, "IN 5-0 (STRB 5 Data 0)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x01, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x00, "IN 5-1 (STRB 5 Data 1)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x02, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x00, "IN 5-2 (STRB 5 Data 2)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x04, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x00, "IN 5-3 (STRB 5 Data 3)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x08, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x00, "IN 5-4 (STRB 5 Data 4)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x10, DEF_STR( On ) )
	PORT_BIT(           0xffe0, IP_ACTIVE_HIGH, IPT_UNUSED )

	PORT_START("IN-6")
	PORT_DIPNAME( 0x01, 0x00, "IN 6-0 (STRB 6 Data 0)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x01, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x00, "IN 6-1 (STRB 6 Data 1)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x02, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x00, "IN 6-2 (STRB 6 Data 2)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x04, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x00, "IN 6-3 (STRB 6 Data 3)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x08, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x00, "IN 6-4 (STRB 6 Data 4)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x10, DEF_STR( On ) )
	PORT_BIT(           0xffe0, IP_ACTIVE_HIGH, IPT_UNUSED )

	PORT_START("IN-7")
	PORT_DIPNAME( 0x01, 0x00, "IN 7-0 (STRB 7 Data 0)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x01, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x00, "IN 7-1 (STRB 7 Data 1)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x02, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x00, "IN 7-2 (STRB 7 Data 2)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x04, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x00, "IN 7-3 (STRB 7 Data 3)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x08, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x00, "IN 7-4 (STRB 7 Data 4)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x10, DEF_STR( On ) )
	PORT_BIT(           0xffe0, IP_ACTIVE_HIGH, IPT_UNUSED )

	PORT_START("IN-8")
	PORT_DIPNAME( 0x01, 0x00, "IN 8-0 (STRB 8 Data 0)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x01, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x00, "IN 8-1 (STRB 8 Data 1)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x02, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x00, "IN 8-2 (STRB 8 Data 2)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x04, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x00, "IN 8-3 (STRB 8 Data 3)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x08, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x00, "IN 8-4 (STRB 8 Data 4)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x10, DEF_STR( On ) )
	PORT_BIT(           0xffe0, IP_ACTIVE_HIGH, IPT_UNUSED )


	PORT_START("IN-9")
	PORT_DIPNAME( 0x01, 0x00, "IN 9-0 (STRB 9 Data 0)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x01, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x00, "IN 9-1 (STRB 9 Data 1)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x02, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x00, "IN 9-2 (STRB 9 Data 2)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x04, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x00, "IN 9-3 (STRB 9 Data 3)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x08, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x00, "IN 9-4 (STRB 9 Data 4)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x10, DEF_STR( On ) )
	PORT_BIT(           0xffe0, IP_ACTIVE_HIGH, IPT_UNUSED )

	PORT_START("IN-10")
	PORT_DIPNAME( 0x01, 0x00, "IN 10-0 (STRB 10 Data 0)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x01, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x00, "IN 10-1 (STRB 10 Data 1)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x02, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x00, "IN 10-2 (STRB 10 Data 2)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x04, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x00, "IN 10-3 (STRB 10 Data 3)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x08, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x00, "IN 10-4 (STRB 10 Data 4)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x10, DEF_STR( On ) )
	PORT_BIT(           0xffe0, IP_ACTIVE_HIGH, IPT_UNUSED )


	PORT_START("IN-11")
	PORT_DIPNAME( 0x01, 0x00, "IN 11-0 (STRB 11 Data 0)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x01, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x00, "IN 11-1 (STRB 11 Data 1)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x02, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x00, "IN 11-2 (STRB 11 Data 2)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x04, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x00, "IN 11-3 (STRB 11 Data 3)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x08, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x00, "IN 11-4 (STRB 11 Data 4)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x10, DEF_STR( On ) )
	PORT_BIT(           0xffe0, IP_ACTIVE_HIGH, IPT_UNUSED )

	// where are 11,12,13,14,15 ?

	PORT_START("IN-16")
	PORT_DIPNAME( 0x01, 0x00, "IN 16-0 (STRB 16 Data 0)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x01, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x00, "IN 16-1 (STRB 16 Data 1)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x02, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x00, "IN 16-2 (STRB 16 Data 2)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x04, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x00, "IN 16-3 (STRB 16 Data 3)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x08, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x00, "IN 16-4 (STRB 16 Data 4)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x10, DEF_STR( On ) )
	PORT_BIT(           0xffe0, IP_ACTIVE_HIGH, IPT_UNUSED )

	PORT_START("IN-17")
	PORT_DIPNAME( 0x01, 0x00, "IN 17-0 (STRB 17 Data 0)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x01, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x00, "IN 17-1 (STRB 17 Data 1)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x02, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x00, "IN 17-2 (STRB 17 Data 2)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x04, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x00, "IN 17-3 (STRB 17 Data 3)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x08, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x00, "IN 17-4 (STRB 17 Data 4)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x10, DEF_STR( On ) )
	PORT_BIT(           0xffe0, IP_ACTIVE_HIGH, IPT_UNUSED )

	PORT_START("IN-18")
	PORT_DIPNAME( 0x01, 0x00, "IN 18-0 (STRB 18 Data 0)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x01, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x00, "IN 18-1 (STRB 18 Data 1)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x02, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x00, "IN 18-2 (STRB 18 Data 2)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x04, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x00, "IN 18-3 (STRB 18 Data 3)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x08, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x00, "IN 18-4 (STRB 18 Data 4)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x10, DEF_STR( On ) )
	PORT_BIT(           0xffe0, IP_ACTIVE_HIGH, IPT_UNUSED )

	PORT_START("IN-19")
	PORT_DIPNAME( 0x01, 0x00, "IN 19-0 (STRB 19 Data 0)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x01, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x00, "IN 19-1 (STRB 19 Data 1)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x02, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x00, "IN 19-2 (STRB 19 Data 2)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x04, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x00, "IN 19-3 (STRB 19 Data 3)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x08, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x00, "IN 19-4 (STRB 19 Data 4)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x10, DEF_STR( On ) )
	PORT_BIT(           0xffe0, IP_ACTIVE_HIGH, IPT_UNUSED )

	PORT_START("IN-20")
	PORT_DIPNAME( 0x01, 0x00, "IN 20-0 (STRB 20 Data 0)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x01, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x00, "IN 20-1 (STRB 20 Data 1)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x02, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x00, "IN 20-2 (STRB 20 Data 2)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x04, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x00, "IN 20-3 (STRB 20 Data 3)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x08, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x00, "IN 20-4 (STRB 20 Data 4)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x10, DEF_STR( On ) )
	PORT_BIT(           0xffe0, IP_ACTIVE_HIGH, IPT_UNUSED )

	PORT_START("IN-COIN")
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 ) // 1 PND
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 ) // 50p
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_COIN3 ) // 20p
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_COIN4 ) // 10p
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_COIN5 ) // ??
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN6 ) // 5p
INPUT_PORTS_END


INPUT_PORTS_START( sc4_base ) // just some fairly generic defaults we map to games where there isn't a specific mapping yet
	PORT_INCLUDE ( sc4_raw )
	PORT_MODIFY("IN-1")
	PORT_BIT( 0x0001, IP_ACTIVE_HIGH, SC45_BUTTON_MATRIX_1_0 ) PORT_NAME("IN1-0")
	PORT_BIT( 0x0002, IP_ACTIVE_HIGH, SC45_BUTTON_MATRIX_1_1 ) PORT_NAME("IN1-1")
	PORT_BIT( 0x0004, IP_ACTIVE_HIGH, SC45_BUTTON_MATRIX_1_2 ) PORT_NAME("IN1-2")
	PORT_BIT( 0x0008, IP_ACTIVE_HIGH, SC45_BUTTON_MATRIX_1_3 ) PORT_NAME("IN1-3")
	PORT_BIT( 0x0010, IP_ACTIVE_HIGH, SC45_BUTTON_MATRIX_1_4 ) PORT_NAME("IN1-4")
	PORT_BIT( 0x0020, IP_ACTIVE_HIGH, SC45_BUTTON_MATRIX_1_5 ) PORT_NAME("IN1-5")

	PORT_MODIFY("IN-2")
	PORT_BIT( 0x0001, IP_ACTIVE_HIGH, SC45_BUTTON_MATRIX_2_0 ) PORT_NAME("IN2-0")
	PORT_BIT( 0x0002, IP_ACTIVE_HIGH, SC45_BUTTON_MATRIX_2_1 ) PORT_NAME("IN2-1")
	PORT_BIT( 0x0004, IP_ACTIVE_HIGH, SC45_BUTTON_MATRIX_2_2 ) PORT_NAME("IN2-2")
	PORT_BIT( 0x0008, IP_ACTIVE_HIGH, SC45_BUTTON_MATRIX_2_3) PORT_NAME("IN2-3")
	PORT_BIT( 0x0010, IP_ACTIVE_HIGH, SC45_BUTTON_MATRIX_2_4) PORT_NAME("IN2-4")
	PORT_BIT( 0x0020, IP_ACTIVE_HIGH, SC45_BUTTON_MATRIX_2_5) PORT_NAME("IN2-5")

	PORT_MODIFY("IN-3")
	PORT_DIPNAME( 0x04, 0x00, "IN 3-2 (STK 4  3.2)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x04, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x00, "IN 3-3 (STK 2  3.3)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x08, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x00, "IN 3-4 (STK 3  3.4)" )
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x10, DEF_STR( On ) )

	PORT_MODIFY("IN-5")
	PORT_DIPNAME( 0x01, 0x00, "IN 5-0 (PRIZ4 5.0)" ) // (PRIZ4 5.0)  // Jackpot key
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x01, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x00, "IN 5-1 (PRIZ35.1)" ) // (PRI31 5.1)
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x02, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x00, "IN 5-2 (PRIZ2 5.2)" ) // (PRIZ2 5.2)
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x04, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x00, "IN 5-3 (PRIZ1 5.3)" ) // (PRIZ1 5.3)
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x08, DEF_STR( On ) )

	PORT_MODIFY("IN-6")
	PORT_DIPNAME( 0x01, 0x00, "IN 6-0 (PERC1 6.0)" ) // (PERC1 6.0)  // %age key
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x01, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x00, "IN 6-1 (PERC2 6.1)" ) // (PERC2 6.1)
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x02, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x00, "IN 6-2 (PERC3 6.2)" ) // (PERC3 6.2)
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x04, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x00, "IN 6-3 (PERC4 6.3)" ) // (PERC4 6.3)
	PORT_DIPSETTING(    0x00, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x08, DEF_STR( On ) )

	PORT_MODIFY("IN-7")
	PORT_BIT( 0x0001, IP_ACTIVE_HIGH, SC45_BUTTON_MATRIX_7_0) PORT_NAME("IN7-0")
	PORT_BIT( 0x0002, IP_ACTIVE_HIGH, SC45_BUTTON_MATRIX_7_1) PORT_NAME("IN7-1")
	PORT_BIT( 0x0004, IP_ACTIVE_HIGH, SC45_BUTTON_MATRIX_7_2) PORT_NAME("IN7-2")
	PORT_BIT( 0x0008, IP_ACTIVE_HIGH, SC45_BUTTON_MATRIX_7_3) PORT_NAME("IN7-3")
	PORT_BIT( 0x0010, IP_ACTIVE_HIGH, SC45_BUTTON_MATRIX_7_4) PORT_NAME("IN7-4")
	PORT_BIT( 0x0020, IP_ACTIVE_HIGH, SC45_BUTTON_MATRIX_7_5) PORT_NAME("IN7-5")

	PORT_MODIFY("IN-8")
	PORT_BIT( 0x0001, IP_ACTIVE_HIGH, SC45_BUTTON_MATRIX_8_0) PORT_NAME("IN8-0")
	PORT_BIT( 0x0002, IP_ACTIVE_HIGH, SC45_BUTTON_MATRIX_8_1) PORT_NAME("IN8-1")
	PORT_BIT( 0x0004, IP_ACTIVE_HIGH, SC45_BUTTON_MATRIX_8_2) PORT_NAME("IN8-2")
	PORT_BIT( 0x0008, IP_ACTIVE_HIGH, SC45_BUTTON_MATRIX_8_3) PORT_NAME("IN8-3")
	PORT_BIT( 0x0010, IP_ACTIVE_HIGH, SC45_BUTTON_MATRIX_8_4) PORT_NAME("IN8-4")
	PORT_BIT( 0x0020, IP_ACTIVE_HIGH, SC45_BUTTON_MATRIX_8_5) PORT_NAME("IN8-5")

	PORT_MODIFY("IN-9")
	PORT_BIT( 0x0001, IP_ACTIVE_HIGH, SC45_BUTTON_MATRIX_9_0) PORT_NAME("IN9-0")
	PORT_BIT( 0x0002, IP_ACTIVE_HIGH, SC45_BUTTON_MATRIX_9_1) PORT_NAME("IN9-1")
	PORT_BIT( 0x0004, IP_ACTIVE_HIGH, SC45_BUTTON_MATRIX_9_2) PORT_NAME("IN9-2")
	PORT_BIT( 0x0008, IP_ACTIVE_HIGH, SC45_BUTTON_MATRIX_9_3) PORT_NAME("IN9-3")
	PORT_BIT( 0x0010, IP_ACTIVE_HIGH, SC45_BUTTON_MATRIX_9_4) PORT_NAME("IN9-4")
	PORT_BIT( 0x0020, IP_ACTIVE_HIGH, SC45_BUTTON_MATRIX_9_5) PORT_NAME("IN9-5")

	PORT_MODIFY("IN-10")
	PORT_BIT( 0x0001, IP_ACTIVE_HIGH, SC45_BUTTON_MATRIX_10_0) PORT_NAME("IN10-0")
	PORT_BIT( 0x0002, IP_ACTIVE_HIGH, SC45_BUTTON_MATRIX_10_1) PORT_NAME("IN10-1")
	PORT_BIT( 0x0004, IP_ACTIVE_HIGH, SC45_BUTTON_MATRIX_10_2) PORT_NAME("IN10-2")
	PORT_BIT( 0x0008, IP_ACTIVE_HIGH, SC45_BUTTON_MATRIX_10_3) PORT_NAME("IN10-3")
	PORT_BIT( 0x0010, IP_ACTIVE_HIGH, SC45_BUTTON_MATRIX_10_4) PORT_NAME("IN10-4")
	PORT_BIT( 0x0020, IP_ACTIVE_HIGH, SC45_BUTTON_MATRIX_10_5) PORT_NAME("IN10-5")

	PORT_MODIFY("IN-20")
	PORT_BIT(0x0001, IP_ACTIVE_HIGH, SC45_BUTTON_MATRIX_20_0) PORT_NAME("Green Button")
	PORT_DIPNAME( 0x0002, 0x0000, "Door Lock" ) // DORLOK20.1
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0002, DEF_STR( On ) )
	PORT_DIPNAME( 0x0004, 0x0000, "Service Door" ) // SERDOR20.2
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0004, DEF_STR( On ) )
	PORT_DIPNAME( 0x0008, 0x0000, "Cashbox Door" ) // CSHDOR20.3
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0008, DEF_STR( On ) )
	PORT_DIPNAME( 0x0010, 0x0000, "Hopper DMP" ) // HOPDMP20.4
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0010, DEF_STR( On ) )
INPUT_PORTS_END

#define MACHINE_FLAGS MACHINE_NOT_WORKING
#define MACHINE_FLAGS_NOSOUND MACHINE_FLAGS|MACHINE_NO_SOUND

bool compare_mbus(uint16_t* rom)
{
	for (int i=0;i<11;i++)
	{
		if (rom[0] != 0xb109)
			return false;

		if (rom[3] != 0x6612)
			return false;

		if (rom[7] != 0x0c81)
			return false;

		if (rom[10] != 0x66ea)
			return false;
	}

	return true;
}

void sc4_state::find_mbus(uint16_t* rom)
{
	for (int i=0;i<(0x100000-0x40)/2;i++)
	{
		bool found = compare_mbus(&rom[i]);

		if (found==true)
		{
			printf("x found at %08x\n", i*2);
			m_chk41addr = i*2;
		}
	}
}


/* default inputs */
INPUT_PORTS_START( sc4 )
	PORT_INCLUDE ( sc4_base )
	PORT_MODIFY("IN-5")
	PORT_DIPNAME( 0x0f, 0x00, "Jackpot Key" )
	SC4_JACKPOT_KEY_SETTINGS
INPUT_PORTS_END



void sc4_state::init_sc4()
{
	uint8_t *src = memregion( "maincpu" )->base();
	// SC4 identification sequence 0x80 0x00 0xf0 0x7d
	if (((src[0] == 0x80) && (src[2] == 0xf0)) || ((src[1] == 0x00) && (src[3] == 0x7d)))
	{
		printf("Confirmed SC4 ROM\n");
	}
	else
	{
		printf("NOT AN SC4 ROM!!!!!\n");
	}

	// there is usually a string in the rom with identification info, often also saying which sound rom should be used!
	// find it.

	int found = find_project_string(machine(), 1, 0);
	if (!found)
		printf("Normal rom pair string not found, checking mismatched / missing rom string\n");

	// help identify roms where one of the pair is missing too
	if (!found)
	{
			found = find_project_string(machine(), 1, 1);
			found = find_project_string(machine(), 1, 2);
	}

	if (!found)
		printf("No suitable string found\n");


	// debug helpers to find strings used for inputs and where the buttons map
//  bfm_sc45_layout_helper(machine());

}

void sc4_state::init_sc4mbus()
{
	init_sc4();
	uint16_t *rom = (uint16_t *)memregion("maincpu")->base();
	find_mbus(rom);
}


ROM_START( sc4tst )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "ftu_sc4_testrig.hi", 0x00000, 0x080000, CRC(dacba04e) SHA1(fb96c790600f55769ddff4868d10b431d07777a4) )
	ROM_LOAD16_BYTE( "ftu_sc4_testrig.lo", 0x00001, 0x080000, CRC(fd313f2e) SHA1(8453842fc10a69758b991541542b8c55c1f3b02f) )
	ROM_REGION( 0x400000, "ymz", ROMREGION_ERASE00 )
ROM_END

ROM_START( sc4hrolr ) // uses RTC on romcard
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "highroller13_hi.bin", 0x00000, 0x080000, CRC(b4138351) SHA1(330b45a4eb4c5bb5432508cc2c6806901ae59a95) )
	ROM_LOAD16_BYTE( "highroller13_lo.bin", 0x00001, 0x080000, CRC(b479cd74) SHA1(c41b156ef2fd46e8658d036ee4d8c4056d0061d2) )

	ROM_REGION( 0x100000, "ymz", 0 )
	ROM_LOAD( "b3a_highroller.bin", 0x0000, 0x100000, CRC(38ce5435) SHA1(e159420e7929fa048f3b2393f761eeed2e1cf3b7) )

	ROM_REGION( 0x100000, "gals", 0 )
	ROM_LOAD( "75585129.ic1.bin", 0x0000, 0x000117, CRC(2454bb33) SHA1(610cde14caef3f2d02f0076b924e015077c3832b) ) /* protected gal16v8 on romcard */
ROM_END

ROM_START( ad4skill )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "skilldicegame_hi.bin", 0x00000, 0x080000, CRC(83d10f1b) SHA1(383a337e924313b1aa1e62451227218a82cd6cd3) )
	ROM_LOAD16_BYTE( "skilldicegame_lo.bin", 0x00001, 0x080000, CRC(e8d35560) SHA1(30efb6dabfc98cf89766bb33e249fed661219798) )

	ROM_REGION( 0x400000, "adder4", 0 )
	ROM_LOAD16_BYTE( "skilldiceadder_hi.bin", 0x00000, 0x080000, CRC(14da612d) SHA1(ad33ef3736538374b9f81868f0041d6b736e2375) )
	ROM_LOAD16_BYTE( "skilldiceadder_lo.bin", 0x00001, 0x080000, CRC(93a2aaf5) SHA1(c6b7c2ac56a9980e01e059f54048472f3e6d8b5b) )

	ROM_REGION( 0x100000, "ymz", 0 )
	ROM_LOAD( "skilldicesound.bin", 0x0000, 0x080000, CRC(aeccd210) SHA1(ee88f67b436092854cc6719cc91db91d905ffc6f) )
ROM_END




ROM_START( sc4bbust )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406101.evn", 0x00000, 0x080000, CRC(b6accc1e) SHA1(dec2c3710967605f12a5e07d0fd43f11fc6e1f03) )
	ROM_LOAD16_BYTE( "95406100.odd", 0x00001, 0x080000, CRC(b3b8ed69) SHA1(6c96e30add751b039b305983ef15cd2c68dfe430) )
	sc_bbust_others
ROM_END


ROM_START( sc4blokq )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404422.odd", 0x00001, 0x080000, CRC(cb1973eb) SHA1(9604bb13aa06491293c3735be93697b4a5dd5bd9) )
	ROM_LOAD16_BYTE( "95404423.evn", 0x00000, 0x080000, CRC(8bf528bb) SHA1(794b346fbd78020cd6a6b82748a298eca7c7536d) )
	sc_bbust_others
ROM_END

ROM_START( sc4blokqa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405422.odd", 0x00001, 0x080000, CRC(e6ff1728) SHA1(3b67d1754da55b5ca57724632991a014d959edd5) )
	ROM_LOAD16_BYTE( "95405423.evn", 0x00000, 0x080000, CRC(8988705b) SHA1(d9fa53889e5b8e68bdd4070dd3db9a2e090ec4d4) )
	sc_bbust_others
ROM_END

ROM_START( sc4blokqb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405424.odd", 0x00001, 0x080000, CRC(dcf048ce) SHA1(f57c9e74fa0c948fc5be17cac7bd601bc09fa4d3) )
	ROM_LOAD16_BYTE( "95405425.evn", 0x00000, 0x080000, CRC(5e1a3264) SHA1(9c9d26d4d7b1686e0e44309b182bf71a171a7176) )
	sc_bbust_others
ROM_END

ROM_START( sc4blokqc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406240.odd", 0x00001, 0x080000, CRC(e579d5d3) SHA1(ccf94d314f604eee53919c6333deff55e908d260) )
	ROM_LOAD16_BYTE( "95406241.evn", 0x00000, 0x080000, CRC(565b4041) SHA1(cc02b2b0d0bf20193821d915cfc49e6701f49d86) )
	sc_bbust_others
ROM_END

ROM_START( sc4blokqd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407100.odd", 0x00001, 0x080000, CRC(37695cfb) SHA1(171a4144af8441a29495ace00cd0371a9b2a3b52) )
	ROM_LOAD16_BYTE( "95407101.evn", 0x00000, 0x080000, CRC(bb08fd97) SHA1(f52d45e092ce6dce2444f96cc98c4b8e2a3a5354) )
	sc_bbust_others
ROM_END

ROM_START( sc4blokqe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407240.odd", 0x00001, 0x080000, CRC(d32b9e02) SHA1(b2401d39c5fa5ffabb05e26987b3f9573c0ed076) )
	ROM_LOAD16_BYTE( "95407241.evn", 0x00000, 0x080000, CRC(c5bff1e6) SHA1(890c5db1a7093ae91ecb0eef033703caadc42859) )
	sc_bbust_others
ROM_END


ROM_START( sc4cvani )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404753.evn", 0x00000, 0x080000, CRC(e8e6d801) SHA1(33e9d1f829e8deba170e81379da0ccf5b2a677d7) )
	ROM_LOAD16_BYTE( "95404752.odd", 0x00001, 0x080000, CRC(8ad0ea8d) SHA1(90119c17fe828bbb52844e32b9be489ef98f6ad8) )
	sc_cvani_others
ROM_END

ROM_START( sc4cvania )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405878.odd", 0x00001, 0x080000, CRC(8788c032) SHA1(7f5f4f4f72339d719a6413e1e47082e3a9bee5ad) )
	ROM_LOAD16_BYTE( "95405879.evn", 0x00000, 0x080000, CRC(803cecd7) SHA1(1670525c4bef19d3b5253d0d14fe60f7736e6150) )
	sc_cvani_others
ROM_END

ROM_START( sc4cvanib )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406118.odd", 0x00001, 0x080000, CRC(5eba8a9b) SHA1(9b498549e08ab6de1dacc3b602d34d0d2b2e97d0) )
	ROM_LOAD16_BYTE( "95406119.evn", 0x00000, 0x080000, CRC(a8be8679) SHA1(6170c737bd2440d0c6afd98027c3f5c409c90eb2) )
	sc_cvani_others
ROM_END

ROM_START( sc4cvanic )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407118.odd", 0x00001, 0x080000, CRC(cad80a25) SHA1(b0f7fa45b6fdeee050337b67f3b59797f0d8f400) )
	ROM_LOAD16_BYTE( "95407119.evn", 0x00000, 0x080000, CRC(567bbb96) SHA1(afb3f308141c3d5d8e9d214b03fa4143a83b756a) )
	sc_cvani_others
ROM_END

ROM_START( sc4cvanid )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408744.odd", 0x00001, 0x080000, CRC(327b8358) SHA1(79bafefdca55747e2b2bfd7e1875659fc8dfcc61) )
	ROM_LOAD16_BYTE( "95408745.evn", 0x00000, 0x080000, CRC(c0197b57) SHA1(a08333429e0cae556e2d9cfdb125dd1fed3652c9) )
	sc_cvani_others
ROM_END

ROM_START( sc4cvanie )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409326.odd", 0x00001, 0x080000, CRC(a6357aa3) SHA1(ab9b0febe49afa7e851cbb3cb1ce77c7bfb28611) )
	ROM_LOAD16_BYTE( "95409327.evn", 0x00000, 0x080000, CRC(e0bba9ba) SHA1(c4946788db0844e40bab240f423aeb32713f0f57) )
	sc_cvani_others
ROM_END

ROM_START( sc4cvanif )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414308.odd", 0x00001, 0x080000, CRC(aac38393) SHA1(3b4e8de63acb9b1134ed162baa6ab282d942e91e) )
	ROM_LOAD16_BYTE( "95414309.evn", 0x00000, 0x080000, CRC(bbedb6bb) SHA1(4c18ad6e54ac8e2ee5ed1c998ffe01c67ff834ef) )
	sc_cvani_others
ROM_END

ROM_START( sc4cvanig )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414310.odd", 0x00001, 0x080000, CRC(a1941e8d) SHA1(0712e303179b590775dfee26ffb5124bdc60e5d1) )
	ROM_LOAD16_BYTE( "95414311.evn", 0x00000, 0x080000, CRC(b155ef1f) SHA1(c6fcee7cd723e1fd4e9f6bec5cdd0f1cfa1c42bd) )
	sc_cvani_others
ROM_END

ROM_START( sc4cvanih )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415308.odd", 0x00001, 0x080000, CRC(8d243b05) SHA1(440ff23b01112d55793b15b91803b09c10e93b4a) )
	ROM_LOAD16_BYTE( "95415309.evn", 0x00000, 0x080000, CRC(99732cdc) SHA1(6bb23eae97124b411ad3e44b65118ca1cdf30d64) )
	sc_cvani_others
ROM_END

ROM_START( sc4cvanii )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415310.odd", 0x00001, 0x080000, CRC(8673a61b) SHA1(5ae94eec0d34ba92418201d4c097328c46eabf6b) )
	ROM_LOAD16_BYTE( "95415311.evn", 0x00000, 0x080000, CRC(93cb7578) SHA1(ddac328917ac628e0ac2732ebc01f2936019b71e) )
	sc_cvani_others
ROM_END


ROM_START( sc4cvclb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408293.evn", 0x00000, 0x080000, CRC(52b2a132) SHA1(120c86d9822081afee9f4a16d871522ab0d52e7a) )
	ROM_LOAD16_BYTE( "95408292.odd", 0x00001, 0x080000, CRC(2b1e111b) SHA1(a9730617512ff76de4dfaa487e0156b2470ace88) )
	sc_cvclb_others
ROM_END

ROM_START( sc4cvclba )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408394.odd", 0x00001, 0x080000, CRC(e1de3d0b) SHA1(9bd3382d6538ca455ae747902cd1a7fe704ca7f4) )
	ROM_LOAD16_BYTE( "95408395.evn", 0x00000, 0x080000, CRC(73727dab) SHA1(1b6b64b8430c0ab8cb6a6b2cb8d8282d4e32ba21) )
	sc_cvclb_others
ROM_END

ROM_START( sc4cvclbb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408430.odd", 0x00001, 0x080000, CRC(f6d20ce7) SHA1(0071bbb6175150b35a185e867800009403298fb8) )
	ROM_LOAD16_BYTE( "95408431.evn", 0x00000, 0x080000, CRC(4e92b4e3) SHA1(d008e4f2a0b6d48376a689b92cbda08a0b015234) )
	sc_cvclb_others
ROM_END

ROM_START( sc4cvclbc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409292.odd", 0x00001, 0x080000, CRC(e2d7476d) SHA1(ea54697110b35a286c1e60fc07ebf6fdf24469d0) )
	ROM_LOAD16_BYTE( "95409293.evn", 0x00000, 0x080000, CRC(26a61688) SHA1(c7128768d031fb03ab31a0b09ccba32913cc5d4e) )
	sc_cvclb_others
ROM_END

ROM_START( sc4cvclbd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409394.odd", 0x00001, 0x080000, CRC(115ec430) SHA1(a7e1a5866e6e50eb1dfc9f990ca9f0ff3be8e77b) )
	ROM_LOAD16_BYTE( "95409395.evn", 0x00000, 0x080000, CRC(8d28167c) SHA1(82e88e2544cb940eebcd237389debcbeaa8a0a38) )
	sc_cvclb_others
ROM_END

ROM_START( sc4cvclbe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409430.odd", 0x00001, 0x080000, CRC(2e43b6fe) SHA1(6916ced089911c53a6ead3025c823675ae47835e) )
	ROM_LOAD16_BYTE( "95409431.evn", 0x00000, 0x080000, CRC(d30e9665) SHA1(d1087018f5dd401a38700669ee08eb247758eecc) )
	sc_cvclb_others
ROM_END

ROM_START( sc4cvclbf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410920.odd", 0x00001, 0x080000, CRC(85edbcc3) SHA1(8789484deceef2d53df3bbed3e0e48b20a46aa2a) )
	ROM_LOAD16_BYTE( "95410921.evn", 0x00000, 0x080000, CRC(28090fef) SHA1(dca69f2ef3e09cadad840dfded89d4c228d4df74) )
	sc_cvclb_others
ROM_END

ROM_START( sc4cvclbg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411920.odd", 0x00001, 0x080000, CRC(815e76f9) SHA1(e3be90da856a86873916e1524dc81143cf546862) )
	ROM_LOAD16_BYTE( "95411921.evn", 0x00000, 0x080000, CRC(a247d382) SHA1(4bab91b42d4c1f0e31a1176656af20ed5b26c1d0) )
	sc_cvclb_others
ROM_END


ROM_START( sc4pstat )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406830.odd", 0x00001, 0x080000, CRC(92800608) SHA1(8b584ae5378bf28797ef0bf8f100f0c2fc52d00e) )
	ROM_LOAD16_BYTE( "95406831.evn", 0x00000, 0x080000, CRC(c8470bf4) SHA1(a3cc8a6ba55c4b017452fbe55ff40b00e5cbf541) )
	sc_pstat_others
ROM_END

ROM_START( sc4pstata )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406938.odd", 0x00001, 0x080000, CRC(d06d63dd) SHA1(8dd55290930dc4d4532a75c86848c56dfe507022) )
	ROM_LOAD16_BYTE( "95406939.evn", 0x00000, 0x080000, CRC(ee8dba94) SHA1(8125751a3dc29e8ea5d396c2b055d25b6e0d1ebc) )
	sc_pstat_others
ROM_END

ROM_START( sc4pstatb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407830.odd", 0x00001, 0x080000, CRC(c1bba3a6) SHA1(67e5f41e15c9c12048dd938ccd6671712e53b252) )
	ROM_LOAD16_BYTE( "95407831.evn", 0x00000, 0x080000, CRC(bcccea76) SHA1(d615e64d0d9051f944eac67273442dc677ab6bb4) )
	sc_pstat_others
ROM_END

ROM_START( sc4pstatc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407938.odd", 0x00001, 0x080000, CRC(4d86ab35) SHA1(2d85731871405ec6679438a267c10879e0d8018b) )
	ROM_LOAD16_BYTE( "95407939.evn", 0x00000, 0x080000, CRC(b49d0ab0) SHA1(0950d7124b8ec18cea397cbb68a7fde8ddf2733c) )
	sc_pstat_others
ROM_END

ROM_START( sc4pstatd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409330.odd", 0x00001, 0x080000, CRC(63ca83de) SHA1(efe8fcccb425ad01b202ec1dc6f9e6e81c90388b) )
	ROM_LOAD16_BYTE( "95409331.evn", 0x00000, 0x080000, CRC(ef9b9518) SHA1(356cc2ab7f8b06708eb4244ca7d7ad83ad5174a0) )
	sc_pstat_others
ROM_END

ROM_START( sc4pstate )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409536.odd", 0x00001, 0x080000, CRC(48891ea9) SHA1(b49b1adfe189aeb113bb59c1840429d37fca1b12) )
	ROM_LOAD16_BYTE( "95409537.evn", 0x00000, 0x080000, CRC(50ed3f7f) SHA1(8d071db1104602eebe9e48a118d4fa0a4532e7cc) )
	sc_pstat_others
ROM_END

ROM_START( sc4pstatf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410772.odd", 0x00001, 0x080000, CRC(fe898786) SHA1(f04cc8aec44b6b27d4fe3100b498389b85227f61) )
	ROM_LOAD16_BYTE( "95410773.evn", 0x00000, 0x080000, CRC(31ac2db0) SHA1(0e9c25b1a70c2941543b4b4fee94bb87d4f2d66b) )
	sc_pstat_others
ROM_END

ROM_START( sc4pstatg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410774.odd", 0x00001, 0x080000, CRC(f0ff037c) SHA1(efc7c5631a733d84352778f87958a507ddaf8014) )
	ROM_LOAD16_BYTE( "95410775.evn", 0x00000, 0x080000, CRC(acd9d42a) SHA1(801ed0499251e513f4f0c72c9de3da9493ba7761) )
	sc_pstat_others
ROM_END

ROM_START( sc4pstath )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410922.odd", 0x00001, 0x080000, CRC(cd7bbf74) SHA1(063893ebb67bec38372cb29c4c3f4e16cfc2720d) )
	ROM_LOAD16_BYTE( "95410923.evn", 0x00000, 0x080000, CRC(5b487dcc) SHA1(b4c4d84ed7dc5ab360e8d1bc9d0da5d4ecf4719f) )
	sc_pstat_others
ROM_END

ROM_START( sc4pstati )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411772.odd", 0x00001, 0x080000, CRC(4e2afc53) SHA1(76138abff8a8a481c0bd0281d79c90967e8158a2) )
	ROM_LOAD16_BYTE( "95411773.evn", 0x00000, 0x080000, CRC(e16d17c1) SHA1(5bbfc5c973a4a2dfdff3c57ab207641ec2d05e1c) )
	sc_pstat_others
ROM_END

ROM_START( sc4pstatj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411774.odd", 0x00001, 0x080000, CRC(19be804e) SHA1(9c407133b0766065ea883532cd213479d6e7aa99) )
	ROM_LOAD16_BYTE( "95411775.evn", 0x00000, 0x080000, CRC(7db91413) SHA1(6d78c8b24a01dd132be23f2866366af12d125e3f) )
	sc_pstat_others
ROM_END


ROM_START( sc4pstatm )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411922.odd", 0x00001, 0x080000, CRC(a3e204ad) SHA1(ff755e30406328e1a1d28403814e289743777f5e) )
	ROM_LOAD16_BYTE( "95411923.evn", 0x00000, 0x080000, CRC(413eb2ae) SHA1(81ec1af213fed3d7d66e2da646ef79b091586eaf) )
	sc_pstat_others
ROM_END

ROM_START( sc4pstatn )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414304.odd", 0x00001, 0x080000, CRC(cc00f2af) SHA1(a76122a25d7d07e81baa99e797ee8b927c00b8ef) )
	ROM_LOAD16_BYTE( "95414305.evn", 0x00000, 0x080000, CRC(341ce188) SHA1(fd0b350c6855187642416e492cab329fa138bb3a) )
	sc_pstat_others
ROM_END

ROM_START( sc4pstato )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414306.odd", 0x00001, 0x080000, CRC(87ed4d56) SHA1(14c2fcbed5765f210315b8234db05c92b7f5ba94) )
	ROM_LOAD16_BYTE( "95414307.evn", 0x00000, 0x080000, CRC(bf6bf908) SHA1(e614f8581a97ce0f44627635c07b4b1c5426d498) )
	sc_pstat_others
ROM_END

ROM_START( sc4pstatp )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415304.odd", 0x00001, 0x080000, CRC(12a9dac0) SHA1(91b27b6ac23788b22da0f3682d21380f1841f27d) )
	ROM_LOAD16_BYTE( "95415305.evn", 0x00000, 0x080000, CRC(c6dc17a6) SHA1(715c4f501b87deab5f271c5ac36efb88ef0ae858) )
	sc_pstat_others
ROM_END

ROM_START( sc4pstatq )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415306.odd", 0x00001, 0x080000, CRC(59446539) SHA1(0583272c1832de1265924a83781c83b711a4318c) )
	ROM_LOAD16_BYTE( "95415307.evn", 0x00000, 0x080000, CRC(4dab0f26) SHA1(2e29e2ff48c29628259271e4c63d199d9280a2fe) )
	sc_pstat_others
ROM_END


ROM_START( sc4spark )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402142.hi", 0x00000, 0x080000, CRC(1ffc404b) SHA1(44e0f02708e6b0b15d3e272e6fa398d573242fba) )
	ROM_LOAD16_BYTE( "95402141.lo", 0x00001, 0x080000, CRC(57db7e17) SHA1(bd9a12ac901b379459dbd13dcde9b6441dfafdc3) )
	sc_spark_others
ROM_END

ROM_START( sc4sparka )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402143.lo", 0x00001, 0x080000, CRC(d06307dc) SHA1(021fb928366aaefc0009ad9a0482294d15fe66a6) )
	ROM_LOAD16_BYTE( "95402144.hi", 0x00000, 0x080000, CRC(0be0d27f) SHA1(d97156c6b386712afb929321d537b46f81cda81e) )
	sc_spark_others
ROM_END

ROM_START( sc4sparkb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403141.lo", 0x00001, 0x080000, CRC(6cb6b647) SHA1(83c862ded1424a120b19323a79fda748a7d49ba3) )
	ROM_LOAD16_BYTE( "95403142.hi", 0x00000, 0x080000, CRC(8e65a90c) SHA1(b801b9d06333a91a2d0b7b86cc6ffc2443833fe0) )
	sc_spark_others
ROM_END

ROM_START( sc4sparkc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403143.lo", 0x00001, 0x080000, CRC(78752813) SHA1(3e8327af899def1e1ea9f7b1d6b44252bd8af9f5) )
	ROM_LOAD16_BYTE( "95403144.hi", 0x00000, 0x080000, CRC(c0f6dd24) SHA1(3231f66bf6a78d86b1e9653add17ff73e620db1c) )
	sc_spark_others
ROM_END

ROM_START( sc4sparkd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404090.lo", 0x00001, 0x080000, CRC(0d2be53b) SHA1(67d1ce2f676e0531e775d5fc54e385c083a9a9a1) )
	ROM_LOAD16_BYTE( "95404091.hi", 0x00000, 0x080000, CRC(0e026041) SHA1(0eca329deeea24ebefcfb46393b8515a5bea4c50) )
	sc_spark_others
ROM_END

ROM_START( sc4sparke )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404092.lo", 0x00001, 0x080000, CRC(c00bc000) SHA1(0fe12fccc219cedd4cee2ef974d2b56099862b15) )
	ROM_LOAD16_BYTE( "95404093.hi", 0x00000, 0x080000, CRC(1d60ce98) SHA1(3011b0f59145d80d8d34207d31abf98331285eff) )
	sc_spark_others
ROM_END



ROM_START( sc4brkfs )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402330.hi", 0x00000, 0x080000, CRC(5fe5753b) SHA1(df165a79dbfdaa85d6d5664c4b64e68581b6b46e) )
	ROM_LOAD16_BYTE( "95402329.lo", 0x00001, 0x080000, CRC(0717309e) SHA1(55a1f8554cfb84bd33f88ea24c781ff6e7323daa) )
	sc_brkfs_others
ROM_END

ROM_START( sc4brkfsa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402331.lo", 0x00001, 0x080000, CRC(a1e71c0a) SHA1(19d5d10924869e5c95a8d72d9aaa9ab09053d4e0) )
	ROM_LOAD16_BYTE( "95402332.hi", 0x00000, 0x080000, CRC(11915577) SHA1(c2202142bff28d274f5a10dd160ba0159547b6b9) )
	sc_brkfs_others
ROM_END

ROM_START( sc4brkfsb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403329.lo", 0x00001, 0x080000, CRC(f80bea0a) SHA1(38b6cdda2968923e1ad82766b89688e5a80a1fad) )
	ROM_LOAD16_BYTE( "95403330.hi", 0x00000, 0x080000, CRC(5e448f73) SHA1(ce9d3a38e681c3e5fde2928426b07d544bf2e544) )
	sc_brkfs_others
ROM_END

ROM_START( sc4brkfsc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403331.lo", 0x00001, 0x080000, CRC(77f05126) SHA1(bf3240a0a7e29d0bf11a241f1068fa1ec4143432) )
	ROM_LOAD16_BYTE( "95403332.hi", 0x00000, 0x080000, CRC(8008bc30) SHA1(501180f871f44fb54fa1c7343a71ff634334f97d) )
	sc_brkfs_others
ROM_END



ROM_START( sc4gslam )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402975.hi", 0x00000, 0x080000, CRC(ac255d4b) SHA1(2bb6017c915a62faa5cb0bca7b67a7c16540861a) )
	ROM_LOAD16_BYTE( "95402974.lo", 0x00001, 0x080000, CRC(6a170906) SHA1(0fc5c87db38abe350c655e5b9595da792ddeaa34) )
	sc_gslam_others
ROM_END

ROM_START( sc4gslama )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402976.lo", 0x00001, 0x080000, CRC(6859579d) SHA1(c9e13dd332d5b92cc1a94962d4d22954fa532b76) )
	ROM_LOAD16_BYTE( "95402977.hi", 0x00000, 0x080000, CRC(5b16cd86) SHA1(5283a97e2ae2c563c82d2b7593e522ab532a3d76) )
	sc_gslam_others
ROM_END

ROM_START( sc4gslamb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403974.lo", 0x00001, 0x080000, CRC(118bd82c) SHA1(68d936f3146beb80239603de8551d8adb8732302) )
	ROM_LOAD16_BYTE( "95403975.hi", 0x00000, 0x080000, CRC(9a027e9e) SHA1(64c650e7a9504785c10af7b7702198235976a690) )
	sc_gslam_others
ROM_END

ROM_START( sc4gslamc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403976.lo", 0x00001, 0x080000, CRC(e645e490) SHA1(7429feb090bc0e17918d804583073d27e7d1cf63) )
	ROM_LOAD16_BYTE( "95403977.hi", 0x00000, 0x080000, CRC(e37850b9) SHA1(79162fdc8db1b3f7c6b0124a6dcc1b97bad33172) )
	sc_gslam_others
ROM_END

ROM_START( sc4gslamd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404078.lo", 0x00001, 0x080000, CRC(e32b12b5) SHA1(6459ae08365c806d9cee083036a8b55f9017d6e2) )
	ROM_LOAD16_BYTE( "95404079.hi", 0x00000, 0x080000, CRC(7d97128c) SHA1(0b7273d460d8aeaf1b8c2507613c994e71b5acd2) )
	sc_gslam_others
ROM_END

ROM_START( sc4gslame )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405078.lo", 0x00001, 0x080000, CRC(24156751) SHA1(de18a563a5e74318af5e91d70ddfc037a99bf3e4) )
	ROM_LOAD16_BYTE( "95405079.hi", 0x00000, 0x080000, CRC(a21017ac) SHA1(b8fab6e558cb0f0390c8f0a60e30eb7da32fcf1e) )
	sc_gslam_others
ROM_END

ROM_START( sc4gslamf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "slam1000.hi", 0x00000, 0x080000, CRC(3fd1a9bd) SHA1(ec5cbed2427bd58f93da1b380f2c44179e460c66) )
	ROM_LOAD16_BYTE( "slam1000.lo", 0x00001, 0x080000, CRC(6499044b) SHA1(50bf83d6a2cabec3c8358401432f9611eaf88dfa) )
	sc_gslam_others
ROM_END




ROM_START( sc4ivply )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406374.lo", 0x00001, 0x080000, CRC(33511be6) SHA1(17d1f87620e1f592f2660815c96dbbbe7a5d1dfe) )
	ROM_LOAD16_BYTE( "95406375.hi", 0x00000, 0x080000, CRC(fb0e1f4f) SHA1(9cf091fb82ec7696ec7c1d3434b2e1a8cf753782) )
	sc_ivply_others
ROM_END




ROM_START( sc4abra )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414374.lo", 0x00001, 0x080000, CRC(f8b83090) SHA1(82aa06f76ea0168b03e6b31132db2e14f279755f) )
	ROM_LOAD16_BYTE( "95414375.hi", 0x00000, 0x080000, CRC(25bd9371) SHA1(ad3900d5a6e032c5094fc651b9ec1767231fb245) )
	sc_abra_others
ROM_END

ROM_START( sc4abraa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415374.lo", 0x00001, 0x080000, CRC(28a54f37) SHA1(64c185f97e9147418bd35538f1607aaebe04ffbb) )
	ROM_LOAD16_BYTE( "95415375.hi", 0x00000, 0x080000, CRC(d59f6b87) SHA1(0dbef7ef213dd557cea91b0ac6c8e71ca45ac23c) )
	sc_abra_others
ROM_END

ROM_START( sc4abrab )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420438.lo", 0x00001, 0x080000, CRC(9ac61925) SHA1(7f3f812a6a3b20fcf5624a631555b564307f17ce) )
	ROM_LOAD16_BYTE( "95420439.hi", 0x00000, 0x080000, CRC(33f4f2c0) SHA1(bc434d1db481f0210810aa2cec7062a502586ecf) )
	sc_abra_others
ROM_END

ROM_START( sc4abrac )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421438.lo", 0x00001, 0x080000, CRC(af623d23) SHA1(ff467317338a5901bfd4a6b2d29da84603e24252) )
	ROM_LOAD16_BYTE( "95421439.hi", 0x00000, 0x080000, CRC(fed59f04) SHA1(3823dab966dad2aa3b1fcd486cb98c291045359a) )
	sc_abra_others
ROM_END

ROM_START( sc4abrad )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95424038.lo", 0x00001, 0x080000, CRC(030b0c4e) SHA1(aa90f47cd55468542ceb9b8713ab7eea4fba0f7e) )
	ROM_LOAD16_BYTE( "95424039.hi", 0x00000, 0x080000, CRC(765156ff) SHA1(856d51584d406f29b033faff8fea9e16adcbc460) )
	sc_abra_others
ROM_END

ROM_START( sc4abrae )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95425038.lo", 0x00001, 0x080000, CRC(eff52c5b) SHA1(606b3a111cf44b119537e347b6baca3fa50f5241) )
	ROM_LOAD16_BYTE( "95425039.hi", 0x00000, 0x080000, CRC(d135072e) SHA1(488f35beb8d1f579520851a2afff783a4dae8a43) )
	sc_abra_others
ROM_END




ROM_START( sc4acesh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406960.lo", 0x00001, 0x080000, CRC(e3c4712d) SHA1(50e17f4a1d1c5a140d6f591fca66b8d5191e1d3c) )
	ROM_LOAD16_BYTE( "95406961.hi", 0x00000, 0x080000, CRC(44b5e0d2) SHA1(16837ecb5dbb96b964cf0ec1d1890a0d914ec675) )
	sc_acesh_others
ROM_END

ROM_START( sc4acesha )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407960.lo", 0x00001, 0x080000, CRC(d4334b67) SHA1(7ecdee5f7a19da5745d8c1bd3ec68ceb51cfe554) )
	ROM_LOAD16_BYTE( "95407961.hi", 0x00000, 0x080000, CRC(3ca49ca9) SHA1(ccf655d4057a405f04597ae450944e828dda626b) )
	sc_acesh_others
ROM_END

ROM_START( sc4aceshb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410470.lo", 0x00001, 0x080000, CRC(9c614444) SHA1(7e691b192503e583052c1e7e1c46c18a60aa05e9) )
	ROM_LOAD16_BYTE( "95410471.hi", 0x00000, 0x080000, CRC(9deb916e) SHA1(ce9cbccef8f26233de7bebf96415b67fe632ee74) )
	sc_acesh_others
ROM_END

ROM_START( sc4aceshc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410476.lo", 0x00001, 0x080000, CRC(511f86b5) SHA1(3d6fe59bd19fb5b3a581781867b3b81dd0f77762) )
	ROM_LOAD16_BYTE( "95410477.hi", 0x00000, 0x080000, CRC(436bd1a7) SHA1(1f88e9f3b99601004c419530537efb063997a156) )
	sc_acesh_others
ROM_END




ROM_START( sc4adren )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412318.lo", 0x00001, 0x080000, CRC(f0b06b9c) SHA1(024cfe8cd6a4b00287f14682354a73fa685b39e7) )
	ROM_LOAD16_BYTE( "95412319.hi", 0x00000, 0x080000, CRC(f6e0b128) SHA1(75b5b001b6da1c19bf895582d2db7adaee69fc1e) )
	sc_adren_others
ROM_END

ROM_START( sc4adrena )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412320.lo", 0x00001, 0x080000, CRC(5df555f0) SHA1(1130b73dd815754eb4ad5efca316fd78608f85ec) )
	ROM_LOAD16_BYTE( "95412321.hi", 0x00000, 0x080000, CRC(42f1069a) SHA1(e3ee74d1d9b8f2dc41c405b89495ced7da36f930) )
	sc_adren_others
ROM_END

ROM_START( sc4adrenb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413318.lo", 0x00001, 0x080000, CRC(d7f16cd4) SHA1(cba61a0e63e160ba07f67465db7fcdb9d054ce25) )
	ROM_LOAD16_BYTE( "95413319.hi", 0x00000, 0x080000, CRC(338e2d52) SHA1(05f4416642a76246725a3d2ba21222922314aadd) )
	sc_adren_others
ROM_END

ROM_START( sc4adrenc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413320.lo", 0x00001, 0x080000, CRC(2965ba19) SHA1(9fbcb639d81a0b2c8d651500a9a100630519f853) )
	ROM_LOAD16_BYTE( "95413321.hi", 0x00000, 0x080000, CRC(6212c41d) SHA1(fa961dedaa25519ac04364e189e800455567dc13) )
	sc_adren_others
ROM_END




ROM_START( sc4alad )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402133.lo", 0x00001, 0x080000, CRC(cff17504) SHA1(5db95fe3c40b47c357dca1587f8d2973568d1259) )
	ROM_LOAD16_BYTE( "95402134.hi", 0x00000, 0x080000, CRC(fc1680c8) SHA1(842d681cf9862c3bd352b6ada9fd5d40ab588230) )
	sc_alad_others
ROM_END

ROM_START( sc4alada )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403133.lo", 0x00001, 0x080000, CRC(79b60ee9) SHA1(672d30742d4ea4845bb8def4dc9945e6554a2c12) )
	ROM_LOAD16_BYTE( "95403134.hi", 0x00000, 0x080000, CRC(a4e43e34) SHA1(366c152666ee81410638794f959cda3b165bb29d) )
	sc_alad_others
ROM_END







ROM_START( sc4a40cl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416356.lo", 0x00001, 0x080000, CRC(d16af940) SHA1(350623ae459f5c93ae97461d69d20e17b85ca7d0) )
	ROM_LOAD16_BYTE( "95416357.hi", 0x00000, 0x080000, CRC(eb467f83) SHA1(e89dcfbf59845922766122e98b202602574aee8a) )
	sc_a40cl_others
ROM_END

ROM_START( sc4a40cla )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416372.lo", 0x00001, 0x080000, CRC(97398b07) SHA1(e84525e64649e4f8b466069f7dd636e1f43bd277) )
	ROM_LOAD16_BYTE( "95416373.hi", 0x00000, 0x080000, CRC(a4c22f0c) SHA1(77d4525922869ec0008c0faae85e45a215a379ad) )
	sc_a40cl_others
ROM_END

ROM_START( sc4a40clb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417356.lo", 0x00001, 0x080000, CRC(b69dc96f) SHA1(409eedbd92fc95f13a7c9026d4cf151f081867e2) )
	ROM_LOAD16_BYTE( "95417357.hi", 0x00000, 0x080000, CRC(85bb5163) SHA1(d81258287d42c752d1e577b36c882e53ba6f3d74) )
	sc_a40cl_others
ROM_END

ROM_START( sc4a40clc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417372.lo", 0x00001, 0x080000, CRC(f0cebb28) SHA1(ca8a80de389c6986b237f8495b00ae40a0baef31) )
	ROM_LOAD16_BYTE( "95417373.hi", 0x00000, 0x080000, CRC(ca3f01ec) SHA1(f0926248ef4ede28a81afcd50256f7567fa157d3) )
	sc_a40cl_others
ROM_END




ROM_START( sc4aztec )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406356.lo", 0x00001, 0x080000, CRC(a67d927d) SHA1(3b038817a2f1171a48a35de56699c2d4f0c89c87) )
	ROM_LOAD16_BYTE( "95406357.hi", 0x00000, 0x080000, CRC(ecf9885f) SHA1(8b1d8685565c4b76eddfad6a0cff9aba7ff2277d) )
	sc_aztec_others
ROM_END


ROM_START( sc4azteca )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406472.lo", 0x00001, 0x080000, CRC(c510a2d2) SHA1(4bdc0d0bacc1adf6f8426ff7c827d1ca022de653) )
	ROM_LOAD16_BYTE( "95406473.hi", 0x00000, 0x080000, CRC(6358495c) SHA1(a628cb4712177e200c762e6ce3a4c258ab70f739) )
	sc_aztec_others
ROM_END


ROM_START( sc4botn )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414710.odd", 0x00001, 0x080000, CRC(2f6f3a0d) SHA1(5635280e4b72afe3e2296e3f3657c98a723bba02) )
	ROM_LOAD16_BYTE( "95414711.evn", 0x00000, 0x080000, CRC(df3d473a) SHA1(11b5ecbf8ed897629ee3bff802b7080eadd47c1a) )
	sc_botn_others
ROM_END

ROM_START( sc4botna )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415710.odd", 0x00001, 0x080000, CRC(18881a99) SHA1(522a104602bf14e5d21d29825a85f718722f18a4) )
	ROM_LOAD16_BYTE( "95415711.evn", 0x00000, 0x080000, CRC(fee029cd) SHA1(aeb5266ee8a286ce3298eecbf50c939bd89bbf9e) )
	sc_botn_others
ROM_END




ROM_START( sc4broll )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404616.lo", 0x00001, 0x080000, CRC(8012362d) SHA1(2c965dc4d738daf2c51a01394685c9ba093a869c) )
	ROM_LOAD16_BYTE( "95404617.hi", 0x00000, 0x080000, CRC(d59b7244) SHA1(fab180e38ba591b9466c78360ad3673f5c291e04) )
	sc_broll_others
ROM_END

ROM_START( sc4brolla )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405616.lo", 0x00001, 0x080000, CRC(878a70b6) SHA1(9413d5229467f4f3f133a887fe6741c9d16af3c4) )
	ROM_LOAD16_BYTE( "95405617.hi", 0x00000, 0x080000, CRC(816c0779) SHA1(4bd726b1debb8d33e92e29831c01a6067f751d1a) )
	sc_broll_others
ROM_END

ROM_START( sc4brollb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406138.lo", 0x00001, 0x080000, CRC(c0f3cf7b) SHA1(403fb2a2a6ab951a0e85f5aa5197860b5af6fbff) )
	ROM_LOAD16_BYTE( "95406139.hi", 0x00000, 0x080000, CRC(37b359dd) SHA1(6f67af8fc17ea0a4b8faacc12dc5cc68b112e956) )
	sc_broll_others
ROM_END

ROM_START( sc4brollc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407138.lo", 0x00001, 0x080000, CRC(d85bea83) SHA1(39aa0e58854397ce3a4b449a9a01f235d566fa86) )
	ROM_LOAD16_BYTE( "95407139.hi", 0x00000, 0x080000, CRC(dde76e49) SHA1(a2c428c8afe5b949e9cbf0214c1b8d891d318229) )
	sc_broll_others
ROM_END


ROM_START( sc4bankb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408338.odd", 0x00001, 0x080000, CRC(65b31889) SHA1(a74ce2e27aa5700c89c0882dbf37f21312004ce9) )
	ROM_LOAD16_BYTE( "95408339.evn", 0x00000, 0x080000, CRC(5d404f28) SHA1(a1c5833ba5381115cd3b2e0b62708df0f03c81ac) )
	sc_bankb_others
ROM_END


ROM_START( sc4bankba )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409338.odd", 0x00001, 0x080000, CRC(b62db492) SHA1(b4c6114c42c59f1bb3cb47c51af41f03aa4f8f98) )
	ROM_LOAD16_BYTE( "95409339.evn", 0x00000, 0x080000, CRC(ee473c08) SHA1(1f9e672bf20c94b0d5dadf16b4e500a7fae244eb) )
	sc_bankb_others
ROM_END






ROM_START( sc4bed )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408048.lo", 0x00001, 0x080000, CRC(199ff9a4) SHA1(4702d782f5d9fc342087e24335207694182b2806) )
	ROM_LOAD16_BYTE( "95408049.hi", 0x00000, 0x080000, CRC(9c033822) SHA1(294e3cd4fcf1c8427f0e848aca5773cca17adcf8) )
	sc_bed_others
ROM_END

ROM_START( sc4beda )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409048.lo", 0x00001, 0x080000, CRC(89c1cd2f) SHA1(30cd1daf34384a3683450ff5d0e44e08f299a99a) )
	ROM_LOAD16_BYTE( "95409049.hi", 0x00000, 0x080000, CRC(f1bde252) SHA1(91620e2ad559e37e85b2f5acc42ec24ca9675503) )
	sc_bed_others
ROM_END

ROM_START( sc4bedb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414626.lo", 0x00001, 0x080000, CRC(4d26157a) SHA1(93f58e631884577cf3baf6e75788168742013361) )
	ROM_LOAD16_BYTE( "95414627.hi", 0x00000, 0x080000, CRC(34d2055b) SHA1(85232661acd9a6265b780547c700a514e9a00d12) )
	sc_bed_others
ROM_END

ROM_START( sc4bedc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414628.lo", 0x00001, 0x080000, CRC(d04a0902) SHA1(ed145edb57d97b13920610f884cbf57efb8d105e) )
	ROM_LOAD16_BYTE( "95414629.hi", 0x00000, 0x080000, CRC(8b254f6c) SHA1(68c0c5090a210baa74886069ec95f7865fd30259) )
	sc_bed_others
ROM_END

ROM_START( sc4bedd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415626.lo", 0x00001, 0x080000, CRC(d8071b11) SHA1(366e87eb73b70f892f4aac6b5d4c6124f03fc195) )
	ROM_LOAD16_BYTE( "95415627.hi", 0x00000, 0x080000, CRC(9098dea8) SHA1(e503c3e7546e07cdb9256fb575fdc72640e23d8c) )
	sc_bed_others
ROM_END

ROM_START( sc4bede )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415628.lo", 0x00001, 0x080000, CRC(01216587) SHA1(b16cd5eb6692298e98374ecafc02a3bb6a8374ed) )
	ROM_LOAD16_BYTE( "95415629.hi", 0x00000, 0x080000, CRC(5b7b2325) SHA1(95bc921d516bcabc5f72d698c64e09234d7c6a2d) )
	sc_bed_others
ROM_END



ROM_START( sc4bedcl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410640.lo", 0x00001, 0x080000, CRC(1a21a4e8) SHA1(8ce99fc663d2f3d7dda473bd5a081d8bb92d14ff) )
	ROM_LOAD16_BYTE( "95410641.hi", 0x00000, 0x080000, CRC(c98869cc) SHA1(968c2d532dde3845e08a1690d51016b8218d52af) )
	sc_bedcl_others
ROM_END

ROM_START( sc4bedcla )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416134.lo", 0x00001, 0x080000, CRC(9f30e058) SHA1(968a766e5348710267e5b3c01c13357e28a11e47) )
	ROM_LOAD16_BYTE( "95416135.hi", 0x00000, 0x080000, CRC(e32d8535) SHA1(2d0696624d1ff97efe815d19e12901dc74b47e6e) )
	sc_bedcl_others
ROM_END

ROM_START( sc4bedclb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416136.lo", 0x00001, 0x080000, CRC(0763400e) SHA1(0cb8801a5ebb92683b4a9b1130d13e265560a747) )
	ROM_LOAD16_BYTE( "95416137.hi", 0x00000, 0x080000, CRC(64111a60) SHA1(df5ac99b67da1f98a9e1f79ac4f73c965678294c) )
	sc_bedcl_others
ROM_END

ROM_START( sc4bedclc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417134.lo", 0x00001, 0x080000, CRC(e67e5027) SHA1(887cc032afa2ddc83e814cc1051863432d386cc2) )
	ROM_LOAD16_BYTE( "95417135.hi", 0x00000, 0x080000, CRC(279f7ce0) SHA1(826c586cc97f85dd36dfdece969ddfbb41833b27) )
	sc_bedcl_others
ROM_END

ROM_START( sc4bedcld )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417136.lo", 0x00001, 0x080000, CRC(7e2df071) SHA1(aacaf1c4a49d62e68902b98994cb6aed3bbb7a02) )
	ROM_LOAD16_BYTE( "95417137.hi", 0x00000, 0x080000, CRC(a0a3e3b5) SHA1(138d682649d0167d1d6bbb1e60a5cbff4a2822b6) )
	sc_bedcl_others
ROM_END



ROM_START( sc4bblas )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406966.lo", 0x00001, 0x080000, CRC(5009dfa2) SHA1(9ed13c78a22b5d7e490eb5205ff7ed1a543c3ff7) )
	ROM_LOAD16_BYTE( "95406967.hi", 0x00000, 0x080000, CRC(64830e10) SHA1(a5c5ec99fb906c94a551c9c3ded79515d3db9898) )
	sc_bblas_others
ROM_END

ROM_START( sc4bblasa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407966.lo", 0x00001, 0x080000, CRC(f3ff24d8) SHA1(6b3f25fbac616ea4d4b38e8f428e7da818ede57b) )
	ROM_LOAD16_BYTE( "95407967.hi", 0x00000, 0x080000, CRC(0a7e20f0) SHA1(0ea65cddf94ead103268d0d3444aaaa46603ea14) )
	sc_bblas_others
ROM_END

ROM_START( sc4bblasb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414284.lo", 0x00001, 0x080000, CRC(401f01c3) SHA1(6b6ca3aa67983b714b6c5f13e31b217e29290e62) )
	ROM_LOAD16_BYTE( "95414285.hi", 0x00000, 0x080000, CRC(e4e98606) SHA1(f8541acd7277ceabdb520e7696bf902452da17a5) )
	sc_bblas_others
ROM_END

ROM_START( sc4bblasc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414630.lo", 0x00001, 0x080000, CRC(b8799cb5) SHA1(5826e3a8876d0a78384584267d7eef6937656036) )
	ROM_LOAD16_BYTE( "95414631.hi", 0x00000, 0x080000, CRC(0177e26f) SHA1(8e91acf82d4c67fd1da66f7784746229083f69ce) )
	sc_bblas_others
ROM_END

ROM_START( sc4bblasd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414632.lo", 0x00001, 0x080000, CRC(cf3dcff8) SHA1(83602af5db08d88b2eefca70fc2f9df841e9c019) )
	ROM_LOAD16_BYTE( "95414633.hi", 0x00000, 0x080000, CRC(a2ad3903) SHA1(ff38e4c9efbbc425c7b02ee33e853ce0e6bdcd77) )
	sc_bblas_others
ROM_END

ROM_START( sc4bblase )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415630.lo", 0x00001, 0x080000, CRC(595b795d) SHA1(8d513952a3c317158a6e4e25dfa4e2c2e8c1ab66) )
	ROM_LOAD16_BYTE( "95415631.hi", 0x00000, 0x080000, CRC(bfd4a0c6) SHA1(21acef7f009a240eaafc6f2b4dc1b17ad24e8412) )
	sc_bblas_others
ROM_END

ROM_START( sc4bblasf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415632.lo", 0x00001, 0x080000, CRC(44f83449) SHA1(cab4b3c63b17467fa0318f524d3feeb6d579bf4d) )
	ROM_LOAD16_BYTE( "95415633.hi", 0x00000, 0x080000, CRC(1c912d92) SHA1(c1f5c1399cb4c56a86c28c4833fa210fc14b8979) )
	sc_bblas_others
ROM_END


ROM_START( sc4bigdl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404658.odd", 0x00001, 0x080000, CRC(71e605a1) SHA1(8889118669c0c72181ec50b83234d77a061f623f) )
	ROM_LOAD16_BYTE( "95404659.evn", 0x00000, 0x080000, CRC(6eeb199c) SHA1(e5062b111d442818caa9b60eeff350ade563d760) )
	sc_bigdl_others
ROM_END

ROM_START( sc4bigdla )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405658.odd", 0x00001, 0x080000, CRC(2937ff83) SHA1(d36131383f213ec2aaa0c1aebb38e5ec25ec28ab) )
	ROM_LOAD16_BYTE( "95405659.evn", 0x00000, 0x080000, CRC(38fe6279) SHA1(0d121063322df609c95bf668d672d6f0490c5718) )
	sc_bigdl_others
ROM_END


ROM_START( sc4bingb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416280.lo", 0x00001, 0x080000, CRC(987d1e51) SHA1(5fba6b545532bf6749543be0978a02600e292368) )
	ROM_LOAD16_BYTE( "95416281.hi", 0x00000, 0x080000, CRC(664987e0) SHA1(c024730b82747896f3b4c87b981d40996d12ac90) )
	sc_bingb_others
ROM_END

ROM_START( sc4bingba )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417280.lo", 0x00001, 0x080000, CRC(b30a43aa) SHA1(fe90208cdd3eb5fb1306c32a01c542730ac59fd1) )
	ROM_LOAD16_BYTE( "95417281.hi", 0x00000, 0x080000, CRC(72d88d9b) SHA1(b2152119842f40afa474195015e38f0c45971205) )
	sc_bingb_others
ROM_END



ROM_START( sc4bb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414574.odd", 0x00001, 0x080000, CRC(6ff4950c) SHA1(5c3c458f7be55aab03a7fde7ff3c4eac0421e89e) )
	ROM_LOAD16_BYTE( "95414575.evn", 0x00000, 0x080000, CRC(ffa0e513) SHA1(0b68114dbbd87a02e90691de4534de224f7b8ad0) )
	sc_bb_others
ROM_END

ROM_START( sc4bba )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415574.odd", 0x00001, 0x080000, CRC(3dc1edac) SHA1(e2c8d4757a470432b4280100d9b6ff40e6eb51af) )
	ROM_LOAD16_BYTE( "95415575.evn", 0x00000, 0x080000, CRC(eb31ef68) SHA1(c666f7cbed7f76fd7806412f248c6d02f5d38028) )
	sc_bb_others
ROM_END



ROM_START( sc4bbclb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408698.odd", 0x00001, 0x080000, CRC(456c8642) SHA1(adbad206dc692f2feb2d9c69a8d92addf0cb4692) )
	ROM_LOAD16_BYTE( "95408699.evn", 0x00000, 0x080000, CRC(46780e5d) SHA1(b5c2cd8c7209d18d312108cce16fda30a5a4aac1) )
	sc_bbclb_others
ROM_END

ROM_START( sc4bbclba )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410110.odd", 0x00001, 0x080000, CRC(e7c636c6) SHA1(fb8c80feca1d2d21b9cf74684d9e897432c813f7) )
	ROM_LOAD16_BYTE( "95410111.evn", 0x00000, 0x080000, CRC(230ad5c8) SHA1(8eebb8bf6cd5acf174adb9723090f842cfab2b1f) )
	sc_bbclb_others
ROM_END

ROM_START( sc4bbclbb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410916.odd", 0x00001, 0x080000, CRC(7d097be7) SHA1(1d0344a4bd21e445bcd1809f73dbb62dabd1d96a) )
	ROM_LOAD16_BYTE( "95410917.evn", 0x00000, 0x080000, CRC(253f98a0) SHA1(59451ee64979642192fa6976a426ea540de90723) )
	sc_bbclb_others
ROM_END

ROM_START( sc4bbclbc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411916.odd", 0x00001, 0x080000, CRC(3dc4b066) SHA1(91bdb0c53da149ec50787995429429dd71deb776) )
	ROM_LOAD16_BYTE( "95411917.evn", 0x00000, 0x080000, CRC(25a0ce98) SHA1(17d4bcec407b3a6488b1d30a9a5651cd8e8b6676) )
	sc_bbclb_others
ROM_END



ROM_START( sc4blast )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412330.odd", 0x00001, 0x080000, CRC(9be05578) SHA1(ca51be09995523d959f646e142dc76b79bda9973) )
	ROM_LOAD16_BYTE( "95412331.evn", 0x00000, 0x080000, CRC(b9dd2235) SHA1(7c5b11cdc1f2fdd3118fc495dfb4ceb0bd6acd5b) )
	sc_roksc_others
ROM_END

ROM_START( sc4blasta )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412332.odd", 0x00001, 0x080000, CRC(3571f826) SHA1(7bb18370bd70d71c4049a40f0676b363b1917467) )
	ROM_LOAD16_BYTE( "95412333.evn", 0x00000, 0x080000, CRC(0a3fc7a1) SHA1(483c7ef030ac48f42538df8cd51d3192b9582ca0) )
	sc_roksc_others
ROM_END

ROM_START( sc4blastb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413330.odd", 0x00001, 0x080000, CRC(789c06d3) SHA1(ad46c911e6525033c549953e1b45acd0a8512378) )
	ROM_LOAD16_BYTE( "95413331.evn", 0x00000, 0x080000, CRC(6a5fecd4) SHA1(a13012664065c1e9e0f02c76da0dfffe7077d3e9) )
	sc_roksc_others
ROM_END

ROM_START( sc4blastc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413332.odd", 0x00001, 0x080000, CRC(474fa683) SHA1(3ebe087be82e2c9cf3f1addd7cc19c8a6a07f772) )
	ROM_LOAD16_BYTE( "95413333.evn", 0x00000, 0x080000, CRC(c3cbc622) SHA1(3410b0b8e8d9cc3abcb396775c127a33fffbe73d) )
	sc_roksc_others
ROM_END

ROM_START( sc4blastd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414620.odd", 0x00001, 0x080000, CRC(2de2f00b) SHA1(f214fe86e7f9a131e0370a4efc724c402c4dfb63) )
	ROM_LOAD16_BYTE( "95414621.evn", 0x00000, 0x080000, CRC(c8309fe3) SHA1(e7e3e81d55ccced33569e3dd3e8f27a075402b49) )
	sc_roksc_others
ROM_END

ROM_START( sc4blaste )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415620.odd", 0x00001, 0x080000, CRC(0d72fde7) SHA1(bbf353792cf8f57a0fa8a7265b027cf102a9ca5f) )
	ROM_LOAD16_BYTE( "95415621.evn", 0x00000, 0x080000, CRC(fde3ad57) SHA1(42120f78c86b051eb904d0c3ee7c7de3d53ae1ad) )
	sc_roksc_others
ROM_END




ROM_START( sc4blue )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400329.lo", 0x00001, 0x080000, CRC(32f4c121) SHA1(06eb684a2888bb9dbae9624c36720fda2e949b4c) )
	ROM_LOAD16_BYTE( "95400330.hi", 0x00000, 0x080000, CRC(7b59182d) SHA1(c775b95665ce7c7a2b73db744e86acd445f9c1b2) )
	sc_blue_others
ROM_END

ROM_START( sc4bluea )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400371.lo", 0x00001, 0x080000, CRC(fc5128a0) SHA1(590d323f476d5d9d95bbf797a20eb00845ed28a1) )
	ROM_LOAD16_BYTE( "95400372.hi", 0x00000, 0x080000, CRC(95ca9b7b) SHA1(47fe6d9895345d25dcccb80d4598c0210a80f38d) )
	sc_blue_others
ROM_END

ROM_START( sc4blueb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400673.lo", 0x00001, 0x080000, CRC(34984f98) SHA1(3b0a07f0cdca6d2cff91b984fb35d1b19c00b1cb) )
	ROM_LOAD16_BYTE( "95400674.hi", 0x00000, 0x080000, CRC(bf9a02d7) SHA1(07f55774cdc39d169ba6f4171fa51dc96a524f94) )
	sc_blue_others
ROM_END

ROM_START( sc4bluec )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401329.lo", 0x00001, 0x080000, CRC(eabcd848) SHA1(9cfa385e4a4db0c87ed8686165150c5325e7ebd8) )
	ROM_LOAD16_BYTE( "95401330.hi", 0x00000, 0x080000, CRC(2cacf287) SHA1(389b53d47979a5670292ac8d459bae072112e0f1) )
	sc_blue_others
ROM_END

ROM_START( sc4blued )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401331.lo", 0x00001, 0x080000, CRC(f1da5967) SHA1(f6aebc89a3963d314f7b1f60678c7888bae211f5) )
	ROM_LOAD16_BYTE( "95401332.hi", 0x00000, 0x080000, CRC(5eac416c) SHA1(5ffde639f02141979d16ae74d514b0add08b4267) )
	sc_blue_others
ROM_END

ROM_START( sc4bluee )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401673.lo", 0x00001, 0x080000, CRC(b9b543fb) SHA1(2d8c5ab99003c8f8a36543972a0e037008a36acd) )
	ROM_LOAD16_BYTE( "95401674.hi", 0x00000, 0x080000, CRC(2d401f00) SHA1(117bfd706c517e9f188fd322939027ef71c573a4) )
	sc_blue_others
ROM_END






ROM_START( sc4bobcl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416880.lo", 0x00001, 0x080000, CRC(94c1b021) SHA1(588bfdf662157382111780b4689cebaa96f32d3c) )
	ROM_LOAD16_BYTE( "95416881.hi", 0x00000, 0x080000, CRC(242e03e0) SHA1(c4d596e5543852d8d1ede9cf9615d4e6ee0f3209) )
	sc_bobcl_others
ROM_END

ROM_START( sc4bobcla )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417880.lo", 0x00001, 0x080000, CRC(d1960747) SHA1(c95db85832cc8acb29c3ae7066d5109f2621e3ae) )
	ROM_LOAD16_BYTE( "95417881.hi", 0x00000, 0x080000, CRC(d40cfb16) SHA1(3a200fbb3c5fed4d1602f43e7ba2858436a6d35f) )
	sc_bobcl_others
ROM_END




ROM_START( sc4bonbx )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414234.lo", 0x00001, 0x080000, CRC(68bd7c9a) SHA1(73a798755ce4cfa8fd95d5c3a2bea2a240432a67) )
	ROM_LOAD16_BYTE( "95414235.hi", 0x00000, 0x080000, CRC(6e5c5047) SHA1(c2a5e98f6a2c519a07833a6f3029d5f3fb65c026) )
	sc_bonbx_others
ROM_END

ROM_START( sc4bonbxa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414420.lo", 0x00001, 0x080000, CRC(17c6e99c) SHA1(9194009c4c1d9a1593fac15a57753673dee468a8) )
	ROM_LOAD16_BYTE( "95414421.hi", 0x00000, 0x080000, CRC(8b2954e3) SHA1(ad39ac956d4bbd3ed4b3101cf20e632337b0d4e4) )
	sc_bonbx_others
ROM_END

ROM_START( sc4bonbxb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414538.lo", 0x00001, 0x080000, CRC(b1560f4d) SHA1(d4d1dc0fd418e8eac617e0a4b8e5b0f67ab248a9) )
	ROM_LOAD16_BYTE( "95414539.hi", 0x00000, 0x080000, CRC(7ff8212d) SHA1(fb52ed0bd0930ed927671755edbfa79ecdbeb211) )
	sc_bonbx_others
ROM_END

ROM_START( sc4bonbxc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415234.lo", 0x00001, 0x080000, CRC(bfb6e866) SHA1(dac56d11eae1268e9f1c389b65de024ca0319a24) )
	ROM_LOAD16_BYTE( "95415235.hi", 0x00000, 0x080000, CRC(9ee1fe89) SHA1(1dfe5f321c376c161ec374c916f3b08a63b7e3e1) )
	sc_bonbx_others
ROM_END

ROM_START( sc4bonbxd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415236.lo", 0x00001, 0x080000, CRC(2b4cef52) SHA1(2114331e0f292f1da0fa4bd852c8317384a71a21) )
	ROM_LOAD16_BYTE( "95415237.hi", 0x00000, 0x080000, CRC(bf8065a6) SHA1(bbf2d5f49f9854b92a7f4e2b5e1fc6cfd5072268) )
	sc_bonbx_others
ROM_END

ROM_START( sc4bonbxe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415238.lo", 0x00001, 0x080000, CRC(0fcac068) SHA1(aba80a5a96ccfd6b9c05e63f28afa5afb810ac5e) )
	ROM_LOAD16_BYTE( "95415239.hi", 0x00000, 0x080000, CRC(73b40b12) SHA1(582cf89b4bb0dcbb189a694158782e7fc2d108ee) )
	sc_bonbx_others
ROM_END

ROM_START( sc4bonbxf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415418.lo", 0x00001, 0x080000, CRC(861ac22e) SHA1(8d238d52045b90dbbe236b59dc13ed36a2a9f6bd) )
	ROM_LOAD16_BYTE( "95415419.hi", 0x00000, 0x080000, CRC(ba75fe01) SHA1(8ca5b3559c1c9c72035d9926a0e68262f7cd958a) )
	sc_bonbx_others
ROM_END

ROM_START( sc4bonbxg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415420.lo", 0x00001, 0x080000, CRC(7235848f) SHA1(65e919674920084f530f96788bc8a75997fe7bbe) )
	ROM_LOAD16_BYTE( "95415421.hi", 0x00000, 0x080000, CRC(1a2feb9c) SHA1(e9fdbf8e944c9b758b822949ad2ef4c20b511597) )
	sc_bonbx_others
ROM_END

ROM_START( sc4bonbxh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415422.lo", 0x00001, 0x080000, CRC(22149879) SHA1(336e41430029385b6c644ed343f65416a1390d8c) )
	ROM_LOAD16_BYTE( "95415423.hi", 0x00000, 0x080000, CRC(14d6ca23) SHA1(88184d5dce82f014867c0b656e3a581f664e9a38) )
	sc_bonbx_others
ROM_END

ROM_START( sc4bonbxi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415424.lo", 0x00001, 0x080000, CRC(2d491a10) SHA1(4f2600d28f4b1429f95ceff39165e28008446ea1) )
	ROM_LOAD16_BYTE( "95415425.hi", 0x00000, 0x080000, CRC(963b5f3b) SHA1(c1383d2aa5bf7ae8127ecab57632981a74b964cf) )
	sc_bonbx_others
ROM_END

ROM_START( sc4bonbxj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415538.lo", 0x00001, 0x080000, CRC(a0aaa513) SHA1(f2320884efc2c7b04bf3dddcac2baba40fd007b9) )
	ROM_LOAD16_BYTE( "95415539.hi", 0x00000, 0x080000, CRC(6ac8d11e) SHA1(dca72f4be29d34f6532d01954565bb4213231a48) )
	sc_bonbx_others
ROM_END

ROM_START( sc4bonbxk )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415540.lo", 0x00001, 0x080000, CRC(7550199f) SHA1(a1ef86e143cdf993f63034c884ea341bbda15488) )
	ROM_LOAD16_BYTE( "95415541.hi", 0x00000, 0x080000, CRC(1d68b1d0) SHA1(f269bd3ba2db0c1fdacf88b50b0f3b335685b281) )
	sc_bonbx_others
ROM_END

ROM_START( sc4bonbxl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415542.lo", 0x00001, 0x080000, CRC(eb8f85d0) SHA1(98b2ef28c800997f26ee50c5d428d4268540eac8) )
	ROM_LOAD16_BYTE( "95415543.hi", 0x00000, 0x080000, CRC(48f17abd) SHA1(761ed166d553306b28084bf243ac4adabbea54e1) )
	sc_bonbx_others
ROM_END




ROM_START( sc4brksp )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404458.lo", 0x00001, 0x080000, CRC(89d6e62f) SHA1(8c6df05583e04f705dd4c4c968b7e84ad9bd9d22) )
	ROM_LOAD16_BYTE( "95404459.hi", 0x00000, 0x080000, CRC(ef22a826) SHA1(e7710f8858d905358d7f78be1ce767c080e4b2fa) )
	sc_brksp_others
ROM_END

ROM_START( sc4brkspa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405458.lo", 0x00001, 0x080000, CRC(31722ba9) SHA1(83b5a20ac2e5f07958f3b42adc5e94cc8dcfeac9) )
	ROM_LOAD16_BYTE( "95405459.hi", 0x00000, 0x080000, CRC(49152b35) SHA1(fba0947c54dbf69bf342f7e192233c83dc4a6025) )
	sc_brksp_others
ROM_END




ROM_START( sc4brix )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404786.lo", 0x00001, 0x080000, CRC(fa9a8b28) SHA1(20e5ba6d4b4fb8fd2f17043a23b676922de28379) )
	ROM_LOAD16_BYTE( "95404787.hi", 0x00000, 0x080000, CRC(883ab49f) SHA1(8697c2703da16f07639ef3d678a4779b4a68d98a) )
	sc_brix_others
ROM_END

ROM_START( sc4brixa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408176.lo", 0x00001, 0x080000, CRC(87e00ec4) SHA1(2692a72c358b4af6c89393495ca27b986b2ff2a7) )
	ROM_LOAD16_BYTE( "95408177.hi", 0x00000, 0x080000, CRC(19dac202) SHA1(f9b122f06dd988063f5eb7c8fd0a7e07006a4224) )
	sc_brix_others
ROM_END

ROM_START( sc4brixb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408510.lo", 0x00001, 0x080000, CRC(5ec0efa2) SHA1(47c56507f5878b7be7e9d222d16aea3904028531) )
	ROM_LOAD16_BYTE( "95408511.hi", 0x00000, 0x080000, CRC(ccb8008d) SHA1(573ed2fdd7e61603d1c6cfa04f1a801b201344ce) )
	sc_brix_others
ROM_END




ROM_START( sc4bugs )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402359.lo", 0x00001, 0x080000, CRC(5fb17f27) SHA1(410070592ac6341aa92df657f92860dba68be579) )
	ROM_LOAD16_BYTE( "95402360.hi", 0x00000, 0x080000, CRC(14e3e3d8) SHA1(701b3c74e18549435deb884f44589c3c23c6a69b) )
	sc_bugs_others
ROM_END

ROM_START( sc4bugsa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402361.lo", 0x00001, 0x080000, CRC(93daa0c9) SHA1(d793611a572463dfd95686bbdbad8323db819b43) )
	ROM_LOAD16_BYTE( "95402362.hi", 0x00000, 0x080000, CRC(5fe0ba99) SHA1(ae17cad184aeb0f29f88509de780c31a78a7f71a) )
	sc_bugs_others
ROM_END

ROM_START( sc4bugsb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403359.lo", 0x00001, 0x080000, CRC(7d6364ca) SHA1(48343572910b753950afd540b9b06fccc16a708e) )
	ROM_LOAD16_BYTE( "95403360.hi", 0x00000, 0x080000, CRC(2d351aad) SHA1(58675bf335c18306ce16d4745c2f3063d102b176) )
	sc_bugs_others
ROM_END

ROM_START( sc4bugsc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403361.lo", 0x00001, 0x080000, CRC(d36a6c8a) SHA1(e469c83a3e01189eb981c900447669aa06571921) )
	ROM_LOAD16_BYTE( "95403362.hi", 0x00000, 0x080000, CRC(33ff9aa1) SHA1(68e064931af100eb846e1b689ec8d8707d44b60a) )
	sc_bugs_others
ROM_END





ROM_START( sc4bulcl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95418032.lo", 0x00001, 0x080000, CRC(1eb87819) SHA1(0ce61f22d5038da059325b17a1e68788dff54460) )
	ROM_LOAD16_BYTE( "95418033.hi", 0x00000, 0x080000, CRC(88c17569) SHA1(7c115440e531266d03e8b47b2e86b85b413d4508) )
	sc_bulcl_others
ROM_END

ROM_START( sc4bulcla )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95418038.lo", 0x00001, 0x080000, CRC(579346ca) SHA1(1a5226b381e3b443ca978f534245f97648d32488) )
	ROM_LOAD16_BYTE( "95418039.hi", 0x00000, 0x080000, CRC(8be73e67) SHA1(44a79674c49938a2547e4a0ece9f9ad2f2604aaa) )
	sc_bulcl_others
ROM_END

ROM_START( sc4bulclb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95418302.lo", 0x00001, 0x080000, CRC(b6a10914) SHA1(94f1709700b925c7c4a29bc7045dd83126cc8175) )
	ROM_LOAD16_BYTE( "95418303.hi", 0x00000, 0x080000, CRC(2de75bac) SHA1(ba2686bfe86cb20d62c1eb3312180db6477e3125) )
	sc_bulcl_others
ROM_END

ROM_START( sc4bulclc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95418362.lo", 0x00001, 0x080000, CRC(2eeb5021) SHA1(2fc456ca457d884c701f32ba6dc0cc367409691f) )
	ROM_LOAD16_BYTE( "95418363.hi", 0x00000, 0x080000, CRC(ff842f06) SHA1(98fa76b55314870869fcbb70fffb1bd81cbb3bb0) )
	sc_bulcl_others
ROM_END

ROM_START( sc4bulcld )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95419032.lo", 0x00001, 0x080000, CRC(386addda) SHA1(397905e5bfde65ac6e6ad6d4531b9faa305bfc4c) )
	ROM_LOAD16_BYTE( "95419033.hi", 0x00000, 0x080000, CRC(006da7dc) SHA1(8af127afc32976146dfc2290bd648f4c910040c5) )
	sc_bulcl_others
ROM_END

ROM_START( sc4bulcle )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95419038.lo", 0x00001, 0x080000, CRC(8365dc75) SHA1(9fc6ee859144b08a791e0357dc4855a50128b8cb) )
	ROM_LOAD16_BYTE( "95419039.hi", 0x00000, 0x080000, CRC(16e44ad9) SHA1(510b6d55591e540700e6a67520330619c8800738) )
	sc_bulcl_others
ROM_END

ROM_START( sc4bulclf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95419302.lo", 0x00001, 0x080000, CRC(0c58753a) SHA1(ab82c97090cd9673a0a26e9853ea3261fc723268) )
	ROM_LOAD16_BYTE( "95419303.hi", 0x00000, 0x080000, CRC(a3257e7a) SHA1(ba08e5d01b4dd4060493c658f8fec767dfa24d1e) )
	sc_bulcl_others
ROM_END

ROM_START( sc4bulclg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95419362.lo", 0x00001, 0x080000, CRC(ee0c0a9b) SHA1(04334c242a9b7a6e9e59c0c0588883780e829788) )
	ROM_LOAD16_BYTE( "95419363.hi", 0x00000, 0x080000, CRC(a77691fa) SHA1(988ca1d9b5cf89a33972f4b0128afcfb2b898134) )
	sc_bulcl_others
ROM_END

ROM_START( sc4bulclh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95422482.lo", 0x00001, 0x080000, CRC(7e158734) SHA1(021648372f8a089325fda98476ad44f66dc4552c) )
	ROM_LOAD16_BYTE( "95422483.hi", 0x00000, 0x080000, CRC(0848d756) SHA1(e428f5771f2bb9e1ebebf0c896ae95aa119faa52) )
	sc_bulcl_others
ROM_END

ROM_START( sc4bulcli )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95423482.lo", 0x00001, 0x080000, CRC(a1d0dff6) SHA1(3146d50a080cb85ac88d6afa1f3f84972b2aa563) )
	ROM_LOAD16_BYTE( "95423483.hi", 0x00000, 0x080000, CRC(6eaa6693) SHA1(36f513bd7bc73c4884464916e5c803ce7da564c7) )
	sc_bulcl_others
ROM_END








ROM_START( sc4canca )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404028.lo", 0x00001, 0x080000, CRC(68b5123f) SHA1(feda4dcc17950c99bb91e081425ee3d18eff98bc) )
	ROM_LOAD16_BYTE( "95404029.hi", 0x00000, 0x080000, CRC(c2d52c27) SHA1(6693e0647f8de08d2594dbeba14ff154fa82d00a) )
	sc_canca
ROM_END


ROM_START( sc4cancaa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405028.lo", 0x00001, 0x080000, CRC(57097d75) SHA1(6eff17e4a0612da502c660e1c864348b5df91339) )
	ROM_LOAD16_BYTE( "95405029.hi", 0x00000, 0x080000, CRC(8312a929) SHA1(d50f70c0d953ec26db72a560e1533c12d84ec281) )
	sc_canca
ROM_END


ROM_START( sc4cancab )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417000.lo", 0x00001, 0x080000, CRC(e3f0f86c) SHA1(f8a2bf2ea1222117b9ff7096ad1920e77838c9f4) )
	ROM_LOAD16_BYTE( "95417001.hi", 0x00000, 0x080000, CRC(9912f9cd) SHA1(c26f90a1c17fcff4be80961f5c9f157418dc0757) )
	sc_canca
ROM_END


ROM_START( sc4cancac )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417002.lo", 0x00001, 0x080000, CRC(690dc008) SHA1(922060edcda9e72161d267d15807584755b2c993) )
	ROM_LOAD16_BYTE( "95417003.hi", 0x00000, 0x080000, CRC(d09ab572) SHA1(e1fdd3e25050b05850764bea086215d5501b486e) )
	sc_canca
ROM_END




ROM_START( sc4captn )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404222.lo", 0x00001, 0x080000, CRC(ddde5d23) SHA1(7b5638c46b8e38f834eb5e6bc49a10f5128041f0) )
	ROM_LOAD16_BYTE( "95404223.hi", 0x00000, 0x080000, CRC(03ab5609) SHA1(5a2161aca95ceaff41872aeb8575da85df4e8bea) )
	sc_captn_others
ROM_END

ROM_START( sc4captna )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404684.lo", 0x00001, 0x080000, CRC(526ea454) SHA1(623927d8742beac94d30c7d9ee570e18ee0bc466) )
	ROM_LOAD16_BYTE( "95404685.hi", 0x00000, 0x080000, CRC(8698ce0a) SHA1(542b730c2675c526f8fdef5f76e51f20231703cd) )
	sc_captn_others
ROM_END

ROM_START( sc4captnb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405084.lo", 0x00001, 0x080000, CRC(0cbe758f) SHA1(df50e45ee55e5de8a7053bb4d19daad370a54f61) )
	ROM_LOAD16_BYTE( "95405085.hi", 0x00000, 0x080000, CRC(a60cae02) SHA1(d64bf27e10a248f5f9f08c376debb02fed18849d) )
	sc_captn_others
ROM_END

ROM_START( sc4captnc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405222.lo", 0x00001, 0x080000, CRC(4924f42d) SHA1(113895ee36d1671384ec01777aaac9dc4bdbbcaa) )
	ROM_LOAD16_BYTE( "95405223.hi", 0x00000, 0x080000, CRC(9f09d8ff) SHA1(3bce6f2cd7df4c006b40c41abaa60b1881fa1e16) )
	sc_captn_others
ROM_END

ROM_START( sc4captnd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405684.lo", 0x00001, 0x080000, CRC(45b61657) SHA1(2cbefd0a95fd5b8fe22d6023e7e8b3a6a568aa3a) )
	ROM_LOAD16_BYTE( "95405685.hi", 0x00000, 0x080000, CRC(e7bc8983) SHA1(fa6de12869fd4a043036e8b309bb1886230cdc4a) )
	sc_captn_others
ROM_END

ROM_START( sc4captne )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406102.odd", 0x00001, 0x080000, CRC(d38d7367) SHA1(037ff25a30f005bb44a408c1262806f80609b7c6) )
	ROM_LOAD16_BYTE( "95406103.evn", 0x00000, 0x080000, CRC(14109c1f) SHA1(4d88cd173adda2f2bcec2634127165bbd127ec7b) )
	sc_captn_others
ROM_END

ROM_START( sc4captnf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407102.odd", 0x00001, 0x080000, CRC(766c1e34) SHA1(f04ea9099c9dfd51781f7b0d86a100ac42937630) )
	ROM_LOAD16_BYTE( "95407103.evn", 0x00000, 0x080000, CRC(6c9eb65c) SHA1(9a43720e9ee8be71dc0c2d526191867eeaa26b7e) )
	sc_captn_others
ROM_END







ROM_START( sc4carry )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1113p12.hi", 0x00000, 0x080000, CRC(dbf6b83d) SHA1(1c0d285b2b3d6ca986ad0ddff6e3c2bd8c3745c0) )
	ROM_LOAD16_BYTE( "p1113p12.lo", 0x00001, 0x080000, CRC(d27305c4) SHA1(0931b4dd015bcbafa4e5059d3f36c656115b5866) )
	sc_carry_others
ROM_END


ROM_START( sc4carrya )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1113s12.hi", 0x00000, 0x080000, CRC(73054e1d) SHA1(8d3e017bcd69c845deb52c314ee40739c27ba4c4) )
	ROM_LOAD16_BYTE( "p1113s12.lo", 0x00001, 0x080000, CRC(80e615f7) SHA1(d4ba4b1ee1766cc9aee8fadeb1841a758fb2fc44) )
	sc_carry_others
ROM_END




ROM_START( sc4cmous )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412114.odd", 0x00001, 0x080000, CRC(f79b62c7) SHA1(a58f58cdfe63ac5a51ca1919d60dadd036d0eec0) )
	ROM_LOAD16_BYTE( "95412115.evn", 0x00000, 0x080000, CRC(dfc00700) SHA1(da9324f35e1d824d394e81dc99218096f78d3f09) )
	sc_cmous_others
ROM_END

ROM_START( sc4cmousa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412116.odd", 0x00001, 0x080000, CRC(4bfc675b) SHA1(522bb99b23d6580273cd9aead497cd6216efac59) )
	ROM_LOAD16_BYTE( "95412117.evn", 0x00000, 0x080000, CRC(5fe9d243) SHA1(f4bf0eb4e097c8d46ef62781d157f462b0c7c896) )
	sc_cmous_others
ROM_END

ROM_START( sc4cmousb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413114.odd", 0x00001, 0x080000, CRC(a4be03d3) SHA1(76d9a7f08f134b0bea680d610c924aab8ec8d040) )
	ROM_LOAD16_BYTE( "95413115.evn", 0x00000, 0x080000, CRC(2fe2fff6) SHA1(ad020f291b06fb75ae3e8607fa96431868fbfdda) )
	sc_cmous_others
ROM_END

ROM_START( sc4cmousc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413116.odd", 0x00001, 0x080000, CRC(1384d852) SHA1(c75e3f05bcb4f8c78c5da800267b60b21e75463d) )
	ROM_LOAD16_BYTE( "95413117.evn", 0x00000, 0x080000, CRC(f6986e01) SHA1(1bdf5acff0151313defc50e529d88b200000f1c4) )
	sc_cmous_others
ROM_END




ROM_START( sc4cburn )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "96000038.lo", 0x00001, 0x080000, CRC(e311ce43) SHA1(c9395c886d76a8aabe35a874f10489cdc02bbd35) )
	ROM_LOAD16_BYTE( "96000039.hi", 0x00000, 0x080000, CRC(0f3815a9) SHA1(7b30e5f246a953c864a074981718f9540ba22daf) )
	sc_hotpr_others
ROM_END


ROM_START( sc4cburna )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "96000040.odd", 0x00001, 0x080000, CRC(da99424b) SHA1(3c74bf35f64a3bc7d301f53a4307c8c4c73059c9) )
	ROM_LOAD16_BYTE( "96000041.evn", 0x00000, 0x080000, CRC(49c6d054) SHA1(88985743f6e56f5c9557a75bb8e0861659d5be28) )
	sc_hotpr_others
ROM_END




ROM_START( sc4cnfr )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400447.lo", 0x00001, 0x080000, CRC(4e7e04ae) SHA1(bb19a62ad8fee65502fed2a282d0064ce4404cd4) )
	ROM_LOAD16_BYTE( "95400448.hi", 0x00000, 0x080000, CRC(8a411de5) SHA1(fad2f0528666ebd47c1a8ac3deb2146df1489bc8) )
	sc_cnfr_others
ROM_END

ROM_START( sc4cnfra )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400449.lo", 0x00001, 0x080000, CRC(181a46cb) SHA1(f09c7fd0c840517965edbbd31f86dd2278599246) )
	ROM_LOAD16_BYTE( "95400450.hi", 0x00000, 0x080000, CRC(a81f2ddc) SHA1(07a6d524ccab3c7e098912df605226d0654cb31c) )
	sc_cnfr_others
ROM_END

ROM_START( sc4cnfrb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400565.lo", 0x00001, 0x080000, CRC(88e0924b) SHA1(51587db4a6af6ea49b1a55fce1c6316f0af78fd1) )
	ROM_LOAD16_BYTE( "95400566.hi", 0x00000, 0x080000, CRC(9f9480f5) SHA1(f5482cbf001c48a4effe2d53ef9b4621ee6be2f4) )
	sc_cnfr_others
ROM_END

ROM_START( sc4cnfrc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400735.lo", 0x00001, 0x080000, CRC(b866a557) SHA1(92bde33704df07178a2e26324aa73e15fe62fb3d) )
	ROM_LOAD16_BYTE( "95400736.hi", 0x00000, 0x080000, CRC(325d9269) SHA1(e5a2fb03cd2728775b187e9d9142b632c6e15af8) )
	sc_cnfr_others
ROM_END

ROM_START( sc4cnfrd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400737.lo", 0x00001, 0x080000, CRC(f632cb6a) SHA1(ddde44ec46144e333be0199caea821a6ecee8764) )
	ROM_LOAD16_BYTE( "95400738.hi", 0x00000, 0x080000, CRC(0b93b60f) SHA1(dbcc3012468f8511fba83bd18c94406660a36375) )
	sc_cnfr_others
ROM_END

ROM_START( sc4cnfre )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401447.lo", 0x00001, 0x080000, CRC(a08b02ba) SHA1(5d6f56937891f042714e90f2c4b2617775282c00) )
	ROM_LOAD16_BYTE( "95401448.hi", 0x00000, 0x080000, CRC(e8b9f8c4) SHA1(8c4b687366c8a5daa2a0325e8e9966afdf7e8618) )
	sc_cnfr_others
ROM_END

ROM_START( sc4cnfrf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401449.lo", 0x00001, 0x080000, CRC(c1de7cb3) SHA1(fc43bd47c2e83e1ed14a6e96e96d215e5f9e27cd) )
	ROM_LOAD16_BYTE( "95401450.hi", 0x00000, 0x080000, CRC(dc94cc5e) SHA1(0d3262ffd9b8236a8c172ac0cb70fefd3f0cbfed) )
	sc_cnfr_others
ROM_END

ROM_START( sc4cnfrg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401565.lo", 0x00001, 0x080000, CRC(fa430cee) SHA1(03154148bc73eaef30330ce41aa59df8d816a5f3) )
	ROM_LOAD16_BYTE( "95401566.hi", 0x00000, 0x080000, CRC(c91ead28) SHA1(d3545f4fc715ad35f7a969397c15551659c405ad) )
	sc_cnfr_others
ROM_END

ROM_START( sc4cnfrh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401735.lo", 0x00001, 0x080000, CRC(faa4b396) SHA1(e00367e61ee6681a571e5dd383b4fbd63ef7d15b) )
	ROM_LOAD16_BYTE( "95401736.hi", 0x00000, 0x080000, CRC(3cba5770) SHA1(e3c41e55f3f6dcb6eb545be6ab5a7558d3208e26) )
	sc_cnfr_others
ROM_END

ROM_START( sc4cnfri )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401737.lo", 0x00001, 0x080000, CRC(2cee9c3c) SHA1(439af1908be3b1fea2c8c65d07ff274a304b22de) )
	ROM_LOAD16_BYTE( "95401738.hi", 0x00000, 0x080000, CRC(b74dac46) SHA1(1e7338e59c49902321a8678e63f7df3ac5a585ac) )
	sc_cnfr_others
ROM_END




ROM_START( sc4cad )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410168.odd", 0x00001, 0x080000, CRC(13d2e302) SHA1(64df52857e517bfe8386004ad16b89e1aa2e518c) )
	ROM_LOAD16_BYTE( "95410169.evn", 0x00000, 0x080000, CRC(090d05fc) SHA1(aeea708199ba4f50156550de6d2b4a08ff468b9d) )
	sc_cad_others
ROM_END

ROM_START( sc4cada )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410170.odd", 0x00001, 0x080000, CRC(43436500) SHA1(f3ac4b2d862083691dc068ba3d66dd8ce9dfa10a) )
	ROM_LOAD16_BYTE( "95410171.evn", 0x00000, 0x080000, CRC(792b539e) SHA1(446a7d693ac79774be8b7da35a1113cf28cc4aac) )
	sc_cad_others
ROM_END

ROM_START( sc4cadb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410776.odd", 0x00001, 0x080000, CRC(f03c0ced) SHA1(8d2397b7cbb711c2a2555441393111cc874b7dfc) )
	ROM_LOAD16_BYTE( "95410777.evn", 0x00000, 0x080000, CRC(63a89373) SHA1(5f1f0bad94ce070b10ec4e95445ba27762ed90bb) )
	sc_cad_others
ROM_END

ROM_START( sc4cadc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410778.odd", 0x00001, 0x080000, CRC(a19ac822) SHA1(9edc5dbdf414ccab400137504a47dd2b700bc59f) )
	ROM_LOAD16_BYTE( "95410779.evn", 0x00000, 0x080000, CRC(830f7598) SHA1(16d2c17e76b89c4634caa3f0559286a1b427ba7e) )
	sc_cad_others
ROM_END

ROM_START( sc4cadd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410814.odd", 0x00001, 0x080000, CRC(0c69919d) SHA1(7ea020fb40ddbef11ddfeb05d117555c25e77e68) )
	ROM_LOAD16_BYTE( "95410815.evn", 0x00000, 0x080000, CRC(05d1300b) SHA1(8e7ade1c9765ffcd05879bc4dcc9dc02275a5fd2) )
	sc_cad_others
ROM_END

ROM_START( sc4cade )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410816.odd", 0x00001, 0x080000, CRC(2f1bdc0e) SHA1(19f4348801b6882b1154b784496257c79a9bfe60) )
	ROM_LOAD16_BYTE( "95410817.evn", 0x00000, 0x080000, CRC(6690813e) SHA1(40983fc3dd512204d2f5b6f5031de4e852bdb50e) )
	sc_cad_others
ROM_END

ROM_START( sc4cadf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411168.odd", 0x00001, 0x080000, CRC(cc599227) SHA1(7e70b3227cc623bd4279b683ea6c2183dd441f70) )
	ROM_LOAD16_BYTE( "95411169.evn", 0x00000, 0x080000, CRC(ba9520e4) SHA1(6748e2b941c8fd78d5f10e83bc539bdb6107c338) )
	sc_cad_others
ROM_END

ROM_START( sc4cadg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411170.odd", 0x00001, 0x080000, CRC(9cc81425) SHA1(4db9ff92950ad17011b70d1eac1c12632b4e9a2c) )
	ROM_LOAD16_BYTE( "95411171.evn", 0x00000, 0x080000, CRC(cab37686) SHA1(51cfd291e9c1254ba3ab10f56e03d23db436232b) )
	sc_cad_others
ROM_END

ROM_START( sc4cadh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411776.odd", 0x00001, 0x080000, CRC(f973f4c7) SHA1(5e24eb12d8acc4c236c280cc9d45e31e0a73589b) )
	ROM_LOAD16_BYTE( "95411777.evn", 0x00000, 0x080000, CRC(becd988b) SHA1(547e3f982e9c2d392d7254521c2fa3a1682f19ce) )
	sc_cad_others
ROM_END

ROM_START( sc4cadi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411778.odd", 0x00001, 0x080000, CRC(6d99f855) SHA1(63236cefbdd3b61c9781c5b1b6a374602ae2dddf) )
	ROM_LOAD16_BYTE( "95411779.evn", 0x00000, 0x080000, CRC(67bc8715) SHA1(b16c478a2d965187818aab4ff948e37ef4baeb3d) )
	sc_cad_others
ROM_END

ROM_START( sc4cadj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411814.odd", 0x00001, 0x080000, CRC(4a4c4853) SHA1(8f2d66a1632baed1f9966169f69409b6b7277e2e) )
	ROM_LOAD16_BYTE( "95411815.evn", 0x00000, 0x080000, CRC(7e83b8e0) SHA1(f8c980b9131d56ae2359ff0beca3305e09ec983f) )
	sc_cad_others
ROM_END

ROM_START( sc4cadk )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411816.odd", 0x00001, 0x080000, CRC(7fc7f90f) SHA1(91e888c28b27d8de42dd1179f3b656f139f87458) )
	ROM_LOAD16_BYTE( "95411817.evn", 0x00000, 0x080000, CRC(5e78d43b) SHA1(8b9042dd49a55d81d8ec6769d3b7546d2c09723c) )
	sc_cad_others
ROM_END

ROM_START( sc4cadl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414570.odd", 0x00001, 0x080000, CRC(ffab5ee7) SHA1(942ea0f8602b4c2a6adfa69b39756dca86293157) )
	ROM_LOAD16_BYTE( "95414571.evn", 0x00000, 0x080000, CRC(96da0b9a) SHA1(0f4e4977053aefb82fcccc2cf5de594c56bb2a05) )
	sc_cad_others
ROM_END

ROM_START( sc4cadm )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415570.odd", 0x00001, 0x080000, CRC(83724d5a) SHA1(fd4d4db9c09b09e36d8e4774be354dad13025678) )
	ROM_LOAD16_BYTE( "95415571.evn", 0x00000, 0x080000, CRC(98a298bb) SHA1(0506a35a99ca3fe66c4c8204d7cae5ab1be72892) )
	sc_cad_others
ROM_END

ROM_START( sc4cadn )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416426.odd", 0x00001, 0x080000, CRC(b31a4f3c) SHA1(9b61e76235a04cab9309f29cc870e32a071d7952) )
	ROM_LOAD16_BYTE( "95416427.evn", 0x00000, 0x080000, CRC(dc724dc7) SHA1(6a27c8a0872a98943f1db339daced3cc1d6a0da4) )
	sc_cad_others
ROM_END

ROM_START( sc4cado )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416428.odd", 0x00001, 0x080000, CRC(d744e04c) SHA1(26ef7f7b65be3e342ca97143c8a86cafb06d2345) )
	ROM_LOAD16_BYTE( "95416429.evn", 0x00000, 0x080000, CRC(7f46c423) SHA1(29769d7b5f03296d1f78119af8836ef82f8c1952) )
	sc_cad_others
ROM_END

ROM_START( sc4cadp )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417426.odd", 0x00001, 0x080000, CRC(7d3e3c74) SHA1(e3d873b7777d4682ad680057358a08bda4e54e87) )
	ROM_LOAD16_BYTE( "95417427.evn", 0x00000, 0x080000, CRC(2f1341a1) SHA1(bb4cc3c2b2897cc46929089a5e971ebd967d0f46) )
	sc_cad_others
ROM_END

ROM_START( sc4cadq )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417428.odd", 0x00001, 0x080000, CRC(8e691238) SHA1(76f0df94549e56f379797169a90f80a10e9d7eef) )
	ROM_LOAD16_BYTE( "95417429.evn", 0x00000, 0x080000, CRC(6935c080) SHA1(50ebf1a213525c8c84ce765a203601cbc0f2b36a) )
	sc_cad_others
ROM_END




ROM_START( sc4cadcl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "91414812.odd", 0x00001, 0x080000, CRC(363aaaf1) SHA1(c6b38aaea1a992a0bda366d92e5fa8a42ff1a7c1) )
	ROM_LOAD16_BYTE( "91414813.evn", 0x00000, 0x080000, CRC(64c2ad49) SHA1(d13b028db6697d6e8a4cb94900b1b770cebe0c72) )
	sc_cadcl_others
ROM_END

ROM_START( sc4cadcla )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "91415812.odd", 0x00001, 0x080000, CRC(5ea10616) SHA1(171eaec9aeef53767085d3ed3d26b411fe37ea87) )
	ROM_LOAD16_BYTE( "91415813.evn", 0x00000, 0x080000, CRC(fa822d67) SHA1(4499ccfaaacae5afefb2d052168e92166480fe54) )
	sc_cadcl_others
ROM_END




ROM_START( sc4cbaz )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408056.lo", 0x00001, 0x080000, CRC(61e1048c) SHA1(25cc77109c1d4f16439affad74924551cdf2ac2d) )
	ROM_LOAD16_BYTE( "95408057.hi", 0x00000, 0x080000, CRC(35ebaa56) SHA1(cda4deb9e5131f0063c04ae88a5ddfdd6a2a2bdd) )
	sc_cbaz_others
ROM_END

ROM_START( sc4cbaza )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408198.lo", 0x00001, 0x080000, CRC(c1b6f768) SHA1(34a8519c6d1e40699a929170c3f438d6ab692fd8) )
	ROM_LOAD16_BYTE( "95408199.hi", 0x00000, 0x080000, CRC(6f36c9f7) SHA1(0637186e549d7031a14b548bdfe081af3024ed12) )
	sc_cbaz_others
ROM_END

ROM_START( sc4cbazb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408222.lo", 0x00001, 0x080000, CRC(a7d61704) SHA1(b3f1a39e735f88af10d61391902f50297d38898e) )
	ROM_LOAD16_BYTE( "95408223.hi", 0x00000, 0x080000, CRC(e242e8f8) SHA1(add3d0d46ad1b4e4cd35e1c41fa2a343017abec1) )
	sc_cbaz_others
ROM_END

ROM_START( sc4cbazc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408234.lo", 0x00001, 0x080000, CRC(baf0e2c9) SHA1(852e56a5673ec8161699df93349bc2387b0bd625) )
	ROM_LOAD16_BYTE( "95408235.hi", 0x00000, 0x080000, CRC(d0330878) SHA1(823f98e4474f5d105126dec7f657d27281d21611) )
	sc_cbaz_others
ROM_END

ROM_START( sc4cbazd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408378.lo", 0x00001, 0x080000, CRC(0cc4756e) SHA1(87c15ef0b718438cfc68f3c29aed5e6ea67c0442) )
	ROM_LOAD16_BYTE( "95408379.hi", 0x00000, 0x080000, CRC(393d7cd2) SHA1(d20d6c14d9ee52da0469b31156e0ff9552416a58) )
	sc_cbaz_others
ROM_END

ROM_START( sc4cbaze )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409056.lo", 0x00001, 0x080000, CRC(0dae003e) SHA1(e837bc4698f5d65b4864d49e77a5a90efd8f89a0) )
	ROM_LOAD16_BYTE( "95409057.hi", 0x00000, 0x080000, CRC(91a171a5) SHA1(c86e530c6330d54a79e5b60875670cfea9f789e6) )
	sc_cbaz_others
ROM_END

ROM_START( sc4cbazf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409198.lo", 0x00001, 0x080000, CRC(547ac730) SHA1(7062f6096369eef21d342a7c35cda0fd617cac5e) )
	ROM_LOAD16_BYTE( "95409199.hi", 0x00000, 0x080000, CRC(9c57c591) SHA1(c0c392fffd68f09fb8cdc9b184a78e7868745413) )
	sc_cbaz_others
ROM_END

ROM_START( sc4cbazg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409222.lo", 0x00001, 0x080000, CRC(b2c70c27) SHA1(d3d128d6d2fbc7cc7505b9b50064984c8f008613) )
	ROM_LOAD16_BYTE( "95409223.hi", 0x00000, 0x080000, CRC(ce602fef) SHA1(9e81f69099bc26cbd773b3e66577bd8a47528a82) )
	sc_cbaz_others
ROM_END

ROM_START( sc4cbazh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409234.lo", 0x00001, 0x080000, CRC(426e5614) SHA1(e0b44aae40eadf1df88179f7070e245609b505c1) )
	ROM_LOAD16_BYTE( "95409235.hi", 0x00000, 0x080000, CRC(176abe31) SHA1(a2ee1d4eef81aff4884dadfdda16dc1c5cd2b67f) )
	sc_cbaz_others
ROM_END

ROM_START( sc4cbazi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409378.lo", 0x00001, 0x080000, CRC(8121f89a) SHA1(04a5201fac956388a788a7828cb6eadfb8a51d40) )
	ROM_LOAD16_BYTE( "95409379.hi", 0x00000, 0x080000, CRC(6f200e30) SHA1(785eb177ae9f952b7e4ee90cb73a51ae0fee3a5e) )
	sc_cbaz_others
ROM_END

ROM_START( sc4cbazj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412192.lo", 0x00001, 0x080000, CRC(73c648d9) SHA1(e3dfe8cdeb6ddf2e1e85b73a49ec9cac0377c688) )
	ROM_LOAD16_BYTE( "95412193.hi", 0x00000, 0x080000, CRC(236b73e7) SHA1(2a8abd9393af205f415e94e276bfc7c363175a38) )
	sc_cbaz_others
ROM_END

ROM_START( sc4cbazk )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413192.lo", 0x00001, 0x080000, CRC(c11e8bd2) SHA1(a5006bd5c4c18ca41611e852a68f2696aa3f3fc8) )
	ROM_LOAD16_BYTE( "95413193.hi", 0x00000, 0x080000, CRC(e73b43ed) SHA1(def6d82f982965feef694802ea75f158e245810b) )
	sc_cbaz_others
ROM_END






ROM_START( sc4cconx )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400903.lo", 0x00001, 0x080000, CRC(9662c017) SHA1(b4c1d85edd89b71dcea277eb09c001ef6272e511) )
	ROM_LOAD16_BYTE( "95400904.hi", 0x00000, 0x080000, CRC(af0dbca4) SHA1(da929472e75338a499a65ee02beb037ded29b418) )
	sc_cconx_others
ROM_END

ROM_START( sc4cconxa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401901.lo", 0x00001, 0x080000, CRC(a52c16a1) SHA1(f7a0a9c9e02dc104634e58f3d560a3b2a4da9435) )
	ROM_LOAD16_BYTE( "95401902.hi", 0x00000, 0x080000, CRC(f51572ef) SHA1(7e0f9179acb7c47909820ba8a9e9efd7eb63068d) )
	sc_cconx_others
ROM_END

ROM_START( sc4cconxb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402089.lo", 0x00001, 0x080000, CRC(276f1b20) SHA1(00d548b122ec07f0b2f8d2790a61c92445d24dfe) )
	ROM_LOAD16_BYTE( "95402090.hi", 0x00000, 0x080000, CRC(e7248393) SHA1(5d4869044a768d28971d05458ea28eda44ec77b4) )
	sc_cconx_others
ROM_END

ROM_START( sc4cconxc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403089.lo", 0x00001, 0x080000, CRC(b1be7740) SHA1(ca316700aead1596f31447091b0a82befbf4d154) )
	ROM_LOAD16_BYTE( "95403090.hi", 0x00000, 0x080000, CRC(85dc66b2) SHA1(5a10a41379e49a4cb1611a7f7aad53804c581ad1) )
	sc_cconx_others
ROM_END

ROM_START( sc4cconxd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403399.lo", 0x00001, 0x080000, CRC(f403ffc5) SHA1(89d56bd23e9667c366974bfbb4cb5af1f8d846ff) )
	ROM_LOAD16_BYTE( "95403400.hi", 0x00000, 0x080000, CRC(0f7535e2) SHA1(41877b1f95c877ca94d3644a0d86803aba0831f6) )
	sc_cconx_others
ROM_END




ROM_START( sc4ccrus )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402812.lo", 0x00001, 0x080000, CRC(795a4bef) SHA1(4398773ecacfb66cf7f7b3ae4ffab8723a049df3) )
	ROM_LOAD16_BYTE( "95402813.hi", 0x00000, 0x080000, CRC(e79f42ba) SHA1(ff365e456183f7bef3b64249eb3b497e1b35845f) )
	sc_ccrus_others
ROM_END

ROM_START( sc4ccrusa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402880.lo", 0x00001, 0x080000, CRC(839c70fa) SHA1(fb7d483c816dd73c11a75438302e276dfe4e504a) )
	ROM_LOAD16_BYTE( "95402881.hi", 0x00000, 0x080000, CRC(715708f7) SHA1(4f2d44bdb4fe4db7ca51a2727c68af25d786fbd4) )
	sc_ccrus_others
ROM_END

ROM_START( sc4ccrusb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403812.lo", 0x00001, 0x080000, CRC(33182ed2) SHA1(0c8582bd578876a6fb093c7b712dfc0030a7ec70) )
	ROM_LOAD16_BYTE( "95403813.hi", 0x00000, 0x080000, CRC(1bb8718d) SHA1(abe1a125c205e745778be928be59ccf3d2b1b698) )
	sc_ccrus_others
ROM_END




ROM_START( sc4cerup )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405244.lo", 0x00001, 0x080000, CRC(18aa9826) SHA1(d9d3abb9ab92a0b0d9da9107ebae2157c41ba7e6) )
	ROM_LOAD16_BYTE( "95405245.hi", 0x00000, 0x080000, CRC(07482460) SHA1(8d6237bc8d51087bc4ff487c6a5e0db05c742a47) )
	sc_cerup_others
ROM_END




ROM_START( sc4cexpl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406596.lo", 0x00001, 0x080000, CRC(08d50241) SHA1(e0818ace6bf77c377d67f2e32b257e34b0c7bcc0) )
	ROM_LOAD16_BYTE( "95406597.hi", 0x00000, 0x080000, CRC(305266ae) SHA1(df3d99955e480e3d75b0fbaae65c6a780e7db46c) )
	sc_cexpl_others
ROM_END

ROM_START( sc4cexpla )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406928.lo", 0x00001, 0x080000, CRC(4930390e) SHA1(9f2a5a65a1e67b51756a602b22c2fbf59e4ecffa) )
	ROM_LOAD16_BYTE( "95406929.hi", 0x00000, 0x080000, CRC(05201df8) SHA1(eadc72a23ccc4f5d690099830444471d1911d0dc) )
	sc_cexpl_others
ROM_END

ROM_START( sc4cexplb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407596.lo", 0x00001, 0x080000, CRC(04c26bd9) SHA1(993f2ce15b979bce9099a78f49bf7c6fb7171f0a) )
	ROM_LOAD16_BYTE( "95407597.hi", 0x00000, 0x080000, CRC(afb31cc8) SHA1(dcef2320a3fa617a84082b48101142610c6985bc) )
	sc_cexpl_others
ROM_END

ROM_START( sc4cexplc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407928.lo", 0x00001, 0x080000, CRC(94c39dfc) SHA1(004acfef0af5153d1fc453d9c1d8a03fd773efa8) )
	ROM_LOAD16_BYTE( "95407929.hi", 0x00000, 0x080000, CRC(72e808ea) SHA1(780d5c2a5d8a8211d0a8dada4c1288a4ad003207) )
	sc_cexpl_others
ROM_END

ROM_START( sc4cexpld )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408354.lo", 0x00001, 0x080000, CRC(7cb241df) SHA1(2e3018a7eb3b8573fcd0acb6c194bc197d46e87f) )
	ROM_LOAD16_BYTE( "95408355.hi", 0x00000, 0x080000, CRC(cd643bee) SHA1(7f22e4c2bf56b400ee56c15b93ed12e6a4a3649a) )
	sc_cexpl_others
ROM_END

ROM_START( sc4cexple )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409354.lo", 0x00001, 0x080000, CRC(51abe89e) SHA1(830b809dd432465f0bc769aed2cedaaf07f94973) )
	ROM_LOAD16_BYTE( "95409355.hi", 0x00000, 0x080000, CRC(f7f1360b) SHA1(76705397348f1c9167cc6e208f08344dd737e627) )
	sc_cexpl_others
ROM_END

ROM_START( sc4cexplf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414634.lo", 0x00001, 0x080000, CRC(fabc6e58) SHA1(03f78ae84ec52b08d255c2ee23165250078ffb52) )
	ROM_LOAD16_BYTE( "95414635.hi", 0x00000, 0x080000, CRC(e7e54948) SHA1(5934c0d95d6a4ddac620f2e95e43234686148e5e) )
	sc_cexpl_others
ROM_END

ROM_START( sc4cexplg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414636.lo", 0x00001, 0x080000, CRC(2b65f0e2) SHA1(ff730568e1eafd4f3d9ef6d052f003c64e2d4477) )
	ROM_LOAD16_BYTE( "95414637.hi", 0x00000, 0x080000, CRC(7925bd41) SHA1(be47858f8417b023d0b428d1112126d79423f9e9) )
	sc_cexpl_others
ROM_END




ROM_START( sc4chand )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402335.lo", 0x00001, 0x080000, CRC(0da0141b) SHA1(f95ff70b42d4754bb0d4120d8eac4b2283d6acd5) )
	ROM_LOAD16_BYTE( "95402336.hi", 0x00000, 0x080000, CRC(4181da5f) SHA1(370c44aaf628248574c277e56207cc0e879403c8) )
	sc_chand_others
ROM_END

ROM_START( sc4chanda )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402337.lo", 0x00001, 0x080000, CRC(199db9a2) SHA1(cae5930ca6b28f7088fa04d1eae35312e787f995) )
	ROM_LOAD16_BYTE( "95402338.hi", 0x00000, 0x080000, CRC(f07cfd86) SHA1(47247efc279c162465cc03d6ae37828bd4e22cfe) )
	sc_chand_others
ROM_END

ROM_START( sc4chandb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403335.lo", 0x00001, 0x080000, CRC(879108e2) SHA1(1c1c497f60a8135c030e5ff3e7051746fbacb0aa) )
	ROM_LOAD16_BYTE( "95403336.hi", 0x00000, 0x080000, CRC(84ef4625) SHA1(801cd63e9bd09c6fedee00e9747d107a51a344f4) )
	sc_chand_others
ROM_END

ROM_START( sc4chandc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403337.lo", 0x00001, 0x080000, CRC(93aca55b) SHA1(b197e6256a478b3ce68ca11d95865067d8109ef0) )
	ROM_LOAD16_BYTE( "95403338.hi", 0x00000, 0x080000, CRC(351261fc) SHA1(9ddb8ed7f595e623a34c8dd1678ecd71f79ce6f5) )
	sc_chand_others
ROM_END




ROM_START( sc4cinv )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400227.lo", 0x00001, 0x080000, CRC(24b47f39) SHA1(891dda304ca8e2cbefc823fcfa07d00890b697e3) )
	ROM_LOAD16_BYTE( "95400228.hi", 0x00000, 0x080000, CRC(210e70b3) SHA1(0d5ab09346b66f716c64df19d9ed4d303f324bc4) )
	sc_cinv_others
ROM_END

ROM_START( sc4cinva )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400229.lo", 0x00001, 0x080000, CRC(8bf8798c) SHA1(cfe3026721dec33287f43dbc2857e6a59ad343ca) )
	ROM_LOAD16_BYTE( "95400230.hi", 0x00000, 0x080000, CRC(82d10f41) SHA1(bb7b6c8e9253def4790cb64d6e521f1c8e01bb64) )
	sc_cinv_others
ROM_END

ROM_START( sc4cinvb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400231.lo", 0x00001, 0x080000, CRC(0f0edf40) SHA1(cbb9ecf2b96d2ee5c8124547dd41cab8d464f5c7) )
	ROM_LOAD16_BYTE( "95400232.hi", 0x00000, 0x080000, CRC(92785b93) SHA1(63c6b5de09b3d9f4d883101e554b6101ed25d828) )
	sc_cinv_others
ROM_END

ROM_START( sc4cinvc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400743.lo", 0x00001, 0x080000, CRC(0475a59e) SHA1(db562dda6831398a30673bdf97de143478eb5fa3) )
	ROM_LOAD16_BYTE( "95400744.hi", 0x00000, 0x080000, CRC(ed638e2b) SHA1(d7084f1056836e7cfd58624f272cd4b457204899) )
	sc_cinv_others
ROM_END

ROM_START( sc4cinvd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400745.lo", 0x00001, 0x080000, CRC(3d71c227) SHA1(4ae11541bf9eccc9ec79f4af6395e30ba025dc24) )
	ROM_LOAD16_BYTE( "95400746.hi", 0x00000, 0x080000, CRC(58a98129) SHA1(88412cc3c4cc9d9cbf2765da3ffeb110f29624ea) )
	sc_cinv_others
ROM_END

ROM_START( sc4cinve )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401227.lo", 0x00001, 0x080000, CRC(43666c44) SHA1(b63015980e266e64d24f9fadfd16b112e738b721) )
	ROM_LOAD16_BYTE( "95401228.hi", 0x00000, 0x080000, CRC(591f0cc8) SHA1(12124045a60a0a0830a7b5662797aaa2fc7b906e) )
	sc_cinv_others
ROM_END

ROM_START( sc4cinvf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401229.lo", 0x00001, 0x080000, CRC(0f5317fe) SHA1(46171a27d7cdac2dda2c2ce7d013d8bc0c1f63e6) )
	ROM_LOAD16_BYTE( "95401230.hi", 0x00000, 0x080000, CRC(109444ae) SHA1(27274218e017f0da2ccfd25503e1bffe39266efe) )
	sc_cinv_others
ROM_END

ROM_START( sc4cinvg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401231.lo", 0x00001, 0x080000, CRC(6f4afe14) SHA1(58d19428dfbb14eb17d3f89113744eedf150df21) )
	ROM_LOAD16_BYTE( "95401232.hi", 0x00000, 0x080000, CRC(bddda645) SHA1(8d1cc6b93f5f6b780882968fe82b0f86e11b9c88) )
	sc_cinv_others
ROM_END

ROM_START( sc4cinvh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401743.lo", 0x00001, 0x080000, CRC(73137ddb) SHA1(f3db49badb3786160f8fb03b774e6d577644f29b) )
	ROM_LOAD16_BYTE( "95401744.hi", 0x00000, 0x080000, CRC(e225b17a) SHA1(c9979b65780f87e1409b656144732d0d37372250) )
	sc_cinv_others
ROM_END

ROM_START( sc4cinvi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401745.lo", 0x00001, 0x080000, CRC(274eca45) SHA1(f4f24c4d072d8a4a796820de26ee16507497c13e) )
	ROM_LOAD16_BYTE( "95401746.hi", 0x00000, 0x080000, CRC(fdddf6aa) SHA1(6d32a140df2272361fd8fca3e70030b9cd38f904) )
	sc_cinv_others
ROM_END




ROM_START( sc4clash )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410494.lo", 0x00001, 0x080000, CRC(328acbeb) SHA1(dbb05173a2af6a2d2cc73b6a99f0693a8bd6d75b) )
	ROM_LOAD16_BYTE( "95410495.hi", 0x00000, 0x080000, CRC(a6681b69) SHA1(410b003485e3a89c62456f4679fd4b0468e0634c) )
	sc_clash_others
ROM_END

ROM_START( sc4clasha )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410496.lo", 0x00001, 0x080000, CRC(99ef95a6) SHA1(2c2353cb8c5805510b67935f47366430d2c1762a) )
	ROM_LOAD16_BYTE( "95410497.hi", 0x00000, 0x080000, CRC(d9d37a25) SHA1(0603761dca1ae8967c03bee0427abfb9c1d6f65c) )
	sc_clash_others
ROM_END

ROM_START( sc4clashb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411494.lo", 0x00001, 0x080000, CRC(ce512069) SHA1(d54f67c01956884bcd19cd652b055ea8109983fb) )
	ROM_LOAD16_BYTE( "95411495.hi", 0x00000, 0x080000, CRC(2e5b9fe4) SHA1(a8e71d6c7e49d318ae2c02dd23fb26b52ccea69c) )
	sc_clash_others
ROM_END

ROM_START( sc4clashc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411496.lo", 0x00001, 0x080000, CRC(5a8582c3) SHA1(621955185bbc8793f585943e782c03d6f8697919) )
	ROM_LOAD16_BYTE( "95411497.hi", 0x00000, 0x080000, CRC(d6954574) SHA1(a04c349f87157fe8939e210c3875cebf210b2985) )
	sc_clash_others
ROM_END

ROM_START( sc4clashd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416264.lo", 0x00001, 0x080000, CRC(7cb7f066) SHA1(c9e9b18a6ecf5933da32532053c3d0c72dbe1d50) )
	ROM_LOAD16_BYTE( "95416265.hi", 0x00000, 0x080000, CRC(0573d517) SHA1(5ad653f61c24896af1c5d5d62989f12bfb181478) )
	sc_clash_others
ROM_END

ROM_START( sc4clashe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416266.lo", 0x00001, 0x080000, CRC(807ce8e5) SHA1(b06e21a063cad4f218df9467da0252a239395683) )
	ROM_LOAD16_BYTE( "95416267.hi", 0x00000, 0x080000, CRC(bafe7eae) SHA1(adf083449690a60553b08ffc5f8898f998c2619c) )
	sc_clash_others
ROM_END

ROM_START( sc4clashf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417264.lo", 0x00001, 0x080000, CRC(fd362d07) SHA1(7257f8597c2b828f8720c35a6449e91010ea8db1) )
	ROM_LOAD16_BYTE( "95417265.hi", 0x00000, 0x080000, CRC(fb29bec0) SHA1(72adbcae927d8394822cccdc975258b12cc96c0c) )
	sc_clash_others
ROM_END

ROM_START( sc4clashg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417266.lo", 0x00001, 0x080000, CRC(029a2e04) SHA1(d213229e50922fbe15011dbad243b6a5bfe7a708) )
	ROM_LOAD16_BYTE( "95417267.hi", 0x00000, 0x080000, CRC(806b734b) SHA1(dcf56ce6e02d409d101726d86fbcdc47a98721ea) )
	sc_clash_others
ROM_END




ROM_START( sc4cr )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404876.odd", 0x00001, 0x080000, CRC(ffd5323a) SHA1(745e90c3fe28de7d361b518cfc5c93bdf7e8bf40) )
	ROM_LOAD16_BYTE( "95404877.evn", 0x00000, 0x080000, CRC(07c84685) SHA1(f0c043c08f605c56a4b448b97de347240794bd72) )
	sc_cr_others
ROM_END

ROM_START( sc4cra )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406648.odd", 0x00001, 0x080000, CRC(e39cee3b) SHA1(e6891488b6c2aba6b844e16525fa206db7c37e81) )
	ROM_LOAD16_BYTE( "95406649.evn", 0x00000, 0x080000, CRC(3ffc1b08) SHA1(54c6cee809bf9a68060261cef9fe0ab58c42fba8) )
	sc_cr_others
ROM_END

ROM_START( sc4crb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407648.odd", 0x00001, 0x080000, CRC(c3c095ee) SHA1(12ce5a56ec3f8c63ce792b38082aa12bd670c9ab) )
	ROM_LOAD16_BYTE( "95407649.evn", 0x00000, 0x080000, CRC(537c6d08) SHA1(a27b75da63203fd9f21e393c9e8a100f915a6bf5) )
	sc_cr_others
ROM_END

ROM_START( sc4crc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408030.odd", 0x00001, 0x080000, CRC(9c8dd79e) SHA1(82d5631e2d5e1c2d98ea2da080e2f7a3d167a819) )
	ROM_LOAD16_BYTE( "95408031.evn", 0x00000, 0x080000, CRC(e5b3d478) SHA1(ba34f9df49ff1363818bc7c72558860cc54e4b61) )
	sc_cr_others
ROM_END

ROM_START( sc4crd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408308.odd", 0x00001, 0x080000, CRC(cc7c041e) SHA1(1a95daf1cb53d3f2b9d1e26be6b50e7becc160e8) )
	ROM_LOAD16_BYTE( "95408309.evn", 0x00000, 0x080000, CRC(c4948c63) SHA1(fa451d85e384175b515fea97b2b34ae53eaebb33) )
	sc_cr_others
ROM_END

ROM_START( sc4cre )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409030.odd", 0x00001, 0x080000, CRC(b6104594) SHA1(1f3a31dff9c7f4fb1d9c9c4669bf8cc1ba4081ae) )
	ROM_LOAD16_BYTE( "95409031.evn", 0x00000, 0x080000, CRC(7769c9af) SHA1(42ee3780e444308ad38de026dee08bf1c31c77f7) )
	sc_cr_others
ROM_END

ROM_START( sc4crf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409308.odd", 0x00001, 0x080000, CRC(a5c41dc2) SHA1(4d63fa341ababca34bad8a8c6a68d4d007ab8d02) )
	ROM_LOAD16_BYTE( "95409309.evn", 0x00000, 0x080000, CRC(1ab2730b) SHA1(ea01f41580dfefc9e011e143a43da334ed4121bc) )
	sc_cr_others
ROM_END



ROM_START( sc4crcl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408194.odd", 0x00001, 0x080000, CRC(d45a4fd2) SHA1(f4bbe26a6ffa7ad13a3ad2402b7b8e051d86bb35) )
	ROM_LOAD16_BYTE( "95408195.evn", 0x00000, 0x080000, CRC(ce24bdc2) SHA1(f9e135772fc319d07e8973da0473cc0799771a76) )
	sc_crcl_others
ROM_END

ROM_START( sc4crcla )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408422.odd", 0x00001, 0x080000, CRC(4fba1f0f) SHA1(76f8e7fe71d8bcf60c7ac983a955c68066d57a53) )
	ROM_LOAD16_BYTE( "95408423.evn", 0x00000, 0x080000, CRC(fba4d637) SHA1(67c655295317fc9c98f3ec34f0d6a36fb0721591) )
	sc_crcl_others
ROM_END

ROM_START( sc4crclb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409194.odd", 0x00001, 0x080000, CRC(28d5b378) SHA1(29404c7a85f1a2c1958c574222128b8cd45ed2df) )
	ROM_LOAD16_BYTE( "95409195.evn", 0x00000, 0x080000, CRC(7dbc98da) SHA1(cd9cf1185cb1f96d583d0f004a7055a6268e3f37) )
	sc_crcl_others
ROM_END

ROM_START( sc4crclc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409422.odd", 0x00001, 0x080000, CRC(17e06e6e) SHA1(c53ef8b7d3a62dda9d101f86b52d032c232971b1) )
	ROM_LOAD16_BYTE( "95409423.evn", 0x00000, 0x080000, CRC(cdab462b) SHA1(f1b597d53b888e42651df9647875a21af8dad1c0) )
	sc_crcl_others
ROM_END

ROM_START( sc4crcld )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410108.odd", 0x00001, 0x080000, CRC(e3d622b0) SHA1(57ca62cd45513fd184f82805051f7219f1aba8fa) )
	ROM_LOAD16_BYTE( "95410109.evn", 0x00000, 0x080000, CRC(7551b505) SHA1(a94c8bccd2808a774981ffc6d470979aba6d9c9f) )
	sc_crcl_others
ROM_END

ROM_START( sc4crcle )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410918.odd", 0x00001, 0x080000, CRC(bae2d17d) SHA1(7809c92c8b5972d55968c2b774b4fe04e1517d75) )
	ROM_LOAD16_BYTE( "95410919.evn", 0x00000, 0x080000, CRC(043a76d0) SHA1(65c08b934c087f08bee373991aa3ac5bcbd96698) )
	sc_crcl_others
ROM_END

ROM_START( sc4crclf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411918.odd", 0x00001, 0x080000, CRC(7bb37fa4) SHA1(1e7ea0639685bfb24547b4dab61a4a961629acf1) )
	ROM_LOAD16_BYTE( "95411919.evn", 0x00000, 0x080000, CRC(bb38ce31) SHA1(13a97f203cb7887b2ea6b379acb3f3df732b1fcf) )
	sc_crcl_others
ROM_END



ROM_START( sc4cashn )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406482.lo", 0x00001, 0x080000, CRC(9cadf511) SHA1(27b02ebea7afb6753df0e2c6486a61da7ffea99b) )
	ROM_LOAD16_BYTE( "95406483.hi", 0x00000, 0x080000, CRC(0993309a) SHA1(15a24d9f618b6f68d983e07896eda45f865661a3) )
	sc_cashn_others
ROM_END




ROM_START( sc4cashm )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404154.lo", 0x00001, 0x080000, CRC(0bcf30c0) SHA1(7c0b0f0812899f9de09b4ba34da0374631f9eedf) )
	ROM_LOAD16_BYTE( "95404155.hi", 0x00000, 0x080000, CRC(a3421610) SHA1(fb6873ae6953d2565a22e6f3397c2bc1b01491da) )
	sc_cashm_others
ROM_END

ROM_START( sc4cashma )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404678.lo", 0x00001, 0x080000, CRC(509fe689) SHA1(f5574ed5a5bbf2c92b6f82f201088d22b826c025) )
	ROM_LOAD16_BYTE( "95404679.hi", 0x00000, 0x080000, CRC(c44e314e) SHA1(0327924b29a05c374a9ff043556e70443e7a0091) )
	sc_cashm_others
ROM_END

ROM_START( sc4cashmb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405154.lo", 0x00001, 0x080000, CRC(56462da7) SHA1(4edd989ff9b0f2941e47392b46b78ab130a7ea38) )
	ROM_LOAD16_BYTE( "95405155.hi", 0x00000, 0x080000, CRC(31caaab1) SHA1(1505c441e73fe1d0050b4fe5fbede0cd108b3a36) )
	sc_cashm_others
ROM_END

ROM_START( sc4cashmc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405678.lo", 0x00001, 0x080000, CRC(45727051) SHA1(6f5836e145cb83d40c445c4f54f02277c7852d9f) )
	ROM_LOAD16_BYTE( "95405679.hi", 0x00000, 0x080000, CRC(02fc0f9c) SHA1(6edf8bbe6d494253c3183bb7869aedbedf2d7b5c) )
	sc_cashm_others
ROM_END

ROM_START( sc4cashmd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406386.lo", 0x00001, 0x080000, CRC(f475bf63) SHA1(d5fb531fb778a8e35a83836dadc2ee7e33ba272b) )
	ROM_LOAD16_BYTE( "95406387.hi", 0x00000, 0x080000, CRC(9db583cb) SHA1(205496712f100e17e91d46849dd341deaa63aa0a) )
	sc_cashm_others
ROM_END

ROM_START( sc4cashme )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407386.lo", 0x00001, 0x080000, CRC(025b77b7) SHA1(793433c1616729a2ccf5489cad52b32a04f9fa15) )
	ROM_LOAD16_BYTE( "95407387.hi", 0x00000, 0x080000, CRC(4d74b9ba) SHA1(3897147aa15c7acb59cc1e502ba712cc944b92be) )
	sc_cashm_others
ROM_END

ROM_START( sc4cashg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408508.lo", 0x00001, 0x080000, CRC(6d96beb0) SHA1(da274c53fce937e01a095693003ecb67379a7980) )
	ROM_LOAD16_BYTE( "95408509.hi", 0x00000, 0x080000, CRC(084450b4) SHA1(553b2a10e36b0c1f26b0c05c4d2b218f4c5b24b1) )
	sc_cashm_german_others
ROM_END




ROM_START( sc4cckey )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410398.lo", 0x00001, 0x080000, CRC(96e93185) SHA1(eb647e8f524339a79aa3a919c9d47cb43162dd81) )
	ROM_LOAD16_BYTE( "95410399.hi", 0x00000, 0x080000, CRC(058e7aeb) SHA1(b6f61cd250d548586328097514be176c85366fa1) )
	sc_cckey_others
ROM_END

ROM_START( sc4cckeya )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410788.lo", 0x00001, 0x080000, CRC(8e9cdf3b) SHA1(9fb8ea12d9595d6b0e7d6cbbc4fb3f58f732e649) )
	ROM_LOAD16_BYTE( "95410789.hi", 0x00000, 0x080000, CRC(51c3ed51) SHA1(50426d5e1124bb77665350ad0a230ef26b678a49) )
	sc_cckey_others
ROM_END

ROM_START( sc4cckeyb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410790.lo", 0x00001, 0x080000, CRC(3d6767af) SHA1(85f5147e569a7ce9ef2395ed8ea21efae701c2ad) )
	ROM_LOAD16_BYTE( "95410791.hi", 0x00000, 0x080000, CRC(24667a0d) SHA1(e2ec4eea414ace0f41fa4cf074bdf7ad8f18d2ed) )
	sc_cckey_others
ROM_END

ROM_START( sc4cckeyc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410924.lo", 0x00001, 0x080000, CRC(ea76e2e6) SHA1(565ee7a0a6c310dab544348172e4b00aa49ceeca) )
	ROM_LOAD16_BYTE( "95410925.hi", 0x00000, 0x080000, CRC(74ae4798) SHA1(6278b79f050e116a04f302a065be869d24349533) )
	sc_cckey_others
ROM_END


ROM_START( sc4cckeyd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411398.lo", 0x00001, 0x080000, CRC(383bbb6f) SHA1(a099e95e90f0851bc9ce844a516ddf1228c1810d) )
	ROM_LOAD16_BYTE( "95411399.hi", 0x00000, 0x080000, CRC(3c58839e) SHA1(0338a9f17953bcf56815682d6c1fe59c2a29d2d6) )
	sc_cckey_others
ROM_END

ROM_START( sc4cckeye )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411788.lo", 0x00001, 0x080000, CRC(b4bb51fb) SHA1(dec9a98da24ebf8e943e39cec8ee7dfade159678) )
	ROM_LOAD16_BYTE( "95411789.hi", 0x00000, 0x080000, CRC(5c0bb85e) SHA1(2a0b73d91cfd6eb260b70e303e041ff4274c631e) )
	sc_cckey_others
ROM_END

ROM_START( sc4cckeyf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411790.lo", 0x00001, 0x080000, CRC(3c07b2ed) SHA1(eea25b20df32ea3bbddfde0e20d2a95b1ad48c06) )
	ROM_LOAD16_BYTE( "95411791.hi", 0x00000, 0x080000, CRC(0079353e) SHA1(f2b0b192c33d677fe477cb9167300e7d8160dc20) )
	sc_cckey_others
ROM_END

ROM_START( sc4cckeyg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411966.lo", 0x00001, 0x080000, CRC(e060fe6b) SHA1(8cbb79f92ad51b23972942cbc419e6c0c84b2ed5) )
	ROM_LOAD16_BYTE( "95411967.hi", 0x00000, 0x080000, CRC(7e139d6b) SHA1(4f7e410093be3da1fe4eda345f61f828b1f37d53) )
	sc_cckey_others
ROM_END

ROM_START( sc4cckeyh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412068.lo", 0x00001, 0x080000, CRC(55357b73) SHA1(d0caf69b096e2184b0bfacbff029cf7abad8d8c2) )
	ROM_LOAD16_BYTE( "95412069.hi", 0x00000, 0x080000, CRC(eb542bba) SHA1(59951b1f52f2306db94b81487d3e31562a2d8bd3) )
	sc_cckey_others
ROM_END

ROM_START( sc4cckeyi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412916.lo", 0x00001, 0x080000, CRC(f99ec6f6) SHA1(0e57c3643bb1d2e95c250af5f1855df48dd78638) )
	ROM_LOAD16_BYTE( "95412917.hi", 0x00000, 0x080000, CRC(0bdf8b04) SHA1(d7cbc1c263650702398e5fb71c98b1fbed475e0a) )
	sc_cckey_others
ROM_END

ROM_START( sc4cckeyj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413068.lo", 0x00001, 0x080000, CRC(011b03f2) SHA1(39990725843577592782bef3771c3e81a882735e) )
	ROM_LOAD16_BYTE( "95413069.hi", 0x00000, 0x080000, CRC(ebefcf57) SHA1(b0ad18af2d4a5a49b78bb7dbae3cd1700f6a50fd) )
	sc_cckey_others
ROM_END

ROM_START( sc4cckeyk )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413916.lo", 0x00001, 0x080000, CRC(8b41a755) SHA1(3791163f0cc1c17b094a8aed972a2c55004ef13c) )
	ROM_LOAD16_BYTE( "95413917.hi", 0x00000, 0x080000, CRC(e659878e) SHA1(b7dab901e9f1aadc033858b81ce0882bbe9d63ed) )
	sc_cckey_others
ROM_END

ROM_START( sc4cckeyl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95412918.lo", 0x00001, 0x080000, CRC(b268d275) SHA1(70b31512c0d8c8e0bf644fa1d5ea723c35c2a629) )
	ROM_LOAD16_BYTE( "95412919.hi", 0x00000, 0x080000, CRC(d96df890) SHA1(65f8d2b4840efd5c3fe65b00ecbf2df7ac81cbe6) )
	sc_cckey_others
ROM_END


ROM_START( sc4cckeym )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95412920.lo", 0x00001, 0x080000, CRC(e047005e) SHA1(b3de1d863cda1945e1207b3df51a8b5172e65bb8) )
	ROM_LOAD16_BYTE( "95412921.hi", 0x00000, 0x080000, CRC(7b903100) SHA1(1df4a5ded5cc195ee020f302870eb8bf4c8a14d3) )
	sc_cckey_others
ROM_END


ROM_START( sc4cckeyn )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95413918.lo", 0x00001, 0x080000, CRC(9062cdaa) SHA1(b8249ca6590aef5041315f1dbbdc019ffb5e8ac2) )
	ROM_LOAD16_BYTE( "95413919.hi", 0x00000, 0x080000, CRC(de714343) SHA1(95f605d91e43c3bcd2a2799ecf385f1e75cc5413) )
	sc_cckey_others
ROM_END

ROM_START( sc4cckeyo )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95413920.lo", 0x00001, 0x080000, CRC(deae622a) SHA1(7f8f7b93c734278d24f69fd4a491b3d4133d8b1d) )
	ROM_LOAD16_BYTE( "95413921.hi", 0x00000, 0x080000, CRC(92671c85) SHA1(4444e3a54393f8fb824eeebfed8f2cbaf07e83ea) )
	sc_cckey_others
ROM_END


ROM_START( sc4ckx )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420660.lo", 0x00001, 0x080000, CRC(2ab46dff) SHA1(362da805615cce7f5c191cb5cf98dd2362cecc10) )
	ROM_LOAD16_BYTE( "95420661.hi", 0x00000, 0x080000, CRC(a8fb772e) SHA1(90aea6a48117a84f17df501deea4416ffee104f1) )
	sc_ckx_others
ROM_END

ROM_START( sc4ckxa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420758.lo", 0x00001, 0x080000, CRC(0a0b0486) SHA1(6eb4ea9f1d6fd8d66b203bcff986ee2f57ed9d77) )
	ROM_LOAD16_BYTE( "95420759.hi", 0x00000, 0x080000, CRC(63637820) SHA1(0e78c2abb2e2ea839b6af68cb8d9dedbb16a1d3d) )
	sc_ckx_others
ROM_END

ROM_START( sc4ckxb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420760.lo", 0x00001, 0x080000, CRC(c1e2be03) SHA1(4e17a0ffe2d34e0d87f173e1450f9cb7ccbd225e) )
	ROM_LOAD16_BYTE( "95420761.hi", 0x00000, 0x080000, CRC(31187236) SHA1(fab73f9e62e43ab9a3104c1a357911cc99f93b38) )
	sc_ckx_others
ROM_END

ROM_START( sc4ckxc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420762.lo", 0x00001, 0x080000, CRC(66f29fe1) SHA1(2eed91eecb15db51117e2351dd80072817b846b8) )
	ROM_LOAD16_BYTE( "95420763.hi", 0x00000, 0x080000, CRC(fe572525) SHA1(d9677d564fd34062e5ba9e6fa382f8bfb4dfd88d) )
	sc_ckx_others
ROM_END

ROM_START( sc4ckxd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421660.lo", 0x00001, 0x080000, CRC(7815578f) SHA1(45525e03c3093c26182490831831a5fba64bef9f) )
	ROM_LOAD16_BYTE( "95421661.hi", 0x00000, 0x080000, CRC(d5ce83b6) SHA1(bbfe89a3f25e358316bd0140f8a0da77dab40bc5) )
	sc_ckx_others
ROM_END

ROM_START( sc4ckxe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421758.lo", 0x00001, 0x080000, CRC(5b7cdf3b) SHA1(8a6aedd2a82c2d02cc954d9ac49b7f4a10ffd0a4) )
	ROM_LOAD16_BYTE( "95421759.hi", 0x00000, 0x080000, CRC(852feca2) SHA1(b537f8090f3d450f9854670c9e13b2c679ddf0d7) )
	sc_ckx_others
ROM_END

ROM_START( sc4ckxf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421760.lo", 0x00001, 0x080000, CRC(001c041b) SHA1(30ca8868a57ff89f258cac9b3d88872b6751958d) )
	ROM_LOAD16_BYTE( "95421761.hi", 0x00000, 0x080000, CRC(6781dc20) SHA1(812cf24aaf3f02dd83b1fc996615fe092809bd47) )
	sc_ckx_others
ROM_END

ROM_START( sc4ckxg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421762.lo", 0x00001, 0x080000, CRC(d6683bf6) SHA1(c2756441043a1097edb540213ca8bef56c432767) )
	ROM_LOAD16_BYTE( "95421763.hi", 0x00000, 0x080000, CRC(6efaba5c) SHA1(abf8dddcdb891396e6dc05a020028fd38890999e) )
	sc_ckx_others
ROM_END




ROM_START( sc4casry )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408018.lo", 0x00001, 0x080000, CRC(a921ad37) SHA1(7e8ab72ebe9c0341912a0a27b1ac114eee5dffb7) )
	ROM_LOAD16_BYTE( "95408019.hi", 0x00000, 0x080000, CRC(91ab8fa0) SHA1(59754315338c1a027d5d6be87da524eb588672d7) )
	sc_casry_others
ROM_END

ROM_START( sc4casrya )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408126.lo", 0x00001, 0x080000, CRC(4f499073) SHA1(93c79e963183f48c43680e0e4f50277e3cf51fbd) )
	ROM_LOAD16_BYTE( "95408127.hi", 0x00000, 0x080000, CRC(0c653000) SHA1(3c4a3958b8476bb7c36e0948b636ab63cf90354d) )
	sc_casry_others
ROM_END

ROM_START( sc4casryb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408158.lo", 0x00001, 0x080000, CRC(0dc6ab53) SHA1(8825515666310afc6559e9f96286f1b78cc16cc3) )
	ROM_LOAD16_BYTE( "95408159.hi", 0x00000, 0x080000, CRC(5e805c7f) SHA1(45d71ab83de1b42ea308a293ddc9a96c5d75c30d) )
	sc_casry_others
ROM_END

ROM_START( sc4casryc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409018.lo", 0x00001, 0x080000, CRC(c5cb046b) SHA1(ad6e0508e084d4f8c7372e808203b14e1abf34ca) )
	ROM_LOAD16_BYTE( "95409019.hi", 0x00000, 0x080000, CRC(4cce8458) SHA1(b7d39278edc311dc915123aab84c0cb2f6df014d) )
	sc_casry_others
ROM_END

ROM_START( sc4casryd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414354.lo", 0x00001, 0x080000, CRC(bbf6bdc9) SHA1(49c8145f3f4937573797538a19f7dfcf5129020f) )
	ROM_LOAD16_BYTE( "95414355.hi", 0x00000, 0x080000, CRC(a61dd5a9) SHA1(8e3f1d7da70960c079301e46fb37eb5f21430fe6) )
	sc_casry_others
ROM_END

ROM_START( sc4casrye )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416042.lo", 0x00001, 0x080000, CRC(1fcd054d) SHA1(7e061ecbd2b53af9dea3dd9b7bdccd555cd31286) )
	ROM_LOAD16_BYTE( "95416043.hi", 0x00000, 0x080000, CRC(61d1a38b) SHA1(631b14ed69cdf39473677ae42ebe3ed0046b221c) )
	sc_casry_others
ROM_END










ROM_START( sc4chick )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406754.lo", 0x00001, 0x080000, CRC(3df14180) SHA1(6239699d788e9becdba091f1fdd358e11c347d1d) )
	ROM_LOAD16_BYTE( "95406755.hi", 0x00000, 0x080000, CRC(3c37d9cd) SHA1(3285a260362f8391a4c9887d452a56881ed0d275) )
	sc_chick_others
ROM_END




ROM_START( sc4chub )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400281.lo", 0x00001, 0x080000, CRC(55cffc3e) SHA1(346b77e8f0fc4ba58cb39157ca7ab411b7a1c8e8) )
	ROM_LOAD16_BYTE( "95400282.hi", 0x00000, 0x080000, CRC(e8b704b3) SHA1(7d14b14b3eb21ea9fd85985bb3acf16d691f00a5) )
	sc_chub_others
ROM_END

ROM_START( sc4chuba )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400803.lo", 0x00001, 0x080000, CRC(ebc45fd0) SHA1(865a5cc1a2ffa5c8125411787999e06568d560ce) )
	ROM_LOAD16_BYTE( "95400804.hi", 0x00000, 0x080000, CRC(5e10c078) SHA1(eb6a00ff568be24721d6254814464cb209bd8171) )
	sc_chub_others
ROM_END

ROM_START( sc4chubb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401803.lo", 0x00001, 0x080000, CRC(3feeeba0) SHA1(e333ed738cee42723528af65c8b3b5b1b24b4bf3) )
	ROM_LOAD16_BYTE( "95401804.hi", 0x00000, 0x080000, CRC(31d342e8) SHA1(302d61748fa24d1c955557118bb4c9cb4997f340) )
	sc_chub_others
ROM_END




ROM_START( sc4cla7 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408544.lo", 0x00001, 0x080000, CRC(1e8f68a5) SHA1(134c951ce4c93954c961c6a752faeeb85d6141bb) )
	ROM_LOAD16_BYTE( "95408545.hi", 0x00000, 0x080000, CRC(39590624) SHA1(75e12f094f344468d15297add4cfdb5bd4342eec) )
	sc_cla7_others
ROM_END

ROM_START( sc4cla7a )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409544.lo", 0x00001, 0x080000, CRC(919ce515) SHA1(23d506c789a3de63eff73b7c18a7e46d5aa4821d) )
	ROM_LOAD16_BYTE( "95409545.hi", 0x00000, 0x080000, CRC(d1ef3f68) SHA1(e72403244167b7534926f22ba471a881f7f88e3e) )
	sc_cla7_others
ROM_END

ROM_START( sc4cla7b )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409804.lo", 0x00001, 0x080000, CRC(cfc6ad00) SHA1(193a3dd9cb8ef04a131fa30b51ad9c21f773108b) )
	ROM_LOAD16_BYTE( "95409805.hi", 0x00000, 0x080000, CRC(1d981e1b) SHA1(cdaf1c68c6b30e2e6b4f958967bd6ef6d73a224d) )
	sc_cla7_others
ROM_END

ROM_START( sc4cla7c )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409806.lo", 0x00001, 0x080000, CRC(0b9494f9) SHA1(2c3f2378615e532df154df6e926a2e366b36b4ae) )
	ROM_LOAD16_BYTE( "95409807.hi", 0x00000, 0x080000, CRC(ce7e1b3e) SHA1(69e125a90aae7a85ed8161fc22ee03d4783f7488) )
	sc_cla7_others
ROM_END




ROM_START( sc4ccogs )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409408.odd", 0x00001, 0x080000, CRC(076fd1b8) SHA1(f6089d6a0e8c8f3408842c219bdfe04c3ef9fe24) )
	ROM_LOAD16_BYTE( "95409409.evn", 0x00000, 0x080000, CRC(c2b9efd8) SHA1(cac26ba46ca51f232057534c66701a33a1de5d4a) )
	sc_ccogs_others
ROM_END




ROM_START( sc4cclas )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402982.lo", 0x00001, 0x080000, CRC(dd14db28) SHA1(d5a79e40597cd92cf65fc85c751e289b18c39bd0) )
	ROM_LOAD16_BYTE( "95402983.hi", 0x00000, 0x080000, CRC(86f90794) SHA1(c58657e67e02eecc70cc7f088817112f9aae892f) )
	sc_cclas_others
ROM_END

ROM_START( sc4cclasa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402984.lo", 0x00001, 0x080000, CRC(775c466b) SHA1(50c5e8b6dcd0ef1a85fefb69368c0a1ece5c13d5) )
	ROM_LOAD16_BYTE( "95402985.hi", 0x00000, 0x080000, CRC(aa059e42) SHA1(825fb2c88f5c0d392972ec67b87a74a5fad74e3e) )
	sc_cclas_others
ROM_END

ROM_START( sc4cclasb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403982.lo", 0x00001, 0x080000, CRC(87c6543e) SHA1(7dfcc37224524cfca68112bf506de0b62e6814fc) )
	ROM_LOAD16_BYTE( "95403983.hi", 0x00000, 0x080000, CRC(af55e3a8) SHA1(24fe9caf9eb9a0eb4ccc2ccd38381cb2041b41c8) )
	sc_cclas_others
ROM_END

ROM_START( sc4cclasc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403984.lo", 0x00001, 0x080000, CRC(85185c89) SHA1(51ec08d434a0e11ced42035f7745043377c6a69d) )
	ROM_LOAD16_BYTE( "95403985.hi", 0x00000, 0x080000, CRC(e707f9ea) SHA1(13538dfb781359755a24f8c757851746a3a94180) )
	sc_cclas_others
ROM_END

ROM_START( sc4cclasd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404120.lo", 0x00001, 0x080000, CRC(b2129489) SHA1(41e87d2e2235837ed2399d7bbcca8a51481ec904) )
	ROM_LOAD16_BYTE( "95404121.hi", 0x00000, 0x080000, CRC(780849f1) SHA1(0aeeb86ceace28d858934e1811a5239eaef07155) )
	sc_cclas_others
ROM_END

ROM_START( sc4cclase )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404778.lo", 0x00001, 0x080000, CRC(acc30fb8) SHA1(f2dc37efb302a87052b32aab4029fb59453b4b97) )
	ROM_LOAD16_BYTE( "95404779.hi", 0x00000, 0x080000, CRC(098905e7) SHA1(2fc5efb61419070b924de108af4f3704eed66e9e) )
	sc_cclas_others
ROM_END

ROM_START( sc4cclasf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405120.lo", 0x00001, 0x080000, CRC(d3414d4b) SHA1(b81366013ed361847ffd3f2613a5cfbaea23a90f) )
	ROM_LOAD16_BYTE( "95405121.hi", 0x00000, 0x080000, CRC(e6d79fe7) SHA1(90cab834eae2db75aec5d1a362106fb2c3d2586d) )
	sc_cclas_others
ROM_END

ROM_START( sc4cclasg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405778.lo", 0x00001, 0x080000, CRC(7ae848a5) SHA1(5153ef7d05a97849a6b4af7c1f605ac76c942f66) )
	ROM_LOAD16_BYTE( "95405779.hi", 0x00000, 0x080000, CRC(cfa46d0d) SHA1(64bf0e4889a85cc9a8f95ecdde6f5d1da26085d5) )
	sc_cclas_others
ROM_END

ROM_START( sc4cclash )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406584.lo", 0x00001, 0x080000, CRC(f857f4bd) SHA1(8f619c255c964668dbdb49b015055297c42a5f68) )
	ROM_LOAD16_BYTE( "95406585.hi", 0x00000, 0x080000, CRC(cfb6c220) SHA1(e69644963b353e83320119a8c8c38e2887b6bbcb) )
	sc_cclas_others
ROM_END

ROM_START( sc4cclasi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406694.lo", 0x00001, 0x080000, CRC(78846b8d) SHA1(9709fc0e3c504bd33f69b3edaa22f62439d8fd91) )
	ROM_LOAD16_BYTE( "95406695.hi", 0x00000, 0x080000, CRC(a89f1d49) SHA1(adfc80802e59fc17872ede25a3e95c82f19f8abd) )
	sc_cclas_others
ROM_END

ROM_START( sc4cclasj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406696.lo", 0x00001, 0x080000, CRC(b1bd9dc5) SHA1(9c93c28f8156d661512b1541e4cd8a86f0612ef4) )
	ROM_LOAD16_BYTE( "95406697.hi", 0x00000, 0x080000, CRC(9571c344) SHA1(16e80d31cfdcf0634ed49d2b5fd4efb9bd331fc0) )
	sc_cclas_others
ROM_END

ROM_START( sc4cclask )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406744.lo", 0x00001, 0x080000, CRC(52943d2e) SHA1(44d7a835e087c957b11ab256ca7bc912f9fe5237) )
	ROM_LOAD16_BYTE( "95406745.hi", 0x00000, 0x080000, CRC(763b7dcd) SHA1(bfaa421e8dd55e1d008bc854c8b0c015f717195e) )
	sc_cclas_others
ROM_END

ROM_START( sc4cclasl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406930.lo", 0x00001, 0x080000, CRC(4491aa95) SHA1(a4a92a66cd1bbc57fe193c1fa860af7f96e08abb) )
	ROM_LOAD16_BYTE( "95406931.hi", 0x00000, 0x080000, CRC(6d634e54) SHA1(e29f7bdbe7534b00ace471a6b246f2a08f94b553) )
	sc_cclas_others
ROM_END

ROM_START( sc4cclasm )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406970.lo", 0x00001, 0x080000, CRC(4fef064e) SHA1(b65ef9ab2fcc7cfe72f0c728591260c2057f8a84) )
	ROM_LOAD16_BYTE( "95406971.hi", 0x00000, 0x080000, CRC(e9319a05) SHA1(ffc3f4cb772593d9ed0ae9294c621ad100d4c45f) )
	sc_cclas_others
ROM_END

ROM_START( sc4cclasn )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406972.lo", 0x00001, 0x080000, CRC(502f0174) SHA1(1d1f9a9e4fc6f978fcd579c4909c19aca20a427b) )
	ROM_LOAD16_BYTE( "95406973.hi", 0x00000, 0x080000, CRC(14eae726) SHA1(907d431355e129c013dab8222e992bc08feb52ae) )
	sc_cclas_others
ROM_END

ROM_START( sc4cclaso )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406978.lo", 0x00001, 0x080000, CRC(5900a448) SHA1(2af8fd172c75a9ce2d41f886e3634b5d46975223) )
	ROM_LOAD16_BYTE( "95406979.hi", 0x00000, 0x080000, CRC(5ff4e432) SHA1(65e72ec1276d194ad2e39c14eed83056381d060e) )
	sc_cclas_others
ROM_END

ROM_START( sc4cclasp )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407584.lo", 0x00001, 0x080000, CRC(e7158a0f) SHA1(296aabee4df111f2369f2429db45861214ae8c25) )
	ROM_LOAD16_BYTE( "95407585.hi", 0x00000, 0x080000, CRC(67645ec3) SHA1(2682734f52fcf3a33e2c4dc8f2fbcd89b7d0e908) )
	sc_cclas_others
ROM_END

ROM_START( sc4cclasq )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407694.lo", 0x00001, 0x080000, CRC(1f9a549a) SHA1(e7a0690c6116ad50ba6727edffbe275a34a6013b) )
	ROM_LOAD16_BYTE( "95407695.hi", 0x00000, 0x080000, CRC(d6f5a80b) SHA1(7af2abda9375662af070b28c26cce61f3a0c7b85) )
	sc_cclas_others
ROM_END

ROM_START( sc4cclasr )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407696.lo", 0x00001, 0x080000, CRC(ef6571d8) SHA1(f323fa4d22b0b6fc30239009d0656d1e1f358cc8) )
	ROM_LOAD16_BYTE( "95407697.hi", 0x00000, 0x080000, CRC(5efad9ec) SHA1(db4049e8985b42e9056f8256bf5a00b965d812f1) )
	sc_cclas_others
ROM_END

ROM_START( sc4cclass )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407744.lo", 0x00001, 0x080000, CRC(c9230322) SHA1(2fb7b25f7303c58403d527e4b2d6522730fa4f05) )
	ROM_LOAD16_BYTE( "95407745.hi", 0x00000, 0x080000, CRC(5a4222b3) SHA1(ad7a38fe18e5437e1a545133bcc0b95e1da6113e) )
	sc_cclas_others
ROM_END

ROM_START( sc4cclast )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407930.lo", 0x00001, 0x080000, CRC(358d3d03) SHA1(ba02bf2e32ead55076ac3ad1cef58c7ed52c5276) )
	ROM_LOAD16_BYTE( "95407931.hi", 0x00000, 0x080000, CRC(f145088c) SHA1(111ad347bf6335b14e8ba3d2283b620a6d8e51ab) )
	sc_cclas_others
ROM_END

ROM_START( sc4cclasu )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407970.lo", 0x00001, 0x080000, CRC(ab7a1e6b) SHA1(4b9e00b5f8bc7e8caf9f35df946cdd7b3696be0a) )
	ROM_LOAD16_BYTE( "95407971.hi", 0x00000, 0x080000, CRC(6ba1e287) SHA1(83a52387796eb1d559b2db772997535e63acee4c) )
	sc_cclas_others
ROM_END

ROM_START( sc4cclasv )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407972.lo", 0x00001, 0x080000, CRC(21cd6a85) SHA1(996923887dc386f5028d08b93993de6a9ca6707f) )
	ROM_LOAD16_BYTE( "95407973.hi", 0x00000, 0x080000, CRC(afa110cf) SHA1(aa81975dd08b8b24d5abe227fd441294de434d49) )
	sc_cclas_others
ROM_END

ROM_START( sc4cclasw )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407978.lo", 0x00001, 0x080000, CRC(4bb5dfb7) SHA1(d637236e9b596df295ded57dac2ea0be383841fa) )
	ROM_LOAD16_BYTE( "95407979.hi", 0x00000, 0x080000, CRC(1ae0e535) SHA1(1d2778cdf4e5595fabd1a03ea9b095c1bc549554) )
	sc_cclas_others
ROM_END

ROM_START( sc4cclasx )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410746.lo", 0x00001, 0x080000, CRC(58b8f1f8) SHA1(76029beb8adad1b5f90d3452931a087bf023fb69) )
	ROM_LOAD16_BYTE( "95410747.hi", 0x00000, 0x080000, CRC(501846a0) SHA1(edc4ed801f96cc6a373113820c09baa37863afbf) )
	sc_cclas_others
ROM_END

ROM_START( sc4cclasy )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410748.lo", 0x00001, 0x080000, CRC(51c3e330) SHA1(ac9a809d7767ca77a138212e622951828b68c09d) )
	ROM_LOAD16_BYTE( "95410749.hi", 0x00000, 0x080000, CRC(48582c9f) SHA1(203036c81cfc816df46eb0daf593357afd75a234) )
	sc_cclas_others
ROM_END

ROM_START( sc4cclasz )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411746.lo", 0x00001, 0x080000, CRC(b4737d12) SHA1(a1cd95eaed2c52c254e298b629eb52d310ab390c) )
	ROM_LOAD16_BYTE( "95411747.hi", 0x00000, 0x080000, CRC(53c4e408) SHA1(e99b59477d33c567fabe496af6f8793f0a5aed0b) )
	sc_cclas_others
ROM_END

ROM_START( sc4cclas0 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411748.lo", 0x00001, 0x080000, CRC(900015f8) SHA1(70b3a6d129558e358f0f77e1b5f7b4b275458eb9) )
	ROM_LOAD16_BYTE( "95411749.hi", 0x00000, 0x080000, CRC(651fdf91) SHA1(9234a2c5b6e2eb8b98013503fa76e8370740ba96) )
	sc_cclas_others
ROM_END

ROM_START( sc4cclas1 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412176.lo", 0x00001, 0x080000, CRC(4f49d67a) SHA1(3c732921baba359980dea01fa992a40b592fa18b) )
	ROM_LOAD16_BYTE( "95412177.hi", 0x00000, 0x080000, CRC(0c5be069) SHA1(73237f14b52a8cd6f1bed26fb2c1df8e1893f855) )
	sc_cclas_others
ROM_END

ROM_START( sc4cclas2 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412178.lo", 0x00001, 0x080000, CRC(0a7cabc4) SHA1(03eafd07bb19be2f0392e0531acc3fa7c0bcfe13) )
	ROM_LOAD16_BYTE( "95412179.hi", 0x00000, 0x080000, CRC(dc1be24f) SHA1(3c1454da68b750cabfa3b3de318520ca8b2e0cd0) )
	sc_cclas_others
ROM_END

ROM_START( sc4cclas3 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413176.lo", 0x00001, 0x080000, CRC(b89473e7) SHA1(026dd4273cdce7122fedaaf5d9e1665a86e6c8fc) )
	ROM_LOAD16_BYTE( "95413177.hi", 0x00000, 0x080000, CRC(5fe8319c) SHA1(ef201862e9ae8c64498d3e89292074298256410b) )
	sc_cclas_others
ROM_END

ROM_START( sc4cclas4 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413178.lo", 0x00001, 0x080000, CRC(7f603811) SHA1(0e930f672379234de65af8d00a5ae0fddf427015) )
	ROM_LOAD16_BYTE( "95413179.hi", 0x00000, 0x080000, CRC(14af9a34) SHA1(b85d1b461e58743bf66dfc4bc6302aabc4b567d9) )
	sc_cclas_others
ROM_END




ROM_START( sc4clclo )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95425336.lo", 0x00001, 0x080000, CRC(be78cc27) SHA1(250d369c0b9c4047f7f4d28df2035095119f1d54) )
	ROM_LOAD16_BYTE( "95425337.hi", 0x00000, 0x080000, CRC(fd8ad271) SHA1(5ef908d5936782ee49f9d7b4013a3125f8092c6a) )
	sc_clclo_others
ROM_END






ROM_START( sc4clucl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416678.lo", 0x00001, 0x080000, CRC(c3282fba) SHA1(9d1c0fdde7f59aae6b61a348f852e05a186eb00f) )
	ROM_LOAD16_BYTE( "95416679.hi", 0x00000, 0x080000, CRC(b87c1eeb) SHA1(e39fed7449b2b6d9ae0a957d02c97f5f309682aa) )
	sc_clucl_others
ROM_END

ROM_START( sc4clucla )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417678.lo", 0x00001, 0x080000, CRC(08ee7c1f) SHA1(ac3df6509d67293e69da9542ad23e87b4f1cbb4f) )
	ROM_LOAD16_BYTE( "95417679.hi", 0x00000, 0x080000, CRC(7254bdc0) SHA1(54bee637fb010515fa49f44c6cb67faecd12e859) )
	sc_clucl_others
ROM_END




ROM_START( sc4colos )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406370.lo", 0x00001, 0x080000, CRC(2cc7180b) SHA1(d995e86f174bd7f44abc11c44da8641f1d7d644d) )
	ROM_LOAD16_BYTE( "95406371.hi", 0x00000, 0x080000, CRC(e88bf209) SHA1(346e3e3cbaf82aa34693152ca67ed7cbea045253) )
	sc_colos_others
ROM_END









ROM_START( sc4cjcl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410968.lo", 0x00001, 0x080000, CRC(40da7b1f) SHA1(05920956c2d0ff0567e83ca12a0b032e58a27b9d) )
	ROM_LOAD16_BYTE( "95410969.hi", 0x00000, 0x080000, CRC(dd39e1e6) SHA1(37a4003f8e24dda2b4d4017ac6c68eb69b594521) )
	sc_cjcl_others
ROM_END

ROM_START( sc4cjcla )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412276.lo", 0x00001, 0x080000, CRC(a2867941) SHA1(892e1eb60c3d6b4e2c2ecda2f78fccba9c709464) )
	ROM_LOAD16_BYTE( "95412277.hi", 0x00000, 0x080000, CRC(986d0408) SHA1(f9f82aaa37d345aabf05096662b34ef2eac05af9) )
	sc_cjcl_others
ROM_END

ROM_START( sc4cjclb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412356.lo", 0x00001, 0x080000, CRC(9ca19307) SHA1(aab9c1b66201db40621dde95ebe4a62dcf1964eb) )
	ROM_LOAD16_BYTE( "95412357.hi", 0x00000, 0x080000, CRC(077c32f9) SHA1(50a5ca9a647a4388628fa375598737e05be6ef19) )
	sc_cjcl_others
ROM_END

ROM_START( sc4cjclc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412516.lo", 0x00001, 0x080000, CRC(d9e554a1) SHA1(545489318be522d5024363386fdc169618a4bdc4) )
	ROM_LOAD16_BYTE( "95412517.hi", 0x00000, 0x080000, CRC(5a6a00b4) SHA1(e7092436bb7e48707651baaef7bc764e5ec2d199) )
	sc_cjcl_others
ROM_END

ROM_START( sc4cjcld )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413276.lo", 0x00001, 0x080000, CRC(d9987f68) SHA1(5fad0bb476afc021140cdcceccced85a2c55eae9) )
	ROM_LOAD16_BYTE( "95413277.hi", 0x00000, 0x080000, CRC(258db231) SHA1(266500b9c8af7e92c5fa5b2698889300c6c46144) )
	sc_cjcl_others
ROM_END

ROM_START( sc4cjcle )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413356.lo", 0x00001, 0x080000, CRC(96105d0d) SHA1(281e7f505aa46fa30bcf54ab2f80731a21b4916d) )
	ROM_LOAD16_BYTE( "95413357.hi", 0x00000, 0x080000, CRC(c90696c5) SHA1(0fb26232c3dffd3bdb3fffaee0a242cc2752250b) )
	sc_cjcl_others
ROM_END

ROM_START( sc4cjclf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413516.lo", 0x00001, 0x080000, CRC(8b35728b) SHA1(d46e25a897b8bb37b0958fca91fb162afb6051f8) )
	ROM_LOAD16_BYTE( "95413517.hi", 0x00000, 0x080000, CRC(2c03efee) SHA1(52db4cde0c372d3c39a00df6a3c97a6da34f5f64) )
	sc_cjcl_others
ROM_END







ROM_START( sc4ctlcl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404804.lo", 0x00001, 0x080000, CRC(7de836c7) SHA1(ba14d6695662bedc5dff2f94fd31dfe9d57a8ddf) )
	ROM_LOAD16_BYTE( "95404805.hi", 0x00000, 0x080000, CRC(83334f9b) SHA1(bc10d7261f0cbdb00424ad7d19cec5137a901f44) )
	sc_ctlcl_others
ROM_END

ROM_START( sc4ctlcla )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405804.lo", 0x00001, 0x080000, CRC(d2c3d594) SHA1(74aafe256a780c6779fb7302a515c35a0f2c6703) )
	ROM_LOAD16_BYTE( "95405805.hi", 0x00000, 0x080000, CRC(bc8344b0) SHA1(97a7d7bab461ef7cfb1fbb89a992e677afcc4b2d) )
	sc_ctlcl_others
ROM_END

ROM_START( sc4ctlclb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412204.lo", 0x00001, 0x080000, CRC(4b17ab16) SHA1(f5063f66e9886fe2442e2379daaa3fb988509368) )
	ROM_LOAD16_BYTE( "95412205.hi", 0x00000, 0x080000, CRC(0de1e4ff) SHA1(82caf4eba559041e2a8990c81ad09eba49084df7) )
	sc_ctlcl_others
ROM_END

ROM_START( sc4ctlclc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413204.lo", 0x00001, 0x080000, CRC(5c46a67a) SHA1(5b3a3caac83e9a137a8d201ab19187d73df1d4a9) )
	ROM_LOAD16_BYTE( "95413205.hi", 0x00000, 0x080000, CRC(4d879bb9) SHA1(cd48570cd8c02cebcee3bd4dc198ede45f3d5b72) )
	sc_ctlcl_others
ROM_END

ROM_START( ad4ctl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASE00 )
	/* not sure which of the regular Cop the Lot roms these go with */

	ROM_REGION( 0x400000, "adder4", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95780008.lo", 0x00001, 0x100000, CRC(0c81e70d) SHA1(0520fbc3afb9c3306e89d32ba8f01b1a83a47de5) )
	ROM_LOAD16_BYTE( "95780009.hi", 0x00000, 0x100000, CRC(458ea63b) SHA1(4ac8290d3255d992be851a13fc45ef2080edc5ce) )
	sc_ctlcl_others
ROM_END

ROM_START( ad4ctla )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASE00 )
	/* not sure which of the regular Cop the Lot roms these go with */

	ROM_REGION( 0x400000, "adder4", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95780024.lo", 0x00001, 0x100000, CRC(654e4428) SHA1(92f6074461bb58120401b0e1981e48d13cd467dd) )
	ROM_LOAD16_BYTE( "95780025.hi", 0x00000, 0x100000, CRC(7397e2e4) SHA1(d22cc364aeb096ca03cd4642a22bd37fe548e2a1) )
	sc_ctlcl_others
ROM_END




ROM_START( sc4crcc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406082.lo", 0x00001, 0x080000, CRC(f3495110) SHA1(3c2242818d02cf34d61c696073855d491bdbee22) )
	ROM_LOAD16_BYTE( "95406083.hi", 0x00000, 0x080000, CRC(284933d6) SHA1(2098319ea687069a4f3080f788655a4c6c2ceaca) )
	sc_crcc_others

	sc_crcc_matrix
ROM_END

ROM_START( sc4crcca )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407082.lo", 0x00001, 0x080000, CRC(97da4079) SHA1(a75a9be748281ff6406e1cf6a28b9271b3410751) )
	ROM_LOAD16_BYTE( "95407083.hi", 0x00000, 0x080000, CRC(15ef1bda) SHA1(77dc2a46cbedcb331bade2d121947b32faace179) )
	sc_crcc_others

	sc_crcc_matrix
ROM_END

ROM_START( sc4crccb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "club-classic-cops-and-robbers_dat_ac_f65ss.lo", 0x00001, 0x080800, CRC(cec630a8) SHA1(91e95a7b89770e75a1b9272e011a7e4dacaa9eeb) )
	ROM_LOAD16_BYTE( "club-classic-cops-and-robbers_dat_ac_f65ss.hi", 0x00000, 0x080800, CRC(17f4f78c) SHA1(fdb7be31f00c75c28468c5754d398d9ded8ee6e3) )
	sc_crcc_others

	sc_crcc_matrix
ROM_END

ROM_START( sc4crccc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "club-classic-cops-and-robbers_pcl_ac_f65ss.lo", 0x00001, 0x080800, CRC(603f4bf1) SHA1(1fc78567a9bc266ee612f17929d5abd66f011001) )
	ROM_LOAD16_BYTE( "club-classic-cops-and-robbers_pcl_ac_f65ss.hi", 0x00000, 0x080800, CRC(d7699839) SHA1(0bcea4750c5d97a0f63a49bc497a631eb76a52bc) )
	sc_crcc_others

	sc_crcc_matrix
ROM_END




ROM_START( sc4crcp )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95422370.lo", 0x00001, 0x080000, CRC(0ac0260a) SHA1(3492804331ade1868526acb31b2d1753fa1679fe) )
	ROM_LOAD16_BYTE( "95422371.hi", 0x00000, 0x080000, CRC(1daca8f2) SHA1(679b4c53202d87aaba3795c9d273b3b31b4017f2) )
	sc_crcp_others
ROM_END

ROM_START( sc4crcpa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95422382.lo", 0x00001, 0x080000, CRC(08ef0c6b) SHA1(466fac875912546a72e43b0b76028c9ee6681ba5) )
	ROM_LOAD16_BYTE( "95422383.hi", 0x00000, 0x080000, CRC(795d8e69) SHA1(cd3bb259a18e2420520b6c3975340b1b3b1727e2) )
	sc_crcp_others
ROM_END


ROM_START( sc4crcpc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95422414.lo", 0x00001, 0x080000, CRC(0704cda8) SHA1(8489648edf16bc3317a56d7468e1aaee8789c606) )
	ROM_LOAD16_BYTE( "95422415.hi", 0x00000, 0x080000, CRC(3d131678) SHA1(e7119d77eb946893bbb1c8453ccd3bb62e3b2359) )
	sc_crcp_others
ROM_END

ROM_START( sc4crcpd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95423370.lo", 0x00001, 0x080000, CRC(b2aa9e5c) SHA1(889c02f66321aba8027dbf01084c5efe4d2de23c) )
	ROM_LOAD16_BYTE( "95423371.hi", 0x00000, 0x080000, CRC(d7d31dac) SHA1(f98d5d2d19cac3478082e5e92c96607fb65fb5f2) )
	sc_crcp_others
ROM_END

ROM_START( sc4crcpe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95423382.lo", 0x00001, 0x080000, CRC(36237b2b) SHA1(e3acccdaa50124ec0c5c8761a53b30a570ed2548) )
	ROM_LOAD16_BYTE( "95423383.hi", 0x00000, 0x080000, CRC(56523529) SHA1(ec984011837542baa74121377a80c5f409930ec1) )
	sc_crcp_others
ROM_END

ROM_START( sc4crcpf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95423414.lo", 0x00001, 0x080000, CRC(c8790664) SHA1(7581764fd4fa25931bba117134d55bbfa7a41d54) )
	ROM_LOAD16_BYTE( "95423415.hi", 0x00000, 0x080000, CRC(f9d6d561) SHA1(9703f23b51def222578179ecb7aed44e2067f79b) )
	sc_crcp_others
ROM_END

ROM_START( sc4crcpg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426012.lo", 0x00001, 0x080000, CRC(f6c3c1c2) SHA1(410ef25804f137363e0f38d9bfc1c53214ad0807) )
	ROM_LOAD16_BYTE( "95426013.hi", 0x00000, 0x080000, CRC(2fe467b8) SHA1(237b97b2a2b7a852f6a7e7d2d7f724f2d7101ae4) )
	sc_crcp_others
ROM_END

ROM_START( sc4crcph )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427012.lo", 0x00001, 0x080000, CRC(ad2e89a5) SHA1(7c5ae7276e45e4c37640315519a02d08ba4c9620) )
	ROM_LOAD16_BYTE( "95427013.hi", 0x00000, 0x080000, CRC(b9f831d6) SHA1(53649f39b75e92253d10f6f924bcced9aebf4ccf) )
	sc_crcp_others
ROM_END

ROM_START( sc4crcpi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95426474.lo", 0x00001, 0x080000, CRC(aa0a4d1d) SHA1(20ad278517287b3cbcd9f8275814635ea3ecb766) )
	ROM_LOAD16_BYTE( "95426475.hi", 0x00000, 0x080000, CRC(ec216472) SHA1(19c4806834348d68e5d80234cd14ecde9cf8112c) )
	ROM_REGION( 0x400000, "ymz", ROMREGION_ERASE00)
ROM_END

ROM_START( sc4crcpj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95427474.lo", 0x00001, 0x080000, CRC(e73365df) SHA1(6d8d918368350da76eec60b71c0dab6de7b970c7) )
	ROM_LOAD16_BYTE( "95427475.hi", 0x00000, 0x080000, CRC(58fd6031) SHA1(beef4a6fd5e0d7bc68e9628073c0ce0035bd9168) )
	ROM_REGION( 0x400000, "ymz", ROMREGION_ERASE00)
ROM_END


ROM_START( sc4crgc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408094.lo", 0x00001, 0x080000, CRC(28d67004) SHA1(9fddbd4df6d5a9fbe8dee1036aa218f043eba048) )
	ROM_LOAD16_BYTE( "95408095.hi", 0x00000, 0x080000, CRC(c93eb0ed) SHA1(23a0fa64eb77ff7596dbfd020562801d4e349b90) )
	sc_crgc_others
ROM_END

ROM_START( sc4crgca )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409996.lo", 0x00001, 0x080000, CRC(85828d04) SHA1(d97eeff91d3fef7dd07b41491c43c5ccfc70fc4e) )
	ROM_LOAD16_BYTE( "95409997.hi", 0x00000, 0x080000, CRC(bb7e5ead) SHA1(ed7641c357f30fd22cb15234de801f695f86733a) )
	sc_crgc_others
ROM_END

ROM_START( sc4crgcb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408166.lo", 0x00001, 0x080000, CRC(132acdbf) SHA1(0dca7f4088fac7ec1c5c7c3e1d444b18304daf93) )
	ROM_LOAD16_BYTE( "95408167.hi", 0x00000, 0x080000, CRC(4ef7c709) SHA1(9f52993848503252c838db3a9dca89dc7d5e0a5d) )
	sc_crgc_others
ROM_END

ROM_START( sc4crgcc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408344.lo", 0x00001, 0x080000, CRC(f80e83ad) SHA1(fd88057a9311b70273f333d1ae73adf164827756) )
	ROM_LOAD16_BYTE( "95408345.hi", 0x00000, 0x080000, CRC(8ebf7b2b) SHA1(c18acb4d9b42a2168d973ecd564fd44de64317d4) )
	sc_crgc_others
ROM_END

ROM_START( sc4crgcd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408412.lo", 0x00001, 0x080000, CRC(8d95a6f3) SHA1(2a94a2fa2d8c7c7418cdb2aba3051b498c319078) )
	ROM_LOAD16_BYTE( "95408413.hi", 0x00000, 0x080000, CRC(25c5d061) SHA1(f07844a9083443b5a7c0262735dffffc2419ed07) )
	sc_crgc_others
ROM_END

ROM_START( sc4crgce )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408606.lo", 0x00001, 0x080000, CRC(3685ee3f) SHA1(20e0b5a539421e5af23bc8d2f38de8a700fba5ce) )
	ROM_LOAD16_BYTE( "95408607.hi", 0x00000, 0x080000, CRC(87168c0d) SHA1(8ec642723223bcd42be1abb7252300a907d151fe) )
	sc_crgc_others
ROM_END

ROM_START( sc4crgcf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408996.lo", 0x00001, 0x080000, CRC(31a5b982) SHA1(331a6d286fb7e8fad92bebdd5596a332f89cd93d) )
	ROM_LOAD16_BYTE( "95408997.hi", 0x00000, 0x080000, CRC(3145b48e) SHA1(c23093780773271f7a8ce507455089ca2a5b4587) )
	sc_crgc_others
ROM_END

ROM_START( sc4crgcg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408998.lo", 0x00001, 0x080000, CRC(01e2faf1) SHA1(af58be5c0a3dc9e21c6e70b14d4a01d008bad16e) )
	ROM_LOAD16_BYTE( "95408999.hi", 0x00000, 0x080000, CRC(73f72639) SHA1(431f1fbe69e9872afe5582408286248e938a6ac9) )
	sc_crgc_others
ROM_END

ROM_START( sc4crgch )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409094.lo", 0x00001, 0x080000, CRC(6ada392d) SHA1(cd4b619bc179011b98ccab13c64eada522696f70) )
	ROM_LOAD16_BYTE( "95409095.hi", 0x00000, 0x080000, CRC(42d196c8) SHA1(99d769a74ee5375a65b1159e16ac6d7890d78257) )
	sc_crgc_others
ROM_END

ROM_START( sc4crgci )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409166.lo", 0x00001, 0x080000, CRC(3f4433a5) SHA1(38ab986c72da1821303635ca18492d59b7c83e45) )
	ROM_LOAD16_BYTE( "95409167.hi", 0x00000, 0x080000, CRC(e9b01ce0) SHA1(1e213c75a8ee91a6e6631b6b6e446e3db0046de1) )
	sc_crgc_others
ROM_END

ROM_START( sc4crgcj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409344.lo", 0x00001, 0x080000, CRC(eb933941) SHA1(1fbf4460ae182fb899e9f22dc9d12ce359cda656) )
	ROM_LOAD16_BYTE( "95409345.hi", 0x00000, 0x080000, CRC(b4ed84de) SHA1(9c0678093df07490326336714c4da78f0c5789ab) )
	sc_crgc_others
ROM_END

ROM_START( sc4crgck )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409412.lo", 0x00001, 0x080000, CRC(8bd3f08b) SHA1(c6f9159f06574c2c60dc19bc06fba7c818925958) )
	ROM_LOAD16_BYTE( "95409413.hi", 0x00000, 0x080000, CRC(295f4d98) SHA1(55840dd97abb347b6e65212fe59fad6fc53ecf51) )
	sc_crgc_others
ROM_END

ROM_START( sc4crgcl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409606.lo", 0x00001, 0x080000, CRC(63beeb71) SHA1(148a8b3ea70e8394748c90d9f34ff33e0a81f260) )
	ROM_LOAD16_BYTE( "95409607.hi", 0x00000, 0x080000, CRC(da789b85) SHA1(94989e9c833a12b38ed263d54ed9fd3dc6b77162) )
	sc_crgc_others
ROM_END

ROM_START( sc4crgcm )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409998.lo", 0x00001, 0x080000, CRC(e1ddccd7) SHA1(8bcd48e99ee204b1ee5c79a4a139d9d93cb5c53c) )
	ROM_LOAD16_BYTE( "95409999.hi", 0x00000, 0x080000, CRC(16c8b2e9) SHA1(4ed92b19ae8e7d4328184881c58a2f62ead38299) )
	sc_crgc_others
ROM_END

ROM_START( sc4crgcn )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410822.lo", 0x00001, 0x080000, CRC(245b3579) SHA1(41c125629a88969fc73e5323c1e58dc0e4ee499d) )
	ROM_LOAD16_BYTE( "95410823.hi", 0x00000, 0x080000, CRC(15388140) SHA1(90b20280a8d64aaf98243289e5129735924588cb) )
	sc_crgc_others
ROM_END

ROM_START( sc4crgco )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410902.lo", 0x00001, 0x080000, CRC(fb3e58b9) SHA1(a5cbef4cf1c63df207f47685dce369a9f0bb502c) )
	ROM_LOAD16_BYTE( "95410903.hi", 0x00000, 0x080000, CRC(4cf0b6e9) SHA1(25f49509781dda48f45811ac401052c1984929e1) )
	sc_crgc_others
ROM_END

ROM_START( sc4crgcp )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411822.lo", 0x00001, 0x080000, CRC(ce575b2e) SHA1(9dc54010b7d4661187a614b8957d4795a94d27ac) )
	ROM_LOAD16_BYTE( "95411823.hi", 0x00000, 0x080000, CRC(2f32da9d) SHA1(d1a7e32c3f4b91d8b1833f10ca42ecd89ff0da05) )
	sc_crgc_others
ROM_END

ROM_START( sc4crgcq )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411902.lo", 0x00001, 0x080000, CRC(c798850b) SHA1(3319e2cd8760345777320a39db7f79a58085aeab) )
	ROM_LOAD16_BYTE( "95411903.hi", 0x00000, 0x080000, CRC(1741fc85) SHA1(280d3da25580dbb1ad128ee2dc4e6a97a28a66f3) )
	sc_crgc_others
ROM_END

ROM_START( sc4crgcr )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412272.lo", 0x00001, 0x080000, CRC(e59b16fc) SHA1(7ae1549e0595654449af641a8b013ef5cd740de6) )
	ROM_LOAD16_BYTE( "95412273.hi", 0x00000, 0x080000, CRC(08362450) SHA1(79feac74582f3e5485546fc9adf76806bc66d295) )
	sc_crgc_others
ROM_END

ROM_START( sc4crgcs )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412444.lo", 0x00001, 0x080000, CRC(d8e39b05) SHA1(a7f2e55e47ee0d1b3f0ca005acb32b26f4d1768d) )
	ROM_LOAD16_BYTE( "95412445.hi", 0x00000, 0x080000, CRC(98f8925c) SHA1(e6ae91e267ca886643d06b686cdddb9f7d1e1574) )
	sc_crgc_others
ROM_END

ROM_START( sc4crgct )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412968.lo", 0x00001, 0x080000, CRC(4313b15b) SHA1(b063b96a3ee4e9dfa720fe978aa30d60cd4f610b) )
	ROM_LOAD16_BYTE( "95412969.hi", 0x00000, 0x080000, CRC(b412e203) SHA1(4ecf6892322882ea0453b929ade5e6e7c3ea9689) )
	sc_crgc_others
ROM_END

ROM_START( sc4crgcu )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413272.lo", 0x00001, 0x080000, CRC(3836dc82) SHA1(dda0e02e2c216b54dd0ca417b2993387537293be) )
	ROM_LOAD16_BYTE( "95413273.hi", 0x00000, 0x080000, CRC(7364acbb) SHA1(1e18d9f10dcb022e132588e00ec1c4d18a99f346) )
	sc_crgc_others
ROM_END

ROM_START( sc4crgcv )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413444.lo", 0x00001, 0x080000, CRC(44300a32) SHA1(4d24ce3efcb73048de6d62d730d6f596a9db1211) )
	ROM_LOAD16_BYTE( "95413445.hi", 0x00000, 0x080000, CRC(d03f70de) SHA1(109bf51f2fb6a50f3f9ab0df856bb57a48914fca) )
	sc_crgc_others
ROM_END

ROM_START( sc4crgcw )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413968.lo", 0x00001, 0x080000, CRC(5d756e67) SHA1(b69e47db4d71c32a7f5a7131652c206b7afd8cd7) )
	ROM_LOAD16_BYTE( "95413969.hi", 0x00000, 0x080000, CRC(45a28f6d) SHA1(1c774d2eea314c663168e8bcbc3ed48c516bab21) )
	sc_crgc_others
ROM_END




ROM_START( sc4corcl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95424078.lo", 0x00001, 0x080000, CRC(73cd70c7) SHA1(d3dea0af57fa949a894e677a51ef490049391a27) )
	ROM_LOAD16_BYTE( "95424079.hi", 0x00000, 0x080000, CRC(725a5f0a) SHA1(7dafea4178e61256d5b874d3828c839fabff1b11) )
	sc_corcl_others
ROM_END

ROM_START( sc4corcla )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95425078.lo", 0x00001, 0x080000, CRC(9cd82a9c) SHA1(995bd60fa1652c6d4f4027ec57e9ed0d6c76766c) )
	ROM_LOAD16_BYTE( "95425079.hi", 0x00000, 0x080000, CRC(ec7cdcf1) SHA1(7494cf259b767f30c4f39f5d05cbf48558599bee) )
	sc_corcl_others
ROM_END




ROM_START( sc4cyc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404970.lo", 0x00001, 0x080000, CRC(663e3409) SHA1(df47da13099f58a8a654904f8433f9236258474e) )
	ROM_LOAD16_BYTE( "95404971.hi", 0x00000, 0x080000, CRC(2ba61311) SHA1(df353431d49822d935645eaf7625d240b5c47edc) )
	sc_cyc_others
ROM_END

ROM_START( sc4cyca )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405970.lo", 0x00001, 0x080000, CRC(eab44b04) SHA1(b3f6c43b54b320b5fb8476f79f4dfd1f41693509) )
	ROM_LOAD16_BYTE( "95405971.hi", 0x00000, 0x080000, CRC(a95e3238) SHA1(af32fedb27507b44f0bca97c8c7935a59ba1decf) )
	sc_cyc_others
ROM_END

ROM_START( sc4cycb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406828.lo", 0x00001, 0x080000, CRC(2eb727fc) SHA1(1228168279a902adb6c4ad9f1b89a9c9566d2cac) )
	ROM_LOAD16_BYTE( "95406829.hi", 0x00000, 0x080000, CRC(455751cb) SHA1(22a7787e97ad2873e0b8d46cb3d2a85c44b432ff) )
	sc_cyc_others
ROM_END

ROM_START( sc4cycc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407828.lo", 0x00001, 0x080000, CRC(1815d8b6) SHA1(d61f43652e20da551cc4075a0135812ad4b47e8a) )
	ROM_LOAD16_BYTE( "95407829.hi", 0x00000, 0x080000, CRC(a23801ee) SHA1(0b8c375f8615462294afd648177ba46c943767a6) )
	sc_cyc_others
ROM_END

ROM_START( sc4cycd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408966.lo", 0x00001, 0x080000, CRC(33099885) SHA1(03ffb60d631ce1db0ea3b6bf3f58fe9317be271d) )
	ROM_LOAD16_BYTE( "95408967.hi", 0x00000, 0x080000, CRC(3c5474c8) SHA1(90e259bb9caa719f6f7bdbb3fe9d248b26197afe) )
	sc_cyc_others
ROM_END




ROM_START( sc4cyccl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410898.lo", 0x00001, 0x080000, CRC(cfa808d0) SHA1(a9b3e1d847db59ad686eb2e0ebd77b87f8fe4971) )
	ROM_LOAD16_BYTE( "95410899.hi", 0x00000, 0x080000, CRC(c4f0f444) SHA1(253c1f04ade2813c2a9837a933798745315a13f5) )
	sc_cyccl_others
ROM_END

ROM_START( sc4cyccla )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410900.lo", 0x00001, 0x080000, CRC(c014291b) SHA1(2bb8724885629ead018ae0d7a78363cf098b0c51) )
	ROM_LOAD16_BYTE( "95410901.hi", 0x00000, 0x080000, CRC(88714020) SHA1(27200edc9024d7fa0584b74ec1501cd947002fce) )
	sc_cyccl_others
ROM_END

ROM_START( sc4cycclb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411898.lo", 0x00001, 0x080000, CRC(58bb0891) SHA1(4bf9a7a4f37c8e90aeb9775525d2e3e1c2867e65) )
	ROM_LOAD16_BYTE( "95411899.hi", 0x00000, 0x080000, CRC(93db23d1) SHA1(e59f0c954718415e096619bd4b4c9cd0607057f2) )
	sc_cyccl_others
ROM_END

ROM_START( sc4cycclc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411900.lo", 0x00001, 0x080000, CRC(5707295a) SHA1(8aa87deed9be1ad1db553b7c27fc66dbbd2f7046) )
	ROM_LOAD16_BYTE( "95411901.hi", 0x00000, 0x080000, CRC(df5a97b5) SHA1(6b34874c379cf68b411ec4f6546bd6b4096d4147) )
	sc_cyccl_others
ROM_END






ROM_START( sc4crzcs )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402099.lo", 0x00001, 0x080000, CRC(a4f80337) SHA1(9960f2aeeb820bf8ea9884239ac68a9939007d81) )
	ROM_LOAD16_BYTE( "95402100.hi", 0x00000, 0x080000, CRC(c463bb66) SHA1(b1a5e53207e9e282ab5e1937843da155ddf29f62) )
	sc_crzcs_others
ROM_END

ROM_START( sc4crzcsa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402101.lo", 0x00001, 0x080000, CRC(75896c5b) SHA1(ee9d40d637add14670dba3668c2fa397198eb7a7) )
	ROM_LOAD16_BYTE( "95402102.hi", 0x00000, 0x080000, CRC(d401814c) SHA1(cc4598363d78bb25c6ea3559eb1fbb607e2c8fd5) )
	sc_crzcs_others
ROM_END

ROM_START( sc4crzcsb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403099.lo", 0x00001, 0x080000, CRC(63df5b51) SHA1(8644e48ca9282a19ba0bf95414905e7bdacedc85) )
	ROM_LOAD16_BYTE( "95403100.hi", 0x00000, 0x080000, CRC(416b585a) SHA1(d71f7625476fba3568bfe58f7f9fc71a35e5812d) )
	sc_crzcs_others
ROM_END

ROM_START( sc4crzcsc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403101.lo", 0x00001, 0x080000, CRC(23e43c74) SHA1(eb7bfa6b39eaf4a74194f57310f6b5a965649766) )
	ROM_LOAD16_BYTE( "95403102.hi", 0x00000, 0x080000, CRC(071c1995) SHA1(f34123b4dceca66da4703ef9d6d51647caf02913) )
	sc_crzcs_others
ROM_END




ROM_START( sc4cclim )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406724.lo", 0x00001, 0x080000, CRC(787224ab) SHA1(16fc90d667991007c59a1048fbe40159cbd3638e) )
	ROM_LOAD16_BYTE( "95406725.hi", 0x00000, 0x080000, CRC(264a8a48) SHA1(3a25188aa1f41c8574553803d31db2709b577dd6) )
	sc_cclim_others
ROM_END

ROM_START( sc4cclima )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407724.lo", 0x00001, 0x080000, CRC(3391d197) SHA1(08c396c012b2a829f760f5ce6517c72f48a18fd6) )
	ROM_LOAD16_BYTE( "95407725.hi", 0x00000, 0x080000, CRC(c81a93b5) SHA1(e4e2d933faaba27b15b1534eecfff1cce0893794) )
	sc_cclim_others
ROM_END

ROM_START( sc4cclimb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410048.lo", 0x00001, 0x080000, CRC(44eeea95) SHA1(49a8858e0dc92367902cb109fbb5882601d43a52) )
	ROM_LOAD16_BYTE( "95410049.hi", 0x00000, 0x080000, CRC(8e66c8b8) SHA1(c492fc253798118f2948ab9f1a194cbe463b4111) )
	sc_cclim_others
ROM_END

ROM_START( sc4cclimc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410050.lo", 0x00001, 0x080000, CRC(a618e745) SHA1(98cb72f390392a274f67ab27f6da6929ec960153) )
	ROM_LOAD16_BYTE( "95410051.hi", 0x00000, 0x080000, CRC(b29e6a73) SHA1(37a4cd7e99f4442b1779bf12f7edd7c8da0de4e8) )
	sc_cclim_others
ROM_END

ROM_START( sc4cclimd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410368.lo", 0x00001, 0x080000, CRC(0e919801) SHA1(a7dfde12faa5ef81d1dfee3b9ac331c04ec04a74) )
	ROM_LOAD16_BYTE( "95410369.hi", 0x00000, 0x080000, CRC(7321ff76) SHA1(3db4026a6a210f13a9e8939467c39d58b654af90) )
	sc_cclim_others
ROM_END

ROM_START( sc4cclime )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410570.lo", 0x00001, 0x080000, CRC(9bd660fd) SHA1(e7851a2362e33797eb35ed653edce6ba38613c82) )
	ROM_LOAD16_BYTE( "95410571.hi", 0x00000, 0x080000, CRC(e7af2b9e) SHA1(0d83bad8c845e1b8c07901b97d76f1ac16d006e7) )
	sc_cclim_others
ROM_END

ROM_START( sc4cclimf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410572.lo", 0x00001, 0x080000, CRC(da0273e7) SHA1(36359f7385de064966f2992e9f4b1214669f09b6) )
	ROM_LOAD16_BYTE( "95410573.hi", 0x00000, 0x080000, CRC(e48df4d2) SHA1(83538ef5e5097899755964377fcf43b6f58a683a) )
	sc_cclim_others
ROM_END

ROM_START( sc4cclimg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410602.lo", 0x00001, 0x080000, CRC(5259b29d) SHA1(54b4183cfcd7853826e0c869ff7d4d48dd02927d) )
	ROM_LOAD16_BYTE( "95410603.hi", 0x00000, 0x080000, CRC(30dbb084) SHA1(f806472fa82a01a0dfd394044163e392f45d1ac0) )
	sc_cclim_others
ROM_END

ROM_START( sc4cclimh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411048.lo", 0x00001, 0x080000, CRC(e242f807) SHA1(7ac2d6b3804cee3721241cc30ee4e61aa6dfde5d) )
	ROM_LOAD16_BYTE( "95411049.hi", 0x00000, 0x080000, CRC(24cbd652) SHA1(0c7b57102c708d7837bbd55c5bf1601a573a6528) )
	sc_cclim_others
ROM_END

ROM_START( sc4cclimi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411368.lo", 0x00001, 0x080000, CRC(63c9e46b) SHA1(bc6aa6483fc386f7a8198742bd62ce2359906679) )
	ROM_LOAD16_BYTE( "95411369.hi", 0x00000, 0x080000, CRC(83d90b83) SHA1(c1fcf31eb14c4dce1091b35e388feab391001529) )
	sc_cclim_others
ROM_END

ROM_START( sc4cclimj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411570.lo", 0x00001, 0x080000, CRC(08c32812) SHA1(da29b94d0b2cf6b87cb4f63d6adcb05a0d3af227) )
	ROM_LOAD16_BYTE( "95411571.hi", 0x00000, 0x080000, CRC(005f2d83) SHA1(df3f8434e69215fc0f355bb0bb21b7139bfc9aef) )
	sc_cclim_others
ROM_END

ROM_START( sc4cclimk )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411572.lo", 0x00001, 0x080000, CRC(49173b08) SHA1(3f7f22c6a866fe5dda56aa14052aedd9f768bb35) )
	ROM_LOAD16_BYTE( "95411573.hi", 0x00000, 0x080000, CRC(037df2cf) SHA1(8a66e43729a10eb459f11ff9051402da380efd6c) )
	sc_cclim_others
ROM_END

ROM_START( sc4ccliml )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411602.lo", 0x00001, 0x080000, CRC(9dca9084) SHA1(e294c7aac3b616893c7f1b14f040cf32534f44ba) )
	ROM_LOAD16_BYTE( "95411603.hi", 0x00000, 0x080000, CRC(fb52e9e7) SHA1(60f21eb0d8391b995c455a7e915812c71beaaba7) )
	sc_cclim_others
ROM_END

ROM_START( sc4cclimm )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1129p12.hi", 0x00000, 0x080000, CRC(3172ea1e) SHA1(d0115e8f005fc0bce277e74fcb4f393cb25f07e1) )
	ROM_LOAD16_BYTE( "p1129p12.lo", 0x00001, 0x080000, CRC(7f631417) SHA1(f037a1746576a6ff85e286291feccb933e887971) )
	sc_cclim_others
ROM_END

ROM_START( sc4cclimn )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1129p41.hi", 0x00000, 0x080000, CRC(e0803c6a) SHA1(d388c7248b07b76a1e8ab5881387e3bfcc01cfdc) )
	ROM_LOAD16_BYTE( "p1129p41.lo", 0x00001, 0x080000, CRC(749068fd) SHA1(c4f789d58546d365367b9f59d51226580dbf0e3c) )
	sc_cclim_others
ROM_END

ROM_START( sc4cclimo )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1129s12.hi", 0x00000, 0x080000, CRC(335e1b6c) SHA1(ca4a854afe6abe1d397fee7bac34ececd6ecf96b) )
	ROM_LOAD16_BYTE( "p1129s12.lo", 0x00001, 0x080000, CRC(eb104c46) SHA1(9cf6d7ca330018a4e1548af4513dff09fcda3f1f) )
	sc_cclim_others
ROM_END

ROM_START( sc4cclimp )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1129s41.hi", 0x00000, 0x080000, CRC(b2298a64) SHA1(69be683390abc5f16599599908ad7738280fb116) )
	ROM_LOAD16_BYTE( "p1129s41.lo", 0x00001, 0x080000, CRC(3f42075a) SHA1(cf0b6cb6cfb104346dd7beeeba4288b63d350481) )
	sc_cclim_others
ROM_END

ROM_START( sc4cclimq )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95412772.lo", 0x00001, 0x080000, CRC(905a1d80) SHA1(04aee5ffb26a3f8c6def2f725f08ca1d8e6b61fd) )
	ROM_LOAD16_BYTE( "95412773.hi", 0x00000, 0x080000, CRC(4aa622ee) SHA1(e999a9ae0e8ef2eb73a9115cd7bf7a8e3904a40b) )
	sc_cclim_others
ROM_END

ROM_START( sc4cclimr )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95412777.hi", 0x00000, 0x080000, CRC(e0e74ca2) SHA1(797fda3a4634bdd3f8ab0b31f7b880c8dbf23904) )
	ROM_LOAD16_BYTE( "95412776.lo", 0x00001, 0x080000, CRC(4aac0ecd) SHA1(192cce8dfc93f44063134bae948d29d10528b83e) )
	sc_cclim_others
ROM_END

ROM_START( sc4cclims )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95412780.lo", 0x00001, 0x080000, CRC(978b1de6) SHA1(e601fbcb188a170f82b9e1092328cb5aea41b899) )
	ROM_LOAD16_BYTE( "95412781.hi", 0x00000, 0x080000, CRC(7b241139) SHA1(62bbf20c3ad5ac251a0e0e017ca0b380ebf671b9) )
	sc_cclim_others
ROM_END

ROM_START( sc4cclimt )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95413772.lo", 0x00001, 0x080000, CRC(3edcdd60) SHA1(40e30b4aba9fc05c781fba6a9d2089dcf1962a76) )
	ROM_LOAD16_BYTE( "95413773.hi", 0x00000, 0x080000, CRC(6da80d25) SHA1(3467e24b60e0f6a7ae07555fc54a8a5bc3f25fed) )
	sc_cclim_others
ROM_END



ROM_START( sc4cclimu )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95413776.lo", 0x00001, 0x080000, CRC(4d339ce0) SHA1(f3152fbd69d277dd0e32179ab4f0c0b7148faa0b) )
	ROM_LOAD16_BYTE( "95413777.hi", 0x00000, 0x080000, CRC(5875fc90) SHA1(996df61833df07f512c1039563917d40e6788f87) )
	sc_cclim_others
ROM_END

ROM_START( sc4cfqps )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402950.odd", 0x00001, 0x080000, CRC(f747af02) SHA1(1bb937c9af221c077f19045ab4c9cc8ec56011b1) )
	ROM_LOAD16_BYTE( "95402951.evn", 0x00000, 0x080000, CRC(41ec3adf) SHA1(03f3dc231715e708c567a44ab3128e77229a8f35) )
	sc_cfqps_others
ROM_END

ROM_START( sc4cfqpsa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403950.lo", 0x00001, 0x080000, CRC(c40ca42e) SHA1(79333b174d3a7a2466f686e06583a3856460a916) )
	ROM_LOAD16_BYTE( "95403951.hi", 0x00000, 0x080000, CRC(a0d2ffe3) SHA1(eed435b55c80aa72857dd5775bd02225ed17173c) )
	sc_cfqps_others
ROM_END

ROM_START( sc4cfqpsb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408138.odd", 0x00001, 0x080000, CRC(e52a968b) SHA1(0b6583dd52616e7fc05b547be8c39640fdc8d07e) )
	ROM_LOAD16_BYTE( "95408139.evn", 0x00000, 0x080000, CRC(e9ca97d6) SHA1(b35766a89359be7611d53ad0a25ebf82726984db) )
	sc_cfqps_others
ROM_END

ROM_START( sc4cfqpsc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408200.odd", 0x00001, 0x080000, CRC(ed4a1958) SHA1(a86e893b42925c13dd0d30f70f3067000b8251d6) )
	ROM_LOAD16_BYTE( "95408201.evn", 0x00000, 0x080000, CRC(9b2f8f47) SHA1(919226138c16b56db1946070747021407e258583) )
	sc_cfqps_others
ROM_END

ROM_START( sc4cfqpsd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409138.odd", 0x00001, 0x080000, CRC(4b9e2020) SHA1(17c44c71bca03d29356d359435eaa2e86abc41e0) )
	ROM_LOAD16_BYTE( "95409139.evn", 0x00000, 0x080000, CRC(3591305e) SHA1(af8a1d4f9dbd4893f0b60c57bd2cfa954d8ebae2) )
	sc_cfqps_others
ROM_END

ROM_START( sc4cfqpse )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409200.odd", 0x00001, 0x080000, CRC(4a116c08) SHA1(d12a335f20b2e89feda852bbc60772c28e962cbd) )
	ROM_LOAD16_BYTE( "95409201.evn", 0x00000, 0x080000, CRC(b756d039) SHA1(8296ac5aa5c3e52fb04b4c80576eca055d90baa1) )
	sc_cfqps_others
ROM_END

ROM_START( sc4cfqpsf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420248.odd", 0x00001, 0x080000, CRC(473bc41d) SHA1(8ce2dc7b72a261a590175f012970f28d452fb565) )
	ROM_LOAD16_BYTE( "95420249.evn", 0x00000, 0x080000, CRC(a57f732f) SHA1(4870bb60ffce969d47f4c57f9653342cdd6cc03d) )
	sc_cfqps_others
ROM_END

ROM_START( sc4cfqpsg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420324.lo", 0x00001, 0x080000, CRC(14f2328e) SHA1(95a05b37e21ce3c729f2000baeb1308008ad13c2) )
	ROM_LOAD16_BYTE( "95420325.hi", 0x00000, 0x080000, CRC(4759bfec) SHA1(0fabe8b9005d05ca7466446719b834120ee3ab9a) )
	sc_cfqps_others
ROM_END

ROM_START( sc4cfqpsh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421248.odd", 0x00001, 0x080000, CRC(a4e924be) SHA1(cfd03c51314b8d90226f1ab19e263e958b159dd9) )
	ROM_LOAD16_BYTE( "95421249.evn", 0x00000, 0x080000, CRC(432e8f7a) SHA1(158972ed5d9b23c345f783628b685907f2a73a0c) )
	sc_cfqps_others
ROM_END

ROM_START( sc4cfqpsi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421324.lo", 0x00001, 0x080000, CRC(f720d22d) SHA1(8e39a6c44dcd4b9987dd750a204b596eec21d55b) )
	ROM_LOAD16_BYTE( "95421325.hi", 0x00000, 0x080000, CRC(a10843b9) SHA1(72ed4ba033b0894981e8da53d882104dbe226a8f) )
	sc_cfqps_others
ROM_END

ROM_START( sc4cfqpsj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "crazyfruits.evn", 0x00000, 0x080000, CRC(7ec3801b) SHA1(77bdd1cbd76bf508c1185c3b4c28a5c14a2bc5a5) )
	ROM_LOAD16_BYTE( "crazyfruits.odd", 0x00001, 0x080000, CRC(d3cd15a1) SHA1(46300e25bf944a0c404670b02fd986ba19899606) )
	sc_cfqps_others
ROM_END

ROM_START( sc4cfqpsk )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "c_fr__pound8.hi", 0x00000, 0x080000, CRC(8657a605) SHA1(05902a98d997aab90147f65dcfbacaa44dca111c) )
	ROM_LOAD16_BYTE( "c_fr__pound8.lo", 0x00001, 0x080000, CRC(246c8c14) SHA1(4e6e5ad37e97efeb6155ce10202ad15f903c61e6) )
	sc_cfqps_others
ROM_END



ROM_START( sc4cfqpsl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95400731.lo", 0x00001, 0x080000, CRC(af64d3e8) SHA1(4bc3c91cd2f686dbc208d197e22d37c8d5a81216) )
	ROM_LOAD16_BYTE( "95400732.hi", 0x00000, 0x080000, CRC(a80fde0f) SHA1(8b99e38aab8b845dfb26555448f1ca669b737ea1) )
	sc_cfqps_others
ROM_END

ROM_START( sc4cfqpsm )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95400733.lo", 0x00001, 0x080000, CRC(a75e21b7) SHA1(526a1d0d2cb7848310beb418f63cd69606e0f158) )
	ROM_LOAD16_BYTE( "95400734.hi", 0x00000, 0x080000, CRC(e6f35697) SHA1(816acbbb254f83030a911928300a6fa244177f51) )
	sc_cfqps_others
ROM_END

ROM_START( sc4cfqpsn )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95401731.lo", 0x00001, 0x080000, CRC(11d8d978) SHA1(b84a3e9ea1d282de05d73b48d20b2a0f49c5384d) )
	ROM_LOAD16_BYTE( "95401732.hi", 0x00000, 0x080000, CRC(796f1e36) SHA1(ec1fc7b9384d7384b5873c572d35cb39de1239ae) )
	sc_cfqps_others
ROM_END

ROM_START( sc4cfqpso )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95401733.lo", 0x00001, 0x080000, CRC(d9b4d65e) SHA1(07196de44372ed477f7b8be40585ab9c01961f8a) )
	ROM_LOAD16_BYTE( "95401734.hi", 0x00000, 0x080000, CRC(17ef0211) SHA1(8d21e5b5a3282fcdd5f6847d860c8c673a71431b) )
	sc_cfqps_others
ROM_END

ROM_START( sc4cfqpsp )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "crfr15hi", 0x00000, 0x080000, CRC(a33f7f3e) SHA1(157f1ad656a423070dbf8f21330ac0382a6e1898) )
	ROM_LOAD16_BYTE( "crfr15lo", 0x00001, 0x080000, CRC(1683f3ef) SHA1(207c92942961ef7590fc4573bc91c9a16e5b9bda) )
	sc_cfqps_others
ROM_END









ROM_START( sc4cfcas )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406710.lo", 0x00001, 0x080000, CRC(a508ee14) SHA1(423a6e9b9f297d30b928051ec8deecd61c33c69e) )
	ROM_LOAD16_BYTE( "95406711.hi", 0x00000, 0x080000, CRC(e18c86fa) SHA1(fdff43a1b07e2c0ac99a0a35e547ef1839c87778) )
	PR6923_CRAZY_FRUITS_SOUNDS11
ROM_END


ROM_START( sc4cfcasa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95410450.lo", 0x00001, 0x080000, CRC(fc8a8427) SHA1(63bd12570a4f9d0a29e7e05a833976697db68e15) )
	ROM_LOAD16_BYTE( "95410451.hi", 0x00000, 0x080000, CRC(99e8270f) SHA1(4529aecc88afc4739ddd5fd072241d577c455a8f) )
	PR6923_CRAZY_FRUITS_SOUNDS11
ROM_END

ROM_START( sc4cfcasb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95410453.hi", 0x00000, 0x080000, CRC(83ceb620) SHA1(a47f4972ef26593e0b9ce13ea4b3f1dc439925f5) )
	ROM_LOAD16_BYTE( "95410452.lo", 0x00001, 0x080000, CRC(59a7f7cd) SHA1(1a40c341f4797e7ec8441f31bc83a59f37eba8fa) )
	PR6923_CRAZY_FRUITS_SOUNDS11
ROM_END

ROM_START( sc4cfcasc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95410530.lo", 0x00001, 0x080000, CRC(0559f269) SHA1(7c9766294f64f35e337796a24955d916231ed89d) )
	ROM_LOAD16_BYTE( "95410531.hi", 0x00000, 0x080000, CRC(5f204ab2) SHA1(b4ba89d8d90175c68eac5346f51dd59465415d69) )
	PR6923_CRAZY_FRUITS_SOUNDS11
ROM_END

ROM_START( sc4cfcasd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95410532.lo", 0x00001, 0x080000, CRC(022b90e3) SHA1(dd8823b75a90896ee843139b696ee5e73f299a87) )
	ROM_LOAD16_BYTE( "95410533.hi", 0x00000, 0x080000, CRC(af410ed5) SHA1(38bbd77789c9db49861dbf72cd23443ad5cfedfb) )
	PR6923_CRAZY_FRUITS_SOUNDS11
ROM_END

ROM_START( sc4cfcase )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95411450.lo", 0x00001, 0x080000, CRC(11cef6cf) SHA1(5364455b3ea34a11844b3b42c1a45a15c2bf126c) )
	ROM_LOAD16_BYTE( "95411451.hi", 0x00000, 0x080000, CRC(f6d4c9e3) SHA1(5011b066375c7faa408ffbd1965c3b5d46836a72) )
	PR6923_CRAZY_FRUITS_SOUNDS11
ROM_END

ROM_START( sc4cfcasf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95411452.lo", 0x00001, 0x080000, CRC(9bd4d90f) SHA1(3a0c3e66a937ecf6d92cd2937a8d942cc40cf436) )
	ROM_LOAD16_BYTE( "95411453.hi", 0x00000, 0x080000, CRC(d2c3aa4a) SHA1(a23c425f2df44525a4d1d95bdfd7ce4985f6a475) )
	PR6923_CRAZY_FRUITS_SOUNDS11
ROM_END

ROM_START( sc4cfcasg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95411530.lo", 0x00001, 0x080000, CRC(3543eb62) SHA1(d0b7afb5e731fbfd12109ba090037d98c69b8a26) )
	ROM_LOAD16_BYTE( "95411531.hi", 0x00000, 0x080000, CRC(3c7955db) SHA1(41e92ed095c81e2f635d1a939c26bbe160eae756) )
	PR6923_CRAZY_FRUITS_SOUNDS11
ROM_END

ROM_START( sc4cfcash )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95411532.lo", 0x00001, 0x080000, CRC(525652ce) SHA1(da5a87c2ac2c2387f267837ceaaf6fbb42019036) )
	ROM_LOAD16_BYTE( "95411533.hi", 0x00000, 0x080000, CRC(2834b509) SHA1(e50e94bda43814268d95cddb42a586bdf73b5efb) )
	PR6923_CRAZY_FRUITS_SOUNDS11
ROM_END

ROM_START( sc4cfcasi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95412738.lo", 0x00001, 0x080000, CRC(3d4acd70) SHA1(3c8396c51b584bf651f7da99fba055ba39d22af9) )
	ROM_LOAD16_BYTE( "95412739.hi", 0x00000, 0x080000, CRC(4d66828e) SHA1(456bd9038fb329c821eb4fd6d19fd41cc6d1ac93) )
	PR6923_CRAZY_FRUITS_SOUNDS11
ROM_END

ROM_START( sc4cfcasj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95412740.lo", 0x00001, 0x080000, CRC(0840e5f9) SHA1(b5dfb30eb91e9f1133fd7ac58ff00186602fc64d) )
	ROM_LOAD16_BYTE( "95412741.hi", 0x00000, 0x080000, CRC(23a85755) SHA1(5ed609ae4ac78294a11fbccd0a024513f97c7d72) )
	PR6923_CRAZY_FRUITS_SOUNDS11
ROM_END

ROM_START( sc4cfcask )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95412754.lo", 0x00001, 0x080000, CRC(9a98fa55) SHA1(5db2401b1bf5904dffc771b54676e87c8eb279a7) )
	ROM_LOAD16_BYTE( "95412755.hi", 0x00000, 0x080000, CRC(9495327b) SHA1(91571feb8d7718bee50ac29f29ef79628f9119cc) )
	PR6923_CRAZY_FRUITS_SOUNDS11
ROM_END

ROM_START( sc4cfcasl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95412764.lo", 0x00001, 0x080000, CRC(46e535cc) SHA1(016232c6b235338f3affe913df08e13889d85f8d) )
	ROM_LOAD16_BYTE( "95412765.hi", 0x00000, 0x080000, CRC(e907ee58) SHA1(9e9988b3c1f8fb4188a8e48338ece87a23181b87) )
	PR6923_CRAZY_FRUITS_SOUNDS11
ROM_END

ROM_START( sc4cfcasm )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95412766.lo", 0x00001, 0x080000, CRC(32339b1f) SHA1(5cdcf2e88bf887d7acc0a33b27ed722d5cba0dfd) )
	ROM_LOAD16_BYTE( "95412767.hi", 0x00000, 0x080000, CRC(8f43e454) SHA1(c4fd0f210fc7074d435362162306b574d8a63d14) )
	PR6923_CRAZY_FRUITS_SOUNDS11
ROM_END



ROM_START( sc4cfcasn )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95413738.lo", 0x00001, 0x080000, CRC(1367c7a9) SHA1(5089435b654173d3005d583fa530eda9a192937d) )
	ROM_LOAD16_BYTE( "95413739.hi", 0x00000, 0x080000, CRC(eb8885e8) SHA1(0cc3d06956bfba56a1ed7315fe4fe3617d484b8f) )
	PR6923_CRAZY_FRUITS_SOUNDS11
ROM_END


ROM_START( sc4cfcaso )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95413740.lo", 0x00001, 0x080000, CRC(5a6465d1) SHA1(9981d9cf86514617a8aa3cb6730e0543b929399e) )
	ROM_LOAD16_BYTE( "95413741.hi", 0x00000, 0x080000, CRC(7655f0d1) SHA1(9d64022d5ea8407eaa5a1d94dede93744845c83a) )
	PR6923_CRAZY_FRUITS_SOUNDS11
ROM_END

ROM_START( sc4cfcasp )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95413754.lo", 0x00001, 0x080000, CRC(17404e08) SHA1(d90399b069328200ff76663062dba12f3394e842) )
	ROM_LOAD16_BYTE( "95413755.hi", 0x00000, 0x080000, CRC(861fb2d1) SHA1(2e4555e8754dbb758819dfa9c1a2e3eb09eacf5f) )
	PR6923_CRAZY_FRUITS_SOUNDS11
ROM_END

ROM_START( sc4cfcasq )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95413762.lo", 0x00001, 0x080000, CRC(f1e94a7f) SHA1(1c8e0579b61d558ee862b948df3c08812b61b3eb) )
	ROM_LOAD16_BYTE( "95413763.hi", 0x00000, 0x080000, CRC(ec7006df) SHA1(164d2078b5a626c0a7d3aa176bc9ab5570e628c1) )
	PR6923_CRAZY_FRUITS_SOUNDS11
ROM_END

ROM_START( sc4cfcasr )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95413764.lo", 0x00001, 0x080000, CRC(0f652c25) SHA1(cf240682e214f1eaead7010d32a6056d8965a654) )
	ROM_LOAD16_BYTE( "95413765.hi", 0x00000, 0x080000, CRC(3fc85422) SHA1(6f14fb68c327b52495386680d1de46006f1e4841) )
	PR6923_CRAZY_FRUITS_SOUNDS11
ROM_END


ROM_START( sc4cfcass )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95402826.lo", 0x00001, 0x080000, CRC(f823b1e3) SHA1(95dd2f8c94dd7f6a204c8056a3a7f2798fb134af) )
	ROM_LOAD16_BYTE( "95402827.hi", 0x00000, 0x080000, CRC(788d2c23) SHA1(88bb8f4e977a2e44af8000f5c135e705f8e55ec4) )
	PR6923_CRAZY_FRUITS_SOUNDS11
ROM_END

ROM_START( sc4cfcast )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95402828.lo", 0x00001, 0x080000, CRC(f5bcb426) SHA1(0f6140cb742c61069f57960c6b94915529ff53b9) )
	ROM_LOAD16_BYTE( "95402829.hi", 0x00000, 0x080000, CRC(29b382e0) SHA1(d513f7069265d73ebac3ccc6b100b262db611b17) )
	PR6923_CRAZY_FRUITS_SOUNDS11
ROM_END

ROM_START( sc4cfcasu )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95403826.lo", 0x00001, 0x080000, CRC(174e86b3) SHA1(3f7ba66f3fa88340ac51bfad49f2feeefc6a1256) )
	ROM_LOAD16_BYTE( "95403827.hi", 0x00000, 0x080000, CRC(f18004de) SHA1(46f421827b118fe3286399157a81530171a133b4) )
	PR6923_CRAZY_FRUITS_SOUNDS11
ROM_END

ROM_START( sc4cfcasv )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95403828.lo", 0x00001, 0x080000, CRC(97231b0e) SHA1(ab200ea31392e1c703bddb18e200f7cda1165a71) )
	ROM_LOAD16_BYTE( "95403829.hi", 0x00000, 0x080000, CRC(a0beaa1d) SHA1(ce532c5c474730cf1cd701eff32aa47f3e1da34d) )
	PR6923_CRAZY_FRUITS_SOUNDS11
ROM_END

ROM_START( sc4cfcasw )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95406072.lo", 0x00001, 0x080000, CRC(147282fb) SHA1(3723fe14a849a7441c9a754d36a33ee3a3bf1716) )
	ROM_LOAD16_BYTE( "95406073.hi", 0x00000, 0x080000, CRC(40343eeb) SHA1(ef95757c47f678c789d940350a77226eafe37d93) )
	PR6923_CRAZY_FRUITS_SOUNDS11
ROM_END

ROM_START( sc4cfcasx )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95407072.lo", 0x00001, 0x080000, CRC(fef2d0ca) SHA1(af7f7138cfe75c293e09c5e7976b736b9e7f1f7b) )
	ROM_LOAD16_BYTE( "95407073.hi", 0x00000, 0x080000, CRC(cb444ef6) SHA1(b851de11a178306033662604c54004c0180d26de) )
	PR6923_CRAZY_FRUITS_SOUNDS11
ROM_END

ROM_START( sc4cfcasy )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95407710.lo", 0x00001, 0x080000, CRC(dd2e2992) SHA1(2ebcdde8112f4cd314f6ff22de8e61c8f81ba6ad) )
	ROM_LOAD16_BYTE( "95407711.hi", 0x00000, 0x080000, CRC(6396f0d4) SHA1(461a326e76249b486755fbf3175b42222ae23144) )
	PR6923_CRAZY_FRUITS_SOUNDS11
ROM_END

ROM_START( sc4cfcasz )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95408162.lo", 0x00001, 0x080000, CRC(30394ef9) SHA1(5c54f3199618bf2aa9a136e05ba6fe16040e4e5b) )
	ROM_LOAD16_BYTE( "95408163.hi", 0x00000, 0x080000, CRC(212c022d) SHA1(3831cf883269e44adc800d8019db8a11f06348e0) )
	PR6923_CRAZY_FRUITS_SOUNDS11
ROM_END

ROM_START( sc4cfcas0 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95408218.lo", 0x00001, 0x080000, CRC(dd7eee33) SHA1(eed75ea486c885b534b2b71b02c1ee34e81bba12) )
	ROM_LOAD16_BYTE( "95408219.hi", 0x00000, 0x080000, CRC(60478c42) SHA1(5e679c87dae3f9ae665dbcb4a00f9feb44e03297) )
	PR6923_CRAZY_FRUITS_SOUNDS11
ROM_END




ROM_START( sc4cfcas1 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95409162.lo", 0x00001, 0x080000, CRC(9a9cd54e) SHA1(a6e683556f08d9bcbc0839eba9934d58c2190d16) )
	ROM_LOAD16_BYTE( "95409163.hi", 0x00000, 0x080000, CRC(0813b353) SHA1(26b122fd878548ffe906ee59194ef61eea78542c) )
	PR6923_CRAZY_FRUITS_SOUNDS11
ROM_END

ROM_START( sc4cfcas2 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95409218.lo", 0x00001, 0x080000, CRC(8c7e3593) SHA1(d94a17fe98eda5c087a0667cbfa607e863d9e84f) )
	ROM_LOAD16_BYTE( "95409219.hi", 0x00000, 0x080000, CRC(cc798847) SHA1(fede2c5ebc2beb21d9b49f58272a60f3fc30a388) )
	PR6923_CRAZY_FRUITS_SOUNDS11
ROM_END






ROM_START( sc4cfcas8 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95410206.lo", 0x00001, 0x080000, CRC(f8479128) SHA1(3228a81e2b72940a8f20fb0102c815455300ab43) )
	ROM_LOAD16_BYTE( "95410207.hi", 0x00000, 0x080000, CRC(2f64a7ef) SHA1(8e05c788c20b35f233070e3a4095addc644b380a) )
	PR6923_CRAZY_FRUITS_SOUNDS11
ROM_END

ROM_START( sc4cfcas9 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95410518.lo", 0x00001, 0x080000, CRC(fcb3b0a6) SHA1(efdb462f91432c8cd26daa31e5adc71e38d2dd4b) )
	ROM_LOAD16_BYTE( "95410519.hi", 0x00000, 0x080000, CRC(3838ca0f) SHA1(8cd53ec06f3c102615034984fa74040a379f81d1) )
	PR6923_CRAZY_FRUITS_SOUNDS11
ROM_END

ROM_START( sc4cfcas3 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95410520.lo", 0x00001, 0x080000, CRC(8f612ecc) SHA1(a7ab7cb115f0242f7825762410d77cce1ad52491) )
	ROM_LOAD16_BYTE( "95410521.hi", 0x00000, 0x080000, CRC(fa54b8de) SHA1(cf0ac9ceba71a3f725e581bee5cd9203f7668992) )
	PR6923_CRAZY_FRUITS_SOUNDS11
ROM_END

ROM_START( sc4cfcas4 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95410522.lo", 0x00001, 0x080000, CRC(a221f34a) SHA1(42d6292ee3661e3aaea56c6b8684ca4c3f54a0d6) )
	ROM_LOAD16_BYTE( "95410523.hi", 0x00000, 0x080000, CRC(8b09aadd) SHA1(10b51d10e6e11253df9b3c5acf2d929fb19772ea) )
	PR6923_CRAZY_FRUITS_SOUNDS11
ROM_END

ROM_START( sc4cfcas5 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95410524.lo", 0x00001, 0x080000, CRC(c309891b) SHA1(d2c8550129b14a6ddf3c251cced1bd450dfa7439) )
	ROM_LOAD16_BYTE( "95410525.hi", 0x00000, 0x080000, CRC(868b6a77) SHA1(09b42a6f3d4371e3b24132d4c45fe062d471a9c2) )
	PR6923_CRAZY_FRUITS_SOUNDS11
ROM_END

ROM_START( sc4cfcas6 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95410996.lo", 0x00001, 0x080000, CRC(d9edbacd) SHA1(e6b0ef17d88dcdb53274a13848e6974cd2127ced) )
	ROM_LOAD16_BYTE( "95410997.hi", 0x00000, 0x080000, CRC(a6118e1b) SHA1(e707da09dcf0aa5d468a00212fc3fcfb3fbcf6ed) )
	PR6923_CRAZY_FRUITS_SOUNDS11
ROM_END

ROM_START( sc4cfcas7 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95410998.lo", 0x00001, 0x080000, CRC(be6fa84d) SHA1(4c426f52c33f52b61ecdfde458e149d0a0222f87) )
	ROM_LOAD16_BYTE( "95410999.hi", 0x00000, 0x080000, CRC(430e4b42) SHA1(9c3316167b48e08c663e0c215bda31c9f78bef0b) )
	PR6923_CRAZY_FRUITS_SOUNDS11
ROM_END



ROM_START( sc4cfcasaa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95411206.lo", 0x00001, 0x080000, CRC(3f747371) SHA1(f04d2f76ff5e0b1fc9096f05d93b555f25a433bd) )
	ROM_LOAD16_BYTE( "95411207.hi", 0x00000, 0x080000, CRC(38a16e88) SHA1(ec94ee1cdb788d5f7e5bc262285bedceb8501c37) )
	PR6923_CRAZY_FRUITS_SOUNDS11
ROM_END

ROM_START( sc4cfcasab )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95411518.lo", 0x00001, 0x080000, CRC(2912f0a9) SHA1(d600c522950e127f957ca341c2ce506223ccf981) )
	ROM_LOAD16_BYTE( "95411519.hi", 0x00000, 0x080000, CRC(1b07aa20) SHA1(4554dff7d38a71241a54f9e56072e6e5c0da9d4b) )
	PR6923_CRAZY_FRUITS_SOUNDS11
ROM_END

ROM_START( sc4cfcasac )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95411520.lo", 0x00001, 0x080000, CRC(f165dba0) SHA1(37ca7bcf72ed84d3c28c405344e6d5cfdc4fb020) )
	ROM_LOAD16_BYTE( "95411521.hi", 0x00000, 0x080000, CRC(44f7fa77) SHA1(06dfdec5e46b45959561223ea960ae5458ba39c9) )
	PR6923_CRAZY_FRUITS_SOUNDS11
ROM_END

ROM_START( sc4cfcasad )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95411522.lo", 0x00001, 0x080000, CRC(4cf8f8dd) SHA1(18f40df751ba13defd1e080f1b76e9aece646ce5) )
	ROM_LOAD16_BYTE( "95411523.hi", 0x00000, 0x080000, CRC(03a57868) SHA1(3e6fe8c3355aeb016b90b8d5307da367f729a5e6) )
	PR6923_CRAZY_FRUITS_SOUNDS11
ROM_END

ROM_START( sc4cfcasae )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95411524.lo", 0x00001, 0x080000, CRC(373e90c5) SHA1(4b1647f9c415f00653546ac31f1ef57d08e579f9) )
	ROM_LOAD16_BYTE( "95411525.hi", 0x00000, 0x080000, CRC(ebaae63f) SHA1(aec39f680821c116eb0549f811a5077b0309499a) )
	PR6923_CRAZY_FRUITS_SOUNDS11
ROM_END

ROM_START( sc4cfcasaf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95411996.lo", 0x00001, 0x080000, CRC(c85682fa) SHA1(7f5cbd1bb96c0a9170916a8d0d3b6fd500d70298) )
	ROM_LOAD16_BYTE( "95411997.hi", 0x00000, 0x080000, CRC(6d805ccf) SHA1(8775f2a55ca8207216787bbea05b8dfa6c8ee4c6) )
	PR6923_CRAZY_FRUITS_SOUNDS11
ROM_END

ROM_START( sc4cfcasag )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95411998.lo", 0x00001, 0x080000, CRC(bb2ca77e) SHA1(47100cf01fbf027316fb7d139a28de86953a5338) )
	ROM_LOAD16_BYTE( "95411999.hi", 0x00000, 0x080000, CRC(0f825875) SHA1(9da6b169074194cd7f54525914bc19da98c94a51) )
	PR6923_CRAZY_FRUITS_SOUNDS11
ROM_END

ROM_START( sc4cfcasah )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95412062.lo", 0x00001, 0x080000, CRC(4c7996c2) SHA1(022ce9a036e8e3b3a1b427d151fca0592855b1ed) )
	ROM_LOAD16_BYTE( "95412063.hi", 0x00000, 0x080000, CRC(06906d31) SHA1(3b8a66f837a4f8bc8d97fee34c29b6cb877beb22) )
	PR6923_CRAZY_FRUITS_SOUNDS11
ROM_END

ROM_START( sc4cfcasai )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95413062.lo", 0x00001, 0x080000, CRC(769b9bed) SHA1(95cf5797e6ff0bb32f80a3616cca92d5ea90a69a) )
	ROM_LOAD16_BYTE( "95413063.hi", 0x00000, 0x080000, CRC(b7ad867f) SHA1(6fe236254e3dfdfe90ab00c7a7fe7676ece8dc1d) )
	PR6923_CRAZY_FRUITS_SOUNDS11
ROM_END







ROM_START( sc4cfcasaj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "p6923p31.hi", 0x00000, 0x080000, CRC(6d87d1f6) SHA1(bc91361ef0ae9b94ee9d437662beca87d5916968) )
	ROM_LOAD16_BYTE( "p6923p31.lo", 0x00001, 0x080000, CRC(21a802b6) SHA1(d2081509edfcbb889628dc910f4db1094eae64dd) )
	PR6923_CRAZY_FRUITS_SOUNDS11
ROM_END

ROM_START( sc4cfcasak )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "p6923p53.hi", 0x00000, 0x080000, CRC(aacd5356) SHA1(a91330815a61d2ca1f8e52d3ac162ab02643798b) )
	ROM_LOAD16_BYTE( "p6923p53.lo", 0x00001, 0x080000, CRC(d50b7e71) SHA1(a6682a5dcd557a2ec27651d420e7bef8459fafff) )
	PR6923_CRAZY_FRUITS_SOUNDS11
ROM_END

ROM_START( sc4cfcasal )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "p6923s31.hi", 0x00000, 0x080000, CRC(e01f2141) SHA1(cd3ecac441f6c1cf67cf2f443dc8465d5bcfc406) )
	ROM_LOAD16_BYTE( "p6923s31.lo", 0x00001, 0x080000, CRC(88d60172) SHA1(0c75342f861f8477ee3393426ca93f7d7da1dc7f) )
	PR6923_CRAZY_FRUITS_SOUNDS11
ROM_END

ROM_START( sc4cfcasam )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "p6923s53.hi", 0x00000, 0x080000, CRC(7c92adb7) SHA1(fe4e2ad8238bccb2ae1ccec4949b27dca5d49d1e) )
	ROM_LOAD16_BYTE( "p6923s53.lo", 0x00001, 0x080000, CRC(e347ca23) SHA1(2b0fadb6e3ac7439a748452b49475f26d93b2e92) )
	PR6923_CRAZY_FRUITS_SOUNDS11
ROM_END



ROM_START( sc4cfcla )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408106.lo", 0x00001, 0x080000, CRC(72b84862) SHA1(971b67eaefc19b330e58a9fe81897fdf82c3a878) )
	ROM_LOAD16_BYTE( "95408107.hi", 0x00000, 0x080000, CRC(85cc91be) SHA1(ac736c3c97e4b0898c96ddb86d3a5e3c19769a8d) )
	sc_cfcla_others
ROM_END

ROM_START( sc4cfclab )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408134.lo", 0x00001, 0x080000, CRC(77708a3d) SHA1(2e35c58b2adf6f18173e8549de4a4ca09c501b4e) )
	ROM_LOAD16_BYTE( "95408135.hi", 0x00000, 0x080000, CRC(2444a443) SHA1(2cbe1a40b1fce85ba5304ba71fd9d7af2f1e4ab1) )
	sc_cfcla_others
ROM_END

ROM_START( sc4cfclac )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408224.lo", 0x00001, 0x080000, CRC(03ec7699) SHA1(5f1abd1baca8a80a277446a9342cf2bd87baaa8a) )
	ROM_LOAD16_BYTE( "95408225.hi", 0x00000, 0x080000, CRC(0f6ea084) SHA1(6773bb4b09186f860606c576c4dcdd4a799e2c59) )
	sc_cfcla_others
ROM_END

ROM_START( sc4cfclad )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409106.lo", 0x00001, 0x080000, CRC(e0ee6148) SHA1(018df803604419c21aea2291378b57d02156cfb7) )
	ROM_LOAD16_BYTE( "95409107.hi", 0x00000, 0x080000, CRC(1f7655ba) SHA1(9e4064995419b27d61f6ada41a4890ca83fee234) )
	sc_cfcla_others
ROM_END

ROM_START( sc4cfclae )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409134.lo", 0x00001, 0x080000, CRC(c994a8f9) SHA1(855cffe9848a9b3f3b1ef5fff11a846f0d353b8a) )
	ROM_LOAD16_BYTE( "95409135.hi", 0x00000, 0x080000, CRC(369d6e9e) SHA1(3a7b9743c9be6ea2e74d99bb6b395f332c860c38) )
	sc_cfcla_others
ROM_END

ROM_START( sc4cfclaf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409224.lo", 0x00001, 0x080000, CRC(b17afef0) SHA1(8fb0111220d8446829e69592c68927bf143fb637) )
	ROM_LOAD16_BYTE( "95409225.hi", 0x00000, 0x080000, CRC(12a0e039) SHA1(ff6c41237645e41fca7208e4ea492a7b2d973144) )
	sc_cfcla_others
ROM_END




ROM_START( sc4cfclb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "cf_club.hi", 0x00000, 0x080000, CRC(ac8ef8f9) SHA1(799795a5d873aa4d896dfb632c1d95ba95e1fe6f) )
	ROM_LOAD16_BYTE( "cf_club.lo", 0x00001, 0x080000, CRC(0fd50a24) SHA1(0b1a5a1f3e879733b949a6ece2f4d872f9c63cf9) )
	sc_cfclb_others
ROM_END

ROM_START( sc4cfclba )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "club-crazy-fruits_dat_acss.hi", 0x00000, 0x080000, CRC(fdd21dab) SHA1(95784b4129d4c9cfdef9907eb74417cd9208872a) )
	ROM_LOAD16_BYTE( "club-crazy-fruits_dat_acss.lo", 0x00001, 0x080000, CRC(7a1ab4e2) SHA1(0bf11dc446855931cbd715a72c3e3977dd859e9f) )
	sc_cfclb_others
ROM_END

ROM_START( sc4cfclbb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "club-crazy-fruits_std_acss.hi", 0x00000, 0x080000, CRC(2d1327da) SHA1(3918efa25b7589798a3e8b886b8faf98e946e2b1) )
	ROM_LOAD16_BYTE( "club-crazy-fruits_std_acss.lo", 0x00001, 0x080000, CRC(7c114170) SHA1(72fa5aff35d1d9f8c8ae08eb18b8acabb47c7f10) )
	sc_cfclb_others
ROM_END





ROM_START( sc4cfdu )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404674.lo", 0x00001, 0x080000, CRC(ac99d914) SHA1(8730754989d2f370e7d5e665a210a4504cbaa85b) )
	ROM_LOAD16_BYTE( "95404675.hi", 0x00000, 0x080000, CRC(4fd4b87b) SHA1(e35fe6afaafe0825c18ef2ac6719a94439ff0989) )
	sc_cfdu_others
ROM_END

ROM_START( sc4cfdua )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405674.lo", 0x00001, 0x080000, CRC(ade7b40d) SHA1(5b9d394f03a683362d6ad01a9773f6116bc96b31) )
	ROM_LOAD16_BYTE( "95405675.hi", 0x00000, 0x080000, CRC(6fa82cc4) SHA1(93fe470fea4c0ff4889163aacec1b95b583c60f6) )
	sc_cfdu_others
ROM_END

ROM_START( sc4cfdub )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406036.lo", 0x00001, 0x080000, CRC(a5377a56) SHA1(f3779a41704c65ef3e39f706c976a3df4a86d6c1) )
	ROM_LOAD16_BYTE( "95406037.hi", 0x00000, 0x080000, CRC(98361695) SHA1(a9950ccf1689c8cae46febf87b5cd54f74c9cb2d) )
	sc_cfdu_others
ROM_END

ROM_START( sc4cfduc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407036.lo", 0x00001, 0x080000, CRC(5143b7ac) SHA1(f5cbfa141f21b698fd87f03ac8da87f7e4215d51) )
	ROM_LOAD16_BYTE( "95407037.hi", 0x00000, 0x080000, CRC(a39db700) SHA1(9699ed670396ae157fdd9c511668d21fd525b88c) )
	sc_cfdu_others
ROM_END


ROM_START( sc4cfgcl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95418104.lo", 0x00001, 0x080000, CRC(8b10b5bc) SHA1(aed9c66988f3c6aa719c66d6ef70b9ea7930122c) )
	ROM_LOAD16_BYTE( "95418105.hi", 0x00000, 0x080000, CRC(b9182340) SHA1(190232c0edb38dcc5e8c719b4cf93b4e12cde336) )
	sc_cfgcl_others
ROM_END

ROM_START( sc4cfgcla )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95418150.lo", 0x00001, 0x080000, CRC(cef9cc8c) SHA1(fd88fc99e5845247ec2d2bb1d6b113c90bf1a951) )
	ROM_LOAD16_BYTE( "95418151.hi", 0x00000, 0x080000, CRC(3850c9c6) SHA1(899e963871873d7628be8cde4fc0bd4d4686fcff) )
	sc_cfgcl_others
ROM_END

ROM_START( sc4cfgclb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95419104.lo", 0x00001, 0x080000, CRC(584167fd) SHA1(5dce3a9028695fe41d113cd8ce13abe1722bf668) )
	ROM_LOAD16_BYTE( "95419105.hi", 0x00000, 0x080000, CRC(3c8f9644) SHA1(67abbad1d54998e292c7364d0711e40da5c1b843) )
	sc_cfgcl_others
ROM_END

ROM_START( sc4cfgclc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95419150.lo", 0x00001, 0x080000, CRC(59d7f74c) SHA1(6bfeb7b2be440998d69deb722ef9d56d526bedab) )
	ROM_LOAD16_BYTE( "95419151.hi", 0x00000, 0x080000, CRC(f0f795b6) SHA1(8b4bec4061a2e28ff8b9fd58e578c77ec2227f93) )
	sc_cfgcl_others
ROM_END




ROM_START( sc4crzgn )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402452.lo", 0x00001, 0x080000, CRC(7483d8a2) SHA1(24c1749b142104d817ffbc0df706d91f0649054b) )
	ROM_LOAD16_BYTE( "95402453.hi", 0x00000, 0x080000, CRC(c48f4d13) SHA1(2f5f5b5e267c4df3cdaec12b7fda79272d3d0843) )
	sc_crzgn_others
ROM_END

ROM_START( sc4crzgna )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402676.lo", 0x00001, 0x080000, CRC(8bd3558f) SHA1(8b5c84f7bc2974f1850b0e03710f90b45ce8bd0d) )
	ROM_LOAD16_BYTE( "95402677.hi", 0x00000, 0x080000, CRC(b2c68af0) SHA1(110a09ad80d144f1d4c8ccf5feb30967b8f82237) )
	sc_crzgn_others
ROM_END

ROM_START( sc4crzgnb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403452.lo", 0x00001, 0x080000, CRC(94e3081f) SHA1(ea80369b6b39b4ec8977a7fe24cb1c1f3e70e52a) )
	ROM_LOAD16_BYTE( "95403453.hi", 0x00000, 0x080000, CRC(4187ae2f) SHA1(436ff74daa1ee45bf7b391565acecbd7da8af69d) )
	sc_crzgn_others
ROM_END

ROM_START( sc4crzgnc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403676.lo", 0x00001, 0x080000, CRC(cef89cf8) SHA1(e54ec7c619442abf7b44d1b6d50c8507121bb300) )
	ROM_LOAD16_BYTE( "95403677.hi", 0x00000, 0x080000, CRC(55368ced) SHA1(ec660706f5de64f433e73a86273a4a8e080333b9) )
	sc_crzgn_others
ROM_END

ROM_START( sc4crzgnd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404186.lo", 0x00001, 0x080000, CRC(d0997272) SHA1(fc3c5712a40fe940dd4781994e6d746da690b0bb) )
	ROM_LOAD16_BYTE( "95404187.hi", 0x00000, 0x080000, CRC(41f50ee9) SHA1(06e0af8d991eb124432b69744de50e7746c1b492) )
	sc_crzgn_others
ROM_END

ROM_START( sc4crzgne )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405186.lo", 0x00001, 0x080000, CRC(4135c3d6) SHA1(f5e3b56a5ffe3c9fd4aa670ee96b3a005457d6e3) )
	ROM_LOAD16_BYTE( "95405187.hi", 0x00000, 0x080000, CRC(89a8de4c) SHA1(4adc348fd9eb1bdd385a38435970673ac49617f0) )
	sc_crzgn_others
ROM_END

ROM_START( sc4crzgnf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406638.lo", 0x00001, 0x080000, CRC(c80a180b) SHA1(7a849df85295a4901a95642ee8265da9d928d1ac) )
	ROM_LOAD16_BYTE( "95406639.hi", 0x00000, 0x080000, CRC(96737d40) SHA1(20e84b085b8ee67c8ddf51d5c9fea6f8fdf6fca8) )
	sc_crzgn_others
ROM_END

ROM_START( sc4crzgng )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406640.lo", 0x00001, 0x080000, CRC(7a08b774) SHA1(1cb245f177760da1a1884974427dec8a241ec0d2) )
	ROM_LOAD16_BYTE( "95406641.hi", 0x00000, 0x080000, CRC(c5b8c740) SHA1(3ceb8b27e3fbbcb6d62f9873c7b7b5f1db984bf0) )
	sc_crzgn_others
ROM_END

ROM_START( sc4crzgnh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406642.lo", 0x00001, 0x080000, CRC(d7b890dc) SHA1(811484235803f0283a8af98b540cf3e7d82257e9) )
	ROM_LOAD16_BYTE( "95406643.hi", 0x00000, 0x080000, CRC(5bc90bfa) SHA1(d62a2d434ccda2ac5fdfd5503f822db94e09c7ca) )
	sc_crzgn_others
ROM_END

ROM_START( sc4crzgni )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407638.lo", 0x00001, 0x080000, CRC(bdceb0a8) SHA1(a258b8045eb06e67e5e424bd288aefe5e3c1f8e7) )
	ROM_LOAD16_BYTE( "95407639.hi", 0x00000, 0x080000, CRC(564a9d5f) SHA1(af02a0c0a4594620540907d9467d3b63005e4230) )
	sc_crzgn_others
ROM_END

ROM_START( sc4crzgnj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407640.lo", 0x00001, 0x080000, CRC(57e4df74) SHA1(1ff36d566b57a5968803abb6c3e89842294f47e7) )
	ROM_LOAD16_BYTE( "95407641.hi", 0x00000, 0x080000, CRC(43ec0214) SHA1(d7964762922b2944d9e3d6ea50418d8cb54c89a5) )
	sc_crzgn_others
ROM_END

ROM_START( sc4crzgnk )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407642.lo", 0x00001, 0x080000, CRC(3473e373) SHA1(724a21668d958e062986a1644b3a313cf8606550) )
	ROM_LOAD16_BYTE( "95407643.hi", 0x00000, 0x080000, CRC(2f42ea78) SHA1(50fe758b6f09bd5af441cf5267f00e3416db12df) )
	sc_crzgn_others
ROM_END

ROM_START( sc4crzgnl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410020.lo", 0x00001, 0x080000, CRC(7301bf72) SHA1(30d1bdb8c08ad69353b2c00b6c83801d76b9cac8) )
	ROM_LOAD16_BYTE( "95410021.hi", 0x00000, 0x080000, CRC(2e1f5465) SHA1(5e6d830183825581426a5a16dad6e076fcc1e86b) )
	sc_crzgn_others
ROM_END

ROM_START( sc4crzgnm )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410022.lo", 0x00001, 0x080000, CRC(cb7e2efa) SHA1(00eb8f7fd1a5b8a45867598c9043be3cf75dc555) )
	ROM_LOAD16_BYTE( "95410023.hi", 0x00000, 0x080000, CRC(c86873fe) SHA1(2be2445cc8a21441484d9f29bc9e64b26513389e) )
	sc_crzgn_others
ROM_END

ROM_START( sc4crzgnn )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410024.lo", 0x00001, 0x080000, CRC(77698831) SHA1(d580d24f95b8a975e7edd188355d192dd2474757) )
	ROM_LOAD16_BYTE( "95410025.hi", 0x00000, 0x080000, CRC(7c8b6324) SHA1(fcf82edbf1e11e2b054658241848f3c6031c23ac) )
	sc_crzgn_others
ROM_END

ROM_START( sc4crzgno )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410026.lo", 0x00001, 0x080000, CRC(10be06c8) SHA1(c01917e0b707fd583cc2ed7c1e3ce6c540b79880) )
	ROM_LOAD16_BYTE( "95410027.hi", 0x00000, 0x080000, CRC(10298e0c) SHA1(f4e1eda6df1d0efef68c6920936d93f08cc3ffb7) )
	sc_crzgn_others
ROM_END

ROM_START( sc4crzgnp )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410028.lo", 0x00001, 0x080000, CRC(905a72ab) SHA1(5898dccaa83d4a2437891cefb7b9408e15b44f97) )
	ROM_LOAD16_BYTE( "95410029.hi", 0x00000, 0x080000, CRC(eb406ca4) SHA1(f73104ca3fed3b8605ff03fe0d8ec212df29db45) )
	sc_crzgn_others
ROM_END

ROM_START( sc4crzgnq )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410366.lo", 0x00001, 0x080000, CRC(a8e9ade8) SHA1(8f4f9012adc35e82b1cc10b9efc8b42fb2eb8cb8) )
	ROM_LOAD16_BYTE( "95410367.hi", 0x00000, 0x080000, CRC(d8381915) SHA1(fe228ad80da15e6cadc2ac6e622dadc6d009bdf5) )
	sc_crzgn_others
ROM_END

ROM_START( sc4crzgnr )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410574.lo", 0x00001, 0x080000, CRC(8dba48b3) SHA1(67a76020720665dd15693360ce7d486273cf908e) )
	ROM_LOAD16_BYTE( "95410575.hi", 0x00000, 0x080000, CRC(8f7f617a) SHA1(fa4fa2446cb15c1b961e32d8ece39198b1f73c9a) )
	sc_crzgn_others
ROM_END

ROM_START( sc4crzgns )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410576.lo", 0x00001, 0x080000, CRC(09681a4f) SHA1(7fa5d871a485467f0076e0549e14d157807d9d83) )
	ROM_LOAD16_BYTE( "95410577.hi", 0x00000, 0x080000, CRC(745628fd) SHA1(be1ef5ba07ed9e4c9fd931e95440ba4d8db41c12) )
	sc_crzgn_others
ROM_END

ROM_START( sc4crzgnt )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410578.lo", 0x00001, 0x080000, CRC(84abd1a5) SHA1(ec77d438563a3856e3541045d5fbbd450f8e285b) )
	ROM_LOAD16_BYTE( "95410579.hi", 0x00000, 0x080000, CRC(7bb2bd0b) SHA1(4f3b59c1296383391cc7b16e3f1258019c58a7fd) )
	sc_crzgn_others
ROM_END

ROM_START( sc4crzgnu )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410580.lo", 0x00001, 0x080000, CRC(4abde8b8) SHA1(3a745514c9eb461656313949564bde999385d7a0) )
	ROM_LOAD16_BYTE( "95410581.hi", 0x00000, 0x080000, CRC(31278443) SHA1(75f49f2d20c0d6123d8d19d1cd98dc7ba7af15d8) )
	sc_crzgn_others
ROM_END

ROM_START( sc4crzgnv )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410596.lo", 0x00001, 0x080000, CRC(29aa05be) SHA1(12d3171d5ad86694ea5a80a4fe829387ceb98cdd) )
	ROM_LOAD16_BYTE( "95410597.hi", 0x00000, 0x080000, CRC(a05af0b5) SHA1(e63b8a698a3576e84ce9a0cc187bf226004ea8b9) )
	sc_crzgn_others
ROM_END

ROM_START( sc4crzgnw )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410598.lo", 0x00001, 0x080000, CRC(d8b370c9) SHA1(f61cbfa97363d62942d74c5769084ede4c4e7c3e) )
	ROM_LOAD16_BYTE( "95410599.hi", 0x00000, 0x080000, CRC(ca0e4d6a) SHA1(e3b0ee85d2fdeb959951cbd8d2b4251a9c91ad13) )
	sc_crzgn_others
ROM_END

ROM_START( sc4crzgnx )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411020.lo", 0x00001, 0x080000, NO_DUMP )
	ROM_LOAD16_BYTE( "95411021.hi", 0x00000, 0x080000, CRC(3066a181) SHA1(350cf285a756328022bae3a0bd8f13d00e29e805) )
	sc_crzgn_others
ROM_END

ROM_START( sc4crzgny )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411022.lo", 0x00001, 0x080000, CRC(4ee458ad) SHA1(7dcbca3828fe99256ecfa1ac3a9b3757717f5fee) )
	ROM_LOAD16_BYTE( "95411023.hi", 0x00000, 0x080000, CRC(5aed69d8) SHA1(53de1e5f19c8f319728fc11085ccc7db7e21cc4b) )
	sc_crzgn_others
ROM_END

ROM_START( sc4crzgnz )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411024.lo", 0x00001, 0x080000, CRC(b6f5548b) SHA1(687c228877ce4b36952ec735eca7cc3e69fce789) )
	ROM_LOAD16_BYTE( "95411025.hi", 0x00000, 0x080000, CRC(e3ab8770) SHA1(569ff2adfee8a08f96e397234dedac9af22903bd) )
	sc_crzgn_others
ROM_END

ROM_START( sc4crzgn0 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411026.lo", 0x00001, 0x080000, CRC(d8996549) SHA1(53830639574e58c8bf96fd3aaeed66e773f119ee) )
	ROM_LOAD16_BYTE( "95411027.hi", 0x00000, 0x080000, CRC(e135da8a) SHA1(cc57026825b2afc6bd21fe43adac05b199607d36) )
	sc_crzgn_others
ROM_END

ROM_START( sc4crzgn1 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411028.lo", 0x00001, 0x080000, CRC(1ac217da) SHA1(fe6043a5c7e8209bd5de87c0facd3a4e7f57196c) )
	ROM_LOAD16_BYTE( "95411029.hi", 0x00000, 0x080000, CRC(eb406ca4) SHA1(f73104ca3fed3b8605ff03fe0d8ec212df29db45) )
	sc_crzgn_others
ROM_END

ROM_START( sc4crzgn2 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411366.lo", 0x00001, 0x080000, CRC(af0c09af) SHA1(7098ba77c9800b2c288bfb9fb24ff946cc0e7c34) )
	ROM_LOAD16_BYTE( "95411367.hi", 0x00000, 0x080000, CRC(2333d361) SHA1(4bf7491df76f59a8698c817ab7e9a309a505d8dc) )
	sc_crzgn_others
ROM_END

ROM_START( sc4crzgn3 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411574.lo", 0x00001, 0x080000, CRC(ebdcb06a) SHA1(995a13cf4f57753da882e71e1023fccc017784fb) )
	ROM_LOAD16_BYTE( "95411575.hi", 0x00000, 0x080000, CRC(6810315f) SHA1(01071ad50661f4e8e2f7e0ce2e90274b57f47a3a) )
	sc_crzgn_others
ROM_END

ROM_START( sc4crzgn4 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411576.lo", 0x00001, 0x080000, CRC(d85d0230) SHA1(0e4e0b2b8067e870265d66c193166773e1f0d983) )
	ROM_LOAD16_BYTE( "95411577.hi", 0x00000, 0x080000, CRC(7853e33c) SHA1(f956dc2148453bfc0d89ace9ea5c01545ade73c9) )
	sc_crzgn_others
ROM_END

ROM_START( sc4crzgn5 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411578.lo", 0x00001, 0x080000, CRC(7498b6c3) SHA1(ee787369e09a7e90237ce3cedd86c3735ad317b2) )
	ROM_LOAD16_BYTE( "95411579.hi", 0x00000, 0x080000, CRC(90d826ef) SHA1(368a5af3bb7a3f85394fd8988515a9cd9798fb66) )
	sc_crzgn_others
ROM_END

ROM_START( sc4crzgn6 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411580.lo", 0x00001, 0x080000, CRC(2266553a) SHA1(7b11dbbe7cb6803e7ed6fbea94dd2355b7eddd82) )
	ROM_LOAD16_BYTE( "95411581.hi", 0x00000, 0x080000, CRC(5003c3ca) SHA1(f15acfc5e14dc62221626f71e32a3788e27d6a73) )
	sc_crzgn_others
ROM_END

ROM_START( sc4crzgn7 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411596.lo", 0x00001, 0x080000, CRC(1a7b740e) SHA1(61121d9c883f36b610ec4b9b0f49fa7dadece011) )
	ROM_LOAD16_BYTE( "95411597.hi", 0x00000, 0x080000, CRC(12632fe5) SHA1(aa4db2387fa88f1ec7150c128ab42f5e568f8bca) )
	sc_crzgn_others
ROM_END

ROM_START( sc4crzgn8 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411598.lo", 0x00001, 0x080000, CRC(4b632ca0) SHA1(2e44df78cf5838f13ab16aa065be1214b9a203d5) )
	ROM_LOAD16_BYTE( "95411599.hi", 0x00000, 0x080000, CRC(42a29fdf) SHA1(4c38e1da31cfd9043bd46b8b549ae74afcfdb497) )
	sc_crzgn_others
ROM_END




ROM_START( sc4crzky )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406634.lo", 0x00001, 0x080000, CRC(777a328a) SHA1(9bfe4fb4ebdc89d43be9d0a11f5a4ccfc2191e12) )
	ROM_LOAD16_BYTE( "95406635.hi", 0x00000, 0x080000, CRC(f8b6e2c3) SHA1(61a7f607c0ab74d82fc04827d4381ced108b9b83) )
	sc_crzky_others
ROM_END

ROM_START( sc4crzkya )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406722.lo", 0x00001, 0x080000, CRC(ebcb20ea) SHA1(9be110af57919efbd9cdff7d89746b557e3e2df0) )
	ROM_LOAD16_BYTE( "95406723.hi", 0x00000, 0x080000, CRC(76613d1b) SHA1(2fed1c09fce5af13d88ab570612fb2f756dabbc8) )
	sc_crzky_others
ROM_END

ROM_START( sc4crzkyb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406802.lo", 0x00001, 0x080000, CRC(2e62ebb1) SHA1(d946ef00f0bb15529e744bcd5f0b83abd135f176) )
	ROM_LOAD16_BYTE( "95406803.hi", 0x00000, 0x080000, CRC(d84dc003) SHA1(fc7dd44bda2ce3ddd23559c226dd81adaea9dba5) )
	sc_crzky_others
ROM_END

ROM_START( sc4crzkyc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407634.lo", 0x00001, 0x080000, CRC(24152e9a) SHA1(8fba1c2d6c51c416fdf6e2ca309a163584647633) )
	ROM_LOAD16_BYTE( "95407635.hi", 0x00000, 0x080000, CRC(67c8ce9d) SHA1(5aa4de657e4fceb85b37e8207b5f852ef0aa22fb) )
	sc_crzky_others
ROM_END

ROM_START( sc4crzkyd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407722.lo", 0x00001, 0x080000, CRC(a58ec2c5) SHA1(0c7582062dc95b4e1cf805d7cd2c9f9364ee006d) )
	ROM_LOAD16_BYTE( "95407723.hi", 0x00000, 0x080000, CRC(9fd991f0) SHA1(e25271116a40b94ff54bcd2230fddf0d880df478) )
	sc_crzky_others
ROM_END

ROM_START( sc4crzkye )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407802.lo", 0x00001, 0x080000, CRC(381a8b19) SHA1(79f899e432d246a7f47fca34204fbfec6cacc6a9) )
	ROM_LOAD16_BYTE( "95407803.hi", 0x00000, 0x080000, CRC(e38e9009) SHA1(9cc8ba488711b9f6cb2d597abdd10c46d2113e39) )
	sc_crzky_others
ROM_END

ROM_START( sc4crzkyf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410042.lo", 0x00001, 0x080000, CRC(cc2479f3) SHA1(8ccebd62df3d69b6ceb252883ba209902eb17060) )
	ROM_LOAD16_BYTE( "95410043.hi", 0x00000, 0x080000, CRC(dc52c32b) SHA1(0b6c0a2c80b73592f70b29f7175c831bf6716bd3) )
	sc_crzky_others
ROM_END

ROM_START( sc4crzkyg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410044.lo", 0x00001, 0x080000, CRC(7ca79549) SHA1(c74fb92f192d9d8bf60ff38a126a3876703385a0) )
	ROM_LOAD16_BYTE( "95410045.hi", 0x00000, 0x080000, NO_DUMP )
	sc_crzky_others
ROM_END

ROM_START( sc4crzkyh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410370.lo", 0x00001, 0x080000, CRC(e53a9a74) SHA1(71a3a3762703a7fa06fd51d1d70e50578a726dcc) )
	ROM_LOAD16_BYTE( "95410371.hi", 0x00000, 0x080000, CRC(0e8bb647) SHA1(785bc1a8b379bf75c31c55aaf35b6c0189946af3) )
	sc_crzky_others
ROM_END

ROM_START( sc4crzkyi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410566.lo", 0x00001, 0x080000, CRC(3a052a51) SHA1(c98f8b53188716ab674b3c2839cf6bc837c88ec2) )
	ROM_LOAD16_BYTE( "95410567.hi", 0x00000, 0x080000, CRC(f6db5850) SHA1(c78ccda4f5f86406f6e5c059980c548a7b0d9751) )
	sc_crzky_others
ROM_END

ROM_START( sc4crzkyj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410568.lo", 0x00001, 0x080000, CRC(3d528d96) SHA1(8b22b4ea0142e33756c92ec232a32a24961810f1) )
	ROM_LOAD16_BYTE( "95410569.hi", 0x00000, 0x080000, CRC(ad86af8e) SHA1(1bf37e6308612dd03e92f4b95b3525b39c007aa8) )
	sc_crzky_others
ROM_END

ROM_START( sc4crzkyk )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410600.lo", 0x00001, 0x080000, CRC(e6445e23) SHA1(71c22c6968673e773c749f34c178ae8424c2b157) )
	ROM_LOAD16_BYTE( "95410601.hi", 0x00000, 0x080000, CRC(2f965559) SHA1(e5ef95e18ff6a0eb953853ea485613876fdff7a2) )
	sc_crzky_others
ROM_END

ROM_START( sc4crzkyl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411042.lo", 0x00001, 0x080000, CRC(17d4aa90) SHA1(20dbd3315cc0dc24a7d00b8709f6ce8821d1b74b) )
	ROM_LOAD16_BYTE( "95411043.hi", 0x00000, 0x080000, CRC(b4116fb1) SHA1(70d728ad4c2a8810465d4b2f4f922c455fa36070) )
	sc_crzky_others
ROM_END

ROM_START( sc4crzkym )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411044.lo", 0x00001, 0x080000, CRC(38cc4eed) SHA1(f3b3de3913787a6459af00f2a17c63eaf757dec3) )
	ROM_LOAD16_BYTE( "95411045.hi", 0x00000, 0x080000, CRC(8524d0e4) SHA1(e0c654a8e1109a5faa3b18e8609a9f4d4a780364) )
	sc_crzky_others
ROM_END

ROM_START( sc4crzkyn )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411370.lo", 0x00001, 0x080000, CRC(4b0ddf07) SHA1(bb99fae2bc11cb6ce23aba74b41d20b0fce38ad3) )
	ROM_LOAD16_BYTE( "95411371.hi", 0x00000, 0x080000, CRC(4d22117f) SHA1(19a9aa05989d41a4485d4bce71989046d3ec2c67) )
	sc_crzky_others
ROM_END

ROM_START( sc4crzkyo )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411566.lo", 0x00001, 0x080000, CRC(74b44419) SHA1(5a3474b39a626567ced29aa5d3e62803b755445f) )
	ROM_LOAD16_BYTE( "95411567.hi", 0x00000, 0x080000, CRC(a051758d) SHA1(20d0e0ebd855e84c5c0ee49a3fff7264c07a47ac) )
	sc_crzky_others
ROM_END

ROM_START( sc4crzkyp )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411568.lo", 0x00001, 0x080000, CRC(73e3e3de) SHA1(d4154b83dac8045d7a578c06ec98a74c79e6d6a1) )
	ROM_LOAD16_BYTE( "95411569.hi", 0x00000, 0x080000, CRC(fb0c8253) SHA1(e9ed1d6ddfa7cb52906b1d0873a1d768c6bb7d50) )
	sc_crzky_others
ROM_END

ROM_START( sc4crzkyq )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411600.lo", 0x00001, 0x080000, CRC(9db954e7) SHA1(ac2a4a2151dd59c1bf18735ff00909cfc869d897) )
	ROM_LOAD16_BYTE( "95411601.hi", 0x00000, 0x080000, CRC(e7c3ae92) SHA1(125f29c228b745828ec74bb964e84f4d16bf728d) )
	sc_crzky_others
ROM_END

ROM_START( sc4crzkyr )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412394.lo", 0x00001, 0x080000, CRC(6f8aac4f) SHA1(9b513aaf02e062499245592da17e7ce78f794c99) )
	ROM_LOAD16_BYTE( "95412395.hi", 0x00000, 0x080000, CRC(680ea9bc) SHA1(9e790db3d856a66a70d1f2432da646444a936d8c) )
	sc_crzky_others
ROM_END

ROM_START( sc4crzkys )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412396.lo", 0x00001, 0x080000, CRC(688dd067) SHA1(fd8cae7da036f741a31a2d40f3e99db499d9fba5) )
	ROM_LOAD16_BYTE( "95412397.hi", 0x00000, 0x080000, CRC(e82f0e8d) SHA1(4158a5aa70324ef18a1cf9358b4d1a9ae3684553) )
	sc_crzky_others
ROM_END

ROM_START( sc4crzkyt )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95412768.lo", 0x00001, 0x080000, CRC(8c0e64b9) SHA1(e67d84d384d4c5bb43c45768d538048fc456b90f) )
	ROM_LOAD16_BYTE( "95412769.hi", 0x00000, 0x080000, CRC(e474d8fb) SHA1(55457253310084b51a7f0b3ad55cd56699505e66) )
	sc_crzky_others
ROM_END

ROM_START( sc4crzkyu )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95412770.lo", 0x00001, 0x080000, CRC(3d588dea) SHA1(0ccb0f2e4997340602ac095b94eb796a289e1125) )
	ROM_LOAD16_BYTE( "95412771.hi", 0x00000, 0x080000, CRC(38deb018) SHA1(5ee9403b969dcccbc7387071923eb3bc4fab8c86) )
	sc_crzky_others
ROM_END

ROM_START( sc4crzkyv )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95413394.lo", 0x00001, 0x080000, CRC(482b8cd6) SHA1(aca9f1ac29d8323b215ed6b62621e357d03e6b60) )
	ROM_LOAD16_BYTE( "95413395.hi", 0x00000, 0x080000, CRC(fecb656d) SHA1(99f52e5c61e71f032e579bc2bc08f415350459f8) )
	sc_crzky_others
ROM_END



ROM_START( sc4crzkyw )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95413396.lo", 0x00001, 0x080000, CRC(02cc445f) SHA1(2ca7589b403e2308d11de81400c4bd5811a3d1c1) )
	ROM_LOAD16_BYTE( "95413397.hi", 0x00000, 0x080000, CRC(88774042) SHA1(3c75da111da7bdbf68bcbc08fc6db26122b04b0b) )
	sc_crzky_others
ROM_END


ROM_START( sc4crzwl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400143.lo", 0x00001, 0x080000, CRC(e3a08e29) SHA1(9e85b34197aa67a1e60e610dfcf4cf900cc473ab) )
	ROM_LOAD16_BYTE( "95400144.hi", 0x00000, 0x080000, CRC(a5bf5824) SHA1(e41652a84d449f85eefaeed4ebc59cd95b35d2ac) )
	sc_crzwl_others
ROM_END

ROM_START( sc4crzwla )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400147.lo", 0x00001, 0x080000, CRC(3e6b5552) SHA1(a6aa55fd7aa718190a6e9b01b0f6c0501b49e3d2) )
	ROM_LOAD16_BYTE( "95400148.hi", 0x00000, 0x080000, CRC(ee857272) SHA1(5760b56164071c0529a0c369d190666dd07949e9) )
	sc_crzwl_others
ROM_END

ROM_START( sc4crzwlb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400221.lo", 0x00001, 0x080000, CRC(670b73c5) SHA1(5145e1fd89decb46bf7654edf136c6549913e132) )
	ROM_LOAD16_BYTE( "95400222.hi", 0x00000, 0x080000, CRC(63bff045) SHA1(51252cf325ddf70fe48a4d6b1728d1f3f8d94025) )
	sc_crzwl_others
ROM_END

ROM_START( sc4crzwlc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400575.lo", 0x00001, 0x080000, CRC(966f3bd7) SHA1(d353e3e06d9bd6b27265352a9b9c6522fa5cda21) )
	ROM_LOAD16_BYTE( "95400576.hi", 0x00000, 0x080000, CRC(6f593a71) SHA1(9798eebf143c5f06026df321a17d89c2d3df6f62) )
	sc_crzwl_others
ROM_END

ROM_START( sc4crzwld )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401143.lo", 0x00001, 0x080000, CRC(d1187eb7) SHA1(aa2f1c4b295cc6856b5a70fd959aa054d15702e9) )
	ROM_LOAD16_BYTE( "95401144.hi", 0x00000, 0x080000, CRC(74df981d) SHA1(65092734a13b24040078094b6c69a58379d5ebc1) )
	sc_crzwl_others
ROM_END

ROM_START( sc4crzwle )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401147.lo", 0x00001, 0x080000, CRC(c88395c6) SHA1(20efe173b8487b8a333540065ee87f406a1cbcb3) )
	ROM_LOAD16_BYTE( "95401148.hi", 0x00000, 0x080000, CRC(f5cd1160) SHA1(b32415eb5aa5f6125d39f3eb371d369b12c61a50) )
	sc_crzwl_others
ROM_END

ROM_START( sc4crzwlf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401221.lo", 0x00001, 0x080000, CRC(41ced3a9) SHA1(e240ce8ed0c0b9fca61211522850f104e856069d) )
	ROM_LOAD16_BYTE( "95401222.hi", 0x00000, 0x080000, CRC(768f0076) SHA1(ee42b8b794b700855c03d67e565363b31f8ffa6d) )
	sc_crzwl_others
ROM_END

ROM_START( sc4crzwlg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401575.lo", 0x00001, 0x080000, CRC(f1c3a87a) SHA1(8ae813b2f232dd1c0b2943506cac59470371ce32) )
	ROM_LOAD16_BYTE( "95401576.hi", 0x00000, 0x080000, CRC(f0b84017) SHA1(124bc1653f33a52aafad11d6fe9e8990bf7765fe) )
	sc_crzwl_others
ROM_END




ROM_START( sc4cccsh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400693.lo", 0x00001, 0x080000, CRC(61896fc5) SHA1(416e97956b5591820c8f51126f823efc0351fe1f) )
	ROM_LOAD16_BYTE( "95400694.hi", 0x00000, 0x080000, CRC(05d6f641) SHA1(0936aa69368e895f21c9ffd51db1dc548c39bd67) )
	sc_cccsh_others
ROM_END

ROM_START( sc4cccsha )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400695.lo", 0x00001, 0x080000, CRC(ca6ed067) SHA1(84e3519c692b0e461038f5e4ec2ac6a6027138f7) )
	ROM_LOAD16_BYTE( "95400696.hi", 0x00000, 0x080000, CRC(455a774a) SHA1(7f0eb430416d954f48de62747258d85f20215664) )
	sc_cccsh_others
ROM_END

ROM_START( sc4cccshb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400837.lo", 0x00001, 0x080000, CRC(eb40ea39) SHA1(56c9a00288149458145c29f2677f9b4eef208f51) )
	ROM_LOAD16_BYTE( "95400838.hi", 0x00000, 0x080000, CRC(062c8198) SHA1(b53101722982075779f02ad189a9a7600678eebc) )
	sc_cccsh_others
ROM_END

ROM_START( sc4cccshc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401693.lo", 0x00001, 0x080000, CRC(e7837b9e) SHA1(6017362906e49b7fa254d183282b6e1ff4c28102) )
	ROM_LOAD16_BYTE( "95401694.hi", 0x00000, 0x080000, CRC(cd1c5bb2) SHA1(7a682252655d67baa0da896af123b23fc1b170fc) )
	sc_cccsh_others
ROM_END

ROM_START( sc4cccshd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401695.lo", 0x00001, 0x080000, CRC(38ef25be) SHA1(ac4923e7466e1b254daed43099ea38eba7d70437) )
	ROM_LOAD16_BYTE( "95401696.hi", 0x00000, 0x080000, CRC(25409cb3) SHA1(fe1216a1237f65635f3cc5584a3cdeedee6f7b88) )
	sc_cccsh_others
ROM_END

ROM_START( sc4cccshe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401837.lo", 0x00001, 0x080000, CRC(8a25ae47) SHA1(ebf00a93e1707070701a77adcb71f47b16bdb99e) )
	ROM_LOAD16_BYTE( "95401838.hi", 0x00000, 0x080000, CRC(99cdfbfe) SHA1(5193d050d4d295f856b0a5a4163c07449a869edf) )
	sc_cccsh_others
ROM_END




ROM_START( sc4ccc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406348.lo", 0x00001, 0x080000, CRC(1c6a6eaa) SHA1(b053cbb9756a36f1a775d18316c9fc305cd97978) )
	ROM_LOAD16_BYTE( "95406349.hi", 0x00000, 0x080000, CRC(c9742e04) SHA1(0662126a8b05bfc3e60bb4a3c659070814f73305) )
	sc_ccc_others
ROM_END









ROM_START( sc4daylt )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400034.p2", 0x00001, 0x080000, CRC(eba8544f) SHA1(cd806adb836d0b6ed3eb16cdcd826ea47efbb9d9) )
	ROM_LOAD16_BYTE( "95400033.p1", 0x00000, 0x080000, CRC(7b8dffcd) SHA1(200323dfdc92faf6853d70583a56e3b37b159f1c) )
	sc_daylt_others
ROM_END

ROM_START( sc4daylta )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400036.p2", 0x00001, 0x080000, CRC(edc97be8) SHA1(a0cf8d63fa34e724ee89e221cf1aa80ce02598c1) )
	ROM_LOAD16_BYTE( "95400035.p1", 0x00000, 0x080000, CRC(775b57e2) SHA1(6a3bc25ecca0cab421092661cb9d622c0934d042) )
	sc_daylt_others
ROM_END

ROM_START( sc4dayltb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400043.lo", 0x00001, 0x080000, CRC(56fb448b) SHA1(009712cd122d10cdb4dfbf472eaaefb879935fce) )
	ROM_LOAD16_BYTE( "95400044.hi", 0x00000, 0x080000, CRC(7f39c4c6) SHA1(09114c34d5ae778494f29a5696725787e4d7c294) )
	sc_daylt_others
ROM_END

ROM_START( sc4dayltc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400045.lo", 0x00001, 0x080000, CRC(6048bbd4) SHA1(d14137a7ff99abfe49ef11e30d4f693eb4c90960) )
	ROM_LOAD16_BYTE( "95400046.hi", 0x00000, 0x080000, CRC(212e9e63) SHA1(41a4cd3f02a96419e0ccd90b1e2c94b3ceddc36a) )
	sc_daylt_others
ROM_END

ROM_START( sc4dayltd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400467.lo", 0x00001, 0x080000, CRC(1e654bb1) SHA1(0a51059d8e4e67b64ec3efe2d09684c954ef3b2f) )
	ROM_LOAD16_BYTE( "95400468.hi", 0x00000, 0x080000, CRC(59e20e36) SHA1(4e90679a7fd4a2a686f628d43e4579b54e0f65b7) )
	sc_daylt_others
ROM_END

ROM_START( sc4daylte )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400469.lo", 0x00001, 0x080000, CRC(475554d3) SHA1(1979061e8abf24a7e36053ad206b8dcb92bba864) )
	ROM_LOAD16_BYTE( "95400470.hi", 0x00000, 0x080000, CRC(0f19fa57) SHA1(76aaba9e09fb7d545a8cf7082604d0c462243410) )
	sc_daylt_others
ROM_END

ROM_START( sc4dayltf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400523.lo", 0x00001, 0x080000, CRC(3b4d349d) SHA1(954b47fa04737d73914c09712d58007de9660fc0) )
	ROM_LOAD16_BYTE( "95400524.hi", 0x00000, 0x080000, CRC(19a92a95) SHA1(0dcc4e22e80bea3ddc3d0f1df37340d319bf05c4) )
	sc_daylt_others
ROM_END

ROM_START( sc4dayltg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401034.p2", 0x00001, 0x080000, CRC(b14778f1) SHA1(7dea26452d7f30a0f1f16c1ad82e2e5362311ea7) )
	ROM_LOAD16_BYTE( "95401033.p1", 0x00000, 0x080000, CRC(c4101114) SHA1(12db5bde5f2f5ed196205def4c94c2d9c7f01bbd) )
	sc_daylt_others
ROM_END

ROM_START( sc4daylth )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401036.p2", 0x00001, 0x080000, CRC(ae8c3a9c) SHA1(c79ad8ac58971532f4dbbad555928651c3b33b30) )
	ROM_LOAD16_BYTE( "95401035.p1", 0x00000, 0x080000, CRC(7487f54a) SHA1(2f9f2f39e7d23fe945d1ee12472003104a08fbde) )
	sc_daylt_others
ROM_END

ROM_START( sc4daylti )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401043.lo", 0x00001, 0x080000, CRC(025dd133) SHA1(3e489f1d9e521b2fcec201eb8988168978d9b48b) )
	ROM_LOAD16_BYTE( "95401044.hi", 0x00000, 0x080000, CRC(6a9662cd) SHA1(014fa14fc756cca096b57d56f0fdf660b316a669) )
	sc_daylt_others
ROM_END

ROM_START( sc4dayltj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401045.lo", 0x00001, 0x080000, CRC(188cf175) SHA1(4771852ac5ece7a40e0ab434d3126c278bb71a56) )
	ROM_LOAD16_BYTE( "95401046.hi", 0x00000, 0x080000, CRC(8b838089) SHA1(703ce9992b66c38da83dc1af29c8ec39b2067261) )
	sc_daylt_others
ROM_END

ROM_START( sc4dayltk )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401467.lo", 0x00001, 0x080000, CRC(fe7ced21) SHA1(88da3772f7af86ec8c24b4e428dbe6a0faf33139) )
	ROM_LOAD16_BYTE( "95401468.hi", 0x00000, 0x080000, CRC(be12082b) SHA1(694a534046f0595562e370b18db60a7c40e84e57) )
	sc_daylt_others
ROM_END

ROM_START( sc4dayltl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401469.lo", 0x00001, 0x080000, CRC(39932455) SHA1(21d7a1653484811e344eaa8e568abcf48270d220) )
	ROM_LOAD16_BYTE( "95401470.hi", 0x00000, 0x080000, CRC(3acac8e3) SHA1(7b2289a1b011cf7ad0f777871c78525f98fd5ac7) )
	sc_daylt_others
ROM_END

ROM_START( sc4dayltm )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401523.lo", 0x00001, 0x080000, CRC(db54920d) SHA1(1ac7b7e011b27528404c7e2ef42796b23f015eba) )
	ROM_LOAD16_BYTE( "95401524.hi", 0x00000, 0x080000, CRC(fe592c88) SHA1(16b79165f2847c74e6708b17be6cc725382129d9) )
	sc_daylt_others
ROM_END



















ROM_START( sc4deepi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408252.lo", 0x00001, 0x080000, CRC(c7f6892f) SHA1(0792b3e59812f7da515c3e95d5ed33eea3b46712) )
	ROM_LOAD16_BYTE( "95408253.hi", 0x00000, 0x080000, CRC(adbf060e) SHA1(2cdda24e129dc24e1d38ad3b5889fa6d9335f1f4) )
	sc_deepi_others
ROM_END

ROM_START( sc4deepia )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408266.lo", 0x00001, 0x080000, CRC(9d81fef2) SHA1(2e6688fe232932425c29cfce26532c85241a802e) )
	ROM_LOAD16_BYTE( "95408267.hi", 0x00000, 0x080000, CRC(7943c442) SHA1(9913b97fcd7751352b86a5cf2059daefe063d362) )
	sc_deepi_others
ROM_END

ROM_START( sc4deepib )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408522.lo", 0x00001, 0x080000, CRC(65039069) SHA1(7bb43c5df91bc364103c9f0201fd1bfb19d72b3d) )
	ROM_LOAD16_BYTE( "95408523.hi", 0x00000, 0x080000, CRC(4e49a56a) SHA1(4f7b973b93944d8dc6b20f4808970193049f322b) )
	sc_deepi_others
ROM_END


ROM_START( sc4deepid )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409266.lo", 0x00001, 0x080000, CRC(9fe37468) SHA1(f9590784fc42f4e87be3baa83f115d734aa9d69c) )
	ROM_LOAD16_BYTE( "95409267.hi", 0x00000, 0x080000, CRC(02114ca9) SHA1(fc0a350785053ac662c0b0f9af95ebf5a0fa7d1c) )
	sc_deepi_others
ROM_END




ROM_START( sc4derby )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404036.lo", 0x00001, 0x080000, CRC(91349f85) SHA1(10e8b89191736e3b75db992c0fda6cd892dd90dc) )
	ROM_LOAD16_BYTE( "95404037.hi", 0x00000, 0x080000, CRC(bea0e501) SHA1(c88a6f2e1b2c59d6d9244a2143658ba014a64fbc) )
	sc_derby_others
ROM_END

ROM_START( sc4derbya )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404038.lo", 0x00001, 0x080000, CRC(4615673e) SHA1(a54dfbdb6b23088f6917404ede04346d43cc26a0) )
	ROM_LOAD16_BYTE( "95404039.hi", 0x00000, 0x080000, CRC(04cc451f) SHA1(5dd238e0f252403a691875878d65ebe89835c02e) )
	sc_derby_others
ROM_END

ROM_START( sc4derbyb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405036.lo", 0x00001, 0x080000, CRC(c809ce64) SHA1(8162bf6187cebb9c27fb77fca02799033d440ed7) )
	ROM_LOAD16_BYTE( "95405037.hi", 0x00000, 0x080000, CRC(0da79621) SHA1(9f4a608acc3b6189267433125afa7c684bd3102d) )
	sc_derby_others
ROM_END

ROM_START( sc4derbyc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405038.lo", 0x00001, 0x080000, CRC(b6c6dca5) SHA1(00c13f81cbe5e97f1b4a4eb2c7094d32015a2702) )
	ROM_LOAD16_BYTE( "95405039.hi", 0x00000, 0x080000, CRC(ae615bf5) SHA1(2ac593015297285852291ef2a31baa8a5a22cd41) )
	sc_derby_others
ROM_END

ROM_START( sc4derbyd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406004.lo", 0x00001, 0x080000, CRC(ecf75c7f) SHA1(52c3e3711ca9aee6b2cf53aea787319d4730520d) )
	ROM_LOAD16_BYTE( "95406005.hi", 0x00000, 0x080000, CRC(b3dc324b) SHA1(744dcc8e8d28e84ab698fa87af6cc96c379eac5d) )
	sc_derby_others
ROM_END

ROM_START( sc4derbye )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407004.lo", 0x00001, 0x080000, CRC(95ae5b75) SHA1(06ec9d10eddbb28642a86a6f95aa31e3b9919329) )
	ROM_LOAD16_BYTE( "95407005.hi", 0x00000, 0x080000, CRC(bf46afb2) SHA1(df6c1dfa93584327df1dd345424984880ce55c19) )
	sc_derby_others
ROM_END








ROM_START( sc4druby )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1439p41.hi", 0x00000, 0x080000, CRC(d90e997b) SHA1(ddf9065bc5cd3710cd900028890e5a1825bf1513) )
	ROM_LOAD16_BYTE( "p1439p41.lo", 0x00001, 0x080000, CRC(20e7ae21) SHA1(e499b8593e5ded1b7e4dbe906fe1b38cb9eff413) )
	sc_druby_others
ROM_END


ROM_START( sc4drubya )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1439p51.hi", 0x00000, 0x080000, CRC(edf42b2a) SHA1(3dde924c44b18a6ab1980f313f20d0c474b43f5a) )
	ROM_LOAD16_BYTE( "p1439p51.lo", 0x00001, 0x080000, CRC(dd701d6c) SHA1(8d0ae2f3426cee81fa3941240312fc3bcf66c2b4) )
	sc_druby_others
ROM_END


ROM_START( sc4drubyb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1439s40.hi", 0x00000, 0x080000, NO_DUMP )
	ROM_LOAD16_BYTE( "p1439s40.lo", 0x00001, 0x080000, CRC(c4b2786d) SHA1(6c6d46cd1d4645a8eb89915c07a8844ce4ccdb26) )
	sc_druby_others
ROM_END


ROM_START( sc4drubyc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1439s41.hi", 0x00000, 0x080000, CRC(d36c4b18) SHA1(2048183e83032af1653e7dc294c6c7ca501d55fe) )
	ROM_LOAD16_BYTE( "p1439s41.lo", 0x00001, 0x080000, CRC(c4b2786d) SHA1(6c6d46cd1d4645a8eb89915c07a8844ce4ccdb26) )
	sc_druby_others
ROM_END


ROM_START( sc4drubyd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1439s51.hi", 0x00000, 0x080000, CRC(e796f949) SHA1(aaffdbbc56f01311ac3dbc04c2a90a7d6f25e7ac) )
	ROM_LOAD16_BYTE( "p1439s51.lo", 0x00001, 0x080000, CRC(3925cb20) SHA1(e9f9ebbad35461f616aae4bf6ab553a58b63d18a) )
	sc_druby_others
ROM_END




ROM_START( sc4disco )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400955.lo", 0x00001, 0x080000, CRC(62c923ef) SHA1(e1590557d5fd0927108ce1b1a134c7ee76525566) )
	ROM_LOAD16_BYTE( "95400956.hi", 0x00000, 0x080000, CRC(728ab93f) SHA1(554b6208074f6ef2bf9045569da569c179c84201) )
	sc_disco_others
ROM_END

ROM_START( sc4discoa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402271.lo", 0x00001, 0x080000, CRC(d93a7b38) SHA1(c15eb45392ecbe2f7596042a1737cc2cecea066d) )
	ROM_LOAD16_BYTE( "95402272.hi", 0x00000, 0x080000, CRC(1902724f) SHA1(c6f6fa5eba7a4cf043f047a2113cbd5c0139a1ef) )
	sc_disco_others
ROM_END

ROM_START( sc4discob )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403341.lo", 0x00001, 0x080000, CRC(6913bf28) SHA1(c721933d63de2b0d5ba4b68541801f3e7677086a) )
	ROM_LOAD16_BYTE( "95403342.hi", 0x00000, 0x080000, CRC(ce9d6c1e) SHA1(68d31f3e003c30d11a349c51823407eccf7f2534) )
	sc_disco_others
ROM_END

ROM_START( sc4discoc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404638.lo", 0x00001, 0x080000, CRC(e088d182) SHA1(51eaa23cab3455412ec5e8407f75f9226e92dbd2) )
	ROM_LOAD16_BYTE( "95404639.hi", 0x00000, 0x080000, CRC(7080c3b5) SHA1(fc8916f5d39dd02083d77e123818cf794d674ff3) )
	sc_disco_others
ROM_END

ROM_START( sc4discod )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405638.lo", 0x00001, 0x080000, CRC(69ed0c5e) SHA1(a90665bf635cb71977cf026fd6a00440a16f3696) )
	ROM_LOAD16_BYTE( "95405639.hi", 0x00000, 0x080000, CRC(face1fd8) SHA1(a830e5fc2f8171296585b3431d45d9a25c6554fb) )
	sc_disco_others
ROM_END







ROM_START( sc4dbldm )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416168.odd", 0x00001, 0x080000, CRC(7724358b) SHA1(ffd57d13d08d05287382e07a433037d6cd002d1e) )
	ROM_LOAD16_BYTE( "95416169.evn", 0x00000, 0x080000, CRC(87b9ce5c) SHA1(b5f9bc9b7428c5537e9d7b2e1adc07a0c78422b2) )
	sc_dbldm_others
ROM_END




ROM_START( sc4dblfr )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406316.odd", 0x00001, 0x080000, CRC(748869bb) SHA1(14c4eac2af716d4c0f281649392a0722a71f08dc) )
	ROM_LOAD16_BYTE( "95406317.evn", 0x00000, 0x080000, CRC(2108c5cd) SHA1(08ef8ef16f0ada502b4f18d05cb63219af3bca29) )
	sc_dblfr_others
ROM_END

ROM_START( sc4dblfra )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406446.odd", 0x00001, 0x080000, CRC(62766923) SHA1(e57ecc5003b82c3a63d509a60da6c1d3a1edc9bb) )
	ROM_LOAD16_BYTE( "95406447.evn", 0x00000, 0x080000, CRC(fe684a83) SHA1(496974fb88856d618849d02f2cc38b0ab633fe02) )
	sc_dblfr_others
ROM_END

ROM_START( sc4dblfrb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407316.odd", 0x00001, 0x080000, CRC(74cf5619) SHA1(2095bb73ab69c7c9f187ec0f3b1a0914da23326c) )
	ROM_LOAD16_BYTE( "95407317.evn", 0x00000, 0x080000, CRC(36da3b26) SHA1(9331c2e8b7ede871e94e918d8071ca6d12c81998) )
	sc_dblfr_others
ROM_END

ROM_START( sc4dblfrc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407446.odd", 0x00001, 0x080000, CRC(62315681) SHA1(0c90b9e919060848bb2b04fb250bf55f841d70c7) )
	ROM_LOAD16_BYTE( "95407447.evn", 0x00000, 0x080000, CRC(e9bab468) SHA1(66a960247792e94039a8e3b56377b480e487fec2) )
	sc_dblfr_others
ROM_END

ROM_START( sc4dblfrd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416680.lo", 0x00001, 0x080000, CRC(f37bc848) SHA1(4e842b8e48432b7cb8aae47ddebfe01eafc112ce) )
	ROM_LOAD16_BYTE( "95416681.hi", 0x00000, 0x080000, CRC(b93dc6c4) SHA1(52044ea62a3edd2009d7c18da285009e75ee817c) )
	sc_dblfr_others
ROM_END

ROM_START( sc4dblfre )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417680.lo", 0x00001, 0x080000, CRC(d60c67d5) SHA1(f3b66d89c4cc625c6497a3eeabd133df7f00384a) )
	ROM_LOAD16_BYTE( "95417681.hi", 0x00000, 0x080000, CRC(a5937541) SHA1(90283dee8352fdbc67c132b01c7306cae58a8004) )
	sc_dblfr_others
ROM_END




ROM_START( sc4darw )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404764.odd", 0x00001, 0x080000, CRC(dabfaa87) SHA1(4225d5ed0985f727ec3899134ff78c3f7a9066c9) )
	ROM_LOAD16_BYTE( "95404765.evn", 0x00000, 0x080000, CRC(2febc6c8) SHA1(1fb4f4f609a0159f97c07fef5ca1c6b1f8788da2) )
	sc_darw_others
ROM_END

ROM_START( sc4darwa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405764.odd", 0x00001, 0x080000, CRC(a4f5508d) SHA1(baadfd23afb0eb2d97eada9860a1b7dc7c9afe6b) )
	ROM_LOAD16_BYTE( "95405765.evn", 0x00000, 0x080000, CRC(e5c365e3) SHA1(5e151aad3f4645b74f3f8f221e6ab48adfc4e8f1) )
	sc_darw_others
ROM_END

ROM_START( sc4darwb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406116.odd", 0x00001, 0x080000, CRC(be00430f) SHA1(f71dbd4ecca434893950f035775b702f99632cec) )
	ROM_LOAD16_BYTE( "95406117.evn", 0x00000, 0x080000, CRC(519cf9ab) SHA1(18cfa58f919247f9b4d9f035723dee9eed43c688) )
	sc_darw_others
ROM_END

ROM_START( sc4darwc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407116.odd", 0x00001, 0x080000, CRC(f3487dcc) SHA1(6a7226421397fee2582a3800b2cd9d514605327d) )
	ROM_LOAD16_BYTE( "95407117.evn", 0x00000, 0x080000, CRC(3ec02d03) SHA1(1d12632bfb54e46dee45acf04148d9dafc19d2ed) )
	sc_darw_others
ROM_END












ROM_START( sc4duckq )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408196.odd", 0x00001, 0x080000, CRC(59931e53) SHA1(adcf354276620862cb7fd39051747a3f53e4c07a) )
	ROM_LOAD16_BYTE( "95408197.evn", 0x00000, 0x080000, CRC(1e57ba10) SHA1(6d6a5dcb14a2ea1b91fa10bc136841ba237568ff) )
	sc_duckq_others
ROM_END

ROM_START( sc4duckqa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409196.odd", 0x00001, 0x080000, CRC(a562a45d) SHA1(65d854e9b7b46b0b8426aa022bf7694f4613bcc3) )
	ROM_LOAD16_BYTE( "95409197.evn", 0x00000, 0x080000, CRC(f7407914) SHA1(66b3abafc75e2545a34846f3e7f8f0940a4d28a1) )
	sc_duckq_others
ROM_END




ROM_START( sc4dyna )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404578.lo", 0x00001, 0x080000, CRC(8b01fbfd) SHA1(91f54598a4b2a6cf9e48197aad58d86085cce25c) )
	ROM_LOAD16_BYTE( "95404579.hi", 0x00000, 0x080000, CRC(49a12485) SHA1(57b3a7e0a0eb035b5a2fa8238ec762b128d4dee9) )
	sc_dyna_others
ROM_END

ROM_START( sc4dynaa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405578.lo", 0x00001, 0x080000, CRC(1446edf7) SHA1(6eb07646f3865044dbd9a3aa829ebe10113fc734) )
	ROM_LOAD16_BYTE( "95405579.hi", 0x00000, 0x080000, CRC(f8db0f45) SHA1(66d995da3868e77c0a3e2f0802c571acf5b8f465) )
	sc_dyna_others
ROM_END




ROM_START( sc4easy )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404392.lo", 0x00001, 0x080000, CRC(6532f057) SHA1(40ce4e1cb784cd9abab1a10d3b95cf72eecea17d) )
	ROM_LOAD16_BYTE( "95404393.hi", 0x00000, 0x080000, CRC(1f969ae9) SHA1(a69ea773b8bd34815a0b5c6791f0237ac00fbd32) )
	sc_easy_others
ROM_END

ROM_START( sc4easya )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404394.lo", 0x00001, 0x080000, CRC(6dd51d85) SHA1(a093d08e4d7a798e7725df3b6a421185396c44c2) )
	ROM_LOAD16_BYTE( "95404395.hi", 0x00000, 0x080000, CRC(b5879699) SHA1(3f680a6905b4affce4add04f13eb0eb44346cf10) )
	sc_easy_others
ROM_END

ROM_START( sc4easyb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404434.lo", 0x00001, 0x080000, CRC(025a3264) SHA1(40d1cb1fa925824a65846f654ba83e7745a6b283) )
	ROM_LOAD16_BYTE( "95404435.hi", 0x00000, 0x080000, CRC(4e75b8f5) SHA1(63967ac37b3ae56ea155b9f80aaba10ff41a37b2) )
	sc_easy_others
ROM_END

ROM_START( sc4easyc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405392.lo", 0x00001, 0x080000, CRC(ec96482b) SHA1(d2f2772c605179c08d34f00088d00df22fce7dad) )
	ROM_LOAD16_BYTE( "95405393.hi", 0x00000, 0x080000, CRC(670ea673) SHA1(23d43528c40ae604c9d6ae3d60a31ee4f4ea14e3) )
	sc_easy_others
ROM_END

ROM_START( sc4easyd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405434.lo", 0x00001, 0x080000, CRC(23095d0f) SHA1(db245af58e0af9deb053be7cd65cec9f9abbb173) )
	ROM_LOAD16_BYTE( "95405435.hi", 0x00000, 0x080000, CRC(e96df0f8) SHA1(b85972c4ea9338b3efe2ff2b2f275b5c53b8ff66) )
	sc_easy_others
ROM_END

ROM_START( sc4easye )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406026.lo", 0x00001, 0x080000, CRC(40b4ad10) SHA1(e9041da69445dd2a08dbb48ec842040c66b788c3) )
	ROM_LOAD16_BYTE( "95406027.hi", 0x00000, 0x080000, CRC(39eb0bf5) SHA1(229044913c99b0a92e5962eda6c70ec62c3e6b72) )
	sc_easy_others
ROM_END

ROM_START( sc4easyf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407026.lo", 0x00001, 0x080000, CRC(3c207a4f) SHA1(9c8c100765b84ed50c2faa81b26efa8e639ef994) )
	ROM_LOAD16_BYTE( "95407027.hi", 0x00000, 0x080000, CRC(d2819011) SHA1(410f40284b3d261d384e0017522518511f6ca105) )
	sc_easy_others
ROM_END


ROM_START( sc4eascs )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406688.lo", 0x00001, 0x080000, CRC(ce13282c) SHA1(2e778af1de70d3e5d11df54ff85fc50645d85f08) )
	ROM_LOAD16_BYTE( "95406689.hi", 0x00000, 0x080000, CRC(a05771f9) SHA1(7c234c49ad2d74ba150cc73da4adbfe733c5b224) )
	PR1326_CAS_EASY_STREAK_SOUNDS11
ROM_END

ROM_START( sc4eascsa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406690.lo", 0x00001, 0x080000, CRC(d72a4e94) SHA1(ef7ad4e2662b68679b84eb9418c1e9fb7c84e29c) )
	ROM_LOAD16_BYTE( "95406691.hi", 0x00000, 0x080000, CRC(9ecd06c6) SHA1(83ef6bbd7a0c84b3e8c40217aa86d9c14b4521a5) )
	PR1326_CAS_EASY_STREAK_SOUNDS11
ROM_END

ROM_START( sc4eascsb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406884.lo", 0x00001, 0x080000, CRC(c17c9f70) SHA1(20c68c07be855e2f7aff1154072ad60140987c5a) )
	ROM_LOAD16_BYTE( "95406885.hi", 0x00000, 0x080000, CRC(26702794) SHA1(06bcad6234be52f812b4fa3468adca13b16afbe4) )
	PR1326_CAS_EASY_STREAK_SOUNDS11
ROM_END

ROM_START( sc4eascsc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406886.lo", 0x00001, 0x080000, CRC(37e0a39a) SHA1(5430af8a09719883a16e88e3c63dfe4f67f7c231) )
	ROM_LOAD16_BYTE( "95406887.hi", 0x00000, 0x080000, CRC(8e11985d) SHA1(7122d1fdf48953adee04d915f61be4e87d80cd26) )
	PR1326_CAS_EASY_STREAK_SOUNDS11
ROM_END


ROM_START( sc4eascsd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407688.lo", 0x00001, 0x080000, CRC(802b17b9) SHA1(ee56083fdb1d45c27095c7561690d1a39154c0e6) )
	ROM_LOAD16_BYTE( "95407689.hi", 0x00000, 0x080000, CRC(713a971c) SHA1(05cd1f6feec4c5296fe9da82a860d56117939cdb) )
	PR1326_CAS_EASY_STREAK_SOUNDS11
ROM_END

ROM_START( sc4eascse )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407690.lo", 0x00001, 0x080000, CRC(37218f3f) SHA1(3fe41a27f60b621f4d39eb8c9cc27a56481227b5) )
	ROM_LOAD16_BYTE( "95407691.hi", 0x00000, 0x080000, CRC(1986504b) SHA1(126fc4486bcd2931849b2b51c694bb54403d7a8e) )
	PR1326_CAS_EASY_STREAK_SOUNDS11
ROM_END

ROM_START( sc4eascsf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407884.lo", 0x00001, 0x080000, CRC(24c50f8a) SHA1(75f98fc5dcffad2f47c012be66c157bd73f00cf3) )
	ROM_LOAD16_BYTE( "95407885.hi", 0x00000, 0x080000, CRC(c3e37e0e) SHA1(eebd6264ee9efb955fee96017af2ea2678bbca8b) )
	PR1326_CAS_EASY_STREAK_SOUNDS11
ROM_END

ROM_START( sc4eascsg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407886.lo", 0x00001, 0x080000, CRC(3a41067c) SHA1(2e776a26a2436ba969220bf41832875a9bcd588c) )
	ROM_LOAD16_BYTE( "95407887.hi", 0x00000, 0x080000, CRC(8c087f60) SHA1(4c0624d4c56d64008b12c8435f901fb732bbe506) )
	PR1326_CAS_EASY_STREAK_SOUNDS11
ROM_END

ROM_START( sc4eascsh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1326p11.hi", 0x00000, 0x080000, CRC(873c7a05) SHA1(201c287c36bf11832ea3736cee7a4b8bbe9f4254) )
	ROM_LOAD16_BYTE( "p1326p11.lo", 0x00001, 0x080000, CRC(14101cd8) SHA1(b808c8447047b98306843e01fa3f2722e9d37769) )
	PR1326_CAS_EASY_STREAK_SOUNDS11
ROM_END

ROM_START( sc4eascsi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1326p41.hi", 0x00000, 0x080000, CRC(490e389d) SHA1(e1cfe5a64804662e637e3f05eea13902e4ba55a8) )
	ROM_LOAD16_BYTE( "p1326p41.lo", 0x00001, 0x080000, CRC(764150c3) SHA1(1f22f22c72379ec9c1ca0da787366a8368c56303) )
	PR1326_CAS_EASY_STREAK_SOUNDS11
ROM_END

ROM_START( sc4eascsj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1326s11.hi", 0x00000, 0x080000, CRC(ae417ea4) SHA1(ba3876cfc9d7787d3a14b100f3f9e185b08cd207) )
	ROM_LOAD16_BYTE( "p1326s11.lo", 0x00001, 0x080000, CRC(3a5cbcd0) SHA1(c36979d9b0045dbb103a665bde5a97a0a4a3b9b0) )
	PR1326_CAS_EASY_STREAK_SOUNDS11
ROM_END

ROM_START( sc4eascsk )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1326s41.hi", 0x00000, 0x080000, CRC(e74d9798) SHA1(67a43317844a455addb624deea516354854ee0a8) )
	ROM_LOAD16_BYTE( "p1326s41.lo", 0x00001, 0x080000, CRC(b74196f7) SHA1(67ab1a115bb2656951f187f47d6c4b9dca7481c0) )
	PR1326_CAS_EASY_STREAK_SOUNDS11
ROM_END
















ROM_START( sc4fastf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406520.lo", 0x00001, 0x080000, CRC(228c868f) SHA1(889ae1e5d417baff64020f20af3bddd1d5ef1478) )
	ROM_LOAD16_BYTE( "95406521.hi", 0x00000, 0x080000, CRC(b2cefd93) SHA1(8660cb32390202ef00f8ab2a0c8f81d6630030fa) )
	sc_fastf_others
ROM_END

ROM_START( sc4fastfa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407520.lo", 0x00001, 0x080000, CRC(26a0557b) SHA1(a93f8cdf102a56c2df56a5cac6571cd909f72da9) )
	ROM_LOAD16_BYTE( "95407521.hi", 0x00000, 0x080000, CRC(a8b832f1) SHA1(87fde3dc3983545addd2eaa3af9726aab2b3743f) )
	sc_fastf_others
ROM_END

ROM_START( sc4fastfb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408082.lo", 0x00001, 0x080000, CRC(c85c8c84) SHA1(5bfd32b96e6b05f314c0964e562929088eb8257b) )
	ROM_LOAD16_BYTE( "95408083.hi", 0x00000, 0x080000, CRC(73a1474e) SHA1(b4ea15cee9da90f8451e82a2e4a17f3b69ca630e) )
	sc_fastf_others
ROM_END

ROM_START( sc4fastfc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409082.lo", 0x00001, 0x080000, CRC(d3b7ef4b) SHA1(1ac796a9154fd643074efd33b7d630fb55b489f2) )
	ROM_LOAD16_BYTE( "95409083.hi", 0x00000, 0x080000, CRC(99f570da) SHA1(f13faf0bbff2e539ee0713acb50ed939d704a5da) )
	sc_fastf_others
ROM_END






ROM_START( sc4ffru )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404050.lo", 0x00001, 0x080000, CRC(d036045d) SHA1(e3ccf3947094cc6ebb21dec2922c7b407f07ffca) )
	ROM_LOAD16_BYTE( "95404051.hi", 0x00000, 0x080000, CRC(0f58d8c5) SHA1(90d8296c422ba51da59825b63d6f5c96368aa136) )
	sc_ffru_others
ROM_END

ROM_START( sc4ffrua )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404980.odd", 0x00001, 0x080000, CRC(c3fee8e1) SHA1(22a9add77a301609f9b531919745f1ac5a9887ca) )
	ROM_LOAD16_BYTE( "95404981.evn", 0x00000, 0x080000, CRC(13988c83) SHA1(94b31887b331952c68b3091a94b82880fc91643f) )
	sc_ffru_others
ROM_END

ROM_START( sc4ffrub )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405050.lo", 0x00001, 0x080000, CRC(eae778b6) SHA1(317c751a981d173f8426518e917749aebc4bba27) )
	ROM_LOAD16_BYTE( "95405051.hi", 0x00000, 0x080000, CRC(780f9bef) SHA1(7edd383137a5dbf2eb69eced075c3e1610784487) )
	sc_ffru_others
ROM_END

ROM_START( sc4ffruc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405980.odd", 0x00001, 0x080000, CRC(7b608cfe) SHA1(17e2e5f4cbbc561090207e0b5806ed398a7e3550) )
	ROM_LOAD16_BYTE( "95405981.evn", 0x00000, 0x080000, CRC(64509991) SHA1(9eccf9db8b7fc68971ce949c0de2d2e806936fbe) )
	sc_ffru_others
ROM_END

ROM_START( sc4ffrud )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406106.odd", 0x00001, 0x080000, CRC(3ad26b02) SHA1(313df363bb5545fb23020fa943cbbc47d3717731) )
	ROM_LOAD16_BYTE( "95406107.evn", 0x00000, 0x080000, CRC(62b76432) SHA1(322fffd89bcbb87fe2e9cbe195269ab371fae0e6) )
	sc_ffru_others
ROM_END

ROM_START( sc4ffrue )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407106.odd", 0x00001, 0x080000, CRC(5c89918a) SHA1(6caf663d4263ab504a6374bebadb9c9e7c54a959) )
	ROM_LOAD16_BYTE( "95407107.evn", 0x00000, 0x080000, CRC(873a3acf) SHA1(bd71c8cbfe2311eb90281b35ea0644cb611c07e8) )
	sc_ffru_others
ROM_END







ROM_START( sc4fever )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404004.lo", 0x00001, 0x080000, CRC(0e523716) SHA1(5ed17450e06af9dfb22950e520819a9b80be4cdb) )
	ROM_LOAD16_BYTE( "95404005.hi", 0x00000, 0x080000, CRC(9662b52b) SHA1(6db549f79a9f2ce036649ae24407f29e8f35d89b) )
	sc_fever_others
ROM_END

ROM_START( sc4fevera )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404088.lo", 0x00001, 0x080000, CRC(5f42bc75) SHA1(85243a31f8970372eae24878fdd06fd1ceed45a6) )
	ROM_LOAD16_BYTE( "95404089.hi", 0x00000, 0x080000, CRC(9a93aaa6) SHA1(2a822a32cc475539b4e664d3dfcea8b8302247d9) )
	sc_fever_others
ROM_END

ROM_START( sc4feverb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404122.lo", 0x00001, 0x080000, CRC(c7f52065) SHA1(18212106c516c605e25bf48680f1d33d33df2366) )
	ROM_LOAD16_BYTE( "95404123.hi", 0x00000, 0x080000, CRC(f232ec9b) SHA1(08c87d4c15fa9f75d3aef3b6e10bbca926be7724) )
	sc_fever_others
ROM_END

ROM_START( sc4feverc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404124.lo", 0x00001, 0x080000, CRC(a401074a) SHA1(6b20007c06111508092dcd93e42bc26d857a2a6d) )
	ROM_LOAD16_BYTE( "95404125.hi", 0x00000, 0x080000, CRC(3b8014d0) SHA1(5cf28bbf1ff11228db89d2b9cefb278c4700fbbd) )
	sc_fever_others
ROM_END

ROM_START( sc4feverd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405004.lo", 0x00001, 0x080000, CRC(ef33ebb8) SHA1(d4790118780a8bdba46ed8b23915408e15526132) )
	ROM_LOAD16_BYTE( "95405005.hi", 0x00000, 0x080000, CRC(ec91c788) SHA1(a6794c08efb91dcc97274992e809955b249cbdc1) )
	sc_fever_others
ROM_END

ROM_START( sc4fevere )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405088.lo", 0x00001, 0x080000, CRC(1af6ed10) SHA1(021f2eda22c0f4bae8f37f8d8a2767057c9a959c) )
	ROM_LOAD16_BYTE( "95405089.hi", 0x00000, 0x080000, CRC(337d40dc) SHA1(3bf4773a8497b8af9ed4f477b8adb72ade968c6f) )
	sc_fever_others
ROM_END

ROM_START( sc4feverf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405122.lo", 0x00001, 0x080000, CRC(ba3e100d) SHA1(8f92bfb78a873a65b254588ef0e82181199f7aba) )
	ROM_LOAD16_BYTE( "95405123.hi", 0x00000, 0x080000, CRC(d682219d) SHA1(0be17d37fc382ec365beb5eb400cf6b21584b3d1) )
	sc_fever_others
ROM_END

ROM_START( sc4feverg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405124.lo", 0x00001, 0x080000, CRC(e3578368) SHA1(1e7b97059c7dcbcc8110e0e7af398c54d9fe9d33) )
	ROM_LOAD16_BYTE( "95405125.hi", 0x00000, 0x080000, CRC(f1078e7b) SHA1(7c212518ca3a40ecf9247376dfd5dfa3539d9971) )
	sc_fever_others
ROM_END

ROM_START( sc4feverh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406012.lo", 0x00001, 0x080000, CRC(d3e3e350) SHA1(46cb90aec65f6676f143b5647a953fa4c6f3aced) )
	ROM_LOAD16_BYTE( "95406013.hi", 0x00000, 0x080000, CRC(3d98bede) SHA1(52640e1d06d606a2a2c8940a83ad0e9b9bffa347) )
	sc_fever_others
ROM_END

ROM_START( sc4feverk )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407012.lo", 0x00001, 0x080000, CRC(d6c56e82) SHA1(9e3fa0cfc7c9541b9637110677fa0b2f0911e450) )
	ROM_LOAD16_BYTE( "95407013.hi", 0x00000, 0x080000, CRC(4bf15184) SHA1(e01acf9551209161692d204723debb85d0822f10) )
	sc_fever_others
ROM_END


ROM_START( sc4fevdt )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406360.lo", 0x00001, 0x080000, CRC(b0ab3f6b) SHA1(48bb1faba0917d16558221f4c0d1cd6a3fa72767) )
	ROM_LOAD16_BYTE( "95406361.hi", 0x00000, 0x080000, CRC(88874cc3) SHA1(8afe9b666eac553380f9bb80459491a7b4811793) )
	sc_fever7157_others
ROM_END

ROM_START( sc4fevdta )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406782.lo", 0x00001, 0x080000, CRC(7346b7d3) SHA1(24ff011729419b454430d7c4f92c4618338d7ec0) )
	ROM_LOAD16_BYTE( "95406783.hi", 0x00000, 0x080000, CRC(fe3d7b7c) SHA1(7618444990a85d227850009393e130d128fe035f) )
	sc_fever7157_others
ROM_END


ROM_START( sc4fevdtb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408920.lo", 0x00001, 0x080000, CRC(e29cf521) SHA1(c0b2211030caf5ff37bb6ffb813a0dfd7b72963d) )
	ROM_LOAD16_BYTE( "95408921.hi", 0x00000, 0x080000, CRC(db3025ee) SHA1(e21f56adee4eda6a77618e82f6b029955e939741) )
	sc_fever7157_others
ROM_END




ROM_START( sc4fpitc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402956.lo", 0x00001, 0x080000, CRC(b2b602bd) SHA1(70c6959bb5c6d3e61fdd9954808e83141cc8a4f4) )
	ROM_LOAD16_BYTE( "95402957.hi", 0x00000, 0x080000, CRC(2e598dab) SHA1(806d738b2b8a201671ecfb45e112d2342025ee64) )
	sc_fpitc_others
ROM_END

ROM_START( sc4fpitca )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402958.lo", 0x00001, 0x080000, CRC(0fdcf790) SHA1(c71e273ee8ae9f11ed9df037aa8f250ca014b949) )
	ROM_LOAD16_BYTE( "95402959.hi", 0x00000, 0x080000, CRC(c53a6bbb) SHA1(16149a78302bf67b9c53d5497fd4e85dfc244d66) )
	sc_fpitc_others
ROM_END

ROM_START( sc4fpitcb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402960.lo", 0x00001, 0x080000, CRC(bf04a142) SHA1(ec0611a975799f7f0c442f805e8157daad0c338e) )
	ROM_LOAD16_BYTE( "95402961.hi", 0x00000, 0x080000, CRC(8b9c46f1) SHA1(410122c661ff83f960c0e0024ad89d06268aa101) )
	sc_fpitc_others
ROM_END

ROM_START( sc4fpitcc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402962.lo", 0x00001, 0x080000, CRC(244ce405) SHA1(ca2fdaa8eb4b40eb84e77317487ffa7eff419051) )
	ROM_LOAD16_BYTE( "95402963.hi", 0x00000, 0x080000, CRC(cb2b0e33) SHA1(65a704ab0b2e4ddc8bbefbf53a0e4ec7b10cb05e) )
	sc_fpitc_others
ROM_END

ROM_START( sc4fpitcd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403956.lo", 0x00001, 0x080000, CRC(131b5532) SHA1(7ca7179f4226e155090143739b58bdce2bbcccfd) )
	ROM_LOAD16_BYTE( "95403957.hi", 0x00000, 0x080000, CRC(a67d44fe) SHA1(5f8a0156de995006680d55cfd164bec9294d4d20) )
	sc_fpitc_others
ROM_END

ROM_START( sc4fpitce )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403958.lo", 0x00001, 0x080000, CRC(c0d150c2) SHA1(a899e69437776722001055b90816fc2f1f585a60) )
	ROM_LOAD16_BYTE( "95403959.hi", 0x00000, 0x080000, CRC(abfdfe42) SHA1(3ca007a9f61acf79c987a870c736fb9c4bbc19d7) )
	sc_fpitc_others
ROM_END

ROM_START( sc4fpitcf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403960.lo", 0x00001, 0x080000, CRC(895da1a6) SHA1(3cfbd9a0fbf15c48e5da949efa3137b20100987d) )
	ROM_LOAD16_BYTE( "95403961.hi", 0x00000, 0x080000, CRC(ab6979e5) SHA1(620164ad61ed5760b96e55fdac1f01aacae4d3c9) )
	sc_fpitc_others
ROM_END

ROM_START( sc4fpitcg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403962.lo", 0x00001, 0x080000, CRC(c4a4c20c) SHA1(d2d47b9a2d77c95f58a5be63be96208f45207fc6) )
	ROM_LOAD16_BYTE( "95403963.hi", 0x00000, 0x080000, CRC(30ce6ed8) SHA1(2a938114a8f7fdc7f1265a345316d954b12d1260) )
	sc_fpitc_others
ROM_END

ROM_START( sc4fpitch )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406002.lo", 0x00001, 0x080000, CRC(4e242877) SHA1(08639639d2f7fb671d66a1e9848726890318ba6d) )
	ROM_LOAD16_BYTE( "95406003.hi", 0x00000, 0x080000, CRC(3577d50e) SHA1(86e622176386c19a39c0cefec19984a0b96b682b) )
	sc_fpitc_others
ROM_END

ROM_START( sc4fpitci )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406064.lo", 0x00001, 0x080000, CRC(980b69ec) SHA1(3b5acb46fdf03c6ce30df5f3e8e12557020f917b) )
	ROM_LOAD16_BYTE( "95406065.hi", 0x00000, 0x080000, CRC(62db32ae) SHA1(001022d7e544b1e9966b361e88be7078936d43a6) )
	sc_fpitc_others
ROM_END

ROM_START( sc4fpitcj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407002.lo", 0x00001, 0x080000, CRC(d54a79ad) SHA1(d76fc8a846ea38fff2d774a8dcfb2661750349f2) )
	ROM_LOAD16_BYTE( "95407003.hi", 0x00000, 0x080000, CRC(0edc749b) SHA1(0da9e663f74596529a0eaa9c695bf720fc90137d) )
	sc_fpitc_others
ROM_END

ROM_START( sc4fpitck )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407064.lo", 0x00001, 0x080000, CRC(989d7797) SHA1(3834d69d57a7e41a36791df63a07e7103e8436d8) )
	ROM_LOAD16_BYTE( "95407065.hi", 0x00000, 0x080000, CRC(b324a4af) SHA1(0bba58991e134ee49102052b22446e8706335233) )
	sc_fpitc_others
ROM_END




ROM_START( sc4fevnx )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406864.lo", 0x00001, 0x080000, CRC(9f7846ac) SHA1(d2e5b218f395974545f5c723259153af0d1b8bf4) )
	ROM_LOAD16_BYTE( "95406865.hi", 0x00000, 0x080000, CRC(f54c556d) SHA1(dce5c622be6324080877dfb09bcb60533caba07b) )
	sc_fevnx_others
ROM_END

ROM_START( sc4fevnxa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406990.lo", 0x00001, 0x080000, CRC(3880104f) SHA1(1c9b7dd1712f524b4dfa9d83cae59699c7c6b3e6) )
	ROM_LOAD16_BYTE( "95406991.hi", 0x00000, 0x080000, CRC(ea59f14b) SHA1(51e36d5ede267872e4dbb59236804034243bba63) )
	sc_fevnx_others
ROM_END






ROM_START( sc4fcc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "club-firecracker_std_ac_varss.hi", 0x00000, 0x080000, CRC(0bbab14f) SHA1(97e5d7d0f57c81232a06f89b486b2833f0ec8640) )
	ROM_LOAD16_BYTE( "club-firecracker_std_ac_varss.lo", 0x00001, 0x080000, CRC(b543a3ad) SHA1(42f1c69adcbf7acc48dad98085f3f9e3a69f3342) )
	sc_fcc_others

	sc_fcc_matrix

ROM_END

ROM_START( sc4fcca )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "club-firecracker_dat_ac_67pss.hi", 0x00000, 0x080000, CRC(f62d7c6b) SHA1(a8e680074880f33313fe4ad6805d70910a4d8a48) )
	ROM_LOAD16_BYTE( "club-firecracker_dat_ac_67pss.lo", 0x00001, 0x080000, CRC(cd7ccd11) SHA1(47f7b89a2b27966de21eb08845108e7da2542227) )
	sc_fcc_others

	sc_fcc_matrix

ROM_END

ROM_START( sc4fccb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "club-firecracker_dat_ac_varss.hi", 0x00000, 0x080000, CRC(5e736802) SHA1(6cd0800ab8a96f057080c857c8ae696da71f58bd) )
	ROM_LOAD16_BYTE( "club-firecracker_dat_ac_varss.lo", 0x00001, 0x080000, CRC(479efeef) SHA1(9115b184c14cc1fb1cdda0db0f6d39b396d424c9) )
	sc_fcc_others

	sc_fcc_matrix
ROM_END

ROM_START( sc4fccc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "club-firecracker_std_ac_67pss.hi", 0x00000, 0x080000, CRC(6b2e08d5) SHA1(11a870bae9881df116727c942e724320557fac83) )
	ROM_LOAD16_BYTE( "club-firecracker_std_ac_67pss.lo", 0x00001, 0x080000, CRC(09315677) SHA1(bd9421275331707e1ad24d7d248b8dddb1bf75ee) )
	sc_fcc_others

	sc_fcc_matrix
ROM_END




ROM_START( sc4fire )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406968.lo", 0x00001, 0x080000, CRC(4fc1f8f3) SHA1(001ca9f18fb11af66a8dc9e11d59981165db5fd1) )
	ROM_LOAD16_BYTE( "95406969.hi", 0x00000, 0x080000, CRC(2a676181) SHA1(aec4c37116f46d16096a297cc7001245f1aa5ec7) )
	sc_fire_others
ROM_END

ROM_START( sc4firea )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407968.lo", 0x00001, 0x080000, CRC(274bb010) SHA1(44c20ddf693f48db871d99cfab1b41b5490bb28b) )
	ROM_LOAD16_BYTE( "95407969.hi", 0x00000, 0x080000, CRC(2af837b9) SHA1(35de190685946709517ac00f61a9138d9ced4561) )
	sc_fire_others
ROM_END




ROM_START( sc4fwp )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402988.lo", 0x00001, 0x080000, CRC(c71b9b6b) SHA1(9eb529cf0355e48f32b8ba7f544b27c2bda77af4) )
	ROM_LOAD16_BYTE( "95402989.hi", 0x00000, 0x080000, CRC(b372b601) SHA1(ca251f96781254d1887eb0bbe1b12726b8eff1e0) )
	sc_fwp_others
ROM_END

ROM_START( sc4fwpa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403986.lo", 0x00001, 0x080000, CRC(5feed224) SHA1(d8c2c70c4d12b3c4a177d8ef32a4b19c9b056872) )
	ROM_LOAD16_BYTE( "95403987.hi", 0x00000, 0x080000, CRC(c5d72a23) SHA1(986bec61f5d3191419a652b2ca0e2e671f2ef81d) )
	sc_fwp_others
ROM_END

ROM_START( sc4fwpb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410344.lo", 0x00001, 0x080000, CRC(99f6aa75) SHA1(867defde4560d6676e3dbc124315be97fc6935c7) )
	ROM_LOAD16_BYTE( "95410345.hi", 0x00000, 0x080000, CRC(5c65c5b0) SHA1(f491fb87ba7eaa8d9d4acc31b1a561ed013c01ec) )
	sc_fwp_others
ROM_END

ROM_START( sc4fwpc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411344.lo", 0x00001, 0x080000, CRC(55442a51) SHA1(022f27999dfedf779d8546541ccbb9cd1da03f0c) )
	ROM_LOAD16_BYTE( "95411345.hi", 0x00000, 0x080000, CRC(f911b233) SHA1(87a9039cff2247888faa83754529e4f8382a520d) )
	sc_fwp_others
ROM_END




ROM_START( sc4fwpcs )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402830.lo", 0x00001, 0x080000, CRC(60667ba0) SHA1(d16f064a0f6c746349695582ad3d1611806ff0ef) )
	ROM_LOAD16_BYTE( "95402831.hi", 0x00000, 0x080000, CRC(eaa38666) SHA1(8a0f8ab18cd97b01c9244a19a22a2cc44584ca52) )
	sc_fwp_others
ROM_END

ROM_START( sc4fwpcsa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402832.lo", 0x00001, 0x080000, CRC(882a61fa) SHA1(d8284d5a9d0fc553fa09c9765e564d49d8e8c86c) )
	ROM_LOAD16_BYTE( "95402833.hi", 0x00000, 0x080000, CRC(17fd9dde) SHA1(d55c1b2bc810410ad74c7d3c74587be5ae581738) )
	sc_fwp_others
ROM_END

ROM_START( sc4fwpcsb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402986.lo", 0x00001, 0x080000, CRC(64b9fc1c) SHA1(72e8b16bb8b16f10a669917a4cef029cb5447225) )
	ROM_LOAD16_BYTE( "95402987.hi", 0x00000, 0x080000, CRC(dc7d47e9) SHA1(f393c791857301192fab520b7e4e5775ed15c71e) )
	sc_fwp_others
ROM_END




ROM_START( sc4fbcrz )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400385.lo", 0x00001, 0x080000, CRC(853d5017) SHA1(37f3cf0aadaee24a82d5cb8b6e1f18d7f4481642) )
	ROM_LOAD16_BYTE( "95400386.hi", 0x00000, 0x080000, CRC(ca1e4ae6) SHA1(67ea3066434f2766c0f16d612767ae694ed7680e) )
	sc_fbcrz_others
ROM_END

ROM_START( sc4fbcrza )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400387.lo", 0x00001, 0x080000, CRC(f24b4dcc) SHA1(9d48bcd22c231cf99dd6874be99a8affe287db9e) )
	ROM_LOAD16_BYTE( "95400388.hi", 0x00000, 0x080000, CRC(ef90f929) SHA1(44b65090438ff5df06e152c9c2b2778ad58db318) )
	sc_fbcrz_others
ROM_END

ROM_START( sc4fbcrzb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400533.lo", 0x00001, 0x080000, CRC(0129ebce) SHA1(9ca5f4d6abf639a9b54ea488d5dfa4f8e079a65e) )
	ROM_LOAD16_BYTE( "95400534.hi", 0x00000, 0x080000, CRC(d2a18245) SHA1(0a574df85255ec4ea8c1e2c884cd3617e5663e33) )
	sc_fbcrz_others
ROM_END

ROM_START( sc4fbcrzc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400747.lo", 0x00001, 0x080000, CRC(c2a2dede) SHA1(709ddc5e2a2da011f07a4d93195a03b2fe9bb191) )
	ROM_LOAD16_BYTE( "95400748.hi", 0x00000, 0x080000, CRC(571f1f23) SHA1(35039811aaa40e47984c576fdc447480ec41a96a) )
	sc_fbcrz_others
ROM_END

ROM_START( sc4fbcrzd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400749.lo", 0x00001, 0x080000, CRC(8709908c) SHA1(d740a66701974351a33dc8bc399bef78a17c1c70) )
	ROM_LOAD16_BYTE( "95400750.hi", 0x00000, 0x080000, CRC(16ac7ea9) SHA1(2e45f74eaf2e00f5a8550197998779cea0596eb7) )
	sc_fbcrz_others
ROM_END

ROM_START( sc4fbcrze )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401385.lo", 0x00001, 0x080000, CRC(25d68337) SHA1(abe7d0c7b9301253ae75c3c61834d551121170fd) )
	ROM_LOAD16_BYTE( "95401386.hi", 0x00000, 0x080000, CRC(9daacb4b) SHA1(4c8bcc331069007a92ace7cac58a26a2533e076e) )
	sc_fbcrz_others
ROM_END

ROM_START( sc4fbcrzf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401387.lo", 0x00001, 0x080000, CRC(faebf38d) SHA1(e92f825265e534fd79011c3ac445524c5a46ee55) )
	ROM_LOAD16_BYTE( "95401388.hi", 0x00000, 0x080000, CRC(96bf2922) SHA1(c2ee3b4f20b6ae7c25b94ee5a24cdfd53ecb2811) )
	sc_fbcrz_others
ROM_END

ROM_START( sc4fbcrzg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401533.lo", 0x00001, 0x080000, CRC(4325328c) SHA1(1de2fe68680267447572c7aca8f5bbe96017d38b) )
	ROM_LOAD16_BYTE( "95401534.hi", 0x00000, 0x080000, CRC(16f1b24f) SHA1(94f18624c036a9b6a543daf80ad3c23c20cf1acc) )
	sc_fbcrz_others
ROM_END

ROM_START( sc4fbcrzh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401747.lo", 0x00001, 0x080000, CRC(af0890aa) SHA1(e35d0bc7db95fccd3bc1e08843c8600c9d3f085f) )
	ROM_LOAD16_BYTE( "95401748.hi", 0x00000, 0x080000, CRC(9d37bc08) SHA1(a58d2b05f568dc7e348c0dd9014ddd2df668e372) )
	sc_fbcrz_others
ROM_END

ROM_START( sc4fbcrzi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401749.lo", 0x00001, 0x080000, CRC(b33d055c) SHA1(b1048654c754f2cd25e94e2eb95529fe7d560566) )
	ROM_LOAD16_BYTE( "95401750.hi", 0x00000, 0x080000, CRC(8476637e) SHA1(2df9d3f9c48057dccd1dd28a5702684c530f3e6c) )
	sc_fbcrz_others
ROM_END




ROM_START( sc4fd7th )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400123.lo", 0x00001, 0x080000, CRC(fff100ca) SHA1(6e6012c9e3d057791a8353995c347152c6e6a452) )
	ROM_LOAD16_BYTE( "95400124.hi", 0x00000, 0x080000, CRC(fa841b79) SHA1(cb1f0c35f2e73f557c57027d2370ed179afa77e2) )
	sc_fd7th_others
ROM_END

ROM_START( sc4fd7tha )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400239.lo", 0x00001, 0x080000, CRC(14692e18) SHA1(e608df4bb40667a168186d29ef8d99d371d3d570) )
	ROM_LOAD16_BYTE( "95400240.hi", 0x00000, 0x080000, CRC(fc290b75) SHA1(41d93251cd1c4a6bab26bfcbc59855b15ba4989d) )
	sc_fd7th_others
ROM_END

ROM_START( sc4fd7thb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400241.lo", 0x00001, 0x080000, CRC(b43e1871) SHA1(678299fdd6b5eb08ca044fdda7aa6538435b5c36) )
	ROM_LOAD16_BYTE( "95400242.hi", 0x00000, 0x080000, CRC(ca0cc6b3) SHA1(a8a43916887f9ae65b0f2510b3df1cbd4ae53800) )
	sc_fd7th_others
ROM_END

ROM_START( sc4fd7thc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400475.lo", 0x00001, 0x080000, CRC(b911fdc7) SHA1(e8698af72b0ccf7b64b56a2b5fdbed1a42f9490c) )
	ROM_LOAD16_BYTE( "95400476.hi", 0x00000, 0x080000, CRC(d8330fc0) SHA1(a96de3319e260d240d4e71ceb2b195bc73264829) )
	sc_fd7th_others
ROM_END

ROM_START( sc4fd7thd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400477.lo", 0x00001, 0x080000, CRC(79508577) SHA1(63222e05eece6f30828179342a065fb093b28d9b) )
	ROM_LOAD16_BYTE( "95400478.hi", 0x00000, 0x080000, CRC(0d0cb0e8) SHA1(f482f79b4f128c22cd121c91b3e5740031bf2473) )
	sc_fd7th_others
ROM_END

ROM_START( sc4fd7the )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401123.lo", 0x00001, 0x080000, CRC(e7b11fd4) SHA1(9127e8b5ba39f6c4553c2b41c43310111d1c4aeb) )
	ROM_LOAD16_BYTE( "95401124.hi", 0x00000, 0x080000, CRC(4586a398) SHA1(ecbe605cb8b5e080f2dd667e09d044c1371b9369) )
	sc_fd7th_others
ROM_END

ROM_START( sc4fd7thf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401239.lo", 0x00001, 0x080000, CRC(606eec9c) SHA1(66c5d4f39d4f7f660ac8d52e7821c2c78ee69b3e) )
	ROM_LOAD16_BYTE( "95401240.hi", 0x00000, 0x080000, CRC(e4bdcacf) SHA1(4207ff3e8f1771d378343d1b03770289c42038d6) )
	sc_fd7th_others
ROM_END

ROM_START( sc4fd7thg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401241.lo", 0x00001, 0x080000, CRC(f8d18ff0) SHA1(95bed4d487473c67f52753356921c06a36d9e8a2) )
	ROM_LOAD16_BYTE( "95401242.hi", 0x00000, 0x080000, CRC(544c469d) SHA1(dbf87656d1f43986c45a6119c38894fd25af8768) )
	sc_fd7th_others
ROM_END

ROM_START( sc4fd7thh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401475.lo", 0x00001, 0x080000, CRC(df44e314) SHA1(0adcdc6014d1b0a69d53977ce02688a86c13b316) )
	ROM_LOAD16_BYTE( "95401476.hi", 0x00000, 0x080000, CRC(509fdd75) SHA1(a038ab9463d2b5a1e7abad9d501c1503de724682) )
	sc_fd7th_others
ROM_END

ROM_START( sc4fd7thi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401477.lo", 0x00001, 0x080000, CRC(c3c16a14) SHA1(e98b7d36fe498851fbb2a0cc5491014c4168c5ae) )
	ROM_LOAD16_BYTE( "95401478.hi", 0x00000, 0x080000, CRC(0cad4aa0) SHA1(da501db50c30e359d295349725b6a6965e3dab3b) )
	sc_fd7th_others
ROM_END




ROM_START( sc4frboo )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400997.lo", 0x00001, 0x080000, CRC(5874bfd8) SHA1(0e965980d3aaeba99a2af6e1d6a192e9ab792d49) )
	ROM_LOAD16_BYTE( "95400998.hi", 0x00000, 0x080000, CRC(a5a68143) SHA1(be8e19df24b216999b272b42c3926faba2a0704c) )
	sc_frboo_others
ROM_END

ROM_START( sc4frbooa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401997.lo", 0x00001, 0x080000, CRC(9753385c) SHA1(315592bcdb295953639bcd4024f655fa838a6a43) )
	ROM_LOAD16_BYTE( "95401998.hi", 0x00000, 0x080000, CRC(b1a8dd00) SHA1(574325504370138a00427cbc2e89b915586a9e66) )
	sc_frboo_others
ROM_END

ROM_START( sc4frboob )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402007.lo", 0x00001, 0x080000, CRC(78c9db57) SHA1(036eaa3e45f806d23b705221b17ee1d3c52bb489) )
	ROM_LOAD16_BYTE( "95402008.hi", 0x00000, 0x080000, CRC(762f00bc) SHA1(96b43f2ef8dbd8f615c5a05a8af52dc40605c2c2) )
	sc_frboo_others
ROM_END

ROM_START( sc4frbooc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403007.lo", 0x00001, 0x080000, CRC(95a9bd51) SHA1(d7939e9f780d06aadeee9e7f5cff1d1fb11c8f9f) )
	ROM_LOAD16_BYTE( "95403008.hi", 0x00000, 0x080000, CRC(c528739c) SHA1(8446e01e7747fca2b85744cf50c6a6fe1939ef7c) )
	sc_frboo_others
ROM_END





ROM_START( sc4frenz )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404436.lo", 0x00001, 0x080000, CRC(6b428b51) SHA1(8460f3156cc2e69d3d05377e33caea7eb66aeb56) )
	ROM_LOAD16_BYTE( "95404437.hi", 0x00000, 0x080000, CRC(4ead56c3) SHA1(4ca9799fd3b01224205624085fe36a05a171f9a5) )
	sc_frenz_others
ROM_END

ROM_START( sc4frenza )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404438.lo", 0x00001, 0x080000, CRC(65c9050c) SHA1(818f60883d7b9563d67eae8041b9fa2ab63a3720) )
	ROM_LOAD16_BYTE( "95404439.hi", 0x00000, 0x080000, CRC(7c113dfa) SHA1(854ec8b001b8b093fac972c782f19afe6235c241) )
	sc_frenz_others
ROM_END

ROM_START( sc4frenzb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405436.lo", 0x00001, 0x080000, CRC(0b3a4357) SHA1(2213cab13f013d8f48dae4e8a8c03f19d35a478e) )
	ROM_LOAD16_BYTE( "95405437.hi", 0x00000, 0x080000, CRC(b3b4c984) SHA1(1c7a48b49de06f9ca0935593c2eaf9c03bd1735d) )
	sc_frenz_others
ROM_END

ROM_START( sc4frenzc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405438.lo", 0x00001, 0x080000, CRC(991343fc) SHA1(e4e2b9fad42c22179d22e3b4efefc1422b117320) )
	ROM_LOAD16_BYTE( "95405439.hi", 0x00000, 0x080000, CRC(ce28e2aa) SHA1(8b4fbcfb96e6caf0edabd6ba80d0ea0e63e275a4) )
	sc_frenz_others
ROM_END

ROM_START( sc4frenzd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406028.lo", 0x00001, 0x080000, CRC(d63540f9) SHA1(48ab12f30b7fbdb539d35f6ffc54da0f7152e699) )
	ROM_LOAD16_BYTE( "95406029.hi", 0x00000, 0x080000, CRC(63e3f490) SHA1(46923775fc67bbf760307ebeb185c15a473e3252) )
	sc_frenz_others
ROM_END

ROM_START( sc4frenze )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407028.lo", 0x00001, 0x080000, CRC(f9f08c54) SHA1(04a55755f1c0966f799ca6b6f77198600af69eb6) )
	ROM_LOAD16_BYTE( "95407029.hi", 0x00000, 0x080000, CRC(63e3f490) SHA1(46923775fc67bbf760307ebeb185c15a473e3252) )
	sc_frenz_others
ROM_END




ROM_START( sc4ftopi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404704.odd", 0x00001, 0x080000, CRC(a838f0cc) SHA1(c67d700baaac6b64cc0e241cfa87d4a38f87d7c7) )
	ROM_LOAD16_BYTE( "95404705.evn", 0x00000, 0x080000, CRC(d025d84e) SHA1(66d8a523a573161cf419deefd02fa7f1ca59339d) )
	sc_ftopi_others
ROM_END

ROM_START( sc4ftopia )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405704.odd", 0x00001, 0x080000, CRC(30f083a5) SHA1(340ee2e605ddd3aee972588c3c455095f710bdf8) )
	ROM_LOAD16_BYTE( "95405705.evn", 0x00000, 0x080000, CRC(17a94aec) SHA1(c9f4d1576f5439e6935bacb7f3ba6c590c56769c) )
	sc_ftopi_others
ROM_END

ROM_START( sc4ftopib )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406646.odd", 0x00001, 0x080000, CRC(a65da5cf) SHA1(2d75a1aa00a88bead5a1af06d5aa981ae00d29a7) )
	ROM_LOAD16_BYTE( "95406647.evn", 0x00000, 0x080000, CRC(4115232f) SHA1(5af4065b9aa45ab55f2d37864ee24929a6e41275) )
	sc_ftopi_others
ROM_END

ROM_START( sc4ftopic )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407646.odd", 0x00001, 0x080000, CRC(c45ddb69) SHA1(444eb64b410d2c9aa9d0fced90c79796d22fbe2f) )
	ROM_LOAD16_BYTE( "95407647.evn", 0x00000, 0x080000, CRC(79622012) SHA1(2a6896998279e0fd0b81ba99167cf05b9f8c6b3f) )
	sc_ftopi_others
ROM_END

ROM_START( sc4ftopid )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408028.odd", 0x00001, 0x080000, CRC(ff994b16) SHA1(9754d6195b909ace734280a0abe2b77e9bd312eb) )
	ROM_LOAD16_BYTE( "95408029.evn", 0x00000, 0x080000, CRC(80f850f7) SHA1(cef3269c70f3a6eca8d6c64c75445fa0451e6cde) )
	sc_ftopi_others
ROM_END

ROM_START( sc4ftopie )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408306.odd", 0x00001, 0x080000, CRC(cdf9b2b2) SHA1(7c4cc9ddbf5aff242c2efbe733746c888d159d57) )
	ROM_LOAD16_BYTE( "95408307.evn", 0x00000, 0x080000, CRC(e3100ef0) SHA1(e976990e567bd3afdbb963289ac679d528d750af) )
	sc_ftopi_others
ROM_END

ROM_START( sc4ftopif )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408660.odd", 0x00001, 0x080000, CRC(04717087) SHA1(94e514e1ce9acdde71f25399ad416b03363a13be) )
	ROM_LOAD16_BYTE( "95408661.evn", 0x00000, 0x080000, CRC(0c58bf56) SHA1(5fb6d053dc37d29bee94c8ca093446b5f75e5e27) )
	sc_ftopi_others
ROM_END

ROM_START( sc4ftopig )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409028.odd", 0x00001, 0x080000, CRC(b3c7510a) SHA1(ce220b9f8352c48f6e69519e9506605ff49b5a0e) )
	ROM_LOAD16_BYTE( "95409029.evn", 0x00000, 0x080000, CRC(d895b833) SHA1(9e479407261c0278c47ac5eeb05a2c3320220c4a) )
	sc_ftopi_others
ROM_END

ROM_START( sc4ftopih )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409306.odd", 0x00001, 0x080000, CRC(244c8915) SHA1(d8093c9fa98cc0b5438c3344ecb74adfb1998fe5) )
	ROM_LOAD16_BYTE( "95409307.evn", 0x00000, 0x080000, CRC(da59a1bd) SHA1(6cc373416b575821d597b586d3f3abfdd2a7d546) )
	sc_ftopi_others
ROM_END

ROM_START( sc4ftopii )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409660.odd", 0x00001, 0x080000, CRC(0ef83ee4) SHA1(d8c81fa951dc2a58fbca8a5dc99ad513f2eed5b4) )
	ROM_LOAD16_BYTE( "95409661.evn", 0x00000, 0x080000, CRC(0381d63f) SHA1(3b1717aa4d87a7ded6b1e454ca27da6c520e94f8) )
	sc_ftopi_others
ROM_END




ROM_START( sc4frsu )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402409.lo", 0x00001, 0x080000, CRC(f28d3873) SHA1(dc84f08e550e80ed94a3658737f3e844503f02f7) )
	ROM_LOAD16_BYTE( "95402410.hi", 0x00000, 0x080000, CRC(cabe32f9) SHA1(68ef4c7459f9ae92f3186f51817710abd1b0fdbd) )
	sc_frsu_others
ROM_END

ROM_START( sc4frsua )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403409.lo", 0x00001, 0x080000, CRC(b82b6741) SHA1(f806b31b6dfed2ea6f64204c0edd19ba217ee001) )
	ROM_LOAD16_BYTE( "95403410.hi", 0x00000, 0x080000, CRC(9c341f24) SHA1(2c8a494234bab0af359fca8fa2ad2b53e33e5b20) )
	sc_frsu_others
ROM_END

ROM_START( sc4frsub )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406532.lo", 0x00001, 0x080000, CRC(f38776fb) SHA1(1ae7b7ce1977d4e4fb574fd6ed990f9803eb0465) )
	ROM_LOAD16_BYTE( "95406533.hi", 0x00000, 0x080000, CRC(7666fedf) SHA1(276937de5917edd46e73d7baccd6d9ed658cb276) )
	sc_frsu_others
ROM_END

ROM_START( sc4frsuc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407532.lo", 0x00001, 0x080000, CRC(a927b36f) SHA1(df3c04c3a7d9e709d45f517740cd1b955c160986) )
	ROM_LOAD16_BYTE( "95407533.hi", 0x00000, 0x080000, CRC(924a5a6a) SHA1(6ade1ea32a2ce9e55f8d23e78cd53ec03c8756b2) )
	sc_frsu_others
ROM_END

ROM_START( sc4frsud )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410052.lo", 0x00001, 0x080000, CRC(66690b85) SHA1(0e1ff97d138b1001dc439c7d30eaa65cd687fa9f) )
	ROM_LOAD16_BYTE( "95410053.hi", 0x00000, 0x080000, CRC(f4c9306a) SHA1(66cf87a6b1125823c8a8792d079955e25f9519c7) )
	sc_frsu_others
ROM_END

ROM_START( sc4frsue )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410056.lo", 0x00001, 0x080000, CRC(f8fc19f2) SHA1(416fb661c6c711a592310e85bedd12159b38e4c5) )
	ROM_LOAD16_BYTE( "95410057.hi", 0x00000, 0x080000, CRC(8ce4554b) SHA1(eea25e6e2b5a54414ed30103e729d22a5970a62d) )
	sc_frsu_others
ROM_END

ROM_START( sc4frsuf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411052.lo", 0x00001, 0x080000, CRC(f6775350) SHA1(3f1cb7c0a7fb9e7d1aea5502d9dc8ce4d25d8866) )
	ROM_LOAD16_BYTE( "95411053.hi", 0x00000, 0x080000, CRC(c78c23e1) SHA1(5a89a27d0eee0731b14d1fb84b87a03478d3b500) )
	sc_frsu_others
ROM_END

ROM_START( sc4frsug )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411056.lo", 0x00001, 0x080000, CRC(5e69e9e1) SHA1(06554561a2e6f8c89423c6c987965318edfb94ad) )
	ROM_LOAD16_BYTE( "95411057.hi", 0x00000, 0x080000, CRC(1578401e) SHA1(1d0ee4f9cacad2e0e54a0ce4b4b0fd6589a3de44) )
	sc_frsu_others
ROM_END






ROM_START( sc4fullt )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412334.odd", 0x00001, 0x080000, CRC(fa047f3c) SHA1(ccd3cdee630c1fc602669ee33b7a5b930e06884e) )
	ROM_LOAD16_BYTE( "95412335.evn", 0x00000, 0x080000, CRC(39214d9f) SHA1(7a8a758856f9bfb068564e295bc4b71782789178) )
	sc_fullt_others
ROM_END

ROM_START( sc4fullta )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412336.odd", 0x00001, 0x080000, CRC(dd162e93) SHA1(4337bd2108ca874f0c3a3570739f1650a056c440) )
	ROM_LOAD16_BYTE( "95412337.evn", 0x00000, 0x080000, CRC(709b89e9) SHA1(02c572e53404b37ca1d94a451d7a64ca7f378631) )
	sc_fullt_others
ROM_END

ROM_START( sc4fulltb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413334.odd", 0x00001, 0x080000, CRC(5c248e22) SHA1(d04d7c581a2e213ce501a8211d8f1d1115f96bda) )
	ROM_LOAD16_BYTE( "95413335.evn", 0x00000, 0x080000, CRC(2c11bdac) SHA1(701a23b3187a8e107ff966c8f29a8d70384720fa) )
	sc_fullt_others
ROM_END

ROM_START( sc4fulltc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413336.odd", 0x00001, 0x080000, CRC(7405e6c9) SHA1(61f1e053ff0f484c9ac9ff5f80a0aff286f6ba6d) )
	ROM_LOAD16_BYTE( "95413337.evn", 0x00000, 0x080000, CRC(469419f5) SHA1(dbba72da97cba59a96d98a647adc7825d591af3b) )
	sc_fullt_others
ROM_END

ROM_START( sc4fulltd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414508.odd", 0x00001, 0x080000, CRC(2b6d1ae2) SHA1(9e09c2e838bb599d2cf155cebe8b9d10e6dc86cb) )
	ROM_LOAD16_BYTE( "95414509.evn", 0x00000, 0x080000, CRC(dc35c2c8) SHA1(4b3a7bbf6e74ff0278d21f3ed755de9dd44fe9bb) )
	sc_fullt_others
ROM_END

ROM_START( sc4fullte )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414510.odd", 0x00001, 0x080000, CRC(c0d8ef81) SHA1(0e6c1d6cb2762a7b777acf10dfd5077bcea65e64) )
	ROM_LOAD16_BYTE( "95414511.evn", 0x00000, 0x080000, CRC(469791e0) SHA1(4335d4cd8cf49181a49130ed0f4a5c9cd6a5b470) )
	sc_fullt_others
ROM_END

ROM_START( sc4fulltf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414738.odd", 0x00001, 0x080000, CRC(84beebed) SHA1(44e36819aca9595cf57d0c9318f6c318fcb342b6) )
	ROM_LOAD16_BYTE( "95414739.evn", 0x00000, 0x080000, CRC(24b419e2) SHA1(e68b3eacfec21fe7ce28b2283f5a02778862d176) )
	sc_fullt_others
ROM_END

ROM_START( sc4fulltg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415508.odd", 0x00001, 0x080000, CRC(a1323401) SHA1(19887bcc93b6fddf6740b01fee12597100f4b8cf) )
	ROM_LOAD16_BYTE( "95415509.evn", 0x00000, 0x080000, CRC(a424beb3) SHA1(9bd29e8c0ba127e5ec8914e76f4b8f2a8cb979b0) )
	sc_fullt_others
ROM_END

ROM_START( sc4fullth )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415510.odd", 0x00001, 0x080000, CRC(61fab1cf) SHA1(a3b6c2588517b01599ef76cba05f3bcaaf33e360) )
	ROM_LOAD16_BYTE( "95415511.evn", 0x00000, 0x080000, CRC(7344a354) SHA1(747ddbd1cb3c4a01cba6ad8447def2995c3fe31d) )
	sc_fullt_others
ROM_END

ROM_START( sc4fullti )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415738.odd", 0x00001, 0x080000, CRC(d8fa18be) SHA1(5667f9eecede0dbb9f042530cec8bfc94e6427a9) )
	ROM_LOAD16_BYTE( "95415739.evn", 0x00000, 0x080000, CRC(24b419e2) SHA1(e68b3eacfec21fe7ce28b2283f5a02778862d176) )
	sc_fullt_others
ROM_END




ROM_START( sc4gamcs )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406354.lo", 0x00001, 0x080000, CRC(846e59e4) SHA1(a9cf980080f571f5b246209a28071da714e326ac) )
	ROM_LOAD16_BYTE( "95406355.hi", 0x00000, 0x080000, CRC(139af67f) SHA1(491a08293324673b42a032a7b3241f573cf4b0e4) )
	sc_gamcs_others
ROM_END





ROM_START( sc4game )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406470.lo", 0x00001, 0x080000, CRC(8471230a) SHA1(d72eef5b8addbaa60d1bda9b3e62b48728ddc818) )
	ROM_LOAD16_BYTE( "95406471.hi", 0x00000, 0x080000, CRC(7dc7e054) SHA1(9ad9c218fb55ba2841fd75dbcdb56d1f1dbc41e9) )
	sc_game_others
ROM_END




ROM_START( sc4ziggy )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420166.lo", 0x00001, 0x080000, CRC(a1f64ad8) SHA1(4cfb0fae5a7d3a8323958019e5813f422d3b3fba) )
	ROM_LOAD16_BYTE( "95420167.hi", 0x00000, 0x080000, CRC(4e4bfed4) SHA1(9a39c9d607cfa905ba93b0e4838ee7cf7d43789a) )
	sc_ziggy_others
ROM_END

ROM_START( sc4ziggya )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420168.lo", 0x00001, 0x080000, CRC(86518fea) SHA1(9a8e97f7bbd7ba51967c06633ff597f03cfb2ea8) )
	ROM_LOAD16_BYTE( "95420169.hi", 0x00000, 0x080000, CRC(bae83963) SHA1(b47302d0d725b6c890349b0449decf6f9a28e77f) )
	sc_ziggy_others
ROM_END

ROM_START( sc4ziggyb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420482.lo", 0x00001, 0x080000, CRC(a46303eb) SHA1(74c997ee7ba5e93ac0eda373761946f21449e177) )
	ROM_LOAD16_BYTE( "95420483.hi", 0x00000, 0x080000, CRC(6bdd94aa) SHA1(44c6e07b5d308020a79d57b0974a134a595982d7) )
	sc_ziggy_others
ROM_END

ROM_START( sc4ziggyc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420484.lo", 0x00001, 0x080000, CRC(b1079049) SHA1(d6addd044b7529e033ced032709d55f77c41cfb3) )
	ROM_LOAD16_BYTE( "95420485.hi", 0x00000, 0x080000, CRC(ddb84927) SHA1(48ee3072c423fe7be5ecc31dc57acc31cd8576b8) )
	sc_ziggy_others
ROM_END

ROM_START( sc4ziggyd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421166.lo", 0x00001, 0x080000, CRC(4cfa8c0c) SHA1(f9612a80dde3d7054c03cb727835db5c40911f2c) )
	ROM_LOAD16_BYTE( "95421167.hi", 0x00000, 0x080000, CRC(6ae0a0fe) SHA1(5802e5fa5e9a18fa66ab541790af1ad62023a4c2) )
	sc_ziggy_others
ROM_END

ROM_START( sc4ziggye )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421168.lo", 0x00001, 0x080000, CRC(9731db4e) SHA1(4e849eabc58dec4420697d2c205db8dd0614fb32) )
	ROM_LOAD16_BYTE( "95421169.hi", 0x00000, 0x080000, CRC(0c997a9e) SHA1(3c13a9986fd2fb057f396f853d6495cd39c8b545) )
	sc_ziggy_others
ROM_END

ROM_START( sc4ziggyf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421482.lo", 0x00001, 0x080000, CRC(19e07df0) SHA1(23862138c21ed236f3ff37b3151ac271450a04af) )
	ROM_LOAD16_BYTE( "95421483.hi", 0x00000, 0x080000, CRC(f70321c1) SHA1(986a98e8a866e31c81bb94fbecb5f8ec826e704d) )
	sc_ziggy_others
ROM_END

ROM_START( sc4ziggyg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421484.lo", 0x00001, 0x080000, CRC(34924d7a) SHA1(721594c08780a80b9019f4cde2a01cde25fd415d) )
	ROM_LOAD16_BYTE( "95421485.hi", 0x00000, 0x080000, CRC(791d07f7) SHA1(b9f0133d60eafab983e3a45c6dbbec03d5b9f24b) )
	sc_ziggy_others
ROM_END







ROM_START( sc4gldcl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416882.lo", 0x00001, 0x080000, CRC(ee246097) SHA1(a17d4a494a6b2d3d608fddb82214e3456d817b15) )
	ROM_LOAD16_BYTE( "95416883.hi", 0x00000, 0x080000, CRC(f05997a4) SHA1(04861a9923ba437717d03f98e1670cfce45ef489) )
	sc_gldcl_others
ROM_END

ROM_START( sc4gldcla )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417882.lo", 0x00001, 0x080000, CRC(0a292b22) SHA1(0af7b907f44969b3e02728ad059679915043ee2c) )
	ROM_LOAD16_BYTE( "95417883.hi", 0x00000, 0x080000, CRC(f0baaee4) SHA1(33dffbeedba02002ec09b58ee2fe4b81a838379b) )
	sc_gldcl_others
ROM_END




ROM_START( sc4gfev )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412110.lo", 0x00001, 0x080000, CRC(8bf4bdd1) SHA1(33106003c3ba80ec918ffd0b5ac61e6cab8cb60a) )
	ROM_LOAD16_BYTE( "95412111.hi", 0x00000, 0x080000, CRC(7eae9cee) SHA1(51f4426c6bcac7e53f296ad1f9592cd3272fd91a) )
	sc_gfev_others
ROM_END

ROM_START( sc4gfeva )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416144.lo", 0x00001, 0x080000, CRC(7d4ed381) SHA1(1ec73ffe04a4ae4fc35bc3e3b74fd98e2e0ad03e) ) // (in monopoply wow)
	ROM_LOAD16_BYTE( "95416145.hi", 0x00000, 0x080000, CRC(6acba346) SHA1(93240893f6a988ee1bd24feb862ed2d0681de976) ) // (in sc4winsp?)
	sc_gfev_others
ROM_END

ROM_START( sc4gfevb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416146.lo", 0x00001, 0x080000, CRC(d4a039fb) SHA1(7a0094695b6bb909250e81cbb096e67d4654957b) )
	ROM_LOAD16_BYTE( "95416147.hi", 0x00000, 0x080000, CRC(3fbce341) SHA1(498081059fa07460c6ede28c4b313747236254b6) )
	sc_gfev_others
ROM_END

ROM_START( sc4db )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416326.lo", 0x00001, 0x080000, CRC(131c13b5) SHA1(9fbce59022f8489d156296c3aa5a0acc1c62ffe5) )
	ROM_LOAD16_BYTE( "95416327.hi", 0x00000, 0x080000, CRC(5dbcfdb6) SHA1(f017cad0eb55937936f3908add20aaa0321e8b8d) )
	sc_gfev_others
ROM_END



ROM_START( sc4ggcas )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406766.lo", 0x00001, 0x080000, CRC(7aead8cd) SHA1(ff75939c2ae19163e15eecd81200f9fa17998cbc) )
	ROM_LOAD16_BYTE( "95406767.hi", 0x00000, 0x080000, CRC(7003bb2e) SHA1(20a7b0b4c85ee99d72e6296d3a905fab0a5a1d69) )
	sc_ggcas_others
ROM_END

ROM_START( sc4ggcasa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406770.lo", 0x00001, 0x080000, CRC(08c703cf) SHA1(c99ff567af7dd3b75d8041147d994d7c8da4dfb8) )
	ROM_LOAD16_BYTE( "95406771.hi", 0x00000, 0x080000, CRC(7b278ba8) SHA1(ec46ac2bc35a477d816b34b42f943e0e2d0c66b0) )
	sc_ggcas_others
ROM_END

ROM_START( sc4ggcasb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406772.lo", 0x00001, 0x080000, CRC(0ae6fcff) SHA1(1aeaa2f55fd72386f4379c1160b0f3f0167ffc86) )
	ROM_LOAD16_BYTE( "95406773.hi", 0x00000, 0x080000, CRC(05c35548) SHA1(cfa131c6e1480ef9a8e5304fa242e3a843937de7) )
	sc_ggcas_others
ROM_END

ROM_START( sc4ggcasc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407766.lo", 0x00001, 0x080000, CRC(f2dee74f) SHA1(9afc799b699dda20a3e91a4e392d705dbe11fc83) )
	ROM_LOAD16_BYTE( "95407767.hi", 0x00000, 0x080000, CRC(d577ccad) SHA1(fb82a45bbe900bad2149986e5df69d236915982f) )
	sc_ggcas_others
ROM_END




ROM_START( sc4gnc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408052.lo", 0x00001, 0x080000, CRC(fea41987) SHA1(0012decf9467db65e7fc5a957909c8bd3223df3f) )
	ROM_LOAD16_BYTE( "95408053.hi", 0x00000, 0x080000, CRC(2794c569) SHA1(9b7bf233fdf3c475ef5617e78b7002dc3fcbf416) )
	sc_gnc_others
ROM_END

ROM_START( sc4gnca )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408392.lo", 0x00001, 0x080000, CRC(0b15aa9c) SHA1(ae15977b13a16f56ab0b38f9a2f3c058ce9fb86a) )
	ROM_LOAD16_BYTE( "95408393.hi", 0x00000, 0x080000, CRC(115ad467) SHA1(8a89c579da9e43b53230d6bf026150b810ca1372) )
	sc_gnc_others
ROM_END

ROM_START( sc4gncb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409052.lo", 0x00001, 0x080000, CRC(0b36a055) SHA1(52dbbc7727e27c7cb802aba46a3b14859feb3b77) )
	ROM_LOAD16_BYTE( "95409053.hi", 0x00000, 0x080000, CRC(73fce66c) SHA1(c0a5a238a654944875802a088de50c66ad00776d) )
	sc_gnc_others
ROM_END

ROM_START( sc4gncc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409392.lo", 0x00001, 0x080000, CRC(87c6be05) SHA1(c15c14d8ca25674ef8f7d7f8a287c57ca75889e4) )
	ROM_LOAD16_BYTE( "95409393.hi", 0x00000, 0x080000, CRC(c2474cbe) SHA1(33fe7d07bf7945035e84560b0c6cb8f3a2c7a9dc) )
	sc_gnc_others
ROM_END

ROM_START( sc4gncd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410358.lo", 0x00001, 0x080000, CRC(75a784dc) SHA1(c53575e3406a6de2e59ac3ae6a14d936d773316b) )
	ROM_LOAD16_BYTE( "95410359.hi", 0x00000, 0x080000, CRC(5d0694fe) SHA1(a4561e12fd221f07dcac9061bba7eadb012b8578) )
	sc_gnc_others
ROM_END

ROM_START( sc4gnce )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411358.lo", 0x00001, 0x080000, CRC(121e42db) SHA1(16f5b557aad85e426a30341e830c1b39d7ebc045) )
	ROM_LOAD16_BYTE( "95411359.hi", 0x00000, 0x080000, CRC(aef8cea0) SHA1(04472d96d7796f00ddf1d25a228c0806797906f7) )
	sc_gnc_others
ROM_END




ROM_START( sc4ghost )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416694.lo", 0x00001, 0x080000, CRC(39c14d15) SHA1(10722511af8aeac80e48c8910ee6a7c3dd0cbaf9) )
	ROM_LOAD16_BYTE( "95416695.hi", 0x00000, 0x080000, CRC(2de60335) SHA1(8249afd9237d7ad8c6643a93084768edf66e3971) )
	sc_ghost_others
ROM_END

ROM_START( sc4ghosta )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416696.lo", 0x00001, 0x080000, CRC(e227eef0) SHA1(a7164790b7cd8fc4522e4a90e193b9e5e00b757d) )
	ROM_LOAD16_BYTE( "95416697.hi", 0x00000, 0x080000, CRC(ed085935) SHA1(459befa07a553c31021c84392a31c93ff633112e) )
	sc_ghost_others
ROM_END

ROM_START( sc4ghostb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416754.lo", 0x00001, 0x080000, CRC(d5a70201) SHA1(025bca73253c6d3a716892d40c590c1ca70831a1) )
	ROM_LOAD16_BYTE( "95416755.hi", 0x00000, 0x080000, CRC(926c7df5) SHA1(79fd13d6f4fc0d9d1f308db6b7793313d2513e26) )
	sc_ghost_others
ROM_END

ROM_START( sc4ghostc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416782.lo", 0x00001, 0x080000, CRC(240534ce) SHA1(9d82334bc7472219b88962fe40afa9030b7c01ab) )
	ROM_LOAD16_BYTE( "95416783.hi", 0x00000, 0x080000, CRC(6c7f4033) SHA1(38c8527392647423124099c32725aadd6b0f0482) )
	sc_ghost_others
ROM_END

ROM_START( sc4ghostd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416958.lo", 0x00001, 0x080000, CRC(1b587f98) SHA1(37242dc9611de47f80fca32a4f35ac69b69bfa0d) )
	ROM_LOAD16_BYTE( "95416959.hi", 0x00000, 0x080000, CRC(6a722439) SHA1(5eeaefde3eb2978aee6578b96e92d9cb402d6ebf) )
	sc_ghost_others
ROM_END

ROM_START( sc4ghoste )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417694.lo", 0x00001, 0x080000, CRC(7e730d77) SHA1(e431deefcadcb36c7732f947b52ad882f84a40ea) )
	ROM_LOAD16_BYTE( "95417695.hi", 0x00000, 0x080000, CRC(48435e53) SHA1(227feaca9cb35f04c3947f1891320ecb7f98389d) )
	sc_ghost_others
ROM_END

ROM_START( sc4ghostf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417754.lo", 0x00001, 0x080000, CRC(f15a8c51) SHA1(8219edf45a1d0c001c2902e6f0849dc0aa9ffa14) )
	ROM_LOAD16_BYTE( "95417755.hi", 0x00000, 0x080000, CRC(39ffcf6f) SHA1(3c50ce386c8d1f037890af3ce1528e30efdfba84) )
	sc_ghost_others
ROM_END

ROM_START( sc4ghostg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417782.lo", 0x00001, 0x080000, CRC(742e0f21) SHA1(0c7589cb2db52f00b42157b21ba79dd9e7d24185) )
	ROM_LOAD16_BYTE( "95417783.hi", 0x00000, 0x080000, CRC(54971536) SHA1(7fdf1fc4a0d661319e4e4668ad70ff2d0ad282e5) )
	sc_ghost_others
ROM_END

ROM_START( sc4ghosth )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417958.lo", 0x00001, 0x080000, CRC(a0281b11) SHA1(cc952039d2ae9117ee9b25e718b1cfa1a6a44056) )
	ROM_LOAD16_BYTE( "95417959.hi", 0x00000, 0x080000, CRC(b69f2462) SHA1(5a17d135c94d78a8cf45b53a7d143ab3d41ba013) )
	sc_ghost_others
ROM_END



ROM_START( sc4ggrid )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408318.odd", 0x00001, 0x080000, CRC(6de26970) SHA1(fe43e4ed9aa2b8843db987f47dbe3a9b3bf78fda) )
	ROM_LOAD16_BYTE( "95408319.evn", 0x00000, 0x080000, CRC(a618fc9f) SHA1(320125c222163eef03b4fb3f663871c22fc4ff61) )
	sc_ggrid_others
ROM_END

ROM_START( sc4ggrida )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408320.odd", 0x00001, 0x080000, CRC(e03bfaa5) SHA1(a85b3198fede56182d4c6a8f66737a43620ad94a) )
	ROM_LOAD16_BYTE( "95408321.evn", 0x00000, 0x080000, CRC(84270b86) SHA1(c78f4c939810bd83832a3bb54e6bf51b5393c741) )
	sc_ggrid_others
ROM_END

ROM_START( sc4ggridb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408524.odd", 0x00001, 0x080000, CRC(19a1d2d0) SHA1(e90cb1c7eee650887510d68094a4f0109469809b) )
	ROM_LOAD16_BYTE( "95408525.evn", 0x00000, 0x080000, CRC(ed20b2b2) SHA1(dbd4119eb72c1304b20839a0ebfa6da0aa0320ae) )
	sc_ggrid_others
ROM_END

ROM_START( sc4ggridc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408526.odd", 0x00001, 0x080000, CRC(173460df) SHA1(dc7dfd9a004d262c1935e0fed2cb9fb6463637be) )
	ROM_LOAD16_BYTE( "95408527.evn", 0x00000, 0x080000, CRC(292672a2) SHA1(60aa190df0c5565eb6b2143e74b9cb106235f87c) )
	sc_ggrid_others
ROM_END

ROM_START( sc4ggridd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408936.odd", 0x00001, 0x080000, CRC(dc598886) SHA1(2ec6cee160b25033421881f5f76f366619ef0f8b) )
	ROM_LOAD16_BYTE( "95408937.evn", 0x00000, 0x080000, CRC(bbf36c44) SHA1(317726644cbced27d89087bd7b58de446c93bbd5) )
	sc_ggrid_others
ROM_END

ROM_START( sc4ggride )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409318.odd", 0x00001, 0x080000, CRC(04527589) SHA1(f1eef3602e819ce19f2ce7fb9d4b4eeaa513eabc) )
	ROM_LOAD16_BYTE( "95409319.evn", 0x00000, 0x080000, CRC(f2ef89a2) SHA1(e6863ef9ed5cb0d43ae9443b6f79fe31e72b177e) )
	sc_ggrid_others
ROM_END

ROM_START( sc4ggridf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409320.odd", 0x00001, 0x080000, CRC(898be65c) SHA1(5086970ee8a95ccd9d291dbf1ac56abec340aab7) )
	ROM_LOAD16_BYTE( "95409321.evn", 0x00000, 0x080000, CRC(d0d07ebb) SHA1(b2f63f3860e904ce138819474b30a851cad52007) )
	sc_ggrid_others
ROM_END

ROM_START( sc4ggridg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409524.odd", 0x00001, 0x080000, CRC(ebdf54c8) SHA1(9e311f79c2d9bc2883f42a0543b49991ba4930bf) )
	ROM_LOAD16_BYTE( "95409525.evn", 0x00000, 0x080000, CRC(81a0c4b2) SHA1(5a167834e657348523d6d421b3926c2f7d947020) )
	sc_ggrid_others
ROM_END

ROM_START( sc4ggridh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409526.odd", 0x00001, 0x080000, CRC(e54ae6c7) SHA1(b71de332efaf2f4340918b2f0fd932c6f772248e) )
	ROM_LOAD16_BYTE( "95409527.evn", 0x00000, 0x080000, CRC(45a604a2) SHA1(41fa4fc68585b5a5e8915d54fc269e87ec414743) )
	sc_ggrid_others
ROM_END

ROM_START( sc4ggridi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410768.odd", 0x00001, 0x080000, CRC(b0badd85) SHA1(5a38c6a3f5cc99d5af0d659d6e61c99d259f4843) )
	ROM_LOAD16_BYTE( "95410769.evn", 0x00000, 0x080000, CRC(32a4e1bb) SHA1(45ff4a4a432bad8208ad4027d70b3ec635791e61) )
	sc_ggrid_others
ROM_END

ROM_START( sc4ggridj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410770.odd", 0x00001, 0x080000, CRC(de71458e) SHA1(1541d3b52d79b2b18d1413fc01a9f83439382a05) )
	ROM_LOAD16_BYTE( "95410771.evn", 0x00000, 0x080000, CRC(1b822802) SHA1(bea0a4d3f285c080ea7d5b7b0a00a8f598923685) )
	sc_ggrid_others
ROM_END

ROM_START( sc4ggridk )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411768.odd", 0x00001, 0x080000, CRC(f40284a1) SHA1(f6a13523ff5f63f8664624118b00dc32edcd8f92) )
	ROM_LOAD16_BYTE( "95411769.evn", 0x00000, 0x080000, CRC(c327e305) SHA1(84c8e0f3516a2f0b81e874f814390407dacf50aa) )
	sc_ggrid_others
ROM_END

ROM_START( sc4ggridl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411770.odd", 0x00001, 0x080000, CRC(7c2ba1dc) SHA1(1f639e88f216564b4b95c78fa3e48f233d5921b0) )
	ROM_LOAD16_BYTE( "95411771.evn", 0x00000, 0x080000, CRC(59990fa4) SHA1(5e25f3c487f4971e74a814fefd84d7b1aba54d04) )
	sc_ggrid_others
ROM_END

ROM_START( sc4ggridm )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414696.odd", 0x00001, 0x080000, CRC(ab6c313c) SHA1(36b1b3917a049088c05f2cdce53f7d851d6be5e6) )
	ROM_LOAD16_BYTE( "95414697.evn", 0x00000, 0x080000, CRC(68c70d54) SHA1(eaec9da8bf9a849925e85e3cfd66c632da2e282f) )
	sc_ggrid_others
ROM_END

ROM_START( sc4ggridn )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415696.odd", 0x00001, 0x080000, CRC(64d6a62b) SHA1(23c999542004aaf100cd05919138b27f56b2c0c1) )
	ROM_LOAD16_BYTE( "95415697.evn", 0x00000, 0x080000, CRC(cc12809f) SHA1(68925025498630b57d23ecc92ad7e8221525d12b) )
	sc_ggrid_others
ROM_END




ROM_START( sc4ggcl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410136.odd", 0x00001, 0x080000, CRC(b833c07d) SHA1(d3eba62f20ae0d72df9ec52ade3b958539046f57) )
	ROM_LOAD16_BYTE( "95410137.evn", 0x00000, 0x080000, CRC(b6b100d7) SHA1(a44bde13b49c975018381c0813ba81d57ae996ca) )
	sc4ggcl_others
ROM_END

ROM_START( sc4ggcla )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410906.odd", 0x00001, 0x080000, CRC(3b662c9d) SHA1(905d24b6d7fc025131df5589566ab6b1ec946208) )
	ROM_LOAD16_BYTE( "95410907.evn", 0x00000, 0x080000, CRC(b4e1e4e9) SHA1(6ba91db947a9bc991d3e04f946973440be043a6d) )
	sc4ggcl_others
ROM_END

ROM_START( sc4ggclb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411906.odd", 0x00001, 0x080000, CRC(359d3081) SHA1(cd471e8dde352f361a7d21bc160ff1edeec140b2) )
	ROM_LOAD16_BYTE( "95411907.evn", 0x00000, 0x080000, CRC(1ed3ac3b) SHA1(4a2697d7a8092f8f84d51e76a46d85b4559553a7) )
	sc4ggcl_others
ROM_END

ROM_START( sc4ggclc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412160.odd", 0x00001, 0x080000, CRC(30861779) SHA1(c722bb4e074298f8c9bee05d0bd5778bcedb10b8) )
	ROM_LOAD16_BYTE( "95412161.evn", 0x00000, 0x080000, CRC(797565dc) SHA1(79f4331904267c45fd41d41871a06352e78f01d7) )
	sc4ggcl_others
ROM_END

ROM_START( sc4ggcld )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413160.odd", 0x00001, 0x080000, CRC(ea99ee09) SHA1(a319083b848fd433d5b3ebc2b41ec80e181ac782) )
	ROM_LOAD16_BYTE( "95413161.evn", 0x00000, 0x080000, CRC(4f7af5c0) SHA1(a86f11fa1692a552229a5ec7213c69f385ab9e33) )
	sc4ggcl_others
ROM_END




ROM_START( sc4goldo )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401621.lo", 0x00001, 0x080000, CRC(9999b865) SHA1(99ff2ed2b73abf278c5c7253702b6cb654c9a946) )
	ROM_LOAD16_BYTE( "95401622.hi", 0x00000, 0x080000, CRC(3075719d) SHA1(392d5fdce4253e0cc88cb0f283605a2b189aa792) )
	sc4goldo_others
ROM_END

ROM_START( sc4goldoa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401639.lo", 0x00001, 0x080000, CRC(b19dd39d) SHA1(418d6d97036072c2ecb48167878c07f2c9549103) )
	ROM_LOAD16_BYTE( "95401640.hi", 0x00000, 0x080000, CRC(4863dc90) SHA1(e5ce472e04c135d3553de66bd34d8b436ef6bd71) )
	sc4goldo_others
ROM_END

ROM_START( sc4goldob )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401645.lo", 0x00001, 0x080000, CRC(9c334258) SHA1(2da7df5fb8fc07128a5f2a0416a070f7f1fc846c) )
	ROM_LOAD16_BYTE( "95401646.hi", 0x00000, 0x080000, CRC(fc55479b) SHA1(c19c70e51ca6dd850afd2d1527af89939a2630fb) )
	sc4goldo_others
ROM_END

ROM_START( sc4goldoc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	// why do we only have a combined rom for this one? (is it same as sc4gocasa once split?)
	ROM_LOAD( "95401643.bin", 0x0000, 0x100000, CRC(3a85490e) SHA1(e1e4765785d631a7ec4e5daa33d1ce4fbdaee446) ) // TODO SPLIT THIS
	sc4goldo_others
ROM_END

ROM_START( sc4gocas )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400623.lo", 0x00001, 0x080000, CRC(51a41329) SHA1(af87c760715996fd5b52a8a45e1bcb6f7f7747bd) )
	ROM_LOAD16_BYTE( "95400624.hi", 0x00000, 0x080000, CRC(f04779ce) SHA1(2268e8502e265a62dbfc6a7d3b64847a2ba0e370) )
	sc4goldo_others
ROM_END

ROM_START( sc4gocasa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401623.lo", 0x00001, 0x080000, CRC(623d6dd3) SHA1(fede0f19764927ee45f2b167c33e73dd3e4e540d) )
	ROM_LOAD16_BYTE( "95401624.hi", 0x00000, 0x080000, CRC(d3e74fd9) SHA1(0d3c1c264290727cf01e6f9f6bec43e52423917c) )
	sc4goldo_others
ROM_END




ROM_START( sc4gshot )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416228.odd", 0x00001, 0x080000, CRC(b8e2b203) SHA1(586be00bc166003705af43d8bb48102e93bb40ea) )
	ROM_LOAD16_BYTE( "95416229.evn", 0x00000, 0x080000, CRC(96d6b36b) SHA1(3d4544805a2c38a42a9eccb3456a7956da506e23) )
	sc_gshot_others
ROM_END

ROM_START( sc4gshota )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416230.lo", 0x00001, 0x080000, CRC(54f4f94d) SHA1(9fb208cda0d963bce229f52fce6b8d44abd98a35) )
	ROM_LOAD16_BYTE( "95416231.hi", 0x00000, 0x080000, CRC(b8893573) SHA1(16037c6aa3b44ab4d49a8cddef16a347b6e0c84c) )
	sc_gshot_others
ROM_END

ROM_START( sc4gshotb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417228.odd", 0x00001, 0x080000, CRC(a087308c) SHA1(5fbe69cbfa177d86351254b30fa8a5a95a3ad726) )
	ROM_LOAD16_BYTE( "95417229.evn", 0x00000, 0x080000, CRC(113c5e8f) SHA1(56c303ff2e7a110c7cc08a720e1ec307892807ce) )
	sc_gshot_others
ROM_END

ROM_START( sc4gshotc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417230.odd", 0x00001, 0x080000, CRC(fab286d5) SHA1(f215bfd5449de3fcda604a0a764c390fc2b55496) )
	ROM_LOAD16_BYTE( "95417231.evn", 0x00000, 0x080000, CRC(98f5a1cc) SHA1(7d16ccdfa5f84ec7a5e7c759c7665d9130ccbfd3) )
	sc_gshot_others
ROM_END




ROM_START( sc4goldw )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408416.lo", 0x00001, 0x080000, CRC(0ba687f0) SHA1(503756236a3e215b38e439d6d2833730cdd866e9) )
	ROM_LOAD16_BYTE( "95408417.hi", 0x00000, 0x080000, CRC(722a756e) SHA1(afbf108239484c177ead3925d990ca45b33951ca) )
	sc_goldw_others
ROM_END

ROM_START( sc4goldwa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409416.lo", 0x00001, 0x080000, CRC(ae198161) SHA1(15ab3284ae508b896129b7474b1f506a73bca982) )
	ROM_LOAD16_BYTE( "95409417.hi", 0x00000, 0x080000, CRC(3f776d99) SHA1(a58c91d2293ed54d2c7ec2ab0eeacb0b16c562e6) )
	sc_goldw_others
ROM_END

ROM_START( sc4goldwb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1431p11.hi", 0x00000, 0x080000, CRC(93910114) SHA1(afb3a756a9667401dbb4299c09139769275a1603) )
	ROM_LOAD16_BYTE( "p1431p11.lo", 0x00001, 0x080000, CRC(636c0743) SHA1(7e5ab61193ee3b387462205a06a9fc386f8fd81d) )
	sc_goldw_others
ROM_END

ROM_START( sc4goldwc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1431s11.hi", 0x00000, 0x080000, CRC(93c51760) SHA1(a3613d5c257772e7c3e02485eb859d306a316625) )
	ROM_LOAD16_BYTE( "p1431s11.lo", 0x00001, 0x080000, CRC(2228a427) SHA1(61133938b6389478833bac49228888d0e27a91a8) )
	sc_goldw_others
ROM_END

ROM_START( sc4goldwd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1437p41.hi", 0x00000, 0x080000, CRC(d5e38e06) SHA1(85be2c45747ee2e80614cd455ea6525e89c53de6) )
	ROM_LOAD16_BYTE( "p1437p41.lo", 0x00001, 0x080000, CRC(c31f15ff) SHA1(57247c02adc0e8f2826f11f6a80ccda1f668b274) )
	PR1436_TRIPLE_CASINO_SOUNDS11
ROM_END

ROM_START( sc4goldwe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1437p51.hi", 0x00000, 0x080000, CRC(d0bcbd31) SHA1(21700e771355776ea3b87113ac4f6b4e55732392) )
	ROM_LOAD16_BYTE( "p1437p51.lo", 0x00001, 0x080000, CRC(62378c3e) SHA1(0f5dc74b1ac4b294d3dac34e303d8a1de1001046) )
	PR1436_TRIPLE_CASINO_SOUNDS11
ROM_END

ROM_START( sc4goldwf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1437s41.hi", 0x00000, 0x080000, CRC(73f7f759) SHA1(6ef14b597ce94693f5db7f1b13daf91f218147a4) )
	ROM_LOAD16_BYTE( "p1437s41.lo", 0x00001, 0x080000, CRC(f49e4698) SHA1(1580070484c998175921f93fd0be2c12e1e7fefc) )
	PR1436_TRIPLE_CASINO_SOUNDS11
ROM_END

ROM_START( sc4goldwg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1437s51.hi", 0x00000, 0x080000, CRC(76a8c46e) SHA1(8bd5e7806bd03ef3b17a3b7291049fdc1708ef07) )
	ROM_LOAD16_BYTE( "p1437s51.lo", 0x00001, 0x080000, CRC(55b6df59) SHA1(6ce2dd23078a9411d8937f1e27f59e6fc337e406) )
	PR1436_TRIPLE_CASINO_SOUNDS11
ROM_END




ROM_START( sc4gx )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400965.lo", 0x00001, 0x080000, CRC(24f32a91) SHA1(5bb3ebb618ccbc31ead3a740a4920d2a5758d8ff) )
	ROM_LOAD16_BYTE( "95400966.hi", 0x00000, 0x080000, CRC(482551a8) SHA1(64a85c2ed6ffd30dbca792fcad8c1d71276a7060) )
	sc_gx_others
ROM_END



ROM_START( sc4gxa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404506.lo", 0x00001, 0x080000, CRC(a5ae17f5) SHA1(10b9b4107b4d4667852b12185923cd1daac40930) )
	ROM_LOAD16_BYTE( "95404507.hi", 0x00000, 0x080000, CRC(d9da7d19) SHA1(313724aa544d660f5bf9c9fa8cf9042c621590dc) )
	sc_gx_others
ROM_END

ROM_START( sc4gxb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405506.lo", 0x00001, 0x080000, CRC(a18a6c36) SHA1(e19b1e7e8855c558ad016eed0e0dbd1506c0a6cc) )
	ROM_LOAD16_BYTE( "95405507.hi", 0x00000, 0x080000, CRC(ce0883f2) SHA1(0969f95cbce2639f7b8be96fd5691e2960781e16) )
	sc_gx_others
ROM_END






ROM_START( sc4gxcasa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400967.lo", 0x00001, 0x080000, CRC(efd704bf) SHA1(11e73d0966caace5d8b7889a6d829848612bc22f) )
	ROM_LOAD16_BYTE( "95400968.hi", 0x00000, 0x080000, CRC(c78378e6) SHA1(a9bccac373f4626ecaac4ea1a25d618471497d32) )
	sc_gxcas_others
ROM_END

ROM_START( sc4gxcasb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401965.lo", 0x00001, 0x080000, CRC(4dc1eb17) SHA1(bf9ce47cf86798c693d8f35c4039f52a49c473e5) )
	ROM_LOAD16_BYTE( "95401966.hi", 0x00000, 0x080000, CRC(1235e18c) SHA1(fd6dd2166d635988b82cf25b346205d33ee5986d) )
	sc_gxcas_others
ROM_END

ROM_START( sc4gxcasc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401967.lo", 0x00001, 0x080000, CRC(ef4eb480) SHA1(ef1394868aa7e516d70c0742c481685d6325470f) )
	ROM_LOAD16_BYTE( "95401968.hi", 0x00000, 0x080000, CRC(7afc98e7) SHA1(c6128ab36d227cda3a552ed740d7308651c42df2) )
	sc_gxcas_others
ROM_END

ROM_START( sc4gxcasd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401969.lo", 0x00001, 0x080000, CRC(8679a51f) SHA1(ccf0d3b6008fa3a5aa802bd987523eb8f8ae8898) )
	ROM_LOAD16_BYTE( "95401970.hi", 0x00000, 0x080000, CRC(6143c28a) SHA1(1458e1b1784d1aa1c1ca4dabde8362ddf6c8f0b2) )
	sc_gxcas_others
ROM_END

ROM_START( sc4gxcase )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406414.lo", 0x00001, 0x080000, CRC(00ec6327) SHA1(95533c2653c09a5badd165dd475a79599e407e07) )
	ROM_LOAD16_BYTE( "95406415.hi", 0x00000, 0x080000, CRC(419314e8) SHA1(04d5e6f4d29fa6cd219cf24c901e61699c038b17) )
	sc_gxcas_others
ROM_END

ROM_START( sc4gxcasf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407414.lo", 0x00001, 0x080000, CRC(4b33fec9) SHA1(0a4cab128ae980aa72e06a507a127dec49dcafb1) )
	ROM_LOAD16_BYTE( "95407415.hi", 0x00000, 0x080000, CRC(3ac19c03) SHA1(cb0f797ade8d6f812fa771d2f61b8d04d4a4a609) )
	sc_gxcas_others
ROM_END




ROM_START( sc4gx3 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406838.lo", 0x00001, 0x080000, CRC(affe53c0) SHA1(70f08858126842e52cee1de9938dd5a0efa349c7) )
	ROM_LOAD16_BYTE( "95406839.hi", 0x00000, 0x080000, CRC(6ef1f574) SHA1(9005d34dbf43cfe6d657944bccf5060c82ea9359) )
	sc_gx3_others
ROM_END

ROM_START( sc4gx3a )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406840.lo", 0x00001, 0x080000, CRC(1a9d8d9a) SHA1(7ec948b9031232015137b16b7e90f4884b856a69) )
	ROM_LOAD16_BYTE( "95406841.hi", 0x00000, 0x080000, CRC(7f5c4b78) SHA1(6370d009fdd41e491fd681d6f73f83fee62e2ba4) )
	sc_gx3_others
ROM_END

ROM_START( sc4gx3b )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406842.lo", 0x00001, 0x080000, CRC(2bc5dc58) SHA1(59db45be261c64c44ca9c2e9cdb7b0112b6f7e3b) )
	ROM_LOAD16_BYTE( "95406843.hi", 0x00000, 0x080000, CRC(f0c1edb0) SHA1(2c1ba2ae57c124c4958de940e0144d3fd1eb9271) )
	sc_gx3_others
ROM_END

ROM_START( sc4gx3c )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406844.lo", 0x00001, 0x080000, CRC(cb93e0d8) SHA1(585a4fb6445d0f7254406334a6d98a012923ffd1) )
	ROM_LOAD16_BYTE( "95406845.hi", 0x00000, 0x080000, CRC(ddf94be9) SHA1(8e20822f93f01db6a676c8f017bfc43110f6a968) )
	sc_gx3_others
ROM_END

ROM_START( sc4gx3d )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407838.lo", 0x00001, 0x080000, CRC(a0c6f51e) SHA1(3b1247794e15fbac08442f30440c583727e50878) )
	ROM_LOAD16_BYTE( "95407839.hi", 0x00000, 0x080000, CRC(c621b37e) SHA1(1f053eea3f494450284c4a7a5136978c2051166a) )
	sc_gx3_others
ROM_END

ROM_START( sc4gx3e )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407840.lo", 0x00001, 0x080000, CRC(0698a60f) SHA1(97a7291e50530336d9775d1526dbb3e05fdbb72d) )
	ROM_LOAD16_BYTE( "95407841.hi", 0x00000, 0x080000, CRC(2a959235) SHA1(163fe57c0f51d00034b5c1609611a5770bf47686) )
	sc_gx3_others
ROM_END

ROM_START( sc4gx3f )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407842.lo", 0x00001, 0x080000, CRC(acd3068e) SHA1(c0fc0493acf77a56333b3f12f312ae2e951c6ad3) )
	ROM_LOAD16_BYTE( "95407843.hi", 0x00000, 0x080000, CRC(89ee3dbb) SHA1(177e02e3702cb7d229b2a19fc64370573baa3668) )
	sc_gx3_others
ROM_END

ROM_START( sc4gx3g )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407844.lo", 0x00001, 0x080000, CRC(fa8ec856) SHA1(25c4b5f71000515ef75bdbb93cfcc330ed264983) )
	ROM_LOAD16_BYTE( "95407845.hi", 0x00000, 0x080000, CRC(3471ded5) SHA1(2039b862bc23436ef8579a3b5a314a4a34e15608) )
	sc_gx3_others
ROM_END




ROM_START( sc4goud )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406468.lo", 0x00001, 0x080000, CRC(b3b7de39) SHA1(c7e5cb4327e6f247baeceda6e814858299309f9d) )
	ROM_LOAD16_BYTE( "95406469.hi", 0x00000, 0x080000, CRC(a0d912fa) SHA1(9d444227f8b8d161600e92dd38f31cf380100369) )
	sc_goud_others
ROM_END




ROM_START( sc4gag )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400381.lo", 0x00001, 0x080000, CRC(d1eed62d) SHA1(21e69bd5153c9839c18a3ef16a395ac9b2723e1b) )
	ROM_LOAD16_BYTE( "95400382.hi", 0x00000, 0x080000, CRC(06625ae8) SHA1(6005098914fb248c2a96bec7e867bc2a70bdbf46) )
	sc_gag_others
ROM_END

ROM_START( sc4gaga )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400619.lo", 0x00001, 0x080000, CRC(b56f01da) SHA1(42e9160909d2861935b3bfb10b46eb9689e26f28) )
	ROM_LOAD16_BYTE( "95400620.hi", 0x00000, 0x080000, CRC(ae208c83) SHA1(1b76de915352fb12ff14d8d88aa2e43876700e01) )
	sc_gag_others
ROM_END

ROM_START( sc4gagb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401383.lo", 0x00001, 0x080000, CRC(5d29d7d0) SHA1(625136fce9f971b5813865a02464b449a5b15c8c) )
	ROM_LOAD16_BYTE( "95401384.hi", 0x00000, 0x080000, CRC(ae3787d9) SHA1(449e3d0d8e74f27bda6c9d8faa04fdd6e9ac44fd) )
	sc_gag_others
ROM_END

ROM_START( sc4gagc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401619.lo", 0x00001, 0x080000, CRC(53a162d3) SHA1(478003404707f5d0d6edf484242a9d93d19b909f) )
	ROM_LOAD16_BYTE( "95401620.hi", 0x00000, 0x080000, CRC(f835f766) SHA1(0523b5e94caad7c54d00a1daac0e2d923e6604d4) )
	sc_gag_others
ROM_END



ROM_START( sc4gcb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416202.lo", 0x00001, 0x080000, CRC(cc3bf680) SHA1(9eb5ef3ef9f54f9586eea3896c5c5503df3509c4) )
	ROM_LOAD16_BYTE( "95416203.hi", 0x00000, 0x080000, CRC(6b80152e) SHA1(9da9f4d9e84e9e4706a16770f307d99368af7293) )
	sc_gcb_others
ROM_END

ROM_START( sc4gcba )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416204.lo", 0x00001, 0x080000, CRC(3ad1f80f) SHA1(55588937af95664f42b8e0c919782c977e47cf41) )
	ROM_LOAD16_BYTE( "95416205.hi", 0x00000, 0x080000, CRC(aafa7c22) SHA1(3e6f62448a630b76835081aec6df01cd746a902a) )
	sc_gcb_others
ROM_END

ROM_START( sc4gcbb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416250.lo", 0x00001, 0x080000, CRC(eaebc504) SHA1(3d0f913ac5a82099b5061ae9d22da106ad74adb3) )
	ROM_LOAD16_BYTE( "95416251.hi", 0x00000, 0x080000, CRC(3f48ac28) SHA1(7aff6aba17e5541ba947d4cea29768bd6c7a35a8) )
	sc_gcb_others
ROM_END

ROM_START( sc4gcbc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416252.lo", 0x00001, 0x080000, CRC(f04cb098) SHA1(dc3dbf2e01b34317e41df6aa5ca4fb5e890e7b2e) )
	ROM_LOAD16_BYTE( "95416253.hi", 0x00000, 0x080000, CRC(53fe6b01) SHA1(15ad70286dfccce999076f9b4b948bfc91bae80d) )
	sc_gcb_others
ROM_END

ROM_START( sc4gcbd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416358.lo", 0x00001, 0x080000, CRC(1fbfbe2d) SHA1(95a605f9ae2648ca1b7dc84a3125f2be3552536d) )
	ROM_LOAD16_BYTE( "95416359.hi", 0x00000, 0x080000, CRC(6c0a35b5) SHA1(63c48613ba132fe449cb3b46e7501d38b195db24) )
	sc_gcb_others
ROM_END

ROM_START( sc4gcbe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416362.lo", 0x00001, 0x080000, CRC(0d7707a5) SHA1(10f3b3a1da9949aac0f0603528fe8bb2228e1f72) )
	ROM_LOAD16_BYTE( "95416363.hi", 0x00000, 0x080000, CRC(f23a9e15) SHA1(e9006a5e6499750a3bb7396a71eeb39596e46912) )
	sc_gcb_others
ROM_END

ROM_START( sc4gcbf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417202.lo", 0x00001, 0x080000, CRC(08f5f698) SHA1(a79c1ea1b2d31685fb4fdd6af56decfb47e01f75) )
	ROM_LOAD16_BYTE( "95417203.hi", 0x00000, 0x080000, CRC(cf5598e5) SHA1(a48ad625fa3a25a1212d117f9d7a7538e24c07de) )
	sc_gcb_others
ROM_END

ROM_START( sc4gcbg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417204.lo", 0x00001, 0x080000, CRC(4f9f5cda) SHA1(71b91e8ff66b7aa65a364230da9a307f56913072) )
	ROM_LOAD16_BYTE( "95417205.hi", 0x00000, 0x080000, CRC(ea03555c) SHA1(8baa5f4bd4c16ecd314e51418f855b7856d4ab90) )
	sc_gcb_others
ROM_END

ROM_START( sc4gcbh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417250.lo", 0x00001, 0x080000, CRC(2a173447) SHA1(48773d3ef13c90faecd369c456f9e7fe4d98fb53) )
	ROM_LOAD16_BYTE( "95417251.hi", 0x00000, 0x080000, CRC(9639106a) SHA1(c5bd833bdec0d5fb657205116f6a3af525005663) )
	sc_gcb_others
ROM_END

ROM_START( sc4gcbi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417252.lo", 0x00001, 0x080000, CRC(7989fec8) SHA1(bf4b5c1ce09ffda902c486ee71b0fb2ab6db340d) )
	ROM_LOAD16_BYTE( "95417253.hi", 0x00000, 0x080000, CRC(47f03742) SHA1(11eeee14240d4e9ff0dd924e1df457ad6e08f329) )
	sc_gcb_others
ROM_END

ROM_START( sc4gcbj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "g_b_cash.lo", 0x00001, 0x080000, CRC(3471962b) SHA1(45579c4e7fca4eef46488fc4e16cbd76af9a7cae) )
	ROM_LOAD16_BYTE( "g_b_cash.hi", 0x00000, 0x080000, CRC(1620b7f9) SHA1(67b7f805a7a7c95b27938710158ca5cb070503d0) )
	sc_gcb_others
ROM_END





ROM_START( sc4greed )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1317p11.hi", 0x00000, 0x080000, CRC(8218f6f7) SHA1(2d8f0df098fa6cf23497eab05b0219a0ca448bda) )
	ROM_LOAD16_BYTE( "p1317p11.lo", 0x00001, 0x080000, CRC(fb049b80) SHA1(dab68cb21be72cdac8165b6915e664b5b3810d30) )
	sc_greed_others
ROM_END

ROM_START( sc4greeda )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1317s11.hi", 0x00000, 0x080000, CRC(3ef66b70) SHA1(8d856a5673374594f77f835017560348bd5f56d2) )
	ROM_LOAD16_BYTE( "p1317s11.lo", 0x00001, 0x080000, CRC(f6f37bb4) SHA1(3645568c8acc7ab4d4f10576b107b40d1c9644b0) )
	sc_greed_others
ROM_END





ROM_START( sc4hf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404214.lo", 0x00001, 0x080000, CRC(b942937b) SHA1(acd32234c251286be8358a55ac7487c550c9b6ac) )
	ROM_LOAD16_BYTE( "95404215.hi", 0x00000, 0x080000, CRC(70742de7) SHA1(e509704e4112657783f4100239ba4bf029c6d6bd) )
	sc_hf_others
ROM_END

ROM_START( sc4hfa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404220.lo", 0x00001, 0x080000, CRC(93a0020e) SHA1(c92457a4b2890a7bd53f229919a51f28d465c2bb) )
	ROM_LOAD16_BYTE( "95404221.hi", 0x00000, 0x080000, CRC(a412bf9c) SHA1(6276e8bbc88ee90aef93ec74357ee85ef0a9c6ae) )
	sc_hf_others
ROM_END

ROM_START( sc4hfb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404286.lo", 0x00001, 0x080000, CRC(6e1fa391) SHA1(14985afb407c96f0ce4b140b1486e2e4db68ae5f) )
	ROM_LOAD16_BYTE( "95404287.hi", 0x00000, 0x080000, CRC(2509ed7d) SHA1(b7b4d968052968a96b23b14c5f6dac29ebeb21a6) )
	sc_hf_others
ROM_END

ROM_START( sc4hfc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405214.lo", 0x00001, 0x080000, CRC(8cb9fc00) SHA1(9c6edeedfea9e6f2b582a429e3d5e82f32caef89) )
	ROM_LOAD16_BYTE( "95405215.hi", 0x00000, 0x080000, CRC(26fe003a) SHA1(641c6712a26f9eb1a757353467e339d2b57b93df) )
	sc_hf_others
ROM_END

ROM_START( sc4hfd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405220.lo", 0x00001, 0x080000, CRC(045d91b8) SHA1(6c3563279b4d615c8ecdee52fb4ba9db8588f107) )
	ROM_LOAD16_BYTE( "95405221.hi", 0x00000, 0x080000, CRC(02c60052) SHA1(76806c4ef25a426e8080a75fbd028c8808597fcd) )
	sc_hf_others
ROM_END

ROM_START( sc4hfe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405286.lo", 0x00001, 0x080000, CRC(6e1d414a) SHA1(f95a35d443779b1addc875061cec752cc1b9b03a) )
	ROM_LOAD16_BYTE( "95405287.hi", 0x00000, 0x080000, CRC(fa11be65) SHA1(42bb3baf08b55d07a8b0787d57ca0875e6f7cfbc) )
	sc_hf_others
ROM_END

ROM_START( sc4hff )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406018.lo", 0x00001, 0x080000, CRC(47b8d0ce) SHA1(0196ee3a78f9165fb6ee529f5a81881edc475276) )
	ROM_LOAD16_BYTE( "95406019.hi", 0x00000, 0x080000, CRC(f7ed0d65) SHA1(eeeebcc309e1605bc0ef7a0433b48dfccade9a7c) )
	sc_hf_others
ROM_END

ROM_START( sc4hfg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407018.lo", 0x00001, 0x080000, CRC(f6f9d680) SHA1(1332423e3f38f00c160d1b790285c4d43584eb14) )
	ROM_LOAD16_BYTE( "95407019.hi", 0x00000, 0x080000, CRC(727ab861) SHA1(866d1e15fb4951fc2d7c574265b6a026e9abe37f) )
	sc_hf_others
ROM_END




ROM_START( sc4hfcl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404100.lo", 0x00001, 0x080000, CRC(73fcd433) SHA1(45ad1b4a47b3a593f3e29f498f13dd9db01d621c) )
	ROM_LOAD16_BYTE( "95404101.hi", 0x00000, 0x080000, CRC(d7322d67) SHA1(29d0fb47c9606db8d86be21d1091876bedfa84b8) )
	sc_hfcl_others
ROM_END


ROM_START( sc4hfcla )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405100.lo", 0x00001, 0x080000, CRC(796fd1f5) SHA1(566cbbc900e3b7bb1872e8fce6b57d6ae7264c8e) )
	ROM_LOAD16_BYTE( "95405101.hi", 0x00000, 0x080000, CRC(70a4543c) SHA1(812438549faeac96f00143a25e037e43cd401562) )
	sc_hfcl_others
ROM_END






ROM_START( sc4heatw )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406486.lo", 0x00001, 0x080000, CRC(79961a2d) SHA1(4a7b8960daa24aa5a496914550e3101f55a060a4) )
	ROM_LOAD16_BYTE( "95406487.hi", 0x00000, 0x080000, CRC(a243da42) SHA1(7c94d43f1ee0b78b257c278d04331447427c94cf) )
	sc_heatw_others
ROM_END




ROM_START( sc4helrd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408462.lo", 0x00001, 0x080000, CRC(d88b84da) SHA1(8f1dee5e598e92ee82e0c323505041d107bbd38e) )
	ROM_LOAD16_BYTE( "95408463.hi", 0x00000, 0x080000, CRC(324102f4) SHA1(68b2e13149e42af5e7fc1e3968ebb086722ea51e) )
	sc_helrd_others
ROM_END




ROM_START( sc4helrs )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408640.lo", 0x00001, 0x080000, CRC(d5426f9d) SHA1(5f36371e396b4735977bceb9f8c58f00483457da) )
	ROM_LOAD16_BYTE( "95408641.hi", 0x00000, 0x080000, CRC(74129aa9) SHA1(01617c4e9b7f64ec49a259cf50e27a05a8096fc8) )
	sc_helrs_others
ROM_END





ROM_START( sc4hdd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400539.lo", 0x00001, 0x080000, CRC(ff4cf9de) SHA1(19638ee079aaf78eded317689e1a1adc28e16e20) )
	ROM_LOAD16_BYTE( "95400540.hi", 0x00000, 0x080000, CRC(d9facac7) SHA1(989f4564ecb13efdb9f3aa351c5b66a8046f6330) )
	sc_hdd_others
ROM_END

ROM_START( sc4hdda )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400713.lo", 0x00001, 0x080000, CRC(a3fd4b50) SHA1(d238d42146081ce45cf0e1464dda1b7bc0a47173) )
	ROM_LOAD16_BYTE( "95400714.hi", 0x00000, 0x080000, CRC(3e70fea8) SHA1(e2b8900f9e19e7e11bbda48ba6d8e33b62d6767d) )
	sc_hdd_others
ROM_END

ROM_START( sc4hddb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400717.lo", 0x00001, 0x080000, CRC(0bc9cd2b) SHA1(c8a2adef36f1ca91ab2f9c0f0a23d6fb4af0e01a) )
	ROM_LOAD16_BYTE( "95400718.hi", 0x00000, 0x080000, CRC(a0120196) SHA1(e1c0972f87b0ce68560e7b2441d7aed45142ddc4) )
	sc_hdd_others
ROM_END

ROM_START( sc4hddc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401539.lo", 0x00001, 0x080000, CRC(e8d5bb5e) SHA1(31254b4d020eca29bcf1a51ce561923f020e884f) )
	ROM_LOAD16_BYTE( "95401540.hi", 0x00000, 0x080000, CRC(da26686f) SHA1(6aa5653268933b0d0df9201824a18348b96646e7) )
	sc_hdd_others
ROM_END

ROM_START( sc4hddd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401541.lo", 0x00001, 0x080000, CRC(7b1e00aa) SHA1(d3896783e79e78a71f2e83d5b7a20bd86caab87f) )
	ROM_LOAD16_BYTE( "95401542.hi", 0x00000, 0x080000, CRC(a96d59d6) SHA1(010d3ad6fcb9df84ca0343ddb1cba972492fe154) )
	sc_hdd_others
ROM_END

ROM_START( sc4hdde )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401713.lo", 0x00001, 0x080000, CRC(1499cccf) SHA1(d47c54033bdb26399838e3234da8d6c30d0da32b) )
	ROM_LOAD16_BYTE( "95401714.hi", 0x00000, 0x080000, CRC(957c1a0a) SHA1(b315221517027707bd0e66486c86e4615814639c) )
	sc_hdd_others
ROM_END

ROM_START( sc4hddf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401717.lo", 0x00001, 0x080000, CRC(bcad4ab4) SHA1(5248f225012d12f978f107bc1810fa96ee437679) )
	ROM_LOAD16_BYTE( "95401718.hi", 0x00000, 0x080000, CRC(0b1ee534) SHA1(ba6c50b2f2cd94f7b614eb9ae38d3a9c4c95e3cf) )
	sc_hdd_others
ROM_END




ROM_START( sc4himi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406314.lo", 0x00001, 0x080000, CRC(3261fa40) SHA1(9c9c6a5c186c04b8944dc53c71eb91b95032099b) )
	ROM_LOAD16_BYTE( "95406315.hi", 0x00000, 0x080000, CRC(00ac5eb7) SHA1(d028d0d807548cb3b35186559961cc4b02610d2c) )
	sc_himi_others
ROM_END

ROM_START( sc4himia ) // was in deep impact set
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409226.lo", 0x00001, 0x080000, CRC(44da86a6) SHA1(d2738b798c0750a7020fbb339be556a7815fbd36) )
	ROM_LOAD16_BYTE( "95409227.hi", 0x00000, 0x080000, CRC(1de99322) SHA1(df7acf9ed94d3ffce3ea416f75b9506bbcc3b8fe) )
	sc_himi_others
ROM_END

ROM_START( sc4himib )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406630.lo", 0x00001, 0x080000, CRC(9b5a9254) SHA1(aeb70462bd0459692556c25645c186990d0239c6) )
	ROM_LOAD16_BYTE( "95406631.hi", 0x00000, 0x080000, CRC(6d672785) SHA1(95f8e5c76c0cc778d1e2cc6f2c442fb33db85f7f) )
	sc_himi_others
ROM_END

ROM_START( sc4himic )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407314.lo", 0x00001, 0x080000, CRC(0bf964fb) SHA1(169e2f58ee7c7a9dad628bde83db8ce02fe46112) )
	ROM_LOAD16_BYTE( "95407315.hi", 0x00000, 0x080000, CRC(d2103c26) SHA1(00ad4ac6dbbf083eacb8a74db3503d5a340111b1) )
	sc_himi_others
ROM_END

ROM_START( sc4himid )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407630.lo", 0x00001, 0x080000, CRC(9857010a) SHA1(8bb8a9f920eadb7db53ab8ce035a89a6fc3c2f76) )
	ROM_LOAD16_BYTE( "95407631.hi", 0x00000, 0x080000, CRC(4ff9bde2) SHA1(2a84d799e23b77a907cb47d20673c298009a6082) )
	sc_himi_others
ROM_END

ROM_START( sc4himie )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408226.lo", 0x00001, 0x080000, CRC(8064537e) SHA1(4a93fc2daa2fd0c00959be35f42b1df16b72d88c) )
	ROM_LOAD16_BYTE( "95408227.hi", 0x00000, 0x080000, CRC(11730edb) SHA1(5768f4e003a81adbde850f0ca6d78e025e35be9f) )
	sc_himi_others
ROM_END

ROM_START( sc4himif )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414294.lo", 0x00001, 0x080000, CRC(1270edf8) SHA1(2b349f483dbdb6340ce0a36373dabded6e64fa9d) )
	ROM_LOAD16_BYTE( "95414295.hi", 0x00000, 0x080000, CRC(c35e8bf3) SHA1(696f1fc8ba4e8fe161df7a1813ed9299df664941) )
	sc_himi_others
ROM_END

ROM_START( sc4himig )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414296.lo", 0x00001, 0x080000, CRC(3483183b) SHA1(e1bf97d764f64be849629b861c74c6471a932309) )
	ROM_LOAD16_BYTE( "95414297.hi", 0x00000, 0x080000, CRC(0288ef0c) SHA1(7b10ff4e83e569a368be6bd41e99943bcd3e90c9) )
	sc_himi_others
ROM_END

ROM_START( sc4himih )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415294.lo", 0x00001, 0x080000, CRC(ee4b8561) SHA1(7b48f86a0b2113b9e62c444013a6ab0204d9bcff) )
	ROM_LOAD16_BYTE( "95415295.hi", 0x00000, 0x080000, CRC(3242df75) SHA1(ca8dc4035bef9035ca1e1677410ca0f54c085c6d) )
	sc_himi_others
ROM_END

ROM_START( sc4himii )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415296.lo", 0x00001, 0x080000, CRC(c8b870a2) SHA1(cc9dfa0b78b41da2d93c310d6b40aa483ab7c92c) )
	ROM_LOAD16_BYTE( "95415297.hi", 0x00000, 0x080000, CRC(f394bb8a) SHA1(3ed3013a0eb3f6daafa2861edde44b9729b0a132) )
	sc_himi_others
ROM_END




ROM_START( sc4hi5 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402946.lo", 0x00001, 0x080000, CRC(f771bfa5) SHA1(f774666e66eaef49ef3fe3ef093b7bd8ab9223f7) )
	ROM_LOAD16_BYTE( "95402947.hi", 0x00000, 0x080000, CRC(41154e48) SHA1(355be96950e238e3caea7d8b51d197c23e4749af) )
	sc_hi5_others
ROM_END

ROM_START( sc4hi5a )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402948.lo", 0x00001, 0x080000, CRC(046caffe) SHA1(15f35b1baa44a9dd2d2772abb9da71a02d565760) )
	ROM_LOAD16_BYTE( "95402949.hi", 0x00000, 0x080000, CRC(c4215d0c) SHA1(4fc25fecc598950cc9edb25ccda516abe3bf9f70) )
	sc_hi5_others
ROM_END

ROM_START( sc4hi5b )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402964.lo", 0x00001, 0x080000, CRC(e6d80d1d) SHA1(fd22d0cd6611e4ec778968103c36bceab1d202f1) )
	ROM_LOAD16_BYTE( "95402965.hi", 0x00000, 0x080000, CRC(c16d6b5b) SHA1(d013ae1dc3739d5259c293ff2e0c2425ceec51ec) )
	sc_hi5_others
ROM_END

ROM_START( sc4hi5c )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402966.lo", 0x00001, 0x080000, CRC(e07555c5) SHA1(98b21c65e238d1a3749d5f0021110df1910db1c4) )
	ROM_LOAD16_BYTE( "95402967.hi", 0x00000, 0x080000, CRC(59302818) SHA1(ea7b3c1604bf403ac2c3dca7bf780203390203dd) )
	sc_hi5_others
ROM_END

ROM_START( sc4hi5d )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403946.lo", 0x00001, 0x080000, CRC(dbbdd3bc) SHA1(df9efd737b467dc502f77cc779923f4eca0ba8d7) )
	ROM_LOAD16_BYTE( "95403947.hi", 0x00000, 0x080000, CRC(785ce105) SHA1(82bccb49096c2b3b2d1440177d80c4d4c78f937d) )
	sc_hi5_others
ROM_END

ROM_START( sc4hi5e )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403948.lo", 0x00001, 0x080000, CRC(f7b890ff) SHA1(47c192f3a9a258bf44b903dabfccc30eeef4cbbb) )
	ROM_LOAD16_BYTE( "95403949.hi", 0x00000, 0x080000, CRC(e45dc9b3) SHA1(c83852613b0bbb90751b32f4441f2688345e1624) )
	sc_hi5_others
ROM_END

ROM_START( sc4hi5f )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403964.lo", 0x00001, 0x080000, CRC(d22732b4) SHA1(95a368d244a3eadc638a9f6c43aeda87db690444) )
	ROM_LOAD16_BYTE( "95403965.hi", 0x00000, 0x080000, CRC(cdf7f6a2) SHA1(415257fc2944b0e989ac76e0f6afa52502bee02d) )
	sc_hi5_others
ROM_END

ROM_START( sc4hi5g )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403966.lo", 0x00001, 0x080000, CRC(5c26b8d9) SHA1(e38042a45981d6336812b614f5b6cdabc53f54d0) )
	ROM_LOAD16_BYTE( "95403967.hi", 0x00000, 0x080000, CRC(57d7ed01) SHA1(286bf64b455e572fe392191885ef3e2076ae22c9) )
	sc_hi5_others
ROM_END




ROM_START( sc4sprng )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406592.lo", 0x00001, 0x080000, CRC(fbf0d51a) SHA1(320430a66f8d793ffa55b1f3bfa293deb9122ca3) )
	ROM_LOAD16_BYTE( "95406593.hi", 0x00000, 0x080000, CRC(5697bdc7) SHA1(c1e3952dd66772159b193a99bf4c8fbf92e0cf2f) )
	sc_sprng_others
ROM_END







ROM_START( sc4hilo )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404330.lo", 0x00001, 0x080000, CRC(8e7952a0) SHA1(1092a4aa124e2d07efe4fc92bd43cc23a49cca9b) )
	ROM_LOAD16_BYTE( "95404331.hi", 0x00000, 0x080000, CRC(02b4fb75) SHA1(6e310c1e80c7f6640d2bb98e8b60b49d86137674) )
	sc_hilo_others
ROM_END

ROM_START( sc4hiloa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404452.lo", 0x00001, 0x080000, CRC(b51678d8) SHA1(3b877975b1386691e2b377916f91e0248bd35f56) )
	ROM_LOAD16_BYTE( "95404453.hi", 0x00000, 0x080000, CRC(cd0aaa2f) SHA1(3c04bb3db09b988b83abc0cc49c6cc30ce2d5648) )
	sc_hilo_others
ROM_END

ROM_START( sc4hilob )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405330.lo", 0x00001, 0x080000, CRC(23067f38) SHA1(6feedee27bfc4f201576e80a072c8624f9848320) )
	ROM_LOAD16_BYTE( "95405331.hi", 0x00000, 0x080000, CRC(271c7551) SHA1(37228f0ceb0104a587671c7841244a8ffd6cf3e1) )
	sc_hilo_others
ROM_END

ROM_START( sc4hiloc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405452.lo", 0x00001, 0x080000, CRC(11bf33a6) SHA1(a8b0419528bb8bee028b27974d03512f37ce201f) )
	ROM_LOAD16_BYTE( "95405453.hi", 0x00000, 0x080000, CRC(002393d5) SHA1(24c288d119e88487273e5918af7bd2dc523ee791) )
	sc_hilo_others
ROM_END

ROM_START( sc4hilod )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406020.lo", 0x00001, 0x080000, CRC(e643afed) SHA1(6cba7056deaa963a7c7edc5a078787071fb64a03) )
	ROM_LOAD16_BYTE( "95406021.hi", 0x00000, 0x080000, CRC(b5206525) SHA1(7e69f45e921fd5ff651557acfd7ffe5afda4aa7c) )
	sc_hilo_others
ROM_END

ROM_START( sc4hiloe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406902.lo", 0x00001, 0x080000, CRC(87f40bc0) SHA1(69d33596d62bbf2019bae2d6cd5b486cdf9d5b45) )
	ROM_LOAD16_BYTE( "95406903.hi", 0x00000, 0x080000, CRC(6b6e2f21) SHA1(e20aa98988e21530d765480c048fa93e3e14c603) )
	sc_hilo_others
ROM_END

ROM_START( sc4hilof )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406904.lo", 0x00001, 0x080000, CRC(0ec41240) SHA1(e011501ae8e75765b451921f04350caaa04b7c2a) )
	ROM_LOAD16_BYTE( "95406905.hi", 0x00000, 0x080000, CRC(55f66037) SHA1(5139cb3e29696ac9dce142852b4db7d301b1761d) )
	sc_hilo_others
ROM_END

ROM_START( sc4hilog )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407020.lo", 0x00001, 0x080000, CRC(b415a33b) SHA1(653a0d9f682e894871720efbd4ee2fba49516ce4) )
	ROM_LOAD16_BYTE( "95407021.hi", 0x00000, 0x080000, CRC(09fe7f6c) SHA1(7c3822ce57ef45cec6e251a545c5378cfa54d999) )
	sc_hilo_others
ROM_END

ROM_START( sc4hiloh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407902.lo", 0x00001, 0x080000, CRC(a76eed40) SHA1(25d15035dce6e37d6cf8b28070166d457e038057) )
	ROM_LOAD16_BYTE( "95407903.hi", 0x00000, 0x080000, CRC(8c31e25f) SHA1(5757c01ec0535eb0ecdbcef2be13cf520b6eca5e) )
	sc_hilo_others
ROM_END

ROM_START( sc4hiloi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407904.lo", 0x00001, 0x080000, CRC(64b34099) SHA1(2e6956fe25072dea9bc1566bae0cdfe1676eb3ef) )
	ROM_LOAD16_BYTE( "95407905.hi", 0x00000, 0x080000, CRC(42e12406) SHA1(3eea44d6f46338e0c22bce2ea8770f885ac0c7fa) )
	sc_hilo_others
ROM_END

ROM_START( sc4hiloj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408332.lo", 0x00001, 0x080000, CRC(4d1c053a) SHA1(6dc8327897e976bd50a13ae73951e10fb3c4dbda) )
	ROM_LOAD16_BYTE( "95408333.hi", 0x00000, 0x080000, CRC(324c50af) SHA1(dcb7e2eb5d08d0827a1ca6480c1aeac7cd705a5f) )
	sc_hilo_others
ROM_END

ROM_START( sc4hilok )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409332.lo", 0x00001, 0x080000, CRC(4ad6ab9a) SHA1(1c7d151dac97758a43d8c85f9b625a37627384f0) )
	ROM_LOAD16_BYTE( "95409333.hi", 0x00000, 0x080000, CRC(cd60de97) SHA1(ec827addeaf97fce58024a2baae75ec1a1bf9bb0) )
	sc_hilo_others
ROM_END







ROM_START( sc4hitsh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406076.lo", 0x00001, 0x080000, CRC(bf3c9301) SHA1(695ac8edd5e6bbf32d9a8bc6d8586ed0e194b1cd) )
	ROM_LOAD16_BYTE( "95406077.hi", 0x00000, 0x080000, CRC(d3081ee2) SHA1(07eb41b8bb0ee3c5dcaf4c9413e7818e2a1c18b8) )
	sc_hitsh_others
ROM_END

ROM_START( sc4hitsha )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406310.lo", 0x00001, 0x080000, CRC(d6a639ab) SHA1(0760d5947de187290603a31deb925e7b9157344d) )
	ROM_LOAD16_BYTE( "95406311.hi", 0x00000, 0x080000, CRC(8e341ac0) SHA1(76a6499605d25c15eb7b35e5b4376313d8d47aa9) )
	sc_hitsh_others
ROM_END

ROM_START( sc4hitshb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407076.lo", 0x00001, 0x080000, CRC(3bb596e6) SHA1(12794b97feb3d7c8fba576b1259ee06662d397df) )
	ROM_LOAD16_BYTE( "95407077.hi", 0x00000, 0x080000, CRC(23e959fe) SHA1(20a94acff8ff31aa0ee209dbaae0d7071e6d8324) )
	sc_hitsh_others
ROM_END

ROM_START( sc4hitshc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407310.lo", 0x00001, 0x080000, CRC(59e647ba) SHA1(8793fd542e85d7b5b7e8fb2364a800e9eea7fd61) )
	ROM_LOAD16_BYTE( "95407311.hi", 0x00000, 0x080000, CRC(b9b4c722) SHA1(8d165226b4f0a437b9da7431b76505386eaa4115) )
	sc_hitsh_others
ROM_END

ROM_START( sc4hitshd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408272.lo", 0x00001, 0x080000, CRC(582e3280) SHA1(e5f78e94bb2d1eb0a673cc1103dd910b9bf1c582) )
	ROM_LOAD16_BYTE( "95408273.hi", 0x00000, 0x080000, CRC(448102ea) SHA1(582190bc07bcc03e0846c68e5b08795de1b3c74b) )
	sc_hitsh_others
ROM_END

ROM_START( sc4hitshe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409272.lo", 0x00001, 0x080000, CRC(2fb376c7) SHA1(ba875aa81c705c9a4b7edb65d9574c720b533769) )
	ROM_LOAD16_BYTE( "95409273.hi", 0x00000, 0x080000, CRC(3d3184d9) SHA1(1825bc5379e58db7319f391fca4d690a0dcb4b9c) )
	sc_hitsh_others
ROM_END




ROM_START( sc4holyw )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402151.lo", 0x00001, 0x080000, CRC(324c130b) SHA1(43088cf08ae96753b72c15c8b10bd508eb21935b) )
	ROM_LOAD16_BYTE( "95402152.hi", 0x00000, 0x080000, CRC(c491344e) SHA1(c67475afbcf736852eb5f63cf821f6d2460a4d19) )
	sc_holyw_others
ROM_END


ROM_START( sc4holywa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402153.lo", 0x00001, 0x080000, CRC(d78406fa) SHA1(2bad25ee032beb35def924cc30cfb0f6cacacbfd) )
	ROM_LOAD16_BYTE( "95402154.hi", 0x00000, 0x080000, CRC(7a23bcb9) SHA1(49ba57c4f45602a8454b405a7750bf08cbb062ea) )
	sc_holyw_others
ROM_END


ROM_START( sc4holywb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403151.lo", 0x00001, 0x080000, CRC(2ed32d9f) SHA1(294ecb67e49db4fbaeece6c50d93bf857562cc03) )
	ROM_LOAD16_BYTE( "95403152.hi", 0x00000, 0x080000, CRC(d09f680d) SHA1(375e2c9dd07c3f1f4c5ab1b1001c6b74dc5e018a) )
	sc_holyw_others
ROM_END


ROM_START( sc4holywc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403153.lo", 0x00001, 0x080000, CRC(2ceb3e73) SHA1(8816b6c2759bc5817a9e9b8484b828f0bb56fe22) )
	ROM_LOAD16_BYTE( "95403154.hi", 0x00000, 0x080000, CRC(e9585b26) SHA1(309c554e2bff84d47337b8b1b73dc4a8aac9c9bf) )
	sc_holyw_others
ROM_END









ROM_START( sc4hotpr )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402476.lo", 0x00001, 0x080000, CRC(e45bbd52) SHA1(a2cea405455404958b2ea7f55bbf38093400aafe) )
	ROM_LOAD16_BYTE( "95402477.hi", 0x00000, 0x080000, CRC(4c1d0262) SHA1(34b266a921f5e4270e099812d6132351e1d9ca32) )
	sc_hotpr_others
ROM_END

ROM_START( sc4hotpra )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402492.lo", 0x00001, 0x080000, CRC(cf58b928) SHA1(356b95461b7981a89a3a5f4c9971bd7ec1aa9a66) )
	ROM_LOAD16_BYTE( "95402493.hi", 0x00000, 0x080000, CRC(ffbb6a98) SHA1(9f561c0ab52019f17302b0f87ec9d9e5631238a7) )
	sc_hotpr_others
ROM_END

ROM_START( sc4hotprb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403476.lo", 0x00001, 0x080000, CRC(6a6a1bf5) SHA1(f72b34efd391234ffa08993a832dc3497a136477) )
	ROM_LOAD16_BYTE( "95403477.hi", 0x00000, 0x080000, CRC(bc2ca284) SHA1(427fde72e3aba0b84b30d2f5bd0d5cbe480be84f) )
	sc_hotpr_others
ROM_END

ROM_START( sc4hotprc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403492.lo", 0x00001, 0x080000, CRC(5a14b4f1) SHA1(421cf94634cb49fe91f7022aa89a7e845e509e93) )
	ROM_LOAD16_BYTE( "95403493.hi", 0x00000, 0x080000, CRC(7717b82d) SHA1(16658464ef1cc8a36a1a95471ec1cb6e97f7b80b) )
	sc_hotpr_others
ROM_END

ROM_START( sc4hotprd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404356.lo", 0x00001, 0x080000, CRC(ec7e87d8) SHA1(2562930cf42ee4c4583532320ab5dd21cb415f50) )
	ROM_LOAD16_BYTE( "95404357.hi", 0x00000, 0x080000, CRC(16081ab9) SHA1(2187eec6b881fedcf6be46259d6f4001ed1a2b25) )
	sc_hotpr_others
ROM_END

ROM_START( sc4hotpre )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405356.lo", 0x00001, 0x080000, CRC(b3857f12) SHA1(a7743a104cef0225148549aba6939c3d36d7045f) )
	ROM_LOAD16_BYTE( "95405357.hi", 0x00000, 0x080000, CRC(abe8ac80) SHA1(7176d8bbbbfb02bce56724ea47b62bd95feaa256) )
	sc_hotpr_others
ROM_END







ROM_START( sc4h6cl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95418148.lo", 0x00001, 0x080000, CRC(bf360c94) SHA1(0f07c1985a85579cbf5aba555cd492552a4b1c34) )
	ROM_LOAD16_BYTE( "95418149.hi", 0x00000, 0x080000, CRC(d540c97c) SHA1(5093a6c94d2dc1436d1add924c1af90cf83ad977) )
	sc_h6cl_others
ROM_END

ROM_START( sc4h6cla )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95418194.lo", 0x00001, 0x080000, CRC(ebe57e4a) SHA1(df9564a00e38988689371e65b56f02ca57fec262) )
	ROM_LOAD16_BYTE( "95418195.hi", 0x00000, 0x080000, CRC(b76d4561) SHA1(0cd08d51ee08a9179a1f700f7718ae43ce6c35fb) )
	sc_h6cl_others
ROM_END

ROM_START( sc4h6clb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95419148.lo", 0x00001, 0x080000, CRC(770d4c2e) SHA1(b7407179c6747601cd83826d14607450eb889e6b) )
	ROM_LOAD16_BYTE( "95419149.hi", 0x00000, 0x080000, CRC(99a8fc66) SHA1(23a6c0a29f060c7d8b419da45dbc6e1078f27ed2) )
	sc_h6cl_others
ROM_END

ROM_START( sc4h6clc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95419194.lo", 0x00001, 0x080000, CRC(9e659d8a) SHA1(5656847d2e1925645238f18bfd9019548339bdeb) )
	ROM_LOAD16_BYTE( "95419195.hi", 0x00000, 0x080000, CRC(29b29377) SHA1(d456b366533a022cafe23b2eaa563aa56c387ce7) )
	sc_h6cl_others
ROM_END







ROM_START( sc4hyde )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404528.lo", 0x00001, 0x080000, CRC(09b4c86d) SHA1(53bc826958a247a2cec07d71898942c13f581e24) )
	ROM_LOAD16_BYTE( "95404529.hi", 0x00000, 0x080000, CRC(d4155fa6) SHA1(79dfc0128239973bea55a818fafbc2f8c5595b4f) )
	sc_hyde_others
ROM_END

ROM_START( sc4hydea )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405528.lo", 0x00001, 0x080000, CRC(80113a3d) SHA1(bdbfdf73dfb2db3cf9bc2f79419c462a6a0c44be) )
	ROM_LOAD16_BYTE( "95405529.hi", 0x00000, 0x080000, CRC(8dd94d2a) SHA1(e87ddd95306d530de8a4ae391518821839b67acf) )
	sc_hyde_others
ROM_END

ROM_START( sc4hydeb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406142.lo", 0x00001, 0x080000, CRC(8366d2dc) SHA1(975689eff5c5fde2ee0fb6dce23a3361b0761d78) )
	ROM_LOAD16_BYTE( "95406143.hi", 0x00000, 0x080000, CRC(bd10d110) SHA1(a9c620d862261770c09eb39eb84d9cdda5319baa) )
	sc_hyde_others
ROM_END

ROM_START( sc4hydec )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407142.lo", 0x00001, 0x080000, CRC(cd8bf33c) SHA1(7eba0c1118de807a13267541860e60b9f63a03e3) )
	ROM_LOAD16_BYTE( "95407143.hi", 0x00000, 0x080000, CRC(59a3239d) SHA1(1cb89245e53a8a35cf86dac5db37ab656e44cc40) )
	sc_hyde_others
ROM_END




ROM_START( sc4hyper )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406308.lo", 0x00001, 0x080000, CRC(027e5d34) SHA1(68812625408bcd4ece0be429974f3a767c9f6b6b) )
	ROM_LOAD16_BYTE( "95406309.hi", 0x00000, 0x080000, CRC(2ac47aae) SHA1(a44a66e50d649cefd070d12ae5c1aa96926376c0) )
	sc_onup_others
ROM_END

ROM_START( sc4hypera )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407162.lo", 0x00001, 0x080000, CRC(e751dc77) SHA1(da52a96b4d32d8c8efeea82b28d4c6abf9afc298) )
	ROM_LOAD16_BYTE( "95407163.hi", 0x00000, 0x080000, CRC(309e2d9e) SHA1(e692251d7a1aa24d080da9bb8f1d7134fdea31be) )
	sc_onup_others
ROM_END









ROM_START( sc4ijclb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95424322.lo", 0x00001, 0x080000, CRC(e43f64c5) SHA1(292df6d16317bc27def74e2f5d47509a07c52f48) )
	ROM_LOAD16_BYTE( "95424323.hi", 0x00000, 0x080000, CRC(a9c5f9e0) SHA1(008262a03cc85858da6f0ba01564ce4703895a6b) )
	sc_ijclb_others
ROM_END




ROM_START( sc4jack )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406962.lo", 0x00001, 0x080000, CRC(a95fce04) SHA1(89d0263fb09fcc9b50027bf67ffd0040af9c865d) )
	ROM_LOAD16_BYTE( "95406963.hi", 0x00000, 0x080000, CRC(e867aecb) SHA1(4438e91863da3876b4a66db0795ff8d631475f6e) )
	sc_jack_others
ROM_END

ROM_START( sc4jacka )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407962.lo", 0x00001, 0x080000, CRC(cd36442b) SHA1(8b6f5a693a3322061d5e4b26c7f8e13be7cfc226) )
	ROM_LOAD16_BYTE( "95407963.hi", 0x00000, 0x080000, CRC(3be5602a) SHA1(b0e88873332e56bcbf5bc6d0d8c5045dd4266ce2) )
	sc_jack_others
ROM_END

ROM_START( sc4jackb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408250.lo", 0x00001, 0x080000, CRC(1f7c738b) SHA1(28210387c42dcb84ff8800bbe349d2a4a57c4081) )
	ROM_LOAD16_BYTE( "95408251.hi", 0x00000, 0x080000, CRC(75a33c8d) SHA1(e44ab3d463d808e12da81bf77661e140f5abc642) )
	sc_jack_others
ROM_END

ROM_START( sc4jackc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409250.lo", 0x00001, 0x080000, CRC(9541120e) SHA1(6546164e1976a966bb8f326b8c989f7b67ae1153) )
	ROM_LOAD16_BYTE( "95409251.hi", 0x00000, 0x080000, CRC(deafd82f) SHA1(b94ef921ee96375483363911cbfebaf4ce84bf62) )
	sc_jack_others
ROM_END

ROM_START( sc4jackd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414342.lo", 0x00001, 0x080000, CRC(00211b9b) SHA1(502cd4a1d25fe5eae58c3cdf83963fee4a9aaea9) )
	ROM_LOAD16_BYTE( "95414343.hi", 0x00000, 0x080000, CRC(38f41741) SHA1(df4091f8ddc12149cfd8c65b726f2e76583f2825) )
	sc_jack_others
ROM_END

ROM_START( sc4jacke )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414344.lo", 0x00001, 0x080000, CRC(19048ce1) SHA1(7a602d6bb4a11659987c40d799a086e3aa3cc28e) )
	ROM_LOAD16_BYTE( "95414345.hi", 0x00000, 0x080000, CRC(83666578) SHA1(26103607be053284ea7789f1dae12cd0cfc0aa08) )
	sc_jack_others
ROM_END

ROM_START( sc4jackf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415342.lo", 0x00001, 0x080000, CRC(8bf53e5d) SHA1(bd997dd053d64f5dee5bf044d4797ab73d06ff7b) )
	ROM_LOAD16_BYTE( "95415343.hi", 0x00000, 0x080000, CRC(4c7ff6c3) SHA1(6acb0f177a7c85923abf4c48f020ce845fa62c18) )
	sc_jack_others
ROM_END

ROM_START( sc4jackg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415344.lo", 0x00001, 0x080000, CRC(1404e8b3) SHA1(db53f3ca3a9d690efd1f5391f24ce60cbbcde343) )
	ROM_LOAD16_BYTE( "95415345.hi", 0x00000, 0x080000, CRC(ed041da0) SHA1(2bd065eb1854fef93c793b7f83cfe3271aaa7cfa) )
	sc_jack_others
ROM_END








ROM_START( sc4jjunc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406800.lo", 0x00001, 0x080000, CRC(b8981604) SHA1(871007167d0d5cae6343679c682eaef1222dac77) )
	ROM_LOAD16_BYTE( "95406801.hi", 0x00000, 0x080000, CRC(d2b9d2c6) SHA1(597707368ed54946b230f0136fb0e694e468e7a2) )
	sc_jjunc_others
ROM_END

ROM_START( sc4jjunca )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406860.lo", 0x00001, 0x080000, CRC(23e4ca88) SHA1(422f5ad5008d6c8a16784473432b59e0a66d95d3) )
	ROM_LOAD16_BYTE( "95406861.hi", 0x00000, 0x080000, CRC(cf0c37a8) SHA1(71a9e48dcc05d33cfdfe4cab92ae1b2858112800) )
	sc_jjunc_others
ROM_END

ROM_START( sc4jjuncb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407800.lo", 0x00001, 0x080000, CRC(897974c6) SHA1(c206f698ecb5e55b1448306a8bb7448c93bf080c) )
	ROM_LOAD16_BYTE( "95407801.hi", 0x00000, 0x080000, CRC(1b4dd345) SHA1(d4473a1d10e722834f66b0ba5f0493b03d235033) )
	sc_jjunc_others
ROM_END

ROM_START( sc4jjuncc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407860.lo", 0x00001, 0x080000, CRC(d89c7947) SHA1(a2c96d82806f25c5eaf7b531252d88cdc09ee089) )
	ROM_LOAD16_BYTE( "95407861.hi", 0x00000, 0x080000, CRC(94222bfc) SHA1(a4bc7d85ca99890d24d83ba4f347d4f19db49026) )
	sc_jjunc_others
ROM_END

ROM_START( sc4jjuncd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408032.lo", 0x00001, 0x080000, CRC(769e2950) SHA1(6144b8d222177b4bfc10919a70eb4ce39b898b43) )
	ROM_LOAD16_BYTE( "95408033.hi", 0x00000, 0x080000, CRC(d0871eaa) SHA1(8175b7129d5ae0f49792f06af82fc2f19161fdfe) )
	sc_jjunc_others
ROM_END

ROM_START( sc4jjunce )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408280.lo", 0x00001, 0x080000, CRC(bf7e7125) SHA1(effd754236c85bae963a9d5c019ef31b1acf24da) )
	ROM_LOAD16_BYTE( "95408281.hi", 0x00000, 0x080000, CRC(9ba5ac3f) SHA1(21aab8ab476edccb127d76d7538533fa3e22dceb) )
	sc_jjunc_others
ROM_END

ROM_START( sc4jjuncf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409032.lo", 0x00001, 0x080000, CRC(da329b3e) SHA1(f546c0dff49b8633f3aa0566c6f8e793718f1723) )
	ROM_LOAD16_BYTE( "95409033.hi", 0x00000, 0x080000, CRC(792c3c23) SHA1(aa137a6baaeff81601e0f0174e13a19029b7105d) )
	sc_jjunc_others
ROM_END

ROM_START( sc4jjuncg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409280.lo", 0x00001, 0x080000, CRC(315237b8) SHA1(4bdbec479fcf8ea1f951a1494ee17b6564dab2d0) )
	ROM_LOAD16_BYTE( "95409281.hi", 0x00000, 0x080000, CRC(738cc34b) SHA1(106c06b4eec272cfe25ea7475b41e1f8f7c4cf4c) )
	sc_jjunc_others
ROM_END

ROM_START( sc4jjunch )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1304p13.hi", 0x00000, 0x080000, CRC(99c6920f) SHA1(917e60be1f1bdb1a8f2b060523a154c7532a8118) )
	ROM_LOAD16_BYTE( "p1304p13.lo", 0x00001, 0x080000, CRC(5a19ff3c) SHA1(d600039dd85f9d2db3925e250587d1cdcbcb9eaf) )
	sc_jjunc_others
ROM_END

ROM_START( sc4jjunci )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1304s13.hi", 0x00000, 0x080000, CRC(39f32b92) SHA1(d8f664c9fc2d099642b706bf8cee2d216dd72aad) )
	ROM_LOAD16_BYTE( "p1304s13.lo", 0x00001, 0x080000, CRC(558a4765) SHA1(36ea6375f09e270630941fab84746938ae46fb79) )
	sc_jjunc_others
ROM_END




ROM_START( sc4jjucl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410318.lo", 0x00001, 0x080000, CRC(044d9908) SHA1(0cbc58f77b729638981ac06590532fa69797e448) )
	ROM_LOAD16_BYTE( "95410319.hi", 0x00000, 0x080000, CRC(1583ec1f) SHA1(22aa9d80f1af130546564d34abc1dd715f9388d1) )
	sc_jjucl_others
ROM_END

ROM_START( sc4jjucla )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410362.lo", 0x00001, 0x080000, CRC(6d2f7d2f) SHA1(05333d9dcf99a93a17efabb56378798680cd302b) )
	ROM_LOAD16_BYTE( "95410363.hi", 0x00000, 0x080000, CRC(e0c11468) SHA1(2bdd844d088085789780c736eb29a36565083964) )
	sc_jjucl_others
ROM_END

ROM_START( sc4jjuclb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411318.lo", 0x00001, 0x080000, CRC(2166860e) SHA1(20b316648036372813ebb7c13727c71e80fa2413) )
	ROM_LOAD16_BYTE( "95411319.hi", 0x00000, 0x080000, CRC(c7c715cc) SHA1(9123671fbd916c5721d2d000dc6a6389190f0a22) )
	sc_jjucl_others
ROM_END

ROM_START( sc4jjuclc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411362.lo", 0x00001, 0x080000, CRC(e14945b6) SHA1(4fd878199cc5ebcaaa6f980304fb59cced401c2a) )
	ROM_LOAD16_BYTE( "95411363.hi", 0x00000, 0x080000, CRC(b90d06e4) SHA1(c216ff7a4e385f3e1ee09bdcf236c583fe674c3a) )
	sc_jjucl_others
ROM_END

ROM_START( sc4jjucld )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411828.lo", 0x00001, 0x080000, CRC(3ed33435) SHA1(530c660abfae693b83f6f3917c98fec8fec0ef70) )
	ROM_LOAD16_BYTE( "95411829.hi", 0x00000, 0x080000, CRC(ed588fed) SHA1(b9454795ce089f04b7af792c523514597de0bd08) )
	sc_jjucl_others
ROM_END

ROM_START( sc4jjucle )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410828.lo", 0x00001, 0x080000, CRC(933e838e) SHA1(501f681aba7a787061947efe92fd8451ce903b2c) )
	ROM_LOAD16_BYTE( "95410829.hi", 0x00000, 0x080000, CRC(b40bcb59) SHA1(e00b5eb3e68f1217bc69622b25f2340aaa6a289d) )
	sc_jjucl_others
ROM_END


ROM_START( sc4jiggn )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402838.lo", 0x00001, 0x080000, CRC(fbfc9210) SHA1(77029afbf7852c4fba5774184bdbef018e447247) )
	ROM_LOAD16_BYTE( "95402839.hi", 0x00000, 0x080000, CRC(b449febc) SHA1(41cc4edb267e108316c5947ae7bb3111de54a12f) )
	sc_jiggn_others
ROM_END




ROM_START( sc4jiggr )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400769.lo", 0x00001, 0x080000, CRC(7b03c84c) SHA1(ce5fe064c928aa3afba6cf43caab7a08af537d98) )
	ROM_LOAD16_BYTE( "95400770.hi", 0x00000, 0x080000, CRC(67fe5696) SHA1(994f482008b526d82801481a84f7d7d0fc82366c) )
	sc_jiggr_others
ROM_END

ROM_START( sc4jiggra )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400917.lo", 0x00001, 0x080000, CRC(280e5e5c) SHA1(202913d7647d34ef91bfba9a5c4c024ed1a4a1f2) )
	ROM_LOAD16_BYTE( "95400918.hi", 0x00000, 0x080000, CRC(67b18ac9) SHA1(18bd935d2900616e744c6587bfd649821ffeba28) )
	sc_jiggr_others
ROM_END

ROM_START( sc4jiggrb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401769.lo", 0x00001, 0x080000, CRC(118c2aa4) SHA1(c401ac1c36435cd90f14a74e8c3d7c7597a5a959) )
	ROM_LOAD16_BYTE( "95401770.hi", 0x00000, 0x080000, CRC(cfb146a4) SHA1(0eef6ce82d4994037caddffdbd58a115663042a3) )
	sc_jiggr_others
ROM_END

ROM_START( sc4jiggrc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401917.lo", 0x00001, 0x080000, CRC(6f879e2b) SHA1(d6d2d55b37e0ec39950b2e67e55bd0ef4b4c402e) )
	ROM_LOAD16_BYTE( "95401918.hi", 0x00000, 0x080000, CRC(70637422) SHA1(5251a43e1bb76a68e901a4cd3f73ecee45e50737) )
	sc_jiggr_others
ROM_END




ROM_START( sc4jive )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408432.lo", 0x00001, 0x080000, CRC(4696c1be) SHA1(81910be1c328faca834c93de3a15d2e0d2d79ae0) )
	ROM_LOAD16_BYTE( "95408433.hi", 0x00000, 0x080000, CRC(3d9369c7) SHA1(bed1ec96e9793af7efe1b10a3cd101cc378e45af) )
	sc_jive_others
ROM_END

ROM_START( sc4jivea )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416058.lo", 0x00001, 0x080000, CRC(22089a87) SHA1(16d4aa89635f796bbfa593e1a3836cbc65191064) )
	ROM_LOAD16_BYTE( "95416059.hi", 0x00000, 0x080000, CRC(39017969) SHA1(c6d9df634a8d4cd31ba464e0d241ac24772b62d6) )
	sc_jive_others
ROM_END

ROM_START( sc4jiveb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416824.lo", 0x00001, 0x080000, CRC(8fd82043) SHA1(4e92abc737ec6286853ba4e300a1ecc8d61e1b07) )
	ROM_LOAD16_BYTE( "95416825.hi", 0x00000, 0x080000, CRC(81a3000d) SHA1(aa863edd7e48c8c8467b73c49be4dc09669c0e5c) )
	sc_jive_others
ROM_END

ROM_START( sc4jivec )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417058.lo", 0x00001, 0x080000, CRC(860c88b5) SHA1(cd7cd906150be0e3b0930d835573b95f85222bd5) )
	ROM_LOAD16_BYTE( "95417059.hi", 0x00000, 0x080000, CRC(6c57f61c) SHA1(611eeb0c9bacf745d2c31dabf3622e9a1665f574) )
	sc_jive_others
ROM_END

ROM_START( sc4jived )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417824.lo", 0x00001, 0x080000, CRC(2bdc3271) SHA1(50bd25be63607df7b995022c5b90fac9d5d34140) )
	ROM_LOAD16_BYTE( "95417825.hi", 0x00000, 0x080000, CRC(d4f58f78) SHA1(629196af54c388b0d2a23554b91a2f40f14f6f07) )
	sc_jive_others
ROM_END




ROM_START( sc4jolly )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406836.odd", 0x00001, 0x080000, CRC(2cc39f4c) SHA1(8d107bf5773e2d00ea296317652f388838737a6b) )
	ROM_LOAD16_BYTE( "95406837.evn", 0x00000, 0x080000, CRC(b0d52da0) SHA1(1cab78990cb7ed5b7663344d0f9a61fb8eefdb48) )
	sc_jolly_others
ROM_END

ROM_START( sc4jollya )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407836.odd", 0x00001, 0x080000, CRC(82681921) SHA1(459ab8cd347ba1c812d5ef7d7fc42b86a9db9794) )
	ROM_LOAD16_BYTE( "95407837.evn", 0x00000, 0x080000, CRC(3ba55dbd) SHA1(a5bc3d4df74f84865cf11700ef113fc942b1a637) )
	sc_jolly_others
ROM_END




ROM_START( sc4juicy )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404882.lo", 0x00001, 0x080000, CRC(ab8732fd) SHA1(2cb28bc7aee41b06aecd36e27ea95451b432287b) )
	ROM_LOAD16_BYTE( "95404883.hi", 0x00000, 0x080000, CRC(3d153f79) SHA1(cb20abdfd4ec55b840494f7cf328038f82867f2d) )
	sc_juicy_others
ROM_END

ROM_START( sc4juicya )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404982.lo", 0x00001, 0x080000, CRC(467c6e2a) SHA1(c281c11c7ea8c22e21a5886881be9a47d143e449) )
	ROM_LOAD16_BYTE( "95404983.hi", 0x00000, 0x080000, CRC(e4afbd59) SHA1(645d124b94b68806f4cc4f929320bf57c6b1a5cc) )
	sc_juicy_others
ROM_END

ROM_START( sc4juicyb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405882.lo", 0x00001, 0x080000, CRC(5634668c) SHA1(14261d853ef9bbe6b7efc6e16ef5929f4613d494) )
	ROM_LOAD16_BYTE( "95405883.hi", 0x00000, 0x080000, CRC(cfe5903e) SHA1(2e80aaaf37aa6d344cf05563d0ef48b556e65fc6) )
	sc_juicy_others
ROM_END

ROM_START( sc4juicyc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405982.lo", 0x00001, 0x080000, CRC(8c7e6c23) SHA1(f9d20a67af07cbf4751aa5e35fd8abf1756b1b52) )
	ROM_LOAD16_BYTE( "95405983.hi", 0x00000, 0x080000, CRC(b1f9322c) SHA1(8e1804e3ba06ae58b0da41b8333654b6d3cc36f2) )
	sc_juicy_others
ROM_END

ROM_START( sc4juicyd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406484.lo", 0x00001, 0x080000, CRC(43d59f13) SHA1(e38a385e3e2e5a65bc2f22b5eb5097664fd35565) )
	ROM_LOAD16_BYTE( "95406485.hi", 0x00000, 0x080000, CRC(17191ba8) SHA1(c7f13e50cdc7a4115b8062d59f2db792255a2058) )
	sc_juicy_others
ROM_END

ROM_START( sc4juicye )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407484.lo", 0x00001, 0x080000, CRC(515b9ac0) SHA1(f2f5e6e2139aa1e1e73d165b94778187c4063c88) )
	ROM_LOAD16_BYTE( "95407485.hi", 0x00000, 0x080000, CRC(08d2882a) SHA1(44262f577b8b17bd833a1ba454d93861bb8e3968) )
	sc_juicy_others
ROM_END

ROM_START( sc4juicyf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408212.lo", 0x00001, 0x080000, CRC(fac3dcb4) SHA1(eb6aa0a97422d735ab26c697448957e23a76e8e1) )
	ROM_LOAD16_BYTE( "95408213.hi", 0x00000, 0x080000, CRC(3b7bd219) SHA1(765a87418034d83b6401ba8c53993edf7aa60973) )
	sc_juicy_others
ROM_END

ROM_START( sc4juicyg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409212.lo", 0x00001, 0x080000, CRC(60c1521e) SHA1(9c0d96918802a49651d44b3850c77b63a3f6f18d) )
	ROM_LOAD16_BYTE( "95409213.hi", 0x00000, 0x080000, CRC(0cd5b84d) SHA1(ef87a40f67207c80b96221a5ad397d5cc803a3b7) )
	sc_juicy_others
ROM_END



ROM_START( sc4juicyi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412202.lo", 0x00001, 0x080000, CRC(be13c0c6) SHA1(400d0b395f15d6d75ef06f16cc99d8005b4de286) )
	ROM_LOAD16_BYTE( "95412203.hi", 0x00000, 0x080000, CRC(ac436887) SHA1(a9b022349a4b22438643b01298b51c1770816c3e) )
	sc_juicy_others
ROM_END

ROM_START( sc4juicyj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413202.lo", 0x00001, 0x080000, CRC(8d0b8d19) SHA1(4e10280fc81d9b08ed6a8c3af8c83ff5c43719f7) )
	ROM_LOAD16_BYTE( "95413203.hi", 0x00000, 0x080000, CRC(100224f6) SHA1(2f7aeca8f074177dab9af8d006a9660380a659eb) )
	sc_juicy_others
ROM_END




ROM_START( sc4jjc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414638.lo", 0x00001, 0x080000, CRC(37a200e7) SHA1(7729e150eff90884e6fad437f02855b0d1f4aaab) )
	ROM_LOAD16_BYTE( "95414639.hi", 0x00000, 0x080000, CRC(8b3d2c9b) SHA1(abfd57df10d092d6ae4262757004d4f5bf031d8f) )
	sc_jjc_others
ROM_END

ROM_START( sc4jjca )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414640.lo", 0x00001, 0x080000, CRC(fb011587) SHA1(8a794e2372719255425908ca3124db9d00a2c671) )
	ROM_LOAD16_BYTE( "95414641.hi", 0x00000, 0x080000, CRC(915a68f4) SHA1(fe4783d9af03e5ddd20a9260c5caf1c607c73ceb) )
	sc_jjc_others
ROM_END

ROM_START( sc4jjcb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415638.lo", 0x00001, 0x080000, CRC(7dd83a80) SHA1(538e2dc8887e9a636f078af131c50b90153c59fe) )
	ROM_LOAD16_BYTE( "95415639.hi", 0x00000, 0x080000, CRC(21903271) SHA1(79e846b208a5235df570323128986085581d2ddf) )
	sc_jjc_others
ROM_END

ROM_START( sc4jjcc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415640.lo", 0x00001, 0x080000, CRC(38391c5f) SHA1(56232d4f2b8c28097368c668e94815c864514b14) )
	ROM_LOAD16_BYTE( "95415641.hi", 0x00000, 0x080000, CRC(2f667c65) SHA1(c208a6273237df8fdbd2a8889d96b705731bb1e0) )
	sc_jjc_others
ROM_END

ROM_START( sc4jjcd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416338.lo", 0x00001, 0x080000, CRC(416c2f1d) SHA1(7aa986a9a52dd857da4443657fbc0f101f03e9f4) )
	ROM_LOAD16_BYTE( "95416339.hi", 0x00000, 0x080000, CRC(e2b1e7a3) SHA1(646714b13061d5f0495414acd549d6ff3ac3ccce) )
	sc_jjc_others
ROM_END

ROM_START( sc4jjce )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417338.lo", 0x00001, 0x080000, CRC(c1bc5ff7) SHA1(0b886ea05f7005150040106adc626f33e9cf3352) )
	ROM_LOAD16_BYTE( "95417339.hi", 0x00000, 0x080000, CRC(8c4cc943) SHA1(e37e47dde1e1d7f6558b715cde0b4a7cd28302f6) )
	sc_jjc_others
ROM_END

ROM_START( sc4jjcf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420416.lo", 0x00001, 0x080000, CRC(53890cdf) SHA1(3f8058bbae4b02c01bd7b9cc0ecb0b790c463784) )
	ROM_LOAD16_BYTE( "95420417.hi", 0x00000, 0x080000, CRC(cbe8b021) SHA1(6b2dcb53d1b121adcdbe4a767dc4e024321b1e0c) )
	sc_jjc_others
ROM_END

ROM_START( sc4jjcg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420418.lo", 0x00001, 0x080000, CRC(318e0964) SHA1(461e32f31cc8d03182475e5d8fd11b72dff15c78) )
	ROM_LOAD16_BYTE( "95420419.hi", 0x00000, 0x080000, CRC(98a5c16c) SHA1(0529fda8b45e423805c03166148df7beb4898259) )
	sc_jjc_others
ROM_END

ROM_START( sc4jjch )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421416.lo", 0x00001, 0x080000, CRC(11a5509a) SHA1(1a5f96f4b061e254af625710156b216dbfaf61e0) )
	ROM_LOAD16_BYTE( "95421417.hi", 0x00000, 0x080000, CRC(7105e0f5) SHA1(f7ed5bc4562cc70f15beb887e4db72af2fd5e65e) )
	sc_jjc_others
ROM_END

ROM_START( sc4jjci )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421418.lo", 0x00001, 0x080000, CRC(fd185e8b) SHA1(e8a662ba2f4d32b0cbc307c38512c3c5cc8b10cb) )
	ROM_LOAD16_BYTE( "95421419.hi", 0x00000, 0x080000, CRC(fb626016) SHA1(9bc68d6450058628aaaf4c62e8269823c7ee6fe5) )
	sc_jjc_others
ROM_END




ROM_START( sc4jjf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400167.lo", 0x00001, 0x080000, CRC(0ae29a7d) SHA1(4ba10d961c00573419be0dbc9ae219a3df9c7299) )
	ROM_LOAD16_BYTE( "95400168.hi", 0x00000, 0x080000, CRC(7de0e701) SHA1(42e5d59fee152a4ba259c5ac472c67c7a292c895) )
	sc_jjf_others
ROM_END

ROM_START( sc4jjfa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400255.lo", 0x00001, 0x080000, CRC(6551fe1e) SHA1(905b11090927cbedea88f2119f5961cae023e5c5) )
	ROM_LOAD16_BYTE( "95400256.hi", 0x00000, 0x080000, CRC(b6487501) SHA1(bcc3c7d1b229e38d414796a95d44f7a35d87a841) )
	sc_jjf_others
ROM_END

ROM_START( sc4jjfb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400257.lo", 0x00001, 0x080000, CRC(25ba4d94) SHA1(f2ff4fbbe8c5fbee3b4439a6e699f4ec7be3964b) )
	ROM_LOAD16_BYTE( "95400258.hi", 0x00000, 0x080000, CRC(de5a9aec) SHA1(3e3c7d2648293be74f4ac3ae755edcc8d7ffb80b) )
	sc_jjf_others
ROM_END

ROM_START( sc4jjfc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400751.lo", 0x00001, 0x080000, CRC(e7988177) SHA1(e4dd00aecca03d28ad2f356ce1e79504840c2ce3) )
	ROM_LOAD16_BYTE( "95400752.hi", 0x00000, 0x080000, CRC(abc8fc7e) SHA1(99c6ff52e1348fb82e07cc45fb5fbf2f6d2059e6) )
	sc_jjf_others
ROM_END

ROM_START( sc4jjfd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400753.lo", 0x00001, 0x080000, CRC(a4190022) SHA1(82e9e68e9f36d3d0b95b91c2e3814f8446099ba2) )
	ROM_LOAD16_BYTE( "95400754.hi", 0x00000, 0x080000, CRC(e21ac2b9) SHA1(830d7b60260f0a89ee474df8ff3beee1bd68c4a9) )
	sc_jjf_others
ROM_END

ROM_START( sc4jjfe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400935.lo", 0x00001, 0x080000, CRC(a970ba37) SHA1(2d49f337752cbde9ea75f61aa3c34582b8e1eb43) )
	ROM_LOAD16_BYTE( "95400936.hi", 0x00000, 0x080000, CRC(fa531aa7) SHA1(0277a7e552a516bccd60408a68d1feca2536c553) )
	sc_jjf_others
ROM_END

ROM_START( sc4jjff )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400937.lo", 0x00001, 0x080000, CRC(b80c9c9a) SHA1(708f1a41fa1ddb6c2e06c9deaabdb33004b0065d) )
	ROM_LOAD16_BYTE( "95400938.hi", 0x00000, 0x080000, CRC(7e56ea21) SHA1(a05168313a60a939c78c880376bd2006042a8f45) )
	sc_jjf_others
ROM_END

ROM_START( sc4jjfg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401167.lo", 0x00001, 0x080000, CRC(e5a2e444) SHA1(7012ba695111ac2d1d651be2e00d73aff017c692) )
	ROM_LOAD16_BYTE( "95401168.hi", 0x00000, 0x080000, CRC(cd3b3689) SHA1(808277be3d2163dde8d9a7ce5b73dff12c4cd518) )
	sc_jjf_others
ROM_END

ROM_START( sc4jjfh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401255.lo", 0x00001, 0x080000, CRC(d271a4a7) SHA1(b72a84c06ae171875da775a5d70556d2fcab5cf6) )
	ROM_LOAD16_BYTE( "95401256.hi", 0x00000, 0x080000, CRC(d6529ef8) SHA1(599b432337b5b3c46749f5f36c6d33b0df862126) )
	sc_jjf_others
ROM_END

ROM_START( sc4jjfi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401257.lo", 0x00001, 0x080000, CRC(2d2e7e17) SHA1(f941e4c1ca3d28954bc5414da67d0c7fba9a0393) )
	ROM_LOAD16_BYTE( "95401258.hi", 0x00000, 0x080000, CRC(786d19ff) SHA1(ad1f3b3172aff887ea9407ca362c15248b59ad1c) )
	sc_jjf_others
ROM_END

ROM_START( sc4jjfj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401751.lo", 0x00001, 0x080000, CRC(4310e660) SHA1(1383e6a497807207560a5d8745e51bbaa155bf95) )
	ROM_LOAD16_BYTE( "95401752.hi", 0x00000, 0x080000, CRC(c891e317) SHA1(d98a8cf225b4c976bb7b822cb6bc5a628a1eb03e) )
	sc_jjf_others
ROM_END

ROM_START( sc4jjfk )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401753.lo", 0x00001, 0x080000, CRC(d837ff93) SHA1(1020cce3817c0b0ad22fab26c2a265d90924d0b0) )
	ROM_LOAD16_BYTE( "95401754.hi", 0x00000, 0x080000, CRC(2437aa53) SHA1(4079cb900f4834d879d1b2f9e116b0ec4f96b4df) )
	sc_jjf_others
ROM_END

ROM_START( sc4jjfl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401935.lo", 0x00001, 0x080000, CRC(2309aba4) SHA1(a08d8363019c72552196a1caabfb449d65775b6e) )
	ROM_LOAD16_BYTE( "95401936.hi", 0x00000, 0x080000, CRC(9b775d2e) SHA1(bf816312b28a81c070ac6230193751ec7687977e) )
	sc_jjf_others
ROM_END

ROM_START( sc4jjfm )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401937.lo", 0x00001, 0x080000, CRC(e73ce3c9) SHA1(942d67857a30380d53f82ff4e92ad51fe4488ada) )
	ROM_LOAD16_BYTE( "95401938.hi", 0x00000, 0x080000, CRC(e3cac8a7) SHA1(48130018e74c32a5863c9d8696c288fbeaee6bdc) )
	sc_jjf_others
ROM_END




ROM_START( sc4jbuck )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402494.lo", 0x00001, 0x080000, CRC(b9c2e7a2) SHA1(3b0805c35e9a19d29c4810d42f1815da5dba61b2) )
	ROM_LOAD16_BYTE( "95402495.hi", 0x00000, 0x080000, CRC(f3d89390) SHA1(91bc7945845f41687d7159cc273bef7d852f3363) )
	sc_jbuck_others
ROM_END

ROM_START( sc4jbucka )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403494.lo", 0x00001, 0x080000, CRC(8357c809) SHA1(9d143108aa0ea04c351be337c1b0f16b5298f1f5) )
	ROM_LOAD16_BYTE( "95403495.hi", 0x00000, 0x080000, CRC(0322e5b7) SHA1(c306db0c71dcb6862ad97cb725e16a56d7a8c4ca) )
	sc_jbuck_others
ROM_END

ROM_START( sc4jbuckb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404074.lo", 0x00001, 0x080000, CRC(e3db5121) SHA1(1114493ea5515a5c45e35ebb279f39a752eec31b) )
	ROM_LOAD16_BYTE( "95404075.hi", 0x00000, 0x080000, CRC(7c3986c2) SHA1(475c4c8dc4b75b54ef61cf67266fb10df16bc2fd) )
	sc_jbuck_others
ROM_END

ROM_START( sc4jbuckc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405074.lo", 0x00001, 0x080000, CRC(feb57a06) SHA1(bde1fa35d6f99f6e67be8c9479c3df64ab3c48e9) )
	ROM_LOAD16_BYTE( "95405075.hi", 0x00000, 0x080000, CRC(b511d792) SHA1(b198fb4a0b2bc5221b3dc367a8f40ea2ccb84516) )
	sc_jbuck_others
ROM_END

ROM_START( sc4jbuckd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "j_bucklo",    0x00001, 0x080000, CRC(8c4a6503) SHA1(e7267c8e58f57928e5f0da433b646633c7e7f5ba) )
	ROM_LOAD16_BYTE( "j_buckhi",    0x00000, 0x080000, CRC(f75eb5d2) SHA1(ccd0bbde1f450c26bb64ffcc9480d53ca9fae74b) )
	sc_jbuck_others
ROM_END




ROM_START( sc4kalei )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412220.odd", 0x00001, 0x080000, CRC(18604726) SHA1(3b1f32c6cf75e1c1edb744feb59fb2303c9ecc5e) )
	ROM_LOAD16_BYTE( "95412221.evn", 0x00000, 0x080000, CRC(f9c58d01) SHA1(668bebcfdfbe5d888d69d4e6d1b483e63f8378ed) )
	sc_kalei_others
ROM_END

ROM_START( sc4kaleia )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412222.odd", 0x00001, 0x080000, CRC(0f417f5c) SHA1(9c54da76ccd31c6dad41ab1eea6453c2879598f3) )
	ROM_LOAD16_BYTE( "95412223.evn", 0x00000, 0x080000, CRC(0a5beeb5) SHA1(899d8b3c0ebe733e875b067c80b0215e901c6a7f) )
	sc_kalei_others
ROM_END

ROM_START( sc4kaleib )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413220.odd", 0x00001, 0x080000, CRC(1731eca0) SHA1(74fe81b2bddcf1ceeaecc996c224384f6a1f9824) )
	ROM_LOAD16_BYTE( "95413221.evn", 0x00000, 0x080000, CRC(cc89e98d) SHA1(bdfce2a956b7bae55a714b78c08a35c1525e6ceb) )
	sc_kalei_others
ROM_END

ROM_START( sc4kaleic )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413222.odd", 0x00001, 0x080000, CRC(c34a9ea9) SHA1(f1b7e010657dc3014b65aebf6bd82b8ba9fdd160) )
	ROM_LOAD16_BYTE( "95413223.evn", 0x00000, 0x080000, CRC(ba1f6905) SHA1(34372198bd94c86eaeb49ed9454081033c106d74) )
	sc_kalei_others
ROM_END

ROM_START( sc4kaleid )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414476.odd", 0x00001, 0x080000, CRC(f5624cfc) SHA1(4270eed4fe75a3c54ff7484031e6ae7a9b8a7298) )
	ROM_LOAD16_BYTE( "95414477.evn", 0x00000, 0x080000, CRC(ad443b30) SHA1(d677ee56a1a90b005ea6ea3c436831c353c9a821) )
	sc_kalei_others
ROM_END

ROM_START( sc4kaleie )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415476.odd", 0x00001, 0x080000, CRC(18b4c3fe) SHA1(8e96dbd39fb07c1a744c1969121e2fdba74568e7) )
	ROM_LOAD16_BYTE( "95415477.evn", 0x00000, 0x080000, CRC(3d7c283f) SHA1(da2efff240baa2c6e290f23591092dbd802922b4) )
	sc_kalei_others
ROM_END




ROM_START( sc4kkong )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414472.lo", 0x00001, 0x080000, CRC(cf4099ca) SHA1(5501c3bdabe84900f77c451a404cfbd40bf1a152) )
	ROM_LOAD16_BYTE( "95414473.hi", 0x00000, 0x080000, CRC(61698cee) SHA1(d74c02affe9c37ce99778407d6b3afc84b45a98f) )
	sc_kkong_others
ROM_END

ROM_START( sc4kkonga )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414580.lo", 0x00001, 0x080000, CRC(5ccc7f67) SHA1(8768f774e8b2607a0b553b6659138fb627caf77d) )
	ROM_LOAD16_BYTE( "95414581.hi", 0x00000, 0x080000, CRC(34878d64) SHA1(35a5e34172df2029263cd240ae8b5efff9794d30) )
	sc_kkong_others
ROM_END

ROM_START( sc4kkongb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415472.lo", 0x00001, 0x080000, CRC(a6f270d0) SHA1(9debde5f2476b981fd5d2e5b24f47d18a4163a00) )
	ROM_LOAD16_BYTE( "95415473.hi", 0x00000, 0x080000, CRC(f06f3391) SHA1(551f29c9d1d35906efff85b164f26ee12bd924ea) )
	sc_kkong_others
ROM_END

ROM_START( sc4kkongc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415580.lo", 0x00001, 0x080000, CRC(ea33d976) SHA1(4dcd684fd997f7031fa51f207051f49fe2e4ce09) )
	ROM_LOAD16_BYTE( "95415581.hi", 0x00000, 0x080000, CRC(39bcead5) SHA1(9f005e57dbfd03fc2f0fb76cd51e91a98a2e9028) )
	sc_kkong_others
ROM_END

ROM_START( sc4kkongd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416720.lo", 0x00001, 0x080000, CRC(4462a7b8) SHA1(c0ee3138e4ecaec4f80696b23222c8991e414224) )
	ROM_LOAD16_BYTE( "95416721.hi", 0x00000, 0x080000, CRC(6b25c34d) SHA1(8511a7ecca3a2a9ba137cd01136a15b14b1d9acc) )
	sc_kkong_others
ROM_END

ROM_START( sc4kkonge )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416770.lo", 0x00001, 0x080000, CRC(d868edc1) SHA1(7f08df6e36adc421e3755e069beeb15f894f284b) )
	ROM_LOAD16_BYTE( "95416771.hi", 0x00000, 0x080000, CRC(6fc88c82) SHA1(10dcac8054ede67759db6f64aa7c952130b2ecca) )
	sc_kkong_others
ROM_END

ROM_START( sc4kkongf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416772.lo", 0x00001, 0x080000, CRC(6cdd2db0) SHA1(7fd6b972f1b2a59d9e1de9ceba28a0c90dff33d6) )
	ROM_LOAD16_BYTE( "95416773.hi", 0x00000, 0x080000, CRC(2b69b383) SHA1(ad5abc445f6dd7341bd29e2e0a359bda04241120) )
	sc_kkong_others
ROM_END

ROM_START( sc4kkongg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417720.lo", 0x00001, 0x080000, CRC(009ad96e) SHA1(49cb002ad376db10e68f6ba25309c68194f24f9f) )
	ROM_LOAD16_BYTE( "95417721.hi", 0x00000, 0x080000, CRC(8c5c2a09) SHA1(d4ab6e21e4110159ca0494b0f83864731653b784) )
	sc_kkong_others
ROM_END

ROM_START( sc4kkongh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417770.lo", 0x00001, 0x080000, CRC(420aeaa9) SHA1(f0f11046e54591f7fa4fcc19bdd8fef6598c8d40) )
	ROM_LOAD16_BYTE( "95417771.hi", 0x00000, 0x080000, CRC(01aaf45a) SHA1(d64b69d2d18ca44e94045d40b71428a1efd6d999) )
	sc_kkong_others
ROM_END

ROM_START( sc4kkongi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417772.lo", 0x00001, 0x080000, CRC(a2ab959f) SHA1(fca03e13fb6e514d7bc1a548c80d41754e2c1b4f) )
	ROM_LOAD16_BYTE( "95417773.hi", 0x00000, 0x080000, CRC(6795513c) SHA1(b7bf4e4f639eb81de23fc5ac57cc2db05a23c3a1) )
	sc_kkong_others
ROM_END

ROM_START( sc4kkongj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "king kon.lo", 0x00001, 0x080000, CRC(95e8e7d1) SHA1(6a6af285a2ebf197ed4f14e5275bdd2d1ebc5283) )
	ROM_LOAD16_BYTE( "king kon.hi", 0x00000, 0x080000, CRC(859bcf1a) SHA1(6d2f81640f2045916b3bad52673de0c14a549a8f) )
	sc_kkong_others
ROM_END




ROM_START( sc4knok )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402460.lo", 0x00001, 0x080000, CRC(01bbbd5d) SHA1(d1bc964004efa8e086393d3ebb54ef0df5e0bfc7) )
	ROM_LOAD16_BYTE( "95402461.hi", 0x00000, 0x080000, CRC(d419b46f) SHA1(1f94e4d548df9b8a83efa4e94f30fe894c5ac9ac) )
	sc_knok_others
ROM_END

ROM_START( sc4knoka )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403460.lo", 0x00001, 0x080000, CRC(abef773b) SHA1(452ebe0345cb1cc2ee9919c79eac781272bae125) )
	ROM_LOAD16_BYTE( "95403461.hi", 0x00000, 0x080000, CRC(3376e44a) SHA1(0e06edb0635aa41250f5035988965154d22a6ff4) )
	sc_knok_others
ROM_END

ROM_START( sc4knokb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406644.lo", 0x00001, 0x080000, CRC(aa84894a) SHA1(a85bcceb8d5a7c30e4503e7e85d800ae22e689a3) )
	ROM_LOAD16_BYTE( "95406645.hi", 0x00000, 0x080000, CRC(15bb7e3a) SHA1(36ca2317ca53d8414b0c826b8885f6c9a4bbe3a2) )
	sc_knok_others
ROM_END

ROM_START( sc4knokc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407644.lo", 0x00001, 0x080000, CRC(067db0c5) SHA1(66ac948515952caa7e8118bae223611ae06440d3) )
	ROM_LOAD16_BYTE( "95407645.hi", 0x00000, 0x080000, CRC(c84123fa) SHA1(a063be3d45179eb7bbf7653bb6f536cb9badf122) )
	sc_knok_others
ROM_END




ROM_START( sc4lasv )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406372.lo", 0x00001, 0x080000, CRC(0db28e23) SHA1(2e78b18dbc7f9941eabfc3ac07c212f69fa30b95) )
	ROM_LOAD16_BYTE( "95406373.hi", 0x00000, 0x080000, CRC(2bac3c58) SHA1(a9b821a98ba6bcf46b0679753d50970bfb0221cc) )
	sc_lasv_others
ROM_END

ROM_START( sc4lasva )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406952.lo", 0x00001, 0x080000, CRC(c28032cc) SHA1(24bc23d6c97d0c52183d42b03b776184a124e1a8) )
	ROM_LOAD16_BYTE( "95406953.hi", 0x00000, 0x080000, CRC(59f73c1d) SHA1(4f488c15f265348c1d6fb15e657ad81a30ae3e50) )
	sc_lasv_others
ROM_END




ROM_START( sc4lir )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404132.lo", 0x00001, 0x080000, CRC(2d9b3911) SHA1(bddcc91f85208e1bdd4c823e36196425d376c1a6) )
	ROM_LOAD16_BYTE( "95404133.hi", 0x00000, 0x080000, CRC(1a3d4273) SHA1(77ee2e8a67a1cb6f59dfadc6f56e465916a34bfc) )
	sc_lir_others
ROM_END

ROM_START( sc4lira )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404136.lo", 0x00001, 0x080000, CRC(4ca0a1b3) SHA1(7ed9d052a64d642fca67630208667d1d0b5de565) )
	ROM_LOAD16_BYTE( "95404137.hi", 0x00000, 0x080000, CRC(ed11a1b5) SHA1(ee19681c277c74d2ea15a0d335d73249c069b9a9) )
	sc_lir_others
ROM_END

ROM_START( sc4lirb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404138.lo", 0x00001, 0x080000, CRC(77a72b5c) SHA1(10a6e2bab024d505c560f0e6cf8aed0058ec286d) )
	ROM_LOAD16_BYTE( "95404139.hi", 0x00000, 0x080000, CRC(491414b0) SHA1(ebf0e82913f0cb0a9d9d7c74b79a0f6442705211) )
	sc_lir_others
ROM_END

ROM_START( sc4lirc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405132.lo", 0x00001, 0x080000, CRC(e2eb1d48) SHA1(fc2add0043e65306cf09c88f63c84bbb5a72330b) )
	ROM_LOAD16_BYTE( "95405133.hi", 0x00000, 0x080000, CRC(157b7d22) SHA1(12736f9b0889aa30fd8730f556e4e3eb5e35fb65) )
	sc_lir_others
ROM_END

ROM_START( sc4lird )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405136.lo", 0x00001, 0x080000, CRC(18e6b65f) SHA1(87852dabbd203ae5fa61e8379bc4779fd74e3548) )
	ROM_LOAD16_BYTE( "95405137.hi", 0x00000, 0x080000, CRC(3e936f54) SHA1(e91ec474bba76e16715eac18fffb83855c9fd992) )
	sc_lir_others
ROM_END

ROM_START( sc4lire )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405138.lo", 0x00001, 0x080000, CRC(4b25c296) SHA1(f7bda012cd28ee58214131b254e3a63b389c39c2) )
	ROM_LOAD16_BYTE( "95405139.hi", 0x00000, 0x080000, CRC(4b694c50) SHA1(3023dbd0f42bc47a89501c4c91b761316c19ed70) )
	sc_lir_others
ROM_END

ROM_START( sc4lirf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406010.lo", 0x00001, 0x080000, CRC(e076a371) SHA1(2bfee5563a0cf5ec0591eba0401840f416ba2e0b) )
	ROM_LOAD16_BYTE( "95406011.hi", 0x00000, 0x080000, CRC(add37113) SHA1(aa5c1569b2a6a2ff9ec30742e604d7e013157e46) )
	sc_lir_others
ROM_END

ROM_START( sc4lirg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406540.lo", 0x00001, 0x080000, CRC(80d13d97) SHA1(3879840b874106c7a79578eb200102797777a261) )
	ROM_LOAD16_BYTE( "95406541.hi", 0x00000, 0x080000, CRC(96a75813) SHA1(12442e1eeff23c9e61bc37f5542cab0923a45a5d) )
	sc_lir_others
ROM_END

ROM_START( sc4lirh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407010.lo", 0x00001, 0x080000, CRC(eb15044f) SHA1(8b09f38a44bc85305f739c819c9dd57ec3cd328d) )
	ROM_LOAD16_BYTE( "95407011.hi", 0x00000, 0x080000, CRC(043d9b69) SHA1(fe89c9c7ad931796179fb9da798f03a503b09f2a) )
	sc_lir_others
ROM_END

ROM_START( sc4liri )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407540.lo", 0x00001, 0x080000, CRC(8acd7c90) SHA1(fea4b8e191efe54af34aef98a37e8123b18830ce) )
	ROM_LOAD16_BYTE( "95407541.hi", 0x00000, 0x080000, CRC(3272d5d8) SHA1(4d57b7a06fccb2e47b7e05273759479ea811b981) )
	sc_lir_others
ROM_END




ROM_START( sc4lined )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406410.lo", 0x00001, 0x080000, CRC(059a6162) SHA1(e0dfcc4314a961e11ac7850b49196dc99c1311e7) )
	ROM_LOAD16_BYTE( "95406411.hi", 0x00000, 0x080000, CRC(d5d1be58) SHA1(e9404372432969403b19da1599670f0493ebdcde) )
	sc_lined_others
ROM_END

ROM_START( sc4lineda )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407410.lo", 0x00001, 0x080000, CRC(43640651) SHA1(36131e4e820d538e1ac66770ab393b4a35cbd861) )
	ROM_LOAD16_BYTE( "95407411.hi", 0x00000, 0x080000, CRC(b4f5f9d1) SHA1(d2cde0ab4fbc5343c21cc87212b76e6dc17fb5f4) )
	sc_lined_others
ROM_END

ROM_START( sc4linedb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410312.lo", 0x00001, 0x080000, CRC(518b3c02) SHA1(29a518e44741a12374da153033bb062a1d880ecb) )
	ROM_LOAD16_BYTE( "95410313.hi", 0x00000, 0x080000, CRC(42a8b9cf) SHA1(f9dc766e89e58dc0a2892a32b443880c964117e8) )
	sc_lined_others
ROM_END

ROM_START( sc4linedc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411312.lo", 0x00001, 0x080000, CRC(e197d593) SHA1(d572fa6648fb54ffa080a4b5afb592649214f08f) )
	ROM_LOAD16_BYTE( "95411313.hi", 0x00000, 0x080000, CRC(7b8fcb79) SHA1(55597a75b3426e17e6f010986f9001fe4cff502f) )
	sc_lined_others
ROM_END

ROM_START( sc4linedd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412044.lo", 0x00001, 0x080000, CRC(20cdffd7) SHA1(c7cda44e258b63d066e6920fee238828b8538e26) )
	ROM_LOAD16_BYTE( "95412045.hi", 0x00000, 0x080000, CRC(95b7d9d5) SHA1(343e7d6643efe3227be3aae350b8f2e692eb444e) )
	sc_lined_others
ROM_END

ROM_START( sc4linede )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413050.lo", 0x00001, 0x080000, CRC(45ee70bb) SHA1(3a0e4faf76d20c37c2252309c6a174efbe60b6fd) )
	ROM_LOAD16_BYTE( "95413051.hi", 0x00000, 0x080000, CRC(ab28333a) SHA1(a6de1dac24dba8c6346b9938418edec638a3d6cf) )
	sc_lined_others
ROM_END

ROM_START( sc4linedf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417004.lo", 0x00001, 0x080000, CRC(ed8b10ea) SHA1(c75d5008ced3eab9b3dfb4ff2aa63412a93f6641) )
	ROM_LOAD16_BYTE( "95417005.hi", 0x00000, 0x080000, CRC(e7ff7a50) SHA1(9ab9e06338b96f35ebc70062685572eab4956cc7) )
	sc_lined_others
ROM_END

ROM_START( sc4linedg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417006.lo", 0x00001, 0x080000, CRC(3c24a417) SHA1(e273681e530cb2a47e1d864f7787a1fe55d47385) )
	ROM_LOAD16_BYTE( "95417007.hi", 0x00000, 0x080000, CRC(ced02db2) SHA1(7ccfbcd1b4b1ac4bf597449df949659c7b6a1d35) )
	sc_lined_others
ROM_END

ROM_START( sc4linedh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417008.lo", 0x00001, 0x080000, CRC(450e112d) SHA1(8c1f90494eb6c393f5906b07c1eda5553c0be236) )
	ROM_LOAD16_BYTE( "95417009.hi", 0x00000, 0x080000, CRC(ef05af27) SHA1(148318f8a156a42350ca6085daf8034bd2b90c8d) )
	sc_lined_others
ROM_END

ROM_START( sc4linedi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417010.lo", 0x00001, 0x080000, CRC(2438a69c) SHA1(5f9c94e5167d70322940999179ca7406c5466b2e) )
	ROM_LOAD16_BYTE( "95417011.hi", 0x00000, 0x080000, CRC(27801fe1) SHA1(aedfdbff63e98dbba48ee986d9638ff4c2635b63) )
	sc_lined_others
ROM_END




ROM_START( sc4ldcas )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404020.lo", 0x00001, 0x080000, CRC(91b6e2a8) SHA1(a748451e1345032277c85c68611a37e3413d1c71) )
	ROM_LOAD16_BYTE( "95404021.hi", 0x00000, 0x080000, CRC(3f78523d) SHA1(bb3198fb2d1b1bad29c839f2b19d783a02b6cb67) )
	sc_lined_others
ROM_END

ROM_START( sc4ldcasa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404022.lo", 0x00001, 0x080000, CRC(9f74e76c) SHA1(1fa9b169cfd7d428cfdc5b1601d9d7297f9b15db) )
	ROM_LOAD16_BYTE( "95404023.hi", 0x00000, 0x080000, CRC(ed927579) SHA1(6333dab7c18ed333a2d14a3057a518cfaf098029) )
	sc_lined_others
ROM_END

ROM_START( sc4ldcasb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404026.lo", 0x00001, 0x080000, CRC(8606d116) SHA1(984a188f3e67264580cd5c1abcc411776b7869e1) )
	ROM_LOAD16_BYTE( "95404027.hi", 0x00000, 0x080000, CRC(a95075fb) SHA1(482b73503993b7604b1b55fb6391987ad500510b) )
	sc_lined_others
ROM_END

ROM_START( sc4ldcasc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405020.lo", 0x00001, 0x080000, CRC(4b88d22f) SHA1(9349fe3054a6ea83cb11bbcca490761821746621) )
	ROM_LOAD16_BYTE( "95405021.hi", 0x00000, 0x080000, CRC(057209e0) SHA1(fd2aac60358f2a69b72e314f65f13d3294c5696f) )
	sc_lined_others
ROM_END

ROM_START( sc4ldcasd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405024.lo", 0x00001, 0x080000, CRC(2992e70b) SHA1(d76dbcbecb57e4bd0ed2611fa9fec6cd4fae5b82) )
	ROM_LOAD16_BYTE( "95405025.hi", 0x00000, 0x080000, CRC(11321f24) SHA1(8733a4120f1eadb5b1f122e41cc074c71b8d1f77) )
	sc_lined_others
ROM_END

ROM_START( sc4ldcase )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405026.lo", 0x00001, 0x080000, CRC(775b7c9a) SHA1(1fd89fc9d156fb2b185c36c39cee69fc7fdf6b87) )
	ROM_LOAD16_BYTE( "95405027.hi", 0x00000, 0x080000, CRC(841786f5) SHA1(211f2bd6cc116dd68e05e0f48b787aeafde5cc93) )
	sc_lined_others
ROM_END




ROM_START( sc4ldvcl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95424452.lo", 0x00001, 0x080000, CRC(4cfa9136) SHA1(f72eee294de4a4f44d1449d331a16ad84d30f902) )
	ROM_LOAD16_BYTE( "95424453.hi", 0x00000, 0x080000, CRC(2ba88d72) SHA1(7444fcd0c784d5026a5b629958cd37bd70780f66) )
	sc_ldvcl_others
ROM_END







ROM_START( sc4lockb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404756.lo", 0x00001, 0x080000, CRC(a2670248) SHA1(a295dbbbccc355b7635b1379c0d375b23ab294a6) )
	ROM_LOAD16_BYTE( "95404757.hi", 0x00000, 0x080000, CRC(dffb76b3) SHA1(37445588cda91a051a0ebf2ffee3a1c8fd14bc8d) )
	sc4lockb_others
ROM_END

ROM_START( sc4lockba )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405756.lo", 0x00001, 0x080000, CRC(a688e6ce) SHA1(fa4c2f892227199025abe9469bfc8e19beb2c528) )
	ROM_LOAD16_BYTE( "95405757.hi", 0x00000, 0x080000, CRC(a6d4a6b8) SHA1(6ce186dcdaddb9463f55b642cf091842616b5f2c) )
	sc4lockb_others
ROM_END

ROM_START( sc4lockbb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406038.lo", 0x00001, 0x080000, CRC(32d179dc) SHA1(95f6966d8ae1220cf1edae543b5eeff04cc6418f) )
	ROM_LOAD16_BYTE( "95406039.hi", 0x00000, 0x080000, CRC(948c0821) SHA1(571501f02fb8aa9d2b78305e6eb407a323c455bd) )
	sc4lockb_others
ROM_END

ROM_START( sc4lockbc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407038.lo", 0x00001, 0x080000, CRC(ca9be660) SHA1(b3a1d3ede11353b62231613a2d49e938a970751e) )
	ROM_LOAD16_BYTE( "95407039.hi", 0x00000, 0x080000, CRC(296cbe18) SHA1(5d06660721762e244078be0fa2e876b9b256bfbb) )
	sc4lockb_others
ROM_END

ROM_START( sc4lockbd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408270.lo", 0x00001, 0x080000, CRC(1f2dc3d2) SHA1(d41f3a17c56c620204d7950d0e47763ec745ff7c) )
	ROM_LOAD16_BYTE( "95408271.hi", 0x00000, 0x080000, CRC(aa555d95) SHA1(b703e308ab7041ca93bacef586cf3448bb96d4a9) )
	sc4lockb_others
ROM_END

ROM_START( sc4lockbe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408350.lo", 0x00001, 0x080000, CRC(f1fe6ad4) SHA1(e1d7fc107a98ec9e7553a4bde586c09ffc4ee51c) )
	ROM_LOAD16_BYTE( "95408351.hi", 0x00000, 0x080000, CRC(37e7738c) SHA1(f988adeac8eb3708be2f43ee76e708b570cb7662) )
	sc4lockb_others
ROM_END

ROM_START( sc4lockbf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409270.lo", 0x00001, 0x080000, CRC(48e6703e) SHA1(ba7d0341d6e53c57d69c8430beacc075c0ac0379) )
	ROM_LOAD16_BYTE( "95409271.hi", 0x00000, 0x080000, CRC(d039790e) SHA1(61ace5427f1785bcb67d66e1f7a4ed6ad0ed9cdf) )
	sc4lockb_others
ROM_END

ROM_START( sc4lockbg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409350.lo", 0x00001, 0x080000, CRC(9bf283d3) SHA1(be5068c4702f3e26311c7c0843dd60304fea147d) )
	ROM_LOAD16_BYTE( "95409351.hi", 0x00000, 0x080000, CRC(d541cb84) SHA1(7562f2e95148c2e30729105522bf9d00996d83c9) )
	sc4lockb_others
ROM_END




ROM_START( sc4lkbcl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406660.lo", 0x00001, 0x080000, CRC(6b47d48e) SHA1(6ddb9638e8bee9b82cdeaa72736fbeb62d0bdeb3) )
	ROM_LOAD16_BYTE( "95406661.hi", 0x00000, 0x080000, CRC(287aeded) SHA1(5b0b9af07bcf9eeae15bc1d4a27681a246ee417f) )
	sc_lkbcl_others
ROM_END

ROM_START( sc4lkbcla )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406810.lo", 0x00001, 0x080000, CRC(67375225) SHA1(5381fe140096d969c313c3e1b7377a569c3301d8) )
	ROM_LOAD16_BYTE( "95406811.hi", 0x00000, 0x080000, CRC(1d7031b2) SHA1(04827897f35009c1e8f0776fcc93eaf5756d072a) )
	sc_lkbcl_others
ROM_END

ROM_START( sc4lkbclb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407660.lo", 0x00001, 0x080000, CRC(e210b047) SHA1(9922637d996ef76078de4206edb065e2a8ed61d3) )
	ROM_LOAD16_BYTE( "95407661.hi", 0x00000, 0x080000, CRC(329374b7) SHA1(8c28e1b788fa902ef23fee414f7706d824860b0e) )
	sc_lkbcl_others
ROM_END

ROM_START( sc4lkbclc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407810.lo", 0x00001, 0x080000, CRC(b9144106) SHA1(6c97a5b3eab1172b89c923f7c4febae826ec19dd) )
	ROM_LOAD16_BYTE( "95407811.hi", 0x00000, 0x080000, CRC(283c553e) SHA1(e68b2c265351e84336644973eb93b79af9058ad7) )
	sc_lkbcl_others
ROM_END

ROM_START( sc4lkbcld )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408040.lo", 0x00001, 0x080000, CRC(2175a392) SHA1(b4e82c8e9b15f644c2a8d00e3d309e3d219105fa) )
	ROM_LOAD16_BYTE( "95408041.hi", 0x00000, 0x080000, CRC(feb97ec7) SHA1(bab549172ac3537180dbffb06a391deb3d24c555) )
	sc_lkbcl_others
ROM_END

ROM_START( sc4lkbcle )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408348.lo", 0x00001, 0x080000, CRC(314bb7de) SHA1(2b469b5904131f8d5a59a2d380ed7e8aa54198cb) )
	ROM_LOAD16_BYTE( "95408349.hi", 0x00000, 0x080000, CRC(a776d21d) SHA1(bfa3c42bfb0d7c35b362df877f1f1acb867bff8c) )
	sc_lkbcl_others
ROM_END

ROM_START( sc4lkbclf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409040.lo", 0x00001, 0x080000, CRC(b468a2c9) SHA1(670297a776188b9b3c36369ff6c0e4a9e741f942) )
	ROM_LOAD16_BYTE( "95409041.hi", 0x00000, 0x080000, CRC(9307a4b7) SHA1(3900ff654cf3161c7c7bec0143841344a6753450) )
	sc_lkbcl_others
ROM_END

ROM_START( sc4lkbclg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409348.lo", 0x00001, 0x080000, CRC(e9535725) SHA1(ce6970d6531f14ba5a63261b0d36fd743f195c12) )
	ROM_LOAD16_BYTE( "95409349.hi", 0x00000, 0x080000, CRC(5391762f) SHA1(719372b44e9b5b0c071e5b991c2757459171881f) )
	sc_lkbcl_others
ROM_END

ROM_START( sc4lkbclh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412198.lo", 0x00001, 0x080000, CRC(9295556a) SHA1(38a8ac18b15982b626ef0e7fbbee2507a57e236b) )
	ROM_LOAD16_BYTE( "95412199.hi", 0x00000, 0x080000, CRC(448574bf) SHA1(f9c9802d7d8fbe195bc1b2c9bbf7a71d81402603) )
	sc_lkbcl_others
ROM_END

ROM_START( sc4lkbcli )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413198.lo", 0x00001, 0x080000, CRC(bb7df1d5) SHA1(1dc0aedcd1bae533d0dd6c3639557eb14fbc115d) )
	ROM_LOAD16_BYTE( "95413199.hi", 0x00000, 0x080000, CRC(9a3cddef) SHA1(f04bbaf181eda99f2d627373f349f0d27bb46bb5) )
	sc_lkbcl_others
ROM_END








ROM_START( sc4lotrf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406762.lo", 0x00001, 0x080000, CRC(c098f523) SHA1(4139fbbf2cb07c2430c17bab2363f1d20842a407) )
	ROM_LOAD16_BYTE( "95406763.hi", 0x00000, 0x080000, CRC(dcd11097) SHA1(fa0fc2873dd568f43d3c45b40a43cb2f1c04739c) )
	sc_lotrf_others
ROM_END

ROM_START( sc4lotrfa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407762.lo", 0x00001, 0x080000, CRC(29d01c1a) SHA1(88dc82758ff0d276558348af7954fa0f05152f60) )
	ROM_LOAD16_BYTE( "95407763.hi", 0x00000, 0x080000, CRC(5815fb7a) SHA1(99d1643d7e0ba1e1692d4c633347861ceade4001) )
	sc_lotrf_others
ROM_END


ROM_START( sc4lotrfb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406882.lo", 0x00001, 0x080000, CRC(ca19c5ac) SHA1(674245c293642425225c0c93b625b03d20c191cf) )
	ROM_LOAD16_BYTE( "95406883.hi", 0x00000, 0x080000, CRC(be029534) SHA1(20d2db4d32a09627a8949e5fcae8f67d6707b545) )
	sc_lotrf_others
ROM_END


ROM_START( sc4lotrfc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407882.lo", 0x00001, 0x080000, CRC(791847c8) SHA1(edfa061f8472f30b134646205ec23e234cfae01d) )
	ROM_LOAD16_BYTE( "95407883.hi", 0x00000, 0x080000, CRC(d1c117a4) SHA1(c250ec310cf18ddc1a9c8666983812b29ff93c53) )
	sc_lotrf_others
ROM_END

ROM_START( sc4lotrfd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408034.lo", 0x00001, 0x080000, CRC(62512ad2) SHA1(9c540efa24caf9c376ccd261fda7c95a56acbadc) )
	ROM_LOAD16_BYTE( "95408035.hi", 0x00000, 0x080000, CRC(432ac324) SHA1(fb9b69f885cc2af8b39dc0820500133d8b136f66) )
	sc_lotrf_others
ROM_END

ROM_START( sc4lotrfe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408036.lo", 0x00001, 0x080000, CRC(74adbd87) SHA1(6320b606eb2ac2188660325a9a21bf69bf61af1c) )
	ROM_LOAD16_BYTE( "95408037.hi", 0x00000, 0x080000, CRC(a0c8a398) SHA1(486a51853878019c371030c70c601cef0b32f54e) )
	sc_lotrf_others
ROM_END

ROM_START( sc4lotrff )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409034.lo", 0x00001, 0x080000, CRC(ff2f081e) SHA1(75a95fe3647a66329f437258805f20ded758ce5a) )
	ROM_LOAD16_BYTE( "95409035.hi", 0x00000, 0x080000, CRC(85252b53) SHA1(a7bbf8c2e32513138d3c7d39ed773ddad30d7d0c) )
	sc_lotrf_others
ROM_END

ROM_START( sc4lotrfg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409036.lo", 0x00001, 0x080000, CRC(223d1401) SHA1(0e654d29476bc3f0c6c8fc529330000df4634add) )
	ROM_LOAD16_BYTE( "95409037.hi", 0x00000, 0x080000, CRC(6cc5d28f) SHA1(070162696f0a043d58df6c8f82a06ff4ad0b6a65) )
	sc_lotrf_others
ROM_END




ROM_START( sc4lotr2 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408152.lo", 0x00001, 0x080000, CRC(051730dc) SHA1(1c6fe0d5a049d9808920854ca295eea816c9c206) )
	ROM_LOAD16_BYTE( "95408153.hi", 0x00000, 0x080000, CRC(a855fa62) SHA1(3a82fabb5ccd69fa9f578c1ee23a3c79cc6ee172) )
	sc_lotr2_others
ROM_END

ROM_START( sc4lotr2a )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408208.lo", 0x00001, 0x080000, CRC(005ce3ea) SHA1(45d2b86e291f231f054b1c4cb81ff745a77a3e44) )
	ROM_LOAD16_BYTE( "95408209.hi", 0x00000, 0x080000, CRC(b2536b9f) SHA1(4c1e7fb5cf9a5df30c0a48516e687ed87ef4fcb1) )
	sc_lotr2_others
ROM_END

ROM_START( sc4lotr2b )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408548.lo", 0x00001, 0x080000, CRC(71697972) SHA1(00c5299293931ff2c135541c29a5cb48cea77e68) )
	ROM_LOAD16_BYTE( "95408549.hi", 0x00000, 0x080000, CRC(65e6822e) SHA1(f9672bf4e5d97e3c7a9b0c0bd6f05852c6f90982) )
	sc_lotr2_others
ROM_END

ROM_START( sc4lotr2c )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409152.lo", 0x00001, 0x080000, CRC(75c4cad2) SHA1(308221608389158d4b3c9215ae6b4fa0694516ab) )
	ROM_LOAD16_BYTE( "95409153.hi", 0x00000, 0x080000, CRC(b43b2980) SHA1(c3bf20d55719f567d5cbb64dabe6289099b89a75) )
	sc_lotr2_others
ROM_END

ROM_START( sc4lotr2d )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409208.lo", 0x00001, 0x080000, CRC(7336a670) SHA1(879f903e63c138f234e411c665db88da02c2f71e) )
	ROM_LOAD16_BYTE( "95409209.hi", 0x00000, 0x080000, CRC(1fdc8f56) SHA1(5132821d8e27f4d8ad7ba02759494cce8a1996b6) )
	sc_lotr2_others
ROM_END

ROM_START( sc4lotr2e )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409548.lo", 0x00001, 0x080000, CRC(e5f8d2ec) SHA1(a5718f9fa4a4705a25425d39230ab8ced037b323) )
	ROM_LOAD16_BYTE( "95409549.hi", 0x00000, 0x080000, CRC(071e670f) SHA1(fb3c39ae927c6eda5e229bca0d5a645807d5c7fc) )
	sc_lotr2_others
ROM_END

ROM_START( sc4lotr2f )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410558.lo", 0x00001, 0x080000, CRC(d60a2b02) SHA1(baca25b69b7669d0c97f9cde09ad26bba0944e32) )
	ROM_LOAD16_BYTE( "95410559.hi", 0x00000, 0x080000, CRC(1a567317) SHA1(a2fa2bbde9ced1e3a7b364c5ee35048169216486) )
	sc_lotr2_others
ROM_END

ROM_START( sc4lotr2g )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410560.lo", 0x00001, 0x080000, CRC(ff25fd5a) SHA1(47a458f3bde3831bd31f4ff43e1650fb8a2189ba) )
	ROM_LOAD16_BYTE( "95410561.hi", 0x00000, 0x080000, CRC(a8afbd1f) SHA1(ae174e04ca288caa75d221fff904e3a215cb0c1d) )
	sc_lotr2_others
ROM_END

ROM_START( sc4lotr2h )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411558.lo", 0x00001, 0x080000, CRC(6476bcf7) SHA1(a8ccfcec79b8cddaa51a96b5a801b7abf481061f) )
	ROM_LOAD16_BYTE( "95411559.hi", 0x00000, 0x080000, CRC(9d23c8cb) SHA1(073644f8bdb41a88dc4107164bd74e27bdc9efb8) )
	sc_lotr2_others
ROM_END

ROM_START( sc4lotr2i )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411560.lo", 0x00001, 0x080000, CRC(fe4828cf) SHA1(71ccc824675082d16f1dc5848a20e06c6c4b5a0c) )
	ROM_LOAD16_BYTE( "95411561.hi", 0x00000, 0x080000, CRC(007ffb15) SHA1(a5ee74bd53a97faa41c428873a23d1eba41ea587) )
	sc_lotr2_others
ROM_END

ROM_START( sc4lotr2j )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412412.lo", 0x00001, 0x080000, CRC(63cde735) SHA1(e6179c97bf18ed8a29ce7fea7cefa3f1a0a638e9) )
	ROM_LOAD16_BYTE( "95412413.hi", 0x00000, 0x080000, CRC(146ae98a) SHA1(e8d3694fd7aeee029196718d901b853bf0b6d1e0) )
	sc_lotr2_others
ROM_END

ROM_START( sc4lotr2k )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413412.lo", 0x00001, 0x080000, CRC(de8c97d7) SHA1(dd8af3671809f8da04b3c6069fc9ca83cc907d8e) )
	ROM_LOAD16_BYTE( "95413413.hi", 0x00000, 0x080000, CRC(e1752ace) SHA1(2524300ec0fa2f760b686c7cc1ee6f200bc2411e) )
	sc_lotr2_others
ROM_END

ROM_START( sc4lotr2l )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1315p11.hi", 0x00000, 0x080000, CRC(6efd6376) SHA1(6820dc5dea912935e0497cb60e45b350e2c62d56) )
	ROM_LOAD16_BYTE( "p1315p11.lo", 0x00001, 0x080000, CRC(cd46cf6e) SHA1(52bdf42178d1e456373e34b9b1d3e5f9ba75b3d5) )
	sc_lotr2_others
ROM_END

ROM_START( sc4lotr2m )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1315s11.hi", 0x00000, 0x080000, CRC(d60911d4) SHA1(16823f0d904746bf0b6025810f68cd0dc687939c) )
	ROM_LOAD16_BYTE( "p1315s11.lo", 0x00001, 0x080000, CRC(54fb7c51) SHA1(c5ce97f0dfc0a30f37fb77e6aca623cc0a44963c) )
	sc_lotr2_others
ROM_END




ROM_START( sc4ltr2c )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408310.lo", 0x00001, 0x080000, CRC(915911cc) SHA1(23f8127be17a72f41a878d1dcb91d01be018f0a9) )
	ROM_LOAD16_BYTE( "95408311.hi", 0x00000, 0x080000, CRC(3b02b5c4) SHA1(7c8ad446df6b6695cce6ebf0c743ff721957f2d9) )
	sc_ltr2c_others
ROM_END

ROM_START( sc4ltr2ca )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408552.lo", 0x00001, 0x080000, CRC(32421027) SHA1(00a2a8dd7aefb2f0695ac9802742cafa61c901c9) )
	ROM_LOAD16_BYTE( "95408553.hi", 0x00000, 0x080000, CRC(ac2e7fea) SHA1(742153d17113618c07a8788432f576e2f54f3d97) )
	sc_ltr2c_others
ROM_END

ROM_START( sc4ltr2cb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408580.lo", 0x00001, 0x080000, CRC(a481bc4a) SHA1(7d55806502fd4e8300c53f9afe6db0571398363e) )
	ROM_LOAD16_BYTE( "95408581.hi", 0x00000, 0x080000, CRC(d5ba2c90) SHA1(e14ef19f137adb7fb5d4a0e9bbfe7f0828fc0bde) )
	sc_ltr2c_others
ROM_END

ROM_START( sc4ltr2cc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408582.lo", 0x00001, 0x080000, CRC(832650ae) SHA1(50e8e3a4f4c8a846258f9a5f4e37a20d0bd889cb) )
	ROM_LOAD16_BYTE( "95408583.hi", 0x00000, 0x080000, CRC(51d6b708) SHA1(97283827944a4e3c40aff1823f22ee1dc23e28e8) )
	sc_ltr2c_others
ROM_END

ROM_START( sc4ltr2cd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409310.lo", 0x00001, 0x080000, CRC(b7ddce2b) SHA1(da5d49ac7bfdf16f802f6fa555eb95b3e192ad18) )
	ROM_LOAD16_BYTE( "95409311.hi", 0x00000, 0x080000, CRC(e2dc12a4) SHA1(24749fbdd0779bfcc7c0a8317699504db5a0be8b) )
	sc_ltr2c_others
ROM_END

ROM_START( sc4ltr2ce )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409552.lo", 0x00001, 0x080000, CRC(9d7d56c8) SHA1(142d8c7a39fe36b80b98c7000a4e158c6af8cf3e) )
	ROM_LOAD16_BYTE( "95409553.hi", 0x00000, 0x080000, CRC(738bf842) SHA1(6f146749c403c48f5968978c8ea9127fe3fa3c29) )
	sc_ltr2c_others
ROM_END

ROM_START( sc4ltr2cf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409580.lo", 0x00001, 0x080000, CRC(8a2b6c03) SHA1(c64ed66d851f3f384a219179dea6226d1d3d5301) )
	ROM_LOAD16_BYTE( "95409581.hi", 0x00000, 0x080000, CRC(64c3cf8f) SHA1(5fb7e3dc294f75106a244bfef899661c7588bb67) )
	sc_ltr2c_others
ROM_END

ROM_START( sc4ltr2cg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409582.lo", 0x00001, 0x080000, CRC(28e31a7b) SHA1(e4326b3e24c60287dc914e1536cbad910db60563) )
	ROM_LOAD16_BYTE( "95409583.hi", 0x00000, 0x080000, CRC(52e38587) SHA1(2c39f8ae729ed760608049cb44688d379d7fbc54) )
	sc_ltr2c_others
ROM_END

ROM_START( sc4ltr2ch )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410832.lo", 0x00001, 0x080000, CRC(c1fda633) SHA1(ded61163a7d725f8260bc4ab58107af1710eec32) )
	ROM_LOAD16_BYTE( "95410833.hi", 0x00000, 0x080000, CRC(af3fcae8) SHA1(a19d9c63fb0a83e358391c358d457b09d93d8b76) )
	sc_ltr2c_others
ROM_END

ROM_START( sc4ltr2ci )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410834.lo", 0x00001, 0x080000, CRC(2f478901) SHA1(b9de9e98bdba53bce60f0a765387f9001eafff5b) )
	ROM_LOAD16_BYTE( "95410835.hi", 0x00000, 0x080000, CRC(83fee9d8) SHA1(bcd35c40ea5be84bf255df15f78a68f9a630f848) )
	sc_ltr2c_others
ROM_END

ROM_START( sc4ltr2cj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411832.lo", 0x00001, 0x080000, CRC(175b6bee) SHA1(3d3885cb8cd26a6c559d36e8e8fefa79ae56d0a6) )
	ROM_LOAD16_BYTE( "95411833.hi", 0x00000, 0x080000, CRC(b477a9fa) SHA1(a7bf76775b58fb8cc9e18c42a75d3da562b5fc31) )
	sc_ltr2c_others
ROM_END

ROM_START( sc4ltr2ck )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411834.lo", 0x00001, 0x080000, CRC(5aa1bc5d) SHA1(1a4e705a189ffb5d9c685faf5cd3f8a088a0dfa6) )
	ROM_LOAD16_BYTE( "95411835.hi", 0x00000, 0x080000, CRC(47318fea) SHA1(a7255a9004493bb609d3e7aaa09631a9cc27b5cc) )
	sc_ltr2c_others
ROM_END




ROM_START( sc4lotrt )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408274.lo", 0x00001, 0x080000, CRC(8d93666b) SHA1(cac29eaae1000abdf69d073109e03050aeee630b) )
	ROM_LOAD16_BYTE( "95408275.hi", 0x00000, 0x080000, CRC(95e4cf85) SHA1(a203564752af2d4821dd384e0a0670d5e565dd5f) )
	sc_lotrt_others
ROM_END

ROM_START( sc4lotrta )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409274.lo", 0x00001, 0x080000, CRC(1f61cac5) SHA1(334bf47adfbf2eddda1200681851084bd4380bbd) )
	ROM_LOAD16_BYTE( "95409275.hi", 0x00000, 0x080000, CRC(09464173) SHA1(9db97e339d66175909217960b2e863b72b238d15) )
	sc_lotrt_others
ROM_END




ROM_START( sc4luck7 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406912.lo", 0x00001, 0x080000, CRC(ccc83e2a) SHA1(a12204a8e7e4fb93cb594152fb605a571ada9c21) )
	ROM_LOAD16_BYTE( "95406913.hi", 0x00000, 0x080000, CRC(3b33e4c7) SHA1(d3912d229bfb0092623bad3a19885449944ab06a) )
	sc_luck7_others
ROM_END

ROM_START( sc4luck7a )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410286.lo", 0x00001, 0x080000, CRC(143befbe) SHA1(f99970ccdcbcc011837ba5e610f6bdf6f1659bf9) )
	ROM_LOAD16_BYTE( "95410287.hi", 0x00000, 0x080000, CRC(3b680008) SHA1(0b0ea12ca15ab136e38fd96f1b3776d25157cb47) )
	sc_luck7_others
ROM_END

ROM_START( sc4luck7b )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410290.lo", 0x00001, 0x080000, CRC(682a2618) SHA1(b360a6566ffda0e7f5e776a12c251ffc1328a84a) )
	ROM_LOAD16_BYTE( "95410291.hi", 0x00000, 0x080000, CRC(d45c4426) SHA1(abf155b1cf3ec3b08c913c9b58f31455d81a5ba6) )
	sc_luck7_others
ROM_END

ROM_START( sc4luck7c )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411286.lo", 0x00001, 0x080000, CRC(d6dce05d) SHA1(cd6d4503016a26f99efce5b25c995304aec8fa29) )
	ROM_LOAD16_BYTE( "95411287.hi", 0x00000, 0x080000, CRC(23fcc1b2) SHA1(99c4ee053c93a5007cfe680e9f30eb8b64b5e1a3) )
	sc_luck7_others
ROM_END

ROM_START( sc4luck7d )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415994.lo", 0x00001, 0x080000, CRC(4d0d5fb6) SHA1(5a54d9fe966852d59cff449625568c212b6da2ed) )
	ROM_LOAD16_BYTE( "95415995.hi", 0x00000, 0x080000, CRC(98220928) SHA1(7c5eac3fc3bb53de42b9600224f0549768939379) )
	sc_luck7_others
ROM_END




ROM_START( sc4luckb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404198.lo", 0x00001, 0x080000, CRC(d5071084) SHA1(c70cce06cda15d8ce9bc38eb659613d97d86391c) )
	ROM_LOAD16_BYTE( "95404199.hi", 0x00000, 0x080000, CRC(382d104c) SHA1(585475851401e3346dea7b79e3f77cd78aaef66b) )
	sc_luckb_others
ROM_END

ROM_START( sc4luckba )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404210.lo", 0x00001, 0x080000, CRC(b8467858) SHA1(0ef2941a98f36ec25a91b0a7970e70269cbbb078) )
	ROM_LOAD16_BYTE( "95404211.hi", 0x00000, 0x080000, CRC(49af67fb) SHA1(e424d5a70cd113b9c14da9490efe1c9763b82a8b) )
	sc_luckb_others
ROM_END

ROM_START( sc4luckbb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404212.lo", 0x00001, 0x080000, CRC(69a62011) SHA1(5ca44dcab6eb6c1e96d672a05df7da9d384daa94) )
	ROM_LOAD16_BYTE( "95404213.hi", 0x00000, 0x080000, CRC(71977d1d) SHA1(cd3bedb10b58f0ca8d0ecf5cf4416409e0c9fd58) )
	sc_luckb_others
ROM_END

ROM_START( sc4luckbc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405198.lo", 0x00001, 0x080000, CRC(d9f7970d) SHA1(1f68be10f90ffbdc141c57b50708fde097fe04c9) )
	ROM_LOAD16_BYTE( "95405199.hi", 0x00000, 0x080000, CRC(6cda6571) SHA1(c93536563f1947cd94274059645a791a4da25a6a) )
	sc_luckb_others
ROM_END

ROM_START( sc4luckbd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405210.lo", 0x00001, 0x080000, CRC(a7190577) SHA1(27adb7ba1bf66521c5b36a34b96a79e94e727984) )
	ROM_LOAD16_BYTE( "95405211.hi", 0x00000, 0x080000, CRC(12817baf) SHA1(79ce82d0af2b4ca7038dadf338d968385d9d64d1) )
	sc_luckb_others
ROM_END

ROM_START( sc4luckbe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405212.lo", 0x00001, 0x080000, CRC(76f95d3e) SHA1(15df0222a31708f71604e0cba1bd1863d426715e) )
	ROM_LOAD16_BYTE( "95405213.hi", 0x00000, 0x080000, CRC(2ab96149) SHA1(3fdae3ba95c3e76688358f01605188f0f98d13d9) )
	sc_luckb_others
ROM_END

ROM_START( sc4luckbf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406556.lo", 0x00001, 0x080000, CRC(42caab1a) SHA1(edb343270922d3f0fcd85c8edfd8a0a21249fbf2) )
	ROM_LOAD16_BYTE( "95406557.hi", 0x00000, 0x080000, CRC(6c5298d7) SHA1(d8eaad0c68fb6c523d3ecf8b3342f6f190715366) )
	sc_luckb_others
ROM_END

ROM_START( sc4luckbg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406720.lo", 0x00001, 0x080000, CRC(617e5530) SHA1(a1cbfecef284ca211fbe952a178a3f21a88b826b) )
	ROM_LOAD16_BYTE( "95406721.hi", 0x00000, 0x080000, CRC(db36db09) SHA1(c96c72d7f61c96b199d0d4e9cc095dbf51bec03e) )
	sc_luckb_others
ROM_END

ROM_START( sc4luckbh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407556.lo", 0x00001, 0x080000, CRC(5f8c33d7) SHA1(b1ee5527d33c35dfe3a7a3b22cf71845a84514d7) )
	ROM_LOAD16_BYTE( "95407557.hi", 0x00000, 0x080000, CRC(6549b7e0) SHA1(9dced4501cd7300f58f194bfa39a13940567001b) )
	sc_luckb_others
ROM_END

ROM_START( sc4luckbi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407720.lo", 0x00001, 0x080000, CRC(ca3b6b30) SHA1(e3b841b5ce34c3653e2c6798e60ec069c01673d7) )
	ROM_LOAD16_BYTE( "95407721.hi", 0x00000, 0x080000, CRC(a20a761a) SHA1(6992c04552eb74001dbbdd7f972039c1a4cff992) )
	sc_luckb_others
ROM_END

ROM_START( sc4luckbj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408232.lo", 0x00001, 0x080000, CRC(4756a519) SHA1(95a833bba170817f4f84cd2bf7d92b6c1f9c4d2f) )
	ROM_LOAD16_BYTE( "95408233.hi", 0x00000, 0x080000, CRC(db609b6f) SHA1(ec6f5f2c103c8b8b64b7dc38e302a0fb1181304f) )
	sc_luckb_others
ROM_END

ROM_START( sc4luckbk )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409232.lo", 0x00001, 0x080000, CRC(2dfdd2c1) SHA1(21690662b28280bca0a033f6ae9407a1130caafb) )
	ROM_LOAD16_BYTE( "95409233.hi", 0x00000, 0x080000, CRC(0943afc6) SHA1(5fd0462edb15ab61e0cfb7f8d761de82811cca71) )
	sc_luckb_others
ROM_END

ROM_START( sc4luckbl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410030.lo", 0x00001, 0x080000, CRC(3259b382) SHA1(83a6d86de7c1fdb954b73a142c029a3f0176b2f5) )
	ROM_LOAD16_BYTE( "95410031.hi", 0x00000, 0x080000, CRC(1000c265) SHA1(8ea5202d2e3f7cf03404b50bbbe9229caf22dc68) )
	sc_luckb_others
ROM_END

ROM_START( sc4luckbm )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410032.lo", 0x00001, 0x080000, CRC(441fda92) SHA1(3dc93712042d67ce8d31dce617bc5befd340caa8) )
	ROM_LOAD16_BYTE( "95410033.hi", 0x00000, 0x080000, CRC(9a5884a6) SHA1(c79d5a444f9130105cf999fc50200159957c8cd6) )
	sc_luckb_others
ROM_END

ROM_START( sc4luckbn )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410034.lo", 0x00001, 0x080000, CRC(6f0c08f4) SHA1(62a656dd8d131dba27f2f960fa8c460016d565d8) )
	ROM_LOAD16_BYTE( "95410035.hi", 0x00000, 0x080000, CRC(f00b16d6) SHA1(be880bbf30779f98d646c0965fbce18c281fde63) )
	sc_luckb_others
ROM_END

ROM_START( sc4luckbo )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410036.lo", 0x00001, 0x080000, CRC(9b750a29) SHA1(7f94ddf82b89fdf85ace86aaebe2399ab6ea5849) )
	ROM_LOAD16_BYTE( "95410037.hi", 0x00000, 0x080000, CRC(a1eee41d) SHA1(e28758e66f66c3c3343a7941e3e86846e73fea8a) )
	sc_luckb_others
ROM_END

ROM_START( sc4luckbp )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410038.lo", 0x00001, 0x080000, CRC(32104c8e) SHA1(0c503de053b058b6bb92c44926318c49d443026a) )
	ROM_LOAD16_BYTE( "95410039.hi", 0x00000, 0x080000, CRC(f18c980d) SHA1(b7035cd07a6443804716fcf7ddb21a246d12460b) )
	sc_luckb_others
ROM_END

ROM_START( sc4luckbq )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410586.lo", 0x00001, 0x080000, CRC(8e9a9b58) SHA1(8f8d750f6e51341d84c6a3c8808bb20d08b74ac3) )
	ROM_LOAD16_BYTE( "95410587.hi", 0x00000, 0x080000, CRC(1214cce0) SHA1(7d5f95197e27e9285ce5167fcbec5a017605a238) )
	sc_luckb_others
ROM_END

ROM_START( sc4luckbr )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410588.lo", 0x00001, 0x080000, CRC(e8270188) SHA1(000f8903471916adc81664a9ba175699c89aa78b) )
	ROM_LOAD16_BYTE( "95410589.hi", 0x00000, 0x080000, CRC(26a7dbe8) SHA1(f89faf2c1c027b29e6780cc0da3374315394ea97) )
	sc_luckb_others
ROM_END

ROM_START( sc4luckbs )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410590.lo", 0x00001, 0x080000, CRC(dbaca336) SHA1(2eb78c89d0477046a0ca1603752d953ec35958e3) )
	ROM_LOAD16_BYTE( "95410591.hi", 0x00000, 0x080000, CRC(6bd51308) SHA1(0d8d1481b5b3c591d94dcb6e853159d79eba8795) )
	sc_luckb_others
ROM_END

ROM_START( sc4luckbt )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411030.lo", 0x00001, 0x080000, CRC(1fba23d6) SHA1(ef9b8719b5d0314b896b3288d1e9656f770def09) )
	ROM_LOAD16_BYTE( "95411031.hi", 0x00000, 0x080000, CRC(5cb4ea7d) SHA1(244805a353de49a6613a6aa417cce29745b29909) )
	sc_luckb_others
ROM_END

ROM_START( sc4luckbu )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411032.lo", 0x00001, 0x080000, CRC(1af5a90d) SHA1(d2a892953d85c2fc0b7ea7e4e545b82ba113bbb0) )
	ROM_LOAD16_BYTE( "95411033.hi", 0x00000, 0x080000, CRC(e855daf5) SHA1(a01bfb4b4c21800e7d0608dc4a91d1b6a6974ae2) )
	sc_luckb_others
ROM_END

ROM_START( sc4luckbv )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411034.lo", 0x00001, 0x080000, CRC(16db1d8f) SHA1(40af45feec9f3dbbe6a2c6b7b5eedfb6e15b5323) )
	ROM_LOAD16_BYTE( "95411035.hi", 0x00000, 0x080000, CRC(42ad9fbe) SHA1(1bac16930866625fa19a736111d54a2db831f8f0) )
	sc_luckb_others
ROM_END

ROM_START( sc4luckbw )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411036.lo", 0x00001, 0x080000, CRC(aa37b87c) SHA1(327ec2da7fe526f9b681ca24d426cbbd860239b7) )
	ROM_LOAD16_BYTE( "95411037.hi", 0x00000, 0x080000, CRC(1fc9fe4b) SHA1(5a8b31ae46f0a06e7b40bbb656ebc6b3b8d9b4c6) )
	sc_luckb_others
ROM_END

ROM_START( sc4luckbx )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411038.lo", 0x00001, 0x080000, CRC(c31ff079) SHA1(eab0d2acf9f3a343c9a71088c2944ccff1808eff) )
	ROM_LOAD16_BYTE( "95411039.hi", 0x00000, 0x080000, CRC(3dfbb430) SHA1(6b9adb872637dd5516029a8921ca99e7254288bd) )
	sc_luckb_others
ROM_END

ROM_START( sc4luckby )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411586.lo", 0x00001, 0x080000, CRC(8f074336) SHA1(2ac3ff2edf02db7eae3da808eb809f732263b0da) )
	ROM_LOAD16_BYTE( "95411587.hi", 0x00000, 0x080000, CRC(ce4f6b68) SHA1(37d6321a7a80261e81e4e42cbc439784b9f41f07) )
	sc_luckb_others
ROM_END

ROM_START( sc4luckbz )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411588.lo", 0x00000, 0x080000, NO_DUMP )
	ROM_LOAD16_BYTE( "95411589.hi", 0x00000, 0x080000, CRC(d9c31c4f) SHA1(e260e5e7acea8433a8235720b70f7a90045abb93) )
	sc_luckb_others
ROM_END

ROM_START( sc4luckb0 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411590.lo", 0x00001, 0x080000, CRC(da317b58) SHA1(6f4ae388476f3030b8197352c51cf4ae83c2f691) )
	ROM_LOAD16_BYTE( "95411591.hi", 0x00000, 0x080000, CRC(b78eb480) SHA1(c748ecd064be708ba908e7689bc6c042d66b2698) )
	sc_luckb_others
ROM_END

ROM_START( sc4luckb1 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95412784.lo", 0x00001, 0x080000, CRC(5aa0e877) SHA1(61f5feb82da6120e9dc1e035ecc5014c0312cfe5) )
	ROM_LOAD16_BYTE( "95412785.hi", 0x00000, 0x080000, CRC(71a8f0a7) SHA1(f15e021e2fd036e3b8e6df527dce247dcbcae183) )
	sc_luckb_others
ROM_END

ROM_START( sc4luckb2 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95412786.lo", 0x00001, 0x080000, CRC(ef399481) SHA1(46f896e061b1e0952f344d8263733b4767f4fd33) )
	ROM_LOAD16_BYTE( "95412787.hi", 0x00000, 0x080000, CRC(afba8e8f) SHA1(5263a8909e930eba04c9d2dd3cc561bc05dd98bc) )
	sc_luckb_others
ROM_END

ROM_START( sc4luckb3 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95413784.lo", 0x00001, 0x080000, CRC(1f6e5750) SHA1(bf66ec55f95263b9609d540ff80ff6e30b412c8f) )
	ROM_LOAD16_BYTE( "95413785.hi", 0x00000, 0x080000, CRC(1d94f72f) SHA1(cdb6f996738d24c5067930ef24699d851a0bbdc6) )
	sc_luckb_others
ROM_END

ROM_START( sc4luckb4 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95413786.lo", 0x00001, 0x080000, CRC(afb1726c) SHA1(676e27a33da21a728129d179339aab761485b22d) )
	ROM_LOAD16_BYTE( "95413787.hi", 0x00000, 0x080000, CRC(c3a6cc83) SHA1(399629c1a80790d4590b97d800f4dcf0ff7f7180) )
	sc_luckb_others
ROM_END



ROM_START( sc4magci )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414684.odd", 0x00001, 0x080000, CRC(f2259f9e) SHA1(09407e802851b995b72647e0b3c3f5dfb06074d0) )
	ROM_LOAD16_BYTE( "95414685.evn", 0x00000, 0x080000, CRC(3899b757) SHA1(ba03059073822a3008c6222232e53c9740a49169) )
	sc_magci_others
ROM_END

ROM_START( sc4magcia )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414686.odd", 0x00001, 0x080000, CRC(1d0dc678) SHA1(6f555e8836cb06cfc62295c0e446790e60e254c2) )
	ROM_LOAD16_BYTE( "95414687.evn", 0x00000, 0x080000, CRC(57a2a756) SHA1(33a7df99b01134129f9c57df3d117552cd3e4517) )
	sc_magci_others
ROM_END

ROM_START( sc4magcib )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414688.odd", 0x00001, 0x080000, CRC(9fb7a7e2) SHA1(6ae25302a4e4b18776b9c7f2c602cecfea63c929) )
	ROM_LOAD16_BYTE( "95414689.evn", 0x00000, 0x080000, CRC(1d506b89) SHA1(6afd34571461800b2c84d444b1ee18f45574a793) )
	sc_magci_others
ROM_END

ROM_START( sc4magcic )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414750.odd", 0x00001, 0x080000, CRC(ba2c853e) SHA1(ce1bb19171636437cc59aa1030d7cb11c917a9a7) )
	ROM_LOAD16_BYTE( "95414751.evn", 0x00000, 0x080000, CRC(fd25ed47) SHA1(556e22439461a295f7f66ec17b69c0647f1259ab) )
	sc_magci_others
ROM_END

ROM_START( sc4magcid )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414768.odd", 0x00001, 0x080000, CRC(85619860) SHA1(621aa8a063f5be226a022f7480ea3acfcd68dac0) )
	ROM_LOAD16_BYTE( "95414769.evn", 0x00000, 0x080000, CRC(9346a662) SHA1(3891e1cb52e5848eb641fdc4714c1bcb64beebdf) )
	sc_magci_others
ROM_END

ROM_START( sc4magcie )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414770.odd", 0x00001, 0x080000, CRC(d6dfaa33) SHA1(c443662c2c18156093937e04bcd7fa1b79d5f743) )
	ROM_LOAD16_BYTE( "95414771.evn", 0x00000, 0x080000, CRC(8b6f7f4c) SHA1(d4b4e6a5bd1a17a2dbe55328a5af4481fad45828) )
	sc_magci_others
ROM_END

ROM_START( sc4magcif )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414772.odd", 0x00001, 0x080000, CRC(4d50f05b) SHA1(86457bb2cbf35ac2c0f45a6d4cbb0ed66ce6336d) )
	ROM_LOAD16_BYTE( "95414773.evn", 0x00000, 0x080000, CRC(43a8b33c) SHA1(32ad816185a45fba2ee003c8dfa3c899bb4db31b) )
	sc_magci_others
ROM_END

ROM_START( sc4magcig )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414838.odd", 0x00001, 0x080000, CRC(992c1a25) SHA1(5342d945bf7688556b918a658c6506a70869ff5f) )
	ROM_LOAD16_BYTE( "95414839.evn", 0x00000, 0x080000, CRC(b95af506) SHA1(e006c24f70c7eec2b92ad3278115c27320987041) )
	sc_magci_others
ROM_END

ROM_START( sc4magcih )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414902.odd", 0x00001, 0x080000, CRC(7114c603) SHA1(63df6b634e05487b3ba5396f61e7aacdd8e7550c) )
	ROM_LOAD16_BYTE( "95414903.evn", 0x00000, 0x080000, CRC(c577c2f6) SHA1(95032c7bdf71a1a513fb59133bd20ec519409843) )
	sc_magci_others
ROM_END

ROM_START( sc4magcii )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414904.odd", 0x00001, 0x080000, CRC(7a6f6426) SHA1(154ab05d57c0af47ab6ed02d49916458aac42608) )
	ROM_LOAD16_BYTE( "95414905.evn", 0x00000, 0x080000, CRC(6de110ce) SHA1(910f52ca9d9716cfcfa12b262554ff06049b10c0) )
	sc_magci_others
ROM_END

ROM_START( sc4magcij )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414906.odd", 0x00001, 0x080000, CRC(8995a5ca) SHA1(28c74f4f9b6e35731046445e9b5e53cea22973ea) )
	ROM_LOAD16_BYTE( "95414907.evn", 0x00000, 0x080000, CRC(0c10d778) SHA1(9d7c983b4e10c5fd25050e56bbb5c1d99cbae8df) )
	sc_magci_others
ROM_END

ROM_START( sc4magcik )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415684.odd", 0x00001, 0x080000, CRC(d34e389b) SHA1(11dd8424e2d22c390621eb423d6cc0d142eb54f3) )
	ROM_LOAD16_BYTE( "95415685.evn", 0x00000, 0x080000, CRC(de571d3a) SHA1(95cc43cf09480c457c22087bd696d00ebb439e70) )
	sc_magci_others
ROM_END

ROM_START( sc4magcil )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415686.odd", 0x00001, 0x080000, CRC(988c1320) SHA1(90f46b0fec3ab57794526e991b002e8a5d85ae20) )
	ROM_LOAD16_BYTE( "95415687.evn", 0x00000, 0x080000, CRC(ff72e15c) SHA1(a9dc73697b44d4ecc3a9627cf877335641f5d7c6) )
	sc_magci_others
ROM_END

ROM_START( sc4magcim )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415688.odd", 0x00001, 0x080000, CRC(1a3672ba) SHA1(c5a7c47e1610466e92731009639f447f9ef0ed95) )
	ROM_LOAD16_BYTE( "95415689.evn", 0x00000, 0x080000, CRC(b5802d83) SHA1(90089e237f0485129cd2e9439e43ed887b047b5a) )
	sc_magci_others
ROM_END

ROM_START( sc4magcin )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415750.odd", 0x00001, 0x080000, CRC(247a654c) SHA1(912beb51fb47fa7e952006d08ff983f3af411b8e) )
	ROM_LOAD16_BYTE( "95415751.evn", 0x00000, 0x080000, CRC(c6111aea) SHA1(5c36e39a1fd50c6b4bafdb6cb98394abf44c78f7) )
	sc_magci_others
ROM_END

ROM_START( sc4magcio )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415768.odd", 0x00001, 0x080000, CRC(1b377812) SHA1(5a3c3c78bb70cedf647f806d3225d3bfb760ef89) )
	ROM_LOAD16_BYTE( "95415769.evn", 0x00000, 0x080000, CRC(a87251cf) SHA1(f78c603e61787cee6dd0c9e11140ae1bd7672caf) )
	sc_magci_others
ROM_END

ROM_START( sc4magcip )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415770.odd", 0x00001, 0x080000, CRC(69cb728e) SHA1(a6372ddd4f6e7418947ee644eac107fbc56479c3) )
	ROM_LOAD16_BYTE( "95415771.evn", 0x00000, 0x080000, CRC(34f29195) SHA1(b818a939a3d457d2d4e791bbdede7d8f05418bdb) )
	sc_magci_others
ROM_END

ROM_START( sc4magciq )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415772.odd", 0x00001, 0x080000, CRC(f24428e6) SHA1(b18c491debfd4fe7b4188a918d1f07e225f1ed2d) )
	ROM_LOAD16_BYTE( "95415773.evn", 0x00000, 0x080000, CRC(fc355de5) SHA1(f296a5f47fcbd91317eff215463aa7b7986854c9) )
	sc_magci_others
ROM_END

ROM_START( sc4magcir )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415838.odd", 0x00001, 0x080000, CRC(e58d13b7) SHA1(776527e45c9d8147d0ca742d695f1038a3d3c870) )
	ROM_LOAD16_BYTE( "95415839.evn", 0x00000, 0x080000, CRC(25672dc8) SHA1(0afeb0b594764960d0de3297feef17c676363746) )
	sc_magci_others
ROM_END

ROM_START( sc4magcis )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415902.odd", 0x00001, 0x080000, CRC(fe2d3c97) SHA1(9041f53deb3fc2af5bc8c69f583459189c0b13b1) )
	ROM_LOAD16_BYTE( "95415903.evn", 0x00000, 0x080000, CRC(780822f7) SHA1(4c1b5b6fd623ceeb0430592fa3461c3802d0d812) )
	sc_magci_others
ROM_END

ROM_START( sc4magcit )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415904.odd", 0x00001, 0x080000, CRC(48297eb3) SHA1(e971690405b243714117bcc5d6e47ac901363195) )
	ROM_LOAD16_BYTE( "95415905.evn", 0x00000, 0x080000, CRC(dda5977e) SHA1(e3dbf9639e792e3623705450259fe474f2fe5dec) )
	sc_magci_others
ROM_END

ROM_START( sc4magciu )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415906.odd", 0x00001, 0x080000, CRC(bbd3bf5f) SHA1(beafcf49ce5cf1db7419df9e08cc3435f01ede74) )
	ROM_LOAD16_BYTE( "95415907.evn", 0x00000, 0x080000, CRC(bc5450c8) SHA1(ec5d634bfb1cc54bb98a309d3d03996ef3c65e60) )
	sc_magci_others
ROM_END




ROM_START( sc4magic )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408202.odd", 0x00001, 0x080000, CRC(6480c47f) SHA1(77e818bf71cb9e03359e4cedf3c5d0be40adf9ed) )
	ROM_LOAD16_BYTE( "95408203.evn", 0x00000, 0x080000, CRC(54682bf5) SHA1(2572ebb837a5666d026be9bcbc76a779c279d429) )
	sc_magic_others
ROM_END

ROM_START( sc4magica )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408370.odd", 0x00001, 0x080000, CRC(4c157395) SHA1(061fc05bc8efbf4b37dd6e2b919b829723d9411d) )
	ROM_LOAD16_BYTE( "95408371.evn", 0x00000, 0x080000, CRC(63e386eb) SHA1(68264c6c3fe577bf91a0d76a67c9ccefcbaf929e) )
	sc_magic_others
ROM_END

ROM_START( sc4magicb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409202.odd", 0x00001, 0x080000, CRC(90a6e7cd) SHA1(bb24462fe9dcbdfc7e894644039f1990c44d30df) )
	ROM_LOAD16_BYTE( "95409203.evn", 0x00000, 0x080000, CRC(c56e948a) SHA1(885cddc36c0aa541a3fa4a03a6ccfb26b07a7064) )
	sc_magic_others
ROM_END

ROM_START( sc4magicc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409370.odd", 0x00001, 0x080000, CRC(b7ea394e) SHA1(b81d9ff9ca228ab9eb064ccbe471d1c2e7a4e5dc) )
	ROM_LOAD16_BYTE( "95409371.evn", 0x00000, 0x080000, CRC(fd3c50fd) SHA1(fecd5b2b9816171986a3aaf4889a13a0c231ae92) )
	sc_magic_others
ROM_END








ROM_START( sc4maxim )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404176.lo", 0x00001, 0x080000, CRC(e94f5a8c) SHA1(93e9bb0b82cc93a510b1b4b73cc1e2175a014b33) )
	ROM_LOAD16_BYTE( "95404177.hi", 0x00000, 0x080000, CRC(255ee22d) SHA1(e8544ecaff119be4cbe8d79463ad4092a9a717a7) )
	sc_maxim_others
ROM_END

ROM_START( sc4maxima )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404364.lo", 0x00001, 0x080000, CRC(d514c3b4) SHA1(bfe4646ee88bb67610ffcc5ecc6457e6549db797) )
	ROM_LOAD16_BYTE( "95404365.hi", 0x00000, 0x080000, CRC(6df0669a) SHA1(0432dd9804e08128ae7dd8a02d1f76b191ec06fa) )
	sc_maxim_others
ROM_END

ROM_START( sc4maximb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405176.lo", 0x00001, 0x080000, CRC(c24dfbd0) SHA1(3ae94e0b89bb181017b730c18c3cad4af16600d2) )
	ROM_LOAD16_BYTE( "95405177.hi", 0x00000, 0x080000, CRC(c1724698) SHA1(7d82e95d2e071d02826bfecb07e43fbfd538382b) )
	sc_maxim_others
ROM_END

ROM_START( sc4maximc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405202.lo", 0x00001, 0x080000, CRC(532d0c71) SHA1(7d3ea71f41707cb43367c630f66893553c1d8f9c) )
	ROM_LOAD16_BYTE( "95405203.hi", 0x00000, 0x080000, CRC(7fa1421c) SHA1(fb1b6fc074a9ec744bb9396b000c7ef25eb99cad) )
	sc_maxim_others
ROM_END

ROM_START( sc4maximd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405364.lo", 0x00001, 0x080000, CRC(79f8b340) SHA1(546a6ce09f65ca09cbcda88afaf52a3680b9c444) )
	ROM_LOAD16_BYTE( "95405365.hi", 0x00000, 0x080000, CRC(bc90a6a3) SHA1(7eb1b129c4bd94696d094a521e670ad73cf05380) )
	sc_maxim_others
ROM_END

ROM_START( sc4maxime )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406730.lo", 0x00001, 0x080000, CRC(c90ea797) SHA1(03a8c34d388dda04db95ba119a38c0ee41c5d479) )
	ROM_LOAD16_BYTE( "95406731.hi", 0x00000, 0x080000, CRC(c60d54a9) SHA1(0bee4710db80098c02afca7b4b1752593b6308c4) )
	sc_maxim_others
ROM_END

ROM_START( sc4maximf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407730.lo", 0x00001, 0x080000, CRC(a5f9b515) SHA1(1df39e1f3ab6f9b8010b5728e7aad936019d4ee4) )
	ROM_LOAD16_BYTE( "95407731.hi", 0x00000, 0x080000, CRC(40464305) SHA1(dfbae3227e4f55c598fef806179695b2ffa27df2) )
	sc_maxim_others
ROM_END

ROM_START( sc4maximg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "maximus.lo",  0x00001, 0x080000, CRC(0a95c98e) SHA1(b72ceba01926b011065b180f1290742b3ff16268) )
	ROM_LOAD16_BYTE( "maximus.hi",  0x00000, 0x080000, CRC(f8b22a33) SHA1(0756f4ce4cc4da0b5920c64e03cb5eb5bc1124c6) )
	sc_maxim_others
ROM_END




ROM_START( sc4maxcc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410910.lo", 0x00001, 0x080000, CRC(b55d74f4) SHA1(8218e7cfd3e3265db5c57fb4c91a8a6660249f3d) )
	ROM_LOAD16_BYTE( "95410911.hi", 0x00000, 0x080000, CRC(1d521acf) SHA1(28568a89288ba7986a8ee1b9de4caaff9f8b2ba8) )
	sc_maxcc_others
ROM_END

ROM_START( sc4maxcca )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410912.lo", 0x00001, 0x080000, CRC(2cf07e5e) SHA1(0bf59aca078bef1ed8c0358f98c1b9d5c4bc2834) )
	ROM_LOAD16_BYTE( "95410913.hi", 0x00000, 0x080000, CRC(1434b7d8) SHA1(8b4ee571ba3fe60db5bbab841f1a26e92670a5c9) )
	sc_maxcc_others
ROM_END

ROM_START( sc4maxccb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411910.lo", 0x00001, 0x080000, CRC(7f811220) SHA1(2e2c61a0345d0f0334a8acb3f1afc28ef4ff99d2) )
	ROM_LOAD16_BYTE( "95411911.hi", 0x00000, 0x080000, CRC(bac46394) SHA1(79144ba540085480b12201b97aa64e6a0f2ab6c5) )
	sc_maxcc_others
ROM_END

ROM_START( sc4maxccc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411912.lo", 0x00001, 0x080000, CRC(e62c188a) SHA1(8bf25fa6318a6de9a92e09f1dc9a4eb2ff728098) )
	ROM_LOAD16_BYTE( "95411913.hi", 0x00000, 0x080000, CRC(b3a2ce83) SHA1(0c6baf68846277ef7effa6566ef849a4c2c18dc6) )
	sc_maxcc_others
ROM_END







ROM_START( sc4miljo )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406346.lo", 0x00001, 0x080000, CRC(5b884ddb) SHA1(b289b0c7d3206ca60af62b6e60cfca778d61d8e0) )
	ROM_LOAD16_BYTE( "95406347.hi", 0x00000, 0x080000, CRC(e89954d8) SHA1(d9642d3ca8b56f85f0825f4154041171cf523efd) )
	sc_miljo_others
ROM_END



ROM_START( sc4milja )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406516.lo", 0x00001, 0x080000, CRC(a315dc08) SHA1(3319a76cfe3f15128f2c1fc4cac2a9137ba4c2ce) )
	ROM_LOAD16_BYTE( "95406517.hi", 0x00000, 0x080000, CRC(865d821a) SHA1(74e3676d1db927d5b5848ee67c376f29b14b976b) )
	sc_milja_others
ROM_END




ROM_START( sc4mbags )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400435.lo", 0x00001, 0x080000, CRC(92164f6e) SHA1(4e53bf94e9fea7fbffd306c1bdcf20b6ac4e7b35) )
	ROM_LOAD16_BYTE( "95400436.hi", 0x00000, 0x080000, CRC(7017062c) SHA1(1ed616fb44fdf689baaede095b3c59e4bab0face) )
	sc_mbags_others
ROM_END

ROM_START( sc4mbagsa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400437.lo", 0x00001, 0x080000, CRC(06e2ab4c) SHA1(7795e7391be900e8f0dc6cf2ed6456898ec6e07a) )
	ROM_LOAD16_BYTE( "95400438.hi", 0x00000, 0x080000, CRC(32dda7ce) SHA1(b30ba112e3a734b01cc6d4424df3844b408d5a6a) )
	sc_mbags_others
ROM_END

ROM_START( sc4mbagsb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401435.lo", 0x00001, 0x080000, CRC(3652929f) SHA1(542413554aa80749308e6d6e4f1a133390be5a9a) )
	ROM_LOAD16_BYTE( "95401436.hi", 0x00000, 0x080000, CRC(70885014) SHA1(4bea5236ed7190c136893fd106f7b9d1c9b7fc3f) )
	sc_mbags_others
ROM_END

ROM_START( sc4mbagsc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401437.lo", 0x00001, 0x080000, CRC(19cf5563) SHA1(1d0ec3beeb242df8cd5af05f65ff4d10986507e3) )
	ROM_LOAD16_BYTE( "95401438.hi", 0x00000, 0x080000, CRC(5e5dd1ce) SHA1(65331052bd3583591bf10762d2f48ce16d7a152f) )
	sc_mbags_others
ROM_END




ROM_START( sc4mgr )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406738.lo", 0x00001, 0x080000, CRC(91e156b8) SHA1(3439f79e68c445960cf41b1f8a2b60e166e28024) )
	ROM_LOAD16_BYTE( "95406739.hi", 0x00000, 0x080000, CRC(2e2ba90c) SHA1(cc09e392dbba2f75ed72fbff8c0ec6e47922cc0b) )
	PR1132_MONEY_GO_ROUND_SOUNDS11
ROM_END

ROM_START( sc4mgra )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407738.lo", 0x00001, 0x080000, CRC(6a1a091d) SHA1(93371ab90612ae6000d3cc6c3ddca8f90db803db) )
	ROM_LOAD16_BYTE( "95407739.hi", 0x00000, 0x080000, CRC(cf883e87) SHA1(74edbaff54944f2a1ed6e7d99bbd9047dbd9fb4b) )
	PR1132_MONEY_GO_ROUND_SOUNDS11
ROM_END

ROM_START( sc4mgrh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1132p72.hi", 0x00000, 0x080000, CRC(9cbaee34) SHA1(1d5713a4cd1f74c97b0879c2850fa994639f2370) )
	ROM_LOAD16_BYTE( "p1132p72.lo", 0x00001, 0x080000, CRC(fb327c69) SHA1(b799290d2e4d0f071e7d97ab4817d8fdc14f9c2c) )
	PR1132_MONEY_GO_ROUND_SOUNDS11
ROM_END

ROM_START( sc4mgri )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1132s72.hi", 0x00000, 0x080000, CRC(06f9462c) SHA1(865c4cab26b9265e0da1e53bdbd9fa1df1e1ea34) )
	ROM_LOAD16_BYTE( "p1132s72.lo", 0x00001, 0x080000, CRC(3392c0b4) SHA1(231cd4ad13cb71d44e1bb0d6b4eb22ecd5240d8c) )
	PR1132_MONEY_GO_ROUND_SOUNDS11
ROM_END

ROM_START( sc4mgrb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410592.lo", 0x00001, 0x080000, CRC(578476ff) SHA1(ca115ed9a04efea369de6dbc7e0c999b0276e24e) )
	ROM_LOAD16_BYTE( "95410593.hi", 0x00000, 0x080000, CRC(63bc7162) SHA1(15629023e84ae7a1b19201d24c0f74edd45c05e3) )
	PR1132_MONEY_GO_ROUND_SOUNDS11_ALT
ROM_END

ROM_START( sc4mgrc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410594.lo", 0x00001, 0x080000, CRC(5d5ef370) SHA1(46253f4e6ac2b226b9cfb0c318dc3537700ab3db) )
	ROM_LOAD16_BYTE( "95410595.hi", 0x00000, 0x080000, CRC(b24010eb) SHA1(ea3ea23665a5b9dcc5f861aad720313f832ee248) )
	PR1132_MONEY_GO_ROUND_SOUNDS11_ALT
ROM_END

ROM_START( sc4mgrd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411592.lo", 0x00001, 0x080000, CRC(d7865d1f) SHA1(203ef7d6ed1babd74c2d5dcc497347fa13a4c909) )
	ROM_LOAD16_BYTE( "95411593.hi", 0x00000, 0x080000, CRC(57ceb99e) SHA1(6997e1e3dbe389d990c97eebd7397546e33bfb73) )
	PR1132_MONEY_GO_ROUND_SOUNDS11_ALT
ROM_END

ROM_START( sc4mgre )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411594.lo", 0x00001, 0x080000, CRC(34ec80ae) SHA1(61373d443505f7656e56d1f0fe0ffdc61153db47) )
	ROM_LOAD16_BYTE( "95411595.hi", 0x00000, 0x080000, CRC(78f7e5f8) SHA1(4993018f22d7223b1b0e1ed11899564298d666e6) )
	PR1132_MONEY_GO_ROUND_SOUNDS11_ALT
ROM_END

ROM_START( sc4mgrf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412214.lo", 0x00001, 0x080000, CRC(569b2591) SHA1(0e5c1e14e9b8de49e02a1efc02f50b6d632a2fa5) )
	ROM_LOAD16_BYTE( "95412215.hi", 0x00000, 0x080000, CRC(cff9aefa) SHA1(c8a973429cec99010a4886ad88cde820ecbdf435) )
	PR1132_MONEY_GO_ROUND_SOUNDS11_ALT
ROM_END

ROM_START( sc4mgrg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413214.lo", 0x00001, 0x080000, CRC(645e63ff) SHA1(78e7e7a210eed5e8086483f7e8dac2a8f2cc4b91) )
	ROM_LOAD16_BYTE( "95413215.hi", 0x00000, 0x080000, CRC(1ce43623) SHA1(9748d4df64f4282a141ca2907630badfdeb5c3bd) )
	PR1132_MONEY_GO_ROUND_SOUNDS11_ALT
ROM_END



ROM_START( sc4mgrj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95412804.lo", 0x00001, 0x080000, CRC(0948aa27) SHA1(14a779bbe176e525b53ce9c7a0ac46d043afb47d) )
	ROM_LOAD16_BYTE( "95412805.hi", 0x00000, 0x080000, CRC(5e7bf5e1) SHA1(f03880f80a154c408717c8d53121d1bde2e62ab5) )
	ROM_REGION( 0x400000, "ymz", ROMREGION_ERASE00)
ROM_END


ROM_START( sc4mgrk )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95412808.lo", 0x00001, 0x080000, CRC(e500f4cf) SHA1(3d0738ab1b927887f257fe063b57ee4a76be6808) )
	ROM_LOAD16_BYTE( "95412809.hi", 0x00000, 0x080000, CRC(4556db34) SHA1(a06dfa1a2389f0ae185b6cc520b7dca75dd9dd6a) )
	ROM_REGION( 0x400000, "ymz", ROMREGION_ERASE00)
ROM_END

ROM_START( sc4mgrl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95413804.lo", 0x00001, 0x080000, CRC(9c1261ca) SHA1(50cce10f8736e982346aba313abd2e761d7286fc) )
	ROM_LOAD16_BYTE( "95413805.hi", 0x00000, 0x080000, CRC(ae77f9cf) SHA1(e88d82e8711b81dce306db558c03141699b2a813) )
	ROM_REGION( 0x400000, "ymz", ROMREGION_ERASE00)
ROM_END


ROM_START( sc4mgrm )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95413808.lo", 0x00001, 0x080000, CRC(80630b04) SHA1(9393d9be92aca4eb4f4a774868fe8b6698139b67) )
	ROM_LOAD16_BYTE( "95413809.hi", 0x00000, 0x080000, CRC(cf0a0b46) SHA1(434acbbe97748b96853f288b14cebcce02151515) )
	ROM_REGION( 0x400000, "ymz", ROMREGION_ERASE00)
ROM_END



ROM_START( sc4mspid )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406748.lo", 0x00001, 0x080000, CRC(6c058276) SHA1(ffbb2a1e304a127fa4549f9ea9e59bfae2c5e094) )
	ROM_LOAD16_BYTE( "95406749.hi", 0x00000, 0x080000, CRC(f2bfbb44) SHA1(3c5edc37ebf81fcff8fce1c6bc0ad1f7e067c46b) )
	sc_mspid_others
ROM_END

ROM_START( sc4mspida )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406750.lo", 0x00001, 0x080000, CRC(83872be6) SHA1(6154db41a864e0e07b07a912db102f971118bd55) )
	ROM_LOAD16_BYTE( "95406751.hi", 0x00000, 0x080000, CRC(0db3369f) SHA1(69b1d8fe06d7629addf6c8ce0da9543795deea0b) )
	sc_mspid_others
ROM_END

ROM_START( sc4mspidb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406878.lo", 0x00001, 0x080000, CRC(74b21587) SHA1(e43577582ad3437574b74fd3be532f6394c5dfd6) )
	ROM_LOAD16_BYTE( "95406879.hi", 0x00000, 0x080000, CRC(4b536557) SHA1(9e610a433ea86480566ddfe61cfff539fee37da3) )
	sc_mspid_others
ROM_END

ROM_START( sc4mspidc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406880.lo", 0x00001, 0x080000, CRC(dc591daa) SHA1(e886ced9a0a8e4e85b49c28497ffa51b09540a51) )
	ROM_LOAD16_BYTE( "95406881.hi", 0x00000, 0x080000, CRC(8c2a3da5) SHA1(9baa87cd559834547b8742f0af8f721d49388825) )
	sc_mspid_others
ROM_END

ROM_START( sc4mspidd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407748.lo", 0x00001, 0x080000, CRC(28f00d53) SHA1(e2e06f70cf4cb689ceba0dae336f249fd181b1b8) )
	ROM_LOAD16_BYTE( "95407749.hi", 0x00000, 0x080000, CRC(a90161cd) SHA1(e368abd3cf5dfc73bdf064054ca51b1c91ac3a1d) )
	sc_mspid_others
ROM_END

ROM_START( sc4mspide )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407750.lo", 0x00001, 0x080000, CRC(6ff48c21) SHA1(1d075660c011718b287d3480f18a6c90d6f63cd8) )
	ROM_LOAD16_BYTE( "95407751.hi", 0x00000, 0x080000, CRC(dcbf02d3) SHA1(366234b594566440736e46a04928928d5af398f9) )
	sc_mspid_others
ROM_END

ROM_START( sc4mspidf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407878.lo", 0x00001, 0x080000, CRC(c82a2b79) SHA1(bd92b6da3c5775e0b4227047d32d42ce263119a9) )
	ROM_LOAD16_BYTE( "95407879.hi", 0x00000, 0x080000, CRC(781c5db8) SHA1(19369ebb7d2e30a56ce604634f3ac793020d6f71) )
	sc_mspid_others
ROM_END

ROM_START( sc4mspidg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407880.lo", 0x00001, 0x080000, CRC(3a123bca) SHA1(40fc796068b7ca8ab0785a22551b6482bd7c14c1) )
	ROM_LOAD16_BYTE( "95407881.hi", 0x00000, 0x080000, CRC(906b8303) SHA1(56a6a5be5f2e5e3fc883ed01e33859121ea10eda) )
	sc_mspid_others
ROM_END

ROM_START( sc4mspidh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412216.lo", 0x00001, 0x080000, CRC(a356b7d9) SHA1(e9926885ffd144c8ac76653f42409dd1e15d2c3b) )
	ROM_LOAD16_BYTE( "95412217.hi", 0x00000, 0x080000, CRC(e309001c) SHA1(44ac7d6a510f926b8b23f46619712b77d48ac308) )
	sc_mspid_others
ROM_END

ROM_START( sc4mspidi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413216.lo", 0x00001, 0x080000, CRC(c01ebf68) SHA1(d0a719436e713272d580fd2feb73dfc33b764a55) )
	ROM_LOAD16_BYTE( "95413217.hi", 0x00000, 0x080000, CRC(96bc4dee) SHA1(1c51cd7ddeef79af384cb0f86dc43bce7315fa27) )
	sc_mspid_others
ROM_END

ROM_START( sc4mspidj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1325p61.hi", 0x00000, 0x080000, CRC(6208c47a) SHA1(7748316b68aa9c3716fc3cf7b220b8b505e55e90) )
	ROM_LOAD16_BYTE( "p1325p61.lo", 0x00001, 0x080000, CRC(ab3bfc9e) SHA1(219e0f666f09552f392619a671c46d20d121355f) )
	sc_mspid_others
ROM_END

ROM_START( sc4mspidk )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1325p71.hi", 0x00000, 0x080000, CRC(ea9bfc91) SHA1(aca22c2136df717be6a4fb656fb0a88cced91216) )
	ROM_LOAD16_BYTE( "p1325p71.lo", 0x00001, 0x080000, CRC(cebb2b61) SHA1(0b2df3671e8fbc7733e7de100aab3edc5ed82bb8) )
	sc_mspid_others
ROM_END

ROM_START( sc4mspidl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1325s61.hi", 0x00000, 0x080000, CRC(8c5b91e2) SHA1(6706296c25fab574139b51b824bf6e8453cd95aa) )
	ROM_LOAD16_BYTE( "p1325s61.lo", 0x00001, 0x080000, CRC(09388780) SHA1(7045d6f31bc6a7fe7b23903faef417116ee7a90a) )
	sc_mspid_others
ROM_END

ROM_START( sc4mspidm )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1325s71.hi", 0x00000, 0x080000, CRC(04c8a909) SHA1(84cbe5e7294f181a845b0aa0f487b8080710df60) )
	ROM_LOAD16_BYTE( "p1325s71.lo", 0x00001, 0x080000, CRC(6cb8507f) SHA1(cdabdd98a658b12b88e2d71d7582ac51c798f301) )
	sc_mspid_others
ROM_END









ROM_START( sc4msclb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404560.lo", 0x00001, 0x080000, CRC(f592f85b) SHA1(74b93613c529c1a1c7d073099485d306d716b6da) )
	ROM_LOAD16_BYTE( "95404561.hi", 0x00000, 0x080000, CRC(0e2737a7) SHA1(8c8b8862cd557ed1633e50908e868c2fac620a6d) )
	sc_msclb_others
ROM_END

ROM_START( sc4msclba )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404562.lo", 0x00001, 0x080000, CRC(ec68d4e7) SHA1(de18a505066909e0e9ea3e130a1fc4c58499e101) )
	ROM_LOAD16_BYTE( "95404563.hi", 0x00000, 0x080000, CRC(38e618ea) SHA1(7a98588bb3ccbf16484a4a2dc1c7dfebb9d2398a) )
	sc_msclb_others
ROM_END

ROM_START( sc4msclbb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404694.lo", 0x00001, 0x080000, CRC(dc31e3e8) SHA1(8c8810395079c75535cb9ae7055d079092a78848) )
	ROM_LOAD16_BYTE( "95404695.hi", 0x00000, 0x080000, CRC(533e6c92) SHA1(a732cdeefb8892eee1acd673d758140829c82549) )
	sc_msclb_others
ROM_END

ROM_START( sc4msclbc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405560.lo", 0x00001, 0x080000, CRC(5fb17d06) SHA1(fd090f0d8353482379148c42c57d87204962b11e) )
	ROM_LOAD16_BYTE( "95405561.hi", 0x00000, 0x080000, CRC(3e95fa7b) SHA1(45f04deaecd11ca747bb78b983da2e7e63536b1b) )
	sc_msclb_others
ROM_END

ROM_START( sc4msclbd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405562.lo", 0x00001, 0x080000, CRC(287268ca) SHA1(e20e45598ffb08ba637b6230599e289a6274c22a) )
	ROM_LOAD16_BYTE( "95405563.hi", 0x00000, 0x080000, CRC(2290d788) SHA1(3e4f862dd03e1fd17e72a7c46e4879581f9137f7) )
	sc_msclb_others
ROM_END

ROM_START( sc4msclbe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405694.lo", 0x00001, 0x080000, CRC(fc7109a3) SHA1(a8a2afd761884dd23a52a575803238eca7b209df) )
	ROM_LOAD16_BYTE( "95405695.hi", 0x00000, 0x080000, CRC(cb684ff5) SHA1(925f9b87c535c6a057bbfd99f7afc252b0b86f52) )
	sc_msclb_others
ROM_END

ROM_START( sc4msclbf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406582.lo", 0x00001, 0x080000, CRC(30df16ef) SHA1(9efbaa66fa555b6031b3c1333f6ae424a2eb7a86) )
	ROM_LOAD16_BYTE( "95406583.hi", 0x00000, 0x080000, CRC(ac0ebd13) SHA1(aabca87b9fa5101cf1ffbbd7eb876c36dcd64a9c) )
	sc_msclb_others
ROM_END

ROM_START( sc4msclbg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407582.lo", 0x00001, 0x080000, CRC(4ed1ba89) SHA1(888af17d5ed0639bae85c6b556e9d34246ffbfb9) )
	ROM_LOAD16_BYTE( "95407583.hi", 0x00000, 0x080000, CRC(94217174) SHA1(2d61ec0c48aa1ba5eceddd22f92a0399e430f406) )
	sc_msclb_others
ROM_END




ROM_START( sc4mtb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400129.lo", 0x00001, 0x080000, CRC(62b2b663) SHA1(afc783cb7cfef9647d0c23a6f3fb0740e551bee8) )
	ROM_LOAD16_BYTE( "95400130.hi", 0x00000, 0x080000, CRC(beef2251) SHA1(e8804cde6fb887674179cd5c97f50f6d098ec31e) )
	sc_mtb_others
ROM_END

ROM_START( sc4mtba )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400245.lo", 0x00001, 0x080000, CRC(4cae99b6) SHA1(193cba2cda19325e0badd1646d96f737880117fb) )
	ROM_LOAD16_BYTE( "95400246.hi", 0x00000, 0x080000, CRC(cfb8885d) SHA1(678e68e0e05d5e7727ec89406e99d431ca700e78) )
	sc_mtb_others
ROM_END

ROM_START( sc4mtbb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400317.lo", 0x00001, 0x080000, CRC(8a0cb5e6) SHA1(09c79d72a58ce479ff37dfdb6ea496a061aec475) )
	ROM_LOAD16_BYTE( "95400318.hi", 0x00000, 0x080000, CRC(193d7b0b) SHA1(3ac2c3496b65b3be2867440e81a37c95a0fca79c) )
	sc_mtb_others
ROM_END

ROM_START( sc4mtbc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400939.lo", 0x00001, 0x080000, CRC(258ad3b3) SHA1(116a9287e3c31ac2ec43752b3c42578bf1c4446b) )
	ROM_LOAD16_BYTE( "95400940.hi", 0x00000, 0x080000, CRC(0ffa3f82) SHA1(01815f6216494cfb3aaee37cdaec63355ec6f329) )
	sc_mtb_others
ROM_END

ROM_START( sc4mtbd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400941.lo", 0x00001, 0x080000, CRC(7ae8e888) SHA1(ee56614a69687b653338487b9aac1a60a00aff48) )
	ROM_LOAD16_BYTE( "95400942.hi", 0x00000, 0x080000, CRC(72d9adc2) SHA1(5e5b10be4a081471b55b82a25a3a6660e3a2880e) )
	sc_mtb_others
ROM_END

ROM_START( sc4mtbe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401129.lo", 0x00001, 0x080000, CRC(5b8f1193) SHA1(58ada999ac499093db2f123006c89c5993e99567) )
	ROM_LOAD16_BYTE( "95401130.hi", 0x00000, 0x080000, CRC(1b0403ea) SHA1(93319e4a900fccdc0f93a6583d8676a7ab188664) )
	sc_mtb_others
ROM_END

ROM_START( sc4mtbf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401245.lo", 0x00001, 0x080000, CRC(f656ba7c) SHA1(9a47388c13420f917e0790efcfb2b3a939a27c53) )
	ROM_LOAD16_BYTE( "95401246.hi", 0x00000, 0x080000, CRC(c3bd439c) SHA1(f64d51f8c23688f11ffdad64cb697433252c87c3) )
	sc_mtb_others
ROM_END

ROM_START( sc4mtbg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401317.lo", 0x00001, 0x080000, CRC(825724b7) SHA1(a467956f63e72f88e13ba95a7af74e6b222d016f) )
	ROM_LOAD16_BYTE( "95401318.hi", 0x00000, 0x080000, CRC(2e0c4767) SHA1(95e272235efc2a96ab7a337e91727e240222c4d4) )
	sc_mtb_others
ROM_END

ROM_START( sc4mtbh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401939.lo", 0x00001, 0x080000, CRC(ff226fb7) SHA1(63410d9f141106af7c1ad11a0ebbe1be026fad6e) )
	ROM_LOAD16_BYTE( "95401940.hi", 0x00000, 0x080000, CRC(9dbf746d) SHA1(1b1c9f3a96401d6c204be7cbe37278f7d792d46e) )
	sc_mtb_others
ROM_END

ROM_START( sc4mtbi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "mtb15hi", 0x00000, 0x080000, CRC(24651e35) SHA1(33c03b791bbb123e53bf3789cfadae425a02eaa4) )
	ROM_LOAD16_BYTE( "mtb15lo", 0x00001, 0x080000, CRC(5aa9718a) SHA1(98b2fe64114b33fa30e3276b9da5959b43abb626) )
	sc_mtb_others
ROM_END

ROM_START( sc4mtbj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "mtbhi", 0x00000, 0x080000, CRC(c1e7fb98) SHA1(edeb4c90581127426da9d7b76642735e55d11cb3) )
	ROM_LOAD16_BYTE( "mtblo", 0x00001, 0x080000, CRC(95cd0534) SHA1(8dabf8d4dbdb029f2ed951bf22bc8bdc5197fcee) )
	sc_mtb_others
ROM_END




ROM_START( sc4mtbcl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400897.lo", 0x00001, 0x080000, CRC(3a39861b) SHA1(e9e8b85e7bb1611c2231d60b869bcc2b6d866c2a) )
	ROM_LOAD16_BYTE( "95400898.hi", 0x00000, 0x080000, CRC(8b17b182) SHA1(6c0cc15b8073363eeab12f6d886bebd0c110378b) )
	sc_mtbcl_others
ROM_END

ROM_START( sc4mtbcla )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400949.lo", 0x00001, 0x080000, CRC(7e55715c) SHA1(190b6d67960daaee5e04f656de794a558b61f76d) )
	ROM_LOAD16_BYTE( "95400950.hi", 0x00000, 0x080000, CRC(ceeaf667) SHA1(d5bf6f7461aaa8267f57cc490680a1609f377a55) )
	sc_mtbcl_others
ROM_END

ROM_START( sc4mtbclb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400971.lo", 0x00001, 0x080000, CRC(d73de9fe) SHA1(2d4785d93ab5d6fdfb813ab4589e3a99584447de) )
	ROM_LOAD16_BYTE( "95400972.hi", 0x00000, 0x080000, CRC(190734b4) SHA1(8afb11c904f9cd6d9fff9ed8837d0ff6b569ebff) )
	sc_mtbcl_others
ROM_END

ROM_START( sc4mtbclc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400979.lo", 0x00001, 0x080000, CRC(1ae05217) SHA1(84b1fdc2ba7fa3978bbba0558017d7c193fd2263) )
	ROM_LOAD16_BYTE( "95400980.hi", 0x00000, 0x080000, CRC(e195e22b) SHA1(28104aec1f4c7525c01c379c63de91fc458627db) )
	sc_mtbcl_others
ROM_END

ROM_START( sc4mtbcld )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401897.lo", 0x00001, 0x080000, CRC(e642920b) SHA1(c24995e6a08114d2e1cb594654623916aa0f8983) )
	ROM_LOAD16_BYTE( "95401898.hi", 0x00000, 0x080000, CRC(8b88e7ba) SHA1(469d94858edeab75393c3fef2a7cd74022cc47b7) )
	sc_mtbcl_others
ROM_END

ROM_START( sc4mtbcle )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401949.lo", 0x00001, 0x080000, CRC(13b00b79) SHA1(22c2822c26e8979e2a38de95ee294a0173afd66b) )
	ROM_LOAD16_BYTE( "95401950.hi", 0x00000, 0x080000, CRC(bb5fbb95) SHA1(245a032654d937b7aca29e05e8c0eb973706eb5f) )
	sc_mtbcl_others
ROM_END

ROM_START( sc4mtbclf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401971.lo", 0x00001, 0x080000, CRC(90b70d6c) SHA1(40c9e16010584d1183286f8d8c497e0569ca8cb4) )
	ROM_LOAD16_BYTE( "95401972.hi", 0x00000, 0x080000, CRC(6ff18dd6) SHA1(9aaf3b15f95603ae1b6f8a4459dfb3e200e866af) )
	sc_mtbcl_others
ROM_END

ROM_START( sc4mtbclg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401979.lo", 0x00001, 0x080000, CRC(0f8a8828) SHA1(b3a7479549a76784d67c528f9d2bd8f2b0f4cccb) )
	ROM_LOAD16_BYTE( "95401980.hi", 0x00000, 0x080000, CRC(1266833c) SHA1(7d6a18bac50b81a9f2b727005887141bdef46e29) )
	sc_mtbcl_others
ROM_END

ROM_START( sc4mtbclh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "club-money-to-burn_dat_ac_1000pnd-50p=1=ss.hi", 0x00000, 0x080000, CRC(6baafc2d) SHA1(fb775e0d431a2c31b3f6c565cddaad5d22d9d122) )
	ROM_LOAD16_BYTE( "club-money-to-burn_dat_ac_1000pnd-50p=1=ss.lo", 0x00001, 0x080000, CRC(179f0787) SHA1(6815167bee95e88cf628128a88065a66983a7489) )
	sc_mtbcl_others
ROM_END

ROM_START( sc4mtbcli )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "club-money-to-burn_dat_ac_1000pnd-50p=2=ss.hi", 0x00000, 0x080000, CRC(3539277b) SHA1(2a602de4a3b3d6336e4f0be15b9bd3d9e730c571) )
	ROM_LOAD16_BYTE( "club-money-to-burn_dat_ac_1000pnd-50p=2=ss.lo", 0x00001, 0x080000, CRC(9cc4bf96) SHA1(db0cd119346b5ffce1bdf19e595f085bef93747d) )
	sc_mtbcl_others
ROM_END

ROM_START( sc4mtbclj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "club-money-to-burn_dat_ac_1000pnd-50p=3=ss.hi", 0x00000, 0x080000, CRC(82d9d864) SHA1(02f6fe87364d5afece5a7289ac74bd66345b53ea) )
	ROM_LOAD16_BYTE( "club-money-to-burn_dat_ac_1000pnd-50p=3=ss.lo", 0x00001, 0x080000, CRC(902c1902) SHA1(15e00073cb818af46a37357304d2a8ef7f256f09) )
	sc_mtbcl_others
ROM_END

ROM_START( sc4mtbclk )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "club-money-to-burn_dat_ac_1000pnd-50p=4=ss.hi", 0x00000, 0x080000, CRC(75a15cea) SHA1(d34e20273c1ed7b7a11100a68462775930f7c39f) )
	ROM_LOAD16_BYTE( "club-money-to-burn_dat_ac_1000pnd-50p=4=ss.lo", 0x00001, 0x080000, CRC(0f5d2f5b) SHA1(725c1cf317cac74a6f0e0185a46ff0c7c399432d) )
	sc_mtbcl_others
ROM_END

ROM_START( sc4mtbcll )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "club-money-to-burn_std_ac_1000pnd-50p=1=ss.hi", 0x00000, 0x080000, CRC(d596e8bc) SHA1(7fbf3bac2c96034714ce3b83077f737c5bd93afc) )
	ROM_LOAD16_BYTE( "club-money-to-burn_std_ac_1000pnd-50p=1=ss.lo", 0x00001, 0x080000, CRC(a4ea7d69) SHA1(ae949626d8784570145bc2e51b6389454ead73f8) )
	sc_mtbcl_others
ROM_END

ROM_START( sc4mtbclm )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "club-money-to-burn_std_ac_1000pnd-50p=2=ss.hi", 0x00000, 0x080000, CRC(decceaa7) SHA1(1e13bf0eb30280932028f0b52cbf45e461f5ab5e) )
	ROM_LOAD16_BYTE( "club-money-to-burn_std_ac_1000pnd-50p=2=ss.lo", 0x00001, 0x080000, CRC(b57323fd) SHA1(b9a43b16ff347cf3d87a064468c0f507c87ef819) )
	sc_mtbcl_others
ROM_END

ROM_START( sc4mtbcln )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "club-money-to-burn_std_ac_1000pnd-50p=3=ss.hi", 0x00000, 0x080000, CRC(64882431) SHA1(21184508e6e2878fa5d8b300ad5f9ef7ab75c223) )
	ROM_LOAD16_BYTE( "club-money-to-burn_std_ac_1000pnd-50p=3=ss.lo", 0x00001, 0x080000, CRC(8ab940ad) SHA1(a4170464e3d08f0a925ab6d302f0d0a7ab03ed94) )
	sc_mtbcl_others
ROM_END

ROM_START( sc4mtbclo )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "club-money-to-burn_std_ac_1000pnd-50p=4=ss.hi", 0x00000, 0x080000, CRC(df931438) SHA1(3fd98e4a3f79b84576c33481296a878dca56afce) )
	ROM_LOAD16_BYTE( "club-money-to-burn_std_ac_1000pnd-50p=4=ss.lo", 0x00001, 0x080000, CRC(dadd9222) SHA1(679a87c43b42662e99ad4307b49b5f3a12e8b28d) )
	sc_mtbcl_others
ROM_END




ROM_START( sc4m2m )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400661.lo", 0x00001, 0x080000, CRC(6775b569) SHA1(59f65a1ef40185bfe85544a6f9a7236bdfcb4955) )
	ROM_LOAD16_BYTE( "95400662.hi", 0x00000, 0x080000, CRC(fbd04554) SHA1(03a5ed0d1caecfe477c02dbe48fd18ac3114dadd) )
	sc_m2m_others
ROM_END

ROM_START( sc4m2ma )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401661.lo", 0x00001, 0x080000, CRC(37f46275) SHA1(cbcfc7d903cb5bc58ed3b98552fec1adba35c130) )
	ROM_LOAD16_BYTE( "95401662.hi", 0x00000, 0x080000, CRC(262a1894) SHA1(7efd6a6f621c253238c323920799ed7de9bda908) )
	sc_m2m_others
ROM_END




ROM_START( sc4monob )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408402.lo", 0x00001, 0x080000, CRC(c1da598d) SHA1(1d27d6bfc253d632154ffe9c7946ad5d8d9213e6) )
	ROM_LOAD16_BYTE( "95408403.hi", 0x00000, 0x080000, CRC(0d8342eb) SHA1(67887e7481b530316ea691658dd173b88ce7cfd8) )
	sc_monob_others
ROM_END

ROM_START( sc4monoba )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408418.lo", 0x00001, 0x080000, CRC(25ea7673) SHA1(14ff3d2a48d66f8d68160d4b3272ae0f13038112) )
	ROM_LOAD16_BYTE( "95408419.hi", 0x00000, 0x080000, CRC(cb7061d5) SHA1(a9460217b961cf557ca0065cd5d52f912c3439a5) )
	sc_monob_others
ROM_END

ROM_START( sc4monobb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408984.lo", 0x00001, 0x080000, CRC(e16bb9bf) SHA1(f3b9fa43bcc10ae632879aa5cae354826410796b) )
	ROM_LOAD16_BYTE( "95408985.hi", 0x00000, 0x080000, CRC(726da105) SHA1(c1f3788408bcf279b94e24df445575de15fc2eda) )
	sc_monob_others
ROM_END

ROM_START( sc4monobc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "monopoly_95408986_std_low.bin",  0x00001, 0x080000, CRC(6be0db3e) SHA1(2771294694e61219e23fc0d5bed3a5d9de500245) )
	ROM_LOAD16_BYTE( "monopoly_95408987_std_high.bin", 0x00000, 0x080000, CRC(66e68347) SHA1(e1fb1b92ac80023dc03467a7a6a0b76edccdbb8c) )
	sc_monob_others
ROM_END

ROM_START( sc4monobd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408988.lo", 0x00001, 0x080000, CRC(463d4019) SHA1(8f4a7918bfb57b29ec0ef7cba581de679a644b12) )
	ROM_LOAD16_BYTE( "95408989.hi", 0x00000, 0x080000, CRC(96746590) SHA1(5adac78a0abd49daec5f45f0dd67336a748cba2a) )
	sc_monob_others
ROM_END

ROM_START( sc4monobe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409402.lo", 0x00001, 0x080000, CRC(c0e526b3) SHA1(9abcc41a19ae04a9955a69a601208170607418d4) )
	ROM_LOAD16_BYTE( "95409403.hi", 0x00000, 0x080000, CRC(5112c591) SHA1(cdb7e621440eeb941807cb074fc9b61e8a747fcb) )
	sc_monob_others
ROM_END

ROM_START( sc4monobf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409418.lo", 0x00001, 0x080000, CRC(1d7b2ed6) SHA1(f2a8793c00ca18dfe9762f12cfdf24e1d212f4f2) )
	ROM_LOAD16_BYTE( "95409419.hi", 0x00000, 0x080000, CRC(4724a186) SHA1(8292d704770eba4da5b838ee32ebef7aff064901) )
	sc_monob_others
ROM_END

ROM_START( sc4monobg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409984.lo", 0x00001, 0x080000, CRC(7cf821df) SHA1(b911265f797c0246e4dd2ea4d876a5ace25913c2) )
	ROM_LOAD16_BYTE( "95409985.hi", 0x00000, 0x080000, CRC(99ac44d3) SHA1(8cf10839b536a190230cce7eb6bc45bc770fdf08) )
	sc_monob_others
ROM_END

ROM_START( sc4monobh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409986.lo", 0x00001, 0x080000, CRC(53783f19) SHA1(027b475755d71fa2f0010a80fb7e1669a6e8bef1) )
	ROM_LOAD16_BYTE( "95409987.hi", 0x00000, 0x080000, CRC(e0e3d8e5) SHA1(16438e7a7198bc0a85c1fa776f1da752fabbee96) )
	sc_monob_others
ROM_END

ROM_START( sc4monobi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409988.lo", 0x00001, 0x080000, CRC(4a8ee93a) SHA1(41bcc485bbfe2b708666120b8ae0eea5200c2f5c) )
	ROM_LOAD16_BYTE( "95409989.hi", 0x00000, 0x080000, CRC(5f8472bf) SHA1(d91602ceb62b57129ad979cda0f56f652da1e650) )
	sc_monob_others
ROM_END

ROM_START( sc4monobj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410490.lo", 0x00001, 0x080000, CRC(d20f7858) SHA1(7342489ab73bada6cd8ecc28d6229016a882a20a) )
	ROM_LOAD16_BYTE( "95410491.hi", 0x00000, 0x080000, CRC(99334dc1) SHA1(923b2019430c4a1a92dfbd8a5c63565d8b15d19c) )
	sc_monob_others
ROM_END

ROM_START( sc4monobk )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410492.lo", 0x00001, 0x080000, CRC(fb5f8442) SHA1(428da1f86dd25f6bb8dfa717a9c81e5dbd6ee6c9) )
	ROM_LOAD16_BYTE( "95410493.hi", 0x00000, 0x080000, CRC(6ce25c4e) SHA1(ef51417b20e19bbeb412db9521ff97806d09439a) )
	sc_monob_others
ROM_END

ROM_START( sc4monobl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411490.lo", 0x00001, 0x080000, CRC(4e3abd7d) SHA1(a20e7196999977ae5c94b53f37666c08e645b61d) )
	ROM_LOAD16_BYTE( "95411491.hi", 0x00000, 0x080000, CRC(ee640eeb) SHA1(ef62e86d3f0dd73e666752f951dd51d1348d03d9) )
	sc_monob_others
ROM_END

ROM_START( sc4monobm )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411492.lo", 0x00001, 0x080000, CRC(1a742a79) SHA1(d30aa680edddb4e5cabf5db054aa2a3b3c5b7b90) )
	ROM_LOAD16_BYTE( "95411493.hi", 0x00000, 0x080000, CRC(23c21c59) SHA1(84c71bff621fac7ce6440e7d24e6e8be8682556c) )
	sc_monob_others
ROM_END




ROM_START( sc4monod )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406090.lo", 0x00001, 0x080000, CRC(cdbb460a) SHA1(cb971677888b3f9c54a784f3a69f93a9a94a488d) )
	ROM_LOAD16_BYTE( "95406091.hi", 0x00000, 0x080000, CRC(8aac1d84) SHA1(fbb7f6467a79f6a55c488621fe6efe981584c5a4) )
	sc_monod_others
ROM_END

ROM_START( sc4monoda )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408334.lo", 0x00001, 0x080000, CRC(a27c87d6) SHA1(4f5b56cd925f6d1fbcd58697dd17370d14e61d53) )
	ROM_LOAD16_BYTE( "95408335.hi", 0x00000, 0x080000, CRC(48564e37) SHA1(1658603b38ec1e94968c036f061f649cb90c1522) )
	sc_monod_others
ROM_END

ROM_START( sc4monodb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408512.lo", 0x00001, 0x080000, CRC(86a22467) SHA1(3bcb64029b95fc34d014067281e139830fa165ef) )
	ROM_LOAD16_BYTE( "95408513.hi", 0x00000, 0x080000, CRC(c86ea35a) SHA1(d2265d556c884919d1fe400e34595aea96ef4f70) )
	sc_monod_others
ROM_END









ROM_START( sc4mclb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416076.lo", 0x00001, 0x080000, CRC(5e535679) SHA1(c187e0862166d3941abb899e25f50952a7a3007b) )
	ROM_LOAD16_BYTE( "95416077.hi", 0x00000, 0x080000, CRC(59fd4d20) SHA1(f1db12abe6fea293961150162a33af3abb759e71) )
	sc_mclb_others
ROM_END

ROM_START( sc4mclba )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416200.lo", 0x00001, 0x080000, CRC(c352ef5c) SHA1(de3c4c30733bbfbe40c71663b80d9978d6c91e4f) )
	ROM_LOAD16_BYTE( "95416201.hi", 0x00000, 0x080000, CRC(9fced013) SHA1(356666bed633da24c8eb6cb5df20fd5dd3eee41f) )
	sc_mclb_others
ROM_END

ROM_START( sc4mclbb )
/* Deleted set 100% same roms was called:  sc4mwwcl (Monopoly Wonders Of The World Club (Mazooma) (Scorpion 4)) */
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416422.lo", 0x00001, 0x080000, CRC(42d8151e) SHA1(c008ce3d50026bfc904d2c79e61e76c4d9c4bd63) )
	ROM_LOAD16_BYTE( "95416423.hi", 0x00000, 0x080000, CRC(6f45ab49) SHA1(2b0686008f5e666a679c2af4e0c167d8a9af5444) )
	sc_mclb_others
ROM_END

ROM_START( sc4mclbc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417076.lo", 0x00001, 0x080000, CRC(0def1790) SHA1(08b52da7573be42b123b1dc9dbb1ba5424b49ff0) )
	ROM_LOAD16_BYTE( "95417077.hi", 0x00000, 0x080000, CRC(889d8d19) SHA1(738509fbf69bf85c6aa6a91723d3414612d28121) )
	sc_mclb_others
ROM_END

ROM_START( sc4mclbd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417200.lo", 0x00001, 0x080000, CRC(aa9b06c0) SHA1(435426c3da265b1d23e89b9844e323c1791e6955) )
	ROM_LOAD16_BYTE( "95417201.hi", 0x00000, 0x080000, CRC(89228288) SHA1(b80852e995033c5aa31d846dd6fa69c112c9446c) )
	sc_mclb_others
ROM_END

ROM_START( sc4mclbe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417422.lo", 0x00001, 0x080000, CRC(699587f5) SHA1(0c0b90b88d04cfd58a122d9a5807328da44d84d4) )
	ROM_LOAD16_BYTE( "95417423.hi", 0x00000, 0x080000, CRC(fe431436) SHA1(cdc55d1caa8c8e0b20440a477fee1f360c35c4f5) )
	sc_mclb_others
ROM_END









ROM_START( sc4mont )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406130.odd", 0x00001, 0x080000, CRC(25f17613) SHA1(57cc7ae5145647aed9420ecc0a02250031b139b1) )
	ROM_LOAD16_BYTE( "95406131.evn", 0x00000, 0x080000, CRC(912c65f2) SHA1(04575ab3fea080fc6391bc4a058cfe16fce55bfa) )
	sc_mont_others
ROM_END

ROM_START( sc4monta )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406456.odd", 0x00001, 0x080000, CRC(e5fe4e0b) SHA1(808007536543d4c32558aef2e0b282344d65257e) )
	ROM_LOAD16_BYTE( "95406457.evn", 0x00000, 0x080000, CRC(31666e1b) SHA1(a7cb32963119ad11d2783be6db851e88d5547651) )
	sc_mont_others
ROM_END

ROM_START( sc4montb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406522.odd", 0x00001, 0x080000, CRC(35c44bdd) SHA1(fdb9dbc57ceea898926af5d8c318c9d0f17f2965) )
	ROM_LOAD16_BYTE( "95406523.evn", 0x00000, 0x080000, CRC(2db29889) SHA1(56a917b2d830b8b15fe5fd8868fb8b1fc0d7c023) )
	sc_mont_others
ROM_END

ROM_START( sc4montc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406600.odd", 0x00001, 0x080000, CRC(7ef84fd0) SHA1(ae70d29f505b0ec14c584220bb053090e4392d52) )
	ROM_LOAD16_BYTE( "95406601.evn", 0x00000, 0x080000, CRC(17fa1234) SHA1(e62a2be4878c8d92f20aa3c3a17e79d49d2ae880) )
	sc_mont_others
ROM_END

ROM_START( sc4montd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406670.odd", 0x00001, 0x080000, CRC(3438bd18) SHA1(c10b6c0a4f65c1d252ca0d62df037ea3710f1cb3) )
	ROM_LOAD16_BYTE( "95406671.evn", 0x00000, 0x080000, CRC(242c6c4a) SHA1(bb02adf354ad8a6b7aecc166024978e34b1f3eb5) )
	sc_mont_others
ROM_END

ROM_START( sc4monte )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407130.odd", 0x00001, 0x080000, CRC(790f0c8c) SHA1(d754f4560fc611be159c84794cc148dcfb4fd857) )
	ROM_LOAD16_BYTE( "95407131.evn", 0x00000, 0x080000, CRC(39fc23f8) SHA1(3fb1443e23afd34bc5af4e69d4b6bf33a103b62b) )
	sc_mont_others
ROM_END

ROM_START( sc4montf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407456.odd", 0x00001, 0x080000, CRC(e67f0701) SHA1(ef52ef52f5ebe9abcebb903dd44b37ddf100fbe2) )
	ROM_LOAD16_BYTE( "95407457.evn", 0x00000, 0x080000, CRC(1ec393cd) SHA1(02892ebf9a1a6cdd43543f37ac42106abc62744b) )
	sc_mont_others
ROM_END

ROM_START( sc4montg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407522.odd", 0x00001, 0x080000, CRC(351104f9) SHA1(42faae9d9d9946cbb817005962bb27a2d4d5c0b4) )
	ROM_LOAD16_BYTE( "95407523.evn", 0x00000, 0x080000, CRC(0d515a0e) SHA1(e6009efbc1d49d136eb2a05dfb88ca579bbe2a36) )
	sc_mont_others
ROM_END

ROM_START( sc4month )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407600.odd", 0x00001, 0x080000, CRC(2206354f) SHA1(3edef3ca4a48bcd1aad076870abe8dc6a36d6eb6) )
	ROM_LOAD16_BYTE( "95407601.evn", 0x00000, 0x080000, CRC(bf2a543e) SHA1(5b690920d9a84f9f881636b36e33556686d3c19f) )
	sc_mont_others
ROM_END

ROM_START( sc4monti )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407670.odd", 0x00001, 0x080000, CRC(8ff0380b) SHA1(349f187294fef7ea1dc627c53c2c24f6fbc877fd) )
	ROM_LOAD16_BYTE( "95407671.evn", 0x00000, 0x080000, CRC(dbd7fdd5) SHA1(c0fc28f95b5a17e68b787ebca7e32f1300c6407c) )
	sc_mont_others
ROM_END




ROM_START( sc4motor )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400263.lo", 0x00001, 0x080000, CRC(cd80f6a1) SHA1(09aea111a65752ec9b9cf077440b68ad7733b46e) )
	ROM_LOAD16_BYTE( "95400264.hi", 0x00000, 0x080000, CRC(22e90f9d) SHA1(a1c2f0741e0752c97576259e282640f363e04ed0) )
	sc_motor_others
ROM_END

ROM_START( sc4motora )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400313.lo", 0x00001, 0x080000, CRC(520c22f5) SHA1(61dd0b2226cdc1f3fadfb86adacaf719f5813a9f) )
	ROM_LOAD16_BYTE( "95400314.hi", 0x00000, 0x080000, CRC(d3c0fb2f) SHA1(fc3bf6b5086e77c145aefa77243c99860461ddce) )
	sc_motor_others
ROM_END

ROM_START( sc4motorb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400739.lo", 0x00001, 0x080000, CRC(1acee4c3) SHA1(63bc20f8f42c41399e56eb051fd90c90b8f55c0f) )
	ROM_LOAD16_BYTE( "95400740.hi", 0x00000, 0x080000, CRC(766729ed) SHA1(56c532c1d4988b25c4dd2f152fd6b7a39fcec46a) )
	sc_motor_others
ROM_END

ROM_START( sc4motorc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400741.lo", 0x00001, 0x080000, CRC(a15d5538) SHA1(a5366e06f144c1d9688d1b2aadcac4675664ffdc) )
	ROM_LOAD16_BYTE( "95400742.hi", 0x00000, 0x080000, CRC(fb1cc6fd) SHA1(7a4c5be6effada69da9fd5c377cd0e1ec13ff6c5) )
	sc_motor_others
ROM_END

ROM_START( sc4motord )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401263.lo", 0x00001, 0x080000, CRC(90594e45) SHA1(96ff726390beb7cdb61f4d550d5ac7fe1ed95cfb) )
	ROM_LOAD16_BYTE( "95401264.hi", 0x00000, 0x080000, CRC(4c14217d) SHA1(a2a9b7fa8b80cfad283e787973838208559c4683) )
	sc_motor_others
ROM_END

ROM_START( sc4motore )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401313.lo", 0x00001, 0x080000, CRC(56e7edfc) SHA1(8bb14e38b1196621300a6b0c7e86f39bdd08fd43) )
	ROM_LOAD16_BYTE( "95401314.hi", 0x00000, 0x080000, CRC(0f9b5ca7) SHA1(80b19c852ff0e9767da23cae8f08501dad55d7b1) )
	sc_motor_others
ROM_END

ROM_START( sc4motorf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401739.lo", 0x00001, 0x080000, CRC(2b1efd88) SHA1(25adffecc8ef5e1333d27b3a03715007cef353ed) )
	ROM_LOAD16_BYTE( "95401740.hi", 0x00000, 0x080000, CRC(626975ae) SHA1(1136ea45699764a27cdd05ad932c9936d63561c8) )
	sc_motor_others
ROM_END

ROM_START( sc4motorg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401741.lo", 0x00001, 0x080000, CRC(20344fa3) SHA1(d508485d312ddfc2fb13b443863d92f1c7e13276) )
	ROM_LOAD16_BYTE( "95401742.hi", 0x00000, 0x080000, CRC(e3880747) SHA1(60193e810abc726dbc9249112da49e6a95f3dc58) )
	sc_motor_others
ROM_END

ROM_START( sc4motorh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "mmay15lo",    0x00001, 0x080000, CRC(da0479fc) SHA1(815a1905714c27915db5bef765e44ba810d362d3) )
	ROM_LOAD16_BYTE( "mmay15hi",    0x00000, 0x080000, CRC(03365045) SHA1(cc041206a7872c5bd90d7feed3b2b760c4e785ab) )
	sc_motor_others
ROM_END




ROM_START( sc4mou )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408478.odd", 0x00001, 0x080000, CRC(05cb62cf) SHA1(70f332e132d259f8b6aaeff2a6b62a10eba5a45c) )
	ROM_LOAD16_BYTE( "95408479.evn", 0x00000, 0x080000, CRC(fab6b737) SHA1(1e3ba6d441c7bf0b9e502a41cf2a6e5040d15aed) )
	sc_mou_others
ROM_END

ROM_START( sc4moua )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409478.odd", 0x00001, 0x080000, CRC(bc02d1df) SHA1(861bb8ea19a3a08d9189f6f5f33e9f0eef460936) )
	ROM_LOAD16_BYTE( "95409479.evn", 0x00000, 0x080000, CRC(5266f13d) SHA1(ae4e38140537719ed2e31d2dca069a5b1c74d147) )
	sc_mou_others
ROM_END

ROM_START( sc4moub )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409480.odd", 0x00001, 0x080000, CRC(9378e444) SHA1(c8dcba47df2b710e802d36d5903f45bb0fd3195b) )
	ROM_LOAD16_BYTE( "95409481.evn", 0x00000, 0x080000, CRC(047e17ac) SHA1(80bb56775f5dbe988bb79024693af592bc8cc638) )
	sc_mou_others
ROM_END




ROM_START( sc4nmtj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414622.lo", 0x00001, 0x080000, CRC(c5187261) SHA1(5b9e891faf7469707b92470a5446c853597e2374) )
	ROM_LOAD16_BYTE( "95414623.hi", 0x00000, 0x080000, CRC(266f953b) SHA1(11f6f3a76a4cd0565646d045b15132dfe32c2107) )
	sc_nmtj_others
ROM_END

ROM_START( sc4nmtja )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420442.lo", 0x00001, 0x080000, CRC(64e269e8) SHA1(25c5c3ff3ee7ce6b7d6f78445c5118eb59d0e54b) )
	ROM_LOAD16_BYTE( "95420443.hi", 0x00000, 0x080000, CRC(de19cd37) SHA1(000f9ffdadc0032549534d49987bbbdee509e655) )
	sc_nmtj_others
ROM_END

ROM_START( sc4nmtjb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420444.lo", 0x00001, 0x080000, CRC(cf082e36) SHA1(5c4dbc75c16ff6b204c692d602d72684ab1b8d3c) )
	ROM_LOAD16_BYTE( "95420445.hi", 0x00000, 0x080000, CRC(7c0d797a) SHA1(f1463d9c276943d4ff8ab98f37561cd17650891c) )
	sc_nmtj_others
ROM_END

ROM_START( sc4nmtjc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421442.lo", 0x00001, 0x080000, CRC(2071bb77) SHA1(8fd3b5cfa2832bd0095d8708f6ed63ded1011d3f) )
	ROM_LOAD16_BYTE( "95421443.hi", 0x00000, 0x080000, CRC(e31d8600) SHA1(448fbd78be8fd4097c2e66b7058dc50cf4f8bc87) )
	sc_nmtj_others
ROM_END

ROM_START( sc4nmtjd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421444.lo", 0x00001, 0x080000, CRC(705b0523) SHA1(80752ae5f151de850a9746506de7cdcc9cf2f2a6) )
	ROM_LOAD16_BYTE( "95421445.hi", 0x00000, 0x080000, CRC(a81ef149) SHA1(7004b0910d133a2756cf37e7496242900a7845d2) )
	sc_nmtj_others
ROM_END







ROM_START( sc4nudit )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408000.lo", 0x00001, 0x080000, CRC(20f44215) SHA1(17c4424edcae989c9d3fbd9a6fbf61d808882bda) )
	ROM_LOAD16_BYTE( "95408001.hi", 0x00000, 0x080000, CRC(dbf69b04) SHA1(dfa4aa0e00ab24436b682c760f5734697bc1b110) )
	sc_nudit_wild_others
ROM_END


ROM_START( sc4nudita )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408002.lo", 0x00001, 0x080000, CRC(30c1e8ae) SHA1(0c7d14461d8324205f98159f5bc2d7c3e9c98c7a) )
	ROM_LOAD16_BYTE( "95408003.hi", 0x00000, 0x080000, CRC(e0e2fbf5) SHA1(973b315170d7ceaf022ba29c8a46088ab6958102) )
	sc_nudit_others
ROM_END


ROM_START( sc4nuditb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409002.lo", 0x00001, 0x080000, CRC(8ddb40dc) SHA1(cfbd0ba995472f5447cc40dcec59aab12a82efab) )
	ROM_LOAD16_BYTE( "95409003.hi", 0x00000, 0x080000, CRC(a1257efb) SHA1(4f21c5f2c992f688e0e592b6790e4d3e6bb7c7ae) )
	sc_nudit_others
ROM_END




ROM_START( sc4oyf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1310p11.hi", 0x00000, 0x080000, CRC(27ca7b83) SHA1(b7f3398d14b6360bc589bd490f3321b790eebe39) )
	ROM_LOAD16_BYTE( "p1310p11.lo", 0x00001, 0x080000, CRC(28504c4b) SHA1(60bff4f680114909520b4d978e7afbb2b22aaeec) )
	sc_oyf_others
ROM_END

ROM_START( sc4oyfa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1310s11.hi", 0x00000, 0x080000, CRC(5028d55f) SHA1(fa96b81b91bc2a0c2708b30721f38a4fb6d1886f) )
	ROM_LOAD16_BYTE( "p1310s11.lo", 0x00001, 0x080000, CRC(9e6a1b0e) SHA1(ff7272cd732c922bc750d95844f7e70f450a6042) )
	sc_oyf_others
ROM_END




ROM_START( sc4opses )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404580.lo", 0x00001, 0x080000, CRC(ee8ddc11) SHA1(dd91b9e8845e057979680406814cd8269d6bde89) )
	ROM_LOAD16_BYTE( "95404581.hi", 0x00000, 0x080000, CRC(36628cef) SHA1(9c863b911bac71a2d1efbc76df13d8ff4c73cc0a) )
	sc_opses_others
ROM_END

ROM_START( sc4opsesa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405580.lo", 0x00001, 0x080000, CRC(61d0aaa5) SHA1(f47232e37fd82f28b3e31811878949549b6df84d) )
	ROM_LOAD16_BYTE( "95405581.hi", 0x00000, 0x080000, CRC(42763b55) SHA1(74b2323f5f997b7ebfa32e571b022ab115a1567f) )
	sc_opses_others
ROM_END




ROM_START( sc4outlw )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400825.lo", 0x00001, 0x080000, CRC(39333b8f) SHA1(2e0919bd311c3aa53cf834954a1b00013615b00a) )
	ROM_LOAD16_BYTE( "95400826.hi", 0x00000, 0x080000, CRC(a242f394) SHA1(67ecead37f703941e2d7827d81d1419f935a4a1a) )
	sc_outlw_others
ROM_END

ROM_START( sc4outlwa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400827.lo", 0x00001, 0x080000, CRC(de4a58e3) SHA1(b364e19e740b0ef2079d6ee8e87c5ddbdaff0308) )
	ROM_LOAD16_BYTE( "95400828.hi", 0x00000, 0x080000, CRC(26b46e52) SHA1(e1aff225b9b2196ae3f80fc6130c244a230207c1) )
	sc_outlw_others
ROM_END

ROM_START( sc4outlwb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401825.lo", 0x00001, 0x080000, CRC(1639b71a) SHA1(1442038fa4f6f03d92437942fe8f796382e82923) )
	ROM_LOAD16_BYTE( "95401826.hi", 0x00000, 0x080000, CRC(e0c6820a) SHA1(6e0c685edc9b7c8a12ac16928fac02c67cfa27f8) )
	sc_outlw_others
ROM_END

ROM_START( sc4outlwc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401827.lo", 0x00001, 0x080000, CRC(f61a55a6) SHA1(ffbfdfcb27b8580f201c98c1960e252014d9f2df) )
	ROM_LOAD16_BYTE( "95401828.hi", 0x00000, 0x080000, CRC(ae87eadf) SHA1(d2d7b7364a1240b2690708a452f42a8c8b076c93) )
	sc_outlw_others
ROM_END




ROM_START( sc4pacmn )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400861.lo", 0x00001, 0x080000, CRC(e4949979) SHA1(a00bcd8c839b14b9e1e4b21a8c78a9be789e0cf7) )
	ROM_LOAD16_BYTE( "95400862.hi", 0x00000, 0x080000, CRC(b1cd18f0) SHA1(b34a7d5595d3abad45e0cd027c217ac5b2bb3088) )
	PACMAN_SOUNDS
ROM_END

ROM_START( sc4pacmna )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402111.lo", 0x00001, 0x080000, CRC(e88b337a) SHA1(ecf7c3fb5e68fba9cbccada502022ae942992699) )
	ROM_LOAD16_BYTE( "95402112.hi", 0x00000, 0x080000, CRC(1f17de1d) SHA1(6576f40cf4dc8fd3de1fd121d5a4c19536da87c5) )
	PACMAN_SOUNDS
ROM_END

ROM_START( sc4pacmnb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403111.lo", 0x00001, 0x080000, CRC(259141c0) SHA1(4bb6d8614696548f1f81f6c2045bde45d4316927) )
	ROM_LOAD16_BYTE( "95403112.hi", 0x00000, 0x080000, CRC(26c12768) SHA1(41dbde2bbc7df5bb9e50d9380f8199345ae1989c) )
	PACMAN_SOUNDS
ROM_END

ROM_START( sc4pacqp )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "96000001.lo", 0x00001, 0x080000, CRC(250b6446) SHA1(d4aaaf1f0eb6833fb90bdfcc46100853db007c98) )
	ROM_LOAD16_BYTE( "96000002.hi", 0x00000, 0x080000, CRC(85aded4d) SHA1(a087e3e82664806858f2b36917a56d5326c79544) )
	QPS_PACP_SOUNDS
ROM_END

ROM_START( sc4pacqpa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "96000016.odd", 0x00001, 0x080000, CRC(4e1d764e) SHA1(9c2a5ae820a95448558ed12cbe87a9353c054d44) )
	ROM_LOAD16_BYTE( "96000017.evn", 0x00000, 0x080000, CRC(eaf93528) SHA1(d4464eae9cd015e33f305d0458086e845199b8cd) )
	QPS_PACP_SOUNDS
ROM_END

ROM_START( sc4pacqpb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "96003001.lo", 0x00001, 0x080000, CRC(70e21196) SHA1(0b7098da61df73f72ba459b1e8e9b6e92aef07d8) )
	ROM_LOAD16_BYTE( "96003002.hi", 0x00000, 0x080000, CRC(4ebbe216) SHA1(aca58b3a2d8368014b738ce15d677dbcca885cc5) )
	QPS_PACP_SOUNDS
ROM_END




ROM_START( sc4paccs )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400871.lo", 0x00001, 0x080000, CRC(25b49ba8) SHA1(6017db6a11531707f15becc1ce0d00e01927cf72) )
	ROM_LOAD16_BYTE( "95400872.hi", 0x00000, 0x080000, CRC(49a6199a) SHA1(eb2ab3f8c094d44e033a7c23158cb382aa35a48b) )
	sc_paccs_others
ROM_END

ROM_START( sc4paccsa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400879.lo", 0x00001, 0x080000, CRC(4ec97800) SHA1(558bd1e63c892bc4423f5f4fcf74750c06cf9e09) )
	ROM_LOAD16_BYTE( "95400880.hi", 0x00000, 0x080000, CRC(16924c38) SHA1(126bca53f0ba2007bfb04e37503059d0302401a4) )
	sc_paccs_others
ROM_END


ROM_START( sc4paccsb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401871.lo", 0x00001, 0x080000, CRC(c43e6e09) SHA1(59ead71e88569c5c008133eb2a8ec3cd0f01425f) )
	ROM_LOAD16_BYTE( "95401872.hi", 0x00000, 0x080000, CRC(a0b1da9e) SHA1(7b48121b38c3752d478ee126a57a2bc447854dba) )
	sc_paccs_others
ROM_END


ROM_START( sc4paccsc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401879.lo", 0x00001, 0x080000, CRC(2547d8a0) SHA1(98b17cd44bea21e976412b5fb16ef26611b4f08b) )
	ROM_LOAD16_BYTE( "95401880.hi", 0x00000, 0x080000, CRC(b03a9913) SHA1(18e18d3ca1cbcc4bd5c49dc247a1a78cd9962009) )
	sc_paccs_others
ROM_END


ROM_START( sc4paccsd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402642.lo", 0x00001, 0x080000, CRC(3a58b754) SHA1(6e11940cef39fec8f066c086f33e3db9fd340c4d) )
	ROM_LOAD16_BYTE( "95402643.hi", 0x00000, 0x080000, CRC(4c9988ff) SHA1(b1ac7ea07e88f10ee6169699ce8b1e9e7f0da67d) )
	sc_paccs_others
ROM_END

ROM_START( sc4paccse )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406408.lo", 0x00001, 0x080000, CRC(8cd20c6c) SHA1(c958f3c4312f85b53fc865e66c3fc89b48813053) )
	ROM_LOAD16_BYTE( "95406409.hi", 0x00000, 0x080000, CRC(861850cf) SHA1(bbd071188b29f5641d6a97bd5f9e7e9174248443) )
	sc_paccs_others
ROM_END


ROM_START( sc4paccsf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407408.lo", 0x00001, 0x080000, CRC(bfbb0455) SHA1(8bbb4134b2b5d23057f9f3f85dd13181d472e974) )
	ROM_LOAD16_BYTE( "95407409.hi", 0x00000, 0x080000, CRC(358075d7) SHA1(7354a26d3d146f9a653f09163b082ee612a37ce3) )
	sc_paccs_others
ROM_END


ROM_START( sc4paccsg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414432.lo", 0x00001, 0x080000, CRC(b5ff58f6) SHA1(d5b707dc1a21b975a2494537851aef53e97bcf1f) )
	ROM_LOAD16_BYTE( "95414433.hi", 0x00000, 0x080000, CRC(a446fd8a) SHA1(4bd318dff1b4ab579196c6067f163bbda874c35c) )
	sc_paccs_others
ROM_END


ROM_START( sc4paccsh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415428.lo", 0x00001, 0x080000, CRC(b9a45ec8) SHA1(abcb37550f9555d6764b8dbf0f96259e1a63cc9a) )
	ROM_LOAD16_BYTE( "95415429.hi", 0x00000, 0x080000, CRC(b974d62c) SHA1(5d72ea9eb22038b4539aff3c322b1842ecc44006) )
	sc_paccs_others
ROM_END


ROM_START( sc4paccsi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415430.lo", 0x00001, 0x080000, CRC(9d9b61af) SHA1(cc3b82ef50ca7d35173fc1e1525c714d70304da5) )
	ROM_LOAD16_BYTE( "95415431.hi", 0x00000, 0x080000, CRC(641cc3ce) SHA1(efaa5707761836683374866553b68cd7cd05795e) )
	sc_paccs_others
ROM_END


ROM_START( sc4paccsj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415432.lo", 0x00001, 0x080000, CRC(b38aef45) SHA1(c9415665965010e6fcffe4168194591efafab83f) )
	ROM_LOAD16_BYTE( "95415433.hi", 0x00000, 0x080000, CRC(c8c68b8a) SHA1(60456c69cb8379474d30824d53e84d6ba73e0da0) )
	sc_paccs_others
ROM_END


ROM_START( sc4paccsk )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415502.lo", 0x00001, 0x080000, CRC(370afd1c) SHA1(7e9f32fe4166e61bffe46d704c9fd5479d63f4ed) )
	ROM_LOAD16_BYTE( "95415503.hi", 0x00000, 0x080000, CRC(f90a4b9a) SHA1(87cf19d6754995003e18cd71b27d395925743a53) )
	sc_paccs_others
ROM_END


ROM_START( sc4paccsl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415504.lo", 0x00001, 0x080000, CRC(0c33ee13) SHA1(16dc6abae6dfa8bad3b4a8a3774af013e9d148a6) )
	ROM_LOAD16_BYTE( "95415505.hi", 0x00000, 0x080000, CRC(a25ff920) SHA1(52c46165ebbd8812df4943dbd5d82e4e4455430b) )
	sc_paccs_others
ROM_END




ROM_START( sc4paccl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406246.lo", 0x00001, 0x080000, CRC(1d855616) SHA1(8cc6ca3019be7efd384ee272725ad2f42307efa8) )
	ROM_LOAD16_BYTE( "95406247.hi", 0x00000, 0x080000, CRC(14c97bd9) SHA1(cb5005251417efacd459fc596b71e1db09a82458) )
	sc_paccl_others
ROM_END

ROM_START( sc4paccla )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406418.lo", 0x00001, 0x080000, CRC(f1d245c6) SHA1(56bedeca7bcb94bb0acfd28d923aa9ad4f8e35c0) )
	ROM_LOAD16_BYTE( "95406419.hi", 0x00000, 0x080000, CRC(3abdcc9c) SHA1(47594fad060345e8e011fb0763f5246ddd3b6e7d) )
	sc_paccl_others
ROM_END

ROM_START( sc4pacclb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407246.lo", 0x00001, 0x080000, CRC(098515f1) SHA1(5dce61b818d5035a65dd52897b32976715a335c6) )
	ROM_LOAD16_BYTE( "95407247.hi", 0x00000, 0x080000, CRC(b1bd0c5a) SHA1(e54320c557822ba99baccac9dbd0c937419bd21f) )
	sc_paccl_others
ROM_END

ROM_START( sc4pacclc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407418.lo", 0x00001, 0x080000, CRC(56614d56) SHA1(5432a4986412f60e220fa2217c98cc98843cc319) )
	ROM_LOAD16_BYTE( "95407419.hi", 0x00000, 0x080000, CRC(87c22c9d) SHA1(d466f8dfaf0b472b0fca11086354c2141ed0559e) )
	sc_paccl_others
ROM_END




ROM_START( sc4pacpl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402323.lo", 0x00001, 0x080000, CRC(cfbed405) SHA1(65d68f1631bf3310b1e835c75474c947f6d4c2be) )
	ROM_LOAD16_BYTE( "95402324.hi", 0x00000, 0x080000, CRC(fad234c2) SHA1(3dbd04c7c21b0a1986e93b2bae447f76822c5a5d) )
	sc_pacpl_others
ROM_END

ROM_START( sc4pacpla )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402325.lo", 0x00001, 0x080000, CRC(6a54edec) SHA1(704f382885c11c46e8e8d3a9d10591ec780d8c37) )
	ROM_LOAD16_BYTE( "95402326.hi", 0x00000, 0x080000, CRC(8cf81ef2) SHA1(12f0974088ff6ac4c165fe56c7f6291d9c9cc192) )
	sc_pacpl_others
ROM_END

ROM_START( sc4pacplb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402389.lo", 0x00001, 0x080000, CRC(784cbdc6) SHA1(860acce2f2379651a209c4f66ffe23b328adb39b) )
	ROM_LOAD16_BYTE( "95402390.hi", 0x00000, 0x080000, CRC(34699bae) SHA1(d42e599a679774263230cfc75104f8c8412bd01f) )
	sc_pacpl_others
ROM_END

ROM_START( sc4pacplc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402606.lo", 0x00001, 0x080000, CRC(788934dc) SHA1(f0ba520072253a5f78f4b29a4dc71e135675defe) )
	ROM_LOAD16_BYTE( "95402607.hi", 0x00000, 0x080000, CRC(f7d100fc) SHA1(6281f5891b60651d1f94a982689dc17a8b2bdefb) )
	sc_pacpl_others
ROM_END

ROM_START( sc4pacpld )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403321.lo", 0x00001, 0x080000, CRC(aefa20af) SHA1(966b9124f5a09e129d68471cfa0a87d813456482) )
	ROM_LOAD16_BYTE( "95403322.hi", 0x00000, 0x080000, CRC(592c15ce) SHA1(c87e47993ede5aa8cbb06a569bd1a9ea5d623274) )
	sc_pacpl_others
ROM_END

ROM_START( sc4pacple )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403325.lo", 0x00001, 0x080000, CRC(20131ca7) SHA1(c76b198745f48ff629a967ffa1130c1c713966e2) )
	ROM_LOAD16_BYTE( "95403326.hi", 0x00000, 0x080000, CRC(056a6037) SHA1(e13eb112ed2e0d2f92c476f84c4992e2790c477a) )
	sc_pacpl_others
ROM_END

ROM_START( sc4pacplf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403389.lo", 0x00001, 0x080000, CRC(153b660c) SHA1(4389778d63a214cc40fa810483c8939eca2d0d1a) )
	ROM_LOAD16_BYTE( "95403390.hi", 0x00000, 0x080000, CRC(bcc5491b) SHA1(54a50df19fbab4c1ab50320b1c4566ab59935f46) )
	sc_pacpl_others
ROM_END

ROM_START( sc4pacplg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403606.lo", 0x00001, 0x080000, CRC(17493df2) SHA1(a55cb6042b4898b6c9a9212cfa89a8c264d7ce7a) )
	ROM_LOAD16_BYTE( "95403607.hi", 0x00000, 0x080000, CRC(ec0635d6) SHA1(b5abca113bb6665e940848fcb0356d417e5d1a8a) )
	sc_pacpl_others
ROM_END

ROM_START( sc4pacplh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "pman_pl.lo",  0x00001, 0x080000, CRC(508f918f) SHA1(203ddc867626ce94988e7465b44f26afed206e3e) )
	ROM_LOAD16_BYTE( "pman_pl.hi",  0x00000, 0x080000, CRC(51b0a5d2) SHA1(8208bbd7072e857967a555abf26cf3f7ef9a0178) )
	sc_pacpl_others
ROM_END




ROM_START( sc4pmani )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406164.lo", 0x00001, 0x080000, CRC(4fce0eed) SHA1(6a81206be65fa897e3e8180b16ffe78e98996def) )
	ROM_LOAD16_BYTE( "95406165.hi", 0x00000, 0x080000, CRC(d059e343) SHA1(98f14670580da750968feb182f37b0eb1e028a6f) )
	sc_pmani_others
ROM_END

ROM_START( sc4pmania )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407164.lo", 0x00001, 0x080000, CRC(e50826d7) SHA1(b0fdcf4a645f3481e82a38dd4cf8859dbdc14eaa) )
	ROM_LOAD16_BYTE( "95407165.hi", 0x00000, 0x080000, CRC(93e33ead) SHA1(fef1e8534506f86e93096f5d07439ace5e8649c1) )
	sc_pmani_others
ROM_END




ROM_START( sc4party )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404102.lo", 0x00001, 0x080000, CRC(5c3f1160) SHA1(d4411fd99fac9d4c2ecfd8965dc2c42da090a344) )
	ROM_LOAD16_BYTE( "95404103.hi", 0x00000, 0x080000, CRC(94f8cefa) SHA1(ee528535ca9840e1cfd315cf04bf8ae79a2f87f4) )
	sc_party_others
ROM_END




ROM_START( sc4paytm )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406364.lo", 0x00001, 0x080000, CRC(21300456) SHA1(41da31b42c99fee5602028c48217f7de6d551c94) )
	ROM_LOAD16_BYTE( "95406365.hi", 0x00000, 0x080000, CRC(805d8f51) SHA1(200cb1bda1360905264b0219e1a4d86a6a84d7c4) )
	sc_paytm_others
ROM_END




ROM_START( sc4pgold )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404232.lo", 0x00001, 0x080000, CRC(47c60203) SHA1(12afaedd713ffeda0fc41715dcb4d411cc05d186) )
	ROM_LOAD16_BYTE( "95404233.hi", 0x00000, 0x080000, CRC(6d8d283d) SHA1(d09b24b67fcb31f2ad13a0904ac00b8ee0b86201) )
	sc_pgold_others
ROM_END

ROM_START( sc4pgolda )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404234.lo", 0x00001, 0x080000, CRC(c3636257) SHA1(3bb86ec3d1b3a447120bc7b5225bfccf67e68c4a) )
	ROM_LOAD16_BYTE( "95404235.hi", 0x00000, 0x080000, CRC(f0879aae) SHA1(01cb5eea7b04597f0b39a14afebfa0d840bb8582) )
	sc_pgold_others
ROM_END

ROM_START( sc4pgoldb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405232.lo", 0x00001, 0x080000, CRC(3560fc87) SHA1(8d03dd35d9e43e449ad55c2ae2ae521e64ef1bec) )
	ROM_LOAD16_BYTE( "95405233.hi", 0x00000, 0x080000, CRC(0a782f76) SHA1(143ce5a7e1e61f767138f375fc399e2209a9df6e) )
	sc_pgold_others
ROM_END

ROM_START( sc4pgoldc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405234.lo", 0x00001, 0x080000, CRC(bb89b7e7) SHA1(c9a4702bbabfa8d94f5fb907fa2a3f740bab0086) )
	ROM_LOAD16_BYTE( "95405235.hi", 0x00000, 0x080000, CRC(8a7284ac) SHA1(4b712f3cd9b9d8bb69a2445bd020ee591c82de35) )
	sc_pgold_others
ROM_END

ROM_START( sc4pgoldd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406014.lo", 0x00001, 0x080000, CRC(9c78f654) SHA1(313659c08a9818ccb93ef9c2a5b43b37619669ba) )
	ROM_LOAD16_BYTE( "95406015.hi", 0x00000, 0x080000, CRC(1c84a153) SHA1(db485aa57672e81849e18c642a8c4099e9e1a93c) )
	sc_pgold_others
ROM_END


ROM_START( sc4pgoldf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407014.lo", 0x00001, 0x080000, CRC(9e7db6c3) SHA1(a38dd6a2110e65e40f96fc02494a3e21846c1bb0) )
	ROM_LOAD16_BYTE( "95407015.hi", 0x00000, 0x080000, CRC(c1e1aaab) SHA1(829113b8c324f4ecf4a6137b6a2a3a6bf062be3b) )
	sc_pgold_others
ROM_END

ROM_START( sc4pglcs )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406734.lo", 0x00001, 0x080000, CRC(e983d8db) SHA1(045c1f73cacfdcbdfbea74ed23287120433e3341) )
	ROM_LOAD16_BYTE( "95406735.hi", 0x00000, 0x080000, CRC(c082ae45) SHA1(5985b87924d44155f55806772e406d3dcaf6b506) )
	SC4_95004316_CAS_PHAR_GOLD_PR1261
ROM_END

ROM_START( sc4pglcsa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408068.lo", 0x00001, 0x080000, CRC(8932a906) SHA1(eca4706dd62e089ad9ce781b2ede0759dbe44bd9) )
	ROM_LOAD16_BYTE( "95408069.hi", 0x00000, 0x080000, CRC(f56fff18) SHA1(e123bd7f1af75218dbbef82b33e7c21d882f8105) )
	SC4_95004316_CAS_PHAR_GOLD_PR1261
ROM_END

ROM_START( sc4pglcsb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408144.lo", 0x00001, 0x080000, CRC(ee67e3d6) SHA1(963160426e1b2d4399132ff0df9662dc1d46c121) )
	ROM_LOAD16_BYTE( "95408145.hi", 0x00000, 0x080000, CRC(b6e8a6d8) SHA1(d443aac2967229de4d6feae634efae9f3cd2f361) )
	SC4_95004316_CAS_PHAR_GOLD_PR1261
ROM_END














ROM_START( sc4ppclb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416532.odd", 0x00001, 0x080000, CRC(7fa63d03) SHA1(2953fc769c93067d9f9833fdf219277dc1c1dd3d) )
	ROM_LOAD16_BYTE( "95416533.evn", 0x00000, 0x080000, CRC(3f202212) SHA1(f611948bea5daa8a52c4c9223d35a62de44520b1) )
	sc_ppclb_others
ROM_END

ROM_START( sc4ppclba )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416582.odd", 0x00001, 0x080000, CRC(574d2e58) SHA1(1356f9048aa200654160cd51bdd3b86f80aac8d1) )
	ROM_LOAD16_BYTE( "95416583.evn", 0x00000, 0x080000, CRC(b89f1718) SHA1(d32640919d71f13b2febf3eb91eb5f230f6f5168) )
	sc_ppclb_others
ROM_END

ROM_START( sc4ppclbb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417532.odd", 0x00001, 0x080000, CRC(d7480cb5) SHA1(c2a718abf8a7557bebef8f6f65d44ff54d8ecb8d) )
	ROM_LOAD16_BYTE( "95417533.evn", 0x00000, 0x080000, CRC(c1e51ffd) SHA1(d0f9028998b17e9faaade381f1b23c48d6f592f1) )
	sc_ppclb_others
ROM_END

ROM_START( sc4ppclbc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417582.odd", 0x00001, 0x080000, CRC(e0598606) SHA1(2f0f009430d869a1a84d7650128d8c6c08cc6fc3) )
	ROM_LOAD16_BYTE( "95417583.evn", 0x00000, 0x080000, CRC(123209f2) SHA1(510d9776d6efad63d5b81b58d23d624e6a91d94f) )
	sc_ppclb_others
ROM_END








ROM_START( sc4pipe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408732.lo", 0x00001, 0x080000, CRC(4cc1d302) SHA1(74a8e5b457a51285245d58c14afe0cf3835c44bf) )
	ROM_LOAD16_BYTE( "95408733.hi", 0x00000, 0x080000, CRC(78dc99e5) SHA1(67f38232da9aeeb33c4e600bcee3b5c552a2a387) )
	sc_pipe_others
ROM_END

ROM_START( sc4pipea )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409732.lo", 0x00001, 0x080000, CRC(35e374ee) SHA1(a79caff9a0dfb0048a83a932d70f9034a9ed2820) )
	ROM_LOAD16_BYTE( "95409733.hi", 0x00000, 0x080000, CRC(391b1ceb) SHA1(e9016f80695f9812676df3899758d38bee19e765) )
	sc_pipe_others
ROM_END




ROM_START( sc4plumb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "91414380.odd", 0x00001, 0x080000, CRC(81fc3ea4) SHA1(21b92e7cbba5b13def9275a994bb96bda7991f5c) )
	ROM_LOAD16_BYTE( "91414381.evn", 0x00000, 0x080000, CRC(3ab009fb) SHA1(e0e1e3a83ebce24c8363c51f4c1e731dced4f818) )
	sc_plumb_others
ROM_END

ROM_START( sc4plumba )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "91415380.odd", 0x00001, 0x080000, CRC(7d72768f) SHA1(1fef9b6701e561814f00484e2e480707d0038410) )
	ROM_LOAD16_BYTE( "91415381.evn", 0x00000, 0x080000, CRC(3a2f5fc3) SHA1(958628b151402cb93ea5a899049889f0e09d87e1) )
	sc_plumb_others
ROM_END

ROM_START( sc4plumbb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410914.odd", 0x00001, 0x080000, CRC(01affca7) SHA1(37b8c3dd3c695bb281afe7ebb19bb929b7220072) )
	ROM_LOAD16_BYTE( "95410915.evn", 0x00000, 0x080000, CRC(05f5a2ee) SHA1(f6e4de5a61495e8325c940c2984cfa2099b7b766) )
	sc_plumb_others
ROM_END

ROM_START( sc4plumbc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411914.odd", 0x00001, 0x080000, CRC(942e5500) SHA1(a211f37198bfdc49859e0abf271a52272e38fc65) )
	ROM_LOAD16_BYTE( "95411915.evn", 0x00000, 0x080000, CRC(eddccd9a) SHA1(d5e21cefb8e5341f3d6cc2d5e93e80fc390fe4a2) )
	sc_plumb_others
ROM_END

ROM_START( sc4plumbd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412162.odd", 0x00001, 0x080000, CRC(092eafc4) SHA1(e23b25850c441569e329cc0edb766d81fc3872cc) )
	ROM_LOAD16_BYTE( "95412163.evn", 0x00000, 0x080000, CRC(7d858ac1) SHA1(c947c345dc8ee9b37f8d05be7741792402956e18) )
	sc_plumb_others
ROM_END

ROM_START( sc4plumbe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413162.odd", 0x00001, 0x080000, CRC(f5a0e7ef) SHA1(306ccc811b551a6090363c6b19177f8552f598be) )
	ROM_LOAD16_BYTE( "95413163.evn", 0x00000, 0x080000, CRC(7d1adcf9) SHA1(f174e06efe0b6e714ab3a8496326a14b1d3ae0d7) )
	sc_plumb_others
ROM_END




ROM_START( sc4polem )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400283.lo", 0x00001, 0x080000, CRC(c03a6f6b) SHA1(d333cf1348e8d52f8402d70e03c7b94638f2550a) )
	ROM_LOAD16_BYTE( "95400284.hi", 0x00000, 0x080000, CRC(3f8a3d85) SHA1(d6388a41002f22094792230b1f381bf7363642ca) )
	sc_polem_others
ROM_END

ROM_START( sc4polema )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400373.lo", 0x00001, 0x080000, CRC(2a7df310) SHA1(b4556d086cc2889b1757c38524f7748d851134bf) )
	ROM_LOAD16_BYTE( "95400374.hi", 0x00000, 0x080000, CRC(e60935ad) SHA1(a453d835e219ea4552026237e99fd00cefa24bc2) )
	sc_polem_others
ROM_END

ROM_START( sc4polemb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400605.lo", 0x00001, 0x080000, CRC(5665ea74) SHA1(e25d11640eb81d8be99d16f33322648e3f5c89b8) )
	ROM_LOAD16_BYTE( "95400606.hi", 0x00000, 0x080000, CRC(9a37c598) SHA1(3865444a452937c79f7c2e2288a0a6bd35b4d635) )
	sc_polem_others
ROM_END

ROM_START( sc4polemc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401285.lo", 0x00001, 0x080000, CRC(42fbf8ac) SHA1(b06e6ffcb750b6bfde8b4f084acd22f0d4edce35) )
	ROM_LOAD16_BYTE( "95401286.hi", 0x00000, 0x080000, CRC(5ae2c2f1) SHA1(3313754716f438761e82dcb5dd64c8c34c09f337) )
	sc_polem_others
ROM_END

ROM_START( sc4polemd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401605.lo", 0x00001, 0x080000, CRC(fffd9ae2) SHA1(f47a3ed9b66526cf4b73d3cc5fda644e329f22bc) )
	ROM_LOAD16_BYTE( "95401606.hi", 0x00000, 0x080000, CRC(280e1ac8) SHA1(275ecae649c323f9d716cc7e549e3e25428cffaf) )
	sc_polem_others
ROM_END



ROM_START( sc4polen )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p7012s13.hi", 0x00000, 0x080000, CRC(83367285) SHA1(4ba6bfc0d33607eadcb0600d0318cf682840008c) )
	ROM_LOAD16_BYTE( "p7012s13.lo", 0x00001, 0x080000, CRC(3ef0e8ce) SHA1(a9c12c8a8c47a2a69559cffc9b9a74f3ceb6ebe8) )
	sc_polen_others
ROM_END




ROM_START( sc4polic )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408136.odd", 0x00001, 0x080000, CRC(94c881c6) SHA1(81c180ec446a43c7281f4cb405d78bde12b67007) )
	ROM_LOAD16_BYTE( "95408137.evn", 0x00000, 0x080000, CRC(4df0a90a) SHA1(63c568f9478e70edae45809d040c2a8e7f3619c0) )
	sc_polic_others
ROM_END

ROM_START( sc4polica )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408514.odd", 0x00001, 0x080000, CRC(fea13625) SHA1(5fa2b331b00b7771c71fd3f3bcda0dbe69321d9f) )
	ROM_LOAD16_BYTE( "95408515.evn", 0x00000, 0x080000, CRC(b847bdd8) SHA1(02ed59dc67ef9acee005e7346fd407de8dba8f22) )
	sc_polic_others
ROM_END

ROM_START( sc4policb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409136.odd", 0x00001, 0x080000, CRC(1df681c9) SHA1(a2afb717e3bd6b95ff22806b0c032317760d1f30) )
	ROM_LOAD16_BYTE( "95409137.evn", 0x00000, 0x080000, CRC(d0f3ddb4) SHA1(216ea4bd35b786947e213915defea2cb7abc5f3a) )
	sc_polic_others
ROM_END

ROM_START( sc4policc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409514.odd", 0x00001, 0x080000, CRC(73288dc0) SHA1(c1350ea22cbeed9d9914eb2437ba067d3194f444) )
	ROM_LOAD16_BYTE( "95409515.evn", 0x00000, 0x080000, CRC(69277de1) SHA1(17fb6ec5de1aad51e83c8a547d8d99e43d5a19b8) )
	sc_polic_others
ROM_END











ROM_START( sc4potsh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410908.odd", 0x00001, 0x080000, CRC(e4982864) SHA1(fce3f36b19bf084fa9f3644f30d5b57b2e370e90) )
	ROM_LOAD16_BYTE( "95410909.evn", 0x00000, 0x080000, CRC(e50630e3) SHA1(52b44577e4d306c9b48e7c52c8906f5b48268c06) )
	sc_potsh_others
ROM_END

ROM_START( sc4potsha )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411908.odd", 0x00001, 0x080000, CRC(1dcd6b86) SHA1(f734801b89a1d12d86aaba0c0af5a4d4216f56dc) )
	ROM_LOAD16_BYTE( "95411909.evn", 0x00000, 0x080000, CRC(f1086ca0) SHA1(5263ad310c514e6f8b8c4dd423e093d8181bdf15) )
	sc_potsh_others
ROM_END






ROM_START( sc4pogbl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95422368.lo", 0x00001, 0x080000, CRC(3d34c353) SHA1(9bab4bf469ed5c877bfb567e0cc4155b0dc37ac1) )
	ROM_LOAD16_BYTE( "95422369.hi", 0x00000, 0x080000, CRC(98b01d37) SHA1(cd7d8334c7dd0078cebf066cc1ae0cc503743036) )
	PR3082_C_POTS_OF_GOLD_SOUNDS11
ROM_END

ROM_START( sc4pogbla )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95422932.lo", 0x00001, 0x080000, CRC(2c5ea61c) SHA1(56a8e6b8392ca76c864eab8060ce1cfbe04b9e54) )
	ROM_LOAD16_BYTE( "95422933.hi", 0x00000, 0x080000, CRC(ec3d6a90) SHA1(5b030940775fdb4286efdd7357a276707d9eabde) )
	PR3082_C_POTS_OF_GOLD_SOUNDS11
ROM_END

ROM_START( sc4pogblb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95423368.lo", 0x00001, 0x080000, CRC(0c9a2240) SHA1(eab6b824b87a570b4236b649d138bb6babf6d766) )
	ROM_LOAD16_BYTE( "95423369.hi", 0x00000, 0x080000, CRC(dd96eb1f) SHA1(c2fae68a39ddf476ac11fc03158ed5ecfbc6fc1a) )
	PR3082_C_POTS_OF_GOLD_SOUNDS11
ROM_END

ROM_START( sc4pogblc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95423932.lo", 0x00001, 0x080000, CRC(03cacc64) SHA1(99b776c529115da34413b507cdc20491bfdbd09e) )
	ROM_LOAD16_BYTE( "95423933.hi", 0x00000, 0x080000, CRC(45faea1b) SHA1(1c6d8072086e2e0345bbae42978338eaa15e8304) )
	PR3082_C_POTS_OF_GOLD_SOUNDS11
ROM_END

ROM_START( sc4pogbld )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95426590.lo", 0x00001, 0x080000, CRC(4bd1ca29) SHA1(9d9a3e14027b259c614780681d337d9004e5dbfd) )
	ROM_LOAD16_BYTE( "95426591.hi", 0x00000, 0x080000, CRC(3e550a37) SHA1(48e2371953744e2d3b59833388be66484293dfe9) )
	PR3082_C_POTS_OF_GOLD_SOUNDS11
ROM_END

ROM_START( sc4pogble )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95427590.lo", 0x00001, 0x080000, CRC(0a7f6c5f) SHA1(cf8eb884d1980a1e136d2b1e52ba4d7dc7904896) )
	ROM_LOAD16_BYTE( "95427591.hi", 0x00000, 0x080000, CRC(8802578a) SHA1(71c16bb83b858d42846703387976460812234771) )
	PR3082_C_POTS_OF_GOLD_SOUNDS11
ROM_END


ROM_START( sc4pwcrz )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400189.lo", 0x00001, 0x080000, CRC(2f864cad) SHA1(21431025b4c4c00496ec039dfb4f7e2623b0924b) )
	ROM_LOAD16_BYTE( "95400190.hi", 0x00000, 0x080000, CRC(6e2fbddf) SHA1(f27ea2a12fcec1d2376b99385babe99ff55c409d) )
	sc_pwcrz_others
ROM_END

ROM_START( sc4pwcrza )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400251.lo", 0x00001, 0x080000, CRC(d7071cf3) SHA1(ec86d2db77058538a34484ae7be79092f83ce289) )
	ROM_LOAD16_BYTE( "95400252.hi", 0x00000, 0x080000, CRC(aa9ffa8f) SHA1(da55df6754c4e80a086303361ef1ed347ef425ba) )
	sc_pwcrz_others
ROM_END

ROM_START( sc4pwcrzb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400253.lo", 0x00001, 0x080000, CRC(94dd9c1d) SHA1(483f58aa3a30a4750d525829087ac1ce0161fc8f) )
	ROM_LOAD16_BYTE( "95400254.hi", 0x00000, 0x080000, CRC(3e9c7197) SHA1(853f8cba3ece11f01958266a27b4492d56b80f34) )
	sc_pwcrz_others
ROM_END

ROM_START( sc4pwcrzc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400755.lo", 0x00001, 0x080000, CRC(a6044a3e) SHA1(e61a56d91ac0f57d53e38280e57b4e6d5a995a0f) )
	ROM_LOAD16_BYTE( "95400756.hi", 0x00000, 0x080000, CRC(b404f71a) SHA1(5c3b794ec6653b3e15c111b549ae57deaa918f1e) )
	sc_pwcrz_others
ROM_END

ROM_START( sc4pwcrzd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400757.lo", 0x00001, 0x080000, CRC(25f9f84e) SHA1(785d22b4250ab2c95766fff4fbff0cde30593155) )
	ROM_LOAD16_BYTE( "95400758.hi", 0x00000, 0x080000, CRC(fe3cdb98) SHA1(302f578750296315ac274887f95450d42350b6b0) )
	sc_pwcrz_others
ROM_END

ROM_START( sc4pwcrze )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401189.lo", 0x00001, 0x080000, CRC(d3bc5e7e) SHA1(a5d10ed68c12aa6e455de1d8671ad76533dd014e) )
	ROM_LOAD16_BYTE( "95401190.hi", 0x00000, 0x080000, CRC(e95a0603) SHA1(d2fac221068b1e816d2218d64a99a17322fe670e) )
	sc_pwcrz_others
ROM_END

ROM_START( sc4pwcrzf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401251.lo", 0x00001, 0x080000, CRC(2e5ce4ce) SHA1(66e4fe46b7029f376b537746956cae62a65c6944) )
	ROM_LOAD16_BYTE( "95401252.hi", 0x00000, 0x080000, NO_DUMP )
	sc_pwcrz_others
ROM_END

ROM_START( sc4pwcrzg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401253.lo", 0x00001, 0x080000, CRC(571369c5) SHA1(d954cca70643ac78400c425f6185a21f2c38af63) )
	ROM_LOAD16_BYTE( "95401254.hi", 0x00000, 0x080000, CRC(72ffc510) SHA1(0197fe9f49329d2f75e82ce1bc03439a87a4d20f) )
	sc_pwcrz_others
ROM_END

ROM_START( sc4pwcrzh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401755.lo", 0x00001, 0x080000, CRC(372f7486) SHA1(43922692ff8cfff0bc1c317911d465444abbc779) )
	ROM_LOAD16_BYTE( "95401756.hi", 0x00000, 0x080000, CRC(44260fec) SHA1(621605bd28ec568b4adf4ebcad983bea0a03c2b8) )
	sc_pwcrz_others
ROM_END

ROM_START( sc4pwcrzi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401757.lo", 0x00001, 0x080000, CRC(f4b4b9b0) SHA1(25764f6325fd99921572c9a3a4df27c040164f4b) )
	ROM_LOAD16_BYTE( "95401758.hi", 0x00000, 0x080000, CRC(3b5247e2) SHA1(b1e086aefd27b5c44cb765a29ff87b27998e2d22) )
	sc_pwcrz_others
ROM_END






ROM_START( sc4pwrsg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406126.odd", 0x00001, 0x080000, CRC(2960dd7c) SHA1(4d19cd24abd2b0f0c7bbc6b69e6482d59021be57) )
	ROM_LOAD16_BYTE( "95406127.evn", 0x00000, 0x080000, CRC(67188a1a) SHA1(bf5453f513a3bdbfa207f2cdde5ce65624ccdff5) )
	sc_pwrsg_others
ROM_END

ROM_START( sc4pwrsga )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406128.odd", 0x00001, 0x080000, CRC(ba811b21) SHA1(c0bfd222af7a7d8626b6bb7fb9f10af974c836dd) )
	ROM_LOAD16_BYTE( "95406129.evn", 0x00000, 0x080000, CRC(bff58455) SHA1(429926241668973e8e053d8a0b1114bf73dd614e) )
	sc_pwrsg_others
ROM_END

ROM_START( sc4pwrsgb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407126.odd", 0x00001, 0x080000, CRC(bfcc64f7) SHA1(9e6779e45e32b89b82540e6b47bd901ac8e636f8) )
	ROM_LOAD16_BYTE( "95407127.evn", 0x00000, 0x080000, CRC(12adc7e8) SHA1(47e6e9721fb2b270e46e3713c6559f2e84579140) )
	sc_pwrsg_others
ROM_END

ROM_START( sc4pwrsgc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407128.odd", 0x00001, 0x080000, CRC(4a6aa63e) SHA1(23c507ccc7c62a14e73dcb3d0d799f68891d6bb8) )
	ROM_LOAD16_BYTE( "95407129.evn", 0x00000, 0x080000, CRC(e57a6249) SHA1(346ce6a4556fcafe47eb969d15a70bcb0e9d4d5f) )
	sc_pwrsg_others
ROM_END






ROM_START( sc4pwrbq )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400945.lo", 0x00001, 0x080000, CRC(8b7067a6) SHA1(df5aaa774cc0021e99e98abc1ac13edb92e68e6e) )
	ROM_LOAD16_BYTE( "95400946.hi", 0x00000, 0x080000, CRC(02804a36) SHA1(9fc1f69f0ed73ca9ee4fb2df3a0b477a9379cf52) )
	MAZOOMA_POWERBALL_SOUNDS
ROM_END

ROM_START( sc4pwrbqa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401945.lo", 0x00001, 0x080000, CRC(ca34aba7) SHA1(2890dbaff083f98ea76bb22d6f453c8bd62591eb) )
	ROM_LOAD16_BYTE( "95401946.hi", 0x00000, 0x080000, CRC(9164fb91) SHA1(5a8b44bc342f483794cdb977ae60f32c905da347) )
	MAZOOMA_POWERBALL_SOUNDS
ROM_END






ROM_START( sc4pir )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406550.lo", 0x00001, 0x080000, CRC(4552e341) SHA1(38a2c2f06c42390ad584dd99f4e3ae9de090b84d) )
	ROM_LOAD16_BYTE( "95406551.hi", 0x00000, 0x080000, CRC(1da38ce3) SHA1(7810b415290f259ad4a27e9c88860f35e2e372ad) )
	sc_pir_others
ROM_END

ROM_START( sc4pira )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406892.lo", 0x00001, 0x080000, CRC(1e56abd5) SHA1(d1accb9c75c3cbbc84cb1b09e9e3cb8ca7477fcc) )
	ROM_LOAD16_BYTE( "95406893.hi", 0x00000, 0x080000, CRC(2c851bc9) SHA1(808cb84e533f16fa9c0768c186cf6a69995f0442) )
	sc_pir_others
ROM_END

ROM_START( sc4pirb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407550.lo", 0x00001, 0x080000, CRC(5a311b60) SHA1(a8366956339c07530119fb03d430ea5123f99d66) )
	ROM_LOAD16_BYTE( "95407551.hi", 0x00000, 0x080000, CRC(72595aa2) SHA1(ac3f7106e19922d47d21498085aff833164516c7) )
	sc_pir_others
ROM_END

ROM_START( sc4pirc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407892.lo", 0x00001, 0x080000, CRC(56622f7f) SHA1(452f940179b5c0c17b1ffd9bdd7e3d184b4ea55a) )
	ROM_LOAD16_BYTE( "95407893.hi", 0x00000, 0x080000, CRC(7630ea84) SHA1(dddab23e48e34a58f89dc7dc73c3ea49796fbbcd) )
	sc_pir_others
ROM_END

ROM_START( sc4pird )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408276.lo", 0x00001, 0x080000, CRC(ddea06a4) SHA1(b231c4aca0e63f9feec9757732459169f9f0b874) )
	ROM_LOAD16_BYTE( "95408277.hi", 0x00000, 0x080000, CRC(bad595ae) SHA1(4a8b59b2c47ba37ef63df30408b6a6ea72dd91f7) )
	sc_pir_others
ROM_END

ROM_START( sc4pire )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409276.lo", 0x00001, 0x080000, CRC(bd702e84) SHA1(af75b80f4fcbe1a72241b492b7050e9447032acc) )
	ROM_LOAD16_BYTE( "95409277.hi", 0x00000, 0x080000, CRC(8e380b6a) SHA1(0e8e8870eae6e1c955cf2c95b0cc77ba478724b0) )
	sc_pir_others
ROM_END

ROM_START( sc4pirf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1303p11.hi", 0x00000, 0x080000, CRC(6ce5b2fc) SHA1(53bc0da9f15934d5e854311056f8dc27fd13be6a) )
	ROM_LOAD16_BYTE( "p1303p11.lo", 0x00001, 0x080000, CRC(674be85d) SHA1(a38e6c2e04d03edf90b51d5c637f46a154798484) )
	sc_pir_others
ROM_END

ROM_START( sc4pirg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1303s11.hi", 0x00000, 0x080000, CRC(934c2667) SHA1(829ce31538491f06d1dbe68481a35fbd9923b08b) )
	ROM_LOAD16_BYTE( "p1303s11.lo", 0x00001, 0x080000, CRC(5bb368be) SHA1(bdd07f05f880df8fa89d4734810ce0836f9cb5b3) )
	sc_pir_others
ROM_END


ROM_START( sc4pen1 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402474.lo", 0x00001, 0x080000, CRC(962934e9) SHA1(de067a858b4501d4691e9972dd2bd2828ce4ba8a) )
	ROM_LOAD16_BYTE( "95402475.hi", 0x00000, 0x080000, CRC(e5a40536) SHA1(dc5861550e5b29635a8aa351b58acb18f3c3afb1) )
	sc_pen1_others
ROM_END

ROM_START( sc4pen1a )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402490.lo", 0x00001, 0x080000, CRC(bb271055) SHA1(19d7874592e9808df2b52ef13e0967c9bc6b2334) )
	ROM_LOAD16_BYTE( "95402491.hi", 0x00000, 0x080000, CRC(2adfee72) SHA1(825c0ca523ad27aab3dd13cdd35a25ecf15ed607) )
	sc_pen1_others
ROM_END

ROM_START( sc4pen1b )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403474.lo", 0x00001, 0x080000, CRC(78026e40) SHA1(658a45827c4bf697fa67041a532eb6e2d36cb2d1) )
	ROM_LOAD16_BYTE( "95403475.hi", 0x00000, 0x080000, CRC(cf1ff33b) SHA1(b13b29cc0cfc56ed003d8efb17b4113d03a78a80) )
	sc_pen1_others
ROM_END

ROM_START( sc4pen1c )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403490.lo", 0x00001, 0x080000, CRC(6aa7ae48) SHA1(30d56b86d648843ebeda8ed2f0e360ead6790e1e) )
	ROM_LOAD16_BYTE( "95403491.hi", 0x00000, 0x080000, CRC(352633ff) SHA1(865d634fddc9467aa52f49bdc13bb8b38194e6a6) )
	sc_pen1_others
ROM_END

ROM_START( sc4pen1d )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p_enemy.hi", 0x00000, 0x080000, CRC(394b222b) SHA1(1bc185c4518f04705438266f42ec74deed44373c) )
	ROM_LOAD16_BYTE( "p_enemy.lo", 0x00001, 0x080000, CRC(03db99ce) SHA1(2880db69e5fc12ff2fc1eccd1e9198c8b28f23d2) )
	sc_pen1_others
ROM_END


ROM_START( sc4quart )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406524.lo", 0x00001, 0x080000, CRC(99a0ae2c) SHA1(a1021485036741c14cb3de0a464795ee3eb7959d) )
	ROM_LOAD16_BYTE( "95406525.hi", 0x00000, 0x080000, CRC(a8ce8629) SHA1(ae892b70d64f0e0e696f66c4931e77c6ecdda088) )
	sc_quart_others
ROM_END

ROM_START( sc4quarta )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406794.lo", 0x00001, 0x080000, CRC(ecc36997) SHA1(1791fc75463cbc23beb2cb1284af207c6c2bcde6) )
	ROM_LOAD16_BYTE( "95406795.hi", 0x00000, 0x080000, CRC(bc4a70f6) SHA1(15774d64073228a19db0bb68b6783b0125c02074) )
	sc_quart_others
ROM_END

ROM_START( sc4quartb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406822.lo", 0x00001, 0x080000, CRC(f645320f) SHA1(d70d8d002e004f943d4862dd6f5a039f4ea153bc) )
	ROM_LOAD16_BYTE( "95406823.hi", 0x00000, 0x080000, CRC(a43f6610) SHA1(2f2be5da9122ab1384c0f8260ec284dcd4f3b8c5) )
	sc_quart_others
ROM_END

ROM_START( sc4quartc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406826.lo", 0x00001, 0x080000, CRC(b558e625) SHA1(588d70decefbf0e509c1347a158d7d31ac04dc5d) )
	ROM_LOAD16_BYTE( "95406827.hi", 0x00000, 0x080000, CRC(50b760fe) SHA1(c6e34e6e42ad9f72eb4563226579a3c29ddccc78) )
	sc_quart_others
ROM_END

ROM_START( sc4quartd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406832.lo", 0x00001, 0x080000, CRC(daa6c09c) SHA1(3cdc353d61034e1ae3acb0d9a53fd848ae80a94a) )
	ROM_LOAD16_BYTE( "95406833.hi", 0x00000, 0x080000, CRC(adbc76a9) SHA1(25c3591dc1d56864eaa7e966d67a8157c3a21675) )
	sc_quart_others
ROM_END

ROM_START( sc4quarte )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406834.lo", 0x00001, 0x080000, CRC(ed0fb304) SHA1(6c1f9e670748eef170adaaf49f726b2275be79ad) )
	ROM_LOAD16_BYTE( "95406835.hi", 0x00000, 0x080000, CRC(60c0223a) SHA1(138d291e49c13075c9ca98a5eceffbc24f611341) )
	sc_quart_others
ROM_END

ROM_START( sc4quartf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406862.lo", 0x00001, 0x080000, CRC(0fcda4b1) SHA1(bea31ba740dbf2180592fb063b9892df9bee461a) )
	ROM_LOAD16_BYTE( "95406863.hi", 0x00000, 0x080000, CRC(bd74a605) SHA1(15e2ef668ce18dcacb8b74d62b46c75a75c1c49e) )
	sc_quart_others
ROM_END

ROM_START( sc4quartg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406900.lo", 0x00001, 0x080000, CRC(c074cd8a) SHA1(bc12dff4678e74dd5cdf14134ba9f227ba376d88) )
	ROM_LOAD16_BYTE( "95406901.hi", 0x00000, 0x080000, CRC(732da92f) SHA1(495a79dedf6eb3dde04a70a4fd850670d8f1708a) )
	sc_quart_others
ROM_END

ROM_START( sc4quarth )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406918.lo", 0x00001, 0x080000, CRC(cf430afd) SHA1(9852252a4593d6da37c9a03a8d77813eb0e495c3) )
	ROM_LOAD16_BYTE( "95406919.hi", 0x00000, 0x080000, CRC(04e43a9d) SHA1(04c2c3ff9269a96367ce8c9316347f51cdf5f310) )
	sc_quart_others
ROM_END

ROM_START( sc4quarti )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406926.lo", 0x00001, 0x080000, CRC(1e43f8e0) SHA1(3dc4ef31d09907a9267b026359300fea0becf5b6) )
	ROM_LOAD16_BYTE( "95406927.hi", 0x00000, 0x080000, CRC(672b9510) SHA1(d7eeff4d850ffb04bbb1efc2d073af351a42ee99) )
	sc_quart_others
ROM_END

ROM_START( sc4quartj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408008.lo", 0x00001, 0x080000, CRC(d7f437d9) SHA1(1edf3b1197757a5fa7268affc8a4bedd0a477d48) )
	ROM_LOAD16_BYTE( "95408009.hi", 0x00000, 0x080000, CRC(2d7dbaf6) SHA1(807baec953ee67af2b52d9637247d8062c446140) )
	sc_quart_others
ROM_END

ROM_START( sc4quartk )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408010.lo", 0x00001, 0x080000, CRC(80c9c166) SHA1(7c329ba6d4582405a7faf26a45b27700595eddbe) )
	ROM_LOAD16_BYTE( "95408011.hi", 0x00000, 0x080000, CRC(50d454b0) SHA1(60052ffe0f8108dac657bb1015a4e6e713b459eb) )
	sc_quart_others
ROM_END

ROM_START( sc4quartl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408088.lo", 0x00001, 0x080000, CRC(72f48774) SHA1(b67b26c3eccfdef7c4b105842ab7aeee875b2a01) )
	ROM_LOAD16_BYTE( "95408089.hi", 0x00000, 0x080000, CRC(bb69a047) SHA1(b83e411b90261948e1fba0c71753d7252ba59e9f) )
	sc_quart_others
ROM_END

ROM_START( sc4quartm )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408214.lo", 0x00001, 0x080000, CRC(6232bc32) SHA1(30b03f7b66b7c2b7fe56ae99a0e845a2c53758a9) )
	ROM_LOAD16_BYTE( "95408215.hi", 0x00000, 0x080000, CRC(b516cbb4) SHA1(8fd2d9920186454a3c1dc75968cf6b42770f15c9) )
	sc_quart_others
ROM_END


ROM_START( sc4qmodo )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "69000164.odd", 0x00001, 0x080000, CRC(7a881601) SHA1(e8bd5aa7c38fda69afbefb4234bb2773e3d3d4cb) )
	ROM_LOAD16_BYTE( "69000165.evn", 0x00000, 0x080000, CRC(d8d3e1f6) SHA1(39cd99c7bc7e148b9b74c1e0e24ccc324be9b391) )
	sc_qmodo_others
ROM_END

ROM_START( sc4qmodoa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "69001164.odd", 0x00001, 0x080000, CRC(d2e6460f) SHA1(46f8a94ab5d50bb6392deb674fce46398901bd0d) )
	ROM_LOAD16_BYTE( "69001165.evn", 0x00000, 0x080000, CRC(53a391eb) SHA1(7cd7d049efcc8f6d73392565c777a2fa97214dfc) )
	sc_qmodo_others
ROM_END

ROM_START( sc4qmodob )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "96000164.odd", 0x00001, 0x080000, CRC(7458baf6) SHA1(67b09ea4e5dd084a70e704651033344e9753091f) )
	ROM_LOAD16_BYTE( "96000165.evn", 0x00000, 0x080000, CRC(e5fbe12e) SHA1(445636889a95f4ef5c6e3f7edae9d232677eef6e) )
	sc_qmodo_others
ROM_END

ROM_START( sc4qmodoc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "96001164.odd", 0x00001, 0x080000, CRC(dc36eaf8) SHA1(67b28c9c18b14c143f6596b7dd4caaccd649ae76) )
	ROM_LOAD16_BYTE( "96001165.evn", 0x00000, 0x080000, CRC(6e8b9133) SHA1(d220ffdb192807f55800d50fdd859a1b96570384) )
	sc_qmodo_others
ROM_END

ROM_START( sc4qmodod )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "96001166.odd", 0x00001, 0x080000, CRC(b51e0720) SHA1(2acf3dd03846c40acaf5f59a6527b8f2dc16b627) )
	ROM_LOAD16_BYTE( "96001167.evn", 0x00000, 0x080000, CRC(f09591d1) SHA1(604dfd124ec01806edd1abf69c15cc77fdfa61a9) )
	sc_qmodo_others
ROM_END


ROM_START( sc4quidr )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406420.odd", 0x00001, 0x080000, CRC(b7ddca3d) SHA1(58c3b2cd9da48af451631f761df7e279f1dc719d) )
	ROM_LOAD16_BYTE( "95406421.evn", 0x00000, 0x080000, CRC(022cde42) SHA1(17cb42c98dc6eab5d3e0d11bd4e4f07e6164a648) )
	sc_quidr_others
ROM_END

ROM_START( sc4quidra )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407420.odd", 0x00001, 0x080000, CRC(4a5fa168) SHA1(b8d2ace1d5e96c897d215ac1a3c9c97d6d9cb904) )
	ROM_LOAD16_BYTE( "95407421.evn", 0x00000, 0x080000, CRC(a66605b1) SHA1(b0a0b738bd24c7d6ad407334b802aaa88bda99b0) )
	sc_quidr_others
ROM_END

ROM_START( sc4quidrb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408148.odd", 0x00001, 0x080000, CRC(108bc690) SHA1(2cca06fae553df95c7cd26d1d764be23bba1f746) )
	ROM_LOAD16_BYTE( "95408149.evn", 0x00000, 0x080000, CRC(17a4f66e) SHA1(b9771298a0c31d8cff6123b05ef45600c9670ca9) )
	sc_quidr_others
ROM_END

ROM_START( sc4quidrc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409148.odd", 0x00001, 0x080000, CRC(66c866ba) SHA1(43ed90a240b0cc143d0dbb6616fcab9704e2cedd) )
	ROM_LOAD16_BYTE( "95409149.evn", 0x00000, 0x080000, CRC(6211bb9c) SHA1(344b3f9c3d653f30af242e12df2f1a23334d59b9) )
	sc_quidr_others
ROM_END




ROM_START( sc4redad )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408932.lo", 0x00001, 0x080000, CRC(d7b60d0a) SHA1(b4dc76c7e29a8da1cf67f9afc1988d25be560829) )
	ROM_LOAD16_BYTE( "95408933.hi", 0x00000, 0x080000, CRC(ad81c654) SHA1(e9db646056cce57d05b8ca59a8ae3a5d9b03bff9) )
	sc_redad_others
ROM_END


ROM_START( sc4redada )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406366.lo", 0x00001, 0x080000, CRC(b1e1d63b) SHA1(920f1ab872bcd6c594569b7272bcfdeb59607cd1) )
	ROM_LOAD16_BYTE( "95406367.hi", 0x00000, 0x080000, CRC(c11107ee) SHA1(a8cebcbc82787d8ccfa07795821f04364ae9ba94) )
	sc_redal_others
ROM_END


ROM_START( sc4rdrag )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414714.odd", 0x00001, 0x080000, CRC(465e2488) SHA1(0d70c1cc2549a60871f0d80cebc8674a042411ac) )
	ROM_LOAD16_BYTE( "95414715.evn", 0x00000, 0x080000, CRC(77ab8399) SHA1(913f02c2242b473348a146adf8a2900b76e587b6) )
	sc_rdrag_others
ROM_END

ROM_START( sc4rdraga )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415714.odd", 0x00001, 0x080000, CRC(6db75b2e) SHA1(18b7a9b9afa8e7cdf505bf28c68cd63e44588cd9) )
	ROM_LOAD16_BYTE( "95415715.evn", 0x00000, 0x080000, CRC(8b13e696) SHA1(8d8f8d23ebd3de1f02968a753b8503ec7d4461a0) )
	sc_rdrag_others
ROM_END



ROM_START( sc4rdragc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416244.odd", 0x00001, 0x080000, CRC(425a82b4) SHA1(2b316ad7e71d339ca38d507800a6b641bf52ee9f) )
	ROM_LOAD16_BYTE( "95416245.evn", 0x00000, 0x080000, CRC(8a7e9138) SHA1(a49d461e46063cc00716b2d6db6ec7a742612a57) )
	sc_rdrag_others
ROM_END




ROM_START( sc4rdragf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417244.odd", 0x00001, 0x080000, CRC(c848d8dc) SHA1(e32416a6676a69e6e603746fada31851b0fae836) )
	ROM_LOAD16_BYTE( "95417245.evn", 0x00000, 0x080000, CRC(bcee571c) SHA1(6b8d95c74124565584c353fe15b037ffa8b78807) )
	sc_rdrag_others
ROM_END


ROM_START( sc4rdrcl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416070.odd", 0x00001, 0x080000, CRC(632a3587) SHA1(e67e1ffe0c667de68b57e638ff2325cbf96b9af9) )
	ROM_LOAD16_BYTE( "95416071.evn", 0x00000, 0x080000, CRC(3fa9ce1b) SHA1(aa4721b7b7228223819fb1eeef97e44cb692cf56) )
	sc_rdrcl_others
ROM_END

ROM_START( sc4rdrcla )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416414.odd", 0x00001, 0x080000, CRC(5a090bb8) SHA1(d03cec018bca0b9922d97be6a0140ac7da376496) )
	ROM_LOAD16_BYTE( "95416415.evn", 0x00000, 0x080000, CRC(a42c8f94) SHA1(0bdf520be9c2b0cb6fc2538563c9a4693951cb58) )
	sc_rdrcl_others
ROM_END

ROM_START( sc4rdrclb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417070.odd", 0x00001, 0x080000, CRC(26060bfd) SHA1(7e71d28525fc1198d2eb868a93cf2c3327df0322) )
	ROM_LOAD16_BYTE( "95417071.evn", 0x00000, 0x080000, CRC(4727e458) SHA1(64b9990dfabeb2b8580a62cfa0dc76746e622651) )
	sc_rdrcl_others
ROM_END


ROM_START( sc4ibiza )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400425.lo", 0x00001, 0x080000, CRC(0543b316) SHA1(25b7e9f9c581d68a2585c52bb6e7e3c15984833f) )
	ROM_LOAD16_BYTE( "95400426.hi", 0x00000, 0x080000, CRC(a70f478b) SHA1(b33b5d61ae4c1d5fa62007de132052b54e835f8e) )
	sc_ibiza_others
ROM_END

ROM_START( sc4ibizaa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400427.lo", 0x00001, 0x080000, CRC(532b9127) SHA1(44f229ff98ced01179d05846e244a61bda45609f) )
	ROM_LOAD16_BYTE( "95400428.hi", 0x00000, 0x080000, CRC(88931a42) SHA1(5eb01bc146c0fb8ea5f94f256ff026cdd9b33447) )
	sc_ibiza_others
ROM_END

ROM_START( sc4ibizab )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400845.lo", 0x00001, 0x080000, CRC(e328f5a0) SHA1(7bb30b505c04bcdc7e606e66320b9b9f3aa90f5e) )
	ROM_LOAD16_BYTE( "95400846.hi", 0x00000, 0x080000, CRC(c60d7bd0) SHA1(7d91a55d30feb6259b883aacd5cd064ee92cfa04) )
	sc_ibiza_others
ROM_END

ROM_START( sc4ibizac )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400847.lo", 0x00001, 0x080000, CRC(3d3f90a8) SHA1(b78a7d3714b6a8d1b5963911b512fe0ba0a61e12) )
	ROM_LOAD16_BYTE( "95400848.hi", 0x00000, 0x080000, CRC(c8fd1e4b) SHA1(ed5ed8acef9001765a9594ec4a02bf7a535b1423) )
	sc_ibiza_others
ROM_END

ROM_START( sc4ibizad )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401425.lo", 0x00001, 0x080000, CRC(0ceab636) SHA1(6bcafe81c7ea7b1ec514d8c3c039642e68c1aada) )
	ROM_LOAD16_BYTE( "95401426.hi", 0x00000, 0x080000, CRC(354a0c64) SHA1(c891916f594a36caad4eba3590b65a2bebf71da0) )
	sc_ibiza_others
ROM_END

ROM_START( sc4ibizae )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401427.lo", 0x00001, 0x080000, CRC(35ca2819) SHA1(fdcde2a0593af963961f10ed8f2d78b91dd462ba) )
	ROM_LOAD16_BYTE( "95401428.hi", 0x00000, 0x080000, CRC(39e93182) SHA1(087f5ac9616ea2aba0c24b0b9ec6a8db71ae65a9) )
	sc_ibiza_others
ROM_END

ROM_START( sc4ibizaf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401845.lo", 0x00001, 0x080000, CRC(72934f1d) SHA1(652c0c8c79b0c3e810d172d8773487783a90d3e9) )
	ROM_LOAD16_BYTE( "95401846.hi", 0x00000, 0x080000, CRC(57949297) SHA1(0b088286a6692951a3d214429324b947f0b7a7eb) )
	sc_ibiza_others
ROM_END

ROM_START( sc4ibizag )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401847.lo", 0x00001, 0x080000, CRC(482da4f3) SHA1(1fb12a3e4e10dea1e33fd4587790a30e1397b102) )
	ROM_LOAD16_BYTE( "95401848.hi", 0x00000, 0x080000, CRC(dc6c1430) SHA1(0a027f8f52b7bcc7a9608c2d67be10686c1060c5) )
	sc_ibiza_others
ROM_END

ROM_START( sc4ibizah )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402081.lo", 0x00001, 0x080000, CRC(2d5822a8) SHA1(a3e4e911f708a4a7c0790080aae60791585c43b3) )
	ROM_LOAD16_BYTE( "95402082.hi", 0x00000, 0x080000, CRC(9e066bbc) SHA1(8ec9df0ce0753e37fadce56a056a942b2393e9a0) )
	sc_ibiza_others
ROM_END

ROM_START( sc4ibizai )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403081.lo", 0x00001, 0x080000, CRC(877bab81) SHA1(40e13e96918bfae99db3e600ae82e1e23d04560c) )
	ROM_LOAD16_BYTE( "95403082.hi", 0x00000, 0x080000, CRC(fc61d8a5) SHA1(92ceb83d01a0a5dfde0e39732a4c57df553145ed) )
	sc_ibiza_others
ROM_END





ROM_START( sc4rhxcl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408050.lo", 0x00001, 0x080000, CRC(aa97ebde) SHA1(888bd0361a52928f5781fca6b1f2d0c4b19e715f) )
	ROM_LOAD16_BYTE( "95408051.hi", 0x00000, 0x080000, CRC(f2b1bb5b) SHA1(3d1e8afa800bb8a2808620246189a7b2cd5acc45) )
	sc_rhxcl_others
ROM_END

ROM_START( sc4rhxcla )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408380.lo", 0x00001, 0x080000, CRC(a6b34b32) SHA1(29008a8164f96704f4f28318c53bb1e7e5e7050d) )
	ROM_LOAD16_BYTE( "95408381.hi", 0x00000, 0x080000, CRC(1bdea091) SHA1(1125631cede992df92891715ff5b6447aaa4bcaa) )
	sc_rhxcl_others
ROM_END

ROM_START( sc4rhxclb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409050.lo", 0x00001, 0x080000, CRC(7ba8ed4e) SHA1(e940d0bf4f555e9089f74177f40dd2f6cf1f5e28) )
	ROM_LOAD16_BYTE( "95409051.hi", 0x00000, 0x080000, CRC(c91a1ace) SHA1(9b3018c397a759cf48e9417e62e6cf98d34f948a) )
	sc_rhxcl_others
ROM_END

ROM_START( sc4rhxclc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409380.lo", 0x00001, 0x080000, CRC(41686594) SHA1(a93702f7887b9fd8cfc69ab0ec8d76903e7bc094) )
	ROM_LOAD16_BYTE( "95409381.hi", 0x00000, 0x080000, CRC(1bdea091) SHA1(1125631cede992df92891715ff5b6447aaa4bcaa) )
	sc_rhxcl_others
ROM_END




ROM_START( sc4r2r )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404582.lo", 0x00001, 0x080000, CRC(e14d88a0) SHA1(a0096849a68403753a11ab41bd8271625bf5f068) )
	ROM_LOAD16_BYTE( "95404583.hi", 0x00000, 0x080000, CRC(fd06feea) SHA1(c634d39ec9e4d0587a52b5bf9cfd99e27027a2ef) )
	sc_r2r_others
ROM_END

ROM_START( sc4r2ra )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405582.lo", 0x00001, 0x080000, CRC(9eff3047) SHA1(366591ad30ebbc1b1ba42b965d006b97842f1745) )
	ROM_LOAD16_BYTE( "95405583.hi", 0x00000, 0x080000, CRC(bd6081ac) SHA1(089d47a2e6da1657680829f55d990a41b2b9144a) )
	sc_r2r_others
ROM_END

ROM_START( sc4r2rb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406140.lo", 0x00001, 0x080000, CRC(144a5fcb) SHA1(4e75518bfda43043bfb412aaa92c650b16bf5704) )
	ROM_LOAD16_BYTE( "95406141.hi", 0x00000, 0x080000, CRC(5607f264) SHA1(494ae1f1dc292e65ade7458ac86926062b578203) )
	sc_r2r_others
ROM_END

ROM_START( sc4r2rc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407140.lo", 0x00001, 0x080000, CRC(2e4ccefd) SHA1(8f7a9f18e3445dc3487814220f9dbe37aef2e9da) )
	ROM_LOAD16_BYTE( "95407141.hi", 0x00000, 0x080000, CRC(2150b14e) SHA1(6b15020f1c3ffb220be077cde3b1002f2838a756) )
	sc_r2r_others
ROM_END


ROM_START( sc4relcz )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404576.lo", 0x00001, 0x080000, CRC(b4a35013) SHA1(af54ef71e891708cf55f2fa0eb93fca1b184a174) )
	ROM_LOAD16_BYTE( "95404577.hi", 0x00000, 0x080000, CRC(d2f649d4) SHA1(bbb6b9f4f8b7f0844189dca0eff8c19e81772b7d) )
	sc_relcz_others
ROM_END

ROM_START( sc4relcza )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405576.lo", 0x00001, 0x080000, CRC(40a1b8e3) SHA1(a18ba3ad2336f407ba6e1a7f829edde5f5a8c357) )
	ROM_LOAD16_BYTE( "95405577.hi", 0x00000, 0x080000, CRC(6313342c) SHA1(5d103a557c48d9d63c259bfdae4277b5b0d140a6) )
	sc_relcz_others
ROM_END

ROM_START( sc4relczb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406022.lo", 0x00001, 0x080000, CRC(c44007bb) SHA1(7191f6982d53c1fb7e75f55bf47c292cfd8a42bb) )
	ROM_LOAD16_BYTE( "95406023.hi", 0x00000, 0x080000, CRC(f53ee5a5) SHA1(65f4556c8f8366cbfaedbb34b44077e2c1534e32) )
	sc_relcz_others
ROM_END

ROM_START( sc4relczc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407022.lo", 0x00001, 0x080000, CRC(0a34c0a3) SHA1(406c40db39f3d97cdf178936117144e37e69f497) )
	ROM_LOAD16_BYTE( "95407023.hi", 0x00000, 0x080000, CRC(5d71f597) SHA1(7905aac99abe5cf4e0fe7833c64998cd2c7658cd) )
	sc_relcz_others
ROM_END


ROM_START( sc4rotc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410882.lo", 0x00001, 0x080000, CRC(2a29ed0c) SHA1(319d34e88e65ea3c1f8c3686760561af6fea6342) )
	ROM_LOAD16_BYTE( "95410883.hi", 0x00000, 0x080000, CRC(a204a346) SHA1(281bff93f769c9d0b7168ef2c1ac8baca4255d8e) )
	sc_rotc_others
ROM_END

ROM_START( sc4rotca )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414572.lo", 0x00001, 0x080000, CRC(c5ffd5a6) SHA1(0abab951024fc537e13fa73e918af64ca84b2ce4) )
	ROM_LOAD16_BYTE( "95414573.hi", 0x00000, 0x080000, CRC(69eafaf6) SHA1(6088df4ef481c335d3f2922a226f4c9f79680618) )
	sc_rotc_others
ROM_END

ROM_START( sc4rotcb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414808.lo", 0x00001, 0x080000, CRC(dc600bd3) SHA1(399557efb637d0ed348d0fa6c4bdc49f6f07ce05) )
	ROM_LOAD16_BYTE( "95414809.hi", 0x00000, 0x080000, CRC(f8da204c) SHA1(060a8eb733a05fbe70aa520ba05b8983cdfcf73f) )
	sc_rotc_others
ROM_END

ROM_START( sc4rotcc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415572.lo", 0x00001, 0x080000, CRC(3840e0d9) SHA1(e17223ce9062bb8958dbc01721b6cb863947613c) )
	ROM_LOAD16_BYTE( "95415573.hi", 0x00000, 0x080000, CRC(0a2cb3a7) SHA1(26e35b7edf2fe9cdf35bd31e8fbf5d6aac805542) )
	sc_rotc_others
ROM_END

ROM_START( sc4rotcd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415808.lo", 0x00001, 0x080000, CRC(cfc11a14) SHA1(a2294bb40c5dc80434342c0557542ccd6a256415) )
	ROM_LOAD16_BYTE( "95415809.hi", 0x00000, 0x080000, CRC(bb60fda2) SHA1(1ec35aa55ad0894b974978e53ffb1d4e559221e0) )
	sc_rotc_others
ROM_END


ROM_START( sc4rvlnx )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408168.lo", 0x00001, 0x080000, CRC(f1c731cf) SHA1(f7b7b7ee0fe04eb38683b01c2be0d3a28175adc5) )
	ROM_LOAD16_BYTE( "95408169.hi", 0x00000, 0x080000, CRC(b3addfca) SHA1(2001653d5d11af59e5f6807e8f209310b4b16a54) )
	sc_rvlnx_others
ROM_END


ROM_START( sc4rvl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406362.lo", 0x00001, 0x080000, CRC(e9a34f99) SHA1(49ea9a2187c6077800b6f44e0cc9a63a3f93b508) )
	ROM_LOAD16_BYTE( "95406363.hi", 0x00000, 0x080000, CRC(4da07a11) SHA1(fa0154e163e3702cb434295ac6ba8961ffa3819c) )
	sc_rvl_others
ROM_END


ROM_START( sc4revo )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404540.lo", 0x00001, 0x080000, CRC(f8ea2471) SHA1(35f1d139c3963c07025cccec1afc85d7a151bf56) )
	ROM_LOAD16_BYTE( "95404541.hi", 0x00000, 0x080000, CRC(0c67e582) SHA1(f6818350f9f123efc9a215f167dce43a90104f67) )
	sc_revo_others
ROM_END

ROM_START( sc4revoa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404542.lo", 0x00001, 0x080000, CRC(9ba820d6) SHA1(21787ba15a288a672d28014f0f3553c619dd20b4) )
	ROM_LOAD16_BYTE( "95404543.hi", 0x00000, 0x080000, CRC(9e622e0f) SHA1(c262765d73006239fd5084ba6ec1a21146464f9a) )
	sc_revo_others
ROM_END

ROM_START( sc4revob )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405540.lo", 0x00001, 0x080000, CRC(bec4cd27) SHA1(d3ebf89d05c251dd202f21fa729f637d80ac1a96) )
	ROM_LOAD16_BYTE( "95405541.hi", 0x00000, 0x080000, CRC(f17e7ac5) SHA1(dad87fa51363d1417e6d1759817d300c8e57fdad) )
	sc_revo_others
ROM_END

ROM_START( sc4revoc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405542.lo", 0x00001, 0x080000, CRC(fbf13fcc) SHA1(88109812bb8b41282aa3375af83b18a52231b438) )
	ROM_LOAD16_BYTE( "95405543.hi", 0x00000, 0x080000, CRC(d29eccb0) SHA1(ad5360ca3ac4b2506619ca9a6ecb0e63a83bfc80) )
	sc_revo_others
ROM_END

ROM_START( sc4revod )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406144.lo", 0x00001, 0x080000, CRC(983403fb) SHA1(a8eb30a37e898ca1908d83cc503d94ebf92d6172) )
	ROM_LOAD16_BYTE( "95406145.hi", 0x00000, 0x080000, CRC(756aad16) SHA1(58a5b98c987eeaf2292679cefc91248d349d3a08) )
	sc_revo_others
ROM_END

ROM_START( sc4revoe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407144.lo", 0x00001, 0x080000, CRC(a4b8628d) SHA1(e8cff42fdace858f54422999078afd1403fdc021) )
	ROM_LOAD16_BYTE( "95407145.hi", 0x00000, 0x080000, CRC(84e9afa8) SHA1(128bd2dc01c7aabaa36a3be379db8073c93482e9) )
	sc_revo_others
ROM_END


ROM_START( sc4rich )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402930.lo", 0x00001, 0x080000, CRC(94654103) SHA1(7c4cd1df0530e4d6f76c3f59849a356f7b6aaab6) )
	ROM_LOAD16_BYTE( "95402931.hi", 0x00000, 0x080000, CRC(c8d8e1cb) SHA1(07853c8f8135366cc85ad8cb3b98de9afefeaf46) )
	sc_rich_others
ROM_END

ROM_START( sc4richa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402934.lo", 0x00001, 0x080000, CRC(250add32) SHA1(0b0c2c339ca08240706630543d60041568237f5e) )
	ROM_LOAD16_BYTE( "95402935.hi", 0x00000, 0x080000, CRC(ec58cd31) SHA1(d58e84e6656e0cc82f88d79f5aa009a6b644b18f) )
	sc_rich_others
ROM_END

ROM_START( sc4richb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402936.lo", 0x00001, 0x080000, CRC(c0271a19) SHA1(243556b9e661f9cebd08cc3985a4430f30e8abcd) )
	ROM_LOAD16_BYTE( "95402937.hi", 0x00000, 0x080000, CRC(f346abb2) SHA1(b1752d1d91eda82e90c0dc9dc21df8ac4f33a369) )
	sc_rich_others
ROM_END

ROM_START( sc4richc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402944.lo", 0x00001, 0x080000, CRC(6dfeaecf) SHA1(627eab4076d323f727a88f129fdf10336debe03e) )
	ROM_LOAD16_BYTE( "95402945.hi", 0x00000, 0x080000, CRC(60fe347d) SHA1(7e8d7373a9a294980c427ce872190eed0ac18ebf) )
	sc_rich_others
ROM_END

ROM_START( sc4richd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403930.lo", 0x00001, 0x080000, CRC(f5e21802) SHA1(eb6cac61f63e8bd4f0d267aa9de0a0eb4d46bdfd) )
	ROM_LOAD16_BYTE( "95403931.hi", 0x00000, 0x080000, CRC(d982b154) SHA1(fb22115bf429569887cc9d6e4ed42efc002ee8c5) )
	sc_rich_others
ROM_END

ROM_START( sc4riche )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403934.lo", 0x00001, 0x080000, CRC(234c2e0c) SHA1(a4bea103d2e14b76586d9793bc63824b4116931d) )
	ROM_LOAD16_BYTE( "95403935.hi", 0x00000, 0x080000, CRC(e535c4c9) SHA1(76dcc670bf06f28e48266e7c76a3714fcbf15d09) )
	sc_rich_others
ROM_END

ROM_START( sc4richf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403936.lo", 0x00001, 0x080000, CRC(335e3640) SHA1(e42384a3e6e848aa189e612409e4922343b99c8f) )
	ROM_LOAD16_BYTE( "95403937.hi", 0x00000, 0x080000, CRC(ab912677) SHA1(4b39d83f14c92f5c7dbb2656d2029d0192a1de58) )
	sc_rich_others
ROM_END

ROM_START( sc4richg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403944.lo", 0x00001, 0x080000, CRC(53b9846a) SHA1(bbd43ecb6b4cad3f582b8535d39c8a0416342112) )
	ROM_LOAD16_BYTE( "95403945.hi", 0x00000, 0x080000, CRC(591c7edf) SHA1(92cd0961760499ae363f4dec7bd8bd35b1ca3f26) )
	sc_rich_others
ROM_END

ROM_START( sc4richh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406452.lo", 0x00001, 0x080000, CRC(cbfa4c42) SHA1(abd848cda7f1bf52829fb32933685147e88b7a05) )
	ROM_LOAD16_BYTE( "95406453.hi", 0x00000, 0x080000, CRC(f0230716) SHA1(757926182944b4181986dc577075ee842e716b4c) )
	sc_rich_others
ROM_END

ROM_START( sc4richi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406454.lo", 0x00001, 0x080000, CRC(6daea98f) SHA1(3a7b5bfa005de416e3e1d496765db6db0383bda2) )
	ROM_LOAD16_BYTE( "95406455.hi", 0x00000, 0x080000, CRC(3c3a3e43) SHA1(6a19c672b60975ed63b3cde0dec9a023ee1b18c2) )
	sc_rich_others
ROM_END

ROM_START( sc4richj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407452.lo", 0x00001, 0x080000, CRC(e0e80388) SHA1(40a80ba6930dde57cc15a8371a5a38ef2a851423) )
	ROM_LOAD16_BYTE( "95407453.hi", 0x00000, 0x080000, CRC(8a39fca7) SHA1(88dbb4a4d01b90bbc8c8c1497dbf55f0bf408556) )
	sc_rich_others
ROM_END

ROM_START( sc4richk )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407454.lo", 0x00001, 0x080000, CRC(0e1ef2e5) SHA1(dc8e037b6b01e014cf0ea2f5cf92b6adfa2d92a6) )
	ROM_LOAD16_BYTE( "95407455.hi", 0x00000, 0x080000, CRC(d52b4419) SHA1(99376a27b4439a2d6dab47c689f06d735d41e102) )
	sc_rich_others
ROM_END

ROM_START( sc4richl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "r_geez.hi", 0x00000, 0x080000, CRC(c1c510bf) SHA1(6b4483e2b732f6b212087c019de2e870b777dd97) )
	ROM_LOAD16_BYTE( "r_geez.lo", 0x00001, 0x080000, CRC(c34f5939) SHA1(a2b0ebafee568ff1a9534c59afe50495c1815d2f) )
	sc_rich_others
ROM_END


ROM_START( sc4rio )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406350.lo", 0x00001, 0x080000, CRC(0f036800) SHA1(3fda07eaf2f325e50e0532e7dc2e4431eb3546ee) )
	ROM_LOAD16_BYTE( "95406351.hi", 0x00000, 0x080000, CRC(90fe320b) SHA1(2ba801956df119211cdb2d4e42b94fffaaf8dad8) )
	sc_rio_others
ROM_END


ROM_START( sc4rttt )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408382.lo", 0x00001, 0x080000, CRC(c11a966b) SHA1(cb6f942426910c550bdc0d86f3bfa8579c35f081) )
	ROM_LOAD16_BYTE( "95408383.hi", 0x00000, 0x080000, CRC(9f7a5fca) SHA1(62405e5d4c3279b5a451aa507c8c6feb9b14e5c8) )
	sc_rttt_others
ROM_END

ROM_START( sc4rttta )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409382.lo", 0x00001, 0x080000, CRC(5b10f3fc) SHA1(5601f1dbe04d6db7a4af2be395b0b4fbb9e1f5df) )
	ROM_LOAD16_BYTE( "95409383.hi", 0x00000, 0x080000, CRC(76f2caf6) SHA1(d13c034f056c170393c0466fc14ee0179cfdbb8b) )
	sc_rttt_others
ROM_END

ROM_START( sc4rtttb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410486.lo", 0x00001, 0x080000, CRC(146ec9ff) SHA1(88d53838b575c37cb492aa7727ac3b5fb6d32156) )
	ROM_LOAD16_BYTE( "95410487.hi", 0x00000, 0x080000, CRC(50b19298) SHA1(e37a83f644eab65fd7d876f9fa4561f0cbe58067) )
	sc_rttt_others
ROM_END

ROM_START( sc4rtttc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410488.lo", 0x00001, 0x080000, CRC(620e861a) SHA1(2416baf3d195134ea34b135ba1c2dc710ec2e1b8) )
	ROM_LOAD16_BYTE( "95410489.hi", 0x00000, 0x080000, CRC(0f6cd4b6) SHA1(fbfb6380186de3be380e28cdecec8bfc1ea29a89) )
	sc_rttt_others
ROM_END

ROM_START( sc4rtttd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411486.lo", 0x00001, 0x080000, CRC(85c38552) SHA1(ed7c9b73fa75f7e589dd734ec584a48101557798) )
	ROM_LOAD16_BYTE( "95411487.hi", 0x00000, 0x080000, CRC(e3b6e1b8) SHA1(c85926dab78169a445894e173f4661d0f2bc05dc) )
	sc_rttt_others
ROM_END

ROM_START( sc4rttte )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411488.lo", 0x00001, 0x080000, CRC(de68cc2f) SHA1(c0ae038a42f2ba227f0195813cb76a2265808600) )
	ROM_LOAD16_BYTE( "95411489.hi", 0x00000, 0x080000, CRC(16c6b97c) SHA1(90223eb9c873d61a9ffa605ddcbdc369c7c87305) )
	sc_rttt_others
ROM_END


ROM_START( sc4roksc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410426.odd", 0x00001, 0x080000, CRC(75b7f366) SHA1(b09cb427f847bae090fcf165ccb80d35ead6bc81) )
	ROM_LOAD16_BYTE( "95410427.evn", 0x00000, 0x080000, CRC(af854394) SHA1(9909840c1c7ccfb3874af0a4be562e7dd8ae6e2a) )
	sc_roksc_others
ROM_END

ROM_START( sc4roksca )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410784.odd", 0x00001, 0x080000, CRC(c1ca7dcb) SHA1(cbb389e6aa0ff50c4c2880c78708345c8a207bec) )
	ROM_LOAD16_BYTE( "95410785.evn", 0x00000, 0x080000, CRC(d5cbba23) SHA1(014d8057f1d28da72c6d5e0169bc73feeed1e918) )
	sc_roksc_others
ROM_END

ROM_START( sc4rokscb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411426.odd", 0x00001, 0x080000, CRC(6d5b8ab9) SHA1(3a223084b6542bc78956ab32a368bfb9b4ee6034) )
	ROM_LOAD16_BYTE( "95411427.evn", 0x00000, 0x080000, CRC(5e06412a) SHA1(1b30fc33418b351efc9cffad9f6fc59833ca899c) )
	sc_roksc_others
ROM_END

ROM_START( sc4rokscc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411784.odd", 0x00001, 0x080000, CRC(d0720971) SHA1(78787976966a00875031130a81d4be6cd5aab866) )
	ROM_LOAD16_BYTE( "95411785.evn", 0x00000, 0x080000, CRC(e265d077) SHA1(a7dd5b0025338d76aa6df015a07192829a3c17be) )
	sc_roksc_others
ROM_END


ROM_START( sc4rogds )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406306.odd", 0x00001, 0x080000, CRC(c552e1fb) SHA1(dbda6881e47e526485225a088103f568cd38b36b) )
	ROM_LOAD16_BYTE( "95406307.evn", 0x00000, 0x080000, CRC(b9a0000b) SHA1(31d21d294331bc0d7c46aaf447fa3562286bc1cc) )
	sc_rogds_others
ROM_END

ROM_START( sc4rogdsa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406658.odd", 0x00001, 0x080000, CRC(87cc73ac) SHA1(dba0d665329d78e028290951657676671bb36317) )
	ROM_LOAD16_BYTE( "95406659.evn", 0x00000, 0x080000, CRC(efdbeb0b) SHA1(0b0d104c6662a5801d33925d76b7b1b0afba149f) )
	sc_rogds_others
ROM_END

ROM_START( sc4rogdsb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407658.odd", 0x00001, 0x080000, CRC(2eeed088) SHA1(399f4f3e3e71ff65086fd2ed73f2a517a3980aee) )
	ROM_LOAD16_BYTE( "95407659.evn", 0x00000, 0x080000, CRC(f80915e0) SHA1(5564c47c5668f97d594840856a5f8c1fe0438ebe) )
	sc_rogds_others
ROM_END

ROM_START( sc4rogdsc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407808.odd", 0x00001, 0x080000, CRC(83bd9eca) SHA1(09dcb17b5326ba1d29a6ebf2ec7c990f7e18c21a) )
	ROM_LOAD16_BYTE( "95407809.evn", 0x00000, 0x080000, CRC(827c0acc) SHA1(3322d5d4598a0680f915347f2c7e748ddcbc78b2) )
	sc_rogds_others
ROM_END

ROM_START( sc4rogdsd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408132.odd", 0x00001, 0x080000, CRC(aa609a02) SHA1(2a7c9a7017b383f2642671ad912273a5182afa51) )
	ROM_LOAD16_BYTE( "95408133.evn", 0x00000, 0x080000, CRC(fa7af483) SHA1(6f1ab822522f8e7867b23727648dd89c93a594da) )
	sc_rogds_others
ROM_END

ROM_START( sc4rogdse )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409042.odd", 0x00001, 0x080000, CRC(8242a18d) SHA1(fe4024530d05479e6cff184b0532d0b94d500054) )
	ROM_LOAD16_BYTE( "95409043.evn", 0x00000, 0x080000, CRC(af0d81c2) SHA1(0a04f02a31fafe883ef16ee2c408a763fbbbbe0f) )
	sc_rogds_others
ROM_END

ROM_START( sc4rogdsf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409132.odd", 0x00001, 0x080000, CRC(40ec2c64) SHA1(c3d8e3ca0e49d996492b4482fc999489b26e2177) )
	ROM_LOAD16_BYTE( "95409133.evn", 0x00000, 0x080000, CRC(9bc1e532) SHA1(81a976d4df906e408d10134f4f183d8cf1eb52f4) )
	sc_rogds_others
ROM_END

ROM_START( sc4rogdsg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409324.odd", 0x00001, 0x080000, CRC(fcd86b8c) SHA1(f8f682a502ad45e79ff982afc77e08801e071cbe) )
	ROM_LOAD16_BYTE( "95409325.evn", 0x00000, 0x080000, CRC(9958fd88) SHA1(298b2c0852339ef7589c88fbb4400277c18818a4) )
	sc_rogds_others
ROM_END


ROM_START( sc4rmo )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402972.lo", 0x00001, 0x080000, CRC(0d12366a) SHA1(311bc72e4684428ec393c04d5b51dbb6f497d548) )
	ROM_LOAD16_BYTE( "95402973.hi", 0x00000, 0x080000, CRC(77ecbc6f) SHA1(1c8b5de1c489b700517313ef8ed6dd157cc20b82) )
	sc_rmo_others
ROM_END

ROM_START( sc4rmoa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403972.lo", 0x00001, 0x080000, CRC(89eb3046) SHA1(480f36e938b69d4df50754267e7d251c549d437f) )
	ROM_LOAD16_BYTE( "95403973.hi", 0x00000, 0x080000, CRC(3b8f08e8) SHA1(3577a9304a6e5ee69c508dacb037f76325f40020) )
	sc_rmo_others
ROM_END


ROM_START( sc4rt )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404322.lo", 0x00001, 0x080000, CRC(53980334) SHA1(8f00235ababdf087aa892047dbdca974043729fd) )
	ROM_LOAD16_BYTE( "95404323.hi", 0x00000, 0x080000, CRC(f74bbef3) SHA1(86a5c120c99771c2ee3197718cb424d0691a39c5) )
	sc_rt_others
ROM_END

ROM_START( sc4rta )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405322.lo", 0x00001, 0x080000, CRC(ab08bdfc) SHA1(a721dffb8ece74bf6552f2ed3e2e886788a648e7) )
	ROM_LOAD16_BYTE( "95405323.hi", 0x00000, 0x080000, CRC(f536e613) SHA1(9457a7897fdc2c20725426a12d9cab0b07be4e6e) )
	sc_rt_others
ROM_END

ROM_START( sc4rtb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405324.lo", 0x00001, 0x080000, CRC(1ee0bf92) SHA1(a8c402468e20743baa5b4c35c7b0e2cf71cc01f9) )
	ROM_LOAD16_BYTE( "95405325.hi", 0x00000, 0x080000, CRC(c96cfed2) SHA1(c48dbcd885d19d06dbec30d794b9f63d9d902f6d) )
	sc_rt_others
ROM_END

ROM_START( sc4rtc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406146.lo", 0x00001, 0x080000, CRC(76cba54b) SHA1(d9017ed89d3c4e13544048e331cf863995f5ae41) )
	ROM_LOAD16_BYTE( "95406147.hi", 0x00000, 0x080000, CRC(b5ac6ae6) SHA1(aeef1cc4aee31cfa2f8b9264e969c37655d98a39) )
	sc_rt_others
ROM_END

ROM_START( sc4rtd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407146.lo", 0x00001, 0x080000, CRC(b30a7f7b) SHA1(3323295e2bb342dfb5246f1c59244d061a0f7c8f) )
	ROM_LOAD16_BYTE( "95407147.hi", 0x00000, 0x080000, CRC(5d1a53aa) SHA1(66d9184c67566745c0db5b5c453a612f3af5b148) )
	sc_rt_others
ROM_END


ROM_START( sc4rtclb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404716.lo", 0x00001, 0x080000, CRC(24355f67) SHA1(e1f3367cc8bb59e1906a1c8ae9ebe355ac3f2183) )
	ROM_LOAD16_BYTE( "95404717.hi", 0x00000, 0x080000, CRC(f285afa5) SHA1(1ad30b6a4fdb045ac259e5f5eaae0a7383cdb804) )
	sc_rtclb_others
ROM_END

ROM_START( sc4rtclba )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405716.lo", 0x00001, 0x080000, CRC(e0b46a90) SHA1(47e7abef29065f9cb0a41aa5b584785952532de7) )
	ROM_LOAD16_BYTE( "95405717.hi", 0x00000, 0x080000, CRC(854dbab7) SHA1(83bb4503adec83f415a09f7f0303577e8caee174) )
	sc_rtclb_others
ROM_END


ROM_START( sc4rollo )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400265.lo", 0x00001, 0x080000, CRC(d0d8bd41) SHA1(bf7e223cbbf7a79d7c62b389a492d0b1842917c6) )
	ROM_LOAD16_BYTE( "95400266.hi", 0x00000, 0x080000, CRC(d5472cf1) SHA1(00b055401fbbf46f12919ec13c34cef2e14d4d1f) )
	sc_rollo_others
ROM_END

ROM_START( sc4rolloa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400267.lo", 0x00001, 0x080000, CRC(da2b4d37) SHA1(a56074f1ba0985d3b20c2e6ffab8e66430f98da7) )
	ROM_LOAD16_BYTE( "95400268.hi", 0x00000, 0x080000, CRC(e91c87cb) SHA1(203e0365b692d445f3c56ee0858e5fae69d53b75) )
	sc_rollo_others
ROM_END

ROM_START( sc4rollob )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400667.lo", 0x00001, 0x080000, CRC(067efd8e) SHA1(ed4c793a95b859d882ef1018a9b5a8a7d2e00443) )
	ROM_LOAD16_BYTE( "95400668.hi", 0x00000, 0x080000, CRC(e1980c74) SHA1(a94fd00cf1750f2051cf2769969d41b8e4e56be3) )
	sc_rollo_others
ROM_END

ROM_START( sc4rolloc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401265.lo", 0x00001, 0x080000, CRC(ce0a645e) SHA1(83f2a9e3de766ae6a7eb54895f519e8185bf4251) )
	ROM_LOAD16_BYTE( "95401266.hi", 0x00000, 0x080000, CRC(29601fc6) SHA1(d764143252da6e5fe9bff960558f0441f0a6baa9) )
	sc_rollo_others
ROM_END

ROM_START( sc4rollod )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401267.lo", 0x00001, 0x080000, CRC(4df4bcd5) SHA1(f4c2f193b0d0d11b62107dd7ddbac631a7cc70f7) )
	ROM_LOAD16_BYTE( "95401268.hi", 0x00000, 0x080000, CRC(55c29d82) SHA1(3c08d26b57954da271ad594ef906c23c45780c83) )
	sc_rollo_others
ROM_END

ROM_START( sc4rolloe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401611.lo", 0x00001, 0x080000, CRC(3a2193ce) SHA1(4a9cae94870c269e594484626c2e7f2d4f8f8f48) )
	ROM_LOAD16_BYTE( "95401612.hi", 0x00000, 0x080000, CRC(75c4080c) SHA1(22294cc777f2ca520b4f49eee77cbf50e0102f84) )
	sc_rollo_others
ROM_END

ROM_START( sc4rollof )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401613.lo", 0x00001, 0x080000, CRC(301ab7c6) SHA1(04db05b2b154c466438a3adf7053e4801da55dee) )
	ROM_LOAD16_BYTE( "95401614.hi", 0x00000, 0x080000, CRC(dc7164d2) SHA1(f8bb3db8d758da605f3f02ed492194caf824fefd) )
	sc_rollo_others
ROM_END




ROM_START( sc4r66 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406780.lo", 0x00001, 0x080000, CRC(8b4efa73) SHA1(48d613a19a3b7486c1975c48c4a89af85d69bf23) )
	ROM_LOAD16_BYTE( "95406781.hi", 0x00000, 0x080000, CRC(2745101d) SHA1(3ea55d20c17285f18bdfa68032df359fbf14b94d) )
	sc_r66_others
ROM_END




ROM_START( sc4rbank )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406094.lo", 0x00001, 0x080000, CRC(de8b2571) SHA1(ad057935168af2021769743547d07100743c9da8) )
	ROM_LOAD16_BYTE( "95406095.hi", 0x00000, 0x080000, CRC(716fb2cc) SHA1(a7bf94c099bf973a1b70f6d51bf48ebc6ff03d19) )
	sc_rbank_others
ROM_END

ROM_START( sc4rbanka )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407094.lo", 0x00001, 0x080000, CRC(04b1d972) SHA1(22525822e359ad55c2954c15c7a69d41760a0db5) )
	ROM_LOAD16_BYTE( "95407095.hi", 0x00000, 0x080000, CRC(d4cd5d6b) SHA1(d05b165d0c4703ebb2941f0a91acafd7c2a65024) )
	sc_rbank_others
ROM_END

ROM_START( sc4rbankb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1111p11.hi", 0x00000, 0x080000, CRC(7c0ed03c) SHA1(c2bbcb840f554710217a6632889966d546d1c067) )
	ROM_LOAD16_BYTE( "p1111p11.lo", 0x00001, 0x080000, CRC(a7c8a2be) SHA1(e44e46efbb1db9bfadef0b4ce9aebd0c5d293fce) )
	sc_rbank_others
ROM_END

ROM_START( sc4rbankc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1111s11.hi", 0x00000, 0x080000, CRC(03ea6d4f) SHA1(1f5eec50a2f630de42224dc1b3d3dc22d8ab3ed6) )
	ROM_LOAD16_BYTE( "p1111s11.lo", 0x00001, 0x080000, CRC(1c0609da) SHA1(cf929fe9497d4f27130ea0ad4d0167b9e47b564d) )
	sc_rbank_others
ROM_END


ROM_START( sc4royle )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404464.lo", 0x00001, 0x080000, CRC(841e5431) SHA1(831780c92dd3287bb603d112e4a1a73e74495488) )
	ROM_LOAD16_BYTE( "95404465.hi", 0x00000, 0x080000, CRC(0ef18715) SHA1(0a589081d8b9ef809c028be5507cae68e159a998) )
	sc_royle_others
ROM_END

ROM_START( sc4roylea )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404466.lo", 0x00001, 0x080000, CRC(a8e2d7bb) SHA1(2213e76e33605300a3bf937d6a789bfe8f1f8597) )
	ROM_LOAD16_BYTE( "95404467.hi", 0x00000, 0x080000, CRC(71c6af83) SHA1(eecf28a757ff026c3e15ed9d1bdc08c6110e701e) )
	sc_royle_others
ROM_END

ROM_START( sc4royleb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405464.lo", 0x00001, 0x080000, CRC(da825d15) SHA1(6dde7438d20ddcbd924e57890537fc170222dc70) )
	ROM_LOAD16_BYTE( "95405465.hi", 0x00000, 0x080000, CRC(3b22b5a1) SHA1(f8e15ea4e918720432f164a7c1be5a9cd0cc6813) )
	sc_royle_others
ROM_END

ROM_START( sc4roylec )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405466.lo", 0x00001, 0x080000, CRC(78461edd) SHA1(a662f0eb8f8a51dd2e6ac0fe1b0e4297f92f60b6) )
	ROM_LOAD16_BYTE( "95405467.hi", 0x00000, 0x080000, CRC(bdcc34a5) SHA1(e1b7d718c6b69c0c92b44c922f82d77b6d4930a1) )
	sc_royle_others
ROM_END

ROM_START( sc4royled )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406098.lo", 0x00001, 0x080000, CRC(2460a66b) SHA1(ecc1ae23754e6803605c6b00a0724fe746ba193e) )
	ROM_LOAD16_BYTE( "95406099.hi", 0x00000, 0x080000, CRC(336665ae) SHA1(5cafd7af10411e4062d8bcbd628491e6f4fcbccf) )
	sc_royle_others
ROM_END

ROM_START( sc4roylee )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406672.lo", 0x00001, 0x080000, CRC(aee4a8fe) SHA1(2b5e4d3ba93fbb81d25ec0fe1193d4720fc0b7f8) )
	ROM_LOAD16_BYTE( "95406673.hi", 0x00000, 0x080000, CRC(1a18dfa1) SHA1(34b584f0cdc3853cda3809652187f195eaa7818d) )
	sc_royle_others
ROM_END

ROM_START( sc4roylef )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406906.lo", 0x00001, 0x080000, CRC(ebff3078) SHA1(f4ad541b35b21ebb0df358736d4e46b48edfbb96) )
	ROM_LOAD16_BYTE( "95406907.hi", 0x00000, 0x080000, CRC(0f077253) SHA1(59713b2bb50774d436b4362ba78786cdea2d3cc4) )
	sc_royle_others
ROM_END

ROM_START( sc4royleg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406908.lo", 0x00001, 0x080000, CRC(2795a21d) SHA1(b7c9f66cc1e4c960362f762a16c882404b9fb5d4) )
	ROM_LOAD16_BYTE( "95406909.hi", 0x00000, 0x080000, CRC(55fa0ec5) SHA1(12161b38f8b118fbbc734bdcf50b99c8d95f5b63) )
	sc_royle_others
ROM_END

ROM_START( sc4royleh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407098.lo", 0x00001, 0x080000, CRC(f32c77b5) SHA1(ae34fdd58dbc2996d2756a340dd01048e830f742) )
	ROM_LOAD16_BYTE( "95407099.hi", 0x00000, 0x080000, CRC(68ac92b6) SHA1(16773d09a825597324d3b367fa2de8c10ff97459) )
	sc_royle_others
ROM_END

ROM_START( sc4roylei )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407672.lo", 0x00001, 0x080000, CRC(e559f474) SHA1(9c141e23b8ecd7053bcca3adcee9a8ccfd6b72a3) )
	ROM_LOAD16_BYTE( "95407673.hi", 0x00000, 0x080000, CRC(b430d148) SHA1(93734e6ccf4de2d43d45cb83a2bcc04f5b9de3aa) )
	sc_royle_others
ROM_END

ROM_START( sc4roylej )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407906.lo", 0x00001, 0x080000, CRC(a5098e31) SHA1(568890e85a26ff1c866d1a8b736c8cfa004560fc) )
	ROM_LOAD16_BYTE( "95407907.hi", 0x00000, 0x080000, CRC(ae2d1838) SHA1(f6ad0c37fdc848a35e4faf990867aa1ea77360a0) )
	sc_royle_others
ROM_END

ROM_START( sc4roylek )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407908.lo", 0x00001, 0x080000, CRC(dbe1fea7) SHA1(190209b1682037c3f8d552f1ed644af2653aa8aa) )
	ROM_LOAD16_BYTE( "95407909.hi", 0x00000, 0x080000, CRC(18d106e9) SHA1(74c78448c36b640095361dd69012c60f61386278) )
	sc_royle_others
ROM_END

ROM_START( sc4roylel )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410204.lo", 0x00001, 0x080000, CRC(355a16d7) SHA1(febf66802f24cc792c6cf19f27df6a3151aec814) )
	ROM_LOAD16_BYTE( "95410205.hi", 0x00000, 0x080000, CRC(f826a3f2) SHA1(4f6b985455ccf327c559f248b7c7d5357b45f626) )
	sc_royle_others
ROM_END

ROM_START( sc4roylem )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411204.lo", 0x00001, 0x080000, CRC(e7b11b3f) SHA1(cd728702b8fcf13fc5ba236322f69a28fd52e1bb) )
	ROM_LOAD16_BYTE( "95411205.hi", 0x00000, 0x080000, CRC(8e6ebfd0) SHA1(4670068164dc530699578affc18837472c4540a7) )
	sc_royle_others
ROM_END




ROM_START( sc4srrmz )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404006.lo", 0x00001, 0x080000, CRC(ffe9dc77) SHA1(ccc632499675f65767a7a3ef275fd723fc09bc27) )
	ROM_LOAD16_BYTE( "95404007.hi", 0x00000, 0x080000, CRC(8fb7c4d0) SHA1(257f187ba4c20ea066392c7b337c7c3331c58175) )
	sc_srrmz_others
ROM_END

ROM_START( sc4srrmza )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406222.lo", 0x00001, 0x080000, CRC(80598518) SHA1(cf84165bb6589ac85925bcfc1981ff873e5a5d5d) )
	ROM_LOAD16_BYTE( "95406223.hi", 0x00000, 0x080000, CRC(1fd3a457) SHA1(c316ed2dae0e36f5c3f982405ef9ad2963cb763e) )
	sc_srrmz_others
ROM_END

ROM_START( sc4srrmzb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407222.lo", 0x00001, 0x080000, CRC(6f7dc46d) SHA1(aeeab3c5ae5b1ae31851023b2ed904f32b969d49) )
	ROM_LOAD16_BYTE( "95407223.hi", 0x00000, 0x080000, CRC(bb06299c) SHA1(9ff3a72c6e4e0781d2334e035b1678f7b2286e52) )
	sc_srrmz_others
ROM_END

ROM_START( sc4srrmzc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410266.lo", 0x00001, 0x080000, CRC(578253f1) SHA1(dabb77e75e063e34b1171f21f67889156c887ea9) )
	ROM_LOAD16_BYTE( "95410267.hi", 0x00000, 0x080000, CRC(e7221566) SHA1(fbedd365d68dd635319a9d450b8e37884d966cae) )
	sc_srrmz_others
ROM_END

ROM_START( sc4srrmzd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410708.lo", 0x00001, 0x080000, CRC(527242f4) SHA1(664bdca3e0eb8b07096e2189e9217eb895aa745b) )
	ROM_LOAD16_BYTE( "95410709.hi", 0x00000, 0x080000, CRC(4f4d535b) SHA1(f0eae6ad80863dfd579a26e4944baede7ee09cff) )
	sc_srrmz_others
ROM_END

ROM_START( sc4srrmze )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411266.lo", 0x00001, 0x080000, CRC(73a583d8) SHA1(b8f62440486342497bee7284325ce08e7758fd74) )
	ROM_LOAD16_BYTE( "95411267.hi", 0x00000, 0x080000, CRC(76bbfc21) SHA1(5d00d407a586f7a0643d053f9f0f8cb1dc61ef49) )
	sc_srrmz_others
ROM_END

ROM_START( sc4srrmzf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411706.lo", 0x00001, 0x080000, CRC(4be8b65d) SHA1(791a932b4f171adeac60bc74d02a7216b293fc26) )
	ROM_LOAD16_BYTE( "95411707.hi", 0x00000, 0x080000, CRC(047312b3) SHA1(493d5aa69e6ddaba4e9896d1dc56b6a9c09411fb) )
	sc_srrmz_others
ROM_END

ROM_START( sc4srrmzg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411708.lo", 0x00001, 0x080000, CRC(7226ed52) SHA1(9e227395ac7ae717f8b089aa95b9cdeb060b5c0f) )
	ROM_LOAD16_BYTE( "95411709.hi", 0x00000, 0x080000, CRC(e6a3b921) SHA1(403c47404979effaac311c89c6f3b398c6e8af48) )
	sc_srrmz_others
ROM_END

ROM_START( sc4srrmzh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414878.lo", 0x00001, 0x080000, CRC(9d7b2bbb) SHA1(a8ae0a0610f951fbf449427ef5ee4e579962cf43) )
	ROM_LOAD16_BYTE( "95414879.hi", 0x00000, 0x080000, CRC(efd3ee89) SHA1(34962978b8905009960d011b28fc963fc4acde0f) )
	sc_srrmz_others
ROM_END

ROM_START( sc4srrmzi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414884.lo", 0x00001, 0x080000, CRC(48d909cd) SHA1(97a56db61e5402da37a379f2063f0e091c665de0) )
	ROM_LOAD16_BYTE( "95414885.hi", 0x00000, 0x080000, CRC(e087b180) SHA1(e847eab2e30219552960317dc4a9ae276f743214) )
	sc_srrmz_others
ROM_END

ROM_START( sc4srrmzj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415878.lo", 0x00001, 0x080000, CRC(e126007d) SHA1(2b37fabde61d4fb13ea8218a0ae48ec73d9f505d) )
	ROM_LOAD16_BYTE( "95415879.hi", 0x00000, 0x080000, CRC(c121443c) SHA1(1b1b3d5ee3b9443b0162176d6fd27fd4009edde1) )
	sc_srrmz_others
ROM_END

ROM_START( sc4srrmzk )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415880.lo", 0x00001, 0x080000, CRC(b64b442a) SHA1(8aea42f76493479acaebabc82c16efc1d5ceedd1) )
	ROM_LOAD16_BYTE( "95415881.hi", 0x00000, 0x080000, CRC(88d6be22) SHA1(94f0a069850bf77e907f13945cfb24d602fca24d) )
	sc_srrmz_others
ROM_END

ROM_START( sc4srrmzl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415882.lo", 0x00001, 0x080000, CRC(e2afb025) SHA1(4897da226711686e0d5cfc90166a6a1bafdcd383) )
	ROM_LOAD16_BYTE( "95415883.hi", 0x00000, 0x080000, CRC(f0630dd0) SHA1(d54ecac93b508dd7717600c27ed0a762abd5d6d1) )
	sc_srrmz_others
ROM_END

ROM_START( sc4srrmzm )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415884.lo", 0x00001, 0x080000, CRC(ee121385) SHA1(486f13da7d94edb82f91ff52127433b3ce36e736) )
	ROM_LOAD16_BYTE( "95415885.hi", 0x00000, 0x080000, CRC(52be6ed0) SHA1(4e7803710a4692ba6b21ea87bce31eb485135433) )
	sc_srrmz_others
ROM_END


ROM_START( sc4srrca )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404146.lo", 0x00001, 0x080000, CRC(f8eadd61) SHA1(13307ed8e90fe23eff44580bbae97587f80e450d) )
	ROM_LOAD16_BYTE( "95404147.hi", 0x00000, 0x080000, CRC(b28cd756) SHA1(880ab382b60073f2d3087850bfc3d1d2e8cc206e) )
	sc_srrmz_others
ROM_END

ROM_START( sc4srrcaa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404148.lo", 0x00001, 0x080000, CRC(5f7ca43a) SHA1(0c26522aba68b0f8de038733cc28a95a6ad70145) )
	ROM_LOAD16_BYTE( "95404149.hi", 0x00000, 0x080000, CRC(d8087251) SHA1(5229d73dc7f4bf32741f514c1cbcfb9111c095f5) )
	sc_srrmz_others
ROM_END

ROM_START( sc4srrcab )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404152.lo", 0x00001, 0x080000, CRC(e66ed420) SHA1(02b75c17db48cb573fbf9aa969ddde1b1298b27a) )
	ROM_LOAD16_BYTE( "95404153.hi", 0x00000, 0x080000, CRC(5070e8d7) SHA1(9894b146409bcb0249f21a6b95eba4b2f44e6519) )
	sc_srrmz_others
ROM_END

ROM_START( sc4srrcac )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405146.lo", 0x00001, 0x080000, CRC(56c01000) SHA1(08445f6200c015d0bff8c873e664d058b3d550fc) )
	ROM_LOAD16_BYTE( "95405147.hi", 0x00000, 0x080000, CRC(777d1d14) SHA1(5e51ae0d98c2d3c298bf972f80f0e91e04df1f52) )
	sc_srrmz_others
ROM_END

ROM_START( sc4srrcad )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405150.lo", 0x00001, 0x080000, CRC(1c46c5e7) SHA1(6a1b06fa5f8ce23cedbf1484c38eee00a73dfab2) )
	ROM_LOAD16_BYTE( "95405151.hi", 0x00000, 0x080000, CRC(9629a2e1) SHA1(029c6c57dad1f4c81b97c10cb1d9f06cd5af1bb3) )
	sc_srrmz_others
ROM_END

ROM_START( sc4srrcae )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405152.lo", 0x00001, 0x080000, CRC(18b4fd76) SHA1(228b68a4ebb5d6809911fa7fa29573293e253db8) )
	ROM_LOAD16_BYTE( "95405153.hi", 0x00000, 0x080000, CRC(6741d4bb) SHA1(fe98ef93a3b99dc5b5d7adb1afb8ac607bb0b6f4) )
	sc_srrmz_others
ROM_END






ROM_START( sc4sidsp )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414360.lo", 0x00001, 0x080000, CRC(400906d3) SHA1(a6c262ae56b9c72468f78994c26fd3890b2d063d) )
	ROM_LOAD16_BYTE( "95414361.hi", 0x00000, 0x080000, CRC(2c6dd3e0) SHA1(ffb7954321a14cc60ae47804c8769b375044dc9a) )
	sc_sidsp_others
ROM_END

ROM_START( sc4sidspa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414362.lo", 0x00001, 0x080000, CRC(872a1583) SHA1(163f2a2ba9b9217a4443ac7ce55140beec32892b) )
	ROM_LOAD16_BYTE( "95414363.hi", 0x00000, 0x080000, CRC(75114c56) SHA1(2b15418b617b162654a984fbe577177f7cdacbd4) )
	sc_sidsp_others
ROM_END

ROM_START( sc4sidspb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415360.lo", 0x00001, 0x080000, CRC(309770ea) SHA1(39563d896cc2b38ee5cd854dab612bb3f46f0621) )
	ROM_LOAD16_BYTE( "95415361.hi", 0x00000, 0x080000, CRC(6e76f446) SHA1(cdb311bdf882297f0ab28580bb196804ae437b76) )
	sc_sidsp_others
ROM_END

ROM_START( sc4sidspc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415362.lo", 0x00001, 0x080000, CRC(d7332409) SHA1(3f193cfdf9f3e34748338bb22caade1a7224f8d9) )
	ROM_LOAD16_BYTE( "95415363.hi", 0x00000, 0x080000, CRC(1ad2cec6) SHA1(9c6926659022984548535e1c643f97e8c515707f) )
	sc_sidsp_others
ROM_END


ROM_START( sc4s2k )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404262.lo", 0x00001, 0x080000, CRC(61f5fb08) SHA1(06ed3b9cc1c4f59e4a756199678979a8b54a956d) )
	ROM_LOAD16_BYTE( "95404263.hi", 0x00000, 0x080000, CRC(eb492e7e) SHA1(cfbcd2004b6ff020024f238387253a8235be0cdd) )
	sc_s2k_others
ROM_END


ROM_START( sc4sirpz )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402472.lo", 0x00001, 0x080000, CRC(c6e6205e) SHA1(8e1cf99c6c5cf75f178c45454e6127a389324988) )
	ROM_LOAD16_BYTE( "95402473.hi", 0x00000, 0x080000, CRC(9b72e647) SHA1(dd8bce757ddca76d81ab0d47a68269432dd5ea64) )
	sc_sirpz_others
ROM_END

ROM_START( sc4sirpza )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406412.lo", 0x00001, 0x080000, CRC(bce7413e) SHA1(8e858e08cfe74960de908d052af872ca7f47ba6c) )
	ROM_LOAD16_BYTE( "95406413.hi", 0x00000, 0x080000, CRC(6ae91540) SHA1(2c0d48ed5415cc03fa8b75964f65d5b9a2f63f72) )
	sc_sirpz_others
ROM_END

ROM_START( sc4sirpzb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407412.lo", 0x00001, 0x080000, CRC(c5d4005e) SHA1(96a9bbd3ea6f7ce30412bc2eb4db802ceac465e2) )
	ROM_LOAD16_BYTE( "95407413.hi", 0x00000, 0x080000, CRC(942c28af) SHA1(d52d248e4827ab191def4fb75feec96356455f5c) )
	sc_sirpz_others
ROM_END


ROM_START( sc4stl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400445.lo", 0x00001, 0x080000, CRC(b68e7c56) SHA1(cdeae27d53cbfb97d64905a77d59fe5fbd8becf1) )
	ROM_LOAD16_BYTE( "95400446.hi", 0x00000, 0x080000, CRC(33aa01e7) SHA1(dcdffcb29c151429e28206661196c73109fe125c) )
	sc_stl_others
ROM_END

ROM_START( sc4stla )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400455.lo", 0x00001, 0x080000, CRC(70f5318e) SHA1(46100f779da2248f2e2c9b6606a0443b7f635878) )
	ROM_LOAD16_BYTE( "95400456.hi", 0x00000, 0x080000, CRC(7186ec69) SHA1(a60018b482285eaba50d786de54733294cbb74ff) )
	sc_stl_others
ROM_END

ROM_START( sc4stlb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400727.lo", 0x00001, 0x080000, CRC(4de69e56) SHA1(12aec2759ea240c4bd40696c633af29b4d9d4102) )
	ROM_LOAD16_BYTE( "95400728.hi", 0x00000, 0x080000, CRC(30d7c59c) SHA1(970dedcb5b701d8b9e5ead869c890b7a85edd760) )
	sc_stl_others
ROM_END

ROM_START( sc4stlc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400729.lo", 0x00001, 0x080000, CRC(a75e26af) SHA1(af0e402f1827046dee34cbce5176be818751da51) )
	ROM_LOAD16_BYTE( "95400730.hi", 0x00000, 0x080000, CRC(c6b400c8) SHA1(f72cce91496068750adb574e70844c5fb41abf17) )
	sc_stl_others
ROM_END

ROM_START( sc4stld )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401445.lo", 0x00001, 0x080000, CRC(1aaa8b8c) SHA1(fde14888b76f7f24b66dfbfb1cb771825f15a4a7) )
	ROM_LOAD16_BYTE( "95401446.hi", 0x00000, 0x080000, CRC(05a591fb) SHA1(30857a7de58d83578e81e120488bf328c85b883d) )
	sc_stl_others
ROM_END

ROM_START( sc4stle )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401455.lo", 0x00001, 0x080000, CRC(05c4e4d0) SHA1(03d7c0721b90e533c00d50225d8554c7dd7c5708) )
	ROM_LOAD16_BYTE( "95401456.hi", 0x00000, 0x080000, CRC(99af831d) SHA1(7fb6eaa3c218050965822a2dec31b27858526fc9) )
	sc_stl_others
ROM_END

ROM_START( sc4stlf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401727.lo", 0x00001, 0x080000, CRC(c3041a35) SHA1(c552a626daf350c176e82276db94baca8e552844) )
	ROM_LOAD16_BYTE( "95401728.hi", 0x00000, 0x080000, CRC(53118ccd) SHA1(8ba68b954a78fbf1ccd9a6666c6ba9375a90cb6b) )
	sc_stl_others
ROM_END

ROM_START( sc4stlg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401729.lo", 0x00001, 0x080000, CRC(6d7eab9c) SHA1(6b06511271c634e98466f5d02d653af63154fed4) )
	ROM_LOAD16_BYTE( "95401730.hi", 0x00000, 0x080000, CRC(e98eab26) SHA1(b40796400c0f2283f405a997e4c23a5f9dc77be1) )
	sc_stl_others
ROM_END


ROM_START( sc4smk7 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p6924s13.hi", 0x00000, 0x080000, CRC(b44ee4ee) SHA1(9386208a15591bbc1e8126f7392bd42022d835cf) )
	ROM_LOAD16_BYTE( "p6924s13.lo", 0x00001, 0x080000, CRC(dee55fcc) SHA1(b6f09f264406f5446f7e4fa849a09930077a8cc7) )
	sc_smk7_others
ROM_END




ROM_START( sc4srr )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406756.lo", 0x00001, 0x080000, CRC(3aab4e4f) SHA1(b53b7b241245a21c886efb801a5ee89d0d8ccf76) )
	ROM_LOAD16_BYTE( "95406757.hi", 0x00000, 0x080000, CRC(f8eb81f3) SHA1(8b9049d1fa089486da84fb86099c7d3c1cade3a7) )
	sc_srr_others
ROM_END

ROM_START( sc4srra )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406774.lo", 0x00001, 0x080000, CRC(576f5be1) SHA1(3697ea7ff205d43ada74e31818ef8f75d4a387b9) )
	ROM_LOAD16_BYTE( "95406775.hi", 0x00000, 0x080000, CRC(0f7a6230) SHA1(36db91b31c99539975c26f6be71456f31d52ca90) )
	sc_srr_others
ROM_END

ROM_START( sc4srrb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407756.lo", 0x00001, 0x080000, CRC(74f51f1a) SHA1(966116eedcc86a1217e8917a918f569cec1c101d) )
	ROM_LOAD16_BYTE( "95407757.hi", 0x00000, 0x080000, CRC(9953f1e1) SHA1(1f6e18ef48d507d88d14155f6d57c52141b85636) )
	sc_srr_others
ROM_END


ROM_START( sc4srrc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407774.lo", 0x00001, 0x080000, CRC(1c33317e) SHA1(7aa11025a7fadcb510c0a165a867bc78fe0c3ec6) )
	ROM_LOAD16_BYTE( "95407775.hi", 0x00000, 0x080000, CRC(4795d34a) SHA1(26f41b9fbda2c3670d99e5956ebe08b4465768de) )
	sc_srr_others
ROM_END




ROM_START( sc4slc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410364.lo", 0x00001, 0x080000, CRC(5edb97a0) SHA1(6a6a3c97a3db1041b39b0eb3a1dd7dec1f088320) )
	ROM_LOAD16_BYTE( "95410365.hi", 0x00000, 0x080000, CRC(b8a37850) SHA1(e96f995f04f87c7054aa73c5726f3830286f3b1a) )
	sc_slc_others
ROM_END

ROM_START( sc4slca )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410824.lo", 0x00001, 0x080000, CRC(9bf7c418) SHA1(3fe26a20efd9c870b9483b483fe659ed74e00ca7) )
	ROM_LOAD16_BYTE( "95410825.hi", 0x00000, 0x080000, CRC(651ed291) SHA1(054d6eded1ca4b530499155358a034ee26059ce9) )
	sc_slc_others
ROM_END

ROM_START( sc4slcb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410904.lo", 0x00001, 0x080000, CRC(ae4fcc00) SHA1(693435829da9cb95c2ac90080ecb96af11c05a8e) )
	ROM_LOAD16_BYTE( "95410905.hi", 0x00000, 0x080000, CRC(318fe4d6) SHA1(30e9d0cc3957ad747baf38b41d278d35cb76e1eb) )
	sc_slc_others
ROM_END

ROM_START( sc4slcc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411364.lo", 0x00001, 0x080000, CRC(1e9ea645) SHA1(450d64e756bd8d184c399ce1797a1220a65e2e2d) )
	ROM_LOAD16_BYTE( "95411365.hi", 0x00000, 0x080000, CRC(ad0cde5b) SHA1(0e9597020603d269f1771e328494b3b51448a50d) )
	sc_slc_others
ROM_END

ROM_START( sc4slcd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411824.lo", 0x00001, 0x080000, CRC(3ce1e43a) SHA1(5f6cbfe0ce93992b890f70483ec7c0277f089c0a) )
	ROM_LOAD16_BYTE( "95411825.hi", 0x00000, 0x080000, CRC(bba77bc1) SHA1(365f8552e8be5d5a541b0ce93b6290bc6365199a) )
	sc_slc_others
ROM_END

ROM_START( sc4slce )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411904.lo", 0x00001, 0x080000, CRC(2c3c30e6) SHA1(196574323ea99a6d06ba9cf1d3674581bd0886f0) )
	ROM_LOAD16_BYTE( "95411905.hi", 0x00000, 0x080000, CRC(730b9548) SHA1(35b6efc63b7733163303c5d3a4c483f1f7bf41f2) )
	sc_slc_others
ROM_END

ROM_START( sc4slcf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412594.lo", 0x00001, 0x080000, CRC(5f11978d) SHA1(c6e5e0a38c3372bcba0fe63b77542c6788730ecf) )
	ROM_LOAD16_BYTE( "95412595.hi", 0x00000, 0x080000, CRC(da2d1421) SHA1(76b7a77d4edebe00d65c199131970638b20d0322) )
	sc_slc_others
ROM_END

ROM_START( sc4slcg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412600.lo", 0x00001, 0x080000, CRC(fbc49ba2) SHA1(771e4077f3878c1666edba557ba1fd5bc96047d5) )
	ROM_LOAD16_BYTE( "95412601.hi", 0x00000, 0x080000, CRC(b3b8fc20) SHA1(4345c7a38070c8f9b0f5656cb7c093719a331f08) )
	sc_slc_others
ROM_END

ROM_START( sc4slch )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412602.lo", 0x00001, 0x080000, CRC(146ec9ab) SHA1(4f148c841914fab8b180e0289cb5a3da6132b364) )
	ROM_LOAD16_BYTE( "95412603.hi", 0x00000, 0x080000, CRC(d11384df) SHA1(3d82beb57cd4adcd4062dbd95ea35c90e3a31d6d) )
	sc_slc_others
ROM_END

ROM_START( sc4slci )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412604.lo", 0x00001, 0x080000, CRC(17042620) SHA1(3029b9d0b98539bb165d20312e6e059c79c53c6e) )
	ROM_LOAD16_BYTE( "95412605.hi", 0x00000, 0x080000, CRC(07cf24fa) SHA1(97f90929195e778365e732272f3268886d61af63) )
	sc_slc_others
ROM_END

ROM_START( sc4slcj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413594.lo", 0x00001, 0x080000, CRC(ab6f448d) SHA1(f97779fc3ea91e3966de2ccbdb71dcda33d9011e) )
	ROM_LOAD16_BYTE( "95413595.hi", 0x00000, 0x080000, CRC(8dbda8b6) SHA1(f86a02c162ace296fc773045dadbcfe965cf67e2) )
	sc_slc_others
ROM_END

ROM_START( sc4slck )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413600.lo", 0x00001, 0x080000, CRC(de181a14) SHA1(88603d7875b826eeeff884d1bf96f64b87151102) )
	ROM_LOAD16_BYTE( "95413601.hi", 0x00000, 0x080000, CRC(b817cc50) SHA1(e256ee3f2d51cabaa4c7eac53c620de106838d61) )
	sc_slc_others
ROM_END

ROM_START( sc4slcl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413602.lo", 0x00001, 0x080000, CRC(f36f9226) SHA1(2574cb9f9cc2af31b8024b57c17112da7139d153) )
	ROM_LOAD16_BYTE( "95413603.hi", 0x00000, 0x080000, CRC(6633f546) SHA1(0554c4a291f943517e0e86bfafc78dd0e90c595e) )
	sc_slc_others
ROM_END

ROM_START( sc4slcm )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413604.lo", 0x00001, 0x080000, CRC(c3a24dab) SHA1(fd994fe7823272bb13e407e52efe707528a20015) )
	ROM_LOAD16_BYTE( "95413605.hi", 0x00000, 0x080000, CRC(aa164a89) SHA1(a3f4826eaca7708f0447b52f29c01fba2e42e767) )
	sc_slc_others
ROM_END

ROM_START( sc4slcn )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1423s12.hi", 0x00000, 0x080000, CRC(207f125d) SHA1(f107859bad37697f3eb90e36aeb31cad1c640a70) )
	ROM_LOAD16_BYTE( "p1423s12.lo", 0x00001, 0x080000, CRC(8234f892) SHA1(b5d551ef2fb44cf0e2c231278b83dc9ccbea69dd) )
	sc_slc_others
ROM_END


ROM_START( sc4solgl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404070.lo", 0x00001, 0x080000, CRC(81f9301e) SHA1(5a4a5d6cd24191d4b4496c049e3f8a4fc5c5c5fe) )
	ROM_LOAD16_BYTE( "95404071.hi", 0x00000, 0x080000, CRC(e1dc5a5d) SHA1(bd24dc8600571549c4b46cf565c16bd15207ab7d) )
	sc_solgl_others
ROM_END

ROM_START( sc4solgla )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404072.lo", 0x00001, 0x080000, CRC(3178ea33) SHA1(e50495daaf821491c54bfb5e5e9926c9340addc6) )
	ROM_LOAD16_BYTE( "95404073.hi", 0x00000, 0x080000, CRC(31099405) SHA1(3b1896c98e82532a536d3275eddb0930e7e708de) )
	sc_solgl_others
ROM_END

ROM_START( sc4solglb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405070.lo", 0x00001, 0x080000, CRC(40a616da) SHA1(3557c8a85cb0482210af05c4cdbb64a5bee41067) )
	ROM_LOAD16_BYTE( "95405071.hi", 0x00000, 0x080000, CRC(b48ad528) SHA1(a806be0170392c58a0b341113ed3d8c813ce350a) )
	sc_solgl_others
ROM_END

ROM_START( sc4solglc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405072.lo", 0x00001, 0x080000, CRC(15aa920a) SHA1(fda0f038a43235501d1f8016bd06f1fea6e15df4) )
	ROM_LOAD16_BYTE( "95405073.hi", 0x00000, 0x080000, CRC(51137ffc) SHA1(d94469e5addf7826c756f98b1e951833ea2bcc98) )
	sc_solgl_others
ROM_END


ROM_START( sc4slih )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410436.lo", 0x00001, 0x080000, CRC(4df1436f) SHA1(a7c516e1d26dc6fca08e64937e91e2174afdf315) )
	ROM_LOAD16_BYTE( "95410437.hi", 0x00000, 0x080000, CRC(254f9b81) SHA1(0462e21acdea74baabfe15bd9282421fd0d15e99) )
	sc_slih_others
ROM_END

ROM_START( sc4sliha )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411436.lo", 0x00001, 0x080000, CRC(5da6d35d) SHA1(66dc43728e38df61d30ae893928fbcdec8a2466b) )
	ROM_LOAD16_BYTE( "95411437.hi", 0x00000, 0x080000, CRC(a3048c2d) SHA1(1c24420464cb8690dcc0a1c317edb432ae61bc0f) )
	sc_slih_others
ROM_END

ROM_START( sc4slihb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412262.lo", 0x00001, 0x080000, CRC(625046fc) SHA1(57a390800ae51f4a5e3f93b0ed87fa2275ea2c78) )
	ROM_LOAD16_BYTE( "95412263.hi", 0x00000, 0x080000, CRC(40db5742) SHA1(e63878a2e45a12392567f60ee6ddfe3af5511be7) )
	sc_slih_others
ROM_END

ROM_START( sc4slihc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413262.lo", 0x00001, 0x080000, CRC(709eaf2f) SHA1(ccb457e1b38387c6d0072a4310856cd09588c03a) )
	ROM_LOAD16_BYTE( "95413263.hi", 0x00000, 0x080000, CRC(a5c95f87) SHA1(bb8f98ddcd78b1d9fddb271592f87d89a4ab51c0) )
	sc_slih_others
ROM_END

ROM_START( sc4slihd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414576.lo", 0x00001, 0x080000, CRC(c77150a5) SHA1(9c23b3432c48166b2a592af6b654c0ae3729c564) )
	ROM_LOAD16_BYTE( "95414577.hi", 0x00000, 0x080000, CRC(ed060c33) SHA1(e5b7bfc55b84c0d42d1bc73f7ad4e21bab9645a6) )
	sc_slih_others
ROM_END

ROM_START( sc4slihe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415576.lo", 0x00001, 0x080000, CRC(e904571b) SHA1(84856fe533d24538b988625a73b67b952c1b6b8a) )
	ROM_LOAD16_BYTE( "95415577.hi", 0x00000, 0x080000, CRC(c17f534d) SHA1(4ceab2363f647dbe8a610e8c12a0ec7d24664a54) )
	sc_slih_others
ROM_END


ROM_START( sc4sace )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408060.odd", 0x00001, 0x080000, CRC(078c36c1) SHA1(4937cc861048189172737b8441d2bc97e359b467) )
	ROM_LOAD16_BYTE( "95408061.evn", 0x00000, 0x080000, CRC(36c760e8) SHA1(2eb760d303486102cfccb924062b77a3679f7bfa) )
	sc_sace_others
ROM_END

ROM_START( sc4sacea )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409060.odd", 0x00001, 0x080000, CRC(a10cdb78) SHA1(34ddb81b622036cd3d86dd48e80e8c036020cbfa) )
	ROM_LOAD16_BYTE( "95409061.evn", 0x00000, 0x080000, CRC(185c314e) SHA1(557da30f4d1679ed17de3308e1e5234deba30c85) )
	sc_sace_others
ROM_END


ROM_START( sc4sbust )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406134.odd", 0x00001, 0x080000, CRC(f8c56233) SHA1(35b87458cc43de52233a771526423fed4648804e) )
	ROM_LOAD16_BYTE( "95406135.evn", 0x00000, 0x080000, CRC(597dc269) SHA1(bc3f4f0118990c9dd25a2f5bc0fe1ee42fc3392e) )
	sc_sbust_others
ROM_END

ROM_START( sc4sbusta )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407134.odd", 0x00001, 0x080000, CRC(49e962a2) SHA1(067fbb01c28f3b6818e0c4e3baeb258916ee08cc) )
	ROM_LOAD16_BYTE( "95407135.evn", 0x00000, 0x080000, CRC(6fed044d) SHA1(fa2dd3b0fa50f205c092d175a2cdffea48d98cb0) )
	sc_sbust_others
ROM_END








ROM_START( sc4splgb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402161.lo", 0x00001, 0x080000, CRC(fd49e035) SHA1(ebfea772f9d60a9429a6545fa495b72d159c1b80) )
	ROM_LOAD16_BYTE( "95402162.hi", 0x00000, 0x080000, CRC(d8050b29) SHA1(0cf738e2ef9f8c9973a3148be338759eb769fe84) )
	sc_splgb_others
ROM_END

ROM_START( sc4splgba )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403161.lo", 0x00001, 0x080000, CRC(5bf7949d) SHA1(0705a98f7819a15de1d18240dc7496e57e2d89b9) )
	ROM_LOAD16_BYTE( "95403162.hi", 0x00000, 0x080000, CRC(cfd7f5c2) SHA1(0368881aaee9e5b153807c01c9ef7021ed10d031) )
	sc_splgb_others
ROM_END


ROM_START( sc4spred )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404606.lo", 0x00001, 0x080000, CRC(14507a00) SHA1(01d711155b66a8252ce8cafb03db585ea624095d) )
	ROM_LOAD16_BYTE( "95404607.hi", 0x00000, 0x080000, CRC(cc4b76d2) SHA1(28a7e1d6ecefd228557e97a90c49bffff0cb335c) )
	sc_spred_others
ROM_END


ROM_START( sc4stag )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406092.lo", 0x00001, 0x080000, CRC(71347a4f) SHA1(5dbaa874b6c4e29d225005815830e5101c5c7105) )
	ROM_LOAD16_BYTE( "95406093.hi", 0x00000, 0x080000, CRC(acc83bb2) SHA1(17ff4b8cad7d63597703cbc2febd0a8a112b3612) )
	sc_stag_others
ROM_END

ROM_START( sc4staga )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407092.lo", 0x00001, 0x080000, CRC(978b63cb) SHA1(50f475de4754e21d9394aee1815e05e4f3c87ae0) )
	ROM_LOAD16_BYTE( "95407093.hi", 0x00000, 0x080000, CRC(cb804832) SHA1(31472e58798e0438f17d7d74a90d04e109fa6231) )
	sc_stag_others
ROM_END


ROM_START( sc4starp )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400153.lo", 0x00001, 0x080000, CRC(7d2efaa9) SHA1(321a15e844a0b3102687c92de7a545cb6906121a) )
	ROM_LOAD16_BYTE( "95400154.hi", 0x00000, 0x080000, CRC(caede04c) SHA1(d623cc28f87f3afd3d2e4bfe27eaa11cd324a5ce) )
	sc_starp_others
ROM_END

ROM_START( sc4starpa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400249.lo", 0x00001, 0x080000, CRC(32a874cb) SHA1(f6d7f3b4c2f6562ec68bfd3357afceb7780c438a) )
	ROM_LOAD16_BYTE( "95400250.hi", 0x00000, 0x080000, CRC(8ff5ce78) SHA1(3a4581e29f09c79bf686caaa92f413f125ca0ee8) )
	sc_starp_others
ROM_END

ROM_START( sc4starpb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400319.lo", 0x00001, 0x080000, CRC(63575baa) SHA1(d8fa40d39a77098c180001f33f0f2ca1891fcf1b) )
	ROM_LOAD16_BYTE( "95400320.hi", 0x00000, 0x080000, CRC(07de30c2) SHA1(2820f3fe2f1450ba3b99d417a1c75fa23170c68d) )
	sc_starp_others
ROM_END

ROM_START( sc4starpc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400839.lo", 0x00001, 0x080000, CRC(7180e28b) SHA1(4cd776032a3d2ee2fddfc6fb96cfa8d26c2d3bf0) )
	ROM_LOAD16_BYTE( "95400840.hi", 0x00000, 0x080000, CRC(7c26b453) SHA1(adca4cb4ce1c9dcbade7b925bc4e2e86da26f3f0) )
	sc_starp_others
ROM_END

ROM_START( sc4starpd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400841.lo", 0x00001, 0x080000, CRC(fda7fa90) SHA1(56c8e52e6e73e77aceaf8a83dd463b66dfd69404) )
	ROM_LOAD16_BYTE( "95400842.hi", 0x00000, 0x080000, CRC(09bf64c4) SHA1(73d24ab4c62cacf572f56e481252483ec4437a50) )
	sc_starp_others
ROM_END

ROM_START( sc4starpe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401153.lo", 0x00001, 0x080000, CRC(642c9f39) SHA1(44772e8e3029f5c28e65a071a67e4cdc12e222fd) )
	ROM_LOAD16_BYTE( "95401154.hi", 0x00000, 0x080000, CRC(550c9a2a) SHA1(39c76f413b74fcd6f15402a69d7d9dd9989273af) )
	sc_starp_others
ROM_END

ROM_START( sc4starpf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401249.lo", 0x00001, 0x080000, CRC(0515b13b) SHA1(0bf59f485655a177b8c7c922685be34f15b5bc9b) )
	ROM_LOAD16_BYTE( "95401250.hi", 0x00000, 0x080000, CRC(9a5a6873) SHA1(22574e541f0798950c2ce4b027cefd2d867f445b) )
	sc_starp_others
ROM_END

ROM_START( sc4starpg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401319.lo", 0x00001, 0x080000, CRC(caa9045a) SHA1(7f0523ad65cb96ca0dee4a3b26e0c1ae29ac5867) )
	ROM_LOAD16_BYTE( "95401320.hi", 0x00000, 0x080000, CRC(5fb3d806) SHA1(cf9df40d6adbb82b1fe76f70e95adeb1ccb0eafd) )
	sc_starp_others
ROM_END

ROM_START( sc4starph )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401839.lo", 0x00001, 0x080000, CRC(68d075de) SHA1(090104001364a19cf50fde46666d00e2f264c15c) )
	ROM_LOAD16_BYTE( "95401840.hi", 0x00000, 0x080000, CRC(2708a807) SHA1(eca3925cea8723e29a42d3d7ec9cae6aab448ba4) )
	sc_starp_others
ROM_END

ROM_START( sc4starpi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401841.lo", 0x00001, 0x080000, CRC(4c27509e) SHA1(bd31292e30305e41f2e4f46422e4763578211174) )
	ROM_LOAD16_BYTE( "95401842.hi", 0x00000, 0x080000, CRC(1356fd9e) SHA1(b2b01e98cfd857621fc29fe7ab3d5a4d6ecbaf3d) )
	sc_starp_others
ROM_END

ROM_START( sc4starpj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "sprz15hi", 0x00000, 0x080000, CRC(ee6891f2) SHA1(b148917e207eabd51ccf4ac87125383e4338e4dc) )
	ROM_LOAD16_BYTE( "sprz15lo", 0x00001, 0x080000, CRC(9bd0fa39) SHA1(2a5b5db8d6318d9050e9650efb37c7dadedc40fd) )
	sc_starp_others
ROM_END

ROM_START( sc4starpk )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "starp247.lo", 0x00001, 0x080000, CRC(075f3e8d) SHA1(1a2e96cff3809a5844f7d76108ae3ef796b1bc97) )
	ROM_LOAD16_BYTE( "starp248.hi", 0x00000, 0x080000, CRC(b1ad5194) SHA1(0ecc9a790bef001ba8ce889639e5a290d322346c) )
	sc_starp_others
ROM_END



ROM_START( sc4strbr )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406478.lo", 0x00001, 0x080000, CRC(44d83c82) SHA1(18d2b7e5436a888cee0edffc9cc89becd6eee492) )
	ROM_LOAD16_BYTE( "95406479.hi", 0x00000, 0x080000, CRC(80bfda8f) SHA1(64b5d7b15339cd72419ab291b87adbc464a6154e) )
	sc_strbr_others
ROM_END

ROM_START( sc4strbra )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406742.lo", 0x00001, 0x080000, CRC(e6581ca0) SHA1(0e9b76524d6395067dbd6813d9299256592c1639) )
	ROM_LOAD16_BYTE( "95406743.hi", 0x00000, 0x080000, CRC(2601e7d6) SHA1(bde092ccae97d5a17fdce96da2b589713a730eff) )
	sc_strbr_others
ROM_END

ROM_START( sc4strbrb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408044.lo", 0x00001, 0x080000, CRC(6ad12ec0) SHA1(5db838200681eb7d4210af1503656001c37154d4) )
	ROM_LOAD16_BYTE( "95408045.hi", 0x00000, 0x080000, CRC(3d34c484) SHA1(7d8b401ac900c0666b849d545143af5c126b0d84) )
	sc_strbr_others
ROM_END

ROM_START( sc4strbrc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408150.lo", 0x00001, 0x080000, CRC(77bc3d52) SHA1(ca03facce088c7773aff292c6e2a8ae0601c6c7d) )
	ROM_LOAD16_BYTE( "95408151.hi", 0x00000, 0x080000, CRC(3ddb4e61) SHA1(789d6bf967d1b790dcf8e9d72c2174f2f3b42fc0) )
	sc_strbr_others
ROM_END

ROM_START( sc4strbrd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408588.lo", 0x00001, 0x080000, CRC(6bf697b3) SHA1(53e326858be5005f03707ea975e39ea5536de23c) )
	ROM_LOAD16_BYTE( "95408589.hi", 0x00000, 0x080000, CRC(c0106b47) SHA1(a794d18d1dca1555f16d26a5b0a7c0f233a51d10) )
	sc_strbr_others
ROM_END


ROM_START( sc4stirc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402938.lo", 0x00001, 0x080000, CRC(7e97188b) SHA1(5e7c7e00b0ef5c25b0a9456bbfa8982786cfde8a) )
	ROM_LOAD16_BYTE( "95402939.hi", 0x00000, 0x080000, CRC(2d9542df) SHA1(3569113432dd4d59ae2e30025da2dae8f78d89a8) )
	sc_stirc_others
ROM_END

ROM_START( sc4stirca )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402942.lo", 0x00001, 0x080000, CRC(0ef41837) SHA1(5ac992670e155a46bf85e1fef5a2b4c5a297666e) )
	ROM_LOAD16_BYTE( "95402943.hi", 0x00000, 0x080000, CRC(a7dcf62d) SHA1(6077b1cb3c2247abcf99fde50226598418b9f0c2) )
	sc_stirc_others
ROM_END

ROM_START( sc4stircb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403938.lo", 0x00001, 0x080000, CRC(4cd8f0d5) SHA1(4057870619de65b9e50dfc5807ca514d576e9f2d) )
	ROM_LOAD16_BYTE( "95403939.hi", 0x00000, 0x080000, CRC(854504d5) SHA1(112de9c078b4d9ede043fcc288ee02a0671b959f) )
	sc_stirc_others
ROM_END

ROM_START( sc4stircc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403940.lo", 0x00001, 0x080000, CRC(8f9e4d10) SHA1(38244028ac3f903971bd6e2786339a321d181df0) )
	ROM_LOAD16_BYTE( "95403941.hi", 0x00000, 0x080000, CRC(335ceb9f) SHA1(453937f56815bbbc87b251366d14d38e979482f9) )
	sc_stirc_others
ROM_END

ROM_START( sc4stircd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404126.lo", 0x00001, 0x080000, CRC(76c867e5) SHA1(5eb6a1e03b397ba847588b5b07d350e74d9e681d) )
	ROM_LOAD16_BYTE( "95404127.hi", 0x00000, 0x080000, CRC(8243db7b) SHA1(d36e3e38aaf6b0fd05757179f715ecaa2d07777a) )
	sc_stirc_others
ROM_END

ROM_START( sc4stirce )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405126.lo", 0x00001, 0x080000, CRC(27a13e76) SHA1(114ab715f6c3e15feee1f36f473faec69815098d) )
	ROM_LOAD16_BYTE( "95405127.hi", 0x00000, 0x080000, CRC(8fe7eaf2) SHA1(f9e8dc96a2a74ab7861f05e88ec42dc893babbca) )
	sc_stirc_others
ROM_END

ROM_START( sc4stircf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406284.lo", 0x00001, 0x080000, CRC(5a1c0cec) SHA1(0a0dac529cf1eae20d552c7d2d6bb27d6dfbc9c7) )
	ROM_LOAD16_BYTE( "95406285.hi", 0x00000, 0x080000, CRC(39171d86) SHA1(66062b366a26e19bbf0b0ae887e2d20be63236e7) )
	sc_stirc_others
ROM_END

ROM_START( sc4stircg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406684.lo", 0x00001, 0x080000, CRC(8e3bce31) SHA1(50d6d55a110e72d0fa1aabafae0f8abe1a6d2343) )
	ROM_LOAD16_BYTE( "95406685.hi", 0x00000, 0x080000, CRC(4f800fca) SHA1(261e63bc273625c07e0a9765183a5723627cf27e) )
	sc_stirc_others
ROM_END

ROM_START( sc4stirch )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407284.lo", 0x00001, 0x080000, CRC(e8e5eb7b) SHA1(3dc0691bf3f0f60c266b57dca121755e5fae0f89) )
	ROM_LOAD16_BYTE( "95407285.hi", 0x00000, 0x080000, CRC(f0e31c05) SHA1(5806213c3c88db20941fb3891c9c79d4cd4da3b1) )
	sc_stirc_others
ROM_END

ROM_START( sc4stirci )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407684.lo", 0x00001, 0x080000, CRC(23c68fc1) SHA1(5a51a98fa7eaa9bca98f47ff10491f505a7a4f1e) )
	ROM_LOAD16_BYTE( "95407685.hi", 0x00000, 0x080000, CRC(5852f121) SHA1(ca6e548a4b687f8f6eba6bd9dc8118b9ec2ab967) )
	sc_stirc_others
ROM_END

ROM_START( sc4stircj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "st_crazy.lo", 0x00001, 0x080000, CRC(2a95d06a) SHA1(b9b17f6f54062f738c0595637c9be8fedac49546) )
	ROM_LOAD16_BYTE( "st_crazy.hi", 0x00000, 0x080000, CRC(0f409002) SHA1(5822c523426eb1a6b3af719720b5b92d2d4ac200) )
	sc_stirc_others
ROM_END


ROM_START( sc4strk )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416484.lo", 0x00001, 0x080000, CRC(1717074a) SHA1(936e10b53f1b44270c96091a60f470425f6baaad) )
	ROM_LOAD16_BYTE( "95416485.hi", 0x00000, 0x080000, CRC(86531e46) SHA1(0b7a911c827bdbabd1d889a04290aca083c43945) )
	sc_strk_others
ROM_END

ROM_START( sc4strka )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416490.lo", 0x00001, 0x080000, CRC(dc674c3e) SHA1(1a087ad8d8931fbfd2ad9d3eca116674941c8805) )
	ROM_LOAD16_BYTE( "95416491.hi", 0x00000, 0x080000, CRC(4c584b02) SHA1(f0dc5c30759fd1d541ab26abcb2af1a21cae2f97) )
	sc_strk_others
ROM_END

ROM_START( sc4strkb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416644.lo", 0x00001, 0x080000, CRC(600c42b5) SHA1(d1af081e9781aa0817960810cf30f931ee345547) )
	ROM_LOAD16_BYTE( "95416645.hi", 0x00000, 0x080000, CRC(5499f16a) SHA1(cdbc7d88b5bf68ce227a3fa8ca45f3b7d8c028cd) )
	sc_strk_others
ROM_END

ROM_START( sc4strkc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416650.lo", 0x00001, 0x080000, CRC(247ae490) SHA1(a96cbeb3c956871cd84900dbb2b0ca6c1e0d198d) )
	ROM_LOAD16_BYTE( "95416651.hi", 0x00000, 0x080000, CRC(21435b22) SHA1(675364e906561c7901137f5cb4927f55d0b633fc) )
	sc_strk_others
ROM_END

ROM_START( sc4strkd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417484.lo", 0x00001, 0x080000, CRC(8c6f8d43) SHA1(e5c4fff6cb92042b5d8f6101e8e5478839c6a7bc) )
	ROM_LOAD16_BYTE( "95417485.hi", 0x00000, 0x080000, CRC(2cfe00ac) SHA1(507f2b7b0992b2644386c895dcfd5531c2b80fe2) )
	sc_strk_others
ROM_END

ROM_START( sc4strke )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417490.lo", 0x00001, 0x080000, CRC(5d690955) SHA1(e87882308d60331d9ff83506f83e9b5e9e43a228) )
	ROM_LOAD16_BYTE( "95417491.hi", 0x00000, 0x080000, CRC(1594598e) SHA1(b0031c0892be74427862f61886282799b0c634f9) )
	sc_strk_others
ROM_END

ROM_START( sc4strkf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417602.lo", 0x00001, 0x080000, CRC(f9e99405) SHA1(37609f6347f89ef39ebc284bf3eb70b97e21a8a6) )
	ROM_LOAD16_BYTE( "95417603.hi", 0x00000, 0x080000, CRC(2034db37) SHA1(13b71d5c7f8886d59b2924ecb54fd52900ea07b6) )
	sc_strk_others
ROM_END

ROM_START( sc4strkg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417604.lo", 0x00001, 0x080000, CRC(d9c257c9) SHA1(861f32993d81e5bef6a4040fe424ef4fca7bb581) )
	ROM_LOAD16_BYTE( "95417605.hi", 0x00000, 0x080000, CRC(2824e731) SHA1(abc7172d8aa1b09fa7a426e5fbec78cba009b5c0) )
	sc_strk_others
ROM_END

ROM_START( sc4strkh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417644.lo", 0x00001, 0x080000, CRC(d4a6f36c) SHA1(140c44d150b26a9214aa9c4daf9cb3a7c8477b17) )
	ROM_LOAD16_BYTE( "95417645.hi", 0x00000, 0x080000, CRC(4d045b2c) SHA1(2dedf3a2fd01c165b7ba5260243a27d013f08092) )
	sc_strk_others
ROM_END

ROM_START( sc4strki )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417646.lo", 0x00001, 0x080000, CRC(328509c5) SHA1(61518a77a5d17259f9b0bd1eca48d16f91fe19b7) )
	ROM_LOAD16_BYTE( "95417647.hi", 0x00000, 0x080000, CRC(51f8dbf3) SHA1(535e7a4d437e5ab98b9ce9dac4b093ed9de84ea0) )
	sc_strk_others
ROM_END

ROM_START( sc4strkj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417648.lo", 0x00001, 0x080000, CRC(237de5b6) SHA1(21fc8de3ba885b7ea9b9077a5c6d041131e4a811) )
	ROM_LOAD16_BYTE( "95417649.hi", 0x00000, 0x080000, CRC(a65b98f8) SHA1(039a153fe962de7eda18f2ff9467e718e941e816) )
	sc_strk_others
ROM_END

ROM_START( sc4strkk )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417650.lo", 0x00001, 0x080000, CRC(5a8f3064) SHA1(ca05350bf10500d4f1b58f9a3c94255eb6a6784a) )
	ROM_LOAD16_BYTE( "95417651.hi", 0x00000, 0x080000, CRC(5a8e85f1) SHA1(b3fab14196b8435ac8d6b9d9980940a336780fc9) )
	sc_strk_others
ROM_END


ROM_START( sc4sahed )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408360.odd", 0x00001, 0x080000, CRC(e3781af6) SHA1(6e7d2d120d04c79f39f2a8794ab95096eae07c1f) )
	ROM_LOAD16_BYTE( "95408361.evn", 0x00000, 0x080000, CRC(396c29b7) SHA1(70d44b4887afa5464edf91c80ff4842bcb15bcbb) )
	sc_sahed_others
ROM_END

ROM_START( sc4saheda )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409360.odd", 0x00001, 0x080000, CRC(a2826287) SHA1(bd45e5a1043ab22d6aef205d3dcacb17b150f6c8) )
	ROM_LOAD16_BYTE( "95409361.evn", 0x00000, 0x080000, CRC(2cc38fbc) SHA1(d9d9baa12347578b057f1c963f3cdc7e3e94b4d6) )
	sc_sahed_others
ROM_END

ROM_START( sc4sahedb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409362.odd", 0x00001, 0x080000, CRC(722143e7) SHA1(a4d2059f103021cc315ce3697eeabc27c90058bd) )
	ROM_LOAD16_BYTE( "95409363.evn", 0x00000, 0x080000, CRC(2448c93e) SHA1(3c5023d768f50ebb96322bc78d5cb51fcb2a2ce7) )
	sc_sahed_others
ROM_END


ROM_START( sc4sf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404372.lo", 0x00001, 0x080000, CRC(07b9d1e2) SHA1(386356eb2c7c8b82978700e466f5d55e82b72b5f) )
	ROM_LOAD16_BYTE( "95404373.hi", 0x00000, 0x080000, CRC(68e73d68) SHA1(43d3229afb3085522dc7e04e0015020e3681cb7d) )
	sc_sf_others
ROM_END

ROM_START( sc4sfa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405226.lo", 0x00001, 0x080000, CRC(943c5377) SHA1(ca9d1fb6cb09f690a82e3bd057c81f88f49b8cb5) )
	ROM_LOAD16_BYTE( "95405227.hi", 0x00000, 0x080000, CRC(79d8a0a0) SHA1(420f8a6f8bc1fbdb442bc7bdfa6c8b8891e3f6c6) )
	sc_sf_others
ROM_END

ROM_START( sc4sfb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405372.lo", 0x00001, 0x080000, CRC(76022704) SHA1(be9158bf045b24777a118b054e2e29f3c1d939f2) )
	ROM_LOAD16_BYTE( "95405373.hi", 0x00000, 0x080000, CRC(94c00e5f) SHA1(9d1fcf9365008ec394e3d497e7115875a7938ccd) )
	sc_sf_others
ROM_END

ROM_START( sc4sfc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406148.lo", 0x00001, 0x080000, CRC(758f60aa) SHA1(3f7d47319c86b0e7ffb08aff8ac0e1c7d22cb175) )
	ROM_LOAD16_BYTE( "95406149.hi", 0x00000, 0x080000, CRC(4383447b) SHA1(cf70714c76c66455fb57e06ea6d4ddc458f897b6) )
	sc_sf_others
ROM_END

ROM_START( sc4sfd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407148.lo", 0x00001, 0x080000, CRC(b5ee08d4) SHA1(f170d32ae41be7d0a01fcee690a1a94380c44d19) )
	ROM_LOAD16_BYTE( "95407149.hi", 0x00000, 0x080000, CRC(aa0bd147) SHA1(a569d7714d6b6ce7d52e02c24cfcdc1e8049805c) )
	sc_sf_others
ROM_END


ROM_START( sc4strx )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406048.lo", 0x00001, 0x080000, CRC(6c20e6ad) SHA1(3cd8b65a15f60e1aadd4917fc2a5b4cc024058b0) )
	ROM_LOAD16_BYTE( "95406049.hi", 0x00000, 0x080000, CRC(a2aa51e1) SHA1(f2bf95fa63b2e1bd0613c278a00063dfa18db9c5) )
	sc_strx_others
ROM_END

ROM_START( sc4strxa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407048.lo", 0x00001, 0x080000, CRC(8903611d) SHA1(1b12dd9d1ab35d625952d12066d027466264bb19) )
	ROM_LOAD16_BYTE( "95407049.hi", 0x00000, 0x080000, CRC(ed8a11f6) SHA1(0d4f37dfecf362c282f17dc55dc0957296a974e6) )
	sc_strx_others
ROM_END

ROM_START( sc4strxb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1114p11.hi", 0x00000, 0x080000, CRC(5d9b7b41) SHA1(20fcc012b15d2b7df9c59b080b9a440bc4b4dfe3) )
	ROM_LOAD16_BYTE( "p1114p11.lo", 0x00001, 0x080000, CRC(d063e941) SHA1(87ad78ff49890b21c2d75a8417885e86479278c3) )
	sc_strx_others
ROM_END

ROM_START( sc4strxc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1114s11.hi", 0x00000, 0x080000, CRC(1664bc68) SHA1(d06799ad3a3b975c30235b25689c3e42c8e832ac) )
	ROM_LOAD16_BYTE( "p1114s11.lo", 0x00001, 0x080000, CRC(3b2ad23c) SHA1(14554d5eb51ae0ecc9fcebdfd50f027f0ece78c3) )
	sc_strx_others
ROM_END


ROM_START( sc4suscl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420890.lo", 0x00001, 0x080000, CRC(0765de84) SHA1(3fe611f34982cd734d7356b54d2cc60acf96b7c4) )
	ROM_LOAD16_BYTE( "95420891.hi", 0x00000, 0x080000, CRC(1cec5a68) SHA1(91485071a476ebf725daae9868e826a5bb45cb46) )
	sc_suscl_others
ROM_END

ROM_START( sc4suscla )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420968.lo", 0x00001, 0x080000, CRC(c588de92) SHA1(18a4215f7bd5f8d4cf6a4b81654496d94d0949c3) )
	ROM_LOAD16_BYTE( "95420969.hi", 0x00000, 0x080000, CRC(49517bf0) SHA1(a62562377c362d227f5028f6607b8b924570f1fb) )
	sc_suscl_others
ROM_END

ROM_START( sc4susclb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421890.lo", 0x00001, 0x080000, CRC(b2a0e9a0) SHA1(e771b3932f68d063592965d40dacd137029c669d) )
	ROM_LOAD16_BYTE( "95421891.hi", 0x00000, 0x080000, CRC(aae9e840) SHA1(a06e8328219eccd7c9c7169fcc49d32fd263d6b7) )
	sc_suscl_others
ROM_END

ROM_START( sc4susclc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421968.lo", 0x00001, 0x080000, CRC(bd6d5125) SHA1(900ddd697279f5cd4c57a9a1827be9fb28c9bb5f) )
	ROM_LOAD16_BYTE( "95421969.hi", 0x00000, 0x080000, CRC(6fa7aaee) SHA1(fb0e9ccf68f7356eeb5b5db630701e25bd2ebed6) )
	sc_suscl_others
ROM_END




ROM_START( sc4s6c )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "club-super-6_dat_ac_250pnd_p65ss.hi", 0x00000, 0x080000, CRC(6a0d7c4a) SHA1(e2529e42d3942ead051632ff106749e6be72baaa) )
	ROM_LOAD16_BYTE( "club-super-6_dat_ac_250pnd_p65ss.lo", 0x00001, 0x080000, CRC(174d52ef) SHA1(c7bc96f2da4eb8b0d707fe7588e7262daa625e44) )
	sc_s6c_others
ROM_END

ROM_START( sc4s6ca )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "club-super-6_dat_ac_250pndss.hi", 0x00000, 0x080000, CRC(1df2874e) SHA1(f3bbcef21d387e0b2e83cb450f050ffd752671bc) )
	ROM_LOAD16_BYTE( "club-super-6_dat_ac_250pndss.lo", 0x00001, 0x080000, CRC(018015a6) SHA1(23a6ae78c2e5787eac98152325da114da8617a1e) )
	sc_s6c_others
ROM_END

ROM_START( sc4s6cb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "club-super-6_std_ac_250pnd_p65ss.hi", 0x00000, 0x080000, CRC(c3e39630) SHA1(cca733107bce7ea1391f03069bc9b8eaee9e1b4c) )
	ROM_LOAD16_BYTE( "club-super-6_std_ac_250pnd_p65ss.lo", 0x00001, 0x080000, CRC(06d1ef16) SHA1(533d245bca5382b9a30b7cf48d2e28b790ec8a69) )
	sc_s6c_others
ROM_END

ROM_START( sc4s6cc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "club-super-6_std_ac_250pndss.hi", 0x00000, 0x080000, CRC(b41c6d34) SHA1(1a64f14ccd57bd5957d653188da9391a9e965d6e) )
	ROM_LOAD16_BYTE( "club-super-6_std_ac_250pndss.lo", 0x00001, 0x080000, CRC(101ca85f) SHA1(d9b41c20e821d6af5e418b954328b7fbcdc3ef31) )
	sc_s6c_others
ROM_END

ROM_START( sc4s6cd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "s6218hi", 0x00000, 0x080000, CRC(4a7709e9) SHA1(d9b8062604c9cb2720fbd6c018b14808ca85b9db) )
	ROM_LOAD16_BYTE( "s6217lo", 0x00001, 0x080000, CRC(ee72e713) SHA1(9ff3201808421fae2bbcab8f15a9b74ac7c6b153) )
	sc_s6c_others
ROM_END

ROM_START( sc4s6ce )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "super6hi", 0x00000, 0x080000, CRC(4ff2489c) SHA1(fedd80cbe9a48682a89bd0c36b4db20a494cdf13) )
	ROM_LOAD16_BYTE( "super6lo", 0x00001, 0x080000, CRC(d11b6a66) SHA1(fd2e647679f764be3b2c21e295f3f10645fef3ec) )
	sc_s6c_others
ROM_END

ROM_START( sc4s6cf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "club_super_six_95400336_vssix_1.9_lo.bin", 0x000000, 0x080000, CRC(828e1e1a) SHA1(00d7ab13e89a28f49dddf5755e09a8a1d23de440) )
	ROM_LOAD16_BYTE( "club_super_six_95400337_vssix_1.9_hi.bin", 0x000001, 0x080000, CRC(2bed3af5) SHA1(56c232cd9d7d8fa0bc02e314edd315f50d07764f) )
	sc_s6c_others
ROM_END

ROM_START( sc4sdr )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400453.lo", 0x00001, 0x080000, CRC(3d8eb03c) SHA1(802b05db351004a999c1881e1d33cbc1af250041) )
	ROM_LOAD16_BYTE( "95400454.hi", 0x00000, 0x080000, CRC(15f51ee7) SHA1(079fe958f159b3af70039b2bcbfa5afe9ecffe53) )
	sc_sdr_others
ROM_END

ROM_START( sc4sdra )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400665.lo", 0x00001, 0x080000, CRC(659950fe) SHA1(c435e45ff7c4dae3416b1c1e9332b35b55814c54) )
	ROM_LOAD16_BYTE( "95400666.hi", 0x00000, 0x080000, CRC(4e94bf2a) SHA1(5f1abae2a5fd5cd317698bb731f49b3887049aad) )
	sc_sdr_others
ROM_END

ROM_START( sc4sdrb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401453.lo", 0x00001, 0x080000, CRC(13622e8c) SHA1(7459caebdebe31c9504b682423452c450e0cac10) )
	ROM_LOAD16_BYTE( "95401454.hi", 0x00000, 0x080000, CRC(01fb42a4) SHA1(f2741aced5571342fb45bc6a8c55f0a968547b59) )
	sc_sdr_others
ROM_END

ROM_START( sc4sdrc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401665.lo", 0x00001, 0x080000, CRC(0660ec51) SHA1(ad61471e5a7879bc793a4f2bc70a2023947aeb31) )
	ROM_LOAD16_BYTE( "95401666.hi", 0x00000, 0x080000, CRC(2cf30c33) SHA1(b79a303dc45622156f70bd500a5aff88595e2de1) )
	sc_sdr_others
ROM_END




ROM_START( sc4sstep )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "96000126.odd", 0x00001, 0x080000, CRC(c7307b16) SHA1(67f920428cfd224e7c1b34ccfd336645e30b04f4) )
	ROM_LOAD16_BYTE( "96000127.evn", 0x00000, 0x080000, CRC(7a380957) SHA1(d800bb9341b216249fd3eae0e97f61da57f50efa) )
	sc_sstep_others
ROM_END

ROM_START( sc4sstepa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "96000128.odd", 0x00001, 0x080000, CRC(762681f6) SHA1(0a7756fa1a82efdec31b8f544e403aeaa3d13ee9) )
	ROM_LOAD16_BYTE( "96000129.evn", 0x00000, 0x080000, CRC(d3d3de05) SHA1(d786ce6a081a04c49096c126832e88c4b29d2d86) )
	sc_sstep_others
ROM_END

ROM_START( sc4sstepb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "96000144.odd", 0x00001, 0x080000, CRC(8e097545) SHA1(cefba0da599b690083631ab14afc617b588f5eea) )
	ROM_LOAD16_BYTE( "96000145.evn", 0x00000, 0x080000, CRC(f6577c93) SHA1(050080efe421e157b4bf19d9e150f096e9dfcd98) )
	sc_sstep_others
ROM_END


ROM_START( sc4supst )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416186.lo", 0x00001, 0x080000, CRC(030d89c3) SHA1(68fd5954d8983de6871bbd606f17f76099bc5f4e) )
	ROM_LOAD16_BYTE( "95416187.hi", 0x00000, 0x080000, CRC(cb7766f3) SHA1(5562d8ceb0ebea053d45698bc411a13eccea6e3c) )
	sc_supst_others
ROM_END

ROM_START( sc4supsta )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417186.lo", 0x00001, 0x080000, CRC(c3fced7d) SHA1(6f13a1c2c1fab635b70c43f625c2caef977685e6) )
	ROM_LOAD16_BYTE( "95417187.hi", 0x00000, 0x080000, CRC(e5ec3755) SHA1(04ea1d3cf5ffa9b9b74d0bf849610e99373629f2) )
	sc_supst_others
ROM_END




ROM_START( sc4taekw )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406628.odd", 0x00001, 0x080000, CRC(2bec7587) SHA1(606d0fab58d86cf089098e8668db99f4a0af63ca) )
	ROM_LOAD16_BYTE( "95406629.evn", 0x00000, 0x080000, CRC(b8357f6e) SHA1(54b5d4327b462f8945bc0dafa8200c3158e871fd) )
	sc_taekw_others
ROM_END

ROM_START( sc4taekwa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407628.odd", 0x00001, 0x080000, CRC(c2c48ca1) SHA1(d324f18c8a86e12d48bb0f453d28bd657e575f9e) )
	ROM_LOAD16_BYTE( "95407629.evn", 0x00000, 0x080000, CRC(72828a7d) SHA1(8224773d5e07af109e949f8d976f714ecc96fcdd) )
	sc_taekw_others
ROM_END

ROM_START( sc4taekwb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409328.odd", 0x00001, 0x080000, CRC(49eaf56b) SHA1(50b15f22b21b48909d1ab0b3a048a3144808fb20) )
	ROM_LOAD16_BYTE( "95409329.evn", 0x00000, 0x080000, CRC(e3709b15) SHA1(74528076471f339cfe1a11dbfa5db3fe893f495d) )
	sc_taekw_others
ROM_END

ROM_START( sc4taekwc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410764.odd", 0x00001, 0x080000, CRC(9c832577) SHA1(53f4c063f518e7e47ee2a62b4e12276905c0a856) )
	ROM_LOAD16_BYTE( "95410765.evn", 0x00000, 0x080000, CRC(b4773b41) SHA1(06a800bdf5717b8a1d87d494fa8fc86627c01917) )
	sc_taekw_others
ROM_END

ROM_START( sc4taekwd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410766.odd", 0x00001, 0x080000, CRC(4b465873) SHA1(2a5fc9c240b709482197e90c283bc10c5a3bedd4) )
	ROM_LOAD16_BYTE( "95410767.evn", 0x00000, 0x080000, CRC(a360986c) SHA1(f855b6065bf1832a8a6bd70de606f42a8b9fd91a) )
	sc_taekw_others
ROM_END

ROM_START( sc4taekwe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411764.odd", 0x00001, 0x080000, CRC(ee13bef8) SHA1(b54577e1eacf051e74d61dd675b740f888364a1d) )
	ROM_LOAD16_BYTE( "95411765.evn", 0x00000, 0x080000, CRC(3cdbe9f4) SHA1(377ae83afb4fa203e51028754856aaf93521f0fe) )
	sc_taekw_others
ROM_END

ROM_START( sc4taekwf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411766.odd", 0x00001, 0x080000, CRC(6140dd05) SHA1(3dd1d50f4cbc1a6ac684da3d7646fa4a3d39e908) )
	ROM_LOAD16_BYTE( "95411767.evn", 0x00000, 0x080000, CRC(8f19c712) SHA1(8f39d589a8841626177a6613ae7afdc680681f26) )
	sc_taekw_others
ROM_END

ROM_START( sc4taekwg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414300.odd", 0x00001, 0x080000, CRC(e46fa6f8) SHA1(6a3c5547fc011a5168eca205697c5fe0810aca6a) )
	ROM_LOAD16_BYTE( "95414301.evn", 0x00000, 0x080000, CRC(1737b892) SHA1(e641cea174c130003435b9b1a13740195d09ec3d) )
	sc_taekw_others
ROM_END

ROM_START( sc4taekwh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414302.odd", 0x00001, 0x080000, CRC(095e61ff) SHA1(6d6340b595047f9eebe9e025092da9b7fbbe0c1a) )
	ROM_LOAD16_BYTE( "95414303.evn", 0x00000, 0x080000, CRC(8bb752bb) SHA1(b06e9949ec8c1374202fcd1116b5535f90198d29) )
	sc_taekw_others
ROM_END

ROM_START( sc4taekwi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415300.odd", 0x00001, 0x080000, CRC(a197c77b) SHA1(fdb1df9647b6409ad477c33088c7bad82ac96530) )
	ROM_LOAD16_BYTE( "95415301.evn", 0x00000, 0x080000, CRC(2f40bbaf) SHA1(ab14d5e5eef0c225bfdfeea9e4939b8d0ed5a1a1) )
	sc_taekw_others
ROM_END

ROM_START( sc4taekwj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415302.odd", 0x00001, 0x080000, CRC(4ca6007c) SHA1(1445aed62119258874f19b8b1113171c9bed9d1c) )
	ROM_LOAD16_BYTE( "95415303.evn", 0x00000, 0x080000, CRC(b3c05186) SHA1(cb442b1e2e51cde3fc705c02da2f6a57e2d3816e) )
	sc_taekw_others
ROM_END


ROM_START( sc4taknt )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406760.lo", 0x00001, 0x080000, CRC(54952a27) SHA1(a2f80249cecd0129dae67ee1b99d90e51ce79db6) )
	ROM_LOAD16_BYTE( "95406761.hi", 0x00000, 0x080000, CRC(9b87e3d0) SHA1(ae1cee519938828eef3c97a04c7b6080e9f13f49) )
	sc_taknt_others
ROM_END


ROM_START( sc4taknta )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407760.lo", 0x00001, 0x080000, CRC(d873b140) SHA1(aedc72d2804ebf1cc4fb04c54ffc1fba1e192363) )
	ROM_LOAD16_BYTE( "95407761.hi", 0x00000, 0x080000, CRC(73e60179) SHA1(9c0913b8a9dba5794a01bdcb02b15cdde976da26) )
	sc_taknt_others
ROM_END


ROM_START( sc4takcl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408062.lo", 0x00001, 0x080000, CRC(bb70c78a) SHA1(187409c94b89b3990e464beef0a78d1c7fe6038d) )
	ROM_LOAD16_BYTE( "95408063.hi", 0x00000, 0x080000, CRC(7f85a7bd) SHA1(5355d2ac07790796379f376ff334d0c17cc913f6) )
	sc_takcl_others
ROM_END

ROM_START( sc4takcla )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408190.lo", 0x00001, 0x080000, CRC(64f7f871) SHA1(96174a64b2d049b1792ae61596c1cf0826daa296) )
	ROM_LOAD16_BYTE( "95408191.hi", 0x00000, 0x080000, CRC(0ac54f9f) SHA1(342c2e0663bec2ff3bbb0f411595766a5e5a1d84) )
	sc_takcl_others
ROM_END

ROM_START( sc4takclb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408192.lo", 0x00001, 0x080000, CRC(aa7275f1) SHA1(8dd3acbd0e60b6eb7c81030c4639b5671a5c1932) )
	ROM_LOAD16_BYTE( "95408193.hi", 0x00000, 0x080000, CRC(237c9b8f) SHA1(c8e06a8a01754651888a5ae37e40108e5fd74f4e) )
	sc_takcl_others
ROM_END

ROM_START( sc4takclc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408216.lo", 0x00001, 0x080000, CRC(e375e8bc) SHA1(9022511264ba3c4eaf6ed4a5ff76ea23ba6bbd83) )
	ROM_LOAD16_BYTE( "95408217.hi", 0x00000, 0x080000, CRC(111fa110) SHA1(61061fb4e487f79806499edcaa7b8d0e47c5858e) )
	sc_takcl_others
ROM_END

ROM_START( sc4takcld )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409062.lo", 0x00001, 0x080000, CRC(e37d3f32) SHA1(db8a9c03312367899f260380a40d621d151c6897) )
	ROM_LOAD16_BYTE( "95409063.hi", 0x00000, 0x080000, CRC(69f6a31e) SHA1(09184ac75cd6e01f04517d71d54b8fc7646aa98d) )
	sc_takcl_others
ROM_END

ROM_START( sc4takcle )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409192.lo", 0x00001, 0x080000, CRC(00202d5f) SHA1(c84bbf956fa711765575a69a29454952455bd0ad) )
	ROM_LOAD16_BYTE( "95409193.hi", 0x00000, 0x080000, CRC(c7503f3a) SHA1(00a3bb77f590f87ac28e7a64f7e329b30ee45b3a) )
	sc_takcl_others
ROM_END

ROM_START( sc4takclf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409216.lo", 0x00001, 0x080000, CRC(2741fd41) SHA1(40a8e44096fd023acbc0e8d88cbb111f7d0b1c74) )
	ROM_LOAD16_BYTE( "95409217.hi", 0x00000, 0x080000, CRC(a70084d3) SHA1(40de10c882948e379823afb088453ab423697feb) )
	sc_takcl_others
ROM_END

ROM_START( sc4takclg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412196.lo", 0x00001, 0x080000, CRC(32082a2c) SHA1(04e8832f8e0786b7308ef047fe82b19eb34a944b) )
	ROM_LOAD16_BYTE( "95412197.hi", 0x00000, 0x080000, CRC(24260e5d) SHA1(5b68817b3a20e542a827284847204fc9b028d27f) )
	sc_takcl_others
ROM_END

ROM_START( sc4takclh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413196.lo", 0x00001, 0x080000, CRC(44d33d52) SHA1(468d2e34c636233b80ded6fad1d0c21ddd0326a0) )
	ROM_LOAD16_BYTE( "95413197.hi", 0x00000, 0x080000, CRC(336ba68e) SHA1(961690dbe9321ab6ac3f4d7445e09d84acbfe3b4) )
	sc_takcl_others
ROM_END

ROM_START( sc4takcli )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1323p11.hi", 0x00000, 0x080000, CRC(957b4bfe) SHA1(9e862d7e2b9fa4ff2b65375706d65abffabbaff1) )
	ROM_LOAD16_BYTE( "p1323p11.lo", 0x00001, 0x080000, CRC(e7fccbef) SHA1(cbde295fc4049004a59d803457a2fdf8566fafe9) )
	sc_takcl_others
ROM_END

ROM_START( sc4takclj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1323s11.hi", 0x00000, 0x080000, CRC(c1d0ae13) SHA1(64427a39b674b48889f7db18213cb8736056e579) )
	ROM_LOAD16_BYTE( "p1323s11.lo", 0x00001, 0x080000, CRC(8a5a3f35) SHA1(4fd85f1be400e401b48f8a0ad621f1d6f314ce8b) )
	sc_takcl_others
ROM_END



ROM_START( sc4ttp )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400569.lo", 0x00001, 0x080000, CRC(b862aba4) SHA1(68c9cd5c05bdc73ea455ecb320760f4b27f70f36) )
	ROM_LOAD16_BYTE( "95400570.hi", 0x00000, 0x080000, CRC(dbd9166f) SHA1(a7690f7f87257daa329fd35a96602fdd38b5c9c7) )
	sc_ttp_others
ROM_END

ROM_START( sc4ttpa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400573.lo", 0x00001, 0x080000, CRC(471e1036) SHA1(4058d341620dc4012a22c1f7a3e804f46ee71f5a) )
	ROM_LOAD16_BYTE( "95400574.hi", 0x00000, 0x080000, CRC(53ec8d50) SHA1(69ef502406b5770ed6d3722a2d48be3f02c047a4) )
	sc_ttp_others
ROM_END

ROM_START( sc4ttpb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400581.lo", 0x00001, 0x080000, CRC(0f14ac4b) SHA1(e02c5474cf236507d4ef980123039c9b125b1171) )
	ROM_LOAD16_BYTE( "95400582.hi", 0x00000, 0x080000, CRC(529fbaad) SHA1(9807f06e202ba5d83432cbcb2ac1ec51f69b60cd) )
	sc_ttp_others
ROM_END

ROM_START( sc4ttpc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401569.lo", 0x00001, 0x080000, CRC(512ca74f) SHA1(d4feb496033db25941f25b5fd2d3aa7d0ea949d5) )
	ROM_LOAD16_BYTE( "95401570.hi", 0x00000, 0x080000, CRC(311277c3) SHA1(43ce32bd16c81862cc65e008ce0f36d354e88ed5) )
	sc_ttp_others
ROM_END

ROM_START( sc4ttpd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401571.lo", 0x00001, 0x080000, CRC(eccd6a24) SHA1(bca44c81b6eebe0d2ae308f1adde34b6670876cc) )
	ROM_LOAD16_BYTE( "95401572.hi", 0x00000, 0x080000, CRC(47e235f0) SHA1(90ea66e3b44519d456730fb8094c6963c13f8e3f) )
	sc_ttp_others
ROM_END

ROM_START( sc4ttpe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401581.lo", 0x00001, 0x080000, CRC(47962069) SHA1(e050947f32a18d0a8a21d7eeabb24ed4cf978b6d) )
	ROM_LOAD16_BYTE( "95401582.hi", 0x00000, 0x080000, CRC(51431805) SHA1(aca3b8431f2eb4eba447b38056171fa15f071799) )
	sc_ttp_others
ROM_END

ROM_START( sc4ttpf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401585.lo", 0x00001, 0x080000, CRC(fab5c9f2) SHA1(347d250fbc4a8d793ccaa0c3e805a8d67b9629db) )
	ROM_LOAD16_BYTE( "95401586.hi", 0x00000, 0x080000, CRC(feba871e) SHA1(e4d3b3bdb90dee9f5eeeedb7e5febd71eefb4e12) )
	sc_ttp_others
ROM_END




ROM_START( sc4tempt )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402638.lo", 0x00001, 0x080000, CRC(f857ccfd) SHA1(bbef92c4682a9a491dcdb82569ca84e7dc2b95ee) )
	ROM_LOAD16_BYTE( "95402639.hi", 0x00000, 0x080000, CRC(97fef5fb) SHA1(dde5375e864d6b84abf67c698bfa1443594b7caa) )
	sc_tempt_others
ROM_END

ROM_START( sc4tempta )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402640.lo", 0x00001, 0x080000, CRC(af1b6f7d) SHA1(3ded3642bb5209267d68188f7718312db3d9d351) )
	ROM_LOAD16_BYTE( "95402641.hi", 0x00000, 0x080000, CRC(4e30a12b) SHA1(0d2cbe6c3e5cd92758b5c840ae3c61f558276c38) )
	sc_tempt_others
ROM_END

ROM_START( sc4temptb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403638.lo", 0x00001, 0x080000, CRC(6d10d513) SHA1(5bbce9d10d547bba4a2639bb57d8f4ffe546f01c) )
	ROM_LOAD16_BYTE( "95403639.hi", 0x00000, 0x080000, CRC(12f616c7) SHA1(04dbe7d7c0649878718214fab945a4dca3a4a3be) )
	sc_tempt_others
ROM_END

ROM_START( sc4temptc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403640.lo", 0x00001, 0x080000, CRC(8d9c0c0d) SHA1(e9d7ce40e0a8c84d128a59fbddcb792e6d8206a9) )
	ROM_LOAD16_BYTE( "95403641.hi", 0x00000, 0x080000, CRC(20cd8fcb) SHA1(6388c3a2f267c6d16f74ebef9f86c86971c90abd) )
	sc_tempt_others
ROM_END

ROM_START( sc4temptd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406062.lo", 0x00001, 0x080000, CRC(902622d4) SHA1(414814fb077a452873c9db0c37b66d6f20cb211b) )
	ROM_LOAD16_BYTE( "95406063.hi", 0x00000, 0x080000, CRC(7a8ec2b0) SHA1(0c7ccfdba6365c4f49ea569a2485f7b42a16dedf) )
	sc_tempt_others
ROM_END

ROM_START( sc4tempte )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406932.lo", 0x00001, 0x080000, CRC(a4dca430) SHA1(7de4040493bf595ecc01fd308b3a4efd46d36c75) )
	ROM_LOAD16_BYTE( "95406933.hi", 0x00000, 0x080000, CRC(93a734e4) SHA1(2312f33b2d84ebe0af1a76423b6394d617cdf703) )
	sc_tempt_others
ROM_END

ROM_START( sc4temptf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407062.lo", 0x00001, 0x080000, CRC(35a32fd4) SHA1(8af600c67b1ec21a127077cf5f50752bc788592d) )
	ROM_LOAD16_BYTE( "95407063.hi", 0x00000, 0x080000, CRC(426697b5) SHA1(a8dfd5b1a4404b5a123a2b1ca075623978490058) )
	sc_tempt_others
ROM_END

ROM_START( sc4temptg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407932.lo", 0x00001, 0x080000, CRC(42445fdd) SHA1(b9b133d183cb4104b493b9ef4535ad54ded95ba8) )
	ROM_LOAD16_BYTE( "95407933.hi", 0x00000, 0x080000, CRC(c5dfc2a2) SHA1(37f302fbaf2b658371545caff6f0b2ebe6228d00) )
	sc_tempt_others
ROM_END


ROM_START( sc4tetri )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402427.lo", 0x00001, 0x080000, CRC(62395361) SHA1(aab5bd825d8070dbe04c224189caa2285d478b30) )
	ROM_LOAD16_BYTE( "95402428.hi", 0x00000, 0x080000, CRC(871c2de1) SHA1(d0d01cb72d9a34b325748d9fc96e2bc4c7ccc801) )
	sc_tetri_others
ROM_END

ROM_START( sc4tetria )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402666.lo", 0x00001, 0x080000, CRC(a0ae1eb6) SHA1(eae2364fdaecd816216f191e3829f056878b1d54) )
	ROM_LOAD16_BYTE( "95402667.hi", 0x00000, 0x080000, CRC(b433a747) SHA1(8d458b4f8d04d39b45ec818de93dc9fdfcf64af2) )
	sc_tetri_others
ROM_END

ROM_START( sc4tetrib )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402718.lo", 0x00001, 0x080000, CRC(7558a203) SHA1(ef5ccf3d5299a9d01c009dc91c888a598bc73d33) )
	ROM_LOAD16_BYTE( "95402719.hi", 0x00000, 0x080000, CRC(73d9b634) SHA1(e2ea94b645fd0d542a3d7ac0968a226a91aa5cea) )
	sc_tetri_others
ROM_END

ROM_START( sc4tetric )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402776.lo", 0x00001, 0x080000, CRC(5485c0af) SHA1(080e37acd6dc6fe54e25872d19749a902cc4d22e) )
	ROM_LOAD16_BYTE( "95402777.hi", 0x00000, 0x080000, CRC(88edd93d) SHA1(729ed52e1bded9b189388387a7ccb263250bebfa) )
	sc_tetri_others
ROM_END

ROM_START( sc4tetrid )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403427.lo", 0x00001, 0x080000, CRC(e1520224) SHA1(ba4da04b3caa4ce7733c12d59c0b831d253a56d5) )
	ROM_LOAD16_BYTE( "95403428.hi", 0x00000, 0x080000, CRC(c8a33bce) SHA1(e21a1c42e9d5ff91c2e003ab99799930aa404f2c) )
	sc_tetri_others
ROM_END

ROM_START( sc4tetrie )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403666.lo", 0x00001, 0x080000, CRC(2e177c3b) SHA1(66b58e3292eac398b10459a33063457c6629e86e) )
	ROM_LOAD16_BYTE( "95403667.hi", 0x00000, 0x080000, CRC(61a8baa2) SHA1(5a74e357fee5575e92c33e0a353a45f63b98c26d) )
	sc_tetri_others
ROM_END

ROM_START( sc4tetrif )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403772.lo", 0x00001, 0x080000, CRC(61e52bae) SHA1(dc5701068d849f2b83eef488c5972aa977e0cefc) )
	ROM_LOAD16_BYTE( "95403773.hi", 0x00000, 0x080000, CRC(135d2457) SHA1(112b71f6a8b5013db42b3b75b8b11eed0f3d9764) )
	sc_tetri_others
ROM_END

ROM_START( sc4tetrig )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403774.lo", 0x00001, 0x080000, CRC(f37ba971) SHA1(6b28f592722592eeae15facc598c2f580321f351) )
	ROM_LOAD16_BYTE( "95403775.hi", 0x00000, 0x080000, CRC(864280e8) SHA1(b513c70254c179131ba84a0f8e863e36440ec8ea) )
	sc_tetri_others
ROM_END

ROM_START( sc4tetrih )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404382.lo", 0x00001, 0x080000, CRC(f17456cb) SHA1(37af77ee1d2016aa730c6dc34a33295a3a3e46ab) )
	ROM_LOAD16_BYTE( "95404383.hi", 0x00000, 0x080000, CRC(ff1c5391) SHA1(a09d97c1f3fcdcfc253e0e91874eda226f6dd5ba) )
	sc_tetri_others
ROM_END

ROM_START( sc4tetrii )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405382.lo", 0x00001, 0x080000, CRC(7a132018) SHA1(a5fa629df3a41e49d5a5ed82da2758bb4fc2470b) )
	ROM_LOAD16_BYTE( "95405383.hi", 0x00000, 0x080000, CRC(03a4369e) SHA1(6c963cdd16207663605edc854c11b4d4cf77ff94) )
	sc_tetri_others
ROM_END

ROM_START( sc4tetrij )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406156.lo", 0x00001, 0x080000, CRC(573e0679) SHA1(5baa0d8ae514e5c7d3aa52107368226e324d521c) )
	ROM_LOAD16_BYTE( "95406157.hi", 0x00000, 0x080000, CRC(e8c90675) SHA1(86e8631040e40fbf003fcc7ecb7bd68239f023c9) )
	sc_tetri_others
ROM_END

ROM_START( sc4tetrik )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407156.lo", 0x00001, 0x080000, CRC(c938c7a0) SHA1(3f4e1f5f24b4a23116f6b84b95831d2028b55296) )
	ROM_LOAD16_BYTE( "95407157.hi", 0x00000, 0x080000, CRC(36eff91d) SHA1(9dbb86fbaa43348e42d12aa6732c6d9881029ddc) )
	sc_tetri_others
ROM_END


ROM_START( sc4lions )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400421.lo", 0x00001, 0x080000, CRC(3caa5aa3) SHA1(e85f8b61a5d2424cf43bf14a3548ada41f585c34) )
	ROM_LOAD16_BYTE( "95400422.hi", 0x00000, 0x080000, CRC(8135ffc1) SHA1(71c686ff116a0fb0da196025c519e59caa8d7539) )
	sc_lions_others
ROM_END

ROM_START( sc4lionsa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400431.lo", 0x00001, 0x080000, CRC(9f51e8d7) SHA1(aaf09fb80852803016851390fa3b095b02cb8609) )
	ROM_LOAD16_BYTE( "95400432.hi", 0x00000, 0x080000, CRC(2b07b713) SHA1(222fb6c2b48484694e1d0c853fdc0e5ff36bfb47) )
	sc_lions_others
ROM_END

ROM_START( sc4lionsb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400597.lo", 0x00001, 0x080000, CRC(8a1ef3f6) SHA1(0a76135465f7d571780fb9e4e096350f29889918) )
	ROM_LOAD16_BYTE( "95400598.hi", 0x00000, 0x080000, CRC(3ad50b6a) SHA1(caae3c7397fabea264238f29c7c85a4d25fa81f9) )
	sc_lions_others
ROM_END

ROM_START( sc4lionsc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401421.lo", 0x00001, 0x080000, CRC(f9f4b4f2) SHA1(ccd165d94d9b30b122a8950235e30191c7193ec2) )
	ROM_LOAD16_BYTE( "95401422.hi", 0x00000, 0x080000, CRC(b69b9595) SHA1(e3ccb82b12f1122a5df663a644975d192b9bc3cb) )
	sc_lions_others
ROM_END

ROM_START( sc4lionsd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401423.lo", 0x00001, 0x080000, CRC(26c0cce6) SHA1(9bd74003c570a61163cc2e09929f4907cfb26144) )
	ROM_LOAD16_BYTE( "95401424.hi", 0x00000, 0x080000, CRC(bd8798ea) SHA1(c6970dddb699ece8f94303b770374e3b6b529e12) )
	sc_lions_others
ROM_END

ROM_START( sc4lionse )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401431.lo", 0x00001, 0x080000, CRC(5a0f0686) SHA1(e994e68a8912b37d3b84724c65713b4193ae04bf) )
	ROM_LOAD16_BYTE( "95401432.hi", 0x00000, 0x080000, CRC(1ca9dd47) SHA1(9eee1ee86a7358b691e675c538f8ac20517484c0) )
	sc_lions_others
ROM_END

ROM_START( sc4lionsf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401597.lo", 0x00001, 0x080000, CRC(5f9b800c) SHA1(f50477c702913af9debccf44b9d9c3f97c4e7cb1) )
	ROM_LOAD16_BYTE( "95401598.hi", 0x00000, 0x080000, CRC(894d2e72) SHA1(50a0af7d7a1f21e76ecea9174b74dff05a7e9cb5) )
	sc_lions_others
ROM_END


ROM_START( sc4tic2 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402644.lo", 0x00001, 0x080000, CRC(5f7e15ee) SHA1(640928d71ce75c447ce4453b7f84c1723aeee41e) )
	ROM_LOAD16_BYTE( "95402645.hi", 0x00000, 0x080000, CRC(3579696d) SHA1(44439ae8fa5c20db1f24657661b6303c42f91709) )
	sc_tic2_others
ROM_END

ROM_START( sc4tic2a )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402646.lo", 0x00001, 0x080000, CRC(87389e10) SHA1(2bff29fde2e73426f01c854caa5899a1db196a84) )
	ROM_LOAD16_BYTE( "95402647.hi", 0x00000, 0x080000, CRC(f6cf7e5c) SHA1(1fc223388e5d94bd4e92740c68a668b92a939917) )
	sc_tic2_others
ROM_END

ROM_START( sc4tic2b )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402672.lo", 0x00001, 0x080000, CRC(49af7863) SHA1(b58eecc60afa4f4937dc04d7c3600af200bdbf29) )
	ROM_LOAD16_BYTE( "95402673.hi", 0x00000, 0x080000, CRC(a5823498) SHA1(5200b3af9cbeb4ab110ee4690d5a8bafb8c83387) )
	sc_tic2_others
ROM_END

ROM_START( sc4tic2c )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403644.lo", 0x00001, 0x080000, CRC(4d906e99) SHA1(28d2b341e414fb342af95766f98ad558df3d8e71) )
	ROM_LOAD16_BYTE( "95403645.hi", 0x00000, 0x080000, CRC(6ff68f71) SHA1(e7a51bea03ca9261304eec6f34e3d6d8aa59d4aa) )
	sc_tic2_others
ROM_END

ROM_START( sc4tic2d )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403672.lo", 0x00001, 0x080000, CRC(1ec80db0) SHA1(196e81d2c72385c341dfee1df9ba123c077dd450) )
	ROM_LOAD16_BYTE( "95403673.hi", 0x00000, 0x080000, CRC(2e6d12bd) SHA1(0bf59b9844ba5946f73011e7af390e35e836c6f6) )
	sc_tic2_others
ROM_END

ROM_START( sc4tic2e )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406404.lo", 0x00001, 0x080000, CRC(03929a61) SHA1(14317ee7a50efb5e25f7e6c928c3830cbd6f9879) )
	ROM_LOAD16_BYTE( "95406405.hi", 0x00000, 0x080000, CRC(fbdb0804) SHA1(c1b6578420ecca300254bae9087b5de07342e902) )
	sc_tic2_others
ROM_END

ROM_START( sc4tic2f )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407404.lo", 0x00001, 0x080000, CRC(0981631b) SHA1(a05389b016098333e1624b059b21247408a8cbe0) )
	ROM_LOAD16_BYTE( "95407405.hi", 0x00000, 0x080000, CRC(f402616d) SHA1(d9319a67f11ee4652fc767cb17e6e59900bc10b9) )
	sc_tic2_others
ROM_END

ROM_START( sc4tic2g )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410220.lo", 0x00001, 0x080000, CRC(0244fbe1) SHA1(3744e169f7d39df905f8a90fbbd5e384fdabdb6f) )
	ROM_LOAD16_BYTE( "95410221.hi", 0x00000, 0x080000, CRC(d5880d07) SHA1(fd9373df8a879db1cbbbdcc417e32fa889b738b9) )
	sc_tic2_others
ROM_END

ROM_START( sc4tic2h )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411220.lo", 0x00001, 0x080000, CRC(6f9e123a) SHA1(178f079dfb51537c24e5dba06e83bf1f0cb76c07) )
	ROM_LOAD16_BYTE( "95411221.hi", 0x00000, 0x080000, CRC(58642ea6) SHA1(6b2bd7b8950200a098d2095f7def0f97db9793f1) )
	sc_tic2_others
ROM_END

ROM_START( sc4tic2i )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412300.lo", 0x00001, 0x080000, CRC(63fd59a5) SHA1(3063fa7b420da2380774e20cbac390da78e268fc) )
	ROM_LOAD16_BYTE( "95412301.hi", 0x00000, 0x080000, CRC(7999b416) SHA1(71ef7b363e467ce79c65acb3973a441f9535826a) )
	sc_tic2_others
ROM_END

ROM_START( sc4tic2j )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413300.lo", 0x00001, 0x080000, CRC(5910aade) SHA1(40c8dd00078e089e5c0c5d17a75896ef199533d2) )
	ROM_LOAD16_BYTE( "95413301.hi", 0x00000, 0x080000, CRC(875c89f9) SHA1(ae68d772c4f73f95091ae499d66c5adddc3cad6c) )
	sc_tic2_others
ROM_END

ROM_START( sc4tic2k )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413306.lo", 0x00001, 0x080000, CRC(730c27ef) SHA1(e3d371b2ab184970177b89c792fd21bb12fbd1cb) )
	ROM_LOAD16_BYTE( "95413307.hi", 0x00000, 0x080000, CRC(46db37f4) SHA1(d54aa4f580255d3fc37cc32f58985d64f0e62954) )
	sc_tic2_others
ROM_END

ROM_START( sc4tic2l )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415932.lo", 0x00001, 0x080000, CRC(7a4c9542) SHA1(e9cede4592e00c3ab7cb9eeb767a679fafd2bb75) )
	ROM_LOAD16_BYTE( "95415933.hi", 0x00000, 0x080000, CRC(628f5395) SHA1(c793f9ae49d9527b1683d429c248ae11bcab421f) )
	sc_tic2_others
ROM_END

ROM_START( sc4tic2m )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415934.lo", 0x00001, 0x080000, CRC(aca05c76) SHA1(dddfe148e6be5f774262d6bad81e0a6f5fe1a3c5) )
	ROM_LOAD16_BYTE( "95415935.hi", 0x00000, 0x080000, CRC(7275ce41) SHA1(2c4989cffb56db1c73d6eaa0c135a579d0b8106f) )
	sc_tic2_others
ROM_END

ROM_START( sc4tic2n )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417026.lo", 0x00001, 0x080000, CRC(80dacc34) SHA1(7d7b48bcc4b2c088961ba63e8d0633b1235ceb72) )
	ROM_LOAD16_BYTE( "95417027.hi", 0x00000, 0x080000, CRC(b42c7e79) SHA1(cf2e47c3014040b97ad4ebcb31a8f782f1ec0077) )
	sc_tic2_others
ROM_END

ROM_START( sc4tic2o )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417028.lo", 0x00001, 0x080000, CRC(97c97c8d) SHA1(e5212802312fb8efdaf5fe13e40b8fcadfe299a8) )
	ROM_LOAD16_BYTE( "95417029.hi", 0x00000, 0x080000, CRC(47082b8e) SHA1(278f380fa0e036bdec08d75f55cbe83e7944bf3b) )
	sc_tic2_others
ROM_END


ROM_START( sc4tbana )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406070.lo", 0x00001, 0x080000, CRC(396f9515) SHA1(abd0f6f67103b99c72f517f000602eff7512d2e5) )
	ROM_LOAD16_BYTE( "95406071.hi", 0x00000, 0x080000, CRC(aae6b041) SHA1(53841ca78007de20642ef15e18017b1519ff0bda) )
	sc_tbana_others
ROM_END

ROM_START( sc4tbanaa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407070.lo", 0x00001, 0x080000, CRC(3797ad14) SHA1(0edca4df3626935e75c30237acdb50f60ce165f2) )
	ROM_LOAD16_BYTE( "95407071.hi", 0x00000, 0x080000, CRC(8a4a53d4) SHA1(026111764dedfd030cb824d89fcf45c7c1ebd76f) )
	sc_tbana_others
ROM_END




ROM_START( sc4tgear )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402185.lo", 0x00001, 0x080000, CRC(a6bdc445) SHA1(b469a0f911a15976c95ed103bf9c9949bed574af) )
	ROM_LOAD16_BYTE( "95402186.hi", 0x00000, 0x080000, CRC(7da86752) SHA1(e389f28da825d661017f29c9e66550f54ae667cb) )
	sc_tgear_others
ROM_END

ROM_START( sc4tgeara )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402365.lo", 0x00001, 0x080000, CRC(2d68a732) SHA1(31e5d64a25df4d563707392673377a606446cdfa) )
	ROM_LOAD16_BYTE( "95402366.hi", 0x00000, 0x080000, CRC(8c2d2aa1) SHA1(16159ef036134b48b6ea0c584fcaf5c3bf9de55f) )
	sc_tgear_others
ROM_END

ROM_START( sc4tgearb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402540.lo", 0x00001, 0x080000, CRC(657c486f) SHA1(218f001fc5d439fc4a617955b7e1d7d8a100a985) )
	ROM_LOAD16_BYTE( "95402541.hi", 0x00000, 0x080000, CRC(d38a9bdc) SHA1(81e6189544fd361d79263ccaf08a6aae320d8439) )
	sc_tgear_others
ROM_END

ROM_START( sc4tgearc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403365.lo", 0x00001, 0x080000, CRC(9a2bc700) SHA1(d2cdc13325e91bea12ade44fcff5bd2cec5d75df) )
	ROM_LOAD16_BYTE( "95403366.hi", 0x00000, 0x080000, CRC(088433d5) SHA1(aa5b7300b21d97b23dfe2f4b9ea507062a509baf) )
	sc_tgear_others
ROM_END

ROM_START( sc4tgeard )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403540.lo", 0x00001, 0x080000, CRC(4c88b6d4) SHA1(847289d2c5bba58338d8f617416b6bbed4ef3a10) )
	ROM_LOAD16_BYTE( "95403541.hi", 0x00000, 0x080000, CRC(354431b1) SHA1(b5be30cefab77c4919d8805989d4d31f51dccb87) )
	sc_tgear_others
ROM_END

ROM_START( sc4tgeare )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404168.lo", 0x00001, 0x080000, CRC(8f78f578) SHA1(e48cf81108a1160f1446e46eeb7ad7601851675c) )
	ROM_LOAD16_BYTE( "95404169.hi", 0x00000, 0x080000, CRC(99ce746b) SHA1(839a31a142ff28405822936453ccc94201dce543) )
	sc_tgear_others
ROM_END

ROM_START( sc4tgearf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405168.lo", 0x00001, 0x080000, CRC(ad7de0ae) SHA1(63a0c834edd2413eecd6de643813a89e1bc2242c) )
	ROM_LOAD16_BYTE( "95405169.hi", 0x00000, 0x080000, CRC(443429ab) SHA1(b8c585423781a8ef4e3558137ff06b90cdedeaf7) )
	sc_tgear_others
ROM_END

ROM_START( sc4tgearg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "top_gear.lo", 0x00001, 0x080000, CRC(96d1e012) SHA1(a5ca70b9371587b169c812873b0ef7848491ba3c) )
	ROM_LOAD16_BYTE( "top_gear.hi", 0x00000, 0x080000, CRC(8e819a04) SHA1(03ebd61123599dbf845a40b38f5dca7b9a6c4182) )
	sc_tgear_others
ROM_END


ROM_START( sc4tload )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408424.lo", 0x00001, 0x080000, CRC(1f8f93d4) SHA1(62349a83010f90931aa8a179db216e138824379e) )
	ROM_LOAD16_BYTE( "95408425.hi", 0x00000, 0x080000, CRC(a4a6930e) SHA1(c75d9ce127ba90e070412888551991cf7d46d7c6) )
	sc_tload_others
ROM_END







ROM_START( sc4ticlb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "club-treasure-island_dat_ac_p65ss.hi", 0x00000, 0x080000, CRC(3529018b) SHA1(aa49f68bb3ca550c77d4667b8373f94856a70cf3) )
	ROM_LOAD16_BYTE( "club-treasure-island_dat_ac_p65ss.lo", 0x00001, 0x080000, CRC(180fcb0a) SHA1(17400ae744af37b5a17389e4f7a639a5e5743b2d) )
	sc_ticlb_others

	sc_ticlb_matrix
ROM_END

ROM_START( sc4ticlba )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "club-treasure-island_dat_acss.hi", 0x00000, 0x080000, CRC(a51b94c9) SHA1(919f44f04a3b60b190e9c87c0d844ddde1268318) )
	ROM_LOAD16_BYTE( "club-treasure-island_dat_acss.lo", 0x00001, 0x080000, CRC(2d685792) SHA1(36329c83f247d284294e381293b25f745ea220da) )
	sc_ticlb_others

	sc_ticlb_matrix
ROM_END

ROM_START( sc4ticlbb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "club-treasure-island_std_ac_p65ss.hi", 0x00000, 0x080000, CRC(856d863b) SHA1(c00ee8032fd0dee6466a30a4bc59ae3b0cd4f556) )
	ROM_LOAD16_BYTE( "club-treasure-island_std_ac_p65ss.lo", 0x00001, 0x080000, CRC(2335e1b7) SHA1(ed32e892c9e71cccb642734e14eaea36f8f1853c) )
	sc_ticlb_others

	sc_ticlb_matrix
ROM_END

ROM_START( sc4ticlbc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "club-treasure-island_std_acss.hi", 0x00000, 0x080000, CRC(155f1379) SHA1(0768ce46a661f1280bd75897b82cca276ce7dbed) )
	ROM_LOAD16_BYTE( "club-treasure-island_std_acss.lo", 0x00001, 0x080000, CRC(16527d2f) SHA1(dba35f4b8f5953a1043610efe627a97b4fca8b74) )
	sc_ticlb_others

	sc_ticlb_matrix
ROM_END


ROM_START( sc4ticlbd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "club_treasure_island_95400361_lo.bin", 0x000001, 0x080000, CRC(15910e0f) SHA1(0ff9614de9a9333510ca269fca7f9b08db2695f9) )
	ROM_LOAD16_BYTE( "club_treasure_island_95400362_hi.bin", 0x000000, 0x080000, CRC(c0e6b3f7) SHA1(1c74fe6a5d6cae9486cf311e9b1c4c69e4552b42) )
	sc_ticlb_others

	sc_ticlb_matrix
ROM_END

ROM_START( sc4ttomb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406814.lo", 0x00001, 0x080000, CRC(9892b8b8) SHA1(c337a325def655a8878a1d04bdb5ddd73d01bf85) )
	ROM_LOAD16_BYTE( "95406815.hi", 0x00000, 0x080000, CRC(c46d3962) SHA1(17dfa99baf02d74b1f8118cf5f38adf66d9fc173) )
	sc_ttomb_others
ROM_END

ROM_START( sc4ttomba )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407814.lo", 0x00001, 0x080000, CRC(56387b17) SHA1(a489b2650e95c8a097285a7d5a9c78203f85e811) )
	ROM_LOAD16_BYTE( "95407815.hi", 0x00000, 0x080000, CRC(bf3fb189) SHA1(69b31379b1fead7c5028a7bae4d9612b24a10c29) )
	sc_ttomb_others
ROM_END

ROM_START( sc4ttombb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1305p12.hi", 0x00000, 0x080000, CRC(d1103ab7) SHA1(7fa4ed1f76e42e188103e13b8b188a304de3d9e4) )
	ROM_LOAD16_BYTE( "p1305p12.lo", 0x00001, 0x080000, CRC(8cb4a6a4) SHA1(bd1e4363024670914a82c65eaa3b37dafddf627a) )
	sc_ttomb_others
ROM_END

ROM_START( sc4ttombc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1305s12.hi", 0x00000, 0x080000, CRC(91276b7c) SHA1(52da930c77a07e234657838bcd35725ba471f747) )
	ROM_LOAD16_BYTE( "p1305s12.lo", 0x00001, 0x080000, CRC(768e8914) SHA1(89bb914f22b08b1abd83d759b6f232d3aa1ff89b) )
	sc_ttomb_others
ROM_END


ROM_START( sc4tridn )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408364.lo", 0x00001, 0x080000, CRC(a3f03319) SHA1(3e48644d66e66b1a1cc4973992fdd87364d43e5a) )
	ROM_LOAD16_BYTE( "95408365.hi", 0x00000, 0x080000, CRC(04f5c0cc) SHA1(8e30ef352fb85e3f4186765c6e413e3985eedaf1) )
	sc_tridn_others
ROM_END

ROM_START( sc4tridna )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409364.lo", 0x00001, 0x080000, CRC(e2e70d26) SHA1(6736b70981c3c255f0ece8c8a704e525b924eeca) )
	ROM_LOAD16_BYTE( "95409365.hi", 0x00000, 0x080000, CRC(8e244a99) SHA1(1a400cb48ce3e26d9d9e76343a7d2c98003dc4b5) )
	sc_tridn_others
ROM_END


ROM_START( sc4tri7 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410994.lo", 0x00001, 0x080000, CRC(1026c0f6) SHA1(fddeb118835adc899da0197777c9923125944df8) )
	ROM_LOAD16_BYTE( "95410995.hi", 0x00000, 0x080000, CRC(49db3a2b) SHA1(503ab7a10ac04e416683bb931d52772bb652d8d1) )
	sc_tri7_others
ROM_END

ROM_START( sc4tri7a )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411994.lo", 0x00001, 0x080000, CRC(0bbb112e) SHA1(a01b89b5fe20b34aa2e76287c6b76c043eab8d9f) )
	ROM_LOAD16_BYTE( "95411995.hi", 0x00000, 0x080000, CRC(ee4d4370) SHA1(8265e738a53d24a17d670a45f4c5f42419b34c0b) )
	sc_tri7_others
ROM_END

ROM_START( sc4tri7b )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412064.lo", 0x00001, 0x080000, CRC(a842778f) SHA1(ad711c31b1e5e89f024476cb41abe75d4efe4141) )
	ROM_LOAD16_BYTE( "95412065.hi", 0x00000, 0x080000, CRC(024c3d0c) SHA1(c586cb58ff512dad2d4ae7c4378af05c2bb4db95) )
	sc_tri7_others
ROM_END

ROM_START( sc4tri7c )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412454.lo", 0x00001, 0x080000, CRC(ee646837) SHA1(59b18e3aecbc32d87eac5bb76402496c39a5e89d) )
	ROM_LOAD16_BYTE( "95412455.hi", 0x00000, 0x080000, CRC(03e09830) SHA1(ef76bc3b8031455378d8486e337b81da3030775f) )
	sc_tri7_others
ROM_END

ROM_START( sc4tri7d )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413064.lo", 0x00001, 0x080000, CRC(37190cc4) SHA1(59c4a685723d502ad636349e4e65b74c2b1475f7) )
	ROM_LOAD16_BYTE( "95413065.hi", 0x00000, 0x080000, CRC(6ef29239) SHA1(ee2997c1685fb0e38aaa534aae95c49717241883) )
	sc_tri7_others
ROM_END

ROM_START( sc4tri7e )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413454.lo", 0x00001, 0x080000, CRC(432af7c8) SHA1(bba8231e2da9a9e85035187e3b03fd019a9c8c6d) )
	ROM_LOAD16_BYTE( "95413455.hi", 0x00000, 0x080000, CRC(18a8fb22) SHA1(4a4558ce2dbee92754f61bc4e8db8964c23a1e53) )
	sc_tri7_others
ROM_END

ROM_START( sc4tri7f )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1328p12.hi", 0x00000, 0x080000, CRC(babba77d) SHA1(53aa064eb8aa2821e4033f1110523bcee0690f7b) )
	ROM_LOAD16_BYTE( "p1328p12.lo", 0x00001, 0x080000, CRC(0e9108f0) SHA1(981d6bbe2201c38195df17ae1de178c907efe901) )
	sc_tri7_others
ROM_END

ROM_START( sc4tri7g )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1328p14.hi", 0x00000, 0x080000, CRC(807b5637) SHA1(fb2bb00348f299d517e55d044a705f6581b7c8de) )
	ROM_LOAD16_BYTE( "p1328p14.lo", 0x00001, 0x080000, CRC(227cc82d) SHA1(18e512cb8a6a07915c17285f2c1c73da90031813) )
	sc_tri7_others
ROM_END

ROM_START( sc4tri7h )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1328p15.hi", 0x00000, 0x080000, CRC(6650d9f4) SHA1(ac157fb98d1d5f7a8724d4d072344bd7ae9a12ca) )
	ROM_LOAD16_BYTE( "p1328p15.lo", 0x00001, 0x080000, CRC(ac654552) SHA1(ed1854336d9910ba7681288b8b4d93db793c6bbb) )
	sc_tri7_others
ROM_END

ROM_START( sc4tri7i )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1328p42.hi", 0x00000, 0x080000, CRC(0e27e529) SHA1(ec1faa8b0807173ca1d2e8900fc1fd17cad03da3) )
	ROM_LOAD16_BYTE( "p1328p42.lo", 0x00001, 0x080000, CRC(e840f88f) SHA1(fa8626dae0cb9a806a01264a4d8503ac71765901) )
	sc_tri7_others
ROM_END

ROM_START( sc4tri7j )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1328p43.hi", 0x00000, 0x080000, CRC(1011729f) SHA1(5e259fc4d332f2bac890dde7dfb64e504b3cb2c2) )
	ROM_LOAD16_BYTE( "p1328p43.lo", 0x00001, 0x080000, CRC(3e0ab5e0) SHA1(fb587e5833bf854ad884ccbe50d9f57f24cef4b0) )
	sc_tri7_others
ROM_END

ROM_START( sc4tri7k )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1328p44.hi", 0x00000, 0x080000, CRC(554e574e) SHA1(b211bfea65c4f993ab9d5bd9d97c3b910a24a806) )
	ROM_LOAD16_BYTE( "p1328p44.lo", 0x00001, 0x080000, CRC(6a788656) SHA1(10e9dff7c56386334b7ceac712ca3ad9d2df0793) )
	sc_tri7_others
ROM_END

ROM_START( sc4tri7l )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1328p45.hi", 0x00000, 0x080000, CRC(0bab79ba) SHA1(03e246aa584797b6613c023c69ad44423c03606d) )
	ROM_LOAD16_BYTE( "p1328p45.lo", 0x00001, 0x080000, CRC(45640e95) SHA1(8b12e67d81767171ff5408d707f04a4b1e21e3b7) )
	sc_tri7_others
ROM_END

ROM_START( sc4tri7m )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1328p93.hi", 0x00000, 0x080000, CRC(f3f210cf) SHA1(c266a39161134fcb16d44bc5417f7e72c382233f) )
	ROM_LOAD16_BYTE( "p1328p93.lo", 0x00001, 0x080000, CRC(eb57adbe) SHA1(33e1031150b7609779c9bc3365700df0e3b7c2f3) )
	sc_tri7_others
ROM_END

ROM_START( sc4tri7n )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1328p94.hi", 0x00000, 0x080000, CRC(e838472e) SHA1(de145ba9a02b1bafc3ee2c7aec93d25193b9cf41) )
	ROM_LOAD16_BYTE( "p1328p94.lo", 0x00001, 0x080000, CRC(a441fefe) SHA1(c861b3db7c16d4ff3b7bb828ccee0f7dd36700e3) )
	sc_tri7_others
ROM_END

ROM_START( sc4tri7o )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1328s12.hi", 0x00000, 0x080000, CRC(98d06e2e) SHA1(f89d74d35a0eb68c939a207c3fe8fb9f34e012b3) )
	ROM_LOAD16_BYTE( "p1328s12.lo", 0x00001, 0x080000, CRC(2942cd24) SHA1(1649efb28459dfbf01d70caaec6a8644c2b86cc5) )
	sc_tri7_others
ROM_END

ROM_START( sc4tri7p )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1328s14.hi", 0x00000, 0x080000, CRC(abd8c9de) SHA1(52526bb32c24f1d7c5183e1fff3ceb0459b6544d) )
	ROM_LOAD16_BYTE( "p1328s14.lo", 0x00001, 0x080000, CRC(46b39cd1) SHA1(4fb0e0c4c9aa59f0d1197393b8df9ac0c5f5f590) )
	sc_tri7_others
ROM_END

ROM_START( sc4tri7q )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1328s15.hi", 0x00000, 0x080000, CRC(455ec1c4) SHA1(b7b7161e19d5841501174fc3366efceec2077e2a) )
	ROM_LOAD16_BYTE( "p1328s15.lo", 0x00001, 0x080000, CRC(8babf31d) SHA1(a746111dacdc55cb9de37c0c7cb4e2572cae956a) )
	sc_tri7_others
ROM_END

ROM_START( sc4tri7r )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1328s42.hi", 0x00000, 0x080000, CRC(e6b01cb3) SHA1(88e5a680c4294f917265c60027b471447fc6941f) )
	ROM_LOAD16_BYTE( "p1328s42.lo", 0x00001, 0x080000, CRC(127a843c) SHA1(6e88ca16f63659c5c58ccbafca4d4f07f7d90b7a) )
	sc_tri7_others
ROM_END

ROM_START( sc4tri7s )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1328s43.hi", 0x00000, 0x080000, CRC(e338ba38) SHA1(4a063151f44fe322d4c8151188ea51b0854c8555) )
	ROM_LOAD16_BYTE( "p1328s43.lo", 0x00001, 0x080000, CRC(7b05833e) SHA1(8442622e810905e95bf8edb0852a45e149f47e2a) )
	sc_tri7_others
ROM_END

ROM_START( sc4tri7t )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1328s44.hi", 0x00000, 0x080000, CRC(702ef9f8) SHA1(a6ccad92fd23af04b73166204fb9f43280002219) )
	ROM_LOAD16_BYTE( "p1328s44.lo", 0x00001, 0x080000, CRC(4ab4597e) SHA1(2a24a4d6eebf15a1b5728d18c36a960cfcb67e76) )
	sc_tri7_others
ROM_END

ROM_START( sc4tri7u )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1328s45.hi", 0x00000, 0x080000, CRC(c66fa7e6) SHA1(71c3c152e0a257401d87dcb9ba1d9ccbc18d367f) )
	ROM_LOAD16_BYTE( "p1328s45.lo", 0x00001, 0x080000, CRC(7a32fb1a) SHA1(c98bcc39740fae3c74cb0edd931ea0d3d86d1fef) )
	sc_tri7_others
ROM_END

ROM_START( sc4tri7v )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1328s93.hi", 0x00000, 0x080000, CRC(c8f46a5c) SHA1(6d21b7d2cd950aa006b6d74c1691dd03314025d5) )
	ROM_LOAD16_BYTE( "p1328s93.lo", 0x00001, 0x080000, CRC(15b328f5) SHA1(4e56ddfbaceab9e662abf6028421e517ff8ab324) )
	sc_tri7_others
ROM_END

ROM_START( sc4tri7w )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1328s94.hi", 0x00000, 0x080000, CRC(543b5016) SHA1(2efdbcdd1db27c2cb0b10e477fbfc55eec210d15) )
	ROM_LOAD16_BYTE( "p1328s94.lo", 0x00001, 0x080000, CRC(0127367f) SHA1(e63d656a1d76ea4d61d5cd432bef6adf58c86f30) )
	sc_tri7_others
ROM_END


ROM_START( sc4trist )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414872.lo", 0x00001, 0x080000, CRC(361abb05) SHA1(a62c0aafee9eac3ec9b42c4b74bb63a6520cb77c) )
	ROM_LOAD16_BYTE( "95414873.hi", 0x00000, 0x080000, CRC(64d2487e) SHA1(2f135190ca532ad8ba3d19dcb7c06833c39ffd83) )
	sc_trist_others
ROM_END

ROM_START( sc4trista )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414874.lo", 0x00001, 0x080000, CRC(d6290c57) SHA1(41daa18354305fe4a530508adb9a21e68b66fdeb) )
	ROM_LOAD16_BYTE( "95414875.hi", 0x00000, 0x080000, CRC(61a96e02) SHA1(ef2fbf2ac53c119ef7de5cf97dd00824b9425032) )
	sc_trist_others
ROM_END

ROM_START( sc4tristb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416232.lo", 0x00001, 0x080000, CRC(0b10d683) SHA1(a2bb6c65c13eaa26cc546135ed668a809b5b7446) )
	ROM_LOAD16_BYTE( "95416233.hi", 0x00000, 0x080000, CRC(aceb2b79) SHA1(5301fedb2f0a84d20db1e0bb655ea3194e2aaa8f) )
	sc_trist_others
ROM_END

ROM_START( sc4tristc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416238.lo", 0x00001, 0x080000, CRC(0440c2f7) SHA1(b44f5ac81d6a140421603547a4e851e0db580d0e) )
	ROM_LOAD16_BYTE( "95416239.hi", 0x00000, 0x080000, CRC(0429ad5e) SHA1(59978dfdc62942f95f92fddd76887b6cc6f80073) )
	sc_trist_others
ROM_END

ROM_START( sc4tristd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416418.lo", 0x00001, 0x080000, CRC(e4989a37) SHA1(07c545f2b38dbd7ba54ba0436d3c31e52e51a38b) )
	ROM_LOAD16_BYTE( "95416419.hi", 0x00000, 0x080000, CRC(aec5d5f8) SHA1(64a20d24b17b113e9f686f570720f445faf6797b) )
	sc_trist_others
ROM_END

ROM_START( sc4triste )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416574.lo", 0x00001, 0x080000, CRC(837e1854) SHA1(c6c6b797a56a94619db83b611d45e96dd7523efa) )
	ROM_LOAD16_BYTE( "95416575.hi", 0x00000, 0x080000, CRC(64238f62) SHA1(d8e3a4d4a906782d9aa56705f1d53f15de5bff5c) )
	sc_trist_others
ROM_END

ROM_START( sc4tristf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416626.lo", 0x00001, 0x080000, CRC(f5aebfd8) SHA1(82f30aeefd8a469989ffbbebf949f2f73395eaf5) )
	ROM_LOAD16_BYTE( "95416627.hi", 0x00000, 0x080000, CRC(3a66d685) SHA1(1a324454ee6f5725430fa8fa908606f5a277cef5) )
	sc_trist_others
ROM_END

ROM_START( sc4tristg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416632.lo", 0x00001, 0x080000, CRC(58307537) SHA1(1aa783734ea833a5bd9e167b5c332e7bc41ee453) )
	ROM_LOAD16_BYTE( "95416633.hi", 0x00000, 0x080000, CRC(31336f19) SHA1(1c21ea319eae54055bddcebcfdfa7f2216090595) )
	sc_trist_others
ROM_END

ROM_START( sc4tristh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416634.lo", 0x00001, 0x080000, CRC(8bbfeee2) SHA1(a32fa36b13a293dce89da7a7ec87a3d53018472d) )
	ROM_LOAD16_BYTE( "95416635.hi", 0x00000, 0x080000, CRC(df28e46c) SHA1(64241771fa1a96a9c646a6fab3ce2494515d1fcc) )
	sc_trist_others
ROM_END

ROM_START( sc4tristi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416636.lo", 0x00001, 0x080000, CRC(ada2434b) SHA1(d22223faedea8686c21f1d341a70278147f4facc) )
	ROM_LOAD16_BYTE( "95416637.hi", 0x00000, 0x080000, CRC(e8c0e8a7) SHA1(7ee848b3a08af66824837466c54a0f3ab6ad6dc3) )
	sc_trist_others
ROM_END

ROM_START( sc4tristj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416642.lo", 0x00001, 0x080000, CRC(70a56cc6) SHA1(9c29307081c02c1b05c832396c5bfa6fdbde8572) )
	ROM_LOAD16_BYTE( "95416643.hi", 0x00000, 0x080000, CRC(24a76a32) SHA1(3669ae6c38781ba26516bc6106970fe8811f9ac7) )
	sc_trist_others
ROM_END

ROM_START( sc4tristk )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417232.lo", 0x00001, 0x080000, CRC(7bb4de17) SHA1(c798d7033026e88ba0559537035fb05f7d58f30c) )
	ROM_LOAD16_BYTE( "95417233.hi", 0x00000, 0x080000, CRC(6a5915ab) SHA1(bdf913cf54c682ce2e38d5fc89b54b0b58342d23) )
	sc_trist_others
ROM_END

ROM_START( sc4tristl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417234.lo", 0x00001, 0x080000, CRC(ac1d42e8) SHA1(1860de568b5b09403f13d6b2ba3769b19acb3e7f) )
	ROM_LOAD16_BYTE( "95417235.hi", 0x00000, 0x080000, CRC(ed94dbab) SHA1(ff8ed44d277add4447befd27818cf4a78e25badf) )
	sc_trist_others
ROM_END

ROM_START( sc4tristm )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417236.lo", 0x00001, 0x080000, CRC(e29f1c8d) SHA1(6254be7ed555e2331a0b5ac0f2d8319145b01a63) )
	ROM_LOAD16_BYTE( "95417237.hi", 0x00000, 0x080000, CRC(294cab6f) SHA1(bfbf01d75504f8c4e829aa887db2812de2dd9fed) )
	sc_trist_others
ROM_END

ROM_START( sc4tristn )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417238.lo", 0x00001, 0x080000, CRC(9eb0fdf7) SHA1(b66a0ad8f0083db2926309ddb9a5cf463b03e763) )
	ROM_LOAD16_BYTE( "95417239.hi", 0x00000, 0x080000, CRC(2b8c5088) SHA1(517291c99827f324f50140dbc4f706dd9d069ac5) )
	sc_trist_others
ROM_END

ROM_START( sc4tristo )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417418.lo", 0x00001, 0x080000, CRC(23d8bb79) SHA1(8e838e2744cbe1129cdb5b0195a2e55355297ed2) )
	ROM_LOAD16_BYTE( "95417419.hi", 0x00000, 0x080000, CRC(3359f77e) SHA1(7a42a53423824b8264a4c619103721cb0b1c0be4) )
	sc_trist_others
ROM_END

ROM_START( sc4tristp )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417574.lo", 0x00001, 0x080000, CRC(0840f093) SHA1(74611230b1c9d8588e492dad38fec9775c49bf33) )
	ROM_LOAD16_BYTE( "95417575.hi", 0x00000, 0x080000, CRC(176608f3) SHA1(db64046341f559c46a705cbef26679ccd080ff5d) )
	sc_trist_others
ROM_END

ROM_START( sc4tristq )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417626.lo", 0x00001, 0x080000, CRC(ddf809b0) SHA1(3aa5c116111f5cb677460a93130f8472cfe5fec8) )
	ROM_LOAD16_BYTE( "95417627.hi", 0x00000, 0x080000, CRC(0e141e79) SHA1(5723a75ae4758b27bff125f646231590bbb7f404) )
	sc_trist_others
ROM_END

ROM_START( sc4tristr )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417628.lo", 0x00001, 0x080000, CRC(1fcc127d) SHA1(cad3ff8055d48cae67394ec4fb3d373e7c24a8ff) )
	ROM_LOAD16_BYTE( "95417629.hi", 0x00000, 0x080000, CRC(ec102d98) SHA1(12cf171a50d639f965955ce0c2a6d5dcbaecefcf) )
	sc_trist_others
ROM_END

ROM_START( sc4trists )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417630.lo", 0x00001, 0x080000, CRC(fa2ad681) SHA1(45312fb02e2ad46d23ba6e62c5a9ecc8f736ffa5) )
	ROM_LOAD16_BYTE( "95417631.hi", 0x00000, 0x080000, CRC(3cc61b01) SHA1(82dcfea953d09f41aae9ae5c281b8fbbe2ef7ec7) )
	sc_trist_others
ROM_END

ROM_START( sc4tristt )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417632.lo", 0x00001, 0x080000, CRC(cd19235e) SHA1(57848f8405799230ab74679f2704a8d0a674fb29) )
	ROM_LOAD16_BYTE( "95417633.hi", 0x00000, 0x080000, CRC(e38f0d88) SHA1(906fb3957f6c320771588cec22cdfcfe290804b9) )
	sc_trist_others
ROM_END

ROM_START( sc4tristu )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417634.lo", 0x00001, 0x080000, CRC(9da385ab) SHA1(34f9e1bb10df6b5c6a12be01d1250447a6b26bb8) )
	ROM_LOAD16_BYTE( "95417635.hi", 0x00000, 0x080000, CRC(d1cf2175) SHA1(1dd72e2c0a9c19ba00df66976b2b2d8d96b72242) )
	sc_trist_others
ROM_END

ROM_START( sc4tristv )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417636.lo", 0x00001, 0x080000, CRC(a2b799ce) SHA1(b9aaf08f4eadf5c1174454b288502c3c4295a5e3) )
	ROM_LOAD16_BYTE( "95417637.hi", 0x00000, 0x080000, CRC(a9076da9) SHA1(84324bdb91ed32e35e16e0a7853b2c53281d138a) )
	sc_trist_others
ROM_END

ROM_START( sc4tristw )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417638.lo", 0x00001, 0x080000, CRC(8c45eb89) SHA1(4a0db371245a9c922958f6d001b4fe370f831ade) )
	ROM_LOAD16_BYTE( "95417639.hi", 0x00000, 0x080000, CRC(4dbad597) SHA1(7f2ef44e34864551aa69075160cdf04578a9f337) )
	sc_trist_others
ROM_END

ROM_START( sc4tristx )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417640.lo", 0x00001, 0x080000, CRC(883b66f2) SHA1(1f9ad6b0213a87a654fe68766d3ee8ea1a8bc593) )
	ROM_LOAD16_BYTE( "95417641.hi", 0x00000, 0x080000, CRC(d06acd1c) SHA1(b4191a97cec37cd4ce687f690f3a9b577b06236f) )
	sc_trist_others
ROM_END

ROM_START( sc4tristy )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417642.lo", 0x00001, 0x080000, CRC(3090f654) SHA1(b03e59fe9d58e2d2f0116e7d2bfb5391b4d4e0d7) )
	ROM_LOAD16_BYTE( "95417643.hi", 0x00000, 0x080000, CRC(81d31db1) SHA1(23379d169434b31675c268d641713a5994fc4a67) )
	sc_trist_others
ROM_END


ROM_START( sc4tub )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404460.lo", 0x00001, 0x080000, CRC(2e876763) SHA1(194ab445923c4377be9725e98ba77d11653df1cd) )
	ROM_LOAD16_BYTE( "95404461.hi", 0x00000, 0x080000, CRC(30a4ebc8) SHA1(7cbaf6b78066954291e71596a8490585c64043e4) )
	sc_tub_others
ROM_END

ROM_START( sc4tuba )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405460.lo", 0x00001, 0x080000, CRC(b7746ce2) SHA1(0ebf630ece5273f7683e1fbe164349d130ea2fb0) )
	ROM_LOAD16_BYTE( "95405461.hi", 0x00000, 0x080000, CRC(afdac796) SHA1(b7d4197e9dedd08ddbaa971cbe71820c0401b460) )
	sc_tub_others
ROM_END

ROM_START( sc4tubb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406032.lo", 0x00001, 0x080000, CRC(11f97160) SHA1(cafd18038d3345702272eabb1ee2bd3d90bf857c) )
	ROM_LOAD16_BYTE( "95406033.hi", 0x00000, 0x080000, CRC(9b4a5f6d) SHA1(21051d01cd937edd9906730c88461b4c5a6d08d0) )
	sc_tub_others
ROM_END

ROM_START( sc4tubc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407032.lo", 0x00001, 0x080000, CRC(0d9920ba) SHA1(2eb8f661bb11cf28a94dfdbbacf07e2a8a41058b) )
	ROM_LOAD16_BYTE( "95407033.hi", 0x00000, 0x080000, CRC(52be5eee) SHA1(1bed67c56da01c4c8048645eb97109dc5fd453eb) )
	sc_tub_others
ROM_END


ROM_START( sc4tfclb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402177.lo", 0x00001, 0x080000, CRC(f74c912a) SHA1(a1709bdf360ed97df200d6e32e16c1196575b9eb) )
	ROM_LOAD16_BYTE( "95402178.hi", 0x00000, 0x080000, CRC(7fa1a196) SHA1(3da86c82ed6115442b7960e5c9c75fa8b62c1c31) )
	sc_tfclb_others
ROM_END

ROM_START( sc4tfclba )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403177.lo", 0x00001, 0x080000, CRC(6b11a6b8) SHA1(ff436b2686bc11e141d49ae2348a47fc7861c5d3) )
	ROM_LOAD16_BYTE( "95403178.hi", 0x00000, 0x080000, CRC(db742c5d) SHA1(e25129d0bcb8542ebb55d2801892f3f517425732) )
	sc_tfclb_others
ROM_END


ROM_START( sc4twilt )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406534.lo", 0x00001, 0x080000, CRC(05db2b5a) SHA1(c5ff8ee24b57f6af08ea74d04034d9ba7e50b5da) )
	ROM_LOAD16_BYTE( "95406535.hi", 0x00000, 0x080000, CRC(c81afcb2) SHA1(25c79f2de750b30f6268260291d773dab8415bba) )
	sc_twilt_others
ROM_END


ROM_START( sc4ufg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400633.lo", 0x00001, 0x080000, CRC(0dd9d476) SHA1(f60b3dff5b26b5ad46d4a96a5d397de265bfde98) )
	ROM_LOAD16_BYTE( "95400634.hi", 0x00000, 0x080000, CRC(7c3678a4) SHA1(508472ec4fa33bf691183909a224607d569ec119) )
	sc_ufg_others
ROM_END

ROM_START( sc4ufga )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401633.lo", 0x00001, 0x080000, CRC(06ed97e6) SHA1(f1acb597187c90e888a8c9810856c9b8f817473c) )
	ROM_LOAD16_BYTE( "95401634.hi", 0x00000, 0x080000, CRC(5ea8e2c3) SHA1(ff4a488a616c10694c4041a85d0dc2729c9da3a9) )
	sc_ufg_others
ROM_END


ROM_START( sc4ufi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400723.lo", 0x00001, 0x080000, CRC(5a08e5e2) SHA1(4cba2a450be3ac78294a94a8964b03d243dd73f0) )
	ROM_LOAD16_BYTE( "95400724.hi", 0x00000, 0x080000, CRC(a47a8c15) SHA1(0996c613d49bea0fdb156a54c9704852f79735e2) )
	sc_ufi_others
ROM_END

ROM_START( sc4ufia )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400725.lo", 0x00001, 0x080000, CRC(abd57573) SHA1(8851454c8d253d862f81af44c7541a110e331873) )
	ROM_LOAD16_BYTE( "95400726.hi", 0x00000, 0x080000, CRC(26b1a723) SHA1(a64129046c2db33c7df96165b2591c056f5fbb0d) )
	sc_ufi_others
ROM_END

ROM_START( sc4ufib )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401723.lo", 0x00001, 0x080000, CRC(38d4e8c8) SHA1(d238bda3b5bc407933575230a1640bd1cc789491) )
	ROM_LOAD16_BYTE( "95401724.hi", 0x00000, 0x080000, CRC(92ea4a31) SHA1(a9b42e039f89ae7ecfcc0b87b9715772e1a06696) )
	sc_ufi_others
ROM_END

ROM_START( sc4ufic )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401725.lo", 0x00001, 0x080000, CRC(09ad97c6) SHA1(9c9628156d2220ad6b1bab490d48fde1ecd6c2c3) )
	ROM_LOAD16_BYTE( "95401726.hi", 0x00000, 0x080000, CRC(2ab46ce2) SHA1(60f74cc855c35a3dc07a32640c14835c1cfe1827) )
	sc_ufi_others
ROM_END

ROM_START( sc4ufid )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401911.lo", 0x00001, 0x080000, CRC(ae68b469) SHA1(de77f117a3d688f0feb9415f00b534d3651f52c7) )
	ROM_LOAD16_BYTE( "95401912.hi", 0x00000, 0x080000, CRC(87b73871) SHA1(598764a0a73705475f15263a79106f22b6d1abe8) )
	sc_ufi_others
ROM_END

ROM_START( sc4ufie )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400911.lo", 0x00000, 0x080000, NO_DUMP )
	ROM_LOAD16_BYTE( "95400912.hi", 0x00000, 0x080000, CRC(d1a24394) SHA1(03d6561f8e66bbaaabd18fb9d5244f12b2202c0d) )
	sc_ufi_others
ROM_END



ROM_START( sc4valnv )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402005.lo", 0x00001, 0x080000, CRC(0b42f5f2) SHA1(02982861f7f3bb1cd20718df539a909dabe929af) )
	ROM_LOAD16_BYTE( "95402006.hi", 0x00000, 0x080000, CRC(5886ecdd) SHA1(4cec3d51871a5956bdeaaa1a4bc37236812d5e16) )
	sc_valnv_others
ROM_END


ROM_START( sc4valqp )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406480.lo", 0x00001, 0x080000, CRC(c0860d85) SHA1(2625c3ba9fadc1c537baef6a619dff50e12e0a13) )
	ROM_LOAD16_BYTE( "95406481.hi", 0x00000, 0x080000, CRC(ddb3439f) SHA1(70ef8e7bdf3234f8e83e47beca54a4d8712257c4) )
	sc_valqp_others
ROM_END


ROM_START( sc4vrgcl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404018.lo", 0x00001, 0x080000, CRC(21ee4f08) SHA1(d2a89dcab0810eb1cccd6176dacbb2bfc96d7450) )
	ROM_LOAD16_BYTE( "95404019.hi", 0x00000, 0x080000, CRC(434fef1c) SHA1(3653a81213599272db63282573b1c639e630c6c2) )
	sc_vrgcl_others
ROM_END

ROM_START( sc4vrgcla )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404076.lo", 0x00001, 0x080000, CRC(65551002) SHA1(fe41d92b0ca3137d1731d6d2cbcc94d4a9da6088) )
	ROM_LOAD16_BYTE( "95404077.hi", 0x00000, 0x080000, CRC(04550388) SHA1(7613591cabdf72628ad5a6915137cba43c427984) )
	sc_vrgcl_others
ROM_END

ROM_START( sc4vrgclb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404548.lo", 0x00001, 0x080000, CRC(e8dacffe) SHA1(cc9034542d684bc300ffdf7a38e047f1ce0ca739) )
	ROM_LOAD16_BYTE( "95404549.hi", 0x00000, 0x080000, CRC(fa5075f9) SHA1(9f87a52c8f9ed64724be87e88fff833253d3e22a) )
	sc_vrgcl_others
ROM_END

ROM_START( sc4vrgclc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404550.lo", 0x00001, 0x080000, CRC(b2ce484f) SHA1(88db91f7bce08ee6e84352286bc33df5d4c9a6d0) )
	ROM_LOAD16_BYTE( "95404551.hi", 0x00000, 0x080000, CRC(66ab7174) SHA1(ec4ae81c1f4e4a9a48c17f8aef798dff216aefc4) )
	sc_vrgcl_others
ROM_END

ROM_START( sc4vrgcld )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404634.lo", 0x00001, 0x080000, CRC(9e2ed4a0) SHA1(44c1707f9489e755f9035c039b7b976b1734630d) )
	ROM_LOAD16_BYTE( "95404635.hi", 0x00000, 0x080000, CRC(8249f713) SHA1(1f361f849985ae70c841b2b40658d40b1aa6b399) )
	sc_vrgcl_others
ROM_END

ROM_START( sc4vrgcle )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405548.lo", 0x00001, 0x080000, CRC(bba0c2bc) SHA1(c3a99423fc62920c5c1ca6fa4f0a3283ca6490f0) )
	ROM_LOAD16_BYTE( "95405549.hi", 0x00000, 0x080000, CRC(b113fe1d) SHA1(7883c34c403c532209137105a15f97f18311bf8a) )
	sc_vrgcl_others
ROM_END

ROM_START( sc4vrgclf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405550.lo", 0x00001, 0x080000, CRC(2c882c37) SHA1(c35645c58d11971cc9d8717a7b27c1082259f07b) )
	ROM_LOAD16_BYTE( "95405551.hi", 0x00000, 0x080000, CRC(ac52e4c3) SHA1(2f93b3f262861e5ba54f70824fcc6961e66f935d) )
	sc_vrgcl_others
ROM_END

ROM_START( sc4vrgclg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405634.lo", 0x00001, 0x080000, CRC(056413e9) SHA1(555dec6eea70135b6a12c0c3c0421618c321fb94) )
	ROM_LOAD16_BYTE( "95405635.hi", 0x00000, 0x080000, CRC(0046f382) SHA1(8d3cb10b44edba2cd677eab2618a610b0f9fa35c) )
	sc_vrgcl_others
ROM_END

ROM_START( sc4vrgclh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412208.lo", 0x00001, 0x080000, CRC(3edfe2d8) SHA1(c4f12de7ed7db8fc317418a8adeea1a2f15ca7ef) )
	ROM_LOAD16_BYTE( "95412209.hi", 0x00000, 0x080000, CRC(509088c4) SHA1(c6b70ecd33599adb379586bbb2cbe3b58fce857b) )
	sc_vrgcl_others
ROM_END

ROM_START( sc4vrgcli )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413208.lo", 0x00001, 0x080000, CRC(4d34246d) SHA1(4bcfae186064f3d1619de336c74e8d4d5d4ef05f) )
	ROM_LOAD16_BYTE( "95413209.hi", 0x00000, 0x080000, CRC(e04b594c) SHA1(d1c5d00e3fc2a4fc6d612ca0d51b83851129398d) )
	sc_vrgcl_others
ROM_END





ROM_START( sc4vmnv )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404300.lo", 0x00001, 0x080000, CRC(97526a2a) SHA1(0e650cd1e33bab5c2a7db6057053b8b15c2efa02) )
	ROM_LOAD16_BYTE( "95404301.hi", 0x00000, 0x080000, CRC(89a6384c) SHA1(022eddf0e93f86f10c78252f24502a319097920c) )
	sc_vmnv_others
ROM_END

ROM_START( sc4vmnva )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p6984s16.hi", 0x00000, 0x080000, CRC(098c104b) SHA1(f9d28ced6d35e95bad7d406fe932dd887a3ed42e) )
	ROM_LOAD16_BYTE( "p6984s16.lo", 0x00001, 0x080000, CRC(957bdbb3) SHA1(2093a331c91d68b0aea66e5dd0b304b958a4b168) )
	sc_vmnv_others
ROM_END


ROM_START( sc4vmclb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404686.lo", 0x00001, 0x080000, CRC(f3d7d462) SHA1(3de86eab8cab50924d5186b57863260dfd555bd7) )
	ROM_LOAD16_BYTE( "95404687.hi", 0x00000, 0x080000, CRC(07d6f891) SHA1(1b5ce61733b3c2adbda439dbca857693d34e3022) )
	PR7132_CLUB_VIVA_SOUNDS11
ROM_END

ROM_START( sc4vmclba )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "viva_mex.hi", 0x00000, 0x080000, CRC(521cea58) SHA1(a2a193c23843195e6f7ef666f3785869e11c9baa) )
	ROM_LOAD16_BYTE( "viva_mex.lo", 0x00001, 0x080000, CRC(ca5260f1) SHA1(1f2901e76db3c2cd615cf1efafc7e965c1ac48f1) )
	PR7132_CLUB_VIVA_SOUNDS11
ROM_END

ROM_START( sc4vmclbb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405686.lo", 0x00001, 0x080000, CRC(5852e489) SHA1(987918f92b2cd53e09696ccf795f0e50ee44cde4) )
	ROM_LOAD16_BYTE( "95405687.hi", 0x00000, 0x080000, CRC(c73c6888) SHA1(6be3c7285ec975a4798c9ff9d59da3466e3b545a) )
	PR7132_CLUB_VIVA_SOUNDS11
ROM_END


ROM_START( sc4wadzl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406160.lo", 0x00001, 0x080000, CRC(c4f62558) SHA1(ba8180828f42cecb94ec58b7ad36575c70d4bce3) )
	ROM_LOAD16_BYTE( "95406161.hi", 0x00000, 0x080000, CRC(6cb9e360) SHA1(a902bd3af145d92fe99fa51b81188f5674d9e6e1) )
	sc_wadzl_others
ROM_END

ROM_START( sc4wadzla )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407160.lo", 0x00001, 0x080000, CRC(9fba6410) SHA1(41d5af5560a0de4e05975d35d5610e79ac985782) )
	ROM_LOAD16_BYTE( "95407161.hi", 0x00000, 0x080000, CRC(00a6c358) SHA1(85ab782804600272daf8de4b591f788dd0777893) )
	sc_wadzl_others
ROM_END


ROM_START( sc4wernr )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402169.lo", 0x00001, 0x080000, CRC(0081b03f) SHA1(07f682278187ccb5cc2adc7be3e457a1966f9fa5) )
	ROM_LOAD16_BYTE( "95402170.hi", 0x00000, 0x080000, CRC(8deb5244) SHA1(e02c051bab6deb009a3b9e4716af29f16e66abba) )
	sc_wernr_others
ROM_END


ROM_START( sc4wrnlt )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402866.lo", 0x00001, 0x080000, CRC(9727bc1e) SHA1(166165dfe43a981d25ea32651b3527113942dc8f) )
	ROM_LOAD16_BYTE( "95402867.hi", 0x00000, 0x080000, CRC(c4a654de) SHA1(f41c867010ac9451ad96bb65b6349a6ab30c4a96) )
	sc_wernr_others
ROM_END


ROM_START( sc4waw )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402846.lo", 0x00001, 0x080000, CRC(fd7afc01) SHA1(62786a0cc5908451c672aece504590cb5596ddd9) )
	ROM_LOAD16_BYTE( "95402847.hi", 0x00000, 0x080000, CRC(0b12e6ee) SHA1(2f06c548b30d635c383d5408f31c264a7a6a7455) )
	sc_waw_others
ROM_END

ROM_START( sc4wawa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403369.lo", 0x00001, 0x080000, CRC(cd9f20c1) SHA1(7fb3c249c3d65a5288d870bc0eb1c331cf08f4e2) )
	ROM_LOAD16_BYTE( "95403370.hi", 0x00000, 0x080000, CRC(e5251c71) SHA1(2f7af921665a2cb458886e3bbc8e1133c5d6a90b) )
	sc_waw_others
ROM_END

ROM_START( sc4wawb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403456.lo", 0x00001, 0x080000, CRC(30765ea8) SHA1(9079118006dee0c49b154ae5462bd10afd129039) )
	ROM_LOAD16_BYTE( "95403457.hi", 0x00000, 0x080000, CRC(8d6f2111) SHA1(ac670aab260ac8d94615e5290e4650b47bf70a67) )
	sc_waw_others
ROM_END

ROM_START( sc4wawc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403846.lo", 0x00001, 0x080000, CRC(fa064a7c) SHA1(0820b51a84be3c102b2809f804b8c1aff7b79c9f) )
	ROM_LOAD16_BYTE( "95403847.hi", 0x00000, 0x080000, CRC(1319716c) SHA1(8960de17a15682a8ccea199207c5b988d0f358ab) )
	sc_waw_others
ROM_END

ROM_START( sc4wawd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404462.lo", 0x00001, 0x080000, CRC(2bbd8487) SHA1(1b2bc57c05b0eae7d7251f53422ada65bc2760ff) )
	ROM_LOAD16_BYTE( "95404463.hi", 0x00000, 0x080000, CRC(6cbc38db) SHA1(d9d7ea5abe4870ff0771bae28723ec7782bf96dc) )
	sc_waw_others
ROM_END

ROM_START( sc4wawe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405462.lo", 0x00001, 0x080000, CRC(82c07e2c) SHA1(94a4d263767435f38d65da8d6231bb28322a4ade) )
	ROM_LOAD16_BYTE( "95405463.hi", 0x00000, 0x080000, CRC(e92b8ddf) SHA1(693da6e5cbb30353d26b419d85d814f83616ae80) )
	sc_waw_others
ROM_END

ROM_START( sc4wawf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "wet__pound15.lo", 0x00001, 0x080000, CRC(3d2bcf6e) SHA1(12bbf1127b4e2046b58d47d01ca593c39d511ab1) )
	ROM_LOAD16_BYTE( "wet__pound15.hi", 0x00000, 0x080000, CRC(ea2e9311) SHA1(febce0cc05f25f11cb911d5583e18271038645c6) )
	sc_waw_others
ROM_END


ROM_START( sc4wdw )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400321.lo", 0x00001, 0x080000, CRC(f9040cc9) SHA1(cf74e5beb74cd94667a2ee9db7a5b53ac8512533) )
	ROM_LOAD16_BYTE( "95400322.hi", 0x00000, 0x080000, CRC(f234fb3c) SHA1(f87875b8c9c3879673e63f7f642f9ee71fe614c9) )
	sc_wdw_others
ROM_END

ROM_START( sc4wdwa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400323.lo", 0x00001, 0x080000, CRC(20d7d0e1) SHA1(bf159a55e4a71f3483ac7da52d9f67db850efaf1) )
	ROM_LOAD16_BYTE( "95400324.hi", 0x00000, 0x080000, CRC(722ad66e) SHA1(43578625d3fe62405468fffeb2555b5cc77fb928) )
	sc_wdw_others
ROM_END

ROM_START( sc4wdwb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400379.lo", 0x00001, 0x080000, CRC(8c9a5ef2) SHA1(310f92c930ce6d1d1cdc6573c8fdc9b32e5f46d8) )
	ROM_LOAD16_BYTE( "95400380.hi", 0x00000, 0x080000, CRC(ead1c62b) SHA1(39e71139810cd2aa7e41a33b237c3ef4154f29c6) )
	sc_wdw_others
ROM_END

ROM_START( sc4wdwc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400759.lo", 0x00001, 0x080000, CRC(f0f38034) SHA1(2c38870ce9051807a519cc54f33f0224546c2abc) )
	ROM_LOAD16_BYTE( "95400760.hi", 0x00000, 0x080000, CRC(70f91f2e) SHA1(d04f630242078b8668d701c221724c596fe3b92e) )
	sc_wdw_others
ROM_END

ROM_START( sc4wdwd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95400761.lo", 0x00001, 0x080000, CRC(6401a1bc) SHA1(bce8992b6d3ff2e322b1a52f1813496e146a1861) )
	ROM_LOAD16_BYTE( "95400762.hi", 0x00000, 0x080000, CRC(ecf12b04) SHA1(483feede4eb4f2e92b3db1e4d7241b278a4d0a67) )
	sc_wdw_others
ROM_END

ROM_START( sc4wdwe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401321.lo", 0x00001, 0x080000, CRC(06e84686) SHA1(102259607447b5563f915c7c2e75aea60c36fdb0) )
	ROM_LOAD16_BYTE( "95401322.hi", 0x00000, 0x080000, CRC(d10b9b13) SHA1(60a4c343e2681465241691e7cb31226e9b24798c) )
	sc_wdw_others
ROM_END

ROM_START( sc4wdwf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401323.lo", 0x00001, 0x080000, CRC(b1a740e7) SHA1(9e09ce3f58aa688dfba77cf66be81967373db2e4) )
	ROM_LOAD16_BYTE( "95401324.hi", 0x00000, 0x080000, CRC(f8fb5c3b) SHA1(64b84bd246d0d305e4eee94f48605a0c51d051ce) )
	sc_wdw_others
ROM_END

ROM_START( sc4wdwg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401379.lo", 0x00001, 0x080000, CRC(a3d1d95c) SHA1(abe3a03dd9ab50f69fd5a144a912eb2118a7b4b9) )
	ROM_LOAD16_BYTE( "95401380.hi", 0x00000, 0x080000, CRC(ab89151d) SHA1(da1ffed3f780917f062a4a4dc1e101ba6ac5998a) )
	sc_wdw_others
ROM_END

ROM_START( sc4wdwh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401759.lo", 0x00001, 0x080000, CRC(99e31f91) SHA1(83e80cc10d54a315df751bd1b8dd646be0c6a1e3) )
	ROM_LOAD16_BYTE( "95401760.hi", 0x00000, 0x080000, CRC(70664916) SHA1(50135cb63bc631d75e0cb22233b932e0b5dd4099) )
	sc_wdw_others
ROM_END

ROM_START( sc4wdwi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401761.lo", 0x00001, 0x080000, CRC(6be8bf00) SHA1(74f7fc5f9989c91958837f12e512a2ea0bc00ab9) )
	ROM_LOAD16_BYTE( "95401762.hi", 0x00000, 0x080000, CRC(a34e3d2b) SHA1(74b97b7a8d07b13d3375c45862faa0e9a751ec84) )
	sc_wdw_others
ROM_END

ROM_START( sc4wdwj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "wdw15hi", 0x0000, 0x080000, CRC(46dcc106) SHA1(647add94380523292436981ce280d523a6569f69) )
	ROM_LOAD16_BYTE( "wdw15lo", 0x00001, 0x080000, CRC(88cc9c2a) SHA1(270a1f17b74f9ac619dd88cbde1c6da6b469ed67) )
	sc_wdw_others
ROM_END


ROM_START( sc4leg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404332.lo", 0x00001, 0x080000, CRC(7ff4995e) SHA1(f74b891c8987687bb9aa359aac5b498383227729) )
	ROM_LOAD16_BYTE( "95404333.hi", 0x00000, 0x080000, CRC(2e0e489c) SHA1(2d3f79c89f7b5ad036637980b1607385c9376839) )
	sc_leg_others
ROM_END

ROM_START( sc4lega )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404334.lo", 0x00001, 0x080000, CRC(93a161e5) SHA1(429fc5d99f12f25a95de824f449a8c4ef939e507) )
	ROM_LOAD16_BYTE( "95404335.hi", 0x00000, 0x080000, CRC(8f1672a7) SHA1(088bf89cf8d189b18a85b3fbcc4ead2a9ca812c9) )
	sc_leg_others
ROM_END

ROM_START( sc4legb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405332.lo", 0x00001, 0x080000, CRC(2e74e812) SHA1(4b08dc2e12ef659da2408f9d28bb6f125adf80a4) )
	ROM_LOAD16_BYTE( "95405333.hi", 0x00000, 0x080000, CRC(99df54d8) SHA1(03d6a106e3a1ebdfce23699b9d642b6201954e70) )
	sc_leg_others
ROM_END

ROM_START( sc4legc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405334.lo", 0x00001, 0x080000, CRC(2ccd6c8a) SHA1(1b8443177beaf83be303a80ac452f39b470acb3a) )
	ROM_LOAD16_BYTE( "95405335.hi", 0x00000, 0x080000, CRC(3dbd4830) SHA1(1c01142fd43692e5ea614e9671d6b1fd9b6a3f50) )
	sc_leg_others
ROM_END

ROM_START( sc4legd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406056.lo", 0x00001, 0x080000, CRC(be74ec40) SHA1(334543a08448ce68b149bcbb756ef6c7b1c14d67) )
	ROM_LOAD16_BYTE( "95406057.hi", 0x00000, 0x080000, CRC(62a94976) SHA1(da9796643b7fa7ad0906ec2447402b28f8fce362) )
	sc_leg_others
ROM_END

ROM_START( sc4lege )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406060.lo", 0x00001, 0x080000, CRC(70b0814f) SHA1(dd63ad9633d46b3b703b79a4a1220ccac4b7959e) )
	ROM_LOAD16_BYTE( "95406061.hi", 0x00000, 0x080000, CRC(4660a459) SHA1(4928a6a6d51dc745dff4663eeda1cf6af8f9ffd4) )
	sc_leg_others
ROM_END

ROM_START( sc4legf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407056.lo", 0x00001, 0x080000, CRC(a7d254b6) SHA1(5e2c8f8d7c633ed0c8fac03a08133c4f54e217e7) )
	ROM_LOAD16_BYTE( "95407057.hi", 0x00000, 0x080000, CRC(f60c7c32) SHA1(4398b77ca86531bae4fd0b052602d837fe1d6f13) )
	sc_leg_others
ROM_END

ROM_START( sc4legg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407060.lo", 0x00001, 0x080000, CRC(b3be6239) SHA1(cbe275e8a13e0a4cbc2c00404971dc97fd7d1ec8) )
	ROM_LOAD16_BYTE( "95407061.hi", 0x00000, 0x080000, CRC(edf316c3) SHA1(8caba456c2133f760999c8ef8e782739432aa987) )
	sc_leg_others
ROM_END

ROM_START( sc4legh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "legionnaire_std_1.5_402862_lo.bin", 0x000001, 0x080000, CRC(598bf2ba) SHA1(f3b02be06458e01bbacfdf4e0fcc3800102a88cd) )
	ROM_LOAD16_BYTE( "legionnaire_std_1.5_402863_hi.bin", 0x000000, 0x080000, CRC(f9c5c16e) SHA1(08c2a937980c4d1183044f5b29680cab918fbdaa) )
	sc_leg_others
ROM_END

ROM_START( sc4legcb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404336.lo", 0x00001, 0x080000, CRC(622b31cd) SHA1(94ecdc0982baa92f5328446627d997f17a865630) )
	ROM_LOAD16_BYTE( "95404337.hi", 0x00000, 0x080000, CRC(229e8a0c) SHA1(f34d4063f2e1e098f6ee6f4ac015af0a423b402b) )
	sc_legcb_others
ROM_END

ROM_START( sc4legcba )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404380.lo", 0x00001, 0x080000, CRC(a66b2f18) SHA1(54bf8fb0613c20c16f9eacd247b83fbe201a4aa8) )
	ROM_LOAD16_BYTE( "95404381.hi", 0x00000, 0x080000, CRC(95f78581) SHA1(ea0c014102d60a54e481fa65564a351d94700d99) )
	sc_legcb_others
ROM_END

ROM_START( sc4legcbb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405336.lo", 0x00001, 0x080000, CRC(593a1dfd) SHA1(c7fe22e129ca4578c1e47ef78eded5ed5da0a787) )
	ROM_LOAD16_BYTE( "95405337.hi", 0x00000, 0x080000, CRC(7f85031d) SHA1(a8712c07147a406706ac5ad800768dc53450400c) )
	sc_legcb_others
ROM_END

ROM_START( sc4legcbc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405380.lo", 0x00001, 0x080000, CRC(eb0c6d92) SHA1(7eb660cdeeb442091c86977a564eb67a1bac167a) )
	ROM_LOAD16_BYTE( "95405381.hi", 0x00000, 0x080000, CRC(db763fde) SHA1(26caad76f16e0936a545a7481d978e19554d35fd) )
	sc_legcb_others
ROM_END

ROM_START( sc4legcbd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412206.lo", 0x00001, 0x080000, CRC(63b1e5a6) SHA1(d9ad4ac6eb57ec13e07ec359a9f214829fee30cd) )
	ROM_LOAD16_BYTE( "95412207.hi", 0x00000, 0x080000, CRC(2e684c7f) SHA1(90e43834bce1e2884428e2f1df3c1528f15c634b) )
	sc_legcb_others
ROM_END

ROM_START( sc4legcbe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413206.lo", 0x00001, 0x080000, CRC(76d29d4e) SHA1(1823fe130110b36fc715b88e35c590fab55618c7) )
	ROM_LOAD16_BYTE( "95413207.hi", 0x00000, 0x080000, CRC(b25594b1) SHA1(f3771534787f6bebd723552424a7968ec46261e4) )
	sc_legcb_others
ROM_END


ROM_START( sc4wldbn )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402876.lo", 0x00001, 0x080000, CRC(267e0d9a) SHA1(e7e615d3d0d036305a239d3d0070d6abeeeff8ed) )
	ROM_LOAD16_BYTE( "95402877.hi", 0x00000, 0x080000, CRC(cea65dca) SHA1(9b7b379683759673258b7f184004b5c6cf31c6d6) )
	sc_wldbn_others
ROM_END

ROM_START( sc4wldbna )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402878.lo", 0x00001, 0x080000, CRC(b7c33be2) SHA1(5a21cbfa053f271b042c1f52e1cff16f30800e51) )
	ROM_LOAD16_BYTE( "95402879.hi", 0x00000, 0x080000, CRC(89ffe589) SHA1(43979e3fe14c876e9778bcf0ccfa16901c632c99) )
	sc_wldbn_others
ROM_END

ROM_START( sc4wldbnb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403876.lo", 0x00001, 0x080000, CRC(ff229596) SHA1(b0e499d709cb19e2eb6efbed31d77672b85f6d2c) )
	ROM_LOAD16_BYTE( "95403877.hi", 0x00000, 0x080000, CRC(680e88e1) SHA1(8b66d1746e6f429774bf637481ee41b3ab016f3d) )
	sc_wldbn_others
ROM_END

ROM_START( sc4wldbnc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403878.lo", 0x00001, 0x080000, CRC(5b3233fb) SHA1(fdc242e12ee4792efb097aeb96cde9292af2d286) )
	ROM_LOAD16_BYTE( "95403879.hi", 0x00000, 0x080000, CRC(60e8268d) SHA1(5f458f40806bb81728a4bb16b532b2391fa52f3c) )
	sc_wldbn_others
ROM_END

ROM_START( sc4wldbnd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410060.lo", 0x00001, 0x080000, CRC(9d73db8c) SHA1(5dfeabdcaa403ad31dd868aa001a5168c904083e) )
	ROM_LOAD16_BYTE( "95410061.hi", 0x00000, 0x080000, CRC(83b5c62c) SHA1(f433e2ca8d80a4e4d55e8c38dada961887f7e56a) )
	sc_wldbn_others
ROM_END

ROM_START( sc4wldbne )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410062.lo", 0x00001, 0x080000, CRC(ba531b0a) SHA1(98d1c3f15e39f7ee51141580479b4b37451fa042) )
	ROM_LOAD16_BYTE( "95410063.hi", 0x00000, 0x080000, CRC(1a7a2fbc) SHA1(16a18e951c47991fd3633cfb42bb9b1d8353f6c8) )
	sc_wldbn_others
ROM_END

ROM_START( sc4wldbnf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410064.lo", 0x00001, 0x080000, CRC(85dcb53d) SHA1(101392cf181d964c258f5bfcad5a4fa84f77caf6) )
	ROM_LOAD16_BYTE( "95410065.hi", 0x00000, 0x080000, CRC(9519c652) SHA1(54d891b9a45c642650a44be164bf94f9f0c47977) )
	sc_wldbn_others
ROM_END

ROM_START( sc4wldbng )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410582.lo", 0x00001, 0x080000, CRC(13299581) SHA1(391c827d1592e8090eebab57e28a482029403d8a) )
	ROM_LOAD16_BYTE( "95410583.hi", 0x00000, 0x080000, CRC(0c59c28f) SHA1(e590df5666f09439aef2352a72b608f032c15309) )
	sc_wldbn_others
ROM_END

ROM_START( sc4wldbnh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410584.lo", 0x00001, 0x080000, CRC(97614a28) SHA1(b4d643b62f3a2eafc83dc51cf356545f1ef315b8) )
	ROM_LOAD16_BYTE( "95410585.hi", 0x00000, 0x080000, CRC(4191c770) SHA1(5d681a9edc3e4d43781285c0d7cd1927790840b4) )
	sc_wldbn_others
ROM_END

ROM_START( sc4wldbni )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411060.lo", 0x00001, 0x080000, CRC(79ae7a14) SHA1(aae8e31b115cf54aa2d1997a5cae93db0e811500) )
	ROM_LOAD16_BYTE( "95411061.hi", 0x00000, 0x080000, CRC(b3a28869) SHA1(5536221e358e5d545369978fec62070d0b4a1257) )
	sc_wldbn_others
ROM_END

ROM_START( sc4wldbnj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411062.lo", 0x00001, 0x080000, CRC(53f2e0b2) SHA1(5472ab467fa1adb8eecc0cce348330d091f3222b) )
	ROM_LOAD16_BYTE( "95411063.hi", 0x00000, 0x080000, CRC(d16c20e7) SHA1(3aa22dbe6e3f4f56ba9614448199a69cf1ded868) )
	sc_wldbn_others
ROM_END

ROM_START( sc4wldbnk )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411064.lo", 0x00001, 0x080000, CRC(63fffbf2) SHA1(361c502d22fc57c698d85d4a5fcd065a021f8ca3) )
	ROM_LOAD16_BYTE( "95411065.hi", 0x00000, 0x080000, CRC(469b08b3) SHA1(7cd44820d517bcbb4588ac5ba7599d60d65dfb6b) )
	sc_wldbn_others
ROM_END

ROM_START( sc4wldbnl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411582.lo", 0x00001, 0x080000, CRC(9e8953aa) SHA1(7823d79838ee2dd0d631b149680bbda25f334460) )
	ROM_LOAD16_BYTE( "95411583.hi", 0x00000, 0x080000, CRC(c896a4bd) SHA1(f78cbf64e85b2ed88e0798aa65e2ce5d7dd9c672) )
	sc_wldbn_others
ROM_END

ROM_START( sc4wldbnm )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411584.lo", 0x00001, 0x080000, CRC(9905864b) SHA1(ef69b35ab10e63ff2e10b94889179aab0c08132c) )
	ROM_LOAD16_BYTE( "95411585.hi", 0x00000, 0x080000, CRC(ea0275ea) SHA1(68e9d894504eab0d8ade5ec770dc35f94cd6e0b5) )
	sc_wldbn_others
ROM_END


ROM_START( sc4wldbnn )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95412798.lo", 0x00001, 0x080000, CRC(b1faf86e) SHA1(59d28f3cc7e6102b6ab30cf8f3cf3ff330400935) )
	ROM_LOAD16_BYTE( "95412799.hi", 0x00000, 0x080000, CRC(154696d1) SHA1(e36ac082a4d9217609d480c8ebe2a62ca5e05030) )
	ROM_REGION( 0x400000, "ymz", ROMREGION_ERASE00)
ROM_END


ROM_START( sc4wldbno )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95412802.lo", 0x00001, 0x080000, CRC(0707c737) SHA1(ab2b5654593af4b2ef3b967b922a140d254832cd) )
	ROM_LOAD16_BYTE( "95412803.hi", 0x00000, 0x080000, CRC(7cba6d55) SHA1(7770f2cb130f54f4f43004b78e1494d0d067f444) )
	ROM_REGION( 0x400000, "ymz", ROMREGION_ERASE00)
ROM_END


ROM_START( sc4wldbnp )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95413798.lo", 0x00001, 0x080000, CRC(78d31c7e) SHA1(4ce9b96b039c2e70b53b7446cc7701cfa0d21f69) )
	ROM_LOAD16_BYTE( "95413799.hi", 0x00000, 0x080000, CRC(969a883b) SHA1(9e23f39bdb3ac8819f2910d3ae86ffc63e36d78b) )
	ROM_REGION( 0x400000, "ymz", ROMREGION_ERASE00)
ROM_END

ROM_START( sc4wldbnq )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95413802.lo", 0x00001, 0x080000, CRC(cda11a90) SHA1(fcf4728cbc93b537283b06f7234dfdd1b1c986fe) )
	ROM_LOAD16_BYTE( "95413803.hi", 0x00000, 0x080000, CRC(4e53c2cc) SHA1(9f844af4abb60c4616c840075e7f7bbadaad70be) )
	ROM_REGION( 0x400000, "ymz", ROMREGION_ERASE00)
ROM_END


ROM_START( sc4wthng )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402371.lo", 0x00001, 0x080000, CRC(8cde578d) SHA1(2fe764c8a018ee7893ac3d6ce54f73455cfc9395) )
	ROM_LOAD16_BYTE( "95402372.hi", 0x00000, 0x080000, CRC(aced6068) SHA1(6519ddecec8ec200161d273d3279b8eb2e752610) )
	sc_wthng_others
ROM_END

ROM_START( sc4wthnga )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402512.lo", 0x00001, 0x080000, CRC(a0e0fe7a) SHA1(076f646b44a1fae6c27ad3875dc8a86214dbe2fc) )
	ROM_LOAD16_BYTE( "95402513.hi", 0x00000, 0x080000, CRC(409e66d8) SHA1(8fe03bca8140b919f77893571ff8d2e9e8079d60) )
	sc_wthng_others
ROM_END

ROM_START( sc4wthngb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403371.lo", 0x00001, 0x080000, CRC(5863f65b) SHA1(f56f613decbd19944f3276045465450788f46d66) )
	ROM_LOAD16_BYTE( "95403372.hi", 0x00000, 0x080000, CRC(6983fc12) SHA1(c68242d707e4ecfe93a6ec2ecb27bb74d4b5e8b6) )
	sc_wthng_others
ROM_END

ROM_START( sc4wthngc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403512.lo", 0x00001, 0x080000, CRC(25798118) SHA1(09d120509e53fb47cee5263c09f9d421eca8a8d3) )
	ROM_LOAD16_BYTE( "95403513.hi", 0x00000, 0x080000, CRC(ab6bab04) SHA1(6561ffcf822c9228a04c79cc4795eb408f41ed95) )
	sc_wthng_others
ROM_END


ROM_START( sc4wthnm )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412264.lo", 0x00001, 0x080000, CRC(bace4f11) SHA1(0a9c3ad35c547c513aca290e03af968050def5e9) )
	ROM_LOAD16_BYTE( "95412265.hi", 0x00000, 0x080000, CRC(cd1c4e6e) SHA1(b564bea4e1a19a533e54163e0fb19b66afbb7317) )
	sc_wthnm_others
ROM_END

ROM_START( sc4wthnma )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413264.lo", 0x00001, 0x080000, CRC(76fe4465) SHA1(2b8612cd55f75337f4f14f71d54f45ed37810bce) )
	ROM_LOAD16_BYTE( "95413265.hi", 0x00000, 0x080000, CRC(d912122d) SHA1(c97f0d6ec98a64ead447ecbbad6b36b054e9100f) )
	sc_wthnm_others
ROM_END

ROM_START( sc4wthnmb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415092.lo", 0x00001, 0x080000, CRC(53ee4489) SHA1(1c12f0abde9827d9bb8dfe90b40c578f5b1933d1) )
	ROM_LOAD16_BYTE( "95415093.hi", 0x00000, 0x080000, CRC(30759ce6) SHA1(d75f0a978f220dfba05e78817bdecf1a47de654b) )
	sc_wthnm_others
ROM_END

ROM_START( sc4wthnmc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415996.lo", 0x00001, 0x080000, CRC(dceffb2f) SHA1(467f4b8e0410c3672c72ab444f7b51666f393a57) )
	ROM_LOAD16_BYTE( "95415997.hi", 0x00000, 0x080000, CRC(0c5d7998) SHA1(75759b0af87f04c459e3fdc1554b215bdc5a6278) )
	sc_wthnm_others
ROM_END

ROM_START( sc4wthnmd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415998.lo", 0x00001, 0x080000, CRC(4caccab6) SHA1(07da95fafbca8982202af67f32a53c5829b2ba24) )
	ROM_LOAD16_BYTE( "95415999.hi", 0x00000, 0x080000, CRC(15552b8f) SHA1(f120b65568eba18d53ee40cb0392663dc6eb4dcd) )
	sc_wthnm_others
ROM_END

ROM_START( sc4wthnme )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417012.lo", 0x00001, 0x080000, CRC(16879c8d) SHA1(f976dee82c6a05a4c2bb016cdc4d1636df11a5b0) )
	ROM_LOAD16_BYTE( "95417013.hi", 0x00000, 0x080000, CRC(687dd1f9) SHA1(b2b5cc21a4a6bd4ef4bf37a235b9f03b323791d4) )
	sc_wthnm_others
ROM_END

ROM_START( sc4wthnmf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417014.lo", 0x00001, 0x080000, CRC(75e910f3) SHA1(bf7794651721797870ee148e2249f53acea3b441) )
	ROM_LOAD16_BYTE( "95417015.hi", 0x00000, 0x080000, CRC(b8d5f355) SHA1(b15c6509358334ffe9f750b96a25e4381e874e1d) )
	sc_wthnm_others
ROM_END



ROM_START( sc4wtc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404512.lo", 0x00001, 0x080000, CRC(d19da45b) SHA1(797ac1f6a97454bdaa1fb1dee892ee242fba8872) )
	ROM_LOAD16_BYTE( "95404513.hi", 0x00000, 0x080000, CRC(5b96b3d8) SHA1(8f9dfdf3ee27ddafcf3372e62cd48a365d0d8b8b) )
	sc_wtc_others
ROM_END

ROM_START( sc4wtca )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404514.lo", 0x00001, 0x080000, CRC(3974cde4) SHA1(63ba97d8a38d4ab4da534c3694de0371c3774d5a) )
	ROM_LOAD16_BYTE( "95404515.hi", 0x00000, 0x080000, CRC(7b49a314) SHA1(eba6dc7c3c9a886fb135e85270f4cb08f10eea75) )
	sc_wtc_others
ROM_END

ROM_START( sc4wtcb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405512.lo", 0x00001, 0x080000, CRC(8f023916) SHA1(7dda48feb0ade5bebb515fbc45bb784ccd541a8f) )
	ROM_LOAD16_BYTE( "95405513.hi", 0x00000, 0x080000, CRC(4f98ef9b) SHA1(ea706a37f848014b13e318944f8ecad03ca00846) )
	sc_wtc_others
ROM_END

ROM_START( sc4wtcc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405516.lo", 0x00001, 0x080000, CRC(53c6c8a6) SHA1(25ea04184f43e842cc6e6772ca3fd1efca86c7c5) )
	ROM_LOAD16_BYTE( "95405517.hi", 0x00000, 0x080000, CRC(b7e08463) SHA1(dbcb1fb6a06ca1f9c100c6beaaec98a153b52572) )
	sc_wtc_others
ROM_END

ROM_START( sc4wtcd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405518.lo", 0x00001, 0x080000, CRC(cdfb2f40) SHA1(d168eb292e35c551cec63769a52259d39b4d55ca) )
	ROM_LOAD16_BYTE( "95405519.hi", 0x00000, 0x080000, CRC(056732b2) SHA1(271cac84becb847e2916897820ba27ad2f62cc4c) )
	sc_wtc_others
ROM_END

ROM_START( sc4wtce )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406406.lo", 0x00001, 0x080000, CRC(d5debf25) SHA1(cee6034eb0994cd4deccf9f42b7b0ca8a971668f) )
	ROM_LOAD16_BYTE( "95406407.hi", 0x00000, 0x080000, CRC(e7aaf70b) SHA1(5ecf4d60ad163f54e5dc34aee3d265b2a92387f6) )
	sc_wtc_others
ROM_END

ROM_START( sc4wtcf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407406.lo", 0x00001, 0x080000, CRC(214864a1) SHA1(680daf34460dcce8080782068f36a6d50b508041) )
	ROM_LOAD16_BYTE( "95407407.hi", 0x00000, 0x080000, CRC(9dc6d390) SHA1(1ae57bf771266ee5f302ad22a3241b0d9409ce3f) )
	sc_wtc_others
ROM_END



ROM_START( sc4wspin )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416048.odd", 0x00001, 0x080000, CRC(52b89fce) SHA1(6a56f58ee44110cfec2da02389089e075daf52c3) )
	ROM_LOAD16_BYTE( "95416049.evn", 0x00000, 0x080000, CRC(bfe02b55) SHA1(ba63d26b04f8962b39f30304806d39db71540457) )
	sc_wspin_others
ROM_END

ROM_START( sc4wspinb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416072.odd", 0x00001, 0x080000, CRC(e70b27cb) SHA1(857284ac95a3354317915dfd94f8d794c29267da) )
	ROM_LOAD16_BYTE( "95416073.evn", 0x00000, 0x080000, CRC(83f8e087) SHA1(da99388b3084728c5ef086610948a8f01c538315) )
	sc_wspin_others
ROM_END

ROM_START( sc4wspinc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417048.odd", 0x00001, 0x080000, CRC(0ee5dd5a) SHA1(383c9c67295ae10fe7f78fc7da01b31e8b10d0ca) )
	ROM_LOAD16_BYTE( "95417049.evn", 0x00000, 0x080000, CRC(7a8eb72f) SHA1(47b29e6ef18c0b1da41f6397e6c52ebf09959228) )
	sc_wspin_others
ROM_END

ROM_START( sc4wspind )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417072.odd", 0x00001, 0x080000, CRC(9c705a07) SHA1(c873115a6769cea96d7276e9d4e89e70c3b7d276) )
	ROM_LOAD16_BYTE( "95417073.evn", 0x00000, 0x080000, CRC(82591acf) SHA1(70bac24f6b7df4b321460342c358cf0ef1c42d86) )
	sc_wspin_others
ROM_END


ROM_START( sc4wwys )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1301p13.hi", 0x00000, 0x080000, CRC(62e6c8a1) SHA1(938df49c93274650ee078ed31980eb6b2465315d) )
	ROM_LOAD16_BYTE( "p1301p13.lo", 0x00001, 0x080000, CRC(5e98d205) SHA1(e73a7f59b9c4ab1772678f2b0d7d88f0bec14dcb) )
	sc_wwys_others
ROM_END

ROM_START( sc4wwysa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1301s13.hi", 0x00000, 0x080000, CRC(78a947e3) SHA1(aea4fa72cae0121c5a15812d227f43080f04a2a2) )
	ROM_LOAD16_BYTE( "p1301s13.lo", 0x00001, 0x080000, CRC(c36ed88b) SHA1(287c6bfa5d1398b7efd21bc9565ad4d2557f7378) )
	sc_wwys_others
ROM_END


ROM_START( sc4winxp )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406368.lo", 0x00001, 0x080000, CRC(cb014e09) SHA1(1e1fc13b5cf6c027a169ad9379e44a41ee22ef18) )
	ROM_LOAD16_BYTE( "95406369.hi", 0x00000, 0x080000, CRC(77128fec) SHA1(c6add5d90e1cb621589e988012eda7b2a3e0a568) )
	sc_winxp_others
ROM_END


ROM_START( sc4winsp )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414890.odd", 0x00001, 0x080000, CRC(b236cae2) SHA1(19511af3f728615fea097d283072844602a228f9) )
	ROM_LOAD16_BYTE( "95414891.evn", 0x00000, 0x080000, CRC(acd72d71) SHA1(8ecebbf828a001b6ce3339c5237fddb05d2135ff) )
	sc_winsp_others
ROM_END

ROM_START( sc4winspa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414892.odd", 0x00001, 0x080000, CRC(0fd370d3) SHA1(7765109f28ee823fbc3066f03e06836702151bd7) )
	ROM_LOAD16_BYTE( "95414893.evn", 0x00000, 0x080000, CRC(21b3d497) SHA1(6b77e3e200c86fc826fa0b50b85cee795f768e65) )
	sc_winsp_others
ROM_END

ROM_START( sc4winspb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414894.odd", 0x00001, 0x080000, CRC(758b2b7d) SHA1(ed115f78317a69e7953321b3bb1293e7b857f0d3) )
	ROM_LOAD16_BYTE( "95414895.evn", 0x00000, 0x080000, CRC(52b8f1c5) SHA1(428b4f3bc55e180763df3707cfb1044a9181e724) )
	sc_winsp_others
ROM_END

ROM_START( sc4winspc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414896.odd", 0x00001, 0x080000, CRC(7ff2ff25) SHA1(c48221410c507d44b83fba6f62c860c1c69b14d1) )
	ROM_LOAD16_BYTE( "95414897.evn", 0x00000, 0x080000, CRC(0f3329e4) SHA1(da2afebd87c95fd74abd8a09c301d8ffe8389550) )
	sc_winsp_others
ROM_END

ROM_START( sc4winspd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414908.odd", 0x00001, 0x080000, CRC(98870df3) SHA1(d3749471939d035288dd957ba63787f1b1be4e74) )
	ROM_LOAD16_BYTE( "95414909.evn", 0x00000, 0x080000, CRC(a725a4a9) SHA1(f5a1e6f3b026c45a9df6b292a422afa59d115cfd) )
	sc_winsp_others
ROM_END

ROM_START( sc4winspe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414984.odd", 0x00001, 0x080000, CRC(b6d64685) SHA1(039f1bdde77acfff313abde8acf6dffcfc6c2a90) )
	ROM_LOAD16_BYTE( "95414985.evn", 0x00000, 0x080000, CRC(bae2b240) SHA1(2946c21d540a1631c6418766aeb2851cf07bf628) )
	sc_winsp_others
ROM_END

ROM_START( sc4winspf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414986.odd", 0x00001, 0x080000, CRC(b417ee3c) SHA1(7767a6a2b01b3073b18c3dd444e7a07b0671905e) )
	ROM_LOAD16_BYTE( "95414987.evn", 0x00000, 0x080000, CRC(b521ec03) SHA1(e5f5e49bc591551a28c70b4aa997577c0652f4d6) )
	sc_winsp_others
ROM_END

ROM_START( sc4winspg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414988.odd", 0x00001, 0x080000, CRC(68c56a33) SHA1(b494de885163227f8d3e735b4b35503273d953d8) )
	ROM_LOAD16_BYTE( "95414989.evn", 0x00000, 0x080000, CRC(b69b4d7e) SHA1(e5ece3ba353e1630d3ae4d2766d8c54f064945a9) )
	sc_winsp_others
ROM_END

ROM_START( sc4winsph )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415892.odd", 0x00001, 0x080000, CRC(58611d9b) SHA1(a31137a1d21281d9defd884080f504d013435cb2) )
	ROM_LOAD16_BYTE( "95415893.evn", 0x00000, 0x080000, CRC(c59f7022) SHA1(2ae528880152d0763037feed3ea46bf19407bd9c) )
	sc_winsp_others
ROM_END

ROM_START( sc4winspi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415894.odd", 0x00001, 0x080000, CRC(6a0793ad) SHA1(ffab484074c896cf8a14463dda6b47cd3ef2a2f9) )
	ROM_LOAD16_BYTE( "95415895.evn", 0x00000, 0x080000, CRC(050c7068) SHA1(744c0f44b79d9ff6fb2bad4ca92d1b55ba585b1c) )
	sc_winsp_others
ROM_END

ROM_START( sc4winspj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415896.odd", 0x00001, 0x080000, CRC(607e47f5) SHA1(c22c3fcfbea5e656eddd8e55533d709a248d98ca) )
	ROM_LOAD16_BYTE( "95415897.evn", 0x00000, 0x080000, CRC(5887a849) SHA1(f5ae020e23f23407f39153413772a51ec9fa9507) )
	sc_winsp_others
ROM_END

ROM_START( sc4winspk )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415908.odd", 0x00001, 0x080000, CRC(e99d8830) SHA1(730c8ae6b9f81835862e6ede7b3d7624b44339fd) )
	ROM_LOAD16_BYTE( "95415909.evn", 0x00000, 0x080000, CRC(8bc3adef) SHA1(6499cc21550867ede2484348ab826ee38c9ef0d6) )
	sc_winsp_others
ROM_END

ROM_START( sc4winspl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415984.odd", 0x00001, 0x080000, CRC(909e4fa8) SHA1(391638f35b2768f67dd0626d5afea00ce4859a28) )
	ROM_LOAD16_BYTE( "95415985.evn", 0x00000, 0x080000, CRC(80e8e99d) SHA1(ac5ee0db62a9ac789e6b02b65e93ddc92a95794c) )
	sc_winsp_others
ROM_END

ROM_START( sc4winspm )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415986.odd", 0x00001, 0x080000, CRC(2c811199) SHA1(d16790e0a4a8161be75060f59fa4156c08dea78c) )
	ROM_LOAD16_BYTE( "95415987.evn", 0x00000, 0x080000, CRC(1ccf0679) SHA1(1941cc6e88a6bf769302c0e260e34f516ff222f1) )
	sc_winsp_others
ROM_END

ROM_START( sc4winspn )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415988.odd", 0x00001, 0x080000, CRC(f0539596) SHA1(513170b69aaff10244a5c556dadbd232ab7e2542) )
	ROM_LOAD16_BYTE( "95415989.evn", 0x00000, 0x080000, CRC(1f75a704) SHA1(13731ec1bdd8a780bf5b117b4de8e95710774155) )
	sc_winsp_others
ROM_END

ROM_START( sc4winspo )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416114.odd", 0x00001, 0x080000, CRC(faefa0a4) SHA1(b75258b7b03c8e3af5294ecfc58db39a686dae10) )
	ROM_LOAD16_BYTE( "95416115.evn", 0x00000, 0x080000, CRC(458dd65c) SHA1(21be5720c5d46286ccc45ac5f46a88ecb6cd35ff) )
	sc_winsp_others
ROM_END

ROM_START( sc4winspp )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416150.odd", 0x00001, 0x080000, CRC(7f0302c3) SHA1(43a4800904f0785996175ffdbbe40c5afa4370b4) )
	ROM_LOAD16_BYTE( "95416151.evn", 0x00000, 0x080000, CRC(e4cd7e83) SHA1(53027a6db2961878a70f7bbfe30a1e661a625323) )
	sc_winsp_others
ROM_END

ROM_START( sc4winspq )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416348.odd", 0x00001, 0x080000, CRC(ef3e1d73) SHA1(93da61a878c2d33ec8f490c8aae39f1baf033100) )
	ROM_LOAD16_BYTE( "95416349.evn", 0x00000, 0x080000, CRC(6a771259) SHA1(83cb40367d3ea9601d343db7baae3e8df81b57b9) )
	sc_winsp_others
ROM_END

ROM_START( sc4winspr )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416350.odd", 0x00001, 0x080000, CRC(b1dacd4f) SHA1(c52554810ad8ff05d737e8f3f4d25e5074aff547) )
	ROM_LOAD16_BYTE( "95416351.evn", 0x00000, 0x080000, CRC(b7744719) SHA1(1443b274b61297f92411c6ac1abe79d07b881c67) )
	sc_winsp_others
ROM_END

ROM_START( sc4winsps )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416352.odd", 0x00001, 0x080000, CRC(8cc72003) SHA1(c87f4128f4131c444fb0dea19e493ce7a38714be) )
	ROM_LOAD16_BYTE( "95416353.evn", 0x00000, 0x080000, CRC(ce28b419) SHA1(8a18c835c561d9a1d5ea8f4ec1e7bde6aef78567) )
	sc_winsp_others
ROM_END

ROM_START( sc4winspt )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416354.odd", 0x00001, 0x080000, CRC(3ba7aed6) SHA1(21f05531c6f919d5467adb2bae02e46283883e24) )
	ROM_LOAD16_BYTE( "95416355.evn", 0x00000, 0x080000, CRC(80c59dc5) SHA1(4bbf03a721ad87101979774e67782bf09d9ca184) )
	sc_winsp_others
ROM_END

ROM_START( sc4winspu )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417114.odd", 0x00001, 0x080000, CRC(250f1566) SHA1(5fb712a2662c6eb70b401a28befdd451026b3127) )
	ROM_LOAD16_BYTE( "95417115.evn", 0x00000, 0x080000, CRC(b5af2eaa) SHA1(51a37c811be4566478ead5fede7f8dc82927ea1e) )
	sc_winsp_others
ROM_END

ROM_START( sc4winspv )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417150.odd", 0x00001, 0x080000, CRC(5223412f) SHA1(ee8c3024e69c78319246c0cadaf46e0d63c69192) )
	ROM_LOAD16_BYTE( "95417151.evn", 0x00000, 0x080000, CRC(6e1cf4d6) SHA1(b0a2afb65580da98fc8379210991433ef25f5801) )
	sc_winsp_others
ROM_END

ROM_START( sc4winspw )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417348.odd", 0x00001, 0x080000, CRC(5e838229) SHA1(8d07f9ef8b1ed071f2a55bb3c6647410c6b7c559) )
	ROM_LOAD16_BYTE( "95417349.evn", 0x00000, 0x080000, CRC(8ffa4ca4) SHA1(b0cfd9fbdaf40f22ae293ef836c3493fca195304) )
	sc_winsp_others
ROM_END

ROM_START( sc4winspx )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417350.odd", 0x00001, 0x080000, CRC(f7bcae3d) SHA1(5b4becb70bdfc8ed971b4b6663ba352493dc378c) )
	ROM_LOAD16_BYTE( "95417351.evn", 0x00000, 0x080000, CRC(8f9c121c) SHA1(876b54be2e9995cc91d3efc45e0a8ac99ad377c1) )
	sc_winsp_others
ROM_END

ROM_START( sc4winspy )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417352.odd", 0x00001, 0x080000, CRC(caa14371) SHA1(a9e3c3823746abde54478168876cb56e4b6e0959) )
	ROM_LOAD16_BYTE( "95417353.evn", 0x00000, 0x080000, CRC(f6c0e11c) SHA1(e383588102ada0a08bd5dea32e654a3882acbed8) )
	sc_winsp_others
ROM_END

ROM_START( sc4winspz )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417354.odd", 0x00001, 0x080000, CRC(0a6afe0c) SHA1(73f4a4eb7c7ebb6f8c685f7c902808819068d8c3) )
	ROM_LOAD16_BYTE( "95417355.evn", 0x00000, 0x080000, CRC(7ddc0282) SHA1(3e79a277276adf384a3065c29d8e358f2a534296) )
	sc_winsp_others
ROM_END

ROM_START( sc4winsp0 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414962.odd", 0x00001, 0x080000, CRC(bcf5b2e5) SHA1(07c15798d23c4adba0cef2281f81ca9f715ae2f1) )
	ROM_LOAD16_BYTE( "95414963.evn", 0x00000, 0x080000, CRC(5f8539af) SHA1(1889e351a3d808d52917d0a56ef4e8ce5cf81042) )
	sc_winsp_others
ROM_END


ROM_START( sc4wondw )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1318p11.hi", 0x00000, 0x080000, CRC(3704daa2) SHA1(2e20ea8473d69432b17d6d3ce6d241dc0331345e) )
	ROM_LOAD16_BYTE( "p1318p11.lo", 0x00001, 0x080000, CRC(5c28ed76) SHA1(b2ce5a73410cba6579bbb37dae23d220611af7a4) )
	sc_wondw_others
ROM_END

ROM_START( sc4wondwa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1318s11.hi", 0x00000, 0x080000, CRC(f9b2dcf0) SHA1(873adac0859ec1d64858be0320369e8b99a0af32) )
	ROM_LOAD16_BYTE( "p1318s11.lo", 0x00001, 0x080000, CRC(603e115b) SHA1(46e626e25b204b657d065531d475c498b74df61f) )
	sc_wondw_others
ROM_END


ROM_START( sc4xmark )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1116p16.hi", 0x00000, 0x080000, CRC(53c9161a) SHA1(0afb525a4b28f5068dc04b274881712190fbdff6) )
	ROM_LOAD16_BYTE( "p1116p16.lo", 0x00001, 0x080000, CRC(f409ccb9) SHA1(ef60f3bae17b3d11e4307b1f93c9a21721f10511) )
	sc_xmark_others
ROM_END

ROM_START( sc4xmarka )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1116s16.hi", 0x00000, 0x080000, CRC(383bbaf0) SHA1(50e3929f8c726c010f434a43b3052af7b03d1dfc) )
	ROM_LOAD16_BYTE( "p1116s16.lo", 0x00001, 0x080000, CRC(534cac1a) SHA1(92006516a77d367c7712ba3474c4bcfa994c47ec) )
	sc_xmark_others
ROM_END


ROM_START( sc4xcash )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408160.lo", 0x00001, 0x080000, CRC(2532bc0a) SHA1(a4886fb10bf3d9c44f702d26dc82a5c54b6ea975) )
	ROM_LOAD16_BYTE( "95408161.hi", 0x00000, 0x080000, CRC(2b6ca545) SHA1(87cdb7c0723b31015560953b2393625ce147c508) )
	sc_xcash_others
ROM_END



ROM_START( sc4s16 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416184.lo", 0x00001, 0x080000, CRC(1c96f8f2) SHA1(5afb19c9ce8f83dda7208f68ed6e6a402662ac16) )
	ROM_LOAD16_BYTE( "95416185.hi", 0x00000, 0x080000, CRC(a03534ee) SHA1(1af3c2defeafdf0fd43bef7112ee63778ba75f0f) )
	sc4s16_others
ROM_END

ROM_START( sc4s16a )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417184.lo", 0x00001, 0x080000, CRC(f126c832) SHA1(afca10926c41be212c36438f47d829c7f5df9ed0) )
	ROM_LOAD16_BYTE( "95417185.hi", 0x00000, 0x080000, CRC(5c8d51e1) SHA1(e6a08ab10dddb04a432578edacbef21fcd68eec6) )
	sc4s16_others
ROM_END






ROM_START( sc4bgold )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404304.lo", 0x00001, 0x080000, NO_DUMP )
	ROM_LOAD16_BYTE( "95404305.hi", 0x00000, 0x080000, CRC(d618e9b5) SHA1(ed333968c917c67fbaf7ef30b674f3a0619d3b02) )
	sc_bgold_others
ROM_END

ROM_START( sc4bgolda )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404604.lo", 0x00001, 0x080000, CRC(3697c961) SHA1(6053e5e05092294c41b78377b8d739309c2e45ac) )
	ROM_LOAD16_BYTE( "95404605.hi", 0x00000, 0x080000, NO_DUMP )
	sc_bgold_others
ROM_END






ROM_START( sc4clbtm )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402149.lo", 0x00001, 0x080000, CRC(f5eee630) SHA1(102ef766562a67cd5c339d755f527252aee924be) )
	ROM_LOAD16_BYTE( "95402150.hi", 0x00000, 0x080000, CRC(19ada5f4) SHA1(3719e29465249026c781fe5226d05770c2e8ce99) )
	sc_clbtm_others

	sc_clbtm_matrix
ROM_END

ROM_START( sc4clbtma )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402257.lo", 0x00001, 0x080000, CRC(fca966af) SHA1(63aa17640405fb858d776799e2388679dfe02a26) )
	ROM_LOAD16_BYTE( "95402258.hi", 0x00000, 0x080000, CRC(24ca572a) SHA1(5daa89c4427e70d0fbebfca116ea6932716f38f8) )
	sc_clbtm_others

	sc_clbtm_matrix
ROM_END

ROM_START( sc4clbtmb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403149.lo", 0x00001, 0x080000, CRC(a62d1945) SHA1(197f9245d46cb156987cbacc1ac0c8230030fdcd) )
	ROM_LOAD16_BYTE( "95403150.hi", 0x00000, 0x080000, CRC(9f79e460) SHA1(0162d5fc54361c5853dc865d61849f766f833545) )
	sc_clbtm_others

	sc_clbtm_matrix
ROM_END

ROM_START( sc4clbtmc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95403257.lo", 0x00001, 0x080000, CRC(5aada808) SHA1(e24bc5e523961abdedfb27ec5cff2cbb6f45bddd) )
	ROM_LOAD16_BYTE( "95403258.hi", 0x00000, 0x080000, CRC(2b133e43) SHA1(805ca1ddf1ebcf6cbbd4b8b41baf5c2056b5bf17) )
	sc_clbtm_others

	sc_clbtm_matrix
ROM_END

ROM_START( sc4clbtmd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404682.lo", 0x00001, 0x080000, CRC(3b9e429f) SHA1(01046d8671a287b2a1c739f84d28c69ba2c8c80f) )
	ROM_LOAD16_BYTE( "95404683.hi", 0x00000, 0x080000, CRC(f84b37ad) SHA1(4430c7fe6274d25de58342d255c4c4a52966b0b3) )
	sc_clbtm_others

	sc_clbtm_matrix
ROM_END

ROM_START( sc4clbtme )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405682.lo", 0x00001, 0x080000, CRC(12d4bf1a) SHA1(2b28c5fdca13e7a70496984e02bfd3b98d60a9ac) )
	ROM_LOAD16_BYTE( "95405683.hi", 0x00000, 0x080000, CRC(b9aba368) SHA1(ec6404447c9e23e179ec1200848d966d5f3f846b) )
	sc_clbtm_others

	sc_clbtm_matrix
ROM_END



ROM_START( sc4dcrls )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414698.odd", 0x00001, 0x080000, CRC(0bb93d5c) SHA1(f3b0c59b731ba3ae2074fe864cbb5ac8f4cac651) )
	ROM_LOAD16_BYTE( "95414699.evn", 0x00000, 0x080000, CRC(f4f800a1) SHA1(3be6ef0c8b15d00db3ccf41646d661e4b82be20e) )
	sc_dcrls_others
ROM_END

ROM_START( sc4dcrlsa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414700.odd", 0x00001, 0x080000, CRC(01aeb959) SHA1(71646ec778408384c8cd973199b4491211763718) )
	ROM_LOAD16_BYTE( "95414701.evn", 0x00000, 0x080000, CRC(f17499f5) SHA1(6f1275d118657dead35dbf79f5a3efcd4a396954) )
	sc_dcrls_others
ROM_END

ROM_START( sc4dcrlsb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414780.odd", 0x00001, 0x080000, CRC(b8331602) SHA1(ad4eb67d6382cf09fbb8e5cb5cf5ebcf24ce70fc) )
	ROM_LOAD16_BYTE( "95414781.evn", 0x00000, 0x080000, CRC(9a7b0030) SHA1(2be0aa919b4661218365ceb6c85d5527c2788077) )
	sc_dcrls_others
ROM_END

ROM_START( sc4dcrlsc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414782.odd", 0x00001, 0x080000, CRC(52ddb034) SHA1(b6a1d432ed94d21555e37b7ef134766ee5dc67e2) )
	ROM_LOAD16_BYTE( "95414783.evn", 0x00000, 0x080000, CRC(2baa047f) SHA1(4fc5e0e9e21d2dbbeabe8c9af6613ecf53edb08f) )
	sc_dcrls_others
ROM_END

ROM_START( sc4dcrlsd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414898.odd", 0x00001, 0x080000, CRC(9459b0df) SHA1(c7220fba49630f2c48aeab95f9cbcab620ccdc70) )
	ROM_LOAD16_BYTE( "95414899.evn", 0x00000, 0x080000, CRC(394ea9d1) SHA1(e0466af5de3097f9cdae0c95e808fcb2df853035) )
	sc_dcrls_others
ROM_END

ROM_START( sc4dcrlse )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414900.odd", 0x00001, 0x080000, CRC(b84e2dbd) SHA1(7d5a4e34f8659ac559b5b65e22d2b4da6d38786a) )
	ROM_LOAD16_BYTE( "95414901.evn", 0x00000, 0x080000, CRC(832918b4) SHA1(ae9700c7d70ead755c4481a07b59bcaa51b893b4) )
	sc_dcrls_others
ROM_END

ROM_START( sc4dcrlsf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415698.odd", 0x00001, 0x080000, CRC(697dc789) SHA1(eb1144fb72f3d850cad8026c62095e89fc2ae285) )
	ROM_LOAD16_BYTE( "95415699.evn", 0x00000, 0x080000, CRC(ed526d6b) SHA1(34ce8f9526564b03ecb1f45d9883942407f53752) )
	sc_dcrls_others
ROM_END

ROM_START( sc4dcrlsg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415700.odd", 0x00001, 0x080000, CRC(636a438c) SHA1(64a1520466e1cafa4784f63034931f652465ae98) )
	ROM_LOAD16_BYTE( "95415701.evn", 0x00000, 0x080000, CRC(e8def43f) SHA1(b849ca3176c89cc62b53ccc0a562ec7bfc53ae4a) )
	sc_dcrls_others
ROM_END

ROM_START( sc4dcrlsh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415780.odd", 0x00001, 0x080000, CRC(fb2a8220) SHA1(16317ed2609c163ef5a23fa3e9c6ba05c94ed763) )
	ROM_LOAD16_BYTE( "95415781.evn", 0x00000, 0x080000, CRC(516d0f6b) SHA1(a12a34d1d5dfd7e13c470d561c519d090fa4d6fa) )
	sc_dcrls_others
ROM_END

ROM_START( sc4dcrlsi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415782.odd", 0x00001, 0x080000, CRC(11c42416) SHA1(9a846a19856a15919186985a4b6383d5ea3a2a30) )
	ROM_LOAD16_BYTE( "95415783.evn", 0x00000, 0x080000, CRC(e0bc0b24) SHA1(eb87b62de2800107fe2153213704fcb7ab8fd586) )
	sc_dcrls_others
ROM_END

ROM_START( sc4dcrlsj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415898.odd", 0x00001, 0x080000, CRC(05d6ccd5) SHA1(ff6fb91f61d4620fd46a3fd8070a3a9fa4335347) )
	ROM_LOAD16_BYTE( "95415899.evn", 0x00000, 0x080000, CRC(6260b585) SHA1(989f32a5007a85c4a469cbadf573c9c3620dfe87) )
	sc_dcrls_others
ROM_END

ROM_START( sc4dcrlsk )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415900.odd", 0x00001, 0x080000, CRC(29c151b7) SHA1(2d3d24e0b2ef0a01d5c4c30feb6502c894986b02) )
	ROM_LOAD16_BYTE( "95415901.evn", 0x00000, 0x080000, CRC(d80704e0) SHA1(ba6622c8c8b47c66133980c8823690325ffa000c) )
	sc_dcrls_others
ROM_END





ROM_START( sc4gcclb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406700.lo", 0x00001, 0x080000, CRC(1f3ae8fb) SHA1(957108b50deb9e43f541fa2b00dfca28422db7ba) )
	ROM_LOAD16_BYTE( "95406701.hi", 0x00000, 0x080000, CRC(d76642cc) SHA1(f650d0a16c7edeec1d0a10848fb01746f6d4f1db) )
	sc_gcclb_others
ROM_END

ROM_START( sc4gcclba )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406702.lo", 0x00001, 0x080000, CRC(556ff86c) SHA1(1c3c46e57f9813ce08772f9b613ae0646bc1c6ea) )
	ROM_LOAD16_BYTE( "95406703.hi", 0x00000, 0x080000, CRC(0e69bda3) SHA1(3cf4d4b8588f41bc1c0754683fec4aed5da2d18a) )
	sc_gcclb_others
ROM_END

ROM_START( sc4gcclbb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406746.lo", 0x00001, 0x080000, CRC(775b97cc) SHA1(b6d6de9f2fbc9b0325c7cd89d2d8c6a20e40f827) )
	ROM_LOAD16_BYTE( "95406747.hi", 0x00000, 0x080000, CRC(e41d5f3a) SHA1(9871c02b2338999eb16ac99c09d20a6f1e64e68b) )
	sc_gcclb_others
ROM_END

ROM_START( sc4gcclbc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406974.lo", 0x00001, 0x080000, CRC(a424b04c) SHA1(8cdd2799964c02e2dbe7f84da8bf111fe1b7f11d) )
	ROM_LOAD16_BYTE( "95406975.hi", 0x00000, 0x080000, CRC(585f9415) SHA1(172380f0bbc650560e626d80fdc34e1c369fb6e6) )
	sc_gcclb_others
ROM_END

ROM_START( sc4gcclbd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406976.lo", 0x00001, 0x080000, CRC(f975389d) SHA1(9d76cac21dd791724c9ed477d341909ad488f80d) )
	ROM_LOAD16_BYTE( "95406977.hi", 0x00000, 0x080000, CRC(94e0a013) SHA1(01712a044ed1c4638092d6e6e57d9fb2557aec0b) )
	sc_gcclb_others
ROM_END

ROM_START( sc4gcclbe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406980.lo", 0x00001, 0x080000, CRC(fb0421c5) SHA1(c0493d68bcebfc5dfb4e6d3224b7ce6c5d8da909) )
	ROM_LOAD16_BYTE( "95406981.hi", 0x00000, 0x080000, CRC(33a244b9) SHA1(6446d2f01c9edfd785f87ac6cc3d9c2e6e3c2b10) )
	sc_gcclb_others
ROM_END

ROM_START( sc4gcclbf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407700.lo", 0x00001, 0x080000, CRC(2001695e) SHA1(d8a1d6aa77ffd51c150ac2dc302c59ece392ee95) )
	ROM_LOAD16_BYTE( "95407701.hi", 0x00000, 0x080000, CRC(7c7cd268) SHA1(09a9be12e7048401e84a5166dfca584eb4ce7b6e) )
	sc_gcclb_others
ROM_END

ROM_START( sc4gcclbg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407702.lo", 0x00001, 0x080000, CRC(29ee1cdc) SHA1(cadb16843f830649aed124faf5f92792b4489449) )
	ROM_LOAD16_BYTE( "95407703.hi", 0x00000, 0x080000, CRC(d1e1aea0) SHA1(3dfb3f74a8551e1fbbd231347e0194ce527a8433) )
	sc_gcclb_others
ROM_END

ROM_START( sc4gcclbh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407746.lo", 0x00001, 0x080000, CRC(e618e6e7) SHA1(49a9923bc514d46da1428c0ba6e6c9df106ebf1f) )
	ROM_LOAD16_BYTE( "95407747.hi", 0x00000, 0x080000, CRC(f12daf09) SHA1(273649df6f55e16e95601de6e4d995f64ee70dd7) )
	sc_gcclb_others
ROM_END

ROM_START( sc4gcclbi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407974.lo", 0x00001, 0x080000, CRC(bb8b187d) SHA1(a0028ed0383fd7aec72cfcde0de73158f47459e7) )
	ROM_LOAD16_BYTE( "95407975.hi", 0x00000, 0x080000, CRC(13ac9693) SHA1(a9e35791c5c51904f0ac4c2ed073544e940b825e) )
	sc_gcclb_others
ROM_END

ROM_START( sc4gcclbj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407976.lo", 0x00001, 0x080000, CRC(21721e51) SHA1(de81051ad8759455c30e6295e7ef4543d19a1a85) )
	ROM_LOAD16_BYTE( "95407977.hi", 0x00000, 0x080000, CRC(e7eb90e6) SHA1(8ffde2059e41167234605a30c73462386948025f) )
	sc_gcclb_others
ROM_END

ROM_START( sc4gcclbk )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407980.lo", 0x00001, 0x080000, CRC(561ca30c) SHA1(18321f96b68edec5f5a4246af0314758626bbf86) )
	ROM_LOAD16_BYTE( "95407981.hi", 0x00000, 0x080000, CRC(c651f342) SHA1(0fb82c53f236d4d132726d1e4373c48c1dae0712) )
	sc_gcclb_others
ROM_END

ROM_START( sc4gcclbl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410756.lo", 0x00001, 0x080000, CRC(796eb631) SHA1(13d1ee86bd0a05474fa14e4698190f4650bfdd7f) )
	ROM_LOAD16_BYTE( "95410757.hi", 0x00000, 0x080000, CRC(17fab5ef) SHA1(5e99358216176f16a62e6c6922a4a7e017ecea3b) )
	sc_gcclb_others
ROM_END

ROM_START( sc4gcclbm )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411756.lo", 0x00001, 0x080000, CRC(9f43dba9) SHA1(116424e0f4e6ba095805d54b84658fc4dad7ee8c) )
	ROM_LOAD16_BYTE( "95411757.hi", 0x00000, 0x080000, CRC(5b990168) SHA1(0d0c9381444138f831a1c3bacc5e95e66f7acc95) )
	sc_gcclb_others
ROM_END

ROM_START( sc4gcclbn )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412166.lo", 0x00001, 0x080000, CRC(f7b4806d) SHA1(906ce4a0acfa98618dae6853abe0442f25364f56) )
	ROM_LOAD16_BYTE( "95412167.hi", 0x00000, 0x080000, CRC(a306f5d0) SHA1(942ccba0e9506ff29f7ff36f173d473a7963a6e2) )
	sc_gcclb_others
ROM_END

ROM_START( sc4gcclbo )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412168.lo", 0x00001, 0x080000, CRC(37dfdeb2) SHA1(7cd4cdb39cf62a7ceb14cc57bd15b8534f386585) )
	ROM_LOAD16_BYTE( "95412169.hi", 0x00000, 0x080000, CRC(cdc31346) SHA1(2a066b346e1945f05bf5ca969aceeacc47f6903b) )
	sc_gcclb_others
ROM_END

ROM_START( sc4gcclbp )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413166.lo", 0x00001, 0x080000, CRC(b5c8b0d3) SHA1(3ed87adecdbfe1305455dc8fbf283b10a73c0df7) )
	ROM_LOAD16_BYTE( "95413167.hi", 0x00000, 0x080000, CRC(9c1d98b6) SHA1(5e62de5f8239c9fbe128976727a13c41f0eff3c9) )
	sc_gcclb_others
ROM_END

ROM_START( sc4gcclbq )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413168.lo", 0x00001, 0x080000, CRC(040346b8) SHA1(23448739b850a1a6b5f1126bf89574624411d95f) )
	ROM_LOAD16_BYTE( "95413169.hi", 0x00000, 0x080000, CRC(326bdfa7) SHA1(4d2eed69bf2d363137282d35d03f32f5968dea55) )
	sc_gcclb_others
ROM_END




ROM_START( sc4onup )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406964.lo", 0x00001, 0x080000, CRC(cca9a87b) SHA1(e761060a55b65dbc4a4b46ccac60c5727f15a03d) )
	ROM_LOAD16_BYTE( "95406965.hi", 0x00000, 0x080000, CRC(644f49a7) SHA1(744f0d217d4d5cb221b8414c8292042595f01b88) )
	sc_onup_others
ROM_END

ROM_START( sc4onupa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407964.lo", 0x00001, 0x080000, CRC(c40c37b9) SHA1(4f41062ee1c487ecd0c5e8b2d26b814e3cf860c7) )
	ROM_LOAD16_BYTE( "95407965.hi", 0x00000, 0x080000, CRC(90a9802d) SHA1(05090cadc227904e159537ba9792956e2799a30b) )
	sc_onup_others
ROM_END




ROM_START( sc4pglcl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412692.lo", 0x00001, 0x080000, CRC(fdc94cd2) SHA1(163d6e10f96dd8f4d0788903bde8dc67525e21d1) )
	ROM_LOAD16_BYTE( "95412693.hi", 0x00000, 0x080000, CRC(a12e1001) SHA1(13c80763ec13152e1176f7140b8e580a2eed99e9) )
	sc_phgcl_others
ROM_END

ROM_START( sc4pglcla )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426522.lo", 0x00001, 0x080000, CRC(b282e34e) SHA1(cb7b8193596514f7d22a1aa69e0d476208269b82) )
	ROM_LOAD16_BYTE( "95426523.hi", 0x00000, 0x080000, CRC(66d53278) SHA1(77e3b2939fcded78f003389d831b375914257934) )
	sc_phgcl_others
ROM_END

ROM_START( sc4pglclb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427522.lo", 0x00001, 0x080000, CRC(b2b6c8ca) SHA1(318bc3f9f0ff73d193d188fddbb0f70856bb2878) )
	ROM_LOAD16_BYTE( "95427523.hi", 0x00000, 0x080000, CRC(a40ae63f) SHA1(00eda3771154648aaa757f88fd7b1188034e1fa6) )
	sc_phgcl_others
ROM_END


ROM_START( sc4pglclc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412964.lo", 0x00001, 0x080000, CRC(2251829c) SHA1(9daf30ddc112830413d8bbff36a291b6aeb8f72a) )
	ROM_LOAD16_BYTE( "95412965.hi", 0x00000, 0x080000, CRC(18e529c0) SHA1(62484b51db99e74f6d8d6d9b985060d4eabf91ec) )
	sc_phgcl_others
ROM_END

ROM_START( sc4pglcld )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412966.lo", 0x00001, 0x080000, CRC(50ef0337) SHA1(4f4873ec66e47d2060aef8a09df594ea4500e803) )
	ROM_LOAD16_BYTE( "95412967.hi", 0x00000, 0x080000, CRC(620e019e) SHA1(4f5c8359f64c3cf83fc34e6469416941415d1e3b) )
	sc_phgcl_others
ROM_END

ROM_START( sc4pglcle )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413964.lo", 0x00001, 0x080000, CRC(53831eb9) SHA1(3aeaf4025dc45d35740f058f58bb838d7299b317) )
	ROM_LOAD16_BYTE( "95413965.hi", 0x00000, 0x080000, CRC(512227b5) SHA1(a12339903414e3e5953b9e7b4dd6ccec8201b250) )
	sc_phgcl_others
ROM_END

ROM_START( sc4pglclf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413966.lo", 0x00001, 0x080000, CRC(65685e8b) SHA1(5c40712e732e8d4d3604fbdc301b2db71672d4ce) )
	ROM_LOAD16_BYTE( "95413967.hi", 0x00000, 0x080000, CRC(836bf5ca) SHA1(685b6a80012b54816ede5529f45e65c4f19d302f) )
	sc_phgcl_others
ROM_END

ROM_START( sc4pglclg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95418058.lo", 0x00001, 0x080000, CRC(1e38ff3d) SHA1(063d0c400b76315bc188b017fd425a1cb505aaa9) )
	ROM_LOAD16_BYTE( "95418059.hi", 0x00000, 0x080000, CRC(53500791) SHA1(00d2b6a0e3a249fb757cde5f88d65db33728cccd) )
	sc_phgcl_others
ROM_END

ROM_START( sc4pglclh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95419058.lo", 0x00001, 0x080000, CRC(2762e348) SHA1(b53c95475d56af25b80cadf780df41b3df648920) )
	ROM_LOAD16_BYTE( "95419059.hi", 0x00000, 0x080000, CRC(6c0faf2d) SHA1(5f3a2ba0422fc7f76577994f0c965f2461511ae2) )
	sc_phgcl_others
ROM_END







ROM_START( sc4po8 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410072.odd", 0x00001, 0x080000, CRC(8074a508) SHA1(c369bfefbc402baac250e3fa92344e90dc39b5f1) )
	ROM_LOAD16_BYTE( "95410073.evn", 0x00000, 0x080000, CRC(c7e741b6) SHA1(6fd91ab8178060a35cd8a03ed432f897016c3874) )
	sc_po8_others
ROM_END

ROM_START( sc4po8a )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410630.odd", 0x00001, 0x080000, CRC(4bba9faa) SHA1(f00b03c83c109ba9a51c6df097678ab5e6d1893f) )
	ROM_LOAD16_BYTE( "95410631.evn", 0x00000, 0x080000, CRC(67ea26f9) SHA1(ee431d26addb4295194d2c64fe41ea932efba7b4) )
	sc_po8_others
ROM_END

ROM_START( sc4po8b )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410632.odd", 0x00001, 0x080000, CRC(ea53cd6c) SHA1(c4d90fbaccf0b99b293a789a3c36c5ce30a383a4) )
	ROM_LOAD16_BYTE( "95410633.evn", 0x00000, 0x080000, CRC(2f3b47cc) SHA1(72a6315d96f78eeb8d275b94698d0a6a6d6dda25) )
	sc_po8_others
ROM_END

ROM_START( sc4po8c )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410780.odd", 0x00001, 0x080000, CRC(71c1e1d8) SHA1(bf6cf3a0558444c0d6b589a829b41897187b3212) )
	ROM_LOAD16_BYTE( "95410781.evn", 0x00000, 0x080000, CRC(d65ab66f) SHA1(9c8433cda1321707fad421ef832df4a71b49ea72) )
	sc_po8_others
ROM_END

ROM_START( sc4po8d )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410782.odd", 0x00001, 0x080000, CRC(5131bf33) SHA1(f5f4ba29468c8acf3461a56a83146215933ad803) )
	ROM_LOAD16_BYTE( "95410783.evn", 0x00000, 0x080000, CRC(32ff0966) SHA1(cf8bd00c7723e995be4c131c79adbb365c7f75e0) )
	sc_po8_others
ROM_END

ROM_START( sc4po8e )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411072.odd", 0x00001, 0x080000, CRC(680a2308) SHA1(d00958222ef7f6194741ebf65ba1c3024380a1fb) )
	ROM_LOAD16_BYTE( "95411073.evn", 0x00000, 0x080000, CRC(23cbe503) SHA1(4533ab7a043d8210215b442fd72c122d23797737) )
	sc_po8_others
ROM_END

ROM_START( sc4po8f )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411074.odd", 0x00001, 0x080000, CRC(d9789364) SHA1(1454819151cd13afb589607c44f521cb9987dead) )
	ROM_LOAD16_BYTE( "95411075.evn", 0x00000, 0x080000, CRC(1f761219) SHA1(3d54e31e3be19062d7837f7342347e48b8033410) )
	sc_po8_others
ROM_END

ROM_START( sc4po8g )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411630.odd", 0x00001, 0x080000, CRC(3fa173cf) SHA1(ee19dd50bad4b482a20bd17d9e8ba7e45b3ae550) )
	ROM_LOAD16_BYTE( "95411631.evn", 0x00000, 0x080000, CRC(e3dc69b5) SHA1(b91e5569ac83bb0e7c8ba9ad4b6b6b4530413f2a) )
	sc_po8_others
ROM_END

ROM_START( sc4po8h )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411632.odd", 0x00001, 0x080000, CRC(9e482109) SHA1(3dda3167f48848bb497b0c218913af16f500dc72) )
	ROM_LOAD16_BYTE( "95411633.evn", 0x00000, 0x080000, CRC(ab0d0880) SHA1(b5de6dc648e5b349ba6f0aa93bf54840ae74f334) )
	sc_po8_others
ROM_END

ROM_START( sc4po8i )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412078.odd", 0x00001, 0x080000, CRC(e36d9e85) SHA1(a33bb8ec8e82e4b25e2716c6b420639d38e58497) )
	ROM_LOAD16_BYTE( "95412079.evn", 0x00000, 0x080000, CRC(0e20cadd) SHA1(885cf084b28d6a341efa568f008c7f5223e2206c) )
	sc_po8_others
ROM_END

ROM_START( sc4po8j )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412080.odd", 0x00001, 0x080000, CRC(02185e38) SHA1(011968d6689d445cc70d63e1c8e5643f2639629e) )
	ROM_LOAD16_BYTE( "95412081.evn", 0x00000, 0x080000, CRC(3463680b) SHA1(b75dd8dfe04299c375e8387cf53e7f10e175900f) )
	sc_po8_others
ROM_END

ROM_START( sc4po8k )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413078.odd", 0x00001, 0x080000, CRC(e30c37b4) SHA1(41a94ca95b297d2d52db4f6c8ef2573e311a7d29) )
	ROM_LOAD16_BYTE( "95413079.evn", 0x00000, 0x080000, CRC(1568a9cf) SHA1(2b76f8e55fa0305895683516f13099116aff6644) )
	sc_po8_others
ROM_END

ROM_START( sc4po8l )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413080.odd", 0x00001, 0x080000, CRC(6069cfe3) SHA1(a5cb4b520a6a24cfdf6bf3eed8ddee043789bf17) )
	ROM_LOAD16_BYTE( "95413081.evn", 0x00000, 0x080000, CRC(b0552747) SHA1(8da04c01ea3129cbda9cf651029e54379d981303) )
	sc_po8_others
ROM_END

ROM_START( sc4po8m )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411780.odd", 0x00001, 0x080000, CRC(ceaecb27) SHA1(4bcc731b0f319d9b7e92743d57f43433bc7b183b) )
	ROM_LOAD16_BYTE( "95411781.evn", 0x00000, 0x080000, CRC(45be07c8) SHA1(e9a18125e577f0d42833140ed181c2308a66d370) )
	sc_po8_others
ROM_END

ROM_START( sc4po8n )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411782.odd", 0x00001, 0x080000, CRC(b5fc0aa3) SHA1(3db5658e19b5fba27c7c1218726e1ee27efcb65f) )
	ROM_LOAD16_BYTE( "95411783.evn", 0x00000, 0x080000, CRC(3d26600f) SHA1(3f7aab34d39bf6383b7d2423e6e6d3edd7b7cc0e) )
	sc_po8_others
ROM_END


ROM_START( sc4wag )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402516.lo", 0x00001, 0x080000, CRC(afae6512) SHA1(f07a6a21492167488117980828d4885de43a2d8f) )
	ROM_LOAD16_BYTE( "95402517.hi", 0x00000, 0x080000, CRC(b69e7146) SHA1(f84754625d9fb9bd7633105de88faf0f9cbbb6b2) )
	sc_wag_others
ROM_END

/*************************************************************************************************************************************************************************/
/*************************************************************************************************************************************************************************/
/*************************************************************************************************************************************************************************/
/*************************************************************************************************************************************************************************/
/* sets below had a mix of SC4/SC5/AD5 roms inside.  They've been split by platform for now */
/*************************************************************************************************************************************************************************/
/*************************************************************************************************************************************************************************/
/*************************************************************************************************************************************************************************/
/*************************************************************************************************************************************************************************/
/*************************************************************************************************************************************************************************/




ROM_START( sc4adjb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95422202.lo", 0x00001, 0x080000, CRC(b46dacd1) SHA1(35765ee76a9f513cf7c2b93310d58a4b5ae2d11a) )
	ROM_LOAD16_BYTE( "95422203.hi", 0x00000, 0x080000, CRC(aa15a168) SHA1(2cb949a1e8a607291682c6c5896e3a21103db1be) )
	sc_adjb_others
ROM_END

ROM_START( sc4adjba )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95422344.lo", 0x00001, 0x080000, CRC(cd8f64f5) SHA1(ac8146c6e003bf0d2d709227a8c071b9261e8768) )
	ROM_LOAD16_BYTE( "95422345.hi", 0x00000, 0x080000, CRC(9931c998) SHA1(792aa5c7aed1ec51a0e32d81ebb0d697f8f2c213) )
	sc_adjb_others
ROM_END

ROM_START( sc4adjbb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95422738.lo", 0x00001, 0x080000, CRC(6e756b04) SHA1(8895d25f914c8058fc1d27e8c27ce30c4eecab92) )
	ROM_LOAD16_BYTE( "95422739.hi", 0x00000, 0x080000, CRC(711891ea) SHA1(18035de1d74b9e4b8dece0d6cd7d42b2fd86869d) )
	sc_adjb_others
ROM_END

ROM_START( sc4adjbc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95423202.lo", 0x00001, 0x080000, CRC(ea5118ac) SHA1(e75f547f2dad9a24f4ca4418b0462e1e62c34707) )
	ROM_LOAD16_BYTE( "95423203.hi", 0x00000, 0x080000, CRC(7cae64e7) SHA1(7b9c108a3ca98fd530b5b512f42abf2308cd1929) )
	sc_adjb_others
ROM_END

ROM_START( sc4adjbd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95423344.lo", 0x00001, 0x080000, CRC(f4f471d2) SHA1(ec9b22538a3b260fbf698d8bca7bb04f49bf7743) )
	ROM_LOAD16_BYTE( "95423345.hi", 0x00000, 0x080000, CRC(d862d1c7) SHA1(2e558422b96cd04bcb7c30be88a3cd1f10fcfc40) )
	sc_adjb_others
ROM_END

ROM_START( sc4adjbe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95423738.lo", 0x00001, 0x080000, CRC(7889520c) SHA1(a3c6a203144a9c95adf5ac9efa974b55b3b2b9a8) )
	ROM_LOAD16_BYTE( "95423739.hi", 0x00000, 0x080000, CRC(4945201a) SHA1(540f0987dbb3d3d3a44baf3127af0b780982d2a8) )
	sc_adjb_others
ROM_END

ROM_START( sc4adjbf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426108.lo", 0x00001, 0x080000, CRC(d5f4a4a8) SHA1(dca003286b6d4d933125aeb6664ad5b66124a914) )
	ROM_LOAD16_BYTE( "95426109.hi", 0x00000, 0x080000, CRC(6ff8b169) SHA1(62fa8ca4172d72cbdf7c59634faba49682ecf675) )
	sc_adjb_others
ROM_END

ROM_START( sc4adjbg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426330.lo", 0x00001, 0x080000, CRC(1ee5674e) SHA1(07ea9fbf3d1563df3680cb47db12cdcd08d1ec4c) )
	ROM_LOAD16_BYTE( "95426331.hi", 0x00000, 0x080000, CRC(1aa9591c) SHA1(461c034b9b8709357f2aa396231141a94fbb3c6b) )
	sc_adjb_others
ROM_END

ROM_START( sc4adjbh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427108.lo", 0x00001, 0x080000, CRC(7596b006) SHA1(52374ce812f404e0d3b852a6497e312e3c34fd53) )
	ROM_LOAD16_BYTE( "95427109.hi", 0x00000, 0x080000, CRC(8de91bb0) SHA1(14ab21c5f0852266cb9989186f1580a8f5f70750) )
	sc_adjb_others
ROM_END

ROM_START( sc4adjbi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427330.lo", 0x00001, 0x080000, CRC(42c6d7d4) SHA1(b7d07d97f89cddb571c50f1da0fe3bdecb3e2c8d) )
	ROM_LOAD16_BYTE( "95427331.hi", 0x00000, 0x080000, CRC(dc2ba1f7) SHA1(ac98205a5820d01257aa3470ef0439b5cfb243cc) )
	sc_adjb_others
ROM_END



ROM_START( sc4adwta )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1940p44.hi", 0x00000, 0x080000, CRC(a90332f1) SHA1(67651afefbf94468b254e5d7bd6ed6b3131032db) )
	ROM_LOAD16_BYTE( "p1940p44.lo", 0x00001, 0x080000, CRC(924ac406) SHA1(7aa4d41b3b1ff38ac550e351db198a81356a9a37) )
	sc_adwta_others
ROM_END

ROM_START( sc4adwtaa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1940s44.hi", 0x00000, 0x080000, CRC(88961ef0) SHA1(e21ba766037a4bc4c7fd14199f7704ed1bfa22a5) )
	ROM_LOAD16_BYTE( "p1940s44.lo", 0x00001, 0x080000, CRC(627fe077) SHA1(e4da21e50501d0f35a392d1e7dbc26057667eacc) )
	sc_adwta_others
ROM_END





ROM_START( sc4a40 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416088.lo", 0x00001, 0x080000, CRC(7bebb61c) SHA1(b1228eb50a66f19c7496306d2fa2d72c768d8b91) )
	ROM_LOAD16_BYTE( "95416089.hi", 0x00000, 0x080000, CRC(8a5bf808) SHA1(02336466dbf1ad7155ffc16111037333d11e09b9) )
	sc_a40_others
ROM_END

ROM_START( sc4a40a )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416090.lo", 0x00001, 0x080000, CRC(23ebede2) SHA1(9b3c3f6fcde88f595d60b986d5f7f551ac6b4dc5) )
	ROM_LOAD16_BYTE( "95416091.hi", 0x00000, 0x080000, CRC(4cc5160d) SHA1(95aa16ec0b36676a17af89b99b47f0d1265e07b5) )
	sc_a40_others
ROM_END

ROM_START( sc4a40b )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417088.lo", 0x00001, 0x080000, CRC(1f4b5227) SHA1(d58751b06a3f69911d3c2275484bdb7f14dfc2ed) )
	ROM_LOAD16_BYTE( "95417089.hi", 0x00000, 0x080000, CRC(bd6ac464) SHA1(811a99bc9b308d5d80dd001e362112bcd13df899) )
	sc_a40_others
ROM_END

ROM_START( sc4a40c )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417090.lo", 0x00001, 0x080000, CRC(6a9853be) SHA1(dcf973cfab45656ca8dc597e385e1c3c2eb4ca9a) )
	ROM_LOAD16_BYTE( "95417091.hi", 0x00000, 0x080000, CRC(4c5a4035) SHA1(363072def1061500102acc0806cad70cbab3be37) )
	sc_a40_others
ROM_END



ROM_START( sc4bantm )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416820.lo", 0x00001, 0x080000, CRC(813d26c0) SHA1(7d6f6874a88ac9c970f07c9bd90132095b2b8bc0) )
	ROM_LOAD16_BYTE( "95416821.hi", 0x00000, 0x080000, CRC(20d748e1) SHA1(61624c5d5e12b8c054e8b2ba723b6e69531826df) )
	sc_bantm_others
ROM_END

ROM_START( sc4bantma )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416822.lo", 0x00001, 0x080000, CRC(070976b0) SHA1(a9a13c58e75e08be87de798d3efb32a8659abc04) )
	ROM_LOAD16_BYTE( "95416823.hi", 0x00000, 0x080000, CRC(4ec8d8bc) SHA1(581c4f6a80bd9cef4293fcbcf3768a9cef4a1858) )
	sc_bantm_others
ROM_END

ROM_START( sc4bantmb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417820.lo", 0x00001, 0x080000, CRC(15996b8f) SHA1(3240b1816c19bc2f55f6e96d4055e730308ad4f0) )
	ROM_LOAD16_BYTE( "95417821.hi", 0x00000, 0x080000, CRC(308d8f89) SHA1(fc4ffe2e00323af9413379df7e8a85525f0ca8fd) )
	sc_bantm_others
ROM_END

ROM_START( sc4bantmc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417822.lo", 0x00001, 0x080000, CRC(b042b55b) SHA1(bad809c3925953dbd9d2e62bc11a5b0d921e9b91) )
	ROM_LOAD16_BYTE( "95417823.hi", 0x00000, 0x080000, CRC(adf313b2) SHA1(acaee4877f025fbbe962110dae12bcd7a0db4785) )
	sc_bantm_others
ROM_END





ROM_START( sc4bar7 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410306.lo", 0x00001, 0x080000, CRC(9b7699a8) SHA1(3f2a6548823f0d9d0019e449806f510df7e73bd9) )
	ROM_LOAD16_BYTE( "95410307.hi", 0x00000, 0x080000, CRC(8082f609) SHA1(e4cc191a668db701ce258d95db771a1372136be6) )
	sc_bar7_others
ROM_END

ROM_START( sc4bar7a )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411306.lo", 0x00001, 0x080000, CRC(4d1d4253) SHA1(7c8adadb579b75bda331d1326f298942d971e1f5) )
	ROM_LOAD16_BYTE( "95411307.hi", 0x00000, 0x080000, CRC(a2833a56) SHA1(09cf14966cbfad627227ce6e2fdaabe1521a6aac) )
	sc_bar7_others
ROM_END

ROM_START( sc4bar7b )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1438p41.hi", 0x00000, 0x080000, CRC(acaa3793) SHA1(a07ddd23f889365654a957747984f5c27bd78bdb) )
	ROM_LOAD16_BYTE( "p1438p41.lo", 0x00001, 0x080000, CRC(ec471c4d) SHA1(441f29eab8eab37c315ba323b3bd5217285f9a91) )
	PR1436_TRIPLE_CASINO_SOUNDS11
ROM_END

ROM_START( sc4bar7c )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1438p51.hi", 0x00000, 0x080000, CRC(46b074fc) SHA1(80860af1dc5ddf34c08a50a50f8bf2fb4038494f) )
	ROM_LOAD16_BYTE( "p1438p51.lo", 0x00001, 0x080000, CRC(e922d2e9) SHA1(2047def0e3ed8baef378c7d7e8ab831667c24ce4) )
	PR1436_TRIPLE_CASINO_SOUNDS11
ROM_END

ROM_START( sc4bar7d )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1438s41.hi", 0x00000, 0x080000, CRC(ae8b7bdd) SHA1(4c705b6849a2fc39e8a84d8ed367366839a99d29) )
	ROM_LOAD16_BYTE( "p1438s41.lo", 0x00001, 0x080000, CRC(73ccf662) SHA1(e6ede55d18f72a1bfc36cc9ae17e9bee17bb3623) )
	PR1436_TRIPLE_CASINO_SOUNDS11
ROM_END

ROM_START( sc4bar7e )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1438s51.hi", 0x00000, 0x080000, CRC(449138b2) SHA1(e538d848ed2558c730f419eeb5cac203ec75c146) )
	ROM_LOAD16_BYTE( "p1438s51.lo", 0x00001, 0x080000, CRC(76a938c6) SHA1(d83e3bd3690b8f7edfcb4f46cc19300a4c6620c1) )
	PR1436_TRIPLE_CASINO_SOUNDS11
ROM_END



ROM_START( sc4batl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95418014.lo", 0x00001, 0x080000, CRC(11fc27fe) SHA1(d1c59918e2ee1a7a40879f3a369d7064360f8e7b) )
	ROM_LOAD16_BYTE( "95418015.hi", 0x00000, 0x080000, CRC(9731462c) SHA1(5324f60ef10089030ed57d1d50127a3a0af2bfd5) )
	sc_batl_others
ROM_END


ROM_START( sc4batla )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95419014.lo", 0x00001, 0x080000, CRC(1e17e7fe) SHA1(c8da03a27ed16c53aeb7637265123ca4474f7b8e) )
	ROM_LOAD16_BYTE( "95419015.hi", 0x00000, 0x080000, CRC(05eb5bfb) SHA1(2f5500c48497671c2aeaee02213417cb58b64666) )
	sc_batl_others
ROM_END



ROM_START( sc4bob )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "94416378.lo", 0x00001, 0x080000, CRC(b0c64d0f) SHA1(3f9e9ca908ef43a293a47e15809246cbd3f6cbcf) )
	ROM_LOAD16_BYTE( "94416379.hi", 0x00000, 0x080000, CRC(42468e3e) SHA1(acc5edd9dba26ca525c632dd3a0c991b7af53075) )
	sc_bob_others
ROM_END

ROM_START( sc4boba )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416380.lo", 0x00001, 0x080000, CRC(985426aa) SHA1(c4e67132e9ad8f6793482615432247190487c989) )
	ROM_LOAD16_BYTE( "95416381.hi", 0x00000, 0x080000, CRC(628b66b1) SHA1(81872397e5225a42a6450a6563c765f5e745346d) )
	sc_bob_others
ROM_END

ROM_START( sc4bobb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416660.lo", 0x00001, 0x080000, CRC(50509f00) SHA1(de131ec8a346974ce8ee20863ad4caf27fe80cf1) )
	ROM_LOAD16_BYTE( "95416661.hi", 0x00000, 0x080000, CRC(7450faed) SHA1(8cefb6eb6d88d48b186379a2e3538fbf32642630) )
	sc_bob_others
ROM_END

ROM_START( sc4bobc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417378.lo", 0x00001, 0x080000, CRC(2db95b6e) SHA1(7f757d93350582d5280544b3c1a39226bbb61b59) )
	ROM_LOAD16_BYTE( "95417379.hi", 0x00000, 0x080000, CRC(fc981e2e) SHA1(4839f15f3fafe03b478ccfb468c6679ac0b0fb00) )
	sc_bob_others
ROM_END

ROM_START( sc4bobd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417380.lo", 0x00001, 0x080000, CRC(9ce4e63a) SHA1(39a27f693ea72bd2cb8841c36211cef57c84e227) )
	ROM_LOAD16_BYTE( "95417381.hi", 0x00000, 0x080000, CRC(a4395863) SHA1(b1b0b816c443fe41d9a25073e32c248ad4d49210) )
	sc_bob_others
ROM_END

ROM_START( sc4bobe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417660.lo", 0x00001, 0x080000, CRC(aeb206a8) SHA1(2747f2f2f062c8f62f7f8087df1c0c84d9ba8e79) )
	ROM_LOAD16_BYTE( "95417661.hi", 0x00000, 0x080000, CRC(796b9d5c) SHA1(a208a0a048e428ffafc2dc0997fde4e83e6c17e1) )
	sc_bob_others
ROM_END

ROM_START( sc4bobf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420296.lo", 0x00001, 0x080000, CRC(a5c1910a) SHA1(11f965d47ddc590cf50b4b18c8cc70eeac5c94dd) )
	ROM_LOAD16_BYTE( "95420297.hi", 0x00000, 0x080000, CRC(408a0f12) SHA1(e1323f4e29be14cf9ddf4b4b2643df7f775cb737) )
	sc_bob_others
ROM_END

ROM_START( sc4bobg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420298.lo", 0x00001, 0x080000, CRC(1f273842) SHA1(a832ff0816aa2646a1421a784bcc79e73f277c1c) )
	ROM_LOAD16_BYTE( "95420299.hi", 0x00000, 0x080000, CRC(41622e25) SHA1(e09b1db7d14a1226b5e66ec22781b57dd28eba9d) )
	sc_bob_others
ROM_END

ROM_START( sc4bobh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421296.lo", 0x00001, 0x080000, CRC(65ed3bcc) SHA1(0770c3d75b701c8d643bedbb334f26453b1a59b6) )
	ROM_LOAD16_BYTE( "95421297.hi", 0x00000, 0x080000, CRC(e7755b24) SHA1(0e153b0dfe1b3bad06c95ee6e9b344ffd1d3f93f) )
	sc_bob_others
ROM_END

ROM_START( sc4bobi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421298.lo", 0x00001, 0x080000, CRC(bd4c64ad) SHA1(56e0a4ffdf83378fdc1e4a02f37ba83030ebb92b) )
	ROM_LOAD16_BYTE( "95421299.hi", 0x00000, 0x080000, CRC(3e755f3e) SHA1(a7dc615e3661c57945f1dbd7fe22669b4048f97f) )
	sc_bob_others
ROM_END


ROM_START( sc4bull )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412934.lo", 0x00001, 0x080000, CRC(769ff771) SHA1(ad3df6d5887349de8e273fd8fcc1889dc9231c6b) )
	ROM_LOAD16_BYTE( "95412935.hi", 0x00000, 0x080000, CRC(3f811a12) SHA1(5fb6e0e2182374fee69839c8d1d046546eb98233) )
	sc_bull_others
ROM_END

ROM_START( sc4bulla )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413934.lo", 0x00001, 0x080000, CRC(bab4d554) SHA1(f020ed79df0fcb8f788045cdb77ddf2b964e1553) )
	ROM_LOAD16_BYTE( "95413935.hi", 0x00000, 0x080000, CRC(c81fb05c) SHA1(992e385553ad755f6ef8df04c9e4524340b870bd) )
	sc_bull_others
ROM_END

ROM_START( sc4bulcs )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95418146.lo", 0x00001, 0x080000, CRC(5f9108e0) SHA1(a62bebdc729f514476d77893eb5d8f5ed8384abb) )
	ROM_LOAD16_BYTE( "95418147.hi", 0x00000, 0x080000, CRC(f782fd6c) SHA1(3602d451e32a00c523453e7e595b4e34214aa55d) )
	sc_bull_others
ROM_END

ROM_START( sc4bullb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95418744.lo", 0x00001, 0x080000, CRC(9699eab6) SHA1(aa7ddaf87cacf0dab0895135bc1ec05afd5ea41f) )
	ROM_LOAD16_BYTE( "95418745.hi", 0x00000, 0x080000, CRC(0c628f17) SHA1(f76df3f7f1ec1163f6850ba086eff4e2b97ed7e0) )
	sc_bull_others
ROM_END

ROM_START( sc4bulcsa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95419146.lo", 0x00001, 0x080000, CRC(407082d7) SHA1(55d4253f4404c579a5b3654a52484512e877cf41) )
	ROM_LOAD16_BYTE( "95419147.hi", 0x00000, 0x080000, CRC(7f5ded3f) SHA1(73eac57938d61f9ba94b673c653d4e1668e85b7d) )
	sc_bull_others
ROM_END

ROM_START( sc4bulcsb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95418830.lo", 0x00001, 0x080000, CRC(f87a11e3) SHA1(c6637f331005df9d4d5a6bd970d79c2367c8f33e) )
	ROM_LOAD16_BYTE( "95418831.hi", 0x00000, 0x080000, CRC(f00cd02c) SHA1(aad7c56f6e62ab234f13b9542da045cac5b961f1) )
	sc_bull_others
ROM_END

ROM_START( sc4bulcsc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95419830.lo", 0x00001, 0x080000, CRC(a652480f) SHA1(de2e1aadbd1d2d64018faef9cd505eb05fbccc71) )
	ROM_LOAD16_BYTE( "95419831.hi", 0x00000, 0x080000, CRC(c47e18d0) SHA1(6a2f89c2f5501d6f0c1af89ce20adc9c1a09deb1) )
	sc_bull_others
ROM_END

ROM_START( sc4bullc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95419744.lo", 0x00001, 0x080000, CRC(0ba6cdb0) SHA1(93ed4e1daf3bd3e23988051995ac53343c83bc64) )
	ROM_LOAD16_BYTE( "95419745.hi", 0x00000, 0x080000, CRC(1a118bb4) SHA1(5a1df4379a7c90f1b26013d949f86f55f579980b) )
	sc_bull_others
ROM_END



ROM_START( sc4butch )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95418332.lo", 0x00001, 0x080000, CRC(0f2fc032) SHA1(2ed1fed3ae1f31dab5b7f00f0f88b388582a4274) )
	ROM_LOAD16_BYTE( "95418333.hi", 0x00000, 0x080000, CRC(d56262e0) SHA1(3ee37fba8f2a5b74c4d2596e76d724ebb03ba007) )
	sc_butch_others
ROM_END

ROM_START( sc4butcha )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95418806.lo", 0x00001, 0x080000, CRC(537a03d8) SHA1(a2c7fc1a48b1d3ff9ee49a1dd7754bb6ddd96017) )
	ROM_LOAD16_BYTE( "95418807.hi", 0x00000, 0x080000, CRC(a98ad35b) SHA1(9c7af483afb5d6312c258a5c1c99c1f2ad1dccce) )
	sc_butch_others
ROM_END

ROM_START( sc4butchb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95418910.lo", 0x00001, 0x080000, CRC(bebc83cd) SHA1(00e4ad7e5b79da90d61005eb2eb56053b919b38a) )
	ROM_LOAD16_BYTE( "95418911.hi", 0x00000, 0x080000, CRC(3a61c8d1) SHA1(b6039a11c97986865e8cfed7acc3adbc8a0506bc) )
	sc_butch_others
ROM_END

ROM_START( sc4butchc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95418948.lo", 0x00001, 0x080000, CRC(fa68a9af) SHA1(b5a2260cf3fa3eb1cb8cdb923a0c69bebe80c893) )
	ROM_LOAD16_BYTE( "95418949.hi", 0x00000, 0x080000, CRC(cacb0b41) SHA1(7ad01a4b31f601000fed8aa18725c26daade087e) )
	sc_butch_others
ROM_END

ROM_START( sc4butchd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95419332.lo", 0x00001, 0x080000, CRC(4ae82089) SHA1(06c7a23bf8196876eed775709cc8e8654a4dd01b) )
	ROM_LOAD16_BYTE( "95419333.hi", 0x00000, 0x080000, CRC(db85a7f9) SHA1(ea08240e65809bb49898fa1464c05f2b22409c9d) )
	sc_butch_others
ROM_END

ROM_START( sc4butche )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95419806.lo", 0x00001, 0x080000, CRC(34b937e8) SHA1(05af5af7d165235798a432d6fe36ba2bc2f846bb) )
	ROM_LOAD16_BYTE( "95419807.hi", 0x00000, 0x080000, CRC(dfe33c01) SHA1(4c8d74e2c54d419911ba138f8d5ddcfeb6f9f883) )
	sc_butch_others
ROM_END

ROM_START( sc4butchf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95419910.lo", 0x00001, 0x080000, CRC(12795f7e) SHA1(2fc36d8be93bff5e62c682b5e8b391862656b1cf) )
	ROM_LOAD16_BYTE( "95419911.hi", 0x00000, 0x080000, CRC(8966bbf1) SHA1(23d323f849c8c67328af4fbdf084c7b68964b5fd) )
	sc_butch_others
ROM_END

ROM_START( sc4butchg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95419948.lo", 0x00001, 0x080000, CRC(2159f20a) SHA1(eb0354ab815656ae35495ab5cbd1e9426c9ad2e3) )
	ROM_LOAD16_BYTE( "95419949.hi", 0x00000, 0x080000, CRC(175efc3b) SHA1(cfaa96cca4ab3e627d3b064d46877ec94fef8cc4) )
	sc_butch_others
ROM_END



ROM_START( sc4cabin )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416548.lo", 0x00001, 0x080000, CRC(f4bf09ed) SHA1(f30ff3c295af7e0483ba2604d83274e5430876be) )
	ROM_LOAD16_BYTE( "95416549.hi", 0x00000, 0x080000, CRC(2b30b64b) SHA1(3d65c3333ebd56c3c39590c2c801cd45d0c74dea) )
	sc_cabin_others
ROM_END

ROM_START( sc4cabina )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416550.lo", 0x00001, 0x080000, CRC(328f9f69) SHA1(5cfe84ba80e24cbee5009a7c1a4d8a723d9985d2) )
	ROM_LOAD16_BYTE( "95416551.hi", 0x00000, 0x080000, CRC(e3ad34f0) SHA1(a4a0fb869c43bfbe9d811ffe9f4ee0833895f7fd) )
	sc_cabin_others
ROM_END

ROM_START( sc4cabinb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416864.lo", 0x00001, 0x080000, CRC(73c39f56) SHA1(5bd300d020c522a723ab55da8031f1d323c481bf) )
	ROM_LOAD16_BYTE( "95416865.hi", 0x00000, 0x080000, CRC(7bb6a60d) SHA1(6f88a09c0cf883c2d61666f8491d8a7d268d1e55) )
	sc_cabin_others
ROM_END

ROM_START( sc4cabinc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416866.lo", 0x00001, 0x080000, CRC(e83a4c5a) SHA1(4724f82f609f008166ff2457eaaa76203a29b603) )
	ROM_LOAD16_BYTE( "95416867.hi", 0x00000, 0x080000, CRC(7eced3f5) SHA1(98bb8251b4ec65a7f5fa4c0a5fb2636376fe0c67) )
	sc_cabin_others
ROM_END

ROM_START( sc4cabind )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417548.lo", 0x00001, 0x080000, CRC(2322f9e3) SHA1(f3a194a017d4e3c08af4316caf22764fb7f68ad4) )
	ROM_LOAD16_BYTE( "95417549.hi", 0x00000, 0x080000, CRC(3b811137) SHA1(59645e6f6c602d6c448576ecc3e8171d35b65cca) )
	sc_cabin_others
ROM_END

ROM_START( sc4cabine )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417550.lo", 0x00001, 0x080000, CRC(fabed74d) SHA1(3c9acd930c146078946132d012d7e713931b4464) )
	ROM_LOAD16_BYTE( "95417551.hi", 0x00000, 0x080000, CRC(4a43de8a) SHA1(2ff08674afa3341b6c6f1330038bf90c6279af80) )
	sc_cabin_others
ROM_END

ROM_START( sc4cabinf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417864.lo", 0x00001, 0x080000, CRC(f551a5c5) SHA1(c8826280430fb69090da1e6ff3f656e45502321d) )
	ROM_LOAD16_BYTE( "95417865.hi", 0x00000, 0x080000, CRC(0de1127e) SHA1(218c48e33ad828d46efe74b73001e31b0afbcbd8) )
	sc_cabin_others
ROM_END

ROM_START( sc4cabing )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417866.lo", 0x00001, 0x080000, CRC(1d287877) SHA1(7730c187ae2941472e695d2130d84e106f7826a7) )
	ROM_LOAD16_BYTE( "95417867.hi", 0x00000, 0x080000, CRC(8035b533) SHA1(ff664de67108f73d3a282079f4ce46fda7fb7b7f) )
	sc_cabin_others
ROM_END

ROM_START( sc4cabinh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420576.lo", 0x00001, 0x080000, CRC(685b132b) SHA1(15f53a9cfb69f77356a1b5251c8e8227b55a05c2) )
	ROM_LOAD16_BYTE( "95420577.hi", 0x00000, 0x080000, CRC(3f5ae984) SHA1(b17a1f911025cdb94bb8bb0bd6f7eae86d3458f9) )
	sc_cabin_others
ROM_END

ROM_START( sc4cabini )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420578.lo", 0x00001, 0x080000, CRC(858102df) SHA1(e8b98596bb93176e22352b1d87dc75f721ce9325) )
	ROM_LOAD16_BYTE( "95420579.hi", 0x00000, 0x080000, CRC(3a99188a) SHA1(80ca9140bb10896ac7443e8550ffd9436decf608) )
	sc_cabin_others
ROM_END

ROM_START( sc4cabinj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420580.lo", 0x00001, 0x080000, CRC(19a56973) SHA1(5dc3a78c90d6dc7ff138bfa54d7e7d8276a5bd8b) )
	ROM_LOAD16_BYTE( "95420581.hi", 0x00000, 0x080000, CRC(beb7da8e) SHA1(9d90bc0f3505f9529e3a8cb3087a0ddc770db592) )
	sc_cabin_others
ROM_END

ROM_START( sc4cabink )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421576.lo", 0x00001, 0x080000, CRC(d56ce8cc) SHA1(297c139d11db4b99ed8ba8e084410062ba77b26c) )
	ROM_LOAD16_BYTE( "95421577.hi", 0x00000, 0x080000, CRC(0a2c4c51) SHA1(f47bb9791ab0afbe3e5755dde719520c0b7b86e3) )
	sc_cabin_others
ROM_END

ROM_START( sc4cabinl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421578.lo", 0x00001, 0x080000, CRC(861d3dcb) SHA1(427b43333aafb8221a5201dfafafe862e5e3b640) )
	ROM_LOAD16_BYTE( "95421579.hi", 0x00000, 0x080000, CRC(43bc1f5e) SHA1(5ee7c2ac4c83bcedd4a880b7f6431e2a2a26728c) )
	sc_cabin_others
ROM_END

ROM_START( sc4cabinm )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421580.lo", 0x00001, 0x080000, CRC(9b080907) SHA1(7e3a4474302a786f28e0372444a165474c61895b) )
	ROM_LOAD16_BYTE( "95421581.hi", 0x00000, 0x080000, CRC(84e91010) SHA1(3e455ec15cdad954b8112cceaf3697f2519db9cc) )
	sc_cabin_others
ROM_END



ROM_START( sc4cari )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420612.lo", 0x00001, 0x080000, CRC(30d13717) SHA1(86ef8ce0e0d4fb304237e0904f3c031b71cf7382) )
	ROM_LOAD16_BYTE( "95420613.hi", 0x00000, 0x080000, CRC(ace67c08) SHA1(315d82bcad9eb36f19a5ee253aef564898c1a8de) )
	sc_cari_others
ROM_END

ROM_START( sc4caria )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420614.lo", 0x00001, 0x080000, CRC(c4254d45) SHA1(7a8b17712f0855fabde2fa911430e4ff35407765) )
	ROM_LOAD16_BYTE( "95420615.hi", 0x00000, 0x080000, CRC(65fdf957) SHA1(d042ec7d8d2b35bef973cf22a623b2dd5730a27b) )
	sc_cari_others
ROM_END

ROM_START( sc4carib )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420648.lo", 0x00001, 0x080000, CRC(4cae3542) SHA1(9e262fd961b6a322cc9d2942acb87cc412243bc3) )
	ROM_LOAD16_BYTE( "95420649.hi", 0x00000, 0x080000, CRC(75845134) SHA1(77b6980cf3045267b6477662aac3e4aba92f4edf) )
	sc_cari_others
ROM_END

ROM_START( sc4caric )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421612.lo", 0x00001, 0x080000, CRC(a129c428) SHA1(652f7f7689aa1d03ba3a5805fa6d6aa84c439c95) )
	ROM_LOAD16_BYTE( "95421613.hi", 0x00000, 0x080000, CRC(29418083) SHA1(d08695736dfd3bd65aa0b249a8a53c236f2298af) )
	sc_cari_others
ROM_END

ROM_START( sc4carid )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421614.lo", 0x00001, 0x080000, CRC(5aff77f4) SHA1(48fae56a98bca531bb6b4e9cfd3c0424b38c4868) )
	ROM_LOAD16_BYTE( "95421615.hi", 0x00000, 0x080000, CRC(7af7c94d) SHA1(0149d750f80556342de2c013bb6c7cfba2b3d6ef) )
	sc_cari_others
ROM_END

ROM_START( sc4carie )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421648.lo", 0x00001, 0x080000, CRC(0e294181) SHA1(eb73dc4076f4c7775848c9c4f659e452e038f35d) )
	ROM_LOAD16_BYTE( "95421649.hi", 0x00000, 0x080000, CRC(9c63b4b5) SHA1(fbc6410b9725ad7e5cb2c75373fa2aa5d898dff8) )
	sc_cari_others
ROM_END

ROM_START( sc4cariq )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "96000070.odd", 0x00001, 0x080000, CRC(65b37a3c) SHA1(560efe0fdbc28f2d0ed900be8c3172c82304dbf1) )
	ROM_LOAD16_BYTE( "96000071.evn", 0x00000, 0x080000, CRC(3c6f2408) SHA1(cf756e2f22be2e912f87231a94d653f250b5c285) )
	sc_cariq_others
ROM_END

ROM_START( sc4cariqa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "96000072.odd", 0x00001, 0x080000, CRC(61c4a366) SHA1(c07cc61c6539dec81a129270ed31f83c5c3cc4db) )
	ROM_LOAD16_BYTE( "96000073.evn", 0x00000, 0x080000, CRC(98eb005d) SHA1(784bb784666a19ee807001556c6cbdbbf3431e34) )
	sc_cariq_others
ROM_END

ROM_START( sc4cariqb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "96000074.odd", 0x00001, 0x080000, CRC(e0f23e84) SHA1(47512a50257aeaec46af524ee0ed87dc573a76e2) )
	ROM_LOAD16_BYTE( "96000075.evn", 0x00000, 0x080000, CRC(1eca734d) SHA1(dc2e28d52f8e191a25b77a87c6335271996ce609) )
	sc_cariq_others
ROM_END

ROM_START( sc4cariqc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "96000076.odd", 0x00001, 0x080000, CRC(dac49cce) SHA1(45553a34c43677070ec4487f097edaf15dbe89d1) )
	ROM_LOAD16_BYTE( "96000077.evn", 0x00000, 0x080000, CRC(e3adc07e) SHA1(f9f128ca375a84afda9be018cac0eb4e0d1a58b3) )
	sc_cariq_others
ROM_END

ROM_START( sc4cariqd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "96000078.lo", 0x00001, 0x080000, CRC(ae6ae590) SHA1(c59d88f7b1c4668dfde5ae95b099c65fdb31c4fb) )
	ROM_LOAD16_BYTE( "96000079.hi", 0x00000, 0x080000, CRC(a3f81488) SHA1(16e9913d3f55be5b18fa785d10d2c7b1b32fc8fe) )
	sc_cariq_others
ROM_END

ROM_START( sc4cariqe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "96000080.odd", 0x00001, 0x080000, CRC(7e2d1c8e) SHA1(e204fb308ff0cbd912d9516a248ab0326166b55e) )
	ROM_LOAD16_BYTE( "96000081.evn", 0x00000, 0x080000, CRC(e35aea8e) SHA1(5069f81ec070c6da3b7225e5d6b4c8ced84b0fbf) )
	sc_cariq_others
ROM_END

ROM_START( sc4cariqf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "96000082.odd", 0x00001, 0x080000, CRC(0b2a8a9a) SHA1(23b8bb2fd9acd0af3a0c10df25a2cba093b8d1f1) )
	ROM_LOAD16_BYTE( "96000083.evn", 0x00000, 0x080000, CRC(feb378a0) SHA1(c498bafe7f86f8da75b4d03aa97f00f12636fff1) )
	sc_cariq_others
ROM_END

ROM_START( sc4cariqg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "96001082.odd", 0x00001, 0x080000, CRC(8d27a472) SHA1(f3be5fc5836addd2cbdc9a491bd092b75846ce18) )
	ROM_LOAD16_BYTE( "96001083.evn", 0x00000, 0x080000, CRC(2eed14e9) SHA1(36eb7c876331cbdd968c644307183a392777aec1) )
	sc_cariq_others
ROM_END


ROM_START( sc4cblas )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95419020.lo", 0x00001, 0x080000, CRC(e508efb9) SHA1(ff96f63f777b62e0a2fe6c19a7b86c158a08dde0) )
	ROM_LOAD16_BYTE( "95419021.hi", 0x00000, 0x080000, CRC(463ee216) SHA1(5c46f6da686816d4363b57ba60ea9977546e43ed) )
	sc_cblas_others
ROM_END


ROM_START( sc4cblasa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95418020.lo", 0x00001, 0x080000, CRC(156f98d1) SHA1(7f7af9b194bd2866f82c7fcd6a19196b2bc7451c) )
	ROM_LOAD16_BYTE( "95418021.hi", 0x00000, 0x080000, CRC(b6a36c59) SHA1(4d0764c1cf3a0036379e29fee244a075a4412800) )
	ROM_REGION( 0x400000, "ymz", ROMREGION_ERASE00)
ROM_END


ROM_START( sc4casxt )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420640.lo", 0x00001, 0x080000, CRC(a3c7226a) SHA1(5c6aaae3d7da11d48172b81c5122b4db64aaa795) )
	ROM_LOAD16_BYTE( "95420641.hi", 0x00000, 0x080000, CRC(2e71cd56) SHA1(68e2e9c35d8bc6cf49cf09b571d89b57943750e3) )
	sc_casxt_others
ROM_END

ROM_START( sc4casxta )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420642.lo", 0x00001, 0x080000, CRC(299b543b) SHA1(4c4cee77ac8db935cb78d41d9589b94b54d7d2a3) )
	ROM_LOAD16_BYTE( "95420643.hi", 0x00000, 0x080000, CRC(7dace7c6) SHA1(ec4d65976bec9ceda68e46816d5f0c4b83807942) )
	sc_casxt_others
ROM_END

ROM_START( sc4casxtb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420644.lo", 0x00001, 0x080000, CRC(64a3ffc3) SHA1(bbea1a1a151730fbbe9c9fc40b6fbc1f4cc37bdc) )
	ROM_LOAD16_BYTE( "95420645.hi", 0x00000, 0x080000, CRC(27b754f3) SHA1(01d2a7db79b9252bcc3cb321ec0b40517a63f9d3) )
	sc_casxt_others
ROM_END

ROM_START( sc4casxtc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421640.lo", 0x00001, 0x080000, CRC(0ade76b5) SHA1(c79006d18783f574b237a65c6ddcf0bcf1bf034f) )
	ROM_LOAD16_BYTE( "95421641.hi", 0x00000, 0x080000, CRC(bc2a1028) SHA1(333abe3c0cc4d8b13ee87838b62a82b158148e36) )
	sc_casxt_others
ROM_END

ROM_START( sc4casxtd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421642.lo", 0x00001, 0x080000, CRC(d7a9d771) SHA1(da1311158abe1fe8afd792120a0acd9a49c8a989) )
	ROM_LOAD16_BYTE( "95421643.hi", 0x00000, 0x080000, CRC(0402f764) SHA1(9a7ae89cd774dc4b6f7ab977544dcad1238496c7) )
	sc_casxt_others
ROM_END

ROM_START( sc4casxte )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421644.lo", 0x00001, 0x080000, CRC(cdbaab1c) SHA1(b40f6c12faec96fe856320560e50a7ff2fa4b234) )
	ROM_LOAD16_BYTE( "95421645.hi", 0x00000, 0x080000, CRC(b5ec898d) SHA1(cbe7b2495e84383c6f72a8865c3b8fae81d8bab5) )
	sc_casxt_others
ROM_END




ROM_START( sc4chavi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95418088.lo", 0x00001, 0x080000, CRC(56f94ca0) SHA1(7ac6f781f15d0397580e0bd379d60953b081cd42) )
	ROM_LOAD16_BYTE( "95418089.hi", 0x00000, 0x080000, CRC(47f83d4a) SHA1(5b33ac69c81a83b0842755b9c91f2fa295e7dc81) )
	sc_chavi_others
ROM_END

ROM_START( sc4chavia )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95418826.lo", 0x00001, 0x080000, CRC(9c9ef425) SHA1(017ed1bf26ef346ee447cff6824e0416d48d91ef) )
	ROM_LOAD16_BYTE( "95418827.hi", 0x00000, 0x080000, CRC(1ff95ad0) SHA1(49b3d1a9fe012565f6e772fb3b3704f55bc5039e) )
	sc_chavi_others
ROM_END

ROM_START( sc4chavib )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95418900.lo", 0x00001, 0x080000, CRC(75dfc3eb) SHA1(5aade49ac5f857d4ece5d3cd04c2a3b7a188241a) )
	ROM_LOAD16_BYTE( "95418901.hi", 0x00000, 0x080000, CRC(3ce6a0a4) SHA1(6fdbf918f8f6408161bfa3eb06086e47543987cd) )
	sc_chavi_others
ROM_END

ROM_START( sc4chavic )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95418950.lo", 0x00001, 0x080000, CRC(f5026eec) SHA1(f535bbd31cc0e08568970f35edc9e366e90669eb) )
	ROM_LOAD16_BYTE( "95418951.hi", 0x00000, 0x080000, CRC(52d24ec0) SHA1(5cd6192b8a8681a49e288e6395dd98769a1e6e30) )
	sc_chavi_others
ROM_END

ROM_START( sc4chavid )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95419088.lo", 0x00001, 0x080000, CRC(c15d6376) SHA1(4ef1d50733bbd63157ff3e04d35d2fa3169d387c) )
	ROM_LOAD16_BYTE( "95419089.hi", 0x00000, 0x080000, CRC(4b894464) SHA1(00885b678bc9d70fd8d6d7af25c5c38bdac38115) )
	sc_chavi_others
ROM_END

ROM_START( sc4chavie )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95419826.lo", 0x00001, 0x080000, CRC(8d6220fb) SHA1(c7f6a4680c202f43765e5df0e6b313c53a1607d7) )
	ROM_LOAD16_BYTE( "95419827.hi", 0x00000, 0x080000, CRC(7d9ee9c9) SHA1(c2acc78e834522630fde6136f6b63dac97f721d6) )
	sc_chavi_others
ROM_END

ROM_START( sc4chavif )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95419900.lo", 0x00001, 0x080000, CRC(81ae49c8) SHA1(bc8bf3373b291eaa946b256b4ba645c1dc178495) )
	ROM_LOAD16_BYTE( "95419901.hi", 0x00000, 0x080000, CRC(8ca22714) SHA1(9fe571bc709f399e7d5190377d45f7a3230d9486) )
	sc_chavi_others
ROM_END

ROM_START( sc4chavig )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95419950.lo", 0x00001, 0x080000, CRC(b61e99f8) SHA1(eeee32e8d53c00fa70458bdb17a68f6083462f2a) )
	ROM_LOAD16_BYTE( "95419951.hi", 0x00000, 0x080000, CRC(9d2fc746) SHA1(84444065825c83724d4413fd4a516991ef5d03cf) )
	sc_chavi_others
ROM_END


ROM_START( sc4chavy )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416844.lo", 0x00001, 0x080000, CRC(37711221) SHA1(8b2fc0261281ffe7230415cb019b4f5cb939009e) )
	ROM_LOAD16_BYTE( "95416845.hi", 0x00000, 0x080000, CRC(0dd320a9) SHA1(2a41b89c40a1aee32e3283ad1449482cf2d355bc) )
	sc_chavy_others
ROM_END

ROM_START( sc4chavya )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416846.lo", 0x00001, 0x080000, CRC(bc307727) SHA1(5937e8fce2a92f0b3bedf02b0887a2f80b5bb27c) )
	ROM_LOAD16_BYTE( "95416847.hi", 0x00000, 0x080000, CRC(dfbb8862) SHA1(815434c40f554ab72b7cdfd780bb64c2643279b4) )
	sc_chavy_others
ROM_END

ROM_START( sc4chavyb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417844.lo", 0x00001, 0x080000, CRC(06affe82) SHA1(81b8d93b0085c8bd1285ee89f62a4e15cb3c600a) )
	ROM_LOAD16_BYTE( "95417845.hi", 0x00000, 0x080000, CRC(2b6f50e1) SHA1(2d4d45a1b4f472b958b00f137e3393f39876cfe0) )
	sc_chavy_others
ROM_END

ROM_START( sc4chavyc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417846.lo", 0x00001, 0x080000, CRC(c6f71785) SHA1(db10ccc08a9e46257d7c8b40e0104be5374dbc1b) )
	ROM_LOAD16_BYTE( "95417847.hi", 0x00000, 0x080000, CRC(3731e982) SHA1(0c244ad65c0b11e8f65d22a4cb97c3b459dc46a1) )
	sc_chavy_others
ROM_END

ROM_START( sc4chavyd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420336.lo", 0x00001, 0x080000, CRC(eae1a753) SHA1(36fc6ec520902cdf8b6f73c740bc50f491594d02) )
	ROM_LOAD16_BYTE( "95420337.hi", 0x00000, 0x080000, CRC(aefa6f50) SHA1(1dc93cc2bca7e724b791b9cfb3584c9539637deb) )
	sc_chavy_others
ROM_END

ROM_START( sc4chavye )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420338.lo", 0x00001, 0x080000, CRC(9fdebb0c) SHA1(bedf8d23568559804443342965b2cf64df6ed5d2) )
	ROM_LOAD16_BYTE( "95420339.hi", 0x00000, 0x080000, CRC(fdf9ad5f) SHA1(c58e5a5fa59333de25a546a8da31bd6be6310fde) )
	sc_chavy_others
ROM_END

ROM_START( sc4chavyf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421336.lo", 0x00001, 0x080000, CRC(476e9be9) SHA1(358ea89949aca559fb48327427402ddd5b0886ec) )
	ROM_LOAD16_BYTE( "95421337.hi", 0x00000, 0x080000, CRC(30b26f24) SHA1(7eca761a22797a6af3aec92a06b0ea5ff446c537) )
	sc_chavy_others
ROM_END

ROM_START( sc4chavyg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421338.lo", 0x00001, 0x080000, CRC(3756ac10) SHA1(9310ccb404923670c746b1c0527df5ee0cba4ab9) )
	ROM_LOAD16_BYTE( "95421339.hi", 0x00000, 0x080000, CRC(0fc25550) SHA1(ea0379141903823ba20acba52e1aaa99cb70aed6) )
	sc_chavy_others
ROM_END


ROM_START( sc4clue )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416502.lo", 0x00001, 0x080000, CRC(86604ff8) SHA1(da4694fa4fa272b3ba49177a4eac92acaea1209c) )
	ROM_LOAD16_BYTE( "95416503.hi", 0x00000, 0x080000, CRC(8cf0f42b) SHA1(714fe0fad59b2d16f12c39aae8943734a5202706) )
	sc_clue_others
ROM_END

ROM_START( sc4cluea )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416504.lo", 0x00001, 0x080000, CRC(5d0c4636) SHA1(b47097c2a6e440fc32fa0b36b55cf8e7c3bb5ebd) )
	ROM_LOAD16_BYTE( "95416505.hi", 0x00000, 0x080000, CRC(953a4273) SHA1(09fa4630aa2b8cb9208221ef818df7ba3564b78d) )
	sc_clue_others
ROM_END

ROM_START( sc4clueb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417502.lo", 0x00001, 0x080000, CRC(0f112ca9) SHA1(6a8ab713685ea04490745d69ad674c7d6ddc3160) )
	ROM_LOAD16_BYTE( "95417503.hi", 0x00000, 0x080000, CRC(5195ffd3) SHA1(1a7ff27e2747f67de3826de03ddf942961228e22) )
	sc_clue_others
ROM_END

ROM_START( sc4cluec )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417504.lo", 0x00001, 0x080000, CRC(21766be6) SHA1(9517a0a8dea3cd9f6f951dda7f8f0499010a626f) )
	ROM_LOAD16_BYTE( "95417505.hi", 0x00000, 0x080000, CRC(b605225c) SHA1(aa3283ce31c0a75b44e09d43995c25b63b5ac74e) )
	sc_clue_others
ROM_END

ROM_START( sc4clued )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420554.lo", 0x00001, 0x080000, CRC(004aa4b4) SHA1(7fadd404add64c710fcb60d7b260fba06fc192f0) )
	ROM_LOAD16_BYTE( "95420555.hi", 0x00000, 0x080000, CRC(0f1e5a37) SHA1(2d4589c3c5e6de99374b471230f14231d1e7258a) )
	sc_clue_others
ROM_END

ROM_START( sc4cluee )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420556.lo", 0x00001, 0x080000, CRC(fd2ca2c4) SHA1(b311e51d03a1d38b1001a0e6fa625a41f6ebf4a2) )
	ROM_LOAD16_BYTE( "95420557.hi", 0x00000, 0x080000, CRC(85a07ec8) SHA1(0e66f7aa5644f8ba7727a5737f6990effd5a41f8) )
	sc_clue_others
ROM_END

ROM_START( sc4cluef )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421554.lo", 0x00001, 0x080000, CRC(43efa639) SHA1(47aef3c613fa125a288843ac5caf02d76dcf2725) )
	ROM_LOAD16_BYTE( "95421555.hi", 0x00000, 0x080000, CRC(742d6413) SHA1(436892f74c15946672f0037095910163fb488f25) )
	sc_clue_others
ROM_END

ROM_START( sc4clueg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421556.lo", 0x00001, 0x080000, CRC(023e4809) SHA1(0f6d8b092ed988b48dba77cecd3117217f43f4f4) )
	ROM_LOAD16_BYTE( "95421557.hi", 0x00000, 0x080000, CRC(dd17ecaa) SHA1(dc264136e21b83fd9a36aa60f34c82af071f4ef4) )
	sc_clue_others
ROM_END



ROM_START( sc4cmani )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1308p11.hi", 0x00000, 0x080000, CRC(23e7409f) SHA1(fd5ed768ad5725a26db9f97f45deab8627af01d9) )
	ROM_LOAD16_BYTE( "p1308p11.lo", 0x00001, 0x080000, CRC(6bf123e0) SHA1(da2fde76ad8781b6d3925c02c3879c5025460021) )
	sc_cmani_others
ROM_END

ROM_START( sc4cmania )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1308s11.hi", 0x00000, 0x080000, CRC(b707e2f3) SHA1(d4872fbadb519729c0cbeeba7843973a67f22b4a) )
	ROM_LOAD16_BYTE( "p1308s11.lo", 0x00001, 0x080000, CRC(0f9409d8) SHA1(351bc2b5bb2f4bd9394ceb51bc75c1bd781d9e86) )
	sc_cmani_others
ROM_END




ROM_START( sc4cj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410986.lo", 0x00001, 0x080000, CRC(9e1adac0) SHA1(9956dc03286e59470a531fed1b15988c068a15c4) )
	ROM_LOAD16_BYTE( "95410987.hi", 0x00000, 0x080000, CRC(10963bc3) SHA1(6217af547ff9b6a7ca79053694be921067c4d901) )
	sc_cj_others
ROM_END

ROM_START( sc4cja )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411986.lo", 0x00001, 0x080000, CRC(080b6a93) SHA1(a3b390561a6756eef99d98291ee308c8cb7ef640) )
	ROM_LOAD16_BYTE( "95411987.hi", 0x00000, 0x080000, CRC(a30e1edb) SHA1(a1b308ad50845c4cdd83cbd8c33b99870e9f5053) )
	sc_cj_others
ROM_END

ROM_START( sc4cjb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412558.lo", 0x00001, 0x080000, CRC(7674e02b) SHA1(05bbebd138b8ab3831db2831169d870acb1f5dfb) )
	ROM_LOAD16_BYTE( "95412559.hi", 0x00000, 0x080000, CRC(b09e9822) SHA1(fbfe52703cb904f8eaf4b0c8b9e124adfd24470a) )
	sc_cj_others
ROM_END

ROM_START( sc4cjc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413558.lo", 0x00001, 0x080000, CRC(29b5cc1c) SHA1(b1265420b1f66f2d75bd0826e98a893a29ab577d) )
	ROM_LOAD16_BYTE( "95413559.hi", 0x00000, 0x080000, CRC(44c8ef1c) SHA1(bc2a630ad76221f4597ec644a6b548b73c51e9bd) )
	sc_cj_others
ROM_END

ROM_START( sc4cjd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410984.lo", 0x00001, 0x080000, NO_DUMP )
	ROM_LOAD16_BYTE( "95410985.hi", 0x00000, 0x080000, CRC(ac594336) SHA1(7f5ab3d31e71b04a7abb6ccf5bb47144730faaa7) )
	sc_cj_others
ROM_END


ROM_START( sc4ctl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406758.lo", 0x00001, 0x080000, CRC(d4446eb7) SHA1(25aaf705b83df02176f73b84ef271b24dce47676) )
	ROM_LOAD16_BYTE( "95406759.hi", 0x00000, 0x080000, CRC(74b3fc22) SHA1(925f3cb16a6e62e2b9e4160ce15f97e14cda355d) )
	sc_ctl_others
ROM_END

ROM_START( sc4ctla )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407758.lo", 0x00001, 0x080000, CRC(cdb926f7) SHA1(d92e42386166053e6aa2f63ecbd227f9cb89f90f) )
	ROM_LOAD16_BYTE( "95407759.hi", 0x00000, 0x080000, CRC(b1fc1032) SHA1(92b34e7391d78a8f1e992dce6f0a3b8a35bf0cda) )
	sc_ctl_others
ROM_END

ROM_START( sc4ctlb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408102.lo", 0x00001, 0x080000, CRC(7186022a) SHA1(a7d32c58fc86d60c46e65bd7153806800fdd79c2) )
	ROM_LOAD16_BYTE( "95408103.hi", 0x00000, 0x080000, CRC(6da649d4) SHA1(bd386437657337a35ec895a9921d9099d2e79bfb) )
	sc_ctl_others
ROM_END

ROM_START( sc4ctlc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408278.lo", 0x00001, 0x080000, CRC(64002efa) SHA1(7375b07f2aa26b46f85d8f5de4e69020a011236e) )
	ROM_LOAD16_BYTE( "95408279.hi", 0x00000, 0x080000, CRC(9adf6fc9) SHA1(7fb4d214fa94789ae3557a95ba87a5c88dc44061) )
	sc_ctl_others
ROM_END

ROM_START( sc4ctld )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409102.lo", 0x00001, 0x080000, CRC(5ea911cf) SHA1(9eb47b7c255275df6ca61d729114fdf4709f8729) )
	ROM_LOAD16_BYTE( "95409103.hi", 0x00000, 0x080000, CRC(829cb2d4) SHA1(50831ebf8ceafbf3120cb1ae5a38aa8c6d6db199) )
	sc_ctl_others
ROM_END

ROM_START( sc4ctle )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409278.lo", 0x00001, 0x080000, CRC(b159af51) SHA1(4ea868782a62b0e93df6cd84fcdbe8f33a505221) )
	ROM_LOAD16_BYTE( "95409279.hi", 0x00000, 0x080000, CRC(2879e6a1) SHA1(0321aafa8a581f4d1e2884104a7dee6c3aab5fb1) )
	sc_ctl_others
ROM_END



ROM_START( sc4crsc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95422152.lo", 0x00001, 0x080000, CRC(dd719a45) SHA1(361295cbb82cac4cfd2e7d24cfe7a786753129b7) )
	ROM_LOAD16_BYTE( "95422153.hi", 0x00000, 0x080000, CRC(eba64f98) SHA1(54437047b4feee53119e5fd8239ec63c52fe9417) )
	sc_crsc_others
ROM_END

ROM_START( sc4crsca )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95422318.lo", 0x00001, 0x080000, CRC(ccc6070f) SHA1(186f1741c89bc62bff0654bfc0875cdfde873a43) )
	ROM_LOAD16_BYTE( "95422319.hi", 0x00000, 0x080000, CRC(6a21ffcf) SHA1(6977d4c09d019223b6cb3dedca1246ef50069958) )
	sc_crsc_others
ROM_END

ROM_START( sc4crscb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95423152.lo", 0x00001, 0x080000, CRC(51ec4234) SHA1(d72adeb3ff8e597425787d5196175cddc04cac9b) )
	ROM_LOAD16_BYTE( "95423153.hi", 0x00000, 0x080000, CRC(3d97a39a) SHA1(dc8e9257b28380efd478fc2e80c5cfa2d8554683) )
	sc_crsc_others
ROM_END

ROM_START( sc4crscc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95423318.lo", 0x00001, 0x080000, CRC(dc508639) SHA1(ac87d7c92e043f711ed7562e041aa362955df118) )
	ROM_LOAD16_BYTE( "95423319.hi", 0x00000, 0x080000, CRC(ef41455f) SHA1(1b7b7debd03afc18701b8fa0e5bd6f00e6fbcc79) )
	sc_crsc_others
ROM_END

ROM_START( sc4crscd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426116.lo", 0x00001, 0x080000, CRC(cd991b6f) SHA1(15da97f4f1225afb5b731a4b7570141b4451d989) )
	ROM_LOAD16_BYTE( "95426117.hi", 0x00000, 0x080000, CRC(01030af9) SHA1(d77391942f4b28289888a409c7e81bd36609b9de) )
	sc_crsc_others
ROM_END

ROM_START( sc4crsce )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426336.lo", 0x00001, 0x080000, CRC(ef76190e) SHA1(6167c6b7e659e7c920933a4979ed6590ce44648e) )
	ROM_LOAD16_BYTE( "95426337.hi", 0x00000, 0x080000, CRC(28114c89) SHA1(674f92c112c0f2a7527e697a032be09fb0acdb6b) )
	sc_crsc_others
ROM_END

ROM_START( sc4crscf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427116.lo", 0x00001, 0x080000, CRC(c73e8d02) SHA1(b6b6d47db75714a063686a91c0a30869719be714) )
	ROM_LOAD16_BYTE( "95427117.hi", 0x00000, 0x080000, CRC(7ba928f6) SHA1(1587767b546288c5f6a0fcebc6b04b008c35d35b) )
	sc_crsc_others
ROM_END

ROM_START( sc4crscg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427336.lo", 0x00001, 0x080000, CRC(17fae461) SHA1(7e8c50ca40aa058bd823cecb64004da0885b2b36) )
	ROM_LOAD16_BYTE( "95427337.hi", 0x00000, 0x080000, CRC(392e4bea) SHA1(15fa1852989aefd45771737cc8c5d62c06354b1d) )
	sc_crsc_others
ROM_END





ROM_START( sc4coro )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416528.lo", 0x00001, 0x080000, CRC(5fb06486) SHA1(8793d3a5e65aa9326c2ae71361045337cc9491c7) )
	ROM_LOAD16_BYTE( "95416529.hi", 0x00000, 0x080000, CRC(30d24daa) SHA1(912d8ebf13105a13b7aa505abef3a17d0327cd28) )
	sc_coro_others
ROM_END

ROM_START( sc4coroa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416530.lo", 0x00001, 0x080000, CRC(2e82f143) SHA1(074db310de493b315f5e0c7cac05103ed919034c) )
	ROM_LOAD16_BYTE( "95416531.hi", 0x00000, 0x080000, CRC(c03c5948) SHA1(aad67f0b14985de08942f43eef23e3a4fb8aeb06) )
	sc_coro_others
ROM_END



ROM_START( sc4corof )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417528.lo", 0x00001, 0x080000, CRC(5efaa5fe) SHA1(87b8b02b658d9f54f914b74c43369477c98ab862) )
	ROM_LOAD16_BYTE( "95417529.hi", 0x00000, 0x080000, CRC(702b64d4) SHA1(54d2ce728ba08aad1ed2b08e9d3045aadc84f411) )
	sc_coro_others
ROM_END

ROM_START( sc4corog )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417530.lo", 0x00001, 0x080000, CRC(5abd5fcf) SHA1(3be0ee04959df05ab7106b3a8016115e5099ad2a) )
	ROM_LOAD16_BYTE( "95417531.hi", 0x00000, 0x080000, CRC(2a686edc) SHA1(0c2ac9d668ae234dd3c2653e23cec70286a12057) )
	sc_coro_others
ROM_END


ROM_START( sc4coroj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420090.lo", 0x00001, 0x080000, CRC(235f14eb) SHA1(677555098be99e52b26fcad96aec69fe25db4f22) )
	ROM_LOAD16_BYTE( "95420091.hi", 0x00000, 0x080000, CRC(0818ca67) SHA1(27df81ecb455a9602029efe8b106cac5cd0680c2) )
	sc_coro_others
ROM_END

ROM_START( sc4corok )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420092.lo", 0x00001, 0x080000, CRC(3cd22ea3) SHA1(51e735c7cd81a2a9c1ea0c94a2d9141fe4158f37) )
	ROM_LOAD16_BYTE( "95420093.hi", 0x00000, 0x080000, CRC(52849e29) SHA1(46d2d119949d747bb0f1fdaa572371cdb227d799) )
	sc_coro_others
ROM_END

ROM_START( sc4corol )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421090.lo", 0x00001, 0x080000, CRC(9ac45c6b) SHA1(2c8bd2f8490d08b8cff2196fa38cb6e700779da9) )
	ROM_LOAD16_BYTE( "95421091.hi", 0x00000, 0x080000, CRC(0bf1289a) SHA1(1343b01dd9f1b543c09b1669e8d157bb6f9d390c) )
	sc_coro_others
ROM_END

ROM_START( sc4corom )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421092.lo", 0x00001, 0x080000, CRC(b21f91f0) SHA1(1583561132f23b3b10752e51a7401cd06003e784) )
	ROM_LOAD16_BYTE( "95421093.hi", 0x00000, 0x080000, CRC(09498b6c) SHA1(c275da993b004855ea691c54ad7b416f2381a2cc) )
	sc_coro_others
ROM_END


ROM_START( sc4coron )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416744.lo", 0x00001, 0x080000, CRC(4582bb6a) SHA1(70189c6602dc258baa8c7273820b1b5e9696ab3e) )
	ROM_LOAD16_BYTE( "95416745.hi", 0x00000, 0x080000, NO_DUMP )
	COROST_TRIPLE_SOUNDS
ROM_END

ROM_START( sc4corotb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416742.lo", 0x00001, 0x080000, CRC(5e3c1a6a) SHA1(8d4c83be4f9192df01aa3b04a1a1ce17fe48a2a7) )
	ROM_LOAD16_BYTE( "95416743.hi", 0x00000, 0x080000, CRC(b9020b30) SHA1(3e27080ef2e341b4ce37e803f54ade398a4eaca6) )
	COROST_TRIPLE_SOUNDS
ROM_END


ROM_START( sc4corotba )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416904.lo", 0x00001, 0x080000, CRC(af23d81c) SHA1(f2854e93a2992671dcfefaa8b980d95b1ec0a483) )
	ROM_LOAD16_BYTE( "95416905.hi", 0x00000, 0x080000, CRC(de4cf602) SHA1(d71cb4f2746d8c0fa2eeb577296ff8b7b08bcd79) )
	COROST_TRIPLE_SOUNDS
ROM_END

ROM_START( sc4coroc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416752.lo", 0x00001, 0x080000, CRC(b50fae97) SHA1(9f3afd1c6dfff818155ccaeaf18889224cd6d625) )
	ROM_LOAD16_BYTE( "95416753.hi", 0x00000, 0x080000, CRC(5a88a8dc) SHA1(4f32fb7744af6f05bc9b6ba82f4ca5ccf6c4fa12) )
	COROST_TRIPLE_SOUNDS
ROM_END

ROM_START( sc4corod )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416838.lo", 0x00001, 0x080000, CRC(0964030d) SHA1(451f681e85223d9bf25e1a3803a178948f3e2cc1) )
	ROM_LOAD16_BYTE( "95416839.hi", 0x00000, 0x080000, CRC(75e641fb) SHA1(655e1694cec922d99b013d52ea008e6f0c87ad4c) )
	COROST_TRIPLE_SOUNDS
ROM_END

ROM_START( sc4coroh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417752.lo", 0x00001, 0x080000, CRC(0860b91a) SHA1(3cb58778418fbffd2fd43d70fcbd4854dc595aa7) )
	ROM_LOAD16_BYTE( "95417753.hi", 0x00000, 0x080000, CRC(0344ba50) SHA1(c8210b78fc2427db3620c5fe5f26b8df28f7e16d) )
	COROST_TRIPLE_SOUNDS
ROM_END

ROM_START( sc4coroi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417838.lo", 0x00001, 0x080000, CRC(1c90823a) SHA1(7787c98aa784f3cfc9443c1ef9ab22dc814b9a80) )
	ROM_LOAD16_BYTE( "95417839.hi", 0x00000, 0x080000, CRC(70d722e7) SHA1(bafeb7828b88a8b25daa9cb6bb4d540f89e71f6f) )
	COROST_TRIPLE_SOUNDS
ROM_END

ROM_START( sc4count )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1929p12.hi", 0x00000, 0x080000, CRC(b5824ce7) SHA1(b7e7a05e187b6d823233218da8a89465d7aa0e75) )
	ROM_LOAD16_BYTE( "p1929p12.lo", 0x00001, 0x080000, CRC(95e21553) SHA1(1f2f67dcf326ed9382944e765b0b508b49efa4c4) )
	sc_count_others
ROM_END

ROM_START( sc4counta )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1929s12.hi", 0x00000, 0x080000, CRC(f97ce062) SHA1(03fe840625a0bd3c4c9c2aa2a93c60e855014d80) )
	ROM_LOAD16_BYTE( "p1929s12.lo", 0x00001, 0x080000, CRC(b377e9d2) SHA1(52e750458af1583682ba951a50b9cd3eefb40987) )
	sc_count_others
ROM_END



ROM_START( sc4czfr )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95402217.lo", 0x00001, 0x080000, CRC(5f86e26e) SHA1(293d3700844dcfaf2a5c6e9a4ee817893e11cf1b) )
	ROM_LOAD16_BYTE( "95402218.hi", 0x00000, 0x080000, CRC(72792a3e) SHA1(efbf3fcf7f7b40424c8b7fa46d35c9bb484fe431) )
	SOUND_95004150_CRAZY_FRUITS_PR6982
ROM_END

ROM_START( sc4czfrd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95401701.lo", 0x00001, 0x080000, CRC(5c09007e) SHA1(d69290225f08ed5dcf0204bd96f7f09a73c2f243) )
	ROM_LOAD16_BYTE( "95401702.hi", 0x00000, 0x080000, NO_DUMP )
	SOUND_95004150_CRAZY_FRUITS_PR6982
ROM_END


ROM_START( sc4czfra )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406394.lo", 0x00001, 0x080000, CRC(c07a8695) SHA1(3670e547530452f21a937f34072d812f25a480ce) )
	ROM_LOAD16_BYTE( "95406395.hi", 0x00000, 0x080000, CRC(e469252b) SHA1(a7861f3d215c70b57263501af158f86257bf37c0) )
	PR1152_CRAZY_FRUITS_SOUNDS11
ROM_END

ROM_START( sc4czfrb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "crazyfruits game rom low.bin", 0x00001, 0x080000, CRC(0afc9913) SHA1(1b3122d9a16a6f58893d241b96aa8b9145c09151) )
	ROM_LOAD16_BYTE( "crazyfruits game rom hi.bin", 0x00000, 0x080000, CRC(15d795db) SHA1(7f3d043c4abfb9116e700381cc4c2096c52e6fe7) )
	sc_czfr_others
ROM_END

ROM_START( sc4czfrc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "qcf-cf.lo", 0x00001, 0x080000, CRC(c7720662) SHA1(081f1e0cc4da26a15be49f5d37c630d2d3d2a7a6) )
	ROM_LOAD16_BYTE( "qcf-cf.hi", 0x00000, 0x080000, CRC(b34c8653) SHA1(8abf0ce18b9a649eb25de71c42348413d5247621) )
	sc_czfr_others
ROM_END



ROM_START( sc4czfre )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95400341.lo", 0x00001, 0x080000, CRC(5de1df69) SHA1(2a83c45ed694b2bf20fc1a748ecd3d00a5271832) )
	ROM_LOAD16_BYTE( "95400342.hi", 0x00000, 0x080000, CRC(903dee0c) SHA1(529fdb14a06564afa3fcf3a2bf04f0c682c4e636) )
	sc_czfr_others
ROM_END

ROM_START( sc4czfrf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95400343.lo", 0x00001, 0x080000, CRC(d525c165) SHA1(818679010490f0266f8a8ada845d227d31bb92a6) )
	ROM_LOAD16_BYTE( "95400344.hi", 0x00000, 0x080000, CRC(afac271f) SHA1(6bc551ed667fbcdd42722222080141d971e709c9) )
	sc_czfr_others
ROM_END

ROM_START( sc4czfrg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95400345.lo", 0x00001, 0x080000, CRC(dbf40752) SHA1(26fe336c235538ed50c4d5b67262bcaf02f74dcc) )
	ROM_LOAD16_BYTE( "95400346.hi", 0x00000, 0x080000, CRC(28344abe) SHA1(c1c361c335114d6102b4ba97809eeb2f8cb2a299) )
	sc_czfr_others
ROM_END

ROM_START( sc4czfrh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95401341.lo", 0x00001, 0x080000, CRC(ea1b83ef) SHA1(8a526c030b8e9547fcdebfaa8316ce77adec9a69) )
	ROM_LOAD16_BYTE( "95401342.hi", 0x00000, 0x080000, CRC(b1e080fb) SHA1(d284191feacb6636cec76e6103ed98003c04c40f) )
	sc_czfr_others
ROM_END

ROM_START( sc4czfri )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95401343.lo", 0x00001, 0x080000, CRC(23aa3367) SHA1(d291432a0595c97d16e8183501a3d622bf20cc86) )
	ROM_LOAD16_BYTE( "95401344.hi", 0x00000, 0x080000, CRC(ba038114) SHA1(3b9835224eae26af435f2d1fd2156a751a48f0e2) )
	sc_czfr_others
ROM_END


ROM_START( sc4czfrj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95401345.lo", 0x00001, 0x080000, CRC(8c3e322a) SHA1(f2663cdd8280fa661ac2aea257acc363525b60c5) )
	ROM_LOAD16_BYTE( "95401346.hi", 0x00000, 0x080000, CRC(d9b74800) SHA1(a3b2e098c7d04fbaf98da69de41636b52e8b497c) )
	sc_czfr_others
ROM_END

ROM_START( sc4czfrk )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95401347.lo", 0x00001, 0x080000, CRC(2bdabad1) SHA1(81479370e7a03d8562162cb4820157d733c5aa87) )
	ROM_LOAD16_BYTE( "95401348.hi", 0x00000, 0x080000, CRC(63212cb9) SHA1(2eb3a00a6f18cc4c63ae84f321735be824e20889) )
	sc_czfr_others
ROM_END


ROM_START( sc4crnjw )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412658.lo", 0x00001, 0x080000, CRC(2d60552e) SHA1(d4bee760079e727909a97a0c78ebcb56686c8ec4) )
	ROM_LOAD16_BYTE( "95412659.hi", 0x00000, 0x080000, CRC(0bd73c6e) SHA1(a388758932d9cf66556fe90cf79975660f5c1ab8) )
	sc_crnjw_others
ROM_END

ROM_START( sc4crnjwa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413658.lo", 0x00001, 0x080000, CRC(9660aff0) SHA1(77593512c29941694bd008b251f6f3aec5176ab8) )
	ROM_LOAD16_BYTE( "95413659.hi", 0x00000, 0x080000, CRC(07d2f7af) SHA1(d4b315655c7f991eb465de1f6cb622ec7ff2b062) )
	sc_crnjw_others
ROM_END



ROM_START( sc4dnd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95418430.lo", 0x00001, 0x080000, CRC(16632394) SHA1(76fc5f14d2b0895959580979d9973094fa10c63a) )
	ROM_LOAD16_BYTE( "95418431.hi", 0x00000, 0x080000, CRC(25e91233) SHA1(8814c4f48299f07e1836ff4e86ceead5fc91e7c9) )
	PR1907_DEAL_OR_NO_DEAL_SOUNDS11
ROM_END


ROM_START( sc4dnda )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95418628.lo", 0x00001, 0x080000, CRC(534e6f8b) SHA1(8b89b691eac6415344e5e91a6682cd2030ffba6e) )
	ROM_LOAD16_BYTE( "95418629.hi", 0x00000, 0x080000, NO_DUMP )
	PR1907_DEAL_OR_NO_DEAL_SOUNDS11
ROM_END


ROM_START( sc4dndb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95419430.lo", 0x00001, 0x080000, CRC(1a510dfc) SHA1(e4620c76a56539e93d90d7f9729b4daab2278256) )
	ROM_LOAD16_BYTE( "95419431.hi", 0x00000, 0x080000, CRC(d4f546b5) SHA1(10d9df4dec30f9f1a9ffecd8c74de491a20ddf56) )
	PR1907_DEAL_OR_NO_DEAL_SOUNDS11
ROM_END



ROM_START( sc4dndc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95419628.lo", 0x00001, 0x080000, CRC(5a6f07e9) SHA1(db21ee37d9a4fd421593fc9eb8ef27bad6788c4e) )
	ROM_LOAD16_BYTE( "95419629.hi", 0x00000, 0x080000, CRC(cdf330d0) SHA1(aa5030ae3ec64ce4ead9688e51bfcd8ff1bbf543) )
	PR1907_DEAL_OR_NO_DEAL_SOUNDS11
ROM_END


ROM_START( sc4dndd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95422036.lo", 0x00001, 0x080000, CRC(b1d048dc) SHA1(bde75f8797ebd668eb8c9cf6c05da562874df540) )
	ROM_LOAD16_BYTE( "95422037.hi", 0x00000, 0x080000, CRC(2d881f97) SHA1(aec94590f8cd7d21b386804b05bd53c9dee17968) )
	PR1907_DEAL_OR_NO_DEAL_SOUNDS11
ROM_END

ROM_START( sc4dnde )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95422866.lo", 0x00001, 0x080000, CRC(9fd063d1) SHA1(6e7dc034c2b6da609b5f200e4ede20faecf1104c) )
	ROM_LOAD16_BYTE( "95422867.hi", 0x00000, 0x080000, CRC(982b5826) SHA1(46be9e735c16300553a4b37d6d98ff97cdb43aed) )
	PR1907_DEAL_OR_NO_DEAL_SOUNDS11
ROM_END

ROM_START( sc4dndf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95422890.lo", 0x00001, 0x080000, CRC(ae29311c) SHA1(1bdd12b2cd2e96d4704da044b251fdb111ee0cb1) )
	ROM_LOAD16_BYTE( "95422891.hi", 0x00000, 0x080000, CRC(d79559eb) SHA1(ef5f27b5b84bf1072f1b5a13ee17143a06515712) )
	PR1907_DEAL_OR_NO_DEAL_SOUNDS11
ROM_END

ROM_START( sc4dndg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95423036.lo", 0x00001, 0x080000, CRC(e484b7fb) SHA1(6489fe890ca6cfeb4eb2043253c6bd0bab185724) )
	ROM_LOAD16_BYTE( "95423037.hi", 0x00000, 0x080000, CRC(e2c36879) SHA1(8978ed2c7c1510b62eedb0353e435b89b648670a) )
	PR1907_DEAL_OR_NO_DEAL_SOUNDS11
ROM_END

ROM_START( sc4dndh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95423866.lo", 0x00001, 0x080000, CRC(f99b2b86) SHA1(f04e154c1887d4225075802e3618261bf444ae26) )
	ROM_LOAD16_BYTE( "95423867.hi", 0x00000, 0x080000, CRC(fe6b1130) SHA1(2de60e150829acde7f7f3453fd5397ed549ab421) )
	PR1907_DEAL_OR_NO_DEAL_SOUNDS11
ROM_END

ROM_START( sc4dndi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95423890.lo", 0x00001, 0x080000, CRC(9b6d3415) SHA1(b14bc3ede63184f7d3be0703e7170f85671350d5) )
	ROM_LOAD16_BYTE( "95423891.hi", 0x00000, 0x080000, CRC(58c37a50) SHA1(09fde3c464bab08b95e2609cde64cf2fc4ddce7e) )
	PR1907_DEAL_OR_NO_DEAL_SOUNDS11
ROM_END

ROM_START( sc4dndj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426154.lo", 0x00001, 0x080000, CRC(17169e09) SHA1(e9e6752ab858c1d14253d27842c269c8e9e6fb5f) )
	ROM_LOAD16_BYTE( "95426155.hi", 0x00000, 0x080000, CRC(7eb7893e) SHA1(a97f35be2d494da19dcb6f4b235a83d46fabba3e) )
	PR1907_DEAL_OR_NO_DEAL_SOUNDS11
ROM_END

ROM_START( sc4dndk )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426170.lo", 0x00001, 0x080000, CRC(99951add) SHA1(640401c6b412dd0d2a186a134e4fb77c92001614) )
	ROM_LOAD16_BYTE( "95426171.hi", 0x00000, 0x080000, CRC(eb65bbe5) SHA1(f8c384086a335a02985ed938b405fffd4aa37bbb) )
	PR1907_DEAL_OR_NO_DEAL_SOUNDS11
ROM_END

ROM_START( sc4dndl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427154.lo", 0x00001, 0x080000, CRC(745ac559) SHA1(3fff27727fdf6b34f4500bc42dacacb9a807e848) )
	ROM_LOAD16_BYTE( "95427155.hi", 0x00000, 0x080000, CRC(15a28fd8) SHA1(4fe2fb60d447bfcc4996643badcf5440b76a951d) )
	PR1907_DEAL_OR_NO_DEAL_SOUNDS11
ROM_END

ROM_START( sc4dndm )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427170.lo", 0x00001, 0x080000, CRC(65cf387c) SHA1(3436e81927c00551d3eaa7d21922187ac9b93bf5) )
	ROM_LOAD16_BYTE( "95427171.hi", 0x00000, 0x080000, CRC(f4021af5) SHA1(15eb7c36da5ab02a14021ec02af5861431bb318b) )
	PR1907_DEAL_OR_NO_DEAL_SOUNDS11
ROM_END

ROM_START( sc4dndn )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95422038.lo", 0x00001, 0x080000, CRC(c75da829) SHA1(b240a1aa1056c3227a23ae1df37eacece7832a97) )
	ROM_LOAD16_BYTE( "95422039.hi", 0x00000, 0x080000, CRC(c9c8661b) SHA1(c8ad798e4807da659015a9925cdb0eb6e837a9af) )
	PR1907_DEAL_OR_NO_DEAL_SOUNDS11
ROM_END

ROM_START( sc4dndo )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95423038.lo", 0x00001, 0x080000, CRC(00adce65) SHA1(917a882f6e28198795ee056914433dd1ce361803) )
	ROM_LOAD16_BYTE( "95423039.hi", 0x00000, 0x080000, CRC(556af108) SHA1(7a88c1311ab7aa7773dd6b70a3891ebbe2d340d1) )
	PR1907_DEAL_OR_NO_DEAL_SOUNDS11
ROM_END

ROM_START( sc4dndcs )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95418528.lo", 0x00001, 0x080000, CRC(d0192ab5) SHA1(b1e98d901ea0ea0010f02e76c296418da775a8c6) )
	ROM_LOAD16_BYTE( "95418529.hi", 0x00000, 0x080000, CRC(10ce1fa8) SHA1(a207f8c1b4fa40f95f5c3cba64648f87efa5d903) )
	PR1907_DEAL_OR_NO_DEAL_SOUNDS11
ROM_END

ROM_START( sc4dndcsa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95418752.lo", 0x00001, 0x080000, CRC(ea381dee) SHA1(e68557b00fbbac86d7c153470050da0980751fa1) )
	ROM_LOAD16_BYTE( "95418753.hi", 0x00000, 0x080000, CRC(de953115) SHA1(98443f633997be3e7a8c43e6692a9dd0c4fcf0b1) )
	PR1907_DEAL_OR_NO_DEAL_SOUNDS11
ROM_END

ROM_START( sc4dndcsb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95419528.lo", 0x00001, 0x080000, CRC(0498ed27) SHA1(cab277385393410dbe54a72f7f778d67aeb42ba1) )
	ROM_LOAD16_BYTE( "95419529.hi", 0x00000, 0x080000, CRC(524a6e36) SHA1(0f8639a83eefcb3cf8f7541a2a77e984cf1688d5) )
	PR1907_DEAL_OR_NO_DEAL_SOUNDS11
ROM_END

ROM_START( sc4dndcsc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95419752.lo", 0x00001, 0x080000, CRC(c06b60aa) SHA1(b40e4412db1f57511e0b2586fd65df56874a23bb) )
	ROM_LOAD16_BYTE( "95419753.hi", 0x00000, 0x080000, CRC(76da2127) SHA1(2a06c5fb329a6045671f92073c2e1dd78dc79769) )
	PR1907_DEAL_OR_NO_DEAL_SOUNDS11
ROM_END

ROM_START( sc4dndcsd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95422796.lo", 0x00001, 0x080000, CRC(e868d7b2) SHA1(9bfab76066bdd8c0189a3b076cb5fcaf9b319974) )
	ROM_LOAD16_BYTE( "95422797.hi", 0x00000, 0x080000, CRC(7814d075) SHA1(2a5b3e2e9c6a2a1823ce952d978287c012e6bdb8) )
	PR1907_DEAL_OR_NO_DEAL_SOUNDS11
ROM_END

ROM_START( sc4dndcse )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95423796.lo", 0x00001, 0x080000, CRC(804c03c4) SHA1(f73c9655c1206334afeb6bbd73bd24579eb1d67c) )
	ROM_LOAD16_BYTE( "95423797.hi", 0x00000, 0x080000, CRC(c5b8ef0a) SHA1(5ae7be45364c8db8425b1ea08d8705c1fe2e032a) )
	PR1907_DEAL_OR_NO_DEAL_SOUNDS11
ROM_END


ROM_START( sc4dndbb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95422664.lo", 0x00001, 0x080000, CRC(af24377d) SHA1(9c1f8373071de8eca8a498453375db26e27aa3e1) )
	ROM_LOAD16_BYTE( "95422665.hi", 0x00000, 0x080000, CRC(31e74642) SHA1(6b738af6d35af25cabb0e7cd7c5a011cae166788) )
	sc_dndbb_others
ROM_END

ROM_START( sc4dndbba )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95422778.lo", 0x00001, 0x080000, CRC(739c07d7) SHA1(3e5d8fea4d357358f16b76f8a692df281d03cdaf) )
	ROM_LOAD16_BYTE( "95422779.hi", 0x00000, 0x080000, CRC(2000c2cf) SHA1(e72ac5615e194f9d691eb9ad88f84ef8f17894a5) )
	sc_dndbb_others
ROM_END

ROM_START( sc4dndbbb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95423664.lo", 0x00001, 0x080000, CRC(06e23750) SHA1(1973d9b37d3c2d73885cbb847fe98e56a635b006) )
	ROM_LOAD16_BYTE( "95423665.hi", 0x00000, 0x080000, CRC(c9ab6972) SHA1(60721c3c0bed04ae3c19bbaf59b993a8f7ea5df7) )
	sc_dndbb_others
ROM_END

ROM_START( sc4dndbbc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95423778.lo", 0x00001, 0x080000, CRC(55a5b079) SHA1(a3739f8e753bf050b67c43dd27d71a53f5bc5b01) )
	ROM_LOAD16_BYTE( "95423779.hi", 0x00000, 0x080000, CRC(3bce6107) SHA1(7f7b11ff409cf3010cb821d5ed8cbc66a25ba8a8) )
	sc_dndbb_others
ROM_END

ROM_START( sc4dndbbd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426050.lo", 0x00001, 0x080000, CRC(db98f3e1) SHA1(72494a2a7a00ccc474e039f68ec107f230c2e104) )
	ROM_LOAD16_BYTE( "95426051.hi", 0x00000, 0x080000, CRC(fa0b3bff) SHA1(9142a3ba525268f56b75b424816524c33bb1df3a) )
	sc_dndbb_others
ROM_END

ROM_START( sc4dndbbe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426308.lo", 0x00001, 0x080000, CRC(959cffe8) SHA1(8600b48081c84eafcdd5cbf3d4047c1713f64253) )
	ROM_LOAD16_BYTE( "95426309.hi", 0x00000, 0x080000, CRC(6038fd5c) SHA1(b8f36483334f2fb215cc5be4f75d9895a1ed4ac0) )
	sc_dndbb_others
ROM_END

ROM_START( sc4dndbbf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426414.lo", 0x00001, 0x080000, CRC(e1cd19de) SHA1(f0a08fd513b96742970abedabea4d4ba6cc8b05b) )
	ROM_LOAD16_BYTE( "95426415.hi", 0x00000, 0x080000, CRC(542e1930) SHA1(d99e54c17147374cbf153c6ed52df9be6d4e03ba) )
	sc_dndbb_others
ROM_END

ROM_START( sc4dndbbg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427050.lo", 0x00001, 0x080000, CRC(72dbb305) SHA1(0cafd52a51ec69b0a30823c972230801c59a9575) )
	ROM_LOAD16_BYTE( "95427051.hi", 0x00000, 0x080000, CRC(020a5849) SHA1(14b2eee9e624a90b74fd9b71f10f8d677a71cfb8) )
	sc_dndbb_others
ROM_END

ROM_START( sc4dndbbh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427308.lo", 0x00001, 0x080000, CRC(beaec88a) SHA1(dbf04c3a2eca9a36a55579b42da2297754d4b455) )
	ROM_LOAD16_BYTE( "95427309.hi", 0x00000, 0x080000, CRC(f5ae80ed) SHA1(4d3353318c29ef7e6b14246177833c46632404d0) )
	sc_dndbb_others
ROM_END

ROM_START( sc4dndbbi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427414.lo", 0x00001, 0x080000, CRC(40f21aca) SHA1(0120c77b4812970d2624e85c05c5195956a1716d) )
	ROM_LOAD16_BYTE( "95427415.hi", 0x00000, 0x080000, CRC(f218587b) SHA1(1ecb5c781fddc4639d4966ff2ecf360d38a9a1e6) )
	sc_dndbb_others
ROM_END



ROM_START( sc4dndcl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95418402.lo", 0x00001, 0x080000, CRC(afd68b50) SHA1(2553aeaeb344bc659da4faf1522a883e5da6962b) )
	ROM_LOAD16_BYTE( "95418403.hi", 0x00000, 0x080000, CRC(47c7a8b5) SHA1(1117caaf9ebf651f0f8988c0a79a0dda2b08e769) )
	sc_dndcl_others
ROM_END

ROM_START( sc4dndcla )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95418404.lo", 0x00001, 0x080000, CRC(ed598e25) SHA1(05d5c0224ff341bbb330ba6dd1dcf303c197439c) )
	ROM_LOAD16_BYTE( "95418405.hi", 0x00000, 0x080000, CRC(4d8b9a91) SHA1(e212d79da66e5dc33da8cca75654857763b02071) )
	sc_dndcl_others
ROM_END

ROM_START( sc4dndclb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95418406.lo", 0x00001, 0x080000, CRC(08151444) SHA1(325e0c3967995bd1b834ae650caefbf6e24059b0) )
	ROM_LOAD16_BYTE( "95418407.hi", 0x00000, 0x080000, CRC(40eb2214) SHA1(8f999222e8aa5706fef8ec789242983eec4b351b) )
	sc_dndcl_others
ROM_END

ROM_START( sc4dndclc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95418702.lo", 0x00001, 0x080000, CRC(d14266fd) SHA1(96c00ef152e557eacb4c27e51f3c034aecf5057b) )
	ROM_LOAD16_BYTE( "95418703.hi", 0x00000, 0x080000, CRC(9b98c457) SHA1(a4729864939793a3311115ecd2aa43d69d979ede) )
	sc_dndcl_others
ROM_END

ROM_START( sc4dndcld )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95419402.lo", 0x00001, 0x080000, CRC(d04aaebd) SHA1(c598dcde46b20760ce3af55820b9c9026ceeb062) )
	ROM_LOAD16_BYTE( "95419403.hi", 0x00000, 0x080000, CRC(b59808a3) SHA1(8ffb0bea90b740a2845c4e88ea2479a479f6e8c6) )
	sc_dndcl_others
ROM_END

ROM_START( sc4dndcle )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95419404.lo", 0x00001, 0x080000, CRC(b618fb02) SHA1(5602b6aca6be28e539e158e0efd67e90762e27dd) )
	ROM_LOAD16_BYTE( "95419405.hi", 0x00000, 0x080000, CRC(5a59647a) SHA1(3f3ad2dd60ddf7d22debebfa105f2f866bab4858) )
	sc_dndcl_others
ROM_END

ROM_START( sc4dndclf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95419406.lo", 0x00001, 0x080000, CRC(5a4f938b) SHA1(db7d5bad6af7cf116b6c74ff236538d7916fce9c) )
	ROM_LOAD16_BYTE( "95419407.hi", 0x00000, 0x080000, CRC(ca3aa841) SHA1(5e9e87aff7b0d4ba2f3a005803aec726eb5abbbe) )
	sc_dndcl_others
ROM_END

ROM_START( sc4dndclg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95419702.lo", 0x00001, 0x080000, CRC(fd9d2ffc) SHA1(1339b014b2ce00db1debdd1e9114924cead84058) )
	ROM_LOAD16_BYTE( "95419703.hi", 0x00000, 0x080000, CRC(c0b6d803) SHA1(89564578207c769065069ba6d498c40cf00a0d11) )
	sc_dndcl_others
ROM_END


ROM_START( sc4dnddd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426042.lo", 0x00001, 0x080000, CRC(b89fdf2c) SHA1(4330486e29841bd04a1db2d6ca25adc8f8f2eb8d) )
	ROM_LOAD16_BYTE( "95426043.hi", 0x00000, 0x080000, CRC(e3f3fc13) SHA1(2b0af338866fe188aec75a02c106e2a38e4e5a9e) )
	sc_dnddd_others
ROM_END

ROM_START( sc4dnddda )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426300.lo", 0x00001, 0x080000, CRC(921e2253) SHA1(28869d93239dd79c1ee79c6a42554cb9f1a612b1) )
	ROM_LOAD16_BYTE( "95426301.hi", 0x00000, 0x080000, CRC(23e2e613) SHA1(357e6e8f3c8967ab5dfa86403206a1a0c86b635a) )
	sc_dnddd_others
ROM_END

ROM_START( sc4dndddb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426408.lo", 0x00001, 0x080000, CRC(773683e0) SHA1(5094f03ea89c95e9b7493fc67cebe16147fe15d0) )
	ROM_LOAD16_BYTE( "95426409.hi", 0x00000, 0x080000, CRC(2ed78190) SHA1(2cdab06d9c9ce5b5da777e9f8519ef517224d27e) )
	sc_dnddd_others
ROM_END

ROM_START( sc4dndddc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426456.lo", 0x00001, 0x080000, CRC(38c66ba2) SHA1(61367219ed92daf00d241ca8fe708bb7678112a0) )
	ROM_LOAD16_BYTE( "95426457.hi", 0x00000, 0x080000, CRC(f19437d0) SHA1(75d07f4e5b124a823d14917ee4000df79e57a521) )
	sc_dnddd_others
ROM_END

ROM_START( sc4dndddd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427042.lo", 0x00001, 0x080000, CRC(7707c79d) SHA1(a1b44af3b70587a09d9786dc086f7d195e1654a1) )
	ROM_LOAD16_BYTE( "95427043.hi", 0x00000, 0x080000, CRC(2881c6f4) SHA1(aab514ce48ff7f4bc58cfb641612a53bb8f7a78d) )
	sc_dnddd_others
ROM_END

ROM_START( sc4dnddde )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427300.lo", 0x00001, 0x080000, CRC(ed013729) SHA1(9d532f99a21227707237ca067ff539a89b9aa7bf) )
	ROM_LOAD16_BYTE( "95427301.hi", 0x00000, 0x080000, CRC(e5c4a338) SHA1(3896ae61ea09e6d889011db8c2346efa5e03525d) )
	sc_dnddd_others
ROM_END

ROM_START( sc4dndddf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427408.lo", 0x00001, 0x080000, CRC(53425c8f) SHA1(f5aaa6e9e6f8e2e3d974e04ed5a433d907fa0e6d) )
	ROM_LOAD16_BYTE( "95427409.hi", 0x00000, 0x080000, CRC(0ffa7f69) SHA1(56279cfd9bd0f5e7303074b0a9da07c5d122fcfb) )
	sc_dnddd_others
ROM_END

ROM_START( sc4dndddg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427456.lo", 0x00001, 0x080000, CRC(7c5bb19f) SHA1(f5b0a88555114424e54c8fb6cf35b44af8a0adf2) )
	ROM_LOAD16_BYTE( "95427457.hi", 0x00000, 0x080000, CRC(b991be45) SHA1(97c17c23dc4068bbc942ddf8160f5037f9796dc2) )
	sc_dnddd_others
ROM_END



ROM_START( sc4dndhf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95422230.lo", 0x00001, 0x080000, CRC(46165367) SHA1(b6febfc3c9afa1597da251137e359e42e39fa5b7) )
	ROM_LOAD16_BYTE( "95422231.hi", 0x00000, 0x080000, CRC(bfa73daa) SHA1(d475fb98035ef91ef85d9695bdbbaf5a09582c4a) )
	sc_dndhf_others
ROM_END


ROM_START( sc4dndhfa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95423230.lo", 0x00001, 0x080000, CRC(e77d4d22) SHA1(81bddb29f9a5af8f675c07ab4786e28dda478390) )
	ROM_LOAD16_BYTE( "95423231.hi", 0x00000, 0x080000, CRC(e911cbcc) SHA1(6bf0cf7081f49adf082dfe5bc489941baf89b177) )
	sc_dndhf_others
ROM_END

ROM_START( sc4dndhfb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426112.lo", 0x00001, 0x080000, CRC(7169a030) SHA1(34048ecf1b9b154b94963830ce033ec081accc84) )
	ROM_LOAD16_BYTE( "95426113.hi", 0x00000, 0x080000, CRC(cffdf2f8) SHA1(8d7ae835f4a50fa8e6ba3f7c9ff7041547b1daaa) )
	sc_dndhf_others
ROM_END

ROM_START( sc4dndhfc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426338.lo", 0x00001, 0x080000, CRC(318a1f6a) SHA1(bef02e8cbaf957ab80e2c29a684490c9aa713244) )
	ROM_LOAD16_BYTE( "95426339.hi", 0x00000, 0x080000, CRC(74f30c4e) SHA1(122a8654a514b1214c56e1d4b2ed5215220951a2) )
	sc_dndhf_others
ROM_END

ROM_START( sc4dndhfd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427112.lo", 0x00001, 0x080000, CRC(56dfc7b2) SHA1(757257d9c9178d587b906840bccdfdfe81095d2c) )
	ROM_LOAD16_BYTE( "95427113.hi", 0x00000, 0x080000, CRC(51bb8182) SHA1(00cebd6109d82b4f3199d6eb21395060f246cc8b) )
	sc_dndhf_others
ROM_END

ROM_START( sc4dndhfe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427338.lo", 0x00001, 0x080000, CRC(c095e219) SHA1(cbed56490174460c882b3c1cc634eac1787bca6b) )
	ROM_LOAD16_BYTE( "95427339.hi", 0x00000, 0x080000, CRC(3221dc69) SHA1(376ce5802c47cdebe364432aa0254d48e077cb0f) )
	sc_dndhf_others
ROM_END

ROM_START( sc4dndhff )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p3038p71.hi", 0x00000, 0x080000, CRC(65c029f2) SHA1(4e6fe3c6994dcc0e110aad0db290ba5cd88b3909) )
	ROM_LOAD16_BYTE( "p3038p71.lo", 0x00001, 0x080000, CRC(ac32e361) SHA1(05aad77374366091d4d92938284e7b913962c022) )
	sc_dndhf_others
ROM_END

ROM_START( sc4dndhfg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p3038p72.hi", 0x00000, 0x080000, CRC(0378fbc2) SHA1(24358afa5c6048c35602525a282ba80fdf0868ec) )
	ROM_LOAD16_BYTE( "p3038p72.lo", 0x00001, 0x080000, CRC(56798d91) SHA1(67c5c7009a5455d1ee512ba80434793a7a9cd1d5) )
	sc_dndhf_others
ROM_END

ROM_START( sc4dndhfh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p3038s71.hi", 0x00000, 0x080000, CRC(a560026f) SHA1(612e25ede337af886f9df29bd9e0f0a01d2599cd) )
	ROM_LOAD16_BYTE( "p3038s71.lo", 0x00001, 0x080000, CRC(439af563) SHA1(791ecfa413407c4907bd90984289c745dc8adf4d) )
	sc_dndhf_others
ROM_END

ROM_START( sc4dndhfi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p3038s72.hi", 0x00000, 0x080000, CRC(8a730875) SHA1(ec2a73a6437f1c5bc9d2d6b64ef051765a686556) )
	ROM_LOAD16_BYTE( "p3038s72.lo", 0x00001, 0x080000, CRC(3742986a) SHA1(23cf803ac3a313fc8a5994ea95351c28bfb282d0) )
	sc_dndhf_others
ROM_END

ROM_START( sc4dndhfj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95423799.hi", 0x00000, 0x080000, CRC(922bf36c) SHA1(18bb9e38340dd0fd7e95fc0f70d0a47ef98f6d9c) )
	ROM_LOAD16_BYTE( "95423798.lo", 0x00001, 0x080000, CRC(310e7ab7) SHA1(8ec44acd16dbe22837d365970e496640d1250355) )
	sc_dndhf_others
ROM_END

ROM_START( sc4dndhfk )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95428082.lo", 0x00001, 0x080000, CRC(161690d0) SHA1(a9eb5d47b067897803814f7c5d5e3436002f278b) )
	ROM_LOAD16_BYTE( "95428083.hi", 0x00000, 0x080000, CRC(72520ead) SHA1(bbaeeaddc98530eb879596a4526d0101a7912d20) )
	sc_dndhf_others
ROM_END

ROM_START( sc4dndhfl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95429082.lo", 0x00001, 0x080000, CRC(5172b542) SHA1(f460fe62b12d709d528135603252951d306c64c9) )
	ROM_LOAD16_BYTE( "95429083.hi", 0x00000, 0x080000, CRC(26040dae) SHA1(9302efb1a92e5c31dc22d0a6b4cc51f26c4bf5ab) )
	sc_dndhf_others
ROM_END


ROM_START( sc4dndys )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426312.lo", 0x00001, 0x080000, CRC(3a13df56) SHA1(84770607327a16ad0f64588578ab423d41be7006) )
	ROM_LOAD16_BYTE( "95426313.hi", 0x00000, 0x080000, CRC(be2643d5) SHA1(64249cff5a31afb5b1e4e17dfdaa2404021c5321) )
	sc_dndys_others
ROM_END

ROM_START( sc4dndysa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427066.lo", 0x00001, 0x080000, CRC(4bdfccda) SHA1(ec72bc6fce4ac4c1074d928270f6018602fabd96) )
	ROM_LOAD16_BYTE( "95427067.hi", 0x00000, 0x080000, CRC(dc8c2384) SHA1(2a936c935730ce61d487659e0be72c7a2164bc73) )
	sc_dndys_others
ROM_END

ROM_START( sc4dndysb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427312.lo", 0x00001, 0x080000, CRC(28fca839) SHA1(b40f8b42f4832fd679499faae6a710b780d3ec27) )
	ROM_LOAD16_BYTE( "95427313.hi", 0x00000, 0x080000, CRC(76764fc0) SHA1(eb6d638e3c68a65abce45cd7b71f7da158b431fc) )
	sc_dndys_others
ROM_END

ROM_START( sc4dndysc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95426066.lo", 0x00001, 0x080000, CRC(9f388484) SHA1(abbf996d8dfca49302eada0d669477ceb6a74706) )
	ROM_LOAD16_BYTE( "95426067.hi", 0x00000, 0x080000, CRC(9f5164c4) SHA1(ed68a46a05bcca3ed1478b6a31ca59cd870ebf7e) )
	sc_dndys_others
ROM_END

ROM_START( sc4dndlp )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426342.lo", 0x00001, 0x080000, CRC(24a01b4a) SHA1(e3418d9c690897244deb4483b1472ed9b6557db3) )
	ROM_LOAD16_BYTE( "95426343.hi", 0x00000, 0x080000, CRC(f8382df6) SHA1(0cbbc0afefd2a61baad30cbdb916572b4d97ea7d) )
	sc_dndlp_others
ROM_END


ROM_START( sc4dndlpa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426458.lo", 0x00001, 0x080000, CRC(ed53f87f) SHA1(0ad1e2c1dc51591fd318294aa1a0be46533c27a6) )
	ROM_LOAD16_BYTE( "95426459.hi", 0x00000, 0x080000, CRC(63264e53) SHA1(07e5ccb9f68f273b74f65d22306352ce33338d34) )
	sc_dndlp_others
ROM_END

ROM_START( sc4dndlpb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426536.lo", 0x00001, 0x080000, CRC(3ee26d10) SHA1(73b9dd35f4c16ce9081c06f47de3c091f7162e0b) )
	ROM_LOAD16_BYTE( "95426537.hi", 0x00000, 0x080000, CRC(9fec9ab5) SHA1(e11c735a921987712d954f0f29434cfb4189be8b) )
	sc_dndlp_others
ROM_END

ROM_START( sc4dndlpc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427342.lo", 0x00001, 0x080000, CRC(230de91f) SHA1(05e39076a18be9848b04045c77e9dcb03b09fcca) )
	ROM_LOAD16_BYTE( "95427343.hi", 0x00000, 0x080000, CRC(0b1e35fe) SHA1(74e269c20aa8676e646285840f3563bd2248982c) )
	sc_dndlp_others
ROM_END

ROM_START( sc4dndlpd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427458.lo", 0x00001, 0x080000, CRC(41936151) SHA1(d68abd4cc7e340a3706b3a52bbe1a79f0ac07a79) )
	ROM_LOAD16_BYTE( "95427459.hi", 0x00000, 0x080000, CRC(87cc61b4) SHA1(d6d2cd815161f41921c4ec28f4259cd6f62bc995) )
	sc_dndlp_others
ROM_END

ROM_START( sc4dndlpe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427536.lo", 0x00001, 0x080000, CRC(e85cf664) SHA1(e35e1bc7818b483a8f8eb67547af2765aaa74213) )
	ROM_LOAD16_BYTE( "95427537.hi", 0x00000, 0x080000, CRC(a5483213) SHA1(b4b332ea7e1740ff07dc5565d1134137896b4640) )
	sc_dndlp_others
ROM_END



ROM_START( sc4dndra )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95422488.lo", 0x00001, 0x080000, CRC(8dd30c4f) SHA1(54705053681607c87b2aa6e3b6d2c9aed1c92030) )
	ROM_LOAD16_BYTE( "95422489.hi", 0x00000, 0x080000, CRC(48836820) SHA1(4063670cc6be1dab844b3de2ea994b799f38d1fe) )
	sc_dndra_others
ROM_END


ROM_START( sc4dndraa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95423488.lo", 0x00001, 0x080000, CRC(fffdb21e) SHA1(f9b4d3f8a3ab4b373eea1d296c3ce923fb95c8d8) )
	ROM_LOAD16_BYTE( "95423489.hi", 0x00000, 0x080000, CRC(b81bae78) SHA1(26824ba85a8462572dc4d94426ba87558e62970e) )
	sc_dndra_others
ROM_END


ROM_START( sc4dndrab )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426082.lo", 0x00001, 0x080000, CRC(77cbb34c) SHA1(75a76d05e6da46d4881e62c8ed95a27e87cf0566) )
	ROM_LOAD16_BYTE( "95426083.hi", 0x00000, 0x080000, CRC(83308070) SHA1(8eb9923095ab98fb416664c6854ddcbd4ec3888e) )
	sc_dndra_others
ROM_END


ROM_START( sc4dndrac )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426416.lo", 0x00001, 0x080000, CRC(35c794c9) SHA1(6ddadede71064ec75d6fc521df5cc9486029948a) )
	ROM_LOAD16_BYTE( "95426417.hi", 0x00000, 0x080000, CRC(ef7802cd) SHA1(6428370e3e5b950f55ae3685b9e82dd213fd92ee) )
	sc_dndra_others
ROM_END


ROM_START( sc4dndrad )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427082.lo", 0x00001, 0x080000, CRC(047d2776) SHA1(85e4210af45990e112324701dafcc3a83b342253) )
	ROM_LOAD16_BYTE( "95427083.hi", 0x00000, 0x080000, CRC(2e91181b) SHA1(fb881415d26a8d2b7f773188e6edffc9babaa582) )
	sc_dndra_others
ROM_END


ROM_START( sc4dndrae )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427416.lo", 0x00001, 0x080000, CRC(f025b0ba) SHA1(ad1bc747e3cb6ece6041fd5335f893148a330b67) )
	ROM_LOAD16_BYTE( "95427417.hi", 0x00000, 0x080000, CRC(4f15c978) SHA1(16ceb67160b3d83d58db2d4fcce2bb5d46fc02b7) )
	sc_dndra_others
ROM_END



ROM_START( sc4dndbd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95422896.lo", 0x00001, 0x080000, CRC(4490c973) SHA1(13b869e65b299de353e8e56049b990e1e7e6aeca) )
	ROM_LOAD16_BYTE( "95422897.hi", 0x00000, 0x080000, CRC(c884eebd) SHA1(fd9b194e8a7a4988abded1325e3ceb993a32edc4) )
	sc_dndbd_others
ROM_END


ROM_START( sc4dndbda )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95423896.lo", 0x00001, 0x080000, CRC(d7ca472f) SHA1(bc9f1ea75fe42a9a1462fc3b4755d6c11853bd21) )
	ROM_LOAD16_BYTE( "95423897.hi", 0x00000, 0x080000, CRC(fbd289b9) SHA1(7dedc2d8ac28ad313dd1688a01bda4cd74fc68b4) )
	sc_dndbd_others
ROM_END

ROM_START( sc4dndbdb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426124.lo", 0x00001, 0x080000, CRC(193b1d2f) SHA1(6c84b99acc231d25d79e8878a14ce01787c44980) )
	ROM_LOAD16_BYTE( "95426125.hi", 0x00000, 0x080000, CRC(266f5e60) SHA1(ad787ff57f7eb40a0f02d3daf573ad01d4fd5e7c) )
	sc_dndbd_others
ROM_END

ROM_START( sc4dndbdc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426302.lo", 0x00001, 0x080000, CRC(8d7a23fb) SHA1(0a9d746eb3f032d774294df9ad1bb0515be21397) )
	ROM_LOAD16_BYTE( "95426303.hi", 0x00000, 0x080000, CRC(5a758509) SHA1(733c8b82e6210e64be1a85a3295273ec0990a8d2) )
	sc_dndbd_others
ROM_END

ROM_START( sc4dndbdd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427124.lo", 0x00001, 0x080000, CRC(707491a9) SHA1(ad1c529bf133c22e0585094342951710d200ae89) )
	ROM_LOAD16_BYTE( "95427125.hi", 0x00000, 0x080000, CRC(4e8f2ced) SHA1(e804bf36dcbfd465af874134afa773a0c1add0fb) )
	sc_dndbd_others
ROM_END

ROM_START( sc4dndbde )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427302.lo", 0x00001, 0x080000, CRC(05659829) SHA1(cdd34e493563361d4a48099bc23c2a4157f29b8e) )
	ROM_LOAD16_BYTE( "95427303.hi", 0x00000, 0x080000, CRC(53dc417e) SHA1(a3feef757f97d0b393c57356734a49f72cdb8dfc) )
	sc_dndbd_others
ROM_END




ROM_START( sc4dndbr )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95422154.lo", 0x00001, 0x080000, CRC(2593d490) SHA1(3d3b4375c37eb8faa65304d874d54b16459ae504) )
	ROM_LOAD16_BYTE( "95422155.hi", 0x00000, 0x080000, CRC(5e8c0dc2) SHA1(eea1d5016870058e65c33cef7da944f9a7854837) )
	sc_dndbr_others
ROM_END

ROM_START( sc4dndbra )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95422322.lo", 0x00001, 0x080000, CRC(e353fb01) SHA1(b65f6d6e0b396a0e9fb939605c118d40676c6d97) )
	ROM_LOAD16_BYTE( "95422323.hi", 0x00000, 0x080000, CRC(27cc917f) SHA1(742e8d9b3a134f3d187febc747966ba378ac9877) )
	sc_dndbr_others
ROM_END

ROM_START( sc4dndbrb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95423154.lo", 0x00001, 0x080000, CRC(6c7bb586) SHA1(1c30a83f451cd168e06dc1249c071b66d01b2c43) )
	ROM_LOAD16_BYTE( "95423155.hi", 0x00000, 0x080000, CRC(5ee307c4) SHA1(df8b23ed617120c041fb5e52746b12aaa77083f7) )
	sc_dndbr_others
ROM_END

ROM_START( sc4dndbrc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95423322.lo", 0x00001, 0x080000, CRC(1c701b97) SHA1(d41eb80fd3f8a9525173a35199cbe7526b0898fe) )
	ROM_LOAD16_BYTE( "95423323.hi", 0x00000, 0x080000, CRC(09b53479) SHA1(3783163cb2a1502c8b59d230116c6d6c42fb356d) )
	sc_dndbr_others
ROM_END

ROM_START( sc4dndbrd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426058.lo", 0x00001, 0x080000, CRC(4d37ac05) SHA1(0bed1340613a52fa22811788d43cb54d61678d12) )
	ROM_LOAD16_BYTE( "95426059.hi", 0x00000, 0x080000, CRC(5b9ffcd7) SHA1(300f5236aec96ba814d01e7b914fa293939e24b4) )
	sc_dndbr_others
ROM_END

ROM_START( sc4dndbre )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426332.lo", 0x00001, 0x080000, CRC(cc96ab08) SHA1(660f1b1f04a8614a98eb61c824d2f77648e6496d) )
	ROM_LOAD16_BYTE( "95426333.hi", 0x00000, 0x080000, CRC(d1cb3f47) SHA1(ba090e2cc438927fc6156e58d7651c7ee97b9091) )
	sc_dndbr_others
ROM_END

ROM_START( sc4dndbrf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427058.lo", 0x00001, 0x080000, CRC(e3cf1d56) SHA1(265815afc3f535cc0433dcea9c541ce054159b8a) )
	ROM_LOAD16_BYTE( "95427059.hi", 0x00000, 0x080000, CRC(5cd8faac) SHA1(7a64d26423c274a9d0a243a3422a7bf37dc8d961) )
	sc_dndbr_others
ROM_END

ROM_START( sc4dndbrg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427332.lo", 0x00001, 0x080000, CRC(639ea20c) SHA1(d4cab4bfdeb843169a11355e34162677a045121b) )
	ROM_LOAD16_BYTE( "95427333.hi", 0x00000, 0x080000, CRC(654cfe7c) SHA1(316f201009343ecfa3397e97407e4f7566950a3a) )
	sc_dndbr_others
ROM_END


ROM_START( sc4dndcc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95422392.lo", 0x00001, 0x080000, CRC(e83c2bea) SHA1(ea4ed543476f637236e03249c493c10693ce27b1) )
	ROM_LOAD16_BYTE( "95422393.hi", 0x00000, 0x080000, CRC(ba88c94a) SHA1(0de684f3ac06899df45ecac9a79f13948f02af70) )
	sc_dndcc_others
ROM_END


ROM_START( sc4dndcca )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95422782.lo", 0x00001, 0x080000, CRC(6e038242) SHA1(e1e4799aaaee1007154490a3cade13567d599799) )
	ROM_LOAD16_BYTE( "95422783.hi", 0x00000, 0x080000, CRC(53f64815) SHA1(cd30f5d5ebeaa2c77ea9205c28ef65ecc5ed9c88) )
	sc_dndcc_others
ROM_END

ROM_START( sc4dndccb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95423392.lo", 0x00001, 0x080000, CRC(1284ed1a) SHA1(ccaa8abd4c0274d1187e58c16a553e8f7c960c87) )
	ROM_LOAD16_BYTE( "95423393.hi", 0x00000, 0x080000, NO_DUMP )
	sc_dndcc_others
ROM_END

ROM_START( sc4dndccc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426054.lo", 0x00001, 0x080000, CRC(0de246f9) SHA1(5d4b884c2539c8aa3b209289f9580b9308018801) )
	ROM_LOAD16_BYTE( "95426055.hi", 0x00000, 0x080000, CRC(29884b00) SHA1(261000bae35d3d48f085f5c6526523f5e076ba35) )
	sc_dndcc_others
ROM_END

ROM_START( sc4dndccd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426322.lo", 0x00001, 0x080000, CRC(133a2d50) SHA1(458dc9a1434a94f022826f24ee929b2a707b56ab) )
	ROM_LOAD16_BYTE( "95426323.hi", 0x00000, 0x080000, CRC(b8d80583) SHA1(dac5c0ef75dae496803ae274ac28ac904b54ab0e) )
	sc_dndcc_others
ROM_END

ROM_START( sc4dndcce )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427054.lo", 0x00001, 0x080000, CRC(30397fe0) SHA1(f8181e08dd7b68e07706bd0b14ff126f4f97aff5) )
	ROM_LOAD16_BYTE( "95427055.hi", 0x00000, 0x080000, CRC(40a88ac1) SHA1(a2212609e27df356b68ab595da3bde789be88280) )
	sc_dndcc_others
ROM_END

ROM_START( sc4dndccf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427322.lo", 0x00001, 0x080000, CRC(3338519c) SHA1(95aefe3c8c688917bde07d9965ff647dbf7280ac) )
	ROM_LOAD16_BYTE( "95427323.hi", 0x00000, 0x080000, CRC(93dc915a) SHA1(cefd62f72d322a7c5729857bec05fc240bdae853) )
	sc_dndcc_others
ROM_END






ROM_START( sc4dnddw )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95422474.lo", 0x00001, 0x080000, CRC(2d263524) SHA1(202f0a095b4a10979105dfa1e189c8eb0f2fb18e) )
	ROM_LOAD16_BYTE( "95422475.hi", 0x00000, 0x080000, CRC(a8ca5e00) SHA1(308f0d38a70e26a9f55731957206c2c11c41a5b8) )
	sc_dnddw_others
ROM_END

ROM_START( sc4dnddwa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95422860.lo", 0x00001, 0x080000, CRC(76bd11a3) SHA1(10f54c15c56fb1350eba3608301a0cb4c370ac3f) )
	ROM_LOAD16_BYTE( "95422861.hi", 0x00000, 0x080000, CRC(d33408b1) SHA1(773b97fc78c13df084f85cfa21cb0da156392631) )
	sc_dnddw_others
ROM_END

ROM_START( sc4dnddwb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95423474.lo", 0x00001, 0x080000, CRC(c0bb2d73) SHA1(7782e69fbdedede4dc625441ce428db922964527) )
	ROM_LOAD16_BYTE( "95423475.hi", 0x00000, 0x080000, CRC(9d33a1d6) SHA1(d78f271521ccf8d89b26b10a3772077caa0267c1) )
	sc_dnddw_others
ROM_END

ROM_START( sc4dnddwc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95423860.lo", 0x00001, 0x080000, CRC(4772372f) SHA1(61479cec9bf0a407670f37d0c81430f26d49d546) )
	ROM_LOAD16_BYTE( "95423861.hi", 0x00000, 0x080000, CRC(78fa784f) SHA1(5891fe0c32a988e7ac36492a33dcd2521b928dc0) )
	sc_dnddw_others
ROM_END

ROM_START( sc4dnddwd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426176.lo", 0x00001, 0x080000, CRC(005ac134) SHA1(d3921e5c6856bd9d7410cd91bff134db2e48361b) )
	ROM_LOAD16_BYTE( "95426177.hi", 0x00000, 0x080000, CRC(6202ee1f) SHA1(17e317e47d4f6b2542c12d6646b8e58b6ece8394) )
	sc_dnddw_others
ROM_END

ROM_START( sc4dnddwe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426418.lo", 0x00001, 0x080000, CRC(5300fde3) SHA1(5bc0ed8507a9c154dc92732fc86f2e12de462eda) )
	ROM_LOAD16_BYTE( "95426419.hi", 0x00000, 0x080000, CRC(c616445b) SHA1(842592e30aa3c94e26348c0f6a3bfaa3d1fc441f) )
	sc_dnddw_others
ROM_END

ROM_START( sc4dnddwf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427176.lo", 0x00001, 0x080000, CRC(579ffa4c) SHA1(b6ef21dbdb4255db29fa9b901fde26ddf01cd823) )
	ROM_LOAD16_BYTE( "95427177.hi", 0x00000, 0x080000, CRC(3f822d83) SHA1(0f2bf3769e7c61b7f1a7c77bd06dfccf64929a57) )
	sc_dnddw_others
ROM_END

ROM_START( sc4dnddwg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427418.lo", 0x00001, 0x080000, CRC(52c3f9f3) SHA1(d1af94901702bde90ddd99aed37a8e6adea13eb9) )
	ROM_LOAD16_BYTE( "95427419.hi", 0x00000, 0x080000, CRC(1f32ae61) SHA1(d6172884ca7a8837c40f40b0a89a4886de1e05fc) )
	sc_dnddw_others
ROM_END





ROM_START( sc4dnddf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95422286.lo", 0x00001, 0x080000, CRC(f39fd151) SHA1(a9464702f2c2219b19653b3ffd16f788b25ee36a) )
	ROM_LOAD16_BYTE( "95422287.hi", 0x00000, 0x080000, CRC(d4cccb65) SHA1(8fabfccfdb04a5583b7aeb9a93678fdc530c324c) )
	sc_dnddf_others
ROM_END


ROM_START( sc4dnddfa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95423286.lo", 0x00001, 0x080000, NO_DUMP )
	ROM_LOAD16_BYTE( "95423287.hi", 0x00000, 0x080000, CRC(76a0bf55) SHA1(01500a0951dcff87ecb5f0d5741e8a6127a7c40b) )
	sc_dnddf_others
ROM_END

ROM_START( sc4dnddfb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426062.lo", 0x00001, 0x080000, CRC(fdf23ab5) SHA1(cb3107c7b47a2563fef817e72d29d7cbef229587) )
	ROM_LOAD16_BYTE( "95426063.hi", 0x00000, 0x080000, CRC(254e02f9) SHA1(5eb0945b209fb04c7fda394a31ad739d12e4fae6) )
	sc_dnddf_others
ROM_END

ROM_START( sc4dnddfc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426328.lo", 0x00001, 0x080000, CRC(f90c2141) SHA1(ac04d077773c1dc308fcd46b7aaa740d0b90e327) )
	ROM_LOAD16_BYTE( "95426329.hi", 0x00000, 0x080000, CRC(ab0ab0c2) SHA1(6d4ee13533bed1260a67227055584e531a211e5b) )
	sc_dnddf_others
ROM_END

ROM_START( sc4dnddfd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427062.lo", 0x00001, 0x080000, CRC(4830608e) SHA1(fc4e5dbfbde09e42a0e806a19ba020c16125fbc9) )
	ROM_LOAD16_BYTE( "95427063.hi", 0x00000, 0x080000, CRC(340810fd) SHA1(08ffaf1cdaaafceb9a40f6550848abc4098c040e) )
	sc_dnddf_others
ROM_END

ROM_START( sc4dnddfe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427328.lo", 0x00001, 0x080000, CRC(8c1f3034) SHA1(6d8c57ca65837fc55a6c089743b3ab496668e424) )
	ROM_LOAD16_BYTE( "95427329.hi", 0x00000, 0x080000, CRC(f6247ac6) SHA1(910a44ae4540582dad41e368f75f48b2f134e642) )
	sc_dnddf_others
ROM_END



ROM_START( sc4dndpg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426254.lo", 0x00001, 0x080000, CRC(110cf29d) SHA1(98b42fc9024ce41e301136725da4732f1ae5577a) )
	ROM_LOAD16_BYTE( "95426255.hi", 0x00000, 0x080000, CRC(bbc13ad8) SHA1(ff3bc9369c74c5e82466363c8a2c1f80ba88d47e) )
	sc_dndpg_others
ROM_END

ROM_START( sc4dndpga )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426340.lo", 0x00001, 0x080000, CRC(351c1470) SHA1(ff43accf90b4c3f062b73537715cb13b26c3f35a) )
	ROM_LOAD16_BYTE( "95426341.hi", 0x00000, 0x080000, CRC(45d767a4) SHA1(0b607e04a43cba6b930dbc5bc97f38e3d02630f7) )
	sc_dndpg_others
ROM_END

ROM_START( sc4dndpgb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426448.lo", 0x00001, 0x080000, CRC(0d0329e4) SHA1(53a4ee777865f9d5e99e82e5406b62a5ebcfe2c6) )
	ROM_LOAD16_BYTE( "95426449.hi", 0x00000, 0x080000, CRC(89a88879) SHA1(61a3eb35caad027bb0255110a171ca1316ce9571) )
	sc_dndpg_others
ROM_END

ROM_START( sc4dndpgc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427254.lo", 0x00001, 0x080000, CRC(fb9b8e54) SHA1(49188caffe06bc268a8b8c18c4afc076d2b72b8f) )
	ROM_LOAD16_BYTE( "95427255.hi", 0x00000, 0x080000, CRC(74c2c2fc) SHA1(de4863f8a7e9e41655f16f09a38c16f2dc865646) )
	sc_dndpg_others
ROM_END

ROM_START( sc4dndpgd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427340.lo", 0x00001, 0x080000, CRC(54ab16c7) SHA1(e3117bf540c4854bb594c27e5de7c590b89c4530) )
	ROM_LOAD16_BYTE( "95427341.hi", 0x00000, 0x080000, CRC(66807abf) SHA1(b75d9cfd7f5e3a7c2407ad7f61daf79ec5a50564) )
	sc_dndpg_others
ROM_END

ROM_START( sc4dndpge )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427448.lo", 0x00001, 0x080000, CRC(db667b80) SHA1(145fd4b7d05eef01bb1d5bb457f809ec8cfff021) )
	ROM_LOAD16_BYTE( "95427449.hi", 0x00000, 0x080000, CRC(a1efbe17) SHA1(3b62e18751a95f39fae8b5a63d9e798b6bad4a3f) )
	sc_dndpg_others
ROM_END






ROM_START( sc4dndtp )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95418706.lo", 0x00001, 0x080000, CRC(2b785234) SHA1(32ff01a1a64d4caa4d539d2e09189fffc635b382) )
	ROM_LOAD16_BYTE( "95418707.hi", 0x00000, 0x080000, CRC(d7c370ff) SHA1(00e49802ba0deb375072057eadacfdac749f6928) )
	sc_dndtp_others
ROM_END

ROM_START( sc4dndtpa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95418730.lo", 0x00001, 0x080000, CRC(435b531f) SHA1(9959786e7d193d8d02438e9e76a8bc068d3175e0) )
	ROM_LOAD16_BYTE( "95418731.hi", 0x00000, 0x080000, CRC(9f5ba2e5) SHA1(1699b4fced4e8118e122022bbcbe3e3c37fa6bba) )
	sc_dndtp_others
ROM_END

ROM_START( sc4dndtpb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95418784.lo", 0x00001, 0x080000, CRC(49a401ea) SHA1(0f0c0b7decd64d5b06d5451691f4bc24ec071c7b) )
	ROM_LOAD16_BYTE( "95418785.hi", 0x00000, 0x080000, CRC(9e2714a1) SHA1(4ff3fda8da04e5971acf2f6491fb6d2a261494b4) )
	sc_dndtp_others
ROM_END

ROM_START( sc4dndtpc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95418872.lo", 0x00001, 0x080000, CRC(fbebbf50) SHA1(2bbabf6dde06ad2f58d030df0b6b5d91b2805604) )
	ROM_LOAD16_BYTE( "95418873.hi", 0x00000, 0x080000, CRC(d3089997) SHA1(6958596a60473861e44aebc48732aeb6a8b11c21) )
	sc_dndtp_others
ROM_END

ROM_START( sc4dndtpd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95418944.lo", 0x00001, 0x080000, CRC(12a79908) SHA1(0c420fb3318ee74e1b0e9a10f0617fff26dbbb2c) )
	ROM_LOAD16_BYTE( "95418945.hi", 0x00000, 0x080000, CRC(1de9312b) SHA1(6878a48bc516dc0b63c7c62b57603e28224460b2) )
	sc_dndtp_others
ROM_END

ROM_START( sc4dndtpe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95419706.lo", 0x00001, 0x080000, CRC(bd8b7ca9) SHA1(ea2e441d9cd3676d8fdd095c7debb561a1059b4e) )
	ROM_LOAD16_BYTE( "95419707.hi", 0x00000, 0x080000, CRC(26df2479) SHA1(6f842e5b72a3c3b91135a60c52be8e54f53309d0) )
	sc_dndtp_others
ROM_END

ROM_START( sc4dndtpf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95419730.lo", 0x00001, 0x080000, CRC(644d007a) SHA1(76ea3e81c53a9db058a386f685d1a534671be2f7) )
	ROM_LOAD16_BYTE( "95419731.hi", 0x00000, 0x080000, CRC(2cc387fd) SHA1(d1cb8774eebf1b623b2dab55654b16c2f7f396b7) )
	sc_dndtp_others
ROM_END

ROM_START( sc4dndtpg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95419784.lo", 0x00001, 0x080000, CRC(c6fd42bd) SHA1(33edc9bcd7d45295a4bc830dc8b8ef817b9751e5) )
	ROM_LOAD16_BYTE( "95419785.hi", 0x00000, 0x080000, CRC(413f47b9) SHA1(0bef7b7fdfb43103cacd9f91b00a4a2fead81f70) )
	sc_dndtp_others
ROM_END

ROM_START( sc4dndtph )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95419872.lo", 0x00001, 0x080000, CRC(e910d376) SHA1(1c225979c8a27bfc9ae244c4aca2184553b040c7) )
	ROM_LOAD16_BYTE( "95419873.hi", 0x00000, 0x080000, CRC(afb8116a) SHA1(3f4416cdcd30a53ad8da12f54089e9c09f98cfac) )
	sc_dndtp_others
ROM_END

ROM_START( sc4dndtpi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95419944.lo", 0x00001, 0x080000, CRC(09446373) SHA1(21c15a082a20294e2b086c74f2da4cb17b052c2c) )
	ROM_LOAD16_BYTE( "95419945.hi", 0x00000, 0x080000, CRC(4499476a) SHA1(261d393b03d2238053ffcf1783cb4a3a6e733ed7) )
	sc_dndtp_others
ROM_END

ROM_START( sc4dndtpj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426142.lo", 0x00001, 0x080000, CRC(776b28c2) SHA1(f6cca110df664dadf0529540678340bc8072c3eb) )
	ROM_LOAD16_BYTE( "95426143.hi", 0x00000, 0x080000, CRC(52253aa7) SHA1(4eae55f63841afefec79c411054d74123f614cc8) )
	sc_dndtp_others
ROM_END

ROM_START( sc4dndtpk )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427142.lo", 0x00001, 0x080000, CRC(adbf6808) SHA1(4cc1ed6d692107ff22c199fba708d52418259cd9) )
	ROM_LOAD16_BYTE( "95427143.hi", 0x00000, 0x080000, CRC(d961af03) SHA1(04f2f2d424257996c496f762403647307c499bd5) )
	sc_dndtp_others
ROM_END

ROM_START( sc4dndtpl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95422006.lo", 0x00001, 0x080000, CRC(78228f0d) SHA1(811cde456cb5dbd53c4cace3033c22c549de8f99) )
	ROM_LOAD16_BYTE( "95422007.hi", 0x00000, 0x080000, CRC(7ae82867) SHA1(60c02d9abc05e1655eef71d5ddfc9b8e6dd3c4aa) )
	sc_dndtp_others
ROM_END

ROM_START( sc4dndtpm )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95423006.lo", 0x00001, 0x080000, CRC(cc8d9bb5) SHA1(0be48fed507ab933590c6a6baa434071d0ddfb60) )
	ROM_LOAD16_BYTE( "95423007.hi", 0x00000, 0x080000, CRC(1858f222) SHA1(679f6c35e9953caf738660c67c372d9bd09e086b) )
	sc_dndtp_others
ROM_END




ROM_START( sc4dndtr )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426468.lo", 0x00001, 0x080000, CRC(dcde303d) SHA1(c885a428d188d66e2cf61b28f3e9c71e3dc61053) )
	ROM_LOAD16_BYTE( "95426469.hi", 0x00000, 0x080000, CRC(175e6a70) SHA1(847f21612b42b50b0788e6980e458312749afbad) )
	sc_dndtr_others
ROM_END


ROM_START( sc4dndtra )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427468.lo", 0x00001, 0x080000, CRC(127334f0) SHA1(c851ce8b355cfb51e6288f117ee41dba7e48969d) )
	ROM_LOAD16_BYTE( "95427469.hi", 0x00000, 0x080000, CRC(537f91aa) SHA1(b63b9ed4d21225f0d979107d2d03a49b6d7ae2a4) )
	sc_dndtr_others
ROM_END



ROM_START( sc4dndww )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95418862.lo", 0x00001, 0x080000, CRC(fbba7955) SHA1(189a9681232ef808d749060eb31aef964e0f5479) )
	ROM_LOAD16_BYTE( "95418863.hi", 0x00000, 0x080000, CRC(7fee6bca) SHA1(b6bded7923d5c00929259f30537d665dbb62867c) )
	sc_dndww_others
ROM_END


ROM_START( sc4dndwwa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95418996.lo", 0x00001, 0x080000, CRC(0d7d5a31) SHA1(bf4db52a1af006b625269f1dafe29f2b76642d04) )
	ROM_LOAD16_BYTE( "95418997.hi", 0x00000, 0x080000, CRC(419a89db) SHA1(015883e4a44067a6d092f7042eb3ef536f4eadb8) )
	sc_dndww_others
ROM_END

ROM_START( sc4dndwwb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95419862.lo", 0x00001, 0x080000, CRC(8e28ba13) SHA1(3a762f4eeedf36249ba1b81d7601033dc59c63ab) )
	ROM_LOAD16_BYTE( "95419863.hi", 0x00000, 0x080000, CRC(99bf979f) SHA1(60d233eb1186787b2525e19fdf825eb489a6db8e) )
	sc_dndww_others
ROM_END

ROM_START( sc4dndwwc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426138.lo", 0x00001, 0x080000, CRC(da9454eb) SHA1(2a1b21b114cc584cc64a57ab59e88eb5274b2432) )
	ROM_LOAD16_BYTE( "95426139.hi", 0x00000, 0x080000, CRC(e96e0f34) SHA1(9233794df825b1a201ee5fa1fb40329cc1c41b73) )
	sc_dndww_others
ROM_END


ROM_START( sc4dndwwd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427138.lo", 0x00001, 0x080000, CRC(a9928e67) SHA1(10c99f2ab3f25e55788633e57f7155e06897a42b) )
	ROM_LOAD16_BYTE( "95427139.hi", 0x00000, 0x080000, CRC(191eed85) SHA1(9896f3e4c24fd649868454f141fd67b7e12ca822) )
	sc_dndww_others
ROM_END

ROM_START( sc4dndwwe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95419996.lo", 0x00001, 0x080000, CRC(524dfc58) SHA1(e7e01c40f01442888a575c874d1423ee8c179849) )
	ROM_LOAD16_BYTE( "95419997.hi", 0x00000, 0x080000, CRC(183005fb) SHA1(78aef9641ff2e4517b546f6316eb6ead5032bc0e) )
	sc_dndww_others
ROM_END


ROM_START( sc4dndcw )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95422294.lo", 0x00001, 0x080000, CRC(f02056ce) SHA1(6de6720064fce6bc52445a33a327493c54c64cb2) )
	ROM_LOAD16_BYTE( "95422295.hi", 0x00000, 0x080000, CRC(408331c2) SHA1(6790fbbe41c92201fbdd4e21f2611a9152543353) )
	sc_dndww_others
ROM_END


ROM_START( sc4dndcwa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95422714.lo", 0x00001, 0x080000, CRC(3ae68439) SHA1(f28c59c59516534ae52acf3ff34fb369ae2e76f4) )
	ROM_LOAD16_BYTE( "95422715.hi", 0x00000, 0x080000, CRC(3ea32328) SHA1(b35cd4b9cdd411706eedc888cc25064e234bf5ed) )
	sc_dndww_others
ROM_END


ROM_START( sc4dndcwb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95423294.lo", 0x00001, 0x080000, CRC(31e3f048) SHA1(3584f654dc180524c35db0cae6d57e8f93bf8d7f) )
	ROM_LOAD16_BYTE( "95423295.hi", 0x00000, 0x080000, CRC(179fa1c3) SHA1(8647a23209c3ec4534b4b08a9d205797d64598df) )
	sc_dndww_others
ROM_END


ROM_START( sc4dndcwc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95423714.lo", 0x00001, 0x080000, CRC(98913f7f) SHA1(30c863b96c3ef7d37f34344354a5efae5f3be80e) )
	ROM_LOAD16_BYTE( "95423715.hi", 0x00000, 0x080000, CRC(eaf9cefe) SHA1(c3bff3c5204ce65e3ed9257962ccdc6196920590) )
	sc_dndww_others
ROM_END



ROM_START( sc4dndwb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95418586.lo", 0x00001, 0x080000, CRC(be0491f5) SHA1(19b5d4f0c19932b550bbecc5090eda4874681e39) )
	ROM_LOAD16_BYTE( "95418587.hi", 0x00000, 0x080000, CRC(dd05943c) SHA1(993c3bd14e9924101e0beae5e6c101163dcd7787) )
	sc_dndwb_others
ROM_END


ROM_START( sc4dndwba )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95418942.lo", 0x00001, 0x080000, CRC(9990ffa1) SHA1(8adb93f05eef8f149103d40d89d93ddb034a91a8) )
	ROM_LOAD16_BYTE( "95418943.hi", 0x00000, 0x080000, CRC(43116ed8) SHA1(1698afb7be73b5bd9702603a740953c6987fef4e) )
	sc_dndwb_others
ROM_END

ROM_START( sc4dndwbb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95419586.lo", 0x00001, 0x080000, CRC(6d1c95d7) SHA1(5326ed6076b83f400fecda5e48b0c89ec275232d) )
	ROM_LOAD16_BYTE( "95419587.hi", 0x00000, 0x080000, CRC(5a702fe0) SHA1(ac6004a2466bbbabe456e2a41e98096a1b37c71c) )
	sc_dndwb_others
ROM_END

ROM_START( sc4dndwbc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95419942.lo", 0x00001, 0x080000, CRC(d12c7227) SHA1(d8d99d9500cdeea4d5b0f304e4e6d4a0df5994af) )
	ROM_LOAD16_BYTE( "95419943.hi", 0x00000, 0x080000, CRC(091f96dc) SHA1(0011c0dbee6ca76cf84e81417156a7339e850075) )
	sc_dndwb_others
ROM_END

ROM_START( sc4dndwbd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95422132.lo", 0x00001, 0x080000, CRC(8f4cd776) SHA1(61ee48942cc597bc2c962be08d95e35652fdcbad) )
	ROM_LOAD16_BYTE( "95422133.hi", 0x00000, 0x080000, CRC(6ea4be95) SHA1(a77791df59db5e269452fea09f4ef8f73b967bd6) )
	sc_dndwb_others
ROM_END

ROM_START( sc4dndwbe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95423132.lo", 0x00001, 0x080000, CRC(54c15f7e) SHA1(9ef7ea85ec528d229dada944b588b3fca4db04ad) )
	ROM_LOAD16_BYTE( "95423133.hi", 0x00000, 0x080000, CRC(909530bf) SHA1(d04e6653a4664e5ca20c80eee597781cc99d6726) )
	sc_dndwb_others
ROM_END

ROM_START( sc4dndwbf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426146.lo", 0x00001, 0x080000, CRC(bd780980) SHA1(22f90f83fbf7286aa5b6c40a6a53af22104b6dec) )
	ROM_LOAD16_BYTE( "95426147.hi", 0x00000, 0x080000, CRC(dc30a7b7) SHA1(e5cd6d452226946a1a683177ff0695e606dccc43) )
	sc_dndwb_others
ROM_END

ROM_START( sc4dndwbg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427146.lo", 0x00001, 0x080000, CRC(2c98681c) SHA1(b616d2546804182b0d41c14fa3133f58224ee8cb) )
	ROM_LOAD16_BYTE( "95427147.hi", 0x00000, 0x080000, CRC(486bc407) SHA1(9dca9bb55ffb090fa8a59da2df0ebba26a5adb5d) )
	sc_dndwb_others
ROM_END




ROM_START( sc4dmine )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406818.lo", 0x00001, 0x080000, CRC(5e2e861e) SHA1(0feaffd721a8e0ad9443e6d0e3f03a52c26dcb7c) )
	ROM_LOAD16_BYTE( "95406819.hi", 0x00000, 0x080000, CRC(6b2a7eeb) SHA1(090f3c0c1c3a3dcf7c90f4a87173e56ff5c2218d) )
	sc_dmine_others
ROM_END

ROM_START( sc4dminea )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407818.lo", 0x00001, 0x080000, CRC(d5f7fd35) SHA1(d281937d8bc1499c9554e8037421ad55284a0621) )
	ROM_LOAD16_BYTE( "95407819.hi", 0x00000, 0x080000, CRC(5f58b617) SHA1(d2acb87838bcd4c109ed1a8d9e460757b7365ef5) )
	sc_dmine_others
ROM_END

ROM_START( sc4dmineb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408096.lo", 0x00001, 0x080000, CRC(9c9f8a16) SHA1(39792c7a83abf490d246323a46e19d3f1c85128d) )
	ROM_LOAD16_BYTE( "95408097.hi", 0x00000, 0x080000, CRC(008dad9d) SHA1(76ead1488b30877e765d5cce21297d40ca090fdd) )
	sc_dmine_others
ROM_END

ROM_START( sc4dminec )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409096.lo", 0x00001, 0x080000, CRC(62ba6707) SHA1(9f80511e34e5ece4c95923bb504164abbd0facdc) )
	ROM_LOAD16_BYTE( "95409097.hi", 0x00000, 0x080000, CRC(2f21f8f2) SHA1(02a3d26eda63fe2dff8cf19bb6612cae0958b847) )
	sc_dmine_others
ROM_END

ROM_START( sc4dmined )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1307p11.hi", 0x00000, 0x080000, CRC(41721838) SHA1(4d08a48bd60ee1ed615aed6af48a0392881a15de) )
	ROM_LOAD16_BYTE( "p1307p11.lo", 0x00001, 0x080000, CRC(cf36f797) SHA1(1d538d3fd9f59b37ac2a60e67f1ad2fca6b29d01) )
	sc_dmine_others
ROM_END

ROM_START( sc4dminee )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1307s11.hi", 0x00000, 0x080000, CRC(177a69e8) SHA1(14b49bb00b120f024d08296e489a470cf258956a) )
	ROM_LOAD16_BYTE( "p1307s11.lo", 0x00001, 0x080000, CRC(ad8678be) SHA1(d16197c807aec0bd61481be427cd4dd37236962b) )
	sc_dmine_others
ROM_END


ROM_START( sc4ddosh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408184.lo", 0x00001, 0x080000, CRC(2125d1b3) SHA1(56ff3ac713d8da3ebac4ebe2e342a60ef665bbec) )
	ROM_LOAD16_BYTE( "95408185.hi", 0x00000, 0x080000, CRC(0ad032e9) SHA1(e6af4938b8b3ea3e25bc5ab426d6506e1ac72df6) )
	sc_ddosh_others
ROM_END

ROM_START( sc4ddosha )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408960.lo", 0x00001, 0x080000, CRC(f7953cd6) SHA1(fc20c51460397b57e976abcf5491f257619dc81f) )
	ROM_LOAD16_BYTE( "95408961.hi", 0x00000, 0x080000, CRC(c340ac9d) SHA1(c08e257f79bec391b5fe642a5609078d64efdc25) )
	sc_ddosh_others
ROM_END

ROM_START( sc4ddoshb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408962.lo", 0x00001, 0x080000, CRC(4e01227d) SHA1(28d9307a4eb5d573ad969d4afe1b630738971b9d) )
	ROM_LOAD16_BYTE( "95408963.hi", 0x00000, 0x080000, CRC(0be07c6c) SHA1(2f1271d3be31ec9fd9fd216ec6a6b7b21e9cbcc1) )
	sc_ddosh_others
ROM_END

ROM_START( sc4ddoshc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409184.lo", 0x00001, 0x080000, CRC(5f521844) SHA1(713a333ab64c93e807f1cdd11680087d9b6992bd) )
	ROM_LOAD16_BYTE( "95409185.hi", 0x00000, 0x080000, CRC(ed2034f4) SHA1(dfadea956683a6d5ec3c9c58d21a331051a7d7ec) )
	sc_ddosh_others
ROM_END

ROM_START( sc4ddoshd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409960.lo", 0x00001, 0x080000, CRC(9ded56aa) SHA1(939c3c2ee623c9d54706ec5a6fe00cc9eb716ab7) )
	ROM_LOAD16_BYTE( "95409961.hi", 0x00000, 0x080000, CRC(53341c67) SHA1(7dd692935564ed9de7c87e371822a7da6e609667) )
	sc_ddosh_others
ROM_END

ROM_START( sc4ddoshe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409962.lo", 0x00001, 0x080000, CRC(884c4711) SHA1(9f2f252605b5fbcd6e468558c120c066155e1bf2) )
	ROM_LOAD16_BYTE( "95409963.hi", 0x00000, 0x080000, CRC(12942dbd) SHA1(a97416b0fe2d434a8df071e73839f53d865c4d9e) )
	sc_ddosh_others
ROM_END

ROM_START( sc4ddoshf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410554.lo", 0x00001, 0x080000, CRC(548d63de) SHA1(b08a7b0df804077d3d21b81cd94fbd6fc07fc586) )
	ROM_LOAD16_BYTE( "95410555.hi", 0x00000, 0x080000, CRC(56333835) SHA1(803944338f3c4ebc909776fe0631fc7053222f5f) )
	sc_ddosh_others
ROM_END

ROM_START( sc4ddoshg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410556.lo", 0x00001, 0x080000, CRC(b802d16c) SHA1(eef08384549d02730b4aeab84e0db419366d7fad) )
	ROM_LOAD16_BYTE( "95410557.hi", 0x00000, 0x080000, CRC(bc129faf) SHA1(2dde04dc0ea7023c17162c05d35b62ffccce6fdf) )
	sc_ddosh_others
ROM_END

ROM_START( sc4ddoshh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411554.lo", 0x00001, 0x080000, CRC(182a5bf0) SHA1(649aab4636e530006b8dbd695f4d74d0d6c9baf6) )
	ROM_LOAD16_BYTE( "95411555.hi", 0x00000, 0x080000, CRC(0ec186c9) SHA1(4c014089ef942efe67de667b14ec66c19476cc90) )
	sc_ddosh_others
ROM_END

ROM_START( sc4ddoshi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411556.lo", 0x00001, 0x080000, CRC(970e1df8) SHA1(b8d614fb50ee472ff42d6fc4135b70948567cc8b) )
	ROM_LOAD16_BYTE( "95411557.hi", 0x00000, 0x080000, CRC(5778044b) SHA1(09a09f666b65de59cde8f2a5ad9ac52297ad4d7a) )
	sc_ddosh_others
ROM_END

ROM_START( sc4ddoshj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412434.lo", 0x00001, 0x080000, CRC(00b23cd4) SHA1(0d2257006b8fdac23caab8dbc696ce131b50f6f3) )
	ROM_LOAD16_BYTE( "95412435.hi", 0x00000, 0x080000, CRC(86ab1cbc) SHA1(e9dbd1421274f5a024187063edd4a138b0835899) )
	sc_ddosh_others
ROM_END

ROM_START( sc4ddoshk )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413434.lo", 0x00001, 0x080000, CRC(b4f57152) SHA1(fcca12a12b7ddd4021828938546c9118bdbbf557) )
	ROM_LOAD16_BYTE( "95413435.hi", 0x00000, 0x080000, CRC(394fba09) SHA1(0fcd4c6b6dade2f2f902c6244c303cb1e344f855) )
	sc_ddosh_others
ROM_END

ROM_START( sc4ddoshl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1309p12.hi", 0x00000, 0x080000, CRC(3a256edb) SHA1(ed4821063e9e717e73875cba0742ca695c402cba) )
	ROM_LOAD16_BYTE( "p1309p12.lo", 0x00001, 0x080000, CRC(e8cc8f24) SHA1(5f5ceaafcefe36e74b9548c4289889cac5914bba) )
	sc_ddosh_others
ROM_END

ROM_START( sc4ddoshm )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1309s12.hi", 0x00000, 0x080000, CRC(a6dbb7b3) SHA1(596655bd858645b242185d00a730c608cba7b519) )
	ROM_LOAD16_BYTE( "p1309s12.lo", 0x00001, 0x080000, CRC(e4dc9264) SHA1(03d01247555a58ed4e0ed048144f615f6a33d351) )
	sc_ddosh_others
ROM_END

ROM_START( sc4ddoshn )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1309p11.hi", 0x00000, 0x080000, CRC(558e6289) SHA1(9dc2247ade861776dc265ac394fe23e1d0a33186) )
	ROM_LOAD16_BYTE( "p1309p11.lo", 0x00001, 0x080000, CRC(9859cdfc) SHA1(7ac807df1f3f303296bd663466b2fbf8c37e330c) )
	sc_ddosh_others
ROM_END

ROM_START( sc4ddosho )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1309s11.hi", 0x00000, 0x080000, CRC(bc0034c4) SHA1(1ce19865e1058710b63403dd9add1a553df460cd) )
	ROM_LOAD16_BYTE( "p1309s11.lo", 0x00001, 0x080000, CRC(79f196ae) SHA1(cf37eb02b2a0ec8ba6482ae33f604b41eae23671) )
	sc_ddosh_others
ROM_END


ROM_START( sc4dhh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95418234.lo", 0x00001, 0x080000, CRC(e54495af) SHA1(3113c0831fed8b1e79a23eefb943389b761174ce) )
	ROM_LOAD16_BYTE( "95418235.hi", 0x00000, 0x080000, CRC(bd8d114b) SHA1(f243831139332b4a4f82cc68494078bee06f8f69) )
	sc_dhh_others
ROM_END

ROM_START( sc4dhha )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95418494.lo", 0x00001, 0x080000, CRC(5b9ae752) SHA1(52fad5c24d205b2e401cd87de89aabfca1e99564) )
	ROM_LOAD16_BYTE( "95418495.hi", 0x00000, 0x080000, CRC(8116da15) SHA1(8057d1536e4647a68eed384060a0ecd23b7d9aba) )
	sc_dhh_others
ROM_END

ROM_START( sc4dhhb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95419234.lo", 0x00001, 0x080000, CRC(d6ebf891) SHA1(33a62ea3e82400e5be27dd007e2c744a357b301a) )
	ROM_LOAD16_BYTE( "95419235.hi", 0x00000, 0x080000, CRC(b0017541) SHA1(18c8d5ab7bea6b229dcbc525aed827f747bc9243) )
	sc_dhh_others
ROM_END

ROM_START( sc4dhhc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95419494.lo", 0x00001, 0x080000, CRC(d0e4b7e3) SHA1(e1f1a17fdef63163770de4505675e70d369327c0) )
	ROM_LOAD16_BYTE( "95419495.hi", 0x00000, 0x080000, CRC(acce7f23) SHA1(3489826d5b8e049838e05679a982e5ed1c8a5cec) )
	sc_dhh_others
ROM_END

ROM_START( sc4dhhd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426230.lo", 0x00001, 0x080000, CRC(af4df562) SHA1(71c6d0b4cf3e9972e947d183b04bc3ea95c78373) )
	ROM_LOAD16_BYTE( "95426231.hi", 0x00000, 0x080000, CRC(613b3a22) SHA1(20c1a67112f3a681b1a5236b68c29a9185308ee3) )
	sc_dhh_others
ROM_END

ROM_START( sc4dhhe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427230.lo", 0x00001, 0x080000, CRC(d46701ec) SHA1(af827fd9368bb480f29f04c452c4c2cf33e120ba) )
	ROM_LOAD16_BYTE( "95427231.hi", 0x00000, 0x080000, CRC(ed311139) SHA1(c0a1fafc3000faf5659f224080671103b3a1665a) )
	sc_dhh_others
ROM_END




ROM_START( sc4dough )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95418044.lo", 0x00001, 0x080000, CRC(c25b15cd) SHA1(85e509a83b443a415480baf74299ee7d7cc58215) )
	ROM_LOAD16_BYTE( "95418045.hi", 0x00000, 0x080000, CRC(76f2652b) SHA1(52bedca861f582d30eb9960f1be70a8392179a34) )
	sc_dough_others
ROM_END

ROM_START( sc4dougha )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95419044.lo", 0x00001, 0x080000, CRC(281fff41) SHA1(3d709f68a2a06d4e57c3ceb34abd8c80740f09f4) )
	ROM_LOAD16_BYTE( "95419045.hi", 0x00000, 0x080000, CRC(37134542) SHA1(77a485ed9c311c83d504d376a07425ece9612b8b) )
	sc_dough_others
ROM_END


ROM_START( sc4ducks )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416442.lo", 0x00001, 0x080000, CRC(d113671a) SHA1(a6f8249138d9f38645daefd150fe213b564b24e1) )
	ROM_LOAD16_BYTE( "95416443.hi", 0x00000, 0x080000, CRC(66b1264c) SHA1(9df605167b69f40f0c4feb8ef2a4d7e760faf2e0) )
	sc_ducks_others
ROM_END

ROM_START( sc4ducksa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416444.lo", 0x00001, 0x080000, CRC(8d0f6ea6) SHA1(5fafaba8931327f57bd0952ed69fac5cc8a8b3d6) )
	ROM_LOAD16_BYTE( "95416445.hi", 0x00000, 0x080000, CRC(56dfa592) SHA1(99eb7a50c9372d1ce3bd1da80106b3e85a2f3c23) )
	sc_ducks_others
ROM_END

ROM_START( sc4ducksb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417442.lo", 0x00001, 0x080000, CRC(21c3aa78) SHA1(7cbb26353e6292392fc4ef0ab3c405cb5788f177) )
	ROM_LOAD16_BYTE( "95417443.hi", 0x00000, 0x080000, CRC(4b70fc11) SHA1(04de2fe94df00f2befaa36b2a2dd79142ace3503) )
	sc_ducks_others
ROM_END

ROM_START( sc4ducksc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417444.lo", 0x00001, 0x080000, CRC(01af39ba) SHA1(92730609807f919dff98dbf1aadd983686d43bbf) )
	ROM_LOAD16_BYTE( "95417445.hi", 0x00000, 0x080000, CRC(b0e04c87) SHA1(9350d7e6d9cf8a4cece3f70b3a7f58ab0c8b4dc4) )
	sc_ducks_others
ROM_END


ROM_START( sc4emmer )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420072.lo", 0x00001, 0x080000, CRC(95baff43) SHA1(b5cefff519282cada89155ef25665f78960149cf) )
	ROM_LOAD16_BYTE( "95420073.hi", 0x00000, 0x080000, CRC(f35cb8e7) SHA1(65e56c481b2df75dbfa3febe9988fe7f1d858b04) )
	sc_emmer_others
ROM_END

ROM_START( sc4emmera )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420074.lo", 0x00001, 0x080000, CRC(04abd6e7) SHA1(e5625b7004bb1870f4f07d44abe87f7371ab85f5) )
	ROM_LOAD16_BYTE( "95420075.hi", 0x00000, 0x080000, CRC(fa8ef4c2) SHA1(71af90541143ccc9a9a8e41a38959a13121255f8) )
	sc_emmer_others
ROM_END

ROM_START( sc4emmerb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421072.lo", 0x00001, 0x080000, CRC(38be125f) SHA1(6de5be192dfb9c6efde712da833a2fbaad896ea4) )
	ROM_LOAD16_BYTE( "95421073.hi", 0x00000, 0x080000, CRC(ca709361) SHA1(556677e27e3164fcf3ac119be5276cac34256451) )
	sc_emmer_others
ROM_END

ROM_START( sc4emmerc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421074.lo", 0x00001, 0x080000, CRC(ea40002d) SHA1(da3949e45bbf7eeba432b72492f156f4dece8582) )
	ROM_LOAD16_BYTE( "95421075.hi", 0x00000, 0x080000, CRC(028ae706) SHA1(5a74859ac7ee5985fcf96d493017b4c8a012a0ac) )
	sc_emmer_others
ROM_END



ROM_START( sc4evol )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410180.lo", 0x00001, 0x080000, CRC(af8ac67c) SHA1(668381e48400a7ad8186212ef00e9881824baf82) )
	ROM_LOAD16_BYTE( "95410181.hi", 0x00000, 0x080000, CRC(ebeb4589) SHA1(1909a92c1e29b2424480aeb21650543da91089cb) )
	sc_evol_others
ROM_END

ROM_START( sc4evola )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410182.lo", 0x00001, 0x080000, CRC(54fefbd7) SHA1(c181e65766e11c397a5376c1abbc6b285fda853a) )
	ROM_LOAD16_BYTE( "95410183.hi", 0x00000, 0x080000, CRC(e8d7216a) SHA1(c93239f759781a8ba854ed19fe6cd328f022c17c) )
	sc_evol_others
ROM_END

ROM_START( sc4evolb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410804.lo", 0x00001, 0x080000, CRC(dac7d3ec) SHA1(918f353d8685b7808da211ec86cc3f456c6141ee) )
	ROM_LOAD16_BYTE( "95410805.hi", 0x00000, 0x080000, CRC(afcd340c) SHA1(91659ded6d0643b3c9b9061a1f08e8f110f548c2) )
	sc_evol_others
ROM_END

ROM_START( sc4evolc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410806.lo", 0x00001, 0x080000, CRC(6207f59d) SHA1(c9a8acb118a3e23a6e8edb7b664b94f68d61c58b) )
	ROM_LOAD16_BYTE( "95410807.hi", 0x00000, 0x080000, CRC(f9cc95ae) SHA1(5f4805e6f45462baf3a8fc920ae8988b6d744f78) )
	sc_evol_others
ROM_END

ROM_START( sc4evold )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411180.lo", 0x00001, 0x080000, CRC(c4f794fb) SHA1(a32f4baa20c8d2d62f09a54e849ddf18ec59bedc) )
	ROM_LOAD16_BYTE( "95411181.hi", 0x00000, 0x080000, CRC(9cbc06a3) SHA1(0687b66effefa68655e5cd58aee58f43a90f33c4) )
	sc_evol_others
ROM_END

ROM_START( sc4evole )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411182.lo", 0x00001, 0x080000, CRC(3f83a950) SHA1(44f21c6a93d15bec6d146d387ea87672358d9d31) )
	ROM_LOAD16_BYTE( "95411183.hi", 0x00000, 0x080000, CRC(9f806240) SHA1(b0678feec7187089ba684e0f736e39491f89fea9) )
	sc_evol_others
ROM_END

ROM_START( sc4evolf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411804.lo", 0x00001, 0x080000, CRC(cd30c648) SHA1(1d3467f96947e62547ad90a762c82887c96b1e20) )
	ROM_LOAD16_BYTE( "95411805.hi", 0x00000, 0x080000, CRC(fba51709) SHA1(69aad603b0bd97cfa0845743570be943f4e9d493) )
	sc_evol_others
ROM_END

ROM_START( sc4evolg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411806.lo", 0x00001, 0x080000, CRC(dd2fa51a) SHA1(81f549f4361f1ad49adc3f8823eab1d417fdce94) )
	ROM_LOAD16_BYTE( "95411807.hi", 0x00000, 0x080000, CRC(4a54b0b6) SHA1(26473624772a8d32b28fc23609daf1c1a5fe3ff7) )
	sc_evol_others
ROM_END

ROM_START( sc4fguy )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426016.lo", 0x00001, 0x080000, CRC(d1a13822) SHA1(9275ee0a1db886bffc7c4390bb3e7408b8e9cd93) )
	ROM_LOAD16_BYTE( "95426017.hi", 0x00000, 0x080000, CRC(0c4616e1) SHA1(f1b460aacbe1ae996f80419b6995e81580bf4876) )
	sc_fguy_others
ROM_END

ROM_START( sc4fguya )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426410.lo", 0x00001, 0x080000, CRC(25cfd853) SHA1(2328165ff0a7752dba37189740a462e9aba0ea2a) )
	ROM_LOAD16_BYTE( "95426411.hi", 0x00000, 0x080000, CRC(c5714aed) SHA1(ae29257ebcf1ddeadb41302b4a448142baa88ad2) )
	sc_fguy_others
ROM_END

ROM_START( sc4fguyb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427016.lo", 0x00001, 0x080000, CRC(6587831c) SHA1(34c1a7498ae82f54fdd68350c86077e6c5232c03) )
	ROM_LOAD16_BYTE( "95427017.hi", 0x00000, 0x080000, CRC(47a8cd87) SHA1(baad39e3fab06d61da9dd725e176c920a5f0657f) )
	sc_fguy_others
ROM_END

ROM_START( sc4fguyc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427410.lo", 0x00001, 0x080000, CRC(af8d7e22) SHA1(22100e1e3588880f9de1bd92348221aae6e3fdc9) )
	ROM_LOAD16_BYTE( "95427411.hi", 0x00000, 0x080000, CRC(0c41bf21) SHA1(a704ed754b7bc62fa3bb75ebc789fe7990e206ec) )
	sc_fguy_others
ROM_END

ROM_START( sc4fguyd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95426070.lo", 0x00001, 0x080000, CRC(fa164ebd) SHA1(b50b261ea503b1f56b99d692805fb90ca67e22e6) )
	ROM_LOAD16_BYTE( "95426071.hi", 0x00000, 0x080000, CRC(304dc4dc) SHA1(045f639ffae4ac54e6554ca5c9a11516f0561e64) )
	ROM_REGION( 0x400000, "ymz", ROMREGION_ERASE00)
ROM_END

ROM_START( sc4fguye )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95427070.lo", 0x00001, 0x080000, CRC(916479f9) SHA1(d9f9a48c671ba046469ddcdcdfbf6de0c7ec703d) )
	ROM_LOAD16_BYTE( "95427071.hi", 0x00000, 0x080000, CRC(85306584) SHA1(f844c345330b5d0983dacb693aa5bacaf5fd60b5) )
	ROM_REGION( 0x400000, "ymz", ROMREGION_ERASE00)
ROM_END

ROM_START( sc4fbspn )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95418098.lo", 0x00001, 0x080000, CRC(525427ef) SHA1(839b52b81d53aa4d0607e274bd575ed2887a0f92) )
	ROM_LOAD16_BYTE( "95418099.hi", 0x00000, 0x080000, CRC(746a44c1) SHA1(5d70aeea4e430ae5d063c7c85b634a8009a67ac1) )
	sc_fbspn_others
ROM_END

ROM_START( sc4fbspna )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95419098.lo", 0x00001, 0x080000, CRC(9edb2064) SHA1(5ee2ba64c99acc92e6d6748451de6def478e38ea) )
	ROM_LOAD16_BYTE( "95419099.hi", 0x00000, 0x080000, CRC(1fdc7007) SHA1(998fd6c328387e13bbf15852f588687f3313c428) )
	sc_fbspn_others
ROM_END

ROM_START( sc4fbspnb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95422200.lo", 0x00001, 0x080000, CRC(1fe607e8) SHA1(0142f4c0d0deee37453158208aed4c3cb0cccc36) )
	ROM_LOAD16_BYTE( "95422201.hi", 0x00000, 0x080000, CRC(0f155036) SHA1(7f294f1ba28d3eeca6898d3cd83692b8cca2bc3c) )
	sc_fbspn_others
ROM_END

ROM_START( sc4fbspnc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95423200.lo", 0x00001, 0x080000, CRC(8a0eb98c) SHA1(fadd69f1af668307f7728e92e282acd3ae8859a5) )
	ROM_LOAD16_BYTE( "95423201.hi", 0x00000, 0x080000, CRC(379e95d8) SHA1(3c677a80b17617c4f1eb8f5ae683b807f16c039d) )
	sc_fbspn_others
ROM_END




ROM_START( sc4fmj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420270.lo", 0x00001, 0x080000, CRC(4c2bd704) SHA1(edf227733a887025f64181410796666fbdbe6f7c) )
	ROM_LOAD16_BYTE( "95420271.hi", 0x00000, 0x080000, CRC(d131fedd) SHA1(d80cda766aab890b933b5573dacfa1e9ebfd44b6) )
	sc_fmj_others
ROM_END

ROM_START( sc4fmja )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420272.lo", 0x00001, 0x080000, CRC(477e538f) SHA1(db7ed34aa111162fe85090ca06a3d80db44b93af) )
	ROM_LOAD16_BYTE( "95420273.hi", 0x00000, 0x080000, CRC(94480a10) SHA1(6d16c74de9dd369814918a589b602651d8db6f5d) )
	sc_fmj_others
ROM_END

ROM_START( sc4fmjb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421270.lo", 0x00001, 0x080000, CRC(c0357d4e) SHA1(a7a5090bb597c6558eacae4dbcd39243af1b2897) )
	ROM_LOAD16_BYTE( "95421271.hi", 0x00000, 0x080000, CRC(a3444399) SHA1(f5544f9e09ba70316735f5f6b7a9353e43846793) )
	sc_fmj_others
ROM_END

ROM_START( sc4fmjc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421272.lo", 0x00001, 0x080000, CRC(ed8b3914) SHA1(304e1592b09702771623a110be1ea12f5bba93d4) )
	ROM_LOAD16_BYTE( "95421273.hi", 0x00000, 0x080000, CRC(7ff83544) SHA1(1bc94725775bbd04792137bf79d02a84f2ba444b) )
	sc_fmj_others
ROM_END



ROM_START( sc4glad )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416726.lo", 0x00001, 0x080000, CRC(a2c85e44) SHA1(81e719ab0b29dcf264dbdd6d67f0d0f95de62a67) )
	ROM_LOAD16_BYTE( "95416727.hi", 0x00000, 0x080000, CRC(2a54392c) SHA1(9ad1874cce0ac5437a2652b03229700aae366d82) )
	sc_glad_others
ROM_END

ROM_START( sc4glada )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416728.lo", 0x00001, 0x080000, CRC(d64caa35) SHA1(7bc5e51a46e6b2b7dcc3f759b9522a98fe6bd82c) )
	ROM_LOAD16_BYTE( "95416729.hi", 0x00000, 0x080000, CRC(d35bc8ed) SHA1(7c174a180df7c39bb93de750c2a6238fd18a750e) )
	sc_glad_others
ROM_END

ROM_START( sc4gladb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417726.lo", 0x00001, 0x080000, CRC(58b916ae) SHA1(4c0e47512802b758e164b6ad6b5d8bf2db2b8e2f) )
	ROM_LOAD16_BYTE( "95417727.hi", 0x00000, 0x080000, CRC(ccd8738e) SHA1(0a9dce84f97ba3fa97415c154cbaf1fe87ee969f) )
	sc_glad_others
ROM_END

ROM_START( sc4gladc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417728.lo", 0x00001, 0x080000, CRC(659080a7) SHA1(480077c90fba6bc473e3082dd5971f9bd6df4c7d) )
	ROM_LOAD16_BYTE( "95417729.hi", 0x00000, 0x080000, CRC(8f577431) SHA1(21e50eb9a1bda0a5f6895a57009bf69dcab50cd3) )
	sc_glad_others
ROM_END

ROM_START( sc4gladd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420536.lo", 0x00001, 0x080000, CRC(62e06fe4) SHA1(d052d6706d48c3b53a1cc9186f4d9505ac76b0b9) )
	ROM_LOAD16_BYTE( "95420537.hi", 0x00000, 0x080000, CRC(0250dc89) SHA1(6ff351e0448643fe9ec64abae3b0afa90e042b41) )
	sc_glad_others
ROM_END

ROM_START( sc4glade )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420538.lo", 0x00001, 0x080000, CRC(b4e9c9ea) SHA1(d92fd73d55ad015e3e828750fb73e860feccada6) )
	ROM_LOAD16_BYTE( "95420539.hi", 0x00000, 0x080000, CRC(5a023a25) SHA1(012649e0273bc35a654778d847c612c78212a102) )
	sc_glad_others
ROM_END

ROM_START( sc4gladf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421536.lo", 0x00001, 0x080000, CRC(b73ada13) SHA1(0f6546d798c01b6ffa2ccb1374db93a80d616e2e) )
	ROM_LOAD16_BYTE( "95421537.hi", 0x00000, 0x080000, CRC(3e8f8b50) SHA1(5b076205191e388204c0c7e62bc9b6c3798c3164) )
	sc_glad_others
ROM_END

ROM_START( sc4gladg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421538.lo", 0x00001, 0x080000, CRC(ee50532a) SHA1(35a28ab419e8a5178a8a253c9d5121617073088f) )
	ROM_LOAD16_BYTE( "95421539.hi", 0x00000, 0x080000, CRC(e47d28f0) SHA1(3eab678f79b602da07c1491c75fc08020c394e8e) )
	sc_glad_others
ROM_END

ROM_START( sc4gd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404358.lo", 0x00001, 0x080000, CRC(03e321ba) SHA1(a30645b2a001033f75ce99c28c36ed2f095edc03) )
	ROM_LOAD16_BYTE( "95404359.hi", 0x00000, 0x080000, CRC(b297dc83) SHA1(e8afce70a181fb951d3b75cd4d0c5feb08e2d0ab) )
	sc_gd_others
ROM_END

ROM_START( sc4gda )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404360.lo", 0x00001, 0x080000, CRC(4985a746) SHA1(f45c805b5a0e2acdc14e0079d1d56762e072ca5a) )
	ROM_LOAD16_BYTE( "95404361.hi", 0x00000, 0x080000, CRC(a813558e) SHA1(cfd1560dfc03e5023f2ae0474f87c0090d716472) )
	sc_gd_others
ROM_END

ROM_START( sc4gdb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405358.lo", 0x00001, 0x080000, CRC(9801b0bb) SHA1(e188b41ccde297597311f678d9a7ac7954835de1) )
	ROM_LOAD16_BYTE( "95405359.hi", 0x00000, 0x080000, CRC(dc2faf38) SHA1(d47e886d593e6849675e119e97ac935bc3d7a23a) )
	sc_gd_others
ROM_END

ROM_START( sc4gdc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405360.lo", 0x00001, 0x080000, CRC(7b4870ed) SHA1(d3eb89520c834cab36ca5c2b92653f1d1d218e36) )
	ROM_LOAD16_BYTE( "95405361.hi", 0x00000, 0x080000, CRC(22e4f4ab) SHA1(379cd01acdfa49f10bb90aeaad288ec1a02b3d5c) )
	sc_gd_others
ROM_END

ROM_START( sc4gdd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406024.lo", 0x00001, 0x080000, CRC(f09c2a2a) SHA1(6b194e51f7aa5620e0b561ad5d0a2eb9057ab0d4) )
	ROM_LOAD16_BYTE( "95406025.hi", 0x00000, 0x080000, CRC(8e3ac428) SHA1(6bb7d6d660e8121d1db260180ab164066fd73475) )
	sc_gd_others
ROM_END

ROM_START( sc4gde )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406496.lo", 0x00001, 0x080000, CRC(b8a36ee5) SHA1(1350aea8c3f9ca2365dea3d19254208ea0226b42) )
	ROM_LOAD16_BYTE( "95406497.hi", 0x00000, 0x080000, CRC(7abb8c9b) SHA1(776ec56ef3f33098ec61c8aa02ce15b7ed2dc3b8) )
	sc_gd_others
ROM_END

ROM_START( sc4gdf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407024.lo", 0x00001, 0x080000, CRC(c05f10d2) SHA1(7fc4165734a3cbf5f4cd77831bf1956d76712fc3) )
	ROM_LOAD16_BYTE( "95407025.hi", 0x00000, 0x080000, CRC(e1f946b8) SHA1(735111ea681d52ab2fa0152347314b194fbbb9c7) )
	sc_gd_others
ROM_END

ROM_START( sc4gdg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407496.lo", 0x00001, 0x080000, CRC(b9843bb7) SHA1(007c0409de40e4d400c00cbe361d2010f13ee446) )
	ROM_LOAD16_BYTE( "95407497.hi", 0x00000, 0x080000, CRC(c88253cb) SHA1(b51e8104695e84db4af9ce4bfab7cc5a90470657) )
	sc_gd_others
ROM_END


ROM_START( sc4gdmz )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414404.lo", 0x00001, 0x080000, CRC(0d68a264) SHA1(d84550a601778f135cea158cfe274ee409ec27d8) )
	ROM_LOAD16_BYTE( "95414405.hi", 0x00000, 0x080000, CRC(7b0da5f0) SHA1(3cabfeebfdc707cfe75c8ef72db741b33abc7ac6) )
	sc_gdmz_others
ROM_END

ROM_START( sc4gdmza )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414406.lo", 0x00001, 0x080000, CRC(04c88d78) SHA1(0a5d26caa8b3861b228ac39c7cc512e4f6359234) )
	ROM_LOAD16_BYTE( "95414407.hi", 0x00000, 0x080000, CRC(b1b5b7e4) SHA1(aef0e8a3d688d22e5465551ddb28c0b0105b6f58) )
	sc_gdmz_others
ROM_END

ROM_START( sc4gdclb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406460.lo", 0x00001, 0x080000, CRC(ffadb4bb) SHA1(6bf47af974b3d7eab99c54f92228b8ee60c92261) )
	ROM_LOAD16_BYTE( "95406461.hi", 0x00000, 0x080000, CRC(70f5e1c3) SHA1(c5bbad6856f7e52827098f8fa7b70658af3741fe) )
	sc_gdclb_others
ROM_END

ROM_START( sc4gdclba )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406462.lo", 0x00001, 0x080000, CRC(621fb28b) SHA1(4ce482a1e31d195cbcabe41c0d5a38f532fe7375) )
	ROM_LOAD16_BYTE( "95406463.hi", 0x00000, 0x080000, CRC(180706d1) SHA1(748b28df7f6860785097b887370f4bb95b23bb7e) )
	sc_gdclb_others
ROM_END

ROM_START( sc4gdclbb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407460.lo", 0x00001, 0x080000, CRC(9c25e8c1) SHA1(ba2780002a72805836c46e9f40a323a2b5c93b0a) )
	ROM_LOAD16_BYTE( "95407461.hi", 0x00000, 0x080000, CRC(e132397f) SHA1(d1e0581467fb5277721472ed3dc808da863bea8f) )
	sc_gdclb_others
ROM_END

ROM_START( sc4gdclbc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407462.lo", 0x00001, 0x080000, CRC(80fb6588) SHA1(4c4aa0d4ca780eb801409dd76d65230827717d1a) )
	ROM_LOAD16_BYTE( "95407463.hi", 0x00000, 0x080000, CRC(ddf6cc93) SHA1(c341c465f8c78d0df78e94d31169eb90cc169e78) )
	sc_gdclb_others
ROM_END

ROM_START( sc4gdclbd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408210.lo", 0x00001, 0x080000, CRC(0718c80a) SHA1(aff944fe848744519151abfdd88558c2c1494e57) )
	ROM_LOAD16_BYTE( "95408211.hi", 0x00000, 0x080000, CRC(0e72bbe9) SHA1(928b8ff011e1954e223e7b58635ae002e74e243b) )
	sc_gdclb_others
ROM_END

ROM_START( sc4gdclbe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409210.lo", 0x00001, 0x080000, CRC(b0e98730) SHA1(e4cb43b59a39410f999e4f2a2886e7c36b2b8543) )
	ROM_LOAD16_BYTE( "95409211.hi", 0x00000, 0x080000, CRC(e7fa2ed5) SHA1(c64377bf4964cfb6c1cadfa28059bdcc56c94e51) )
	sc_gdclb_others
ROM_END

ROM_START( sc4gdclbf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412200.lo", 0x00001, 0x080000, CRC(f57f37bd) SHA1(bc0d0e448f08e00a94468ffe1a6bdb2b5808cb47) )
	ROM_LOAD16_BYTE( "95412201.hi", 0x00000, 0x080000, CRC(17a6fa5e) SHA1(305ca3f40fa2cb2dfe7461baf19b7077149d943c) )
	sc_gdclb_others
ROM_END

ROM_START( sc4gdclbg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412842.lo", 0x00001, 0x080000, CRC(a4676445) SHA1(bb34b9d3453254b4d19f5f13c2dca65e9b7fede3) )
	ROM_LOAD16_BYTE( "95412843.hi", 0x00000, 0x080000, CRC(21912d54) SHA1(50bbc4adf70468b079835b4e210d6ff648db95ff) )
	sc_gdclb_others
ROM_END

ROM_START( sc4gdclbh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413200.lo", 0x00001, 0x080000, CRC(245188b5) SHA1(36ead091cc28d070ce306ca3c9216491c709e2b3) )
	ROM_LOAD16_BYTE( "95413201.hi", 0x00000, 0x080000, CRC(367b94a9) SHA1(6df95d32eb447a351098548da76314300bf8ff5c) )
	sc_gdclb_others
ROM_END

ROM_START( sc4gdclbi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413842.lo", 0x00001, 0x080000, CRC(b5418f2a) SHA1(7d806eddfb3e9a1aed29d1962d48c96656c929d0) )
	ROM_LOAD16_BYTE( "95413843.hi", 0x00000, 0x080000, CRC(163f4700) SHA1(2513bf9e306f2eafc48d29a2d79f7c26d6d384b3) )
	sc_gdclb_others
ROM_END

ROM_START( sc4gdclbj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1137p11.hi", 0x00000, 0x080000, CRC(8213a2c4) SHA1(a547fc96a53800643bddc66b6f596baacf888a92) )
	ROM_LOAD16_BYTE( "p1137p11.lo", 0x00001, 0x080000, CRC(686d17fb) SHA1(3c6da76aa4872b1a5bf09e984fbac0cfdf4f6264) )
	sc_gdclb_others
ROM_END

ROM_START( sc4gdclbk )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1137s11.hi", 0x00000, 0x080000, CRC(252c2257) SHA1(6b5acdc869d1eb535711016c1840f6ff9acde8aa) )
	ROM_LOAD16_BYTE( "p1137s11.lo", 0x00001, 0x080000, CRC(9316e977) SHA1(4ef00e2ae9ca533a684dbd625924eca8fec56c4f) )
	sc_gdclb_others
ROM_END



ROM_START( sc4gbcas )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404206.lo", 0x00001, 0x080000, CRC(bb853407) SHA1(498aad2cf2948d7fbef673186823af6a7dbefd95) )
	ROM_LOAD16_BYTE( "95404207.hi", 0x00000, 0x080000, CRC(e4e3f4e0) SHA1(ab8a8eeb5661d6cd3ba5d25f2082c06ac8c343ac) )
	PR1034_GOLDEN_BALLS_SOUNDS11
ROM_END

ROM_START( sc4gbcasa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404208.lo", 0x00001, 0x080000, CRC(fc080cdf) SHA1(86084d70e6a68afe3f788661b78cfb8df3bb8104) )
	ROM_LOAD16_BYTE( "95404209.hi", 0x00000, 0x080000, CRC(6e4d776f) SHA1(0e09b78e138215ceb693bc86a3fe6d7320da9431) )
	PR1034_GOLDEN_BALLS_SOUNDS11
ROM_END

ROM_START( sc4gbcasb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405206.lo", 0x00001, 0x080000, CRC(79f07bc2) SHA1(e85ddf4c18573894ecd51fec283cc46becb8aacc) )
	ROM_LOAD16_BYTE( "95405207.hi", 0x00000, 0x080000, CRC(2c295913) SHA1(a8bf658c6ec248e7f53485ce778e9679eae62504) )
	PR1034_GOLDEN_BALLS_SOUNDS11
ROM_END

ROM_START( sc4gbcasc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95405208.lo", 0x00001, 0x080000, CRC(3e7d431a) SHA1(b7975bbbf369d2eb6b8fad5c7200e0007ce2365b) )
	ROM_LOAD16_BYTE( "95405209.hi", 0x00000, 0x080000, CRC(a687da9c) SHA1(99cbc48fe2c9c28fb3b68107402af10c2a7f39fa) )
	PR1034_GOLDEN_BALLS_SOUNDS11
ROM_END

ROM_START( sc4gball )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412492.lo", 0x00001, 0x080000, CRC(963ccd4c) SHA1(4d6991080ed4dcd3affce900ee7d467319100d6a) )
	ROM_LOAD16_BYTE( "95412493.hi", 0x00000, 0x080000, CRC(b259c913) SHA1(a2bf15be684bcb762937e4af49fc1139ade8ea34) )
	sc_gball_pthers
ROM_END

ROM_START( sc4gballa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412612.lo", 0x00001, 0x080000, CRC(612afa6d) SHA1(886fd0df386abbc1609792d3e0e35b6554693c14) )
	ROM_LOAD16_BYTE( "95412613.hi", 0x00000, 0x080000, CRC(2e2826d2) SHA1(f4b20ae1758e55a16cab6338d0f712212776d8dd) )
	sc_gball_pthers
ROM_END

ROM_START( sc4gballb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413492.lo", 0x00001, 0x080000, CRC(3123302d) SHA1(61aea9ee55dd906bd17ed3181c5ac252eadbbc03) )
	ROM_LOAD16_BYTE( "95413493.hi", 0x00000, 0x080000, CRC(2924e1e3) SHA1(ee78f5bf6f69dc5e8186f699bd52a3c19f986e73) )
	sc_gball_pthers
ROM_END

ROM_START( sc4gballc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413612.lo", 0x00001, 0x080000, CRC(1b48a527) SHA1(9f740ddbeef6f3a0f11d3f79f0893faa25bffe29) )
	ROM_LOAD16_BYTE( "95413613.hi", 0x00000, 0x080000, CRC(f40d7d2b) SHA1(ef6fc79fa39240291d26b6e99c6d78630b282464) )
	sc_gball_pthers
ROM_END


ROM_START( sc4ggtb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95404762.lo", 0x00001, 0x080000, CRC(613941b4) SHA1(4cc1441b94913d597aa88e973f4514f7278a7cc9) )
	ROM_LOAD16_BYTE( "95404763.hi", 0x00000, 0x080000, CRC(dbec56ec) SHA1(bb8f2395afe8d5193320d627ef722c1105b02826) )
	sc_ggame_others
ROM_END

ROM_START( sc4ggtba )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408086.lo", 0x00001, 0x080000, CRC(cd0b1ea9) SHA1(ebb5023afa24876d97417db50aa675ec29c0351b) )
	ROM_LOAD16_BYTE( "95408087.hi", 0x00000, 0x080000, CRC(0a21efb4) SHA1(5765431cfc61521d55e00d7fdd2e08f0ef37d12c) )
	sc_ggame_others
ROM_END

ROM_START( sc4ggtbb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414970.lo", 0x00001, 0x080000, CRC(ab09550b) SHA1(004f858e5bcda93653efa511f66d33cbe20a44d1) )
	ROM_LOAD16_BYTE( "95414971.hi", 0x00000, 0x080000, CRC(e137aff0) SHA1(50a2a6819034ca5b04c704da7cfaf029bdc031b2) )
	sc_ggame_others
ROM_END



ROM_START( sc4ggamea )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406768.lo", 0x00001, 0x080000, CRC(dec1db93) SHA1(b22b874e5ff5da46e2783544781d7387f98dd4fc) )
	ROM_LOAD16_BYTE( "95406769.hi", 0x00000, 0x080000, CRC(1f47e9d3) SHA1(88bf64e5e7e7bcbba2cc24ac8b9421b38b6b060e) )
	sc_ggame_others
ROM_END

ROM_START( sc4ggameb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407768.lo", 0x00001, 0x080000, CRC(4f1398fa) SHA1(3c3c9710330dfe7af06daa44bf6ba8efdc14eacc) )
	ROM_LOAD16_BYTE( "95407769.hi", 0x00000, 0x080000, CRC(8039c58d) SHA1(ac0fc21e93daff06d383bba3f309867b054aafc4) )
	sc_ggame_others
ROM_END

ROM_START( sc4ggamec )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407770.lo", 0x00001, 0x080000, CRC(94da2109) SHA1(6bf1bc1b916678c727d64070d334e9418f3be95c) )
	ROM_LOAD16_BYTE( "95407771.hi", 0x00000, 0x080000, CRC(dff20663) SHA1(3f3525c45602220ba1bd8599ae9f5e4cc249899a) )
	sc_ggame_others
ROM_END

ROM_START( sc4ggamed )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407772.lo", 0x00001, 0x080000, CRC(bb731d03) SHA1(f56dd26c93cb5fad4e7b5d43ad7395be35482b1f) )
	ROM_LOAD16_BYTE( "95407773.hi", 0x00000, 0x080000, CRC(7148b4ca) SHA1(caf979f86fea994fb0d304c8c88920aba89ca25e) )
	sc_ggame_others
ROM_END


ROM_START( sc4ggamef )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410406.lo", 0x00001, 0x080000, CRC(6cdefd78) SHA1(f1d1646c852ecf25551b93ff97666508ec1676c7) )
	ROM_LOAD16_BYTE( "95410407.hi", 0x00000, 0x080000, CRC(40c516b5) SHA1(e9dff71a3674acc8ab9b7e6386d42f30322af3ca) )
	sc_ggame_others
ROM_END

ROM_START( sc4ggameg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411406.lo", 0x00001, 0x080000, CRC(1e277091) SHA1(8d4482c94c943e2d53ce4d7da74bf8f8410d275d) )
	ROM_LOAD16_BYTE( "95411407.hi", 0x00000, 0x080000, CRC(02de3113) SHA1(ba9f402561595dfc676049801afffa53b1ab572b) )
	sc_ggame_others
ROM_END

ROM_START( sc4ggame )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414396.lo", 0x00001, 0x080000, CRC(1160d686) SHA1(b531823f970e09c808e2efb284951c79091b44b1) )
	ROM_LOAD16_BYTE( "95414397.hi", 0x00000, 0x080000, CRC(886f6d1f) SHA1(532f53d2c9956212ea0086b92c1dad38b417ad9e) )
	sc_ggame_others
ROM_END

ROM_START( sc4ggamei )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414398.lo", 0x00001, 0x080000, CRC(9449cf3a) SHA1(7bad738f8fd64c776b12578fd038397d4d079206) )
	ROM_LOAD16_BYTE( "95414399.hi", 0x00000, 0x080000, CRC(c8af9c12) SHA1(8701aad284a792993af2a06dddd20d19b0f4ceb6) )
	sc_ggame_others
ROM_END

ROM_START( sc4ggamej )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414478.lo", 0x00001, 0x080000, CRC(4128b79c) SHA1(b5fc3e3c8a20afa895a575d4d1552f722cbb5bc5) )
	ROM_LOAD16_BYTE( "95414479.hi", 0x00000, 0x080000, CRC(34a4ef4c) SHA1(2cb5ef9b6423b5fd7dabf55548d85ea54b24ff0c) )
	sc_ggame_others
ROM_END

ROM_START( sc4ggamek )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414552.lo", 0x00001, 0x080000, CRC(7136db21) SHA1(d871a0cb436a0f7e83bcd70dc389949763d76511) )
	ROM_LOAD16_BYTE( "95414553.hi", 0x00000, 0x080000, CRC(d05c13d2) SHA1(d5dde00f8af3856e515ed9a1a9e8c936a7cfd288) )
	sc_ggame_others
ROM_END

ROM_START( sc4ggamel )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414792.lo", 0x00001, 0x080000, CRC(448b5aee) SHA1(85e091be6bb04f81b176e69b0db6aab7f293d1fa) )
	ROM_LOAD16_BYTE( "95414793.hi", 0x00000, 0x080000, CRC(3a9c62c7) SHA1(15899911c70e35cc2a9c803707791b1cab4d8251) )
	sc_ggame_others
ROM_END

ROM_START( sc4ggamem )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414798.lo", 0x00001, 0x080000, CRC(2bd29667) SHA1(c8bf4fb72b858dbb777cf90bbc6e0368111aea79) )
	ROM_LOAD16_BYTE( "95414799.hi", 0x00000, 0x080000, CRC(3c8358b2) SHA1(0243526b7490f518793db804e85cc952e6000881) )
	sc_ggame_others
ROM_END

ROM_START( sc4ggamen )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414814.lo", 0x00001, 0x080000, CRC(2d2a1fb6) SHA1(62b6496a12e022991f63adeae32e2b9cca974b52) )
	ROM_LOAD16_BYTE( "95414815.hi", 0x00000, 0x080000, CRC(0257ff33) SHA1(1c9eb81d55fd7c76572e9f84a7765c16471cf110) )
	sc_ggame_others
ROM_END





ROM_START( sc4ggamep )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415396.lo", 0x00001, 0x080000, CRC(c7fa8e2e) SHA1(6b79f66de1c4f229add425ba2135ec41bd478606) )
	ROM_LOAD16_BYTE( "95415397.hi", 0x00000, 0x080000, CRC(2ff91444) SHA1(581ab1010085ca9ba487f7ff3e47ea6b0adb0ce1) )
	sc_ggame_others
ROM_END

ROM_START( sc4ggameq )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415398.lo", 0x00001, 0x080000, CRC(42d39792) SHA1(4b24ef5632da8a4a4b699e212b7e3f2c6db4b801) )
	ROM_LOAD16_BYTE( "95415399.hi", 0x00000, 0x080000, CRC(bcbb2ba8) SHA1(a6586e8296d461359e5cdc7e2a53e745c4d47927) )
	sc_ggame_others
ROM_END

ROM_START( sc4ggamer )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415400.lo", 0x00001, 0x080000, CRC(3b3e1044) SHA1(496fa7b86e3e5077ee926ef06daacdd741d58e93) )
	ROM_LOAD16_BYTE( "95415401.hi", 0x00000, 0x080000, CRC(d2a1f5cf) SHA1(5b733aa8d84f5b28ea3d877b93616d38aa97a13f) )
	sc_ggame_others
ROM_END

ROM_START( sc4ggames )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415404.lo", 0x00001, 0x080000, CRC(b59fc2d1) SHA1(236b233f5ece509d1628e3ff1cf051dfa044f5d8) )
	ROM_LOAD16_BYTE( "95415405.hi", 0x00000, 0x080000, CRC(a88f6b11) SHA1(ad171d40710854b96bc37167f5c6217b7331e9bf) )
	sc_ggame_others
ROM_END

ROM_START( sc4ggamet )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415406.lo", 0x00001, 0x080000, CRC(bc3fedcd) SHA1(d656f7dc0fa30759125b9c997a41192246044434) )
	ROM_LOAD16_BYTE( "95415407.hi", 0x00000, 0x080000, CRC(62377905) SHA1(e151253110a1425fbb89fe60035d439edfb677b5) )
	sc_ggame_others
ROM_END

ROM_START( sc4ggameu )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415408.lo", 0x00001, 0x080000, CRC(cc1ee1af) SHA1(6cd6ccca6bad75599c34ac86f79af3c9c15df626) )
	ROM_LOAD16_BYTE( "95415409.hi", 0x00000, 0x080000, CRC(19d9f4c5) SHA1(99567dc1500f54e8bddeccaac8f2757a80c7541e) )
	sc_ggame_others
ROM_END

ROM_START( sc4ggamev )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415552.lo", 0x00001, 0x080000, CRC(18c42d46) SHA1(9b18345670067258664aba6a6c1fe86cd8b9ca4c) )
	ROM_LOAD16_BYTE( "95415553.hi", 0x00000, 0x080000, CRC(ca2adcb0) SHA1(9ffb8deb732443772e500bdef58e41e0836f3f3c) )
	sc_ggame_others
ROM_END

ROM_START( sc4ggamew )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415792.lo", 0x00001, 0x080000, CRC(9312f684) SHA1(9fd3ada1125225b3b448415a1b9dc9f7e56461a6) )
	ROM_LOAD16_BYTE( "95415793.hi", 0x00000, 0x080000, CRC(0dad5eab) SHA1(5bae9c5403d2db632811c7fcd170975c254a3567) )
	sc_ggame_others
ROM_END

ROM_START( sc4ggamex )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415794.lo", 0x00001, 0x080000, CRC(c07ac457) SHA1(5a5bd8be1647d1ac0e25d63787289417c978db6f) )
	ROM_LOAD16_BYTE( "95415795.hi", 0x00000, 0x080000, CRC(7afe1d2d) SHA1(edb6b2d4c8f62895f5f77538c689ace93746f69d) )
	sc_ggame_others
ROM_END

ROM_START( sc4ggamey )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415796.lo", 0x00001, 0x080000, CRC(4e748bde) SHA1(3228fb7e3d39d8c0d44a08d9cb53555907e6cffe) )
	ROM_LOAD16_BYTE( "95415797.hi", 0x00000, 0x080000, CRC(43f71bb6) SHA1(5835abd2e978c9964c63718eed2907636e0997ee) )
	sc_ggame_others
ROM_END

ROM_START( sc4ggamez )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415798.lo", 0x00001, 0x080000, CRC(ff9798a5) SHA1(f7d72f113425c6a914cc3c6f3c6d393ae6455fbd) )
	ROM_LOAD16_BYTE( "95415799.hi", 0x00000, 0x080000, CRC(cfe254d4) SHA1(a233532461420d77a0c93df3a697d3352472a35f) )
	sc_ggame_others
ROM_END

ROM_START( sc4ggame0 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415814.lo", 0x00001, 0x080000, CRC(cd739a1d) SHA1(b80f07958ef26d071668d89260db46b1cb22231c) )
	ROM_LOAD16_BYTE( "95415815.hi", 0x00000, 0x080000, CRC(e745f7f6) SHA1(6adb886f2e270fd826768cb7f701e2fdc5f4c161) )
	sc_ggame_others
ROM_END

ROM_START( sc4ggame1 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416054.lo", 0x00001, 0x080000, CRC(9aa4dec5) SHA1(5a2f80df9f69836e08a576c1121addaa76558ee6) )
	ROM_LOAD16_BYTE( "95416055.hi", 0x00000, 0x080000, CRC(212eb6a8) SHA1(e45b6ba6747d9969a1515cc88434ba2e5965574b) )
	sc_ggame_others
ROM_END


ROM_START( sc4ggame2 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417054.lo", 0x00001, 0x080000, CRC(d3e2fdbc) SHA1(80d4a40a5a00c7452d3985bfcbf663751f731052) )
	ROM_LOAD16_BYTE( "95417055.hi", 0x00000, 0x080000, CRC(57d80fca) SHA1(30de0c5fc027be579b88ef6134078deb959a6e5e) )
	sc_ggame_others
ROM_END

ROM_START( sc4ggame3 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417382.lo", 0x00001, 0x080000, CRC(51e487cc) SHA1(8de22e386ac9107b85d8c79ca7e8c662a81a5dfc) )
	ROM_LOAD16_BYTE( "95417383.hi", 0x00000, 0x080000, CRC(b48c6cf0) SHA1(7a83fcdec48a95335ed190c7e9951e081b645d3b) )
	sc_ggame_others
ROM_END

ROM_START( sc4ggame4 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417384.lo", 0x00001, 0x080000, CRC(51b709db) SHA1(26a4ae234604867f3b8f02ec3de5dc506f99df97) )
	ROM_LOAD16_BYTE( "95417385.hi", 0x00000, 0x080000, CRC(9acb994e) SHA1(629f6f8d0908b088bfe0eda01b2bdaad5de65607) )
	sc_ggame_others
ROM_END

ROM_START( sc4ggame5 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417386.lo", 0x00001, 0x080000, CRC(a219590a) SHA1(8191d239939914b0d4ec5884a35c8eafada1cccf) )
	ROM_LOAD16_BYTE( "95417387.hi", 0x00000, 0x080000, CRC(74fbce05) SHA1(8297336ed15f27cf6b15396377774211b79f8d49) )
	sc_ggame_others
ROM_END

ROM_START( sc4ggame6 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417388.lo", 0x00001, 0x080000, CRC(761ebd3c) SHA1(f7690ca38ea445dc4171aed786004e467729ce8d) )
	ROM_LOAD16_BYTE( "95417389.hi", 0x00000, 0x080000, CRC(ba347108) SHA1(e78a3f963ae39fbbc1751370335e5a9034fac47a) )
	sc_ggame_others
ROM_END

ROM_START( sc4ggame7 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420132.lo", 0x00001, 0x080000, CRC(76d87518) SHA1(bd21827b69206b6c4cb9ac97bd80759e472bce7d) )
	ROM_LOAD16_BYTE( "95420133.hi", 0x00000, 0x080000, CRC(c2e67303) SHA1(8bdab8dc5544a21c2527dcac5ef055e3134106ea) )
	sc_ggame_others
ROM_END

ROM_START( sc4ggame8 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420142.lo", 0x00001, 0x080000, CRC(74478a35) SHA1(daf6257b7efdd620989c062fa645179066315d39) )
	ROM_LOAD16_BYTE( "95420143.hi", 0x00000, 0x080000, CRC(02ce5d1b) SHA1(4d52b0f953d339762d87488c49a827c060681746) )
	sc_ggame_others
ROM_END

ROM_START( sc4ggame9 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420144.lo", 0x00001, 0x080000, CRC(5d5003b4) SHA1(d47c6ed8f7c186d53c01166323782576a222ee48) )
	ROM_LOAD16_BYTE( "95420145.hi", 0x00000, 0x080000, CRC(b4d8a5f6) SHA1(4140e2e52e6faa8bedeacdd1c3cc3d67708ce12d) )
	sc_ggame_others
ROM_END

ROM_START( sc4ggameaa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420154.lo", 0x00001, 0x080000, CRC(9fbc7014) SHA1(4e710a91501bc63bdc0dc4a13d5decfa241ebfa7) )
	ROM_LOAD16_BYTE( "95420155.hi", 0x00000, 0x080000, CRC(1eedeb0f) SHA1(dae2fcfb38e85117023555ef62281af83ae4900c) )
	sc_ggame_others
ROM_END

ROM_START( sc4ggameab )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421132.lo", 0x00001, 0x080000, CRC(6bcb4a83) SHA1(b67cbb9f632de3c82a22b0aa73ca4aff2c2b2c78) )
	ROM_LOAD16_BYTE( "95421133.hi", 0x00000, 0x080000, CRC(27c3abbc) SHA1(14cf53851597267d3e86254d0a21f0e2abbf9bba) )
	sc_ggame_others
ROM_END

ROM_START( sc4ggameac )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421142.lo", 0x00001, 0x080000, CRC(68328252) SHA1(4b233adf661135d7e2bf35fca115b58648f3a58b) )
	ROM_LOAD16_BYTE( "95421143.hi", 0x00000, 0x080000, CRC(ff55c3b8) SHA1(558059252a59ade38b8686fde6ffc9e6c3153fad) )
	sc_ggame_others
ROM_END

ROM_START( sc4ggamead )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421144.lo", 0x00001, 0x080000, CRC(cb267fff) SHA1(0c06432f5ca41227293ed281b87f717e72f59ba8) )
	ROM_LOAD16_BYTE( "95421145.hi", 0x00000, 0x080000, CRC(4dfac3ec) SHA1(68036cd8521130081a999d9cd3a01cb27b5a437b) )
	sc_ggame_others
ROM_END


ROM_START( sc4ggameae )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421154.lo", 0x00001, 0x080000, CRC(66098ecf) SHA1(86b5971d26ec95440170508e7a6dc6cbfc6b3b19) )
	ROM_LOAD16_BYTE( "95421155.hi", 0x00000, 0x080000, CRC(fc87ee07) SHA1(d979843995213a0a6f6efede741309cdcdaba231) )
	sc_ggame_others
ROM_END


ROM_START( sc4ggdlx )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414956.lo", 0x00001, 0x080000, CRC(e6ce40a0) SHA1(6f8abc42fac35492393f72d3edfa81273e5a7c6e) )
	ROM_LOAD16_BYTE( "95414957.hi", 0x00000, 0x080000, CRC(627dbb26) SHA1(2798b7c154dd1d177b11926c76b78ab8b1150d82) )
	sc_ggame_others
ROM_END

ROM_START( sc4ggdlxa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414958.lo", 0x00001, 0x080000, CRC(3b02336b) SHA1(fedab56766e09924c1b5fc26ad9bcddda00fcc98) )
	ROM_LOAD16_BYTE( "95414959.hi", 0x00000, 0x080000, CRC(ac32f8bb) SHA1(784553498a7b5264afb90ffb597d6a070ef1612c) )
	sc_ggame_others
ROM_END


ROM_START( sc4ggdlxb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414992.lo", 0x00001, 0x080000, CRC(8d9880be) SHA1(7ba6f3eb8145f56f4c360fdfdb525135e4e053f1) )
	ROM_LOAD16_BYTE( "95414993.hi", 0x00000, 0x080000, CRC(1cfaa847) SHA1(69186bb277ccb644901a26a6e061250f0f5fed13) )
	sc_ggame_others
ROM_END


ROM_START( sc4ggdlxc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415956.lo", 0x00001, 0x080000, CRC(3b91d734) SHA1(89764625ad2f7f86e291caf53551656590473cab) )
	ROM_LOAD16_BYTE( "95415957.hi", 0x00000, 0x080000, CRC(925f43d0) SHA1(d6a994b35a16822ec4f5832a953b6504a27ebac6) )
	sc_ggame_others
ROM_END

ROM_START( sc4ggdlxd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415958.lo", 0x00001, 0x080000, CRC(7d2f448f) SHA1(b0d8612b6848648266f50eef902fe780eaef31c7) )
	ROM_LOAD16_BYTE( "95415959.hi", 0x00000, 0x080000, CRC(09d9d900) SHA1(5e56343c25bc643072dbe2a707ab6f7e245d6cae) )
	sc_ggame_others
ROM_END

ROM_START( sc4ggdlxe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415992.lo", 0x00001, 0x080000, CRC(208987db) SHA1(cd4221f93a4979262e32fed351585784bed8cf83) )
	ROM_LOAD16_BYTE( "95415993.hi", 0x00000, 0x080000, CRC(0bb70094) SHA1(694dae18704c26ae4f3162c383ecfdb3328796a5) )
	sc_ggame_others
ROM_END

ROM_START( sc4ggdlxf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416016.lo", 0x00001, 0x080000, CRC(943624e5) SHA1(f0a976991bc295e334ae33dedf2752760e586c57) )
	ROM_LOAD16_BYTE( "95416017.hi", 0x00000, 0x080000, CRC(f0155180) SHA1(51a5e99937508a15317d56588cf261da8dbd8b01) )
	sc_ggame_others
ROM_END

ROM_START( sc4ggdlxg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417016.lo", 0x00001, 0x080000, CRC(34529f02) SHA1(ea720107b89d69d1e673dd00a8e59eb670074817) )
	ROM_LOAD16_BYTE( "95417017.hi", 0x00000, 0x080000, CRC(cc7bc094) SHA1(3f62b9ee21a330d5bcda8a3f89f23322ffcb6e1f) )
	sc_ggame_others
ROM_END



ROM_START( sc4gggtb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420668.lo", 0x00001, 0x080000, CRC(afef1626) SHA1(00d676d0ed36ddf4255d26779b3319f61319a9cb) )
	ROM_LOAD16_BYTE( "95420669.hi", 0x00000, 0x080000, CRC(6e273bf4) SHA1(35ba48892d9ea3ebcb00fe33c75ee8e7949ab421) )
	sc_ggg_others
ROM_END

ROM_START( sc4gggtba )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421668.lo", 0x00001, 0x080000, CRC(6f4964c8) SHA1(14948920079694791a66dfd1f07b22badb37bc57) )
	ROM_LOAD16_BYTE( "95421669.hi", 0x00000, 0x080000, CRC(23e030cf) SHA1(85c70ee8f906b48fb75692ec9e58e9072b1a731d) )
	sc_ggg_others
ROM_END

ROM_START( sc4ggg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420672.lo", 0x00001, 0x080000, CRC(88dd2622) SHA1(dc30308638d602b76e828cf6c98605a9fb7589b6) )
	ROM_LOAD16_BYTE( "95420673.hi", 0x00000, 0x080000, CRC(309a5f9c) SHA1(8592127022c84bc6c36b744d8fa0d44a1bc38ea0) )
	sc_ggg_others
ROM_END

ROM_START( sc4gggb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420674.lo", 0x00001, 0x080000, CRC(f83a5079) SHA1(f8e6918690bc77288d2097aeb5af4634e5522fb6) )
	ROM_LOAD16_BYTE( "95420675.hi", 0x00000, 0x080000, CRC(ccf7c8ff) SHA1(c3996b17f1fce785e3892e6d5434a1d27b61c6f0) )
	sc_ggg_others
ROM_END

ROM_START( sc4gggc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420676.lo", 0x00001, 0x080000, CRC(6c841f87) SHA1(873d60cb447529f42d9f4cb0fe05d7c464a10473) )
	ROM_LOAD16_BYTE( "95420677.hi", 0x00000, 0x080000, CRC(afe9543e) SHA1(2c88a0120762372f349c098c8e346626198a7635) )
	sc_ggg_others
ROM_END

ROM_START( sc4gggd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420678.lo", 0x00001, 0x080000, CRC(5d1d9b26) SHA1(ed9594f794bd582bb54e7c824272b7321b6ae2ef) )
	ROM_LOAD16_BYTE( "95420679.hi", 0x00000, 0x080000, CRC(910215de) SHA1(5e3d9ab721ea008e33754ac8237f82c106114364) )
	sc_ggg_others
ROM_END

ROM_START( sc4ggge )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420680.lo", 0x00001, 0x080000, CRC(22a8623d) SHA1(22b75306e0ee2944e45739c380fc68338f2534f7) )
	ROM_LOAD16_BYTE( "95420681.hi", 0x00000, 0x080000, CRC(47dfc03a) SHA1(6fcacce499a8aa17a5b821511ffb1434d7f43850) )
	sc_ggg_others
ROM_END

ROM_START( sc4gggf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420682.lo", 0x00001, 0x080000, CRC(e41bce75) SHA1(db07f0c55c40d527df795f860ff2317af1f030b3) )
	ROM_LOAD16_BYTE( "95420683.hi", 0x00000, 0x080000, CRC(00b92dfe) SHA1(e77cc410b591eace2b40c5fdbf0fc99f0379c771) )
	sc_ggg_others
ROM_END


ROM_START( sc4gggg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420688.lo", 0x00001, 0x080000, CRC(6d7c6ed2) SHA1(1004c5027a60ba2cbdd488a4d49fee3c369d7c74) )
	ROM_LOAD16_BYTE( "95420689.hi", 0x00000, 0x080000, CRC(cea637a1) SHA1(68120b95741938d1df56eeb933968eb80144ec5a) )
	sc_ggg_others
ROM_END

ROM_START( sc4gggh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420690.lo", 0x00001, 0x080000, CRC(f06826b2) SHA1(ed3f6d54252fe6edd4490c86291212365c2f9177) )
	ROM_LOAD16_BYTE( "95420691.hi", 0x00000, 0x080000, CRC(74cf31ec) SHA1(d0e68a25266e7f9308afb4840329a638d97e3ad1) )
	sc_ggg_others
ROM_END

ROM_START( sc4gggi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420692.lo", 0x00001, 0x080000, CRC(c95de79a) SHA1(be7b6ca447fdc1d4ffaab2b4c6c69e344d093e1e) )
	ROM_LOAD16_BYTE( "95420693.hi", 0x00000, 0x080000, CRC(9ca69f5d) SHA1(1fecabfc8ee0b97bdf73116b16adf02fd214e15d) )
	sc_ggg_others
ROM_END



ROM_START( sc4gggk )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421672.lo", 0x00001, 0x080000, CRC(297c1a93) SHA1(7d0f8c6998f2970e1cce94a71fa4327edc66e07e) )
	ROM_LOAD16_BYTE( "95421673.hi", 0x00000, 0x080000, CRC(0e6202a7) SHA1(b47433908dbc8f1a237d13bf013a292c8e10d457) )
	sc_ggg_others
ROM_END

ROM_START( sc4gggl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421674.lo", 0x00001, 0x080000, CRC(f1934864) SHA1(a30341ba11dbea1bd4b93e9d9a8fcd60e763aec3) )
	ROM_LOAD16_BYTE( "95421675.hi", 0x00000, 0x080000, CRC(c37e7f84) SHA1(206d6fe336c4d66f1ee36b0d3d98ff3a58768281) )
	sc_ggg_others
ROM_END

ROM_START( sc4gggm )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421676.lo", 0x00001, 0x080000, CRC(c6b43eaf) SHA1(7e714ef9d84bb525539e311aed7e12a0a51d4476) )
	ROM_LOAD16_BYTE( "95421677.hi", 0x00000, 0x080000, CRC(96b39469) SHA1(c4f431d74d961d04b733b92867e46330c362c314) )
	sc_ggg_others
ROM_END

ROM_START( sc4gggn )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421678.lo", 0x00001, 0x080000, CRC(8e5aa825) SHA1(e30fa52a7fabcdda207e3bb604d0808006d0a24b) )
	ROM_LOAD16_BYTE( "95421679.hi", 0x00000, 0x080000, CRC(8ea625d4) SHA1(2875e8fc4b861e13f2a97e3e6c38d1ac83ade5dd) )
	sc_ggg_others
ROM_END

ROM_START( sc4gggo )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421680.lo", 0x00001, 0x080000, CRC(8dc426bb) SHA1(018c11e6c21a7b72b76fae1fbd76e35d5214d7b5) )
	ROM_LOAD16_BYTE( "95421681.hi", 0x00000, 0x080000, CRC(921c9d46) SHA1(da705ca5476263bb7417bf3de4fa73a24625172e) )
	sc_ggg_others
ROM_END

ROM_START( sc4gggp )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421682.lo", 0x00001, 0x080000, CRC(df227550) SHA1(8a34835e18209328d84d83a1a4070e5838cca664) )
	ROM_LOAD16_BYTE( "95421683.hi", 0x00000, 0x080000, CRC(8320c53a) SHA1(97ffa9b10b27915c23dba5beb8b823b6a1010cd7) )
	sc_ggg_others
ROM_END

ROM_START( sc4gggq )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421688.lo", 0x00001, 0x080000, CRC(cb3c7d09) SHA1(8fc0a9dedb94f67997ebed49123a9fe9bb39725e) )
	ROM_LOAD16_BYTE( "95421689.hi", 0x00000, 0x080000, CRC(2c2c2eeb) SHA1(a76754657ca3d036c3d054ad8e5feabb8bf3e71f) )
	sc_ggg_others
ROM_END

ROM_START( sc4gggr )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421690.lo", 0x00001, 0x080000, CRC(70c1a986) SHA1(b3e9d41de8fd766677ad1bbc53e3b328f273f7d3) )
	ROM_LOAD16_BYTE( "95421691.hi", 0x00000, 0x080000, CRC(471dca82) SHA1(d5e73c4fd18029262abd50b1b0269c3c3b6bd5c9) )
	sc_ggg_others
ROM_END

ROM_START( sc4gggs )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421692.lo", 0x00001, 0x080000, CRC(cb08f4ab) SHA1(6fd4ccc6224e3fb9038caadf0934e66bc88c0990) )
	ROM_LOAD16_BYTE( "95421693.hi", 0x00000, 0x080000, CRC(c0109286) SHA1(d66dd01f87dd6ecc212b935bdc8719b7cfa92315) )
	sc_ggg_others
ROM_END


ROM_START( sc4gunp )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95422362.lo", 0x00001, 0x080000, CRC(219225d1) SHA1(b2134e94a8fba0644003789942ea8689caf27b0c) )
	ROM_LOAD16_BYTE( "95422363.hi", 0x00000, 0x080000, CRC(8f489ba6) SHA1(f2f92332a09b4a811b01601fcae05a75ea418a0d) )
	sc_gunp_others
ROM_END

ROM_START( sc4gunpa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95422754.lo", 0x00001, 0x080000, CRC(1231d677) SHA1(8c9af6034319a66a1aef442f5d4fdc206900f433) )
	ROM_LOAD16_BYTE( "95422755.hi", 0x00000, 0x080000, CRC(67d94e3c) SHA1(02ee70541cd0991a34c6652b197c4f97e581f0b0) )
	sc_gunp_others
ROM_END

ROM_START( sc4gunpb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95423362.lo", 0x00001, 0x080000, CRC(76983aff) SHA1(4a857e9980f0d624d5983ad6bba9b1ac114ce79a) )
	ROM_LOAD16_BYTE( "95423363.hi", 0x00000, 0x080000, CRC(5ec237ef) SHA1(3521a763f7a4d22b79c79895888385b02e8720c5) )
	sc_gunp_others
ROM_END

ROM_START( sc4gunpc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95423754.lo", 0x00001, 0x080000, CRC(ce12c28a) SHA1(65f77811d6cc518e592c855577b373497d856e4d) )
	ROM_LOAD16_BYTE( "95423755.hi", 0x00000, 0x080000, CRC(0f92017c) SHA1(ad30f594e0c338675957547aa1de521b8406d633) )
	sc_gunp_others
ROM_END

ROM_START( sc4gunpd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426120.lo", 0x00001, 0x080000, CRC(d72e3bb4) SHA1(cd9c074f05e10c216a3392689a9a0ff217a235c1) )
	ROM_LOAD16_BYTE( "95426121.hi", 0x00000, 0x080000, CRC(657ab411) SHA1(8aacb39864580fb8724fa9284142f94518efb707) )
	sc_gunp_others
ROM_END

ROM_START( sc4gunpe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426320.lo", 0x00001, 0x080000, CRC(b3d3761b) SHA1(38a9990809998329e03e7a6559efedfba0e7bc4c) )
	ROM_LOAD16_BYTE( "95426321.hi", 0x00000, 0x080000, CRC(bc368bf0) SHA1(a1466380a4aac8fd96a9c987870e2ede9c26ce9b) )
	sc_gunp_others
ROM_END

ROM_START( sc4gunpf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427120.lo", 0x00001, 0x080000, CRC(b9761612) SHA1(df6826e0a8bb9076ed6b6142688c54d56f4233ed) )
	ROM_LOAD16_BYTE( "95427121.hi", 0x00000, 0x080000, CRC(557a4665) SHA1(f757a4d3f61dde1c41de833872dfa9411121414e) )
	sc_gunp_others
ROM_END

ROM_START( sc4gunpg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427320.lo", 0x00001, 0x080000, CRC(ad4fddfd) SHA1(528a5c924f0d6cd5aefccfc2d25bc37648c9f846) )
	ROM_LOAD16_BYTE( "95427321.hi", 0x00000, 0x080000, CRC(67a534ad) SHA1(04bf1bf63222855374d106b07940241d09c899c8) )
	sc_gunp_others
ROM_END



ROM_START( sc4hapnt )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406816.lo", 0x00001, 0x080000, CRC(e94df07f) SHA1(6c37860d0284cc1526c1711a9033d2d413ad09dd) )
	ROM_LOAD16_BYTE( "95406817.hi", 0x00000, 0x080000, CRC(a7d5a893) SHA1(fcff244e8b53be074fdb8b4f88bd1f514ada8ab4) )
	sc_hapnt_others
ROM_END

ROM_START( sc4hapnta )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95407816.lo", 0x00001, 0x080000, CRC(5cec8ea5) SHA1(c2724e8cbf261a209b0b55cde22493d9ac3c19b0) )
	ROM_LOAD16_BYTE( "95407817.hi", 0x00000, 0x080000, CRC(ad3ae839) SHA1(018d2850c94d379aaba98d9815c29932748745c7) )
	sc_hapnt_others
ROM_END


ROM_START( sc4hapntb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1306p12.hi", 0x00000, 0x080000, CRC(ea9c501d) SHA1(31f912906f562a04053f58ebed3c7bd0ee864434) )
	ROM_LOAD16_BYTE( "p1306p12.lo", 0x00001, 0x080000, CRC(d9aaad4c) SHA1(d0ccc3fc4cf1dbda58086fdba2ea0f10f94e1cd6) )
	sc_hapnt_others
ROM_END

ROM_START( sc4hapntc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1306p13.hi", 0x00000, 0x080000, CRC(6170b940) SHA1(82f06d54d5ea732b9e88a4ef97ce353c7db6e826) )
	ROM_LOAD16_BYTE( "p1306p13.lo", 0x00001, 0x080000, CRC(d99cff8f) SHA1(abf1e61cdfe3bc77f69bb9f981795d7978bcf23a) )
	sc_hapnt_others
ROM_END

ROM_START( sc4hapntd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1306s12.hi", 0x00000, 0x080000, CRC(dee0e44f) SHA1(0d60c2fafaa1fe334a44dc02e139d371048fc7f7) )
	ROM_LOAD16_BYTE( "p1306s12.lo", 0x00001, 0x080000, CRC(1c4a3b34) SHA1(3310133354ebfde433763136447a5f6d6a470d02) )
	sc_hapnt_others
ROM_END

ROM_START( sc4hapnte )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1306s13.hi", 0x00000, 0x080000, CRC(3fa9a172) SHA1(106c083892e83e3b925ca1b78b1653bc3ec2a81f) )
	ROM_LOAD16_BYTE( "p1306s13.lo", 0x00001, 0x080000, CRC(6aa080d1) SHA1(124750321d644714f6c41c9e1a19f6707ac43d5f) )
	sc_hapnt_others
ROM_END


ROM_START( sc4hntcs )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410960.lo", 0x00001, 0x080000, CRC(09417c00) SHA1(6d10896bf290c58b6fdb5d735b423a9d2d7db5d7) )
	ROM_LOAD16_BYTE( "95410961.hi", 0x00000, 0x080000, CRC(1611d748) SHA1(b5ae00a1e1adaf7af912b3f21fb0cb7b414996b3) )
	PR1327_CAS_HAPPY_NOTES_SOUNDS11
ROM_END

ROM_START( sc4hntcsa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410962.lo", 0x00001, 0x080000, CRC(64cecf4d) SHA1(63f9f5fda15e99a249ea869105461c6d1db0a01a) )
	ROM_LOAD16_BYTE( "95410963.hi", 0x00000, 0x080000, CRC(758108b0) SHA1(4de01d82d5ff01adfa3907adb46d666dd9979734) )
	PR1327_CAS_HAPPY_NOTES_SOUNDS11
ROM_END

ROM_START( sc4hntcsb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410964.lo", 0x00001, 0x080000, CRC(fbf06b79) SHA1(8130b8e9f5f18399963db8206f0d6c9ed0312fce) )
	ROM_LOAD16_BYTE( "95410965.hi", 0x00000, 0x080000, CRC(a91e7166) SHA1(accaa8c05fe6754b4ac1cf6eee5713de239968ce) )
	PR1327_CAS_HAPPY_NOTES_SOUNDS11
ROM_END

ROM_START( sc4hntcsc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411960.lo", 0x00001, 0x080000, CRC(e4f6033a) SHA1(4bde8c65639e952e4120c9a538d709a88e2a7982) )
	ROM_LOAD16_BYTE( "95411961.hi", 0x00000, 0x080000, CRC(f17e876d) SHA1(d90123e04cbf6935ee9cb62abbfa5cabed83c5b5) )
	PR1327_CAS_HAPPY_NOTES_SOUNDS11
ROM_END

ROM_START( sc4hntcsd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1327p12.hi", 0x00000, 0x080000, CRC(734e60a9) SHA1(911f6c78654020c03b3e906aaf37ce3751f1cd85) )
	ROM_LOAD16_BYTE( "p1327p12.lo", 0x00001, 0x080000, CRC(d1d169e0) SHA1(41540cee7ca0fbe2c93a1c55f0f9e9c933ed3f3b) )
	PR1327_CAS_HAPPY_NOTES_SOUNDS11
ROM_END

ROM_START( sc4hntcse )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1327p13.hi", 0x00000, 0x080000, CRC(d3141aff) SHA1(7c75e47e1a36979177c220edacc5256342c3405c) )
	ROM_LOAD16_BYTE( "p1327p13.lo", 0x00001, 0x080000, CRC(b80f3b5f) SHA1(8b1a3daadb67b2d147184ab7213abe9c29791c6e) )
	PR1327_CAS_HAPPY_NOTES_SOUNDS11
ROM_END

ROM_START( sc4hntcsf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1327p41.hi", 0x00000, 0x080000, CRC(0a0305da) SHA1(c4ed571484d13fed4adc83d96efabe060a6452a5) )
	ROM_LOAD16_BYTE( "p1327p41.lo", 0x00001, 0x080000, CRC(f0001712) SHA1(5be70eaedca55c4098f236763206cf32bb1c2f4d) )
	PR1327_CAS_HAPPY_NOTES_SOUNDS11
ROM_END

ROM_START( sc4hntcsg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1327p42.hi", 0x00000, 0x080000, CRC(c6e34460) SHA1(63ae0930b7b0c104edee067a0e13c0a90c2d027c) )
	ROM_LOAD16_BYTE( "p1327p42.lo", 0x00001, 0x080000, CRC(5e055385) SHA1(e42d7a0668bd1069c20bd519c37c1fcacabffca2) )
	PR1327_CAS_HAPPY_NOTES_SOUNDS11
ROM_END

ROM_START( sc4hntcsh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1327p61.hi", 0x00000, 0x080000, CRC(27a9e9a7) SHA1(9489bc73a3c806de974a82397c0d9907bcb3b9e5) )
	ROM_LOAD16_BYTE( "p1327p61.lo", 0x00001, 0x080000, CRC(13e3aa1e) SHA1(8bd97ebaa6b56e7c846cde2f0b61173c0332c120) )
	PR1327_CAS_HAPPY_NOTES_SOUNDS11
ROM_END

ROM_START( sc4hntcsi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1327p71.hi", 0x00000, 0x080000, CRC(d9dd087b) SHA1(872a52bcb42df863407ecb796c8768a02d3614d9) )
	ROM_LOAD16_BYTE( "p1327p71.lo", 0x00001, 0x080000, CRC(6cbef7a3) SHA1(11b6dfd24595fd10ce31e3f6361c6e9a7880c6c9) )
	PR1327_CAS_HAPPY_NOTES_SOUNDS11
ROM_END

ROM_START( sc4hntcsj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1327s12.hi", 0x00000, 0x080000, CRC(9aae872b) SHA1(fbd5e402d76bfc2b6e99667b6c97f851fc482abb) )
	ROM_LOAD16_BYTE( "p1327s12.lo", 0x00001, 0x080000, CRC(c964b987) SHA1(422b59e1ca2f3105ab28b33f4e95314ee525882f) )
	PR1327_CAS_HAPPY_NOTES_SOUNDS11
ROM_END

ROM_START( sc4hntcsk )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1327s13.hi", 0x00000, 0x080000, CRC(fc9dcc8a) SHA1(20605e20cdb643384f6c268d9e52b362d7369786) )
	ROM_LOAD16_BYTE( "p1327s13.lo", 0x00001, 0x080000, CRC(e2327afa) SHA1(bd1c0334fbc5ae4c386ad389c0ec619344c26656) )
	PR1327_CAS_HAPPY_NOTES_SOUNDS11
ROM_END

ROM_START( sc4hntcsl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1327s41.hi", 0x00000, 0x080000, CRC(cd03727a) SHA1(fcc802a62d60bb084ea3451730bc685212510320) )
	ROM_LOAD16_BYTE( "p1327s41.lo", 0x00001, 0x080000, CRC(346db912) SHA1(83435a6bb3e557a04040cd221474df23c39aa507) )
	PR1327_CAS_HAPPY_NOTES_SOUNDS11
ROM_END

ROM_START( sc4hntcsm )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1327s42.hi", 0x00000, 0x080000, CRC(2e737e11) SHA1(d1447129e4e1c888527371f1c5625d3e33d9d187) )
	ROM_LOAD16_BYTE( "p1327s42.lo", 0x00001, 0x080000, CRC(3916863f) SHA1(d0949fcc6414f42d292a04bd0d056cc51a10371c) )
	PR1327_CAS_HAPPY_NOTES_SOUNDS11
ROM_END

ROM_START( sc4hntcsn )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1327s61.hi", 0x00000, 0x080000, CRC(d3429de9) SHA1(4e5f6df4a557dd0aa061a0b7c4d30b8f01f1d958) )
	ROM_LOAD16_BYTE( "p1327s61.lo", 0x00001, 0x080000, CRC(76e3c418) SHA1(1ec77288096715894e0e179024205fa9dfaf3909) )
	PR1327_CAS_HAPPY_NOTES_SOUNDS11
ROM_END

ROM_START( sc4hntcso )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1327s71.hi", 0x00000, 0x080000, CRC(2d367c35) SHA1(547ba184d465d90296150595f35a5c4b012108c6) )
	ROM_LOAD16_BYTE( "p1327s71.lo", 0x00001, 0x080000, CRC(09be99a5) SHA1(d9ad6f8e4487c7fdf129478ac967d4c0aa96a0f9) )
	PR1327_CAS_HAPPY_NOTES_SOUNDS11
ROM_END

ROM_START( sc4hntcsp )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95412788.lo", 0x00001, 0x080000, CRC(0d8a0e36) SHA1(258ec8aea8f820919e106eda3e774054bb3758cb) )
	ROM_LOAD16_BYTE( "95412789.hi", 0x00000, 0x080000, CRC(0b15f31a) SHA1(051e047374fc3848a052b85bc54fb2a940fcd2f5) )
	ROM_REGION( 0x400000, "ymz", ROMREGION_ERASE00)
ROM_END


ROM_START( sc4hntcsq )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95412792.lo", 0x00001, 0x080000, CRC(bebb8a6d) SHA1(e1009ff6af9f9cf5df3d3f35dcac254e9ba25a84) )
	ROM_LOAD16_BYTE( "95412793.hi", 0x00000, 0x080000, CRC(fd18bea6) SHA1(94f3cf2ed593705a3caa788b3b55864f5726bb75) )
	ROM_REGION( 0x400000, "ymz", ROMREGION_ERASE00)
ROM_END


ROM_START( sc4hntcsr )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95413788.lo", 0x00001, 0x080000, CRC(223e4033) SHA1(47a335ef0d6964c7ef2ab8e4308323ac9269d98b) )
	ROM_LOAD16_BYTE( "95413789.hi", 0x00000, 0x080000, CRC(51709e9c) SHA1(d9e3b4ebcf7acf0fc8bc546019b93338fe1a60e6) )
	ROM_REGION( 0x400000, "ymz", ROMREGION_ERASE00)
ROM_END


ROM_START( sc4hntcss )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95413792.lo", 0x00001, 0x080000, CRC(1dfb13f0) SHA1(aab4c67090ea5024fe8ac1b36855398cbd8f0eed) )
	ROM_LOAD16_BYTE( "95413793.hi", 0x00000, 0x080000, CRC(57bf0935) SHA1(34ed890464aafd08e41d622367cc82898e9e07de) )
	ROM_REGION( 0x400000, "ymz", ROMREGION_ERASE00)
ROM_END


ROM_START( sc4helld )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406678.lo", 0x00001, 0x080000, CRC(82fa3872) SHA1(c1de9c35998c710f22eb95283a5a21d8e22c3c30) )
	ROM_LOAD16_BYTE( "95406679.hi", 0x00000, 0x080000, CRC(5b0767c6) SHA1(3a0245ba5822ab3faa74a1f06a3c1648fa97ee40) )
	SC4_95004211_HELLS_BELLS_PR6945
ROM_END

ROM_START( sc4hellb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408312.lo", 0x00001, 0x080000, CRC(45a2dd84) SHA1(cdd41b2f58138c9dc92cca8f8f92945b7a18d285) )
	ROM_LOAD16_BYTE( "95408313.hi", 0x00000, 0x080000, CRC(ec35f697) SHA1(752b7aa50145024f55ad7c2724f76dcda22f8701) )
	sc_hellb_others
ROM_END

ROM_START( sc4hellbb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408314.lo", 0x00001, 0x080000, CRC(9c97a1b8) SHA1(856508f6e744a66d8f7d5a4627397c8f3ea48fe8) )
	ROM_LOAD16_BYTE( "95408315.hi", 0x00000, 0x080000, CRC(5939c02c) SHA1(0fc62072c1cc76b287f259d32a0ea0e0856a34b4) )
	sc_hellb_others
ROM_END

ROM_START( sc4hellbc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408608.lo", 0x00001, 0x080000, CRC(773f7c6a) SHA1(5f4f07f9a29feeda398711f93c4c8fe80d039f54) )
	ROM_LOAD16_BYTE( "95408609.hi", 0x00000, 0x080000, CRC(bdeecd6f) SHA1(f703548e968539cf82d6894264e18c96de77a514) )
	sc_hellb_others
ROM_END

ROM_START( sc4hellbd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409312.lo", 0x00001, 0x080000, CRC(928296c5) SHA1(384fe4873a83991513db17405b2fbf442e3fe275) )
	ROM_LOAD16_BYTE( "95409313.hi", 0x00000, 0x080000, CRC(5fadd38f) SHA1(7775ddfdddf075f3aeb1acc4034bbceeec451fb0) )
	sc_hellb_others
ROM_END

ROM_START( sc4hellbe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409314.lo", 0x00001, 0x080000, CRC(4bb7eaf9) SHA1(8c3545fd3d8b3dde3ef5125ce612024caed395db) )
	ROM_LOAD16_BYTE( "95409315.hi", 0x00000, 0x080000, CRC(eaa1e534) SHA1(9ec4d0027abbce8e2f3cb50042fb8ae2043da00d) )
	sc_hellb_others
ROM_END

ROM_START( sc4hellbf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409608.lo", 0x00001, 0x080000, CRC(57c4bff8) SHA1(7df25f3441c0582c1c4b4b5bf1f32d486c9f22ff) )
	ROM_LOAD16_BYTE( "95409609.hi", 0x00000, 0x080000, CRC(139b8f22) SHA1(72c1eb1b61df3ba6872f1ad6831a6bd7a5dc8582) )
	sc_hellb_others
ROM_END

ROM_START( sc4hellbg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410546.lo", 0x00001, 0x080000, CRC(4746bac9) SHA1(ecbf0cc1887726fef2bfb8d3c358a1eb5970f1ba) )
	ROM_LOAD16_BYTE( "95410547.hi", 0x00000, 0x080000, CRC(b6a1da76) SHA1(d28d3fdc65623fbaed57b2f23e0705b5e353fd81) )
	sc_hellb_others
ROM_END

ROM_START( sc4hellbh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410548.lo", 0x00001, 0x080000, CRC(e5c3aaa3) SHA1(af13fff2f73400abdb207bce0476ec8b2a305e7c) )
	ROM_LOAD16_BYTE( "95410549.hi", 0x00000, 0x080000, CRC(428d932a) SHA1(edf6b155b7cfc5882a2ac13aa3bf554073a6b06f) )
	sc_hellb_others
ROM_END

ROM_START( sc4hellbi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411546.lo", 0x00001, 0x080000, CRC(8ca1aa2b) SHA1(90409c9c0ae4a482d657c47ef90945303c49075d) )
	ROM_LOAD16_BYTE( "95411547.hi", 0x00000, 0x080000, CRC(8d0a7be3) SHA1(11880744f08028045cea20f4a4c6878d2e5c0338) )
	sc_hellb_others
ROM_END

ROM_START( sc4hellbj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411548.lo", 0x00001, 0x080000, CRC(60105ee2) SHA1(8f0ebefab1d59ae089e0e29af2bfef06fa50b89a) )
	ROM_LOAD16_BYTE( "95411549.hi", 0x00000, 0x080000, CRC(a5e2c30f) SHA1(89bbced9533fe3a6c4c21c92a78d4f5909911315) )
	sc_hellb_others
ROM_END





ROM_START( sc4hill )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412982.lo", 0x00001, 0x080000, CRC(74eb09da) SHA1(ddc5d159727b0a5c7c17d5e14e5ab1e6214bcc39) )
	ROM_LOAD16_BYTE( "95412983.hi", 0x00000, 0x080000, CRC(4b5de886) SHA1(df937ca98c29fd7d7f2b2943e9b625d0f67cc555) )
	sc_hill_others
ROM_END

ROM_START( sc4hilla )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413982.lo", 0x00001, 0x080000, CRC(e649181c) SHA1(d197bc0d65e59b0ac0b1fd344213b5db3b11d7af) )
	ROM_LOAD16_BYTE( "95413983.hi", 0x00000, 0x080000, CRC(054304e1) SHA1(14ca648a59cc66a262abb2f0628a6f39772490eb) )
	sc_hill_others
ROM_END


ROM_START( sc4hiss )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416506.odd", 0x00001, 0x080000, CRC(b362b7e0) SHA1(6363a70310debd9948f152d6f08fa323dd7fbb09) )
	ROM_LOAD16_BYTE( "95416507.evn", 0x00000, 0x080000, CRC(9c933840) SHA1(a781bf357eb6012a2d28b4d5369f8a60e2a92cab) )
	sc_hiss_others
ROM_END

ROM_START( sc4hissa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416508.odd", 0x00001, 0x080000, CRC(a65a629a) SHA1(cef0a84f669852655b1b085f52f9dbc11d0f9cfb) )
	ROM_LOAD16_BYTE( "95416509.evn", 0x00000, 0x080000, CRC(2c09b047) SHA1(0854921111d7303e66fdd67c0f917a6f2574641b) )
	sc_hiss_others
ROM_END

ROM_START( sc4hissb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417506.odd", 0x00001, 0x080000, CRC(1c39af09) SHA1(0b9b502afc1d4c885eaeec6b2ac75bc44545c4bb) )
	ROM_LOAD16_BYTE( "95417507.evn", 0x00000, 0x080000, CRC(2d7645b8) SHA1(b809e577cddb42293d8dca1b84bee549e1b32a1c) )
	sc_hiss_others
ROM_END

ROM_START( sc4hissc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417508.odd", 0x00001, 0x080000, CRC(460ebd98) SHA1(7de8e862c88d4b41c158eeb7a41c7ace43a43f24) )
	ROM_LOAD16_BYTE( "95417509.evn", 0x00000, 0x080000, CRC(8b009f24) SHA1(879211fa80e48d0842bd610580c698cfc28ced76) )
	sc_hiss_others
ROM_END

ROM_START( sc4hissd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420402.lo", 0x00001, 0x080000, CRC(99eaf0ba) SHA1(cf57f189587fa19658ffc55a05fba113eb5e40e2) )
	ROM_LOAD16_BYTE( "95420403.hi", 0x00000, 0x080000, CRC(36e4a630) SHA1(b825d0141c805d6e4cb5ed5753987037014f7256) )
	sc_hiss_others
ROM_END

ROM_START( sc4hisse )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420404.lo", 0x00001, 0x080000, CRC(4fab20fd) SHA1(85d0f10d8499ab03474e96df7c1e0e816e89441a) )
	ROM_LOAD16_BYTE( "95420405.hi", 0x00000, 0x080000, CRC(4b42567e) SHA1(d7faa5786ab4ade9ef80e081e8475b54e0c9da1e) )
	sc_hiss_others
ROM_END

ROM_START( sc4hissf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421402.lo", 0x00001, 0x080000, CRC(376f101f) SHA1(f4124abf660c55e68561ec810fb306c8f00a53bf) )
	ROM_LOAD16_BYTE( "95421403.hi", 0x00000, 0x080000, CRC(aee8c285) SHA1(d4d03a8d17b8ab36e66f143ffc03325a267f3833) )
	sc_hiss_others
ROM_END

ROM_START( sc4hissg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421404.lo", 0x00001, 0x080000, CRC(88453934) SHA1(0e77d294aba735b9dcab5a3c5fc931066582f7e3) )
	ROM_LOAD16_BYTE( "95421405.hi", 0x00000, 0x080000, CRC(7425dd4f) SHA1(3c4ed325911092f923d37d321585e2bb23d71e66) )
	sc_hiss_others
ROM_END


ROM_START( sc4hotdg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1936p42.hi", 0x00000, 0x080000, CRC(ae8e4467) SHA1(555ae664fdb85ad4ea3abd20189c5823ebcc16a0) )
	ROM_LOAD16_BYTE( "p1936p42.lo", 0x00001, 0x080000, CRC(1975e059) SHA1(f7edd621806c4045f3f3fb062c41e943f2028639) )
	sc_hotdg_others
ROM_END

ROM_START( sc4hotdga )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1936p43.hi", 0x00000, 0x080000, NO_DUMP )
	ROM_LOAD16_BYTE( "p1936p43.lo", 0x00001, 0x080000, CRC(61ba3980) SHA1(38c3ffdd004467e39679eafc04892d5c4dbf761f) )
	sc_hotdg_others
ROM_END

ROM_START( sc4hotdgb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1936s42.hi", 0x00000, 0x080000, CRC(1bb44d7b) SHA1(f2cae86ad19cd2c951410a4cbb71b20e87a90ef8) )
	ROM_LOAD16_BYTE( "p1936s42.lo", 0x00001, 0x080000, CRC(78c3dbad) SHA1(c78225d108dc1a8f289ae6fd7f6cd78a6eea1d00) )
	sc_hotdg_others
ROM_END

ROM_START( sc4hotdgc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1936s43.hi", 0x00000, 0x080000, CRC(476c133f) SHA1(2c65820aec9fe4b9e92a2b3dc694f380856dea21) )
	ROM_LOAD16_BYTE( "p1936s43.lo", 0x00001, 0x080000, CRC(e8057da6) SHA1(dc9415d58a449597df8edce880eae19630b04c98) )
	sc_hotdg_others
ROM_END


ROM_START( sc4hotrd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95418238.lo", 0x00001, 0x080000, CRC(97fcc01f) SHA1(35b382fc2b3b582f234818e358d8785bb1617563) )
	ROM_LOAD16_BYTE( "95418239.hi", 0x00000, 0x080000, CRC(452c6e2f) SHA1(badc230257478bb2c73eca9015b9e3e8ca8f022b) )
	sc_hotrd_others
ROM_END

ROM_START( sc4hotrda )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95419238.lo", 0x00001, 0x080000, CRC(59f27ac5) SHA1(01714e84735c6b694c33971ec5027cf553ab1422) )
	ROM_LOAD16_BYTE( "95419239.hi", 0x00000, 0x080000, CRC(a1e79c90) SHA1(caacc2bda9aa86b42d343dda55134da29b774372) )
	sc_hotrd_others
ROM_END

ROM_START( sc4hotsh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426494.lo", 0x00001, 0x080000, CRC(83f08974) SHA1(1fca9ceb3e8cfee276069b5b60a41554b8dc2068) )
	ROM_LOAD16_BYTE( "95426495.hi", 0x00000, 0x080000, CRC(10224462) SHA1(0d7b0689629daabb2f84f7c3d441b8b7d5637c83) )
	sc_hotsh_others
ROM_END

ROM_START( sc4hotsha )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427494.lo", 0x00001, 0x080000, CRC(8dece917) SHA1(7d2ef3b1d3c67e056e8217cf08588787445b9b62) )
	ROM_LOAD16_BYTE( "95427495.hi", 0x00000, 0x080000, CRC(57d41514) SHA1(f3e24b6ed4495b928b75f9be9705f45cea9872ea) )
	sc_hotsh_others
ROM_END


ROM_START( sc4hotwd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408156.lo", 0x00001, 0x080000, CRC(6d2f16a8) SHA1(5aeb5f1fb578e40e093c39e97af0ae68d9b84a61) )
	ROM_LOAD16_BYTE( "95408157.hi", 0x00000, 0x080000, CRC(40bfed36) SHA1(faade9bbce8dca6f29dd2e115208507935ce5c06) )
	sc_hotwd_others
ROM_END

ROM_START( sc4hotwda )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95409156.lo", 0x00001, 0x080000, CRC(c625f300) SHA1(b0cab60d228f0bd2f66289068f0ae8473acf6fe7) )
	ROM_LOAD16_BYTE( "95409157.hi", 0x00000, 0x080000, CRC(854e2774) SHA1(7bc443d5add0bc886fec7f336766556bb18ec5cf) )
	sc_hotwd_others
ROM_END

ROM_START( sc4hotwdb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410742.lo", 0x00001, 0x080000, CRC(cf7541be) SHA1(82f1b92a36446c7bf1d8b474aa42b91e247cd959) )
	ROM_LOAD16_BYTE( "95410743.hi", 0x00000, 0x080000, CRC(f10d6882) SHA1(6f02952351ad16a88057bb8a702748b0f257fb1a) )
	sc_hotwd_others
ROM_END

ROM_START( sc4hotwdc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410744.lo", 0x00001, 0x080000, CRC(15b02f78) SHA1(471192b014e4fbaaff9279aff56397f3e536d405) )
	ROM_LOAD16_BYTE( "95410745.hi", 0x00000, 0x080000, CRC(d2e37099) SHA1(5c52c24239328b1d8e944cd4728de7c8c77a9905) )
	sc_hotwd_others
ROM_END

ROM_START( sc4hotwdd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411742.lo", 0x00001, 0x080000, CRC(5e52421a) SHA1(faa1c38a9f57c79e417fa3fda40a7e2f5f50a4f9) )
	ROM_LOAD16_BYTE( "95411743.hi", 0x00000, 0x080000, CRC(fe4b57d3) SHA1(26d3924f7a298fc5b3a23ad1cb54cf5593c5af4a) )
	sc_hotwd_others
ROM_END

ROM_START( sc4hotwde )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411744.lo", 0x00001, 0x080000, CRC(4e133d64) SHA1(939cbc5e4e28c5c519b2c0288f64220fcb1a4cd2) )
	ROM_LOAD16_BYTE( "95411745.hi", 0x00000, 0x080000, CRC(5437310d) SHA1(1b02985c32e9e381a85c57da8c8e1bf79e1ad7a6) )
	sc_hotwd_others
ROM_END



ROM_START( sc4celeb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412450.lo", 0x00001, 0x080000, CRC(3ecc7ad0) SHA1(d412f6ee4d5555b81cd8880624fe0d6a74bd7162) )
	ROM_LOAD16_BYTE( "95412451.hi", 0x00000, 0x080000, CRC(b2175e84) SHA1(5c9b1fad20e1e24878f412f3aae1633a76fdc2d6) )
	sc_celeb_others
ROM_END

ROM_START( sc4celeba )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412452.lo", 0x00001, 0x080000, CRC(ad38d950) SHA1(ceece0814ee5d436b39fe115020812da89c14e47) )
	ROM_LOAD16_BYTE( "95412453.hi", 0x00000, 0x080000, CRC(252415c3) SHA1(6b561d9301105050804b2cac18703fe862fff651) )
	sc_celeb_others
ROM_END

ROM_START( sc4celebb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412582.lo", 0x00001, 0x080000, CRC(e7be047f) SHA1(a0c20b6463f48cd632ed9e97ba12b8cba818baa8) )
	ROM_LOAD16_BYTE( "95412583.hi", 0x00000, 0x080000, CRC(3195b94e) SHA1(af6d32f6e0f1c55ab8e066782e12a0dfef8b6c30) )
	sc_celeb_others
ROM_END

ROM_START( sc4celebc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413452.lo", 0x00001, 0x080000, CRC(1ee613ac) SHA1(df7db88998cb92aab42af3b8c71181ae57dc2887) )
	ROM_LOAD16_BYTE( "95413453.hi", 0x00000, 0x080000, CRC(dbe1282c) SHA1(4a400ecc8c1eabad236e80b937988894723c4209) )
	sc_celeb_others
ROM_END

ROM_START( sc4celebd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413582.lo", 0x00001, 0x080000, CRC(94dcfcb6) SHA1(f5ce361a949ce881fa1fc77522947aadc7659f6b) )
	ROM_LOAD16_BYTE( "95413583.hi", 0x00000, 0x080000, CRC(c69cb5d8) SHA1(bf57bad6445807d338d2eeab626bf93dd363b834) )
	sc_celeb_others
ROM_END



ROM_START( sc4inspn )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416704.lo", 0x00001, 0x080000, CRC(15151797) SHA1(06ac7be17907f79d86cb7bffe3abf1ee63d7ff0f) )
	ROM_LOAD16_BYTE( "95416705.hi", 0x00000, 0x080000, CRC(a783b113) SHA1(55c9ca17144b4806e719cba923f5f906173e2ef5) )
	sc_inspn_others
ROM_END

ROM_START( sc4inspna )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417704.lo", 0x00001, 0x080000, CRC(8abaac7b) SHA1(f1028f673738973e843a986a751bcb122c891fff) )
	ROM_LOAD16_BYTE( "95417705.hi", 0x00000, 0x080000, CRC(793a1843) SHA1(194a176c4b145111fdd70496dbe1809b36816b29) )
	sc_inspn_others
ROM_END




ROM_START( sc4ijob )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95422378.lo", 0x00001, 0x080000, CRC(8ff38628) SHA1(8477f92c088475c3256c361a608130d86cab23f9) )
	ROM_LOAD16_BYTE( "95422379.hi", 0x00000, 0x080000, CRC(319f75dc) SHA1(7629b792052fe0107f4a00da6b8166c985c96e06) )
	sc_ijob_others
ROM_END

ROM_START( sc4ijoba )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95422480.lo", 0x00001, 0x080000, CRC(31919954) SHA1(aba76190875478b78a672660bf925d869d5f0391) )
	ROM_LOAD16_BYTE( "95422481.hi", 0x00000, 0x080000, CRC(7fc2155a) SHA1(5de2de8f5061e1cf3d6c0d3628397beb249aabf9) )
	sc_ijob_others
ROM_END

ROM_START( sc4ijobb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95422590.lo", 0x00001, 0x080000, CRC(00deb75d) SHA1(dff99d61579f53bd41e6885293e30b30b0dfc9db) )
	ROM_LOAD16_BYTE( "95422591.hi", 0x00000, 0x080000, CRC(07479fb5) SHA1(0a4a4e9751e51ea2bb9444861537bf64338594fb) )
	sc_ijob_others
ROM_END

ROM_START( sc4ijobc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95422876.lo", 0x00001, 0x080000, CRC(fc9a9eb1) SHA1(2c8db0cf9d60c52b06a11cd973b634759e111d36) )
	ROM_LOAD16_BYTE( "95422877.hi", 0x00000, 0x080000, CRC(eb99528b) SHA1(46897348b16f8a6172502bdb1bebdb2db49a2aed) )
	sc_ijob_others
ROM_END

ROM_START( sc4ijobd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95423378.lo", 0x00001, 0x080000, CRC(c692c68e) SHA1(145b0deba57d596cc66d6c1a7259f78ea3823f94) )
	ROM_LOAD16_BYTE( "95423379.hi", 0x00000, 0x080000, CRC(920e7a50) SHA1(3af29d8f77a80c5a9d8e8c3d200fd1898777340a) )
	sc_ijob_others
ROM_END

ROM_START( sc4ijobe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95423480.lo", 0x00001, 0x080000, CRC(fd21edb8) SHA1(424827cb40fa9fb5ac831319b711c06a41b3b136) )
	ROM_LOAD16_BYTE( "95423481.hi", 0x00000, 0x080000, CRC(9249f37d) SHA1(7624bccaf5382a0fb4645bf77bd5415b2a2f6eaf) )
	sc_ijob_others
ROM_END

ROM_START( sc4ijobf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95423590.lo", 0x00001, 0x080000, CRC(434b2e9f) SHA1(5c586b81ff8ae36a1bc42a08643d7bcac2c4930c) )
	ROM_LOAD16_BYTE( "95423591.hi", 0x00000, 0x080000, CRC(896cfc1e) SHA1(6fe85fd536d2285d1921bafd5fceceac59093479) )
	sc_ijob_others
ROM_END

ROM_START( sc4ijobg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95423876.lo", 0x00001, 0x080000, CRC(3909f9c2) SHA1(e7ca2b26d9102aa8404239ca445089f05c720554) )
	ROM_LOAD16_BYTE( "95423877.hi", 0x00000, 0x080000, CRC(49549dd0) SHA1(eae316d5402958232777548ae6ff478721a8b3fa) )
	sc_ijob_others
ROM_END

ROM_START( sc4ijobh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426318.lo", 0x00001, 0x080000, CRC(9a1d2dd1) SHA1(6016ad4db9d566ada8642ea8fd377e3133754e37) )
	ROM_LOAD16_BYTE( "95426319.hi", 0x00000, 0x080000, CRC(23a5204c) SHA1(ade96b23d61e29b5b194a0acef97b07d7671a026) )
	sc_ijob_others
ROM_END

ROM_START( sc4ijobi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426462.lo", 0x00001, 0x080000, CRC(16a3ae53) SHA1(ea8b612828e83db585d02def48adc69bcdc1910d) )
	ROM_LOAD16_BYTE( "95426463.hi", 0x00000, 0x080000, CRC(73bf9be3) SHA1(afec4aba74496a1b3bf3472907904cb07f681987) )
	sc_ijob_others
ROM_END

ROM_START( sc4ijobj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427318.lo", 0x00001, 0x080000, CRC(abcd391a) SHA1(3e7403dc84e162a734dee9f43b8fe07c3d073a04) )
	ROM_LOAD16_BYTE( "95427319.hi", 0x00000, 0x080000, CRC(30538c7f) SHA1(b4ab374e0a5ed644274ba767644c839dadd3e79a) )
	sc_ijob_others
ROM_END

ROM_START( sc4ijobk )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427462.lo", 0x00001, 0x080000, CRC(9d305f15) SHA1(4a224e679447e46dafeed3b89b658b9c47e41d48) )
	ROM_LOAD16_BYTE( "95427463.hi", 0x00000, 0x080000, CRC(32d24f72) SHA1(182f56fc3e459a9801fb726b0f54a62b632f937f) )
	sc_ijob_others
ROM_END

ROM_START( sc4ijobl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95426078.lo", 0x00001, 0x080000, CRC(42ea72a3) SHA1(da088b0322fe3fc89348f63d7b342d68be37906b) )
	ROM_LOAD16_BYTE( "95426079.hi", 0x00000, 0x080000, CRC(49c8cf07) SHA1(af9a27e36ca5652ac542ad69b9987dd454b5bb7d) )
	ROM_REGION( 0x400000, "ymz", ROMREGION_ERASE00)
ROM_END

ROM_START( sc4ijobm )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95427079.hi", 0x00000, 0x080000, CRC(ff90a41d) SHA1(cc3011bdee10214c2de15683c436f1819a3b0e89) )
	ROM_LOAD16_BYTE( "95427078.lo", 0x00001, 0x080000, CRC(9b27c381) SHA1(f9a66f13d3ea1ca206ec2904e6944ed41fe83e63) )
	ROM_REGION( 0x400000, "ymz", ROMREGION_ERASE00)
ROM_END

ROM_START( sc4jjok )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412718.lo", 0x00001, 0x080000, CRC(a7e2c60e) SHA1(febe7eb08d56d8d1b47a92952ee2b7c8d96cb5f3) )
	ROM_LOAD16_BYTE( "95412719.hi", 0x00000, 0x080000, CRC(a8d93a87) SHA1(d996cd711780aa316ea7e0ee40f20a3455742bf1) )
	sc_jjok_others
ROM_END

ROM_START( sc4jjoka )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413718.lo", 0x00001, 0x080000, CRC(331295e4) SHA1(4bf8b112acc881b4094a2b72c999d439c94e81b8) )
	ROM_LOAD16_BYTE( "95413719.hi", 0x00000, 0x080000, CRC(910fc3f2) SHA1(dc8949aa989793754584817b9a56c6dfcfc0e705) )
	sc_jjok_others
ROM_END


ROM_START( sc4ldvl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95424524.lo", 0x00001, 0x080000, CRC(d323cf89) SHA1(f384be0014e08984cb72275ac3ee163a0975222b) )
	ROM_LOAD16_BYTE( "95424525.hi", 0x00000, 0x080000, CRC(fa2b69df) SHA1(84b86aec04f9ba8001a61665d6d4ced7918eb18a) )
	sc_ldvl_others
ROM_END

ROM_START( sc4ldvla )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95424536.lo", 0x00001, 0x080000, CRC(25c304a4) SHA1(7c9210b08e392f9501708c39d33e9546ad60f174) )
	ROM_LOAD16_BYTE( "95424537.hi", 0x00000, 0x080000, CRC(90499606) SHA1(0c017102c286c9b76aa93a426866c4ee1f3da10b) )
	sc_ldvl_others
ROM_END

ROM_START( sc4ldvlb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95425524.lo", 0x00001, 0x080000, CRC(446a9058) SHA1(a934093af01643c0b66e50b04e419abb90e7e952) )
	ROM_LOAD16_BYTE( "95425525.hi", 0x00000, 0x080000, CRC(34a5acdb) SHA1(c23f7571a103cb645689e13c99891174fb783dde) )
	sc_ldvl_others
ROM_END

ROM_START( sc4ldvlc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95425536.lo", 0x00001, 0x080000, CRC(7d5fc6d2) SHA1(eec138baa134c9ca4e67bbe9ed4a73e06e4ac4ae) )
	ROM_LOAD16_BYTE( "95425537.hi", 0x00000, 0x080000, CRC(a271557a) SHA1(7f7f41164c4bda1cefef3ffd68858677917fef7e) )
	sc_ldvl_others
ROM_END


ROM_START( sc4lotrr )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410614.lo", 0x00001, 0x080000, CRC(39b70b6b) SHA1(feda79017224771822335ecd2a9c29c69c7a5aa5) )
	ROM_LOAD16_BYTE( "95410615.hi", 0x00000, 0x080000, CRC(d5ed8948) SHA1(63f4fd2fd5de601cee0668d74b7a8907643024ff) )
	sc_lotrr_others
ROM_END

ROM_START( sc4lotrra )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410950.lo", 0x00001, 0x080000, CRC(33875be7) SHA1(4e2e092e2fa2f1ea9b7f4d748b182f5a9c3ffb3b) )
	ROM_LOAD16_BYTE( "95410951.hi", 0x00000, 0x080000, CRC(72553949) SHA1(91927ef336d367364f61dfe31c016c8e9a2bf99e) )
	sc_lotrr_others
ROM_END

ROM_START( sc4lotrrb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411614.lo", 0x00001, 0x080000, CRC(a2a230d7) SHA1(55f0a16bed44bceb94c578ed84df857ce95cf55d) )
	ROM_LOAD16_BYTE( "95411615.hi", 0x00000, 0x080000, CRC(bb8ff190) SHA1(9fec418e8d7540a54c79885236a30016acac70cf) )
	sc_lotrr_others
ROM_END

ROM_START( sc4lotrrc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95411950.lo", 0x00001, 0x080000, CRC(ec42d7f8) SHA1(fed3e0d17f671db1d3060ab9d8ae5d9bf45c9dbc) )
	ROM_LOAD16_BYTE( "95411951.hi", 0x00000, 0x080000, CRC(a7c31009) SHA1(d2db9721b6967810e02f28ce20e1ded6598eb08d) )
	sc_lotrr_others
ROM_END

ROM_START( sc4lotrrd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412574.lo", 0x00001, 0x080000, CRC(653cd804) SHA1(35e3346581a72ad29af7afe88266910d46ef043a) )
	ROM_LOAD16_BYTE( "95412575.hi", 0x00000, 0x080000, CRC(4227e0b5) SHA1(a8ae560d537cfb1bea25ac271deed501e7b655f4) )
	sc_lotrr_others
ROM_END

ROM_START( sc4lotrre )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413574.lo", 0x00001, 0x080000, CRC(796749c9) SHA1(e2940a8184537570b71eabd675ea44f23055798c) )
	ROM_LOAD16_BYTE( "95413575.hi", 0x00000, 0x080000, CRC(147949f2) SHA1(39b435b8f5b9490b3592e0cbff01b11c249ecea2) )
	sc_lotrr_others
ROM_END


ROM_START( sc4manic )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95418956.lo", 0x00001, 0x080000, CRC(a01460ca) SHA1(b7112e726e016d3e6a8d5a61dd64cf3bd0e7188b) )
	ROM_LOAD16_BYTE( "95418957.hi", 0x00000, 0x080000, CRC(35fb3cdb) SHA1(984bf00ff02b83cc44fa6746b1ab7ce7a387171d) )
	sc_manic_others
ROM_END

ROM_START( sc4manica )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95419956.lo", 0x00001, 0x080000, CRC(e58342d1) SHA1(aa3cd1c8324f1731d84ad087ddb7b620b8c1c8f1) )
	ROM_LOAD16_BYTE( "95419957.hi", 0x00000, 0x080000, CRC(686efd41) SHA1(6d30b574886603bc737a132adbc4d5ac503a4527) )
	sc_manic_others
ROM_END



ROM_START( sc4manicb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426162.lo", 0x00001, 0x080000, CRC(47fb3a1f) SHA1(41cff1666dd890fc18ac8110f6e4c598ca5826a9) )
	ROM_LOAD16_BYTE( "95426163.hi", 0x00000, 0x080000, CRC(525a8b91) SHA1(1e8a6bc2c140bbfe4297ec786535c30412dad05c) )
	sc_manic_others
ROM_END

ROM_START( sc4manicc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427162.lo", 0x00001, 0x080000, CRC(355bd2da) SHA1(035f715984db286db6f7ebd0c751fea872d4c98b) )
	ROM_LOAD16_BYTE( "95427163.hi", 0x00000, 0x080000, CRC(98fd15cf) SHA1(39f012e15d0d6f687399388a3ab0abbaf5f7db89) )
	sc_manic_others
ROM_END

ROM_START( sc4manicd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p3034p41.hi", 0x00000, 0x080000, CRC(b021f5cd) SHA1(5822c6589345e6796137cd58d747823ff2ee78c2) )
	ROM_LOAD16_BYTE( "p3034p41.lo", 0x00001, 0x080000, CRC(56afeb99) SHA1(763eda91a8a9a2579bc9bfdba1adf39098ed7b56) )
	sc_manic_others
ROM_END

ROM_START( sc4manice )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p3034p42.hi", 0x00000, 0x080000, CRC(21c7fcbc) SHA1(4956e633f1dec59d023dc91ab56fb0e04f270645) )
	ROM_LOAD16_BYTE( "p3034p42.lo", 0x00001, 0x080000, CRC(760fafaa) SHA1(6295cf876527f20f77614443150a0ae48a07bc83) )
	sc_manic_others
ROM_END

ROM_START( sc4manicf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p3034s41.hi", 0x00000, 0x080000, CRC(02ab75c4) SHA1(f93bcea32f8801c5deb1020e96d074bdb5197099) )
	ROM_LOAD16_BYTE( "p3034s41.lo", 0x00001, 0x080000, CRC(37f8f591) SHA1(761ca79e6accf6df9324d3a81f174326a5b26a9c) )
	sc_manic_others
ROM_END

ROM_START( sc4manicg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p3034s42.hi", 0x00000, 0x080000, CRC(ed59aa13) SHA1(9e21f633307592d7a33ba0c14a78e2bd986f60e0) )
	ROM_LOAD16_BYTE( "p3034s42.lo", 0x00001, 0x080000, CRC(41ed34c8) SHA1(ca4b30a6575b06dcb6b56ab872cfa70b1eac5c46) )
	sc_manic_others
ROM_END


ROM_START( sc4mmm )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416950.lo", 0x00001, 0x080000, CRC(a1a42db1) SHA1(c676e72a9a4c9399dfbcc98b50845dd71cc96c1c) )
	ROM_LOAD16_BYTE( "95416951.hi", 0x00000, 0x080000, CRC(ff46f109) SHA1(3163548003146b20da266c5344a2e610eed346e6) )
	sc_mmm_others
ROM_END

ROM_START( sc4mmma )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416952.lo", 0x00001, 0x080000, CRC(060e7fe9) SHA1(6a01b5a07673ca618c887dc5f0f2427a72692404) )
	ROM_LOAD16_BYTE( "95416953.hi", 0x00000, 0x080000, CRC(dbecceb1) SHA1(4f3643003a67c52c9e5e4c31bc596fdd567a9b1a) )
	sc_mmm_others
ROM_END

ROM_START( sc4mmmb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417950.lo", 0x00001, 0x080000, CRC(c46efd5f) SHA1(18a5bdd00fce9db6ca12c46407affc5cff69021e) )
	ROM_LOAD16_BYTE( "95417951.hi", 0x00000, 0x080000, CRC(544a15ab) SHA1(4997c67cc238265238835764a25cff54588e2a4b) )
	sc_mmm_others
ROM_END

ROM_START( sc4mmmc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417952.lo", 0x00001, 0x080000, CRC(92a4b8a1) SHA1(0f88fa8544dd6608b5ddce31c2ba7b5525ccc4e6) )
	ROM_LOAD16_BYTE( "95417953.hi", 0x00000, 0x080000, CRC(3127af1d) SHA1(2d483b8cc1c873bbcf5f142f10cc0399d5bc49ec) )
	sc_mmm_others
ROM_END

ROM_START( sc4mmmd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420200.lo", 0x00001, 0x080000, CRC(6eff6750) SHA1(bb8f7904653299f5caaf5a9ef8dfcdcf75da5bda) )
	ROM_LOAD16_BYTE( "95420201.hi", 0x00000, 0x080000, CRC(5b1ad2e0) SHA1(97a3436ced57849a5a852248d342c875c015d61e) )
	sc_mmm_others
ROM_END

ROM_START( sc4mmme )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420202.lo", 0x00001, 0x080000, CRC(6eda8626) SHA1(1a5a563d0c3b6cccbf8c8fad7b4b01010a6cb6cb) )
	ROM_LOAD16_BYTE( "95420203.hi", 0x00000, 0x080000, CRC(079e8554) SHA1(2d7b5fe3b452bd3a33139bb73b43a5230dd7501b) )
	sc_mmm_others
ROM_END

ROM_START( sc4mmmf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421200.lo", 0x00001, 0x080000, CRC(d1233024) SHA1(aed4e4b89f89f4bc1beaa49310c3237548d270a0) )
	ROM_LOAD16_BYTE( "95421201.hi", 0x00000, 0x080000, CRC(5a088c8d) SHA1(624f410471f8d40cc172784b9ec41883b3424161) )
	sc_mmm_others
ROM_END

ROM_START( sc4mmmg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421202.lo", 0x00001, 0x080000, CRC(4203aa0b) SHA1(3f7c2e83c71a6308778cbf693c587399f96fafa6) )
	ROM_LOAD16_BYTE( "95421203.hi", 0x00000, 0x080000, CRC(13f57bf4) SHA1(0d72148383b265dd2d0bf94915e044e907a56e28) )
	sc_mmm_others
ROM_END


ROM_START( sc4mmad )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416294.lo", 0x00001, 0x080000, CRC(146f4b75) SHA1(70ac7756ac6e52113bf7afef783622e07a300ef6) )
	ROM_LOAD16_BYTE( "95416295.hi", 0x00000, 0x080000, CRC(ce018f99) SHA1(10df73d5ded92f033653f9f2ba35fd57bc4b49be) )
	sc_mmad_others
ROM_END

ROM_START( sc4mmada )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416296.lo", 0x00001, 0x080000, CRC(74669322) SHA1(e3c101e30e3e26e1bf4335cd47cb9de0bcb29341) )
	ROM_LOAD16_BYTE( "95416297.hi", 0x00000, 0x080000, CRC(23d37d4b) SHA1(474018fcbbfa49255217f75a43c27a4dc0bde424) )
	sc_mmad_others
ROM_END

ROM_START( sc4mmadb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417294.lo", 0x00001, 0x080000, CRC(46b0ff94) SHA1(47b6014fee6837a519142a48b4b5c4d030f409f7) )
	ROM_LOAD16_BYTE( "95417295.hi", 0x00000, 0x080000, CRC(2699252c) SHA1(8bfa96fc12c867e8e09d71064f91babd39f13e16) )
	sc_mmad_others
ROM_END

ROM_START( sc4mmadc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417296.lo", 0x00001, 0x080000, CRC(28e3e75a) SHA1(bacbc50c0e85f3ee5606fda43515bfa6721af0ac) )
	ROM_LOAD16_BYTE( "95417297.hi", 0x00000, 0x080000, CRC(35b8870e) SHA1(b7f2ff0c31fe31a92477aebc8503489e271e793a) )
	sc_mmad_others
ROM_END

ROM_START( sc4mmadd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420542.lo", 0x00001, 0x080000, CRC(e0844e00) SHA1(e29fed68345b99e206cf075cf549f3ecfcd6050c) )
	ROM_LOAD16_BYTE( "95420543.hi", 0x00000, 0x080000, CRC(9c38669e) SHA1(f80c7ba42c1e1b0581d3aa427388689e80c2a054) )
	sc_mmad_others
ROM_END

ROM_START( sc4mmade )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420544.lo", 0x00001, 0x080000, CRC(d1ad5cae) SHA1(5215d303068d9b221d0b65fca33d8b87dcb2ef6d) )
	ROM_LOAD16_BYTE( "95420545.hi", 0x00000, 0x080000, NO_DUMP )
	sc_mmad_others
ROM_END

ROM_START( sc4mmadf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421542.lo", 0x00001, 0x080000, CRC(c0e2e259) SHA1(1bf56cd771dcc62922e86adf9ad83301d731666b) )
	ROM_LOAD16_BYTE( "95421543.hi", 0x00000, 0x080000, NO_DUMP )
	sc_mmad_others
ROM_END

ROM_START( sc4mmadg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421544.lo", 0x00001, 0x080000, CRC(05cec61a) SHA1(6ab25749d7d893ff03a96d64a2fb556706b6c89a) )
	ROM_LOAD16_BYTE( "95421545.hi", 0x00000, 0x080000, NO_DUMP )
	sc_mmad_others
ROM_END



ROM_START( sc4monsp )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95406476.lo", 0x00001, 0x080000, CRC(c1ffdeb4) SHA1(e600dc090edc996cd5bde0bf24b3c1fa0e4a19aa) )
	ROM_LOAD16_BYTE( "95406477.hi", 0x00000, 0x080000, CRC(3a932d01) SHA1(e1488f522680bda99b4b46a61acc8e3c1ad68f6f) )
	sc_monsp_others
ROM_END



ROM_START( sc4mono5 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412170.lo", 0x00001, 0x080000, CRC(7907cfb1) SHA1(1a2df03640b8ac67d17410a472dab7392873b044) )
	ROM_LOAD16_BYTE( "95412171.hi", 0x00000, 0x080000, CRC(4b7999be) SHA1(0f5431ebd037a44269aba212190ea4e6f10bb621) )
	sc_mono_alt_others
ROM_END

ROM_START( sc4mono5a )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95413170.lo", 0x00001, 0x080000, CRC(e99a0858) SHA1(013c8455cf09f341bb43e048edfd0a9755c3ae37) )
	ROM_LOAD16_BYTE( "95413171.hi", 0x00000, 0x080000, CRC(071a2d39) SHA1(60500127a6b3666bc1b220ed73090476820a3584) )
	sc_mono_alt_others
ROM_END



ROM_START( sc4monop )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95408978.lo", 0x00001, 0x080000, CRC(b27e529e) SHA1(f73fcd70f48789914958890972a85ee1f1e7cb58) )
	ROM_LOAD16_BYTE( "95408979.hi", 0x00000, 0x080000, CRC(04da3c33) SHA1(9f9d1b5410813b54d671f2f0272f93e0edefa769) )
	sc_mcas_others
ROM_END

ROM_START( sc4monopa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95410474.lo", 0x00001, 0x080000, CRC(5612849e) SHA1(154aa5e60e09a70d3b8e22df959d9593ffd5c25c) )
	ROM_LOAD16_BYTE( "95410475.hi", 0x00000, 0x080000, CRC(e28212d1) SHA1(e3b0fe66f675ed3b7ebdab244e2e269842bed143) )
	sc_mcas_others
ROM_END

ROM_START( sc4monoa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414454.lo", 0x00001, 0x080000, CRC(9faebd13) SHA1(d9253d30803836cc8e8eff6587878659a9e71267) )
	ROM_LOAD16_BYTE( "95414455.hi", 0x00000, 0x080000, CRC(1b17644b) SHA1(ac53525daf0780345ea87749a5f8e8e9dc5b61c8) )
	sc_mcas_others
ROM_END

ROM_START( sc4monoaa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414460.lo", 0x00001, 0x080000, CRC(24a640d4) SHA1(030a5b1349f35bd2c1fff3135d2a9562999b61fc) )
	ROM_LOAD16_BYTE( "95414461.hi", 0x00000, 0x080000, CRC(ca2658cf) SHA1(f78eaa3fb69392b5c673e4801e177a424a8f90d5) )
	sc_mcas_others
ROM_END

ROM_START( sc4monoan )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414462.lo", 0x00001, 0x080000, CRC(a7a0c83e) SHA1(a24ab4d8667c6fab2bc1893e0a3b50bedac43b5c) )
	ROM_LOAD16_BYTE( "95414463.hi", 0x00000, 0x080000, CRC(6410d507) SHA1(9b6aeb4683456bd82269a717124c49162cb24878) )
	sc_mcas_others
ROM_END

ROM_START( sc4monoal )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414480.lo", 0x00001, 0x080000, CRC(9d678b10) SHA1(26a2aad2f7bced36e8ad85c3fbe781df7d698c09) )
	ROM_LOAD16_BYTE( "95414481.hi", 0x00000, 0x080000, CRC(e0f15310) SHA1(6ee52b756f3a69c96cec5d7363abb5e77227004f) )
	sc_mcas_others
ROM_END


ROM_START( sc4monoab )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414784.lo", 0x00001, 0x080000, CRC(95591c90) SHA1(24dfc4dd1d3b6d05d9a123bcd7441c78f10866a5) )
	ROM_LOAD16_BYTE( "95414785.hi", 0x00000, 0x080000, CRC(09bbf43c) SHA1(78e94f61333ccf7603fd083832cee7114584f593) )
	sc_mcas_others
ROM_END

ROM_START( sc4monoac )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414790.lo", 0x00001, 0x080000, CRC(a7121df8) SHA1(638902dea8950c5668911b3c88a7bcf0af8acf54) )
	ROM_LOAD16_BYTE( "95414791.hi", 0x00000, 0x080000, CRC(04036214) SHA1(b1ee059c44f285e10b56ef5643d8fc68bb1e4502) )
	sc_mcas_others
ROM_END

ROM_START( sc4monoad )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415454.lo", 0x00001, 0x080000, CRC(895b8d82) SHA1(017baf2971576638d4b8f976f99e6097abe193e4) )
	ROM_LOAD16_BYTE( "95415455.hi", 0x00000, 0x080000, CRC(37f16d0d) SHA1(76430ae7e79d9312cbea4b49e0b707f46b48a56a) )
	sc_mcas_others
ROM_END

ROM_START( sc4monoae )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415456.lo", 0x00001, 0x080000, CRC(3031b8db) SHA1(7aaa9b9a585f0407177d4959992000d2d77d4d69) )
	ROM_LOAD16_BYTE( "95415457.hi", 0x00000, 0x080000, CRC(990f11aa) SHA1(549ae1cebaf6069300dc9048cae448235d69587b) )
	sc_mcas_others
ROM_END

ROM_START( sc4monoaf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415458.lo", 0x00001, 0x080000, CRC(81091dce) SHA1(64d5352d2c5bccba496834dc2fe61e3fe9c3e49f) )
	ROM_LOAD16_BYTE( "95415459.hi", 0x00000, 0x080000, CRC(a9a46e04) SHA1(82a019172bf2a6d440528e19c7c09e7b1c9d4980) )
	sc_mcas_others
ROM_END

ROM_START( sc4monoag )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415460.lo", 0x00001, 0x080000, CRC(05fd1a11) SHA1(69eeb4430453ce192f8f3869096e59c7335ea598) )
	ROM_LOAD16_BYTE( "95415461.hi", 0x00000, 0x080000, CRC(41c97eea) SHA1(246f7ad3f0a73e407b2ca6818bd02b1832f75344) )
	sc_mcas_others
ROM_END

ROM_START( sc4monoao )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415462.lo", 0x00001, 0x080000, CRC(82fab458) SHA1(22b78f17eec971cc4ccd5cf61387983f53c88f2e) )
	ROM_LOAD16_BYTE( "95415463.hi", 0x00000, 0x080000, CRC(ade4d484) SHA1(ae6e942780c951370b9ed19d209d533dc20e9f59) )
	sc_mcas_others
ROM_END

ROM_START( sc4monoap )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415464.lo", 0x00001, 0x080000, CRC(33346ded) SHA1(2cca98cc51bdf2f3ebffdab9983c48d6689e0562) )
	ROM_LOAD16_BYTE( "95415465.hi", 0x00000, 0x080000, CRC(7f27c676) SHA1(ee5972174f307b11cc4c18fa3b2dd3b0543e8401) )
	sc_mcas_others
ROM_END

ROM_START( sc4monoaq )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415466.lo", 0x00001, 0x080000, CRC(560cc6d3) SHA1(d3df1cd502dc9e2513d89389e8d13f6b1967c99c) )
	ROM_LOAD16_BYTE( "95415467.hi", 0x00000, 0x080000, CRC(7046b475) SHA1(a3bba1a11490c16495148d3a8f486d1716f877a5) )
	sc_mcas_others
ROM_END

ROM_START( sc4monoam )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415480.lo", 0x00001, 0x080000, CRC(bbd9a433) SHA1(0de5c345569eb266affaf4bdb8c0d353cf757f9b) )
	ROM_LOAD16_BYTE( "95415481.hi", 0x00000, 0x080000, CRC(26dc3bfa) SHA1(07c2e9c0a68e860ddeb6ece5e9c94eaf572dd93d) )
	sc_mcas_others
ROM_END

ROM_START( sc4monoah )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415784.lo", 0x00001, 0x080000, CRC(7f9bcded) SHA1(117e652fcc29c2acffba13c0082ac10c52f983e2) )
	ROM_LOAD16_BYTE( "95415785.hi", 0x00000, 0x080000, CRC(0885584c) SHA1(6ad16cb00535198b21a91574f953f2437c57ef1d) )
	sc_mcas_others
ROM_END

ROM_START( sc4monoai )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415786.lo", 0x00001, 0x080000, CRC(e639315d) SHA1(24fad5452338161e0838249229118608fb09417f) )
	ROM_LOAD16_BYTE( "95415787.hi", 0x00000, 0x080000, CRC(7e896b08) SHA1(887d25ee15784d7b672f1d85d17fa04d98ed2e31) )
	sc_mcas_others
ROM_END

ROM_START( sc4monoaj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415788.lo", 0x00001, 0x080000, CRC(cf1ab197) SHA1(eb1e166e057505b0957871c2f00b08a97428ca7c) )
	ROM_LOAD16_BYTE( "95415789.hi", 0x00000, 0x080000, CRC(297508d2) SHA1(86933487fb4c63f0159f3b133509db503a4c6229) )
	sc_mcas_others
ROM_END

ROM_START( sc4monoak )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415790.lo", 0x00001, 0x080000, CRC(34ff1c8e) SHA1(41cfc6ef8ed15fa4ce368bd9a4e0a7e46be362db) )
	ROM_LOAD16_BYTE( "95415791.hi", 0x00000, 0x080000, CRC(83e98ff0) SHA1(70c69ee23383530d0252b84895027af2d651fd07) )
	sc_mcas_others
ROM_END


ROM_START( sc4monog )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414944.lo", 0x00001, 0x080000, CRC(cae778fc) SHA1(b1424f98457d1d5b7011b72cc5dfb3d1f7128ad0) )
	ROM_LOAD16_BYTE( "95414945.hi", 0x00000, 0x080000, CRC(eb440e83) SHA1(8e57cf0d88611fd6d63422cc345c603fe4bd8f95) )
	sc_monop_others
ROM_END

ROM_START( sc4monoga )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414946.lo", 0x00001, 0x080000, CRC(f01aedbc) SHA1(a8c7bb88faf502611875cd632aef0829dc56b469) )
	ROM_LOAD16_BYTE( "95414947.hi", 0x00000, 0x080000, CRC(b9186430) SHA1(b8d71dce43ee9fc4e8634af0d2d5d35bf290ba91) )
	sc_monop_others
ROM_END

ROM_START( sc4mondx )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414948.lo", 0x00001, 0x080000, CRC(6f7cfe6e) SHA1(d4953815dfd058714715470bba4264c08dbfeaba) )
	ROM_LOAD16_BYTE( "95414949.hi", 0x00000, 0x080000, CRC(cecf1348) SHA1(79adbfd95ecda81cb4dccab05f162133a916ca89) )
	sc_monop_others
ROM_END


ROM_START( sc4monot )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414492.lo", 0x00001, 0x080000, CRC(8164956c) SHA1(25e79399fd2e23ebbde581896c32285b4fbb06f2) )
	ROM_LOAD16_BYTE( "95414493.hi", 0x00000, 0x080000, CRC(3bf28321) SHA1(efddbb5ebcf15d74361d039334d21d52cc5fbe9a) )
	sc_monl_sounds
ROM_END

ROM_START( sc4monota )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414494.lo", 0x00001, 0x080000, CRC(2b2bab9c) SHA1(8b1b4bb62e2f7d47d4b598815731b969f3ab1ba5) )
	ROM_LOAD16_BYTE( "95414495.hi", 0x00000, 0x080000, CRC(ba683acd) SHA1(15b33c74abd86d26f7b03b121e766e240a23ec1e) )
	sc_monl_sounds
ROM_END

ROM_START( sc4monotb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415494.lo", 0x00001, 0x080000, CRC(2bffa151) SHA1(fbf1a18a92434a19be1f53d2967ffca3b0b1da6a) )
	ROM_LOAD16_BYTE( "95415495.hi", 0x00000, 0x080000, CRC(1f831b76) SHA1(bf91cb69d482aba9df2fc5e7404df8f610b57ae3) )
	sc_monl_sounds
ROM_END

ROM_START( sc4monotc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95412230.lo", 0x00001, 0x080000, CRC(45f66b73) SHA1(049bfdbd73a2b8bd8222d0c1eca15c2d64414d51) )
	ROM_LOAD16_BYTE( "95412231.hi", 0x00000, 0x080000, CRC(aac2e0a4) SHA1(e652dfd759128323b0f40a4a4118186a1dd57c23) )
	sc_monl_sounds
ROM_END




ROM_START( sc4mondxa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415948.lo", 0x00001, 0x080000, CRC(b26d891e) SHA1(092dfbbe19b79a07df2d635b19da62a8c54d60e1) )
	ROM_LOAD16_BYTE( "95415949.hi", 0x00000, 0x080000, CRC(05d91c13) SHA1(593478fb88717a32e36a67d277a8620d643dfec5) )
	sc_monop_others
ROM_END

ROM_START( sc4mondxd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415950.lo", 0x00001, 0x080000, CRC(a83ec6a0) SHA1(df9312566530018e3b7f5af5b083423d7a738040) )
	ROM_LOAD16_BYTE( "95415951.hi", 0x00000, 0x080000, CRC(9af2b8b7) SHA1(1e810de18cae8cd2cf77e542d705956d25bd9492) )
	sc_monop_others
ROM_END

ROM_START( sc4mondxe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416018.lo", 0x00001, 0x080000, CRC(63401675) SHA1(1b30d39abbdfda0b159ffab9a45148ac5115f957) )
	ROM_LOAD16_BYTE( "95416019.hi", 0x00000, 0x080000, CRC(751deaca) SHA1(c9a0bccbc944b9f20975b2c9f487ad10c6b91239) )
	sc_monop_others
ROM_END




ROM_START( sc4mondxb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95416020.lo", 0x00001, 0x080000, CRC(03cc1803) SHA1(f8930e37be21059de9ffc1f0265f12dbc3122e86) )
	ROM_LOAD16_BYTE( "95416021.hi", 0x00000, 0x080000, CRC(bb5f02b2) SHA1(8de3181c611fae440f26d901913dac40adf7804e) )
	sc_monop_others
ROM_END

ROM_START( sc4mondxf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417018.lo", 0x00001, 0x080000, CRC(1165520d) SHA1(b463c530f5581d5f8f168a606931efdbb859a2a2) )
	ROM_LOAD16_BYTE( "95417019.hi", 0x00000, 0x080000, CRC(dd52faf8) SHA1(ee58ec64dfffbaf0d846afd7ea7ba4a513b33869) )
	sc_monop_others
ROM_END

ROM_START( sc4mondxc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95417020.lo", 0x00001, 0x080000, CRC(a7ce7d6f) SHA1(481bb67c33e45af67f3e25adb0c96737a8c81fa4) )
	ROM_LOAD16_BYTE( "95417021.hi", 0x00000, 0x080000, CRC(e293103e) SHA1(d8f0e3292ae545c729e639b097ead9aedacc7089) )
	sc_monop_others
ROM_END



ROM_START( sc4mondxg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414950.lo", 0x00001, 0x080000, CRC(43b9da3c) SHA1(bcc473bb48e090052aab6646738fff750aeaa9f0) )
	ROM_LOAD16_BYTE( "95414951.hi", 0x00000, 0x080000, CRC(1360c672) SHA1(837b4172f05119555fc6874aeaacfdbeeacb381c) )
	sc_monop_others
ROM_END



ROM_START( sc4cmon )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414526.lo", 0x00001, 0x080000, CRC(ea7dba5e) SHA1(e3f803ab5c4101e11084c30279679839ba60717f) )
	ROM_LOAD16_BYTE( "95414527.hi", 0x00000, 0x080000, CRC(7095fbe4) SHA1(8b52db7d3ca52747a31894a88c7acecda1af277a) )
	sc_monl_sounds
ROM_END


ROM_START( sc4cmona )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414752.lo", 0x00001, 0x080000, CRC(21b6f763) SHA1(7dab85f5da4ec18fe94b5f203ce7724784ff36d1) )
	ROM_LOAD16_BYTE( "95414753.hi", 0x00000, 0x080000, CRC(bbe5e9a6) SHA1(7b915f711453ad0e078ab2aabd384138782a395a) )
	sc_monl_sounds
ROM_END


ROM_START( sc4cmonb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95414758.lo", 0x00001, 0x080000, CRC(bcda234a) SHA1(e48563941e26fa900587ac2de4b0858deb467b3e) )
	ROM_LOAD16_BYTE( "95414759.hi", 0x00000, 0x080000, CRC(f8c45405) SHA1(f22c42cc38e2da2ca6f74c57cfbca3c2c9bd924f) )
	sc_monl_sounds
ROM_END


ROM_START( sc4cmonc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415520.lo", 0x00001, 0x080000, CRC(9293c6ac) SHA1(ac65fb59f0da86031ea8f3b1249f75685105152c) )
	ROM_LOAD16_BYTE( "95415521.hi", 0x00000, 0x080000, CRC(91534979) SHA1(d1618a40c4cc32e0f825603b58cacaa9b59da0a3) )
	sc_monl_sounds
ROM_END


ROM_START( sc4cmond )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415522.lo", 0x00001, 0x080000, CRC(2cbfa7b1) SHA1(295dee90b19612879d12c72675f8327a2b1f0d2a) )
	ROM_LOAD16_BYTE( "95415523.hi", 0x00000, 0x080000, CRC(c13bb898) SHA1(c52449e479c5629851a89d43f9f29fb7373aeca2) )
	sc_monl_sounds
ROM_END


ROM_START( sc4cmone )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415526.lo", 0x00001, 0x080000, CRC(b91074c3) SHA1(a78b20fc978a177910dbbe45a37dc5d7d5d5f9dc) )
	ROM_LOAD16_BYTE( "95415527.hi", 0x00000, 0x080000, CRC(d5e18c67) SHA1(842d28e15b8e24f551449144947ceabafd54d021) )
	sc_monl_sounds
ROM_END


ROM_START( sc4cmonf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415752.lo", 0x00001, 0x080000, CRC(a001f1a7) SHA1(29405a687e1c8bad471f96fdb2b091074f33475b) )
	ROM_LOAD16_BYTE( "95415753.hi", 0x00000, 0x080000, CRC(8e36db12) SHA1(155999b67b4a01a5d5f8e460e2a49e023cb63146) )
	sc_monl_sounds
ROM_END


ROM_START( sc4cmong )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415754.lo", 0x00001, 0x080000, CRC(6fdff81a) SHA1(3517866f169aa05d728878b974b971cb9b1d198c) )
	ROM_LOAD16_BYTE( "95415755.hi", 0x00000, 0x080000, CRC(3c94180f) SHA1(498d617c29b99159e34dab4a07d8c38e22331176) )
	sc_monl_sounds
ROM_END


ROM_START( sc4cmonh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415756.lo", 0x00001, 0x080000, CRC(c28a5aed) SHA1(2b6fa68615352ef327a5d44bced1456d89c50e1c) )
	ROM_LOAD16_BYTE( "95415757.hi", 0x00000, 0x080000, CRC(f9242cf5) SHA1(32cfd4cdc121d74673b6ebd04ccee1d0328b7928) )
	sc_monl_sounds
ROM_END


ROM_START( sc4cmoni )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95415758.lo", 0x00001, 0x080000, CRC(2625469c) SHA1(517be2d09e6c3146a8ecf57fbedc1a9a28a2cf44) )
	ROM_LOAD16_BYTE( "95415759.hi", 0x00000, 0x080000, CRC(3daac87f) SHA1(8748df6ea4e7f35b71f53b79c16979d23b8981f9) )
	sc_monl_sounds
ROM_END


ROM_START( sc4mcask )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420670.lo", 0x00001, 0x080000, CRC(26a16559) SHA1(1fe601d1dd6f98c83b548accae07864ee84a2f45) )
	ROM_LOAD16_BYTE( "95420671.hi", 0x00000, 0x080000, CRC(d314c373) SHA1(47b2d10eccc2b25ac3324f5c5b95e9d092e1d2ff) )
	sc_mcas_others
ROM_END

ROM_START( sc4mcas )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420694.lo", 0x00001, 0x080000, CRC(902cbf08) SHA1(974b67da5c2da372a47569add00f593328de8f30) )
	ROM_LOAD16_BYTE( "95420695.hi", 0x00000, 0x080000, CRC(4b5d5013) SHA1(e0600a752b7ca05fd1f274ee3bbe8e14257a8806) )
	sc_mcas_others
ROM_END


ROM_START( sc4mcasm )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420696.lo", 0x00001, 0x080000, CRC(cd735a28) SHA1(cf98337fc4e6097d2aa12ab76499fa0205a86e40) )
	ROM_LOAD16_BYTE( "95420697.hi", 0x00000, 0x080000, CRC(6200691c) SHA1(0912e9bc70a1a320b0705fbe2bc08eda4c329934) )
	sc_mcas_others
ROM_END


ROM_START( sc4mcasn )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420698.lo", 0x00001, 0x080000, CRC(2ff54095) SHA1(312e91a6051ef5e8cf96bca0cf1a585f1ed98a3c) )
	ROM_LOAD16_BYTE( "95420699.hi", 0x00000, 0x080000, CRC(e1ac2e31) SHA1(562fbf0bf8fb23c36e8515413ffcc38991ca5543) )
	sc_mcas_others
ROM_END


ROM_START( sc4mcaso )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420700.lo", 0x00001, 0x080000, CRC(437922ab) SHA1(dde8166aeff6b59f9f144d6706f2e39fdba697f5) )
	ROM_LOAD16_BYTE( "95420701.hi", 0x00000, 0x080000, CRC(2823836d) SHA1(92d915bd185a51d9827de77093531cf3e0808101) )
	sc_mcas_others
ROM_END


ROM_START( sc4mcasp )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420702.lo", 0x00001, 0x080000, CRC(a701f08b) SHA1(937e04b45d292672bfe09973a997302b1428e2bb) )
	ROM_LOAD16_BYTE( "95420703.hi", 0x00000, 0x080000, CRC(f982b514) SHA1(74e9518795311d29a85ae87ba76ae9d154f46bf6) )
	sc_mcas_others
ROM_END


ROM_START( sc4mcasq )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420704.lo", 0x00001, 0x080000, CRC(b3e43ead) SHA1(4d9af5ca64121cde14ff80933f0a8482e55a4749) )
	ROM_LOAD16_BYTE( "95420705.hi", 0x00000, 0x080000, CRC(bf4377b9) SHA1(80a07b4952873e7c7f936aaeec1516be8f65db90) )
	sc_mcas_others
ROM_END


ROM_START( sc4mcasr )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420706.lo", 0x00001, 0x080000, CRC(498e2e13) SHA1(d8c7580a6b9e1143b940d28c37cc7729d57046c9) )
	ROM_LOAD16_BYTE( "95420707.hi", 0x00000, 0x080000, CRC(e672491b) SHA1(34ce10a9e3e999a3eefeadd354c3ebe7f42952e5) )
	sc_mcas_others
ROM_END


ROM_START( sc4mcass )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420708.lo", 0x00001, 0x080000, CRC(a3b73f0c) SHA1(4f816bd2cf10f97f954cc0b42cf5b0e147904d84) )
	ROM_LOAD16_BYTE( "95420709.hi", 0x00000, 0x080000, CRC(510bf078) SHA1(12d632ee343a37575d2eb2e059ec7f05330938a6) )
	sc_mcas_others
ROM_END


ROM_START( sc4mcast )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420710.lo", 0x00001, 0x080000, CRC(bae1c0fe) SHA1(2630a115d3c47c75a4fbc152ab890234ac1c41e7) )
	ROM_LOAD16_BYTE( "95420711.hi", 0x00000, 0x080000, CRC(f7283222) SHA1(5cdc163ffd8f1d31f107506d1af668563e7f165b) )
	sc_mcas_others
ROM_END


ROM_START( sc4mcasu )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420740.lo", 0x00001, 0x080000, CRC(9d96a632) SHA1(264c74f98e86bf1358b8534e316e0df08cdabdb3) )
	ROM_LOAD16_BYTE( "95420741.hi", 0x00000, 0x080000, CRC(925773b7) SHA1(0b8088128eee9e6b82b5f1417fb649cb7926f088) )
	sc_mcas_others
ROM_END


ROM_START( sc4mcasv )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421694.lo", 0x00001, 0x080000, CRC(16e77364) SHA1(c4d7de59bb0bdd5d49efdc60c8d0d68ae3dbe9f4) )
	ROM_LOAD16_BYTE( "95421695.hi", 0x00000, 0x080000, CRC(6360bc4a) SHA1(c9b689eadb055505bf2df8864bf3321b5df4060b) )
	sc_mcas_others
ROM_END


ROM_START( sc4mcasw )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421696.lo", 0x00001, 0x080000, CRC(9aad3266) SHA1(59f3a0895a22bd5a828cf872423ac30628ae326c) )
	ROM_LOAD16_BYTE( "95421697.hi", 0x00000, 0x080000, CRC(b4c09ab7) SHA1(d6f3bcff3c9db175bd4594f733b8fbda3ebb3aca) )
	sc_mcas_others
ROM_END


ROM_START( sc4mcasx )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421698.lo", 0x00001, 0x080000, CRC(254b46c7) SHA1(d37ad7c5ed0f90ce1d17a5b0a6973a92342d8ded) )
	ROM_LOAD16_BYTE( "95421699.hi", 0x00000, 0x080000, CRC(5aea25f7) SHA1(b30d2e6ce8fa8da2d8c78e27edd958b79287a690) )
	sc_mcas_others
ROM_END


ROM_START( sc4mcasy )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421700.lo", 0x00001, 0x080000, CRC(05e1bec3) SHA1(7ab1652379ffbc83d6d1c73c85973e802bbcc66b) )
	ROM_LOAD16_BYTE( "95421701.hi", 0x00000, 0x080000, CRC(4c09c20e) SHA1(e0eae6c57428427a9ddc7abedc45aac8599c384b) )
	sc_mcas_others
ROM_END


ROM_START( sc4mcasz )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421702.lo", 0x00001, 0x080000, CRC(9d27d446) SHA1(6a7f04e99851e09e56e4d5f9746d36882ffc016e) )
	ROM_LOAD16_BYTE( "95421703.hi", 0x00000, 0x080000, CRC(1a4cb8ab) SHA1(df9ff6c4278c36745a4ca702ab73efb77c334da0) )
	sc_mcas_others
ROM_END


ROM_START( sc4mcas0 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421704.lo", 0x00001, 0x080000, CRC(39d716cf) SHA1(6672254ac9f6c36cf164388a7a2de0f7ae0508c5) )
	ROM_LOAD16_BYTE( "95421705.hi", 0x00000, 0x080000, CRC(1e98fc98) SHA1(afee5b517089f490b5c4438c3902d2385be42b3a) )
	sc_mcas_others
ROM_END


ROM_START( sc4mcas1 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421706.lo", 0x00001, 0x080000, CRC(82886cd7) SHA1(781ee991fc97cc13f91417abaa98ec5ac400a502) )
	ROM_LOAD16_BYTE( "95421707.hi", 0x00000, 0x080000, CRC(4011a5d6) SHA1(27f2c94d5a39926cc837ccf86363c7b969ce6858) )
	sc_mcas_others
ROM_END


ROM_START( sc4mcas2 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421708.lo", 0x00001, 0x080000, CRC(4c8b9c5b) SHA1(42c333c83f2a55189ba70330bb3e8215abac05fb) )
	ROM_LOAD16_BYTE( "95421709.hi", 0x00000, 0x080000, CRC(5bbfc742) SHA1(361a76bff97f9dd645ec80ff390686bb4b7270ec) )
	sc_mcas_others
ROM_END


ROM_START( sc4mcas3 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421710.lo", 0x00001, 0x080000, CRC(1ae0d8a7) SHA1(42eb858e5402465e735a1d4d4f0aace7c82b9a82) )
	ROM_LOAD16_BYTE( "95421711.hi", 0x00000, 0x080000, CRC(5075729f) SHA1(bd66ad9445b86aeab2876bab1923c8bc6ca9058f) )
	sc_mcas_others
ROM_END


ROM_START( sc4mcas4 )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421740.lo", 0x00001, 0x080000, CRC(f46033be) SHA1(4dc7541fb2c0741127b1232db2028e4836a67318) )
	ROM_LOAD16_BYTE( "95421741.hi", 0x00000, 0x080000, CRC(80c9040d) SHA1(6e6a81189f815071d6275f7399bd6b0fdc8e59a3) )
	sc_mcas_others
ROM_END


ROM_START( sc4mdm )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426472.lo", 0x00001, 0x080000, CRC(4c1bc459) SHA1(6bcbf1caea6910f716cc3636bb5fdf17b99f8734) )
	ROM_LOAD16_BYTE( "95426473.hi", 0x00000, 0x080000, CRC(e52a5716) SHA1(4707cc1857dc72fd42f5cd208f547d20f4dfd303) )
	sc_mdm_others
ROM_END

ROM_START( sc4mdma )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427472.lo", 0x00001, 0x080000, CRC(a3f5058b) SHA1(8fa6a8e04ae5bd8a78b2479abc97e565c30b2050) )
	ROM_LOAD16_BYTE( "95427473.hi", 0x00000, 0x080000, CRC(6b862885) SHA1(ba908f12ab5a7803571650fbac1dec4f51a8d983) )
	sc_mdm_others
ROM_END



ROM_START( sc4mhn )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95424522.lo", 0x00001, 0x080000, CRC(47173638) SHA1(95f5c59bb1d986b1be0cab4d0642f7c6f2cc4d21) )
	ROM_LOAD16_BYTE( "95424523.hi", 0x00000, 0x080000, CRC(25dd80e8) SHA1(8da9991f3afaa7f0d7e6fbd9ca91fd8ebd66208f) )
	sc_mhn_others
ROM_END

ROM_START( sc4mhna )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95425522.lo", 0x00001, 0x080000, CRC(589587c2) SHA1(43d5d74e4c0809168025a90790956b501b7ae759) )
	ROM_LOAD16_BYTE( "95425523.hi", 0x00000, 0x080000, CRC(b6d31074) SHA1(ca5358100c46e978aba5520a551371dd31dbe341) )
	sc_mhn_others
ROM_END


ROM_START( sc4mhp )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95422180.lo", 0x00001, 0x080000, CRC(26b8e28a) SHA1(ccfad8b9decfdb3db130ec21410cc1575ee7d1dc) )
	ROM_LOAD16_BYTE( "95422181.hi", 0x00000, 0x080000, CRC(12cea172) SHA1(7b59efc59c17826587efb9ac13888a175810c5b7) )
	sc_mhp_others
ROM_END


ROM_START( sc4mhpa )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95422352.lo", 0x00001, 0x080000, CRC(a407395d) SHA1(a61050c97161a3fb602fc1287d7b016798bd84e2) )
	ROM_LOAD16_BYTE( "95422353.hi", 0x00000, 0x080000, CRC(b8e09157) SHA1(546f87587ff32d631e4483934f9c16a165a36dc3) )
	sc_mhp_others
ROM_END

ROM_START( sc4mhpb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95422388.lo", 0x00001, 0x080000, CRC(d11a0c9e) SHA1(2ddea95b8098b5fa0b05b2ba6cd33f3d4b460d61) )
	ROM_LOAD16_BYTE( "95422389.hi", 0x00000, 0x080000, CRC(f402886f) SHA1(94e1c0802969d9714816d3fbcf1558d7105a3dc3) )
	sc_mhp_others
ROM_END

ROM_START( sc4mhpc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95423180.lo", 0x00001, 0x080000, CRC(0a33f3d1) SHA1(31d1457927a91454d649633de64b056e1075d63f) )
	ROM_LOAD16_BYTE( "95423181.hi", 0x00000, 0x080000, CRC(ccf2df4f) SHA1(380380f0e929d4f9677690752a2940fedfb51394) )
	sc_mhp_others
ROM_END

ROM_START( sc4mhpd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95423352.lo", 0x00001, 0x080000, CRC(c38d46b5) SHA1(eab08c165a1ec97421d2b641e2a1dfaf9f52b6cf) )
	ROM_LOAD16_BYTE( "95423353.hi", 0x00000, 0x080000, CRC(f244c7e0) SHA1(e0e1bb889999e7ec8ab785654219ff78d29a4f34) )
	sc_mhp_others
ROM_END

ROM_START( sc4mhpe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95423388.lo", 0x00001, 0x080000, CRC(aa09bd94) SHA1(769c5f4438f36361e5f0a1af935d3d2004559a4b) )
	ROM_LOAD16_BYTE( "95423389.hi", 0x00000, 0x080000, CRC(a69ab735) SHA1(b87a9de0d61c203da9ecb9cb6e24734275e32f37) )
	sc_mhp_others
ROM_END

ROM_START( sc4mhpf )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426094.lo", 0x00001, 0x080000, CRC(358eec20) SHA1(2a30491ecf4361860e4663df662924c79fc69211) )
	ROM_LOAD16_BYTE( "95426095.hi", 0x00000, 0x080000, CRC(ce2b8548) SHA1(0ef8cfba7a36a44ed3bf614c40ba1577546f59b7) )
	sc_mhp_others
ROM_END

ROM_START( sc4mhpg )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426334.lo", 0x00001, 0x080000, CRC(377dbfb0) SHA1(d2ee6ffbcc545843defd0de1c5afd5dc4f979200) )
	ROM_LOAD16_BYTE( "95426335.hi", 0x00000, 0x080000, CRC(d7bfcc80) SHA1(25dc446557822ef682a35ffe083b72cbdde56f88) )
	sc_mhp_others
ROM_END

ROM_START( sc4mhph )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95426506.lo", 0x00001, 0x080000, CRC(73becfde) SHA1(9985fe2acfeb5a13f80e839247b5d29c0cacf87b) )
	ROM_LOAD16_BYTE( "95426507.hi", 0x00000, 0x080000, CRC(c8508d0b) SHA1(35c67b0c4ce5f3805a5481a8d558e67004357184) )
	sc_mhp_others
ROM_END

ROM_START( sc4mhpi )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427094.lo", 0x00001, 0x080000, CRC(3fa03265) SHA1(d95defc1021774a70eab43a7b589ea1e505d2f73) )
	ROM_LOAD16_BYTE( "95427095.hi", 0x00000, 0x080000, CRC(c695e5d1) SHA1(301369834eb7fd7d4c1b86aa886d0e41a145255b) )
	sc_mhp_others
ROM_END


ROM_START( sc4mhpj )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427334.lo", 0x00001, 0x080000, CRC(f04a3764) SHA1(8ac0e6e75f904e2d199af8d14d40e32fd1b8edf6) )
	ROM_LOAD16_BYTE( "95427335.hi", 0x00000, 0x080000, CRC(2575f364) SHA1(1f28b768fae048ef95263354126717c3a37941b7) )
	sc_mhp_others
ROM_END

ROM_START( sc4mhpk )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95427506.lo", 0x00001, 0x080000, CRC(1d225c6b) SHA1(8b3aa35436d6a2ba1f4537df90493268f596d941) )
	ROM_LOAD16_BYTE( "95427507.hi", 0x00000, 0x080000, CRC(497ac6c3) SHA1(6aa76eb22ff5290710e13b01d0bac463cda01ead) )
	sc_mhp_others
ROM_END

ROM_START( sc4mhpl )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95419854.lo", 0x00001, 0x080000, CRC(8a7b8323) SHA1(7927df3bb734548c8464ca94a188f597b4d8f6aa) )
	ROM_LOAD16_BYTE( "95419855.hi", 0x00000, 0x080000, CRC(52609c25) SHA1(af11b20e91f1a70dedd9570e9a14a6ff769641cc) )
	sc_mhp_others
ROM_END

ROM_START( sc4mhpm )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95422770.lo", 0x00001, 0x080000, CRC(7aa35ca2) SHA1(67be41eb0d19b4bf1e5013249f08eb761c5c1276) )
	ROM_LOAD16_BYTE( "95422771.hi", 0x00000, 0x080000, CRC(30a79dad) SHA1(4854f4427cf46bbe874b2937c22261df4c21af4e) )
	sc_mhp_others
ROM_END

ROM_START( sc4mhpn )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF)
	ROM_LOAD16_BYTE( "95423770.lo", 0x00001, 0x080000, CRC(fb478f7a) SHA1(cbf813b0ccfcaf6ce1a241c9479dc75d3623043c) )
	ROM_LOAD16_BYTE( "95423771.hi", 0x00000, 0x080000, CRC(6217ded1) SHA1(c96041c00211a0cc938267cc39e968cdb93b48f0) )
	sc_mhp_others
ROM_END

ROM_START( sc4mhpo )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95418854.lo", 0x000001, 0x080000, CRC(fa81d1e1) SHA1(f26c45d1fffc52a10f2b08fc2f607f63f94413b6) )
	ROM_LOAD16_BYTE( "95418855.hi", 0x000000, 0x080000, CRC(3e7fbc1d) SHA1(0f82fa92f9e08161e3b74e1890b88a4fdcb23d12) )
	sc_mhp_others
ROM_END


ROM_START( sc4mmb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1931p11.hi", 0x00000, 0x080000, CRC(9452efb8) SHA1(c05e354130194fec75c29affb1e6c33cbe2cf84b) )
	ROM_LOAD16_BYTE( "p1931p11.lo", 0x00001, 0x080000, CRC(41d918b9) SHA1(6522370275e4a49e2e77d03ae2b2205ba992a494) )
	sc_mmb_others
ROM_END

ROM_START( sc4mmba )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "p1931s11.hi", 0x00000, 0x080000, CRC(c173fc27) SHA1(65dd2d16e3919731a9f20db458604c3272a20dec) )
	ROM_LOAD16_BYTE( "p1931s11.lo", 0x00001, 0x080000, CRC(393f7e53) SHA1(9990e5c0c7c12b3883d856b41cbb3141719f97a5) )
	sc_mmb_others
ROM_END



ROM_START( sc4mrh )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420628.lo", 0x00001, 0x080000, CRC(1ea11921) SHA1(ba10d6d8120563b8c141d867d1a019948c7ee9bd) )
	ROM_LOAD16_BYTE( "95420629.hi", 0x00000, 0x080000, CRC(70f2cbad) SHA1(f13f5e8816ca8d5940d6eb69641bfd5aa86480ed) )
	sc_mrh_others
ROM_END

ROM_START( sc4mrha )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420630.lo", 0x00001, 0x080000, CRC(5fd725e1) SHA1(6e12ed87a99a07e5a677ce138adfd0c4585bb9eb) )
	ROM_LOAD16_BYTE( "95420631.hi", 0x00000, 0x080000, CRC(afcbfcc2) SHA1(a9656cb6f644cafba5a64508dab2cdeeb1a457a1) )
	sc_mrh_others
ROM_END

ROM_START( sc4mrhb )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95420632.lo", 0x00001, 0x080000, CRC(aeb00018) SHA1(fd402b878b4e7b2d6206e4a8a444639f7b56f9a6) )
	ROM_LOAD16_BYTE( "95420633.hi", 0x00000, 0x080000, CRC(72e50b04) SHA1(9eb0b5b2c7d8ca15556d0cf65fb304092a77e69a) )
	sc_mrh_others
ROM_END

ROM_START( sc4mrhc )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421628.lo", 0x00001, 0x080000, CRC(151d9586) SHA1(41817b82e067e16adf9f47bd53d527a51ffb628f) )
	ROM_LOAD16_BYTE( "95421629.hi", 0x00000, 0x080000, CRC(691f2f62) SHA1(8e1602b64f8848088da0da6410c2eef3959165a1) )
	sc_mrh_others
ROM_END

ROM_START( sc4mrhd )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421630.lo", 0x00001, 0x080000, CRC(e4b078ce) SHA1(bc59f503beff592f420c7337c805a1c184e4a118) )
	ROM_LOAD16_BYTE( "95421631.hi", 0x00000, 0x080000, CRC(13526f8e) SHA1(a326dd4d09b12b664101c5db6ccf037162137654) )
	sc_mrh_others
ROM_END

ROM_START( sc4mrhe )
	ROM_REGION( 0x400000, "maincpu", ROMREGION_ERASEFF )
	ROM_LOAD16_BYTE( "95421632.lo", 0x00001, 0x080000, CRC(a5