/*****************************************************************************

Copyright (c) 2006, 2009, Oracle and/or its affiliates. All Rights Reserved.

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation; version 2 of the License.

This program is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License along with
this program; if not, write to the Free Software Foundation, Inc.,
51 Franklin Street, Fifth Floor, Boston, MA 02110-1335 USA

*****************************************************************************/

/**************************************************//**
@file include/row0ext.ic
Caching of externally stored column prefixes

Created September 2006 Marko Makela
*******************************************************/

#include "rem0types.h"
#include "btr0types.h"

/********************************************************************//**
Looks up a column prefix of an externally stored column.
@return column prefix, or NULL if the column is not stored externally,
or pointer to field_ref_zero if the BLOB pointer is unset */
UNIV_INLINE
const byte*
row_ext_lookup_ith(
/*===============*/
	const row_ext_t*	ext,	/*!< in/out: column prefix cache */
	ulint			i,	/*!< in: index of ext->ext[] */
	ulint*			len)	/*!< out: length of prefix, in bytes,
					at most ext->max_len */
{
	ut_ad(ext);
	ut_ad(len);
	ut_ad(i < ext->n_ext);

	*len = ext->len[i];

	ut_ad(*len <= ext->max_len);
	ut_ad(ext->max_len > 0);

	if (*len == 0) {
		/* The BLOB could not be fetched to the cache. */
		return(field_ref_zero);
	} else {
		return(ext->buf + i * ext->max_len);
	}
}

/********************************************************************//**
Looks up a column prefix of an externally stored column.
@return column prefix, or NULL if the column is not stored externally,
or pointer to field_ref_zero if the BLOB pointer is unset */
UNIV_INLINE
const byte*
row_ext_lookup(
/*===========*/
	const row_ext_t*	ext,	/*!< in: column prefix cache */
	ulint			col,	/*!< in: column number in the InnoDB
					table object, as reported by
					dict_col_get_no(); NOT relative to the
					records in the clustered index */
	ulint*			len)	/*!< out: length of prefix, in bytes,
					at most ext->max_len */
{
	ulint	i;

	ut_ad(ext);
	ut_ad(len);

	for (i = 0; i < ext->n_ext; i++) {
		if (col == ext->ext[i]) {
			return(row_ext_lookup_ith(ext, i, len));
		}
	}

	return(NULL);
}
