/*
 * @(#)PyraminxGL.c
 *
 * Copyright 2023 - 2024  David A. Bagley, bagleyd AT verizon.net
 *
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the author not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.
 *
 * This program is distributed in the hope that it will be "useful",
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */

#ifdef HAVE_OPENGL

/* Methods file for PyraminxGL */

#include "PyraminxP.h"
#include "PyraminxGLP.h"

#define glTranslate(x,y,z) glTranslatef((float) x, (float) y, (float) z)

static float front_shininess[] =
{60.0};
static float front_specular[] =
{(float) 0.7, (float) 0.7, (float) 0.7, 1.0};
static float ambient[] =
{0.0, 0.0, 0.0, 1.0};
static float diffuse[] =
{1.0, 1.0, 1.0, 1.0};
static float position0[] =
{1.0, 1.0, 1.0, 0.0};
static float position1[] =
{-1.0, -1.0, 1.0, 0.0};
static float position2[] =
{-1.0, 1.0, -1.0, 0.0};
static float lmodel_ambient[] =
{0.5, 0.5, 0.5, 1.0};
static float lmodel_twoside[] =
{GL_TRUE};

static float face_material[MAX_FACES + 2][4] =
{{1.0, 1.0, 1.0, 1.0},
{1.0, 1.0, 1.0, 1.0},
{1.0, 1.0, 1.0, 1.0},
{1.0, 1.0, 1.0, 1.0},
{0.0, 0.0, 0.0, 1.0},
{0.0, 0.0, 0.0, 1.0}};

/* Blue Red Yellow Green */
/*-
 * Color labels mapping:
 * =====================
 *
 *	        +
 *	       /|\
 *	      / | \
 *           /  |  \
 *          /   |   \
 *         /(0) | (1)\
 *        /BLUE | RED \
 *       /  TOP | RIGHT\
 *      /       |     3 \
 *     /2     --+--    1 \
 *    /0 1 3-- (2) ---2 0 \
 *   /  ---  YELLOW   ---  \
 *  /2--      LEFT       -- \
 * /0 1 3                    \
 * +--------------------------+
 * Flip side: BOTTOM (3) GREEN
 */

static Boolean madeCurrent = False;

#ifdef WINVER
static HGLRC hRC = NULL;
#else
static GLXContext *glXContext = (GLXContext *) NULL;
static Boolean setValuesPuzzleGL(Widget current, Widget request, Widget renew);
static void resizePuzzleGL(PyraminxGLWidget w);
static void initializePuzzleGL(Widget request, Widget renew);
static void exposePuzzleGL(Widget renew, XEvent *event, Region region);
static void movePuzzleGLTl(PyraminxGLWidget w,
	XEvent *event, char **args, int nArgs);
static void movePuzzleGLTop(PyraminxGLWidget w,
	XEvent *event, char **args, int nArgs);
static void movePuzzleGLTr(PyraminxGLWidget w,
	XEvent *event, char **args, int nArgs);
static void movePuzzleGLLeft(PyraminxGLWidget w,
	XEvent *event, char **args, int nArgs);
static void movePuzzleGLRight(PyraminxGLWidget w,
	XEvent *event, char **args, int nArgs);
static void movePuzzleGLBl(PyraminxGLWidget w,
	XEvent *event, char **args, int nArgs);
static void movePuzzleGLBottom(PyraminxGLWidget w,
	XEvent *event, char **args, int nArgs);
static void movePuzzleGLBr(PyraminxGLWidget w,
	XEvent *event, char **args, int nArgs);

static char translationsGL[] =
"<KeyPress>q: Quit()\n\
 Ctrl<KeyPress>C: Quit()\n\
 <KeyPress>osfCancel: Hide()\n\
 <KeyPress>Escape: Hide()\n\
 <KeyPress>osfEscape: Hide()\n\
 Ctrl<KeyPress>[: Hide()\n\
 <KeyPress>0x1B: Hide()\n\
 <KeyPress>0x2E: Speed()\n\
 <KeyPress>0x3E: Speed()\n\
 <KeyPress>0x3C: Slow()\n\
 <KeyPress>0x2C: Slow()\n\
 Shift<KeyPress>2: Sound()\n\
 <KeyPress>F11: MoveCcw()\n\
 <KeyPress>KP_Divide: MoveCcw()\n\
 <KeyPress>R5: MoveCcw()\n\
 <KeyPress>Home: MoveTl()\n\
 <KeyPress>KP_7: MoveTl()\n\
 <KeyPress>R7: MoveTl()\n\
 <KeyPress>Up: MoveTop()\n\
 <KeyPress>osfUp: MoveTop()\n\
 <KeyPress>KP_Up: MoveTop()\n\
 <KeyPress>KP_8: MoveTop()\n\
 <KeyPress>R8: MoveTop()\n\
 <KeyPress>Prior: MoveTr()\n\
 <KeyPress>KP_9: MoveTr()\n\
 <KeyPress>R9: MoveTr()\n\
 <KeyPress>Left: MoveLeft()\n\
 <KeyPress>osfLeft: MoveLeft()\n\
 <KeyPress>KP_Left: MoveLeft()\n\
 <KeyPress>KP_4: MoveLeft()\n\
 <KeyPress>R10: MoveLeft()\n\
 <KeyPress>F12: MoveCw()\n\
 <KeyPress>Begin: MoveCw()\n\
 <KeyPress>KP_5: MoveCw()\n\
 <KeyPress>R11: MoveCw()\n\
 <KeyPress>Right: MoveRight()\n\
 <KeyPress>osfRight: MoveRight()\n\
 <KeyPress>KP_Right: MoveRight()\n\
 <KeyPress>KP_6: MoveRight()\n\
 <KeyPress>R12: MoveRight()\n\
 <KeyPress>End: MoveBl()\n\
 <KeyPress>KP_1: MoveBl()\n\
 <KeyPress>R13: MoveBl()\n\
 <KeyPress>Down: MoveBottom()\n\
 <KeyPress>osfDown: MoveBottom()\n\
 <KeyPress>KP_Down: MoveBottom()\n\
 <KeyPress>KP_2: MoveBottom()\n\
 <KeyPress>R14: MoveBottom()\n\
 <KeyPress>Next: MoveBr()\n\
 <KeyPress>KP_3: MoveBr()\n\
 <KeyPress>R15: MoveBr()\n\
 <Btn1Down>: Select()\n\
 <Btn1Up>: Release()\n\
 <Btn2Down>: PracticeMaybe()\n\
 <Btn2Down>(2+): Practice2()\n\
 <Btn3Down>: RandomizeMaybe()\n\
 <Btn3Down>(2+): Randomize2()\n\
 <Btn4Down>: MoveTop()\n\
 <Btn5Down>: MoveBottom()\n\
 <KeyPress>g: Get()\n\
 <KeyPress>w: Write()\n\
 <KeyPress>u: Undo()\n\
 <KeyPress>r: Redo()\n\
 <KeyPress>c: Clear()\n\
 <KeyPress>z: Randomize()\n\
 <KeyPress>s: Solve()\n\
 <KeyPress>f: Find()\n\
 <KeyPress>p: Practice()\n\
 <KeyPress>i: Increment()\n\
 <KeyPress>d: Decrement()\n\
 <KeyPress>o: Orientize()\n\
 <KeyPress>2: Period2()\n\
 <KeyPress>3: Period3()\n\
 <KeyPress>b: Both()\n\
 <KeyPress>y: Sticky()\n\
 <KeyPress>v: View()\n\
 <EnterWindow>: Enter()\n\
 <LeaveWindow>: Leave()";

static XtActionsRec actionsListGL[] =
{
	{(char *) "Quit", (XtActionProc) quitPuzzle},
	{(char *) "Hide", (XtActionProc) hidePuzzle},
	{(char *) "MoveCcw", (XtActionProc) movePuzzleCcw},
	{(char *) "MoveTl", (XtActionProc) movePuzzleGLTl},
	{(char *) "MoveTop", (XtActionProc) movePuzzleGLTop},
	{(char *) "MoveTr", (XtActionProc) movePuzzleGLTr},
	{(char *) "MoveLeft", (XtActionProc) movePuzzleGLLeft},
	{(char *) "MoveCw", (XtActionProc) movePuzzleCw},
	{(char *) "MoveBr", (XtActionProc) movePuzzleGLBr},
	{(char *) "MoveRight", (XtActionProc) movePuzzleGLRight},
	{(char *) "MoveBl", (XtActionProc) movePuzzleGLBl},
	{(char *) "MoveBottom", (XtActionProc) movePuzzleGLBottom},
	{(char *) "MoveBr", (XtActionProc) movePuzzleGLBr},
	{(char *) "Select", (XtActionProc) selectPuzzle},
	{(char *) "Release", (XtActionProc) releasePuzzle},
	{(char *) "PracticeMaybe", (XtActionProc) practicePuzzleWithQuery},
	{(char *) "Practice2", (XtActionProc) practicePuzzleWithDoubleClick},
	{(char *) "RandomizeMaybe", (XtActionProc) randomizePuzzleWithQuery},
	{(char *) "Randomize2", (XtActionProc) randomizePuzzleWithDoubleClick},
	{(char *) "Get", (XtActionProc) getPuzzle},
	{(char *) "Write", (XtActionProc) writePuzzle},
	{(char *) "Undo", (XtActionProc) undoPuzzle},
	{(char *) "Redo", (XtActionProc) redoPuzzle},
	{(char *) "Clear", (XtActionProc) clearPuzzle},
	{(char *) "Randomize", (XtActionProc) randomizePuzzle},
	{(char *) "Solve", (XtActionProc) solvePuzzle},
	{(char *) "Find", (XtActionProc) findPuzzle},
	{(char *) "Practice", (XtActionProc) practicePuzzle},
	{(char *) "Increment", (XtActionProc) incrementPuzzle},
	{(char *) "Decrement", (XtActionProc) decrementPuzzle},
	{(char *) "Orientize", (XtActionProc) orientizePuzzle},
	{(char *) "Period2", (XtActionProc) period2ModePuzzle},
	{(char *) "Period3", (XtActionProc) period3ModePuzzle},
	{(char *) "Both", (XtActionProc) bothModePuzzle},
	{(char *) "Sticky", (XtActionProc) stickyModePuzzle},
	{(char *) "View", (XtActionProc) viewPuzzle},
	{(char *) "Speed", (XtActionProc) speedUpPuzzle},
	{(char *) "Slow", (XtActionProc) slowDownPuzzle},
	{(char *) "Sound", (XtActionProc) toggleSoundPuzzle},
	{(char *) "Enter", (XtActionProc) enterPuzzle},
	{(char *) "Leave", (XtActionProc) leavePuzzle}
};

static XtResource resourcesGL[] =
{
	{XtNwidth, XtCWidth, XtRDimension, sizeof (Dimension),
	 XtOffset(PyraminxWidget, core.width),
	 XtRString, (caddr_t) "300"},
	{XtNheight, XtCHeight, XtRDimension, sizeof (Dimension),
	 XtOffset(PyraminxWidget, core.height),
	 XtRString, (caddr_t) "400"},
	{XtNmono, XtCMono, XtRBoolean, sizeof (Boolean),
	 XtOffset(PyraminxWidget, pyraminx.mono),
	 XtRString, (caddr_t) "FALSE"},
	{XtNreverseVideo, XtCReverseVideo, XtRBoolean, sizeof (Boolean),
	 XtOffset(PyraminxWidget, pyraminx.reverse),
	 XtRString, (caddr_t) "FALSE"},
	{XtNforeground, XtCForeground, XtRPixel, sizeof (Pixel),
	 XtOffset(PyraminxWidget, pyraminx.foreground),
	 XtRString, (caddr_t) XtDefaultForeground},
	{XtNbackground, XtCBackground, XtRPixel, sizeof (Pixel),
	 XtOffset(PyraminxWidget, pyraminx.background),
	 XtRString, (caddr_t) "#AEB2C3" /*XtDefaultBackground*/},
	{XtNframeColor, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(PyraminxWidget, pyraminx.frameColor),
	 XtRString, (caddr_t) "Cyan" /*XtDefaultForeground*/},
	{XtNfaceColor0, XtCLabel, XtRString, sizeof (String),
	 XtOffset(PyraminxWidget, pyraminx.faceName[0]),
	 XtRString, (caddr_t) "Blue"},
	{XtNfaceColor1, XtCLabel, XtRString, sizeof (String),
	 XtOffset(PyraminxWidget, pyraminx.faceName[1]),
	 XtRString, (caddr_t) "Red"},
	{XtNfaceColor2, XtCLabel, XtRString, sizeof (String),
	 XtOffset(PyraminxWidget, pyraminx.faceName[2]),
	 XtRString, (caddr_t) "Yellow"},
	{XtNfaceColor3, XtCLabel, XtRString, sizeof (String),
	 XtOffset(PyraminxWidget, pyraminx.faceName[3]),
	 XtRString, (caddr_t) "Green"},
	{XtNpieceBorder, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(PyraminxWidget, pyraminx.borderColor),
	 XtRString, (caddr_t) "gray25" /*XtDefaultForeground*/},
	{XtNdelay, XtCDelay, XtRInt, sizeof (int),
	 XtOffset(PyraminxWidget, pyraminx.delay),
	 XtRString, (caddr_t) "10"},
	{XtNsound, XtCSound, XtRBoolean, sizeof (Boolean),
	 XtOffset(PyraminxWidget, pyraminx.sound),
	 XtRString, (caddr_t) "FALSE"},
	{XtNmoveSound, XtCMoveSound, XtRString, sizeof (String),
	 XtOffset(PyraminxWidget, pyraminx.moveSound),
	 XtRString, (caddr_t) MOVESOUND},
	{XtNfont, XtCFont, XtRString, sizeof (String),
	 XtOffset(PyraminxWidget, pyraminx.font),
	 XtRString, (caddr_t) "9x15bold"},
	{XtNview, XtCView, XtRInt, sizeof (int),
	 XtOffset(PyraminxWidget, pyraminx.view),
	 XtRString, (caddr_t) "1"},
	{XtNsize, XtCSize, XtRInt, sizeof (int),
	 XtOffset(PyraminxWidget, pyraminx.size),
	 XtRString, (caddr_t) "3"}, /*DEFAULT_FACETS */
	{XtNsticky, XtCSticky, XtRBoolean, sizeof (Boolean),
	 XtOffset(PyraminxWidget, pyraminx.sticky),
	 XtRString, (caddr_t) "FALSE"}, /* DEFAULT_STICKY */
	{XtNmode, XtCMode, XtRInt, sizeof (int),
	 XtOffset(PyraminxWidget, pyraminx.mode),
	 XtRString, (caddr_t) "3"}, /*DEFAULT_MODE */
	{XtNorient, XtCOrient, XtRBoolean, sizeof (Boolean),
	 XtOffset(PyraminxWidget, pyraminx.orient),
	 XtRString, (caddr_t) "FALSE"},	/* DEFAULT_ORIENT */
	{XtNpractice, XtCPractice, XtRBoolean, sizeof (Boolean),
	 XtOffset(PyraminxWidget, pyraminx.practice),
	 XtRString, (caddr_t) "TRUE"}, /* DEFAULT_PRACTICE */
	{XtNuserName, XtCUserName, XtRString, sizeof (String),
	 XtOffset(PyraminxWidget, pyraminx.userName),
	 XtRString, (caddr_t) ""},
	{XtNscoreFile, XtCScoreFile, XtRString, sizeof (String),
	 XtOffset(PyraminxWidget, pyraminx.scoreFile),
	 XtRString, (caddr_t) ""},
	{XtNscoreOnly, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(PyraminxWidget, pyraminx.scoreOnly),
	 XtRString, (caddr_t) "FALSE"},
	{XtNversionOnly, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(PyraminxWidget, pyraminx.versionOnly),
	 XtRString, (caddr_t) "FALSE"},
	{XtNmenu, XtCMenu, XtRInt, sizeof (int),
	 XtOffset(PyraminxWidget, pyraminx.menu),
	 XtRString, (caddr_t) "999"}, /* ACTION_IGNORE */
	{XtNstart, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(PyraminxWidget, pyraminx.started),
	 XtRString, (caddr_t) "FALSE"},
	{XtNcheat, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(PyraminxWidget, pyraminx.cheat),
	 XtRString, (caddr_t) "FALSE"},
	{XtNface, XtCFace, XtRInt, sizeof (int),
	 XtOffset(PyraminxWidget, pyraminx.currentFace),
	 XtRString, (caddr_t) "-1"},
	{XtNpos, XtCPos, XtRInt, sizeof (int),
	 XtOffset(PyraminxWidget, pyraminx.currentPosition),
	 XtRString, (caddr_t) "-1"},
	{XtNdirection, XtCDirection, XtRInt, sizeof (int),
	 XtOffset(PyraminxWidget, pyraminx.currentDirection),
	 XtRString, (caddr_t) "-1"},
	{XtNstyle, XtCStyle, XtRInt, sizeof (int),
	 XtOffset(PyraminxWidget, pyraminx.currentStyle),
	 XtRString, (caddr_t) "-1"},
	{XtNcontrol, XtCControl, XtRInt, sizeof (int),
	 XtOffset(PyraminxWidget, pyraminx.currentControl),
	 XtRString, (caddr_t) "0"},
	{XtNfast, XtCFast, XtRInt, sizeof (int),
	 XtOffset(PyraminxWidget, pyraminx.currentFast),
	 XtRString, (caddr_t) "1"},
	{XtNpixmapSize, XtCPixmapSize, XtRInt, sizeof (int),
	 XtOffset(PyraminxWidget, pyraminx.pixmapSize),
	 XtRString, (caddr_t) "64"},
	{XtNselectCallback, XtCCallback, XtRCallback, sizeof (caddr_t),
	 XtOffset(PyraminxWidget, pyraminx.select),
	 XtRCallback, (caddr_t) NULL}
};

PyraminxGLClassRec pyraminxGLClassRec =
{
	{
		(WidgetClass) & pyraminxClassRec,	/* superclass */
		(char *) "PyraminxGL",	/* class name */
		sizeof (PyraminxGLRec),	/* widget size */
		NULL,		/* class initialize */
		NULL,		/* class part initialize */
		FALSE,		/* class inited */
		(XtInitProc) initializePuzzleGL,	/* initialize */
		NULL,		/* initialize hook */
		XtInheritRealize,	/* realize */
		actionsListGL,	/* actions */
		XtNumber(actionsListGL),	/* num actions */
		resourcesGL,	/* resources */
		XtNumber(resourcesGL),	/* num resources */
		NULLQUARK,	/* xrm class */
		TRUE,		/* compress motion */
		TRUE,		/* compress exposure */
		TRUE,		/* compress enterleave */
		TRUE,		/* visible interest */
		NULL,		/* destroy */
		(XtWidgetProc) resizePuzzleGL,	/* resize */
		(XtExposeProc) exposePuzzleGL,	/* expose */
		(XtSetValuesFunc) setValuesPuzzleGL,	/* set values */
		NULL,		/* set values hook */
		XtInheritSetValuesAlmost,	/* set values almost */
		NULL,		/* get values hook */
		XtInheritAcceptFocus,	/* accept focus */
		XtVersion,	/* version */
		NULL,		/* callback private */
		translationsGL,	/* tm table */
		NULL,		/* query geometry */
		NULL,		/* display accelerator */
		NULL		/* extension */
	},
	{
		0		/* ignore */
	},
	{
		0		/* ignore */
	}
};

WidgetClass pyraminxGLWidgetClass = (WidgetClass) & pyraminxGLClassRec;
#endif

static void
pickColor(PyraminxGLWidget w, int c, int mono, float *material)
{
#ifdef WINVER
	struct tagColor {
		int red, green, blue;
	} color;
#define MAX_INTENSITY 0xFF
#else
	XColor color;
#define MAX_INTENSITY 0xFFFF
#endif

	switch (c) {
	case TOP_FACE:
	case RIGHT_FACE:
	case LEFT_FACE:
	case BOTTOM_FACE:
#ifdef WINVER
		color.red = GetRValue(w->pyraminx.faceGC[c]);
		color.green = GetGValue(w->pyraminx.faceGC[c]);
		color.blue = GetBValue(w->pyraminx.faceGC[c]);
#else
		color.pixel = w->pyraminx.faceColor[c];
		XQueryColor(XtDisplay(w), DefaultColormapOfScreen(XtScreen(w)),
			&color);
#endif
		break;
	case BORDER_FACE:
#ifdef WINVER
		color.red = GetRValue(w->pyraminx.borderGC);
		color.green = GetGValue(w->pyraminx.borderGC);
		color.blue = GetBValue(w->pyraminx.borderGC);
#else
		color.pixel = w->pyraminx.borderColor;
		XQueryColor(XtDisplay(w), DefaultColormapOfScreen(XtScreen(w)),
			&color);
#endif
		break;
	case NO_FACE:
	default:
#ifdef WINVER
		color.red = GetRValue(w->pyraminx.inverseGC);
		color.green = GetGValue(w->pyraminx.inverseGC);
		color.blue = GetBValue(w->pyraminx.inverseGC);
#else
		color.pixel = w->pyraminx.background;
		XQueryColor(XtDisplay(w), DefaultColormapOfScreen(XtScreen(w)),
			&color);
#endif
		break;
	}
#ifdef DEBUG
	(void) printf("%d: i%d %d %d\n", c, color.red, color.green, color.blue);
#endif
	if (mono) {
		/* really GrayScale */
		float intensity = (float) (0.3 * color.red +
			0.59 * color.green + 0.11 * color.blue);
#ifdef DEBUG
		(void) printf("m%g\n", intensity);
#endif
		material[0] = (float) intensity / MAX_INTENSITY;
		material[1] = material[0];
		material[2] = material[0];
	} else {
		material[0] = (float) color.red / MAX_INTENSITY;
		material[1] = (float) color.green / MAX_INTENSITY;
		material[2] = (float) color.blue / MAX_INTENSITY;
	}
#ifdef DEBUG
	(void) printf("%d: f%g %g %g\n", c, material[0], material[1], material[2]);
#endif
}

typedef float Matrix[3][3];
typedef float Vector[3];

static void
matrixMatrix(Matrix a, Matrix b, Matrix x)
{
	int i, j;

	for (i = 0; i < 3; i++) {
		for (j = 0; j < 3; j++) {
			x[i][j] = a[i][0] * b[0][j] +
				a[i][1] * b[1][j] +
				a[i][2] * b[2][j];
		}
	}
}

static void
matrixVector(Matrix a, Vector v, Vector x)
{
	int i;

	for (i = 0; i < 3; i++) {
		x[i] = a[i][0] * v[0] +
			a[i][1] * v[1] +
			a[i][2] * v[2];
	}
}

static void
setMatrix(float theta, float x, float y, float z, Matrix a)
{
	float s = (float) sin(theta * M_PI / 180.0);
	float c = (float) cos(theta * M_PI / 180.0);

	if (x != 0.0 && y == 0.0 && z == 0.0) {
		a[0][0] = 1.0, a[0][1] = 0.0; a[0][2] = 0.0;
		a[1][0] = 0.0, a[1][1] = c; a[1][2] = -s;
		a[2][0] = 0.0, a[2][1] = s; a[2][2] = c;
	} else if (y != 0.0 && x == 0.0 && z == 0.0) {
		a[0][0] = c, a[0][1] = 0.0; a[0][2] = s;
		a[1][0] = 0.0, a[1][1] = 1.0; a[1][2] = 0.0;
		a[2][0] = -s, a[2][1] = 0.0; a[2][2] = c;
	} else if (z != 0.0 && x == 0.0 && y == 0.0) {
		a[0][0] = c, a[0][1] = -s; a[0][2] = 0.0;
		a[1][0] = s, a[1][1] = c; a[1][2] = 0.0;
		a[2][0] = 0.0, a[2][1] = 0.0; a[2][2] = 1.0;
	}
}

static void
drawStickerlessTetra(void)
{
	glPushMatrix();
	glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
		face_material[BORDER_FACE]);
	glBegin(GL_TRIANGLES);
	/* OPP GREEN FRONT */
	glNormal3f(-1.0, 1.0, 1.0);
	glVertex3f(-FACET_TETRA, FACET_LENGTH, FACET_LENGTH);
	glVertex3f(-FACET_LENGTH, FACET_LENGTH, FACET_TETRA);
	glVertex3f(-FACET_LENGTH, FACET_TETRA, FACET_LENGTH);
	/* OPP RED LEFT */
	glNormal3f(-1.0, -1.0, -1.0);
	glVertex3f(-FACET_LENGTH, -FACET_TETRA, -FACET_LENGTH);
	glVertex3f(-FACET_TETRA, -FACET_LENGTH, -FACET_LENGTH);
	glVertex3f(-FACET_LENGTH, -FACET_LENGTH, -FACET_TETRA);
	/* OPP YELLOW TOP */
	glNormal3f(1.0, 1.0, -1.0);
	glVertex3f(FACET_TETRA, FACET_LENGTH, -FACET_LENGTH);
	glVertex3f(FACET_LENGTH, FACET_LENGTH, -FACET_TETRA);
	glVertex3f(FACET_LENGTH, FACET_TETRA, -FACET_LENGTH);
	/* OPP BLUE RIGHT */
	glNormal3f(1.0, -1.0, 1.0);
	glVertex3f(FACET_LENGTH, -FACET_TETRA, FACET_LENGTH);
	glVertex3f(FACET_LENGTH, -FACET_LENGTH, FACET_TETRA);
	glVertex3f(FACET_TETRA, -FACET_LENGTH, FACET_LENGTH);

	/* TOP_FACE */
	glNormal3f(-1.0, 1.0, -1.0);
	glVertex3f(-FACET_LENGTH, -FACET_TETRA, -FACET_LENGTH);
	glVertex3f(-FACET_LENGTH, FACET_LENGTH, FACET_TETRA);
	glVertex3f(FACET_TETRA, FACET_LENGTH, -FACET_LENGTH);
	/* RIGHT_FACE */
	glNormal3f(1.0, 1.0, 1.0);
	glVertex3f(FACET_LENGTH, FACET_LENGTH, -FACET_TETRA);
	glVertex3f(-FACET_TETRA, FACET_LENGTH, FACET_LENGTH);
	glVertex3f(FACET_LENGTH, -FACET_TETRA, FACET_LENGTH);
	/* LEFT_FACE */
	glNormal3f(-1.0, -1.0, 1.0);
	glVertex3f(FACET_TETRA, -FACET_LENGTH, FACET_LENGTH);
	glVertex3f(-FACET_LENGTH, FACET_TETRA, FACET_LENGTH);
	glVertex3f(-FACET_LENGTH, -FACET_LENGTH, -FACET_TETRA);
	/* BOTTOM_FACE */
	glNormal3f(1.0, -1.0, -1.0);
	glVertex3f(FACET_LENGTH, FACET_TETRA, -FACET_LENGTH);
	glVertex3f(FACET_LENGTH, -FACET_LENGTH, FACET_TETRA);
	glVertex3f(-FACET_TETRA, -FACET_LENGTH, -FACET_LENGTH);
	glEnd();

	glBegin(GL_QUADS);
	/* LEFT BAR */
	glNormal3f(-1.0, 0.0, 0.0);
	glVertex3f(-FACET_LENGTH, -FACET_TETRA, -FACET_LENGTH);
	glVertex3f(-FACET_LENGTH, -FACET_LENGTH, -FACET_TETRA);
	glVertex3f(-FACET_LENGTH, FACET_TETRA, FACET_LENGTH);
	glVertex3f(-FACET_LENGTH, FACET_LENGTH, FACET_TETRA);
	/* TOP BAR */
	glNormal3f(1.0, 0.0, 0.0);
	glVertex3f(FACET_LENGTH, FACET_LENGTH, -FACET_TETRA);
	glVertex3f(FACET_TETRA, FACET_LENGTH, -FACET_LENGTH);
	glVertex3f(-FACET_LENGTH, FACET_LENGTH, FACET_TETRA);
	glVertex3f(-FACET_TETRA, FACET_LENGTH, FACET_LENGTH);
	/* RIGHT BAR */
	glNormal3f(0.0, 0.0, 1.0);
	glVertex3f(FACET_TETRA, -FACET_LENGTH, FACET_LENGTH);
	glVertex3f(FACET_LENGTH, -FACET_TETRA, FACET_LENGTH);
	glVertex3f(-FACET_TETRA, FACET_LENGTH, FACET_LENGTH);
	glVertex3f(-FACET_LENGTH, FACET_TETRA, FACET_LENGTH);
	/* FAR LEFT BAR */
	glNormal3f(0.0, 0.0, -1.0);
	glVertex3f(-FACET_TETRA, -FACET_LENGTH, -FACET_LENGTH);
	glVertex3f(-FACET_LENGTH, -FACET_TETRA, -FACET_LENGTH);
	glVertex3f(FACET_TETRA, FACET_LENGTH, -FACET_LENGTH);
	glVertex3f(FACET_LENGTH, FACET_TETRA, -FACET_LENGTH);
	/* FAR RIGHT BAR */
	glNormal3f(1.0, 0.0, 0.0);
	glVertex3f(FACET_LENGTH, FACET_TETRA, -FACET_LENGTH);
	glVertex3f(FACET_LENGTH, FACET_LENGTH, -FACET_TETRA);
	glVertex3f(FACET_LENGTH, -FACET_TETRA, FACET_LENGTH);
	glVertex3f(FACET_LENGTH, -FACET_LENGTH, FACET_TETRA);
	/* BOTTOM BAR */
	glNormal3f(0.0, -1.0, 0.0);
	glVertex3f(-FACET_LENGTH, -FACET_LENGTH, -FACET_TETRA);
	glVertex3f(-FACET_TETRA, -FACET_LENGTH, -FACET_LENGTH);
	glVertex3f(FACET_LENGTH, -FACET_LENGTH, FACET_TETRA);
	glVertex3f(FACET_TETRA, -FACET_LENGTH, FACET_LENGTH);
	glEnd();
	glPopMatrix();
}

static void
drawStickerlessOcta(int topFace, int rightFace, int bottomFace, int leftFace,
		int topBack, int rightBack, int bottomBack, int leftBack)
{
	glPushMatrix();
	glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
		face_material[BORDER_FACE]);
	glBegin(GL_QUADS);
	if (topFace != NO_POSITION || rightFace != NO_POSITION) {
		glNormal3f(1.0, 1.0, 0.0);
		glVertex3f(CUT, FACET_OCTA, BAR_WIDTH); /* TOP RIGHT BAR */
		glVertex3f(FACET_OCTA, CUT, BAR_WIDTH);
		glVertex3f(FACET_OCTA, CUT, -BAR_WIDTH);
		glVertex3f(CUT, FACET_OCTA, -BAR_WIDTH);
	}
	if (rightFace != NO_POSITION || bottomFace != NO_POSITION) {
		glNormal3f(0.0, 1.0, 1.0);
		glVertex3f(-BAR_WIDTH, CUT, FACET_OCTA); /* TOP FRONT RIGHT BAR */
		glVertex3f(BAR_WIDTH, CUT, FACET_OCTA);
		glVertex3f(BAR_WIDTH, FACET_OCTA, CUT);
		glVertex3f(-BAR_WIDTH, FACET_OCTA, CUT);
	}
	if (bottomFace != NO_POSITION || leftFace != NO_POSITION) {
		glNormal3f(-1.0, 1.0, 0.0);
		glVertex3f(-FACET_OCTA, CUT, BAR_WIDTH); /* TOP FRONT LEFT BAR */
		glVertex3f(-CUT, FACET_OCTA, BAR_WIDTH);
		glVertex3f(-CUT, FACET_OCTA, -BAR_WIDTH);
		glVertex3f(-FACET_OCTA, CUT, -BAR_WIDTH);
	}
	if (topFace != NO_POSITION || leftFace != NO_POSITION) {
		glNormal3f(0.0, 1.0, -1.0);
		glVertex3f(BAR_WIDTH, CUT, -FACET_OCTA); /* TOP LEFT BAR */
		glVertex3f(-BAR_WIDTH, CUT, -FACET_OCTA);
		glVertex3f(-BAR_WIDTH, FACET_OCTA, -CUT);
		glVertex3f(BAR_WIDTH, FACET_OCTA, -CUT);
	}

	if (topFace != NO_POSITION || bottomBack != NO_POSITION) {
		glNormal3f(1.0, 0.0, -1.0);
		glVertex3f(FACET_OCTA, BAR_WIDTH, -CUT); /* BACK BOTTOM BAR */
		glVertex3f(FACET_OCTA, -BAR_WIDTH, -CUT);
		glVertex3f(CUT, -BAR_WIDTH, -FACET_OCTA);
		glVertex3f(CUT, BAR_WIDTH, -FACET_OCTA);
	}
	if (rightFace != NO_POSITION || rightBack != NO_POSITION) {
		glNormal3f(1.0, 0.0, 1.0);
		glVertex3f(FACET_OCTA, -BAR_WIDTH, CUT); /* FAR RIGHT BAR */
		glVertex3f(FACET_OCTA, BAR_WIDTH, CUT);
		glVertex3f(CUT, BAR_WIDTH, FACET_OCTA);
		glVertex3f(CUT, -BAR_WIDTH, FACET_OCTA);
	}
	if (bottomFace != NO_POSITION || topBack != NO_POSITION) {
		glNormal3f(-1.0, 0.0, 1.0);
		glVertex3f(-FACET_OCTA, -BAR_WIDTH, CUT); /* BOTTOM BAR */
		glVertex3f(-CUT, -BAR_WIDTH, FACET_OCTA);
		glVertex3f(-CUT, BAR_WIDTH, FACET_OCTA);
		glVertex3f(-FACET_OCTA, BAR_WIDTH, CUT);
	}
	if (leftFace != NO_POSITION || leftBack != NO_POSITION) {
		glNormal3f(-1.0, 0.0, -1.0);
		glVertex3f(-FACET_OCTA, BAR_WIDTH, -CUT); /* FAR LEFT BAR */
		glVertex3f(-CUT, BAR_WIDTH, -FACET_OCTA);
		glVertex3f(-CUT, -BAR_WIDTH, -FACET_OCTA);
		glVertex3f(-FACET_OCTA, -BAR_WIDTH, -CUT);
	}

	if (rightBack != NO_POSITION || bottomBack != NO_POSITION) {
		glNormal3f(1.0, -1.0, 0.0);
		glVertex3f(FACET_OCTA, -CUT, -BAR_WIDTH); /* BACK RIGHT BAR */
		glVertex3f(FACET_OCTA, -CUT, BAR_WIDTH);
		glVertex3f(CUT, -FACET_OCTA, BAR_WIDTH);
		glVertex3f(CUT, -FACET_OCTA, -BAR_WIDTH);
	}
	if (topBack != NO_POSITION || rightBack != NO_POSITION) {
		glNormal3f(0.0, -1.0, 1.0);
		glVertex3f(-BAR_WIDTH, -CUT, FACET_OCTA); /* BACK UPPER RIGHT BAR */
		glVertex3f(-BAR_WIDTH, -FACET_OCTA, CUT);
		glVertex3f(BAR_WIDTH, -FACET_OCTA, CUT);
		glVertex3f(BAR_WIDTH, -CUT, FACET_OCTA);
	}
	if (topBack != NO_POSITION || leftBack != NO_POSITION) {
		glNormal3f(-1.0, -1.0, 0.0);
		glVertex3f(-CUT, -FACET_OCTA, -BAR_WIDTH); /* BACK UPPER LEFT BAR */
		glVertex3f(-CUT, -FACET_OCTA, BAR_WIDTH);
		glVertex3f(-FACET_OCTA, -CUT, BAR_WIDTH);
		glVertex3f(-FACET_OCTA, -CUT, -BAR_WIDTH);
	}
	if (bottomBack != NO_POSITION || leftBack != NO_POSITION) {
		glNormal3f(0.0, -1.0, -1.0);
		glVertex3f(BAR_WIDTH, -CUT, -FACET_OCTA); /* LEFT BAR */
		glVertex3f(BAR_WIDTH, -FACET_OCTA, -CUT);
		glVertex3f(-BAR_WIDTH, -FACET_OCTA, -CUT);
		glVertex3f(-BAR_WIDTH, -CUT, -FACET_OCTA);
	}
	glEnd();

	/* POINTS */
	if (bottomFace != NO_POSITION || leftFace != NO_POSITION
			|| topBack != NO_POSITION || leftBack != NO_POSITION) {
		glBegin(GL_POLYGON);
		glNormal3f(-1.0, 0.0, 0.0);
		glVertex3f(-FACET_OCTA, -CUT, -BAR_WIDTH); /* LEFT */
		glVertex3f(-FACET_OCTA, -CUT, BAR_WIDTH);
		glVertex3f(-FACET_OCTA, -BAR_WIDTH, CUT);
		glVertex3f(-FACET_OCTA, BAR_WIDTH, CUT);
		glVertex3f(-FACET_OCTA, CUT, BAR_WIDTH);
		glVertex3f(-FACET_OCTA, CUT, -BAR_WIDTH);
		glVertex3f(-FACET_OCTA, BAR_WIDTH, -CUT);
		glVertex3f(-FACET_OCTA, -BAR_WIDTH, -CUT);
		glEnd();
	    if (topFace == NO_POSITION && rightFace == NO_POSITION
			&& rightBack == NO_POSITION && bottomBack == NO_POSITION) {
		/* +0, 1, -1, -1, -1, 5, +6, -1 */
		glBegin(GL_POLYGON);
		glNormal3f(1.0, 0.0, 0.0);
		glVertex3f(0, -BAR_WIDTH, -FACET_OCTA);
		glVertex3f(0, BAR_WIDTH, -FACET_OCTA);
		glVertex3f(0, FACET_OCTA, -BAR_WIDTH);
		glVertex3f(0, FACET_OCTA, BAR_WIDTH);
		glVertex3f(0, BAR_WIDTH, FACET_OCTA);
		glVertex3f(0, -BAR_WIDTH, FACET_OCTA);
		glVertex3f(0, -FACET_OCTA, BAR_WIDTH);
		glVertex3f(0, -FACET_OCTA, -BAR_WIDTH);
		glEnd();
	    }
	}
	if (topFace != NO_POSITION || rightFace != NO_POSITION
			|| rightBack != NO_POSITION || bottomBack != NO_POSITION) {
		glBegin(GL_POLYGON);
		glNormal3f(1.0, 0.0, 0.0);
		glVertex3f(FACET_OCTA, -BAR_WIDTH, -CUT); /* BACK RIGHT */
		glVertex3f(FACET_OCTA, BAR_WIDTH, -CUT);
		glVertex3f(FACET_OCTA, CUT, -BAR_WIDTH);
		glVertex3f(FACET_OCTA, CUT, BAR_WIDTH);
		glVertex3f(FACET_OCTA, BAR_WIDTH, CUT);
		glVertex3f(FACET_OCTA, -BAR_WIDTH, CUT);
		glVertex3f(FACET_OCTA, -CUT, BAR_WIDTH);
		glVertex3f(FACET_OCTA, -CUT, -BAR_WIDTH);
		glEnd();
	    if (bottomFace == NO_POSITION && leftFace == NO_POSITION
			&& topBack == NO_POSITION && leftBack == NO_POSITION) {
		 /* -1, -1, +2, 3, 4, -1, -1, +7 */
		glBegin(GL_POLYGON);
		glNormal3f(-1.0, 0.0, 0.0);
		glVertex3f(0, -FACET_OCTA, -BAR_WIDTH);
		glVertex3f(0, -FACET_OCTA, BAR_WIDTH);
		glVertex3f(0, -BAR_WIDTH, FACET_OCTA);
		glVertex3f(0, BAR_WIDTH, FACET_OCTA);
		glVertex3f(0, FACET_OCTA, BAR_WIDTH);
		glVertex3f(0, FACET_OCTA, -BAR_WIDTH);
		glVertex3f(0, BAR_WIDTH, -FACET_OCTA);
		glVertex3f(0, -BAR_WIDTH, -FACET_OCTA);
		glEnd();
	    }
	}
	if (topFace != NO_POSITION || rightFace != NO_POSITION
			|| bottomFace != NO_POSITION || leftFace != NO_POSITION) {
		glBegin(GL_POLYGON);
		glNormal3f(0.0, 1.0, 0.0);
		glVertex3f(-CUT, FACET_OCTA, -BAR_WIDTH); /* TOP */
		glVertex3f(-CUT, FACET_OCTA, BAR_WIDTH);
		glVertex3f(-BAR_WIDTH, FACET_OCTA, CUT);
		glVertex3f(BAR_WIDTH, FACET_OCTA, CUT);
		glVertex3f(CUT, FACET_OCTA, BAR_WIDTH);
		glVertex3f(CUT, FACET_OCTA, -BAR_WIDTH);
		glVertex3f(BAR_WIDTH, FACET_OCTA, -CUT);
		glVertex3f(-BAR_WIDTH, FACET_OCTA, -CUT);
		glEnd();
	    if (topBack == NO_POSITION && rightBack == NO_POSITION
			&& bottomBack == NO_POSITION && leftBack == NO_POSITION) {
		/* -1, -1, -1, -1, 4, +5, 6, +7 */
		glBegin(GL_POLYGON);
		glNormal3f(0.0, -1.0, 0.0);
		glVertex3f(-BAR_WIDTH, 0, -FACET_OCTA);
		glVertex3f(BAR_WIDTH, 0, -FACET_OCTA);
		glVertex3f(FACET_OCTA, 0, -BAR_WIDTH);
		glVertex3f(FACET_OCTA, 0, BAR_WIDTH);
		glVertex3f(BAR_WIDTH, 0, FACET_OCTA);
		glVertex3f(-BAR_WIDTH, 0, FACET_OCTA);
		glVertex3f(-FACET_OCTA, 0, BAR_WIDTH);
		glVertex3f(-FACET_OCTA, 0, -BAR_WIDTH);
		glEnd();
	    }
	}
	if (topBack != NO_POSITION || rightBack != NO_POSITION
			|| bottomBack != NO_POSITION || leftBack != NO_POSITION) {
		glBegin(GL_POLYGON);
		glNormal3f(0.0, -1.0, 0.0);
		glVertex3f(-BAR_WIDTH, -FACET_OCTA, -CUT); /* BOTTOM */
		glVertex3f(BAR_WIDTH, -FACET_OCTA, -CUT);
		glVertex3f(CUT, -FACET_OCTA, -BAR_WIDTH);
		glVertex3f(CUT, -FACET_OCTA, BAR_WIDTH);
		glVertex3f(BAR_WIDTH, -FACET_OCTA, CUT);
		glVertex3f(-BAR_WIDTH, -FACET_OCTA, CUT);
		glVertex3f(-CUT, -FACET_OCTA, BAR_WIDTH);
		glVertex3f(-CUT, -FACET_OCTA, -BAR_WIDTH);
		glEnd();
	    if (topFace == NO_POSITION && rightFace == NO_POSITION
			&& bottomFace == NO_POSITION && leftFace == NO_POSITION) {
		 /* +0, 1, +2, 3, -1, -1, -1, -1 */
		glBegin(GL_POLYGON);
		glNormal3f(0.0, 1.0, 0.0);
		glVertex3f(-FACET_OCTA, 0, -BAR_WIDTH);
		glVertex3f(-FACET_OCTA, 0, BAR_WIDTH);
		glVertex3f(-BAR_WIDTH, 0, FACET_OCTA);
		glVertex3f(BAR_WIDTH, 0, FACET_OCTA);
		glVertex3f(FACET_OCTA, 0, BAR_WIDTH);
		glVertex3f(FACET_OCTA, 0, -BAR_WIDTH);
		glVertex3f(BAR_WIDTH, 0, -FACET_OCTA);
		glVertex3f(-BAR_WIDTH, 0, -FACET_OCTA);
		glEnd();
	    }
	}
	if (rightFace != NO_POSITION || bottomFace != NO_POSITION
			|| topBack != NO_POSITION || rightBack != NO_POSITION) {
		glBegin(GL_POLYGON);
		glNormal3f(0.0, 0.0, 1.0);
		glVertex3f(-BAR_WIDTH, -CUT, FACET_OCTA); /* RIGHT */
		glVertex3f(BAR_WIDTH, -CUT, FACET_OCTA);
		glVertex3f(CUT, -BAR_WIDTH, FACET_OCTA);
		glVertex3f(CUT, BAR_WIDTH, FACET_OCTA);
		glVertex3f(BAR_WIDTH, CUT, FACET_OCTA);
		glVertex3f(-BAR_WIDTH, CUT, FACET_OCTA);
		glVertex3f(-CUT, BAR_WIDTH, FACET_OCTA);
		glVertex3f(-CUT, -BAR_WIDTH, FACET_OCTA);
		glEnd();
	    if (topFace == NO_POSITION && leftFace == NO_POSITION
			&& bottomBack == NO_POSITION && leftBack == NO_POSITION) {
		/* +0, -1, -1, 3, -1, -1, 6, +7 */
		glBegin(GL_POLYGON);
		glNormal3f(0.0, 0.0, -1.0);
		glVertex3f(-FACET_OCTA, BAR_WIDTH, 0);
		glVertex3f(-BAR_WIDTH, FACET_OCTA, 0);
		glVertex3f(BAR_WIDTH, FACET_OCTA, 0);
		glVertex3f(FACET_OCTA, BAR_WIDTH, 0);
		glVertex3f(FACET_OCTA, -BAR_WIDTH, 0);
		glVertex3f(BAR_WIDTH, -FACET_OCTA, 0);
		glVertex3f(-BAR_WIDTH, -FACET_OCTA, 0);
		glVertex3f(-FACET_OCTA, -BAR_WIDTH, 0);
		glVertex3f(-FACET_OCTA, BAR_WIDTH, 0);
		glEnd();
	    }
	}
	if (topFace != NO_POSITION || leftFace != NO_POSITION
			|| bottomBack != NO_POSITION || leftBack != NO_POSITION) {
		glBegin(GL_POLYGON);
		glNormal3f(0.0, 0.0, -1.0);
		glVertex3f(-CUT, -BAR_WIDTH, -FACET_OCTA); /* BACK LEFT */
		glVertex3f(-CUT, BAR_WIDTH, -FACET_OCTA);
		glVertex3f(-BAR_WIDTH, CUT, -FACET_OCTA);
		glVertex3f(BAR_WIDTH, CUT, -FACET_OCTA);
		glVertex3f(CUT, BAR_WIDTH, -FACET_OCTA);
		glVertex3f(CUT, -BAR_WIDTH, -FACET_OCTA);
		glVertex3f(BAR_WIDTH, -CUT, -FACET_OCTA);
		glVertex3f(-BAR_WIDTH, -CUT, -FACET_OCTA);
		glEnd();
	    if (rightFace == NO_POSITION && bottomFace == NO_POSITION
		    && topBack == NO_POSITION && rightBack == NO_POSITION) {
		/* -1, 1, +2, -1, 4, +5, -1, -1 */
		glBegin(GL_POLYGON);
		glNormal3f(0.0, 0.0, 1.0);
		glVertex3f(-BAR_WIDTH, -FACET_OCTA, 0);
		glVertex3f(BAR_WIDTH, -FACET_OCTA, 0);
		glVertex3f(FACET_OCTA, -BAR_WIDTH, 0);
		glVertex3f(FACET_OCTA, BAR_WIDTH, 0);
		glVertex3f(BAR_WIDTH, FACET_OCTA, 0);
		glVertex3f(-BAR_WIDTH, FACET_OCTA, 0);
		glVertex3f(-FACET_OCTA, BAR_WIDTH, 0);
		glVertex3f(-FACET_OCTA, -BAR_WIDTH, 0);
		glEnd();
	    }
	}

	/* FACES */
	/* TOP BEHIND */
	if (topFace != NO_POSITION) {
		glBegin(GL_POLYGON);
		glNormal3f(1.0, 1.0, -1.0);
		glVertex3f(CUT, BAR_WIDTH, -FACET_OCTA);
		glVertex3f(BAR_WIDTH, CUT, -FACET_OCTA);
		glVertex3f(BAR_WIDTH, FACET_OCTA, -CUT);
		glVertex3f(CUT, FACET_OCTA, -BAR_WIDTH);
		glVertex3f(FACET_OCTA, CUT, -BAR_WIDTH);
		glVertex3f(FACET_OCTA, BAR_WIDTH, -CUT);
		glEnd();
	}
	/* TOP RIGHT */
	if (rightFace != NO_POSITION) {
		glBegin(GL_POLYGON);
		glNormal3f(1.0, 1.0, 1.0);
		glVertex3f(FACET_OCTA, BAR_WIDTH, CUT);
		glVertex3f(FACET_OCTA, CUT, BAR_WIDTH);
		glVertex3f(CUT, FACET_OCTA, BAR_WIDTH);
		glVertex3f(BAR_WIDTH, FACET_OCTA, CUT);
		glVertex3f(BAR_WIDTH, CUT, FACET_OCTA);
		glVertex3f(CUT, BAR_WIDTH, FACET_OCTA);
		glEnd();
	}
	/* FRONT */
	if (bottomFace != NO_POSITION) {
		glBegin(GL_POLYGON);
		glNormal3f(-1.0, 1.0, 1.0);
		glVertex3f(-CUT, BAR_WIDTH, FACET_OCTA); /* RIGHT */
		glVertex3f(-BAR_WIDTH, CUT, FACET_OCTA);
		glVertex3f(-BAR_WIDTH, FACET_OCTA, CUT); /* TOP */
		glVertex3f(-CUT, FACET_OCTA, BAR_WIDTH);
		glVertex3f(-FACET_OCTA, CUT, BAR_WIDTH); /* LEFT */
		glVertex3f(-FACET_OCTA, BAR_WIDTH, CUT);
		glEnd();
	}
	/* TOP LEFT */
	if (leftFace != NO_POSITION) {
		glBegin(GL_POLYGON);
		glNormal3f(-1.0, 1.0, -1.0);
		glVertex3f(-FACET_OCTA, BAR_WIDTH, -CUT); /* LEFT BOTTOM */
		glVertex3f(-FACET_OCTA, CUT, -BAR_WIDTH);
		glVertex3f(-CUT, FACET_OCTA, -BAR_WIDTH); /* TOP */
		glVertex3f(-BAR_WIDTH, FACET_OCTA, -CUT);
		glVertex3f(-BAR_WIDTH, CUT, -FACET_OCTA); /* FAR LEFT */
		glVertex3f(-CUT, BAR_WIDTH, -FACET_OCTA);
		glEnd();
	}
	/* BOTTOM */
	if (topBack != NO_POSITION) {
		glBegin(GL_POLYGON);
		glNormal3f(-1.0, -1.0, 1.0);
		glVertex3f(-FACET_OCTA, -BAR_WIDTH, CUT);
		glVertex3f(-FACET_OCTA, -CUT, BAR_WIDTH);
		glVertex3f(-CUT, -FACET_OCTA, BAR_WIDTH);
		glVertex3f(-BAR_WIDTH, -FACET_OCTA, CUT);
		glVertex3f(-BAR_WIDTH, -CUT, FACET_OCTA);
		glVertex3f(-CUT, -BAR_WIDTH, FACET_OCTA);
		glEnd();
	}
	/* BOTTOM RIGHT */
	if (rightBack != NO_POSITION) {
		glBegin(GL_POLYGON);
		glNormal3f(1.0, -1.0, 1.0);
		glVertex3f(CUT, -BAR_WIDTH, FACET_OCTA);
		glVertex3f(BAR_WIDTH, -CUT, FACET_OCTA);
		glVertex3f(BAR_WIDTH, -FACET_OCTA, CUT);
		glVertex3f(CUT, -FACET_OCTA, BAR_WIDTH);
		glVertex3f(FACET_OCTA, -CUT, BAR_WIDTH);
		glVertex3f(FACET_OCTA, -BAR_WIDTH, CUT);
		glEnd();
	}
	/* BACK */
	if (bottomBack != NO_POSITION) {
		glBegin(GL_POLYGON);
		glNormal3f(1.0, -1.0, -1.0);
		glVertex3f(FACET_OCTA, -BAR_WIDTH, -CUT);
		glVertex3f(FACET_OCTA, -CUT, -BAR_WIDTH);
		glVertex3f(CUT, -FACET_OCTA, -BAR_WIDTH);
		glVertex3f(BAR_WIDTH, -FACET_OCTA, -CUT);
		glVertex3f(BAR_WIDTH, -CUT, -FACET_OCTA);
		glVertex3f(CUT, -BAR_WIDTH, -FACET_OCTA);
		glEnd();
	}
	/* BOTTOM LEFT */
	if (leftBack != NO_POSITION) {
		glBegin(GL_POLYGON);
		glNormal3f(-1.0, -1.0, -1.0);
		glVertex3f(-CUT, -BAR_WIDTH, -FACET_OCTA);
		glVertex3f(-BAR_WIDTH, -CUT, -FACET_OCTA);
		glVertex3f(-BAR_WIDTH, -FACET_OCTA, -CUT);
		glVertex3f(-CUT, -FACET_OCTA, -BAR_WIDTH);
		glVertex3f(-FACET_OCTA, -CUT, -BAR_WIDTH);
		glVertex3f(-FACET_OCTA, -BAR_WIDTH, -CUT);
		glEnd();
	}
	glPopMatrix();
}

/* This helps to Map sizes for the piece faces */
static int truncUnit[MAX_FACES][6][3] =
{
	{ /* 0 Blue */
		{-2, 3, -1},
		{-1, 3, -2},
		{-1, 2, -3},
		{-2, 1, -3},
		{-3, 1, -2},
		{-3, 2, -1}
	}, { /* 1 Red */
		{1, 2, 3},
		{2, 1, 3},
		{3, 1, 2},
		{3, 2, 1},
		{2, 3, 1},
		{1, 3, 2}
	}, { /* 2 Yellow */
		{-3, -1, 2},
		{-3, -2, 1},
		{-2, -3, 1},
		{-1, -3, 2},
		{-1, -2, 3},
		{-2, -1, 3}
	}, { /* 3 Green */
		{1, -2, -3},
		{2, -1, -3},
		{3, -1, -2},
		{3, -2, -1},
		{2, -3, -1},
		{1, -3, -2}
	}
};

static float stickerCoord[6][3];

static void setStickerCoord(int face, float a, float b, float c)
{
	int i, j, value;
	for (i = 0; i < 6; i++) {
		for (j = 0; j < 3; j++) {
			value = truncUnit[face][i][j];
			switch(value) {
			case -1:
				stickerCoord[i][j] = -a;
				break;
			case 1:
				stickerCoord[i][j] = a;
				break;
			case -2:
				stickerCoord[i][j] = -b;
				break;
			case 2:
				stickerCoord[i][j] = b;
				break;
			case -3:
				stickerCoord[i][j] = -c;
				break;
			case 3:
				stickerCoord[i][j] = c;
				break;
			}
		}
	}
}

static void
drawSticker(PyraminxGLWidget w, int face, int position, Boolean tetra)
{
	float center[3], corner[3], mid[3], small[3];
	float width = ((tetra) ? 0.12f : 0.03f); /* TODO */
	int color = (int) w->pyraminx.facetLoc[face][position].face;
	int dir = (int) w->pyraminx.facetLoc[face][position].rotation;
	int i, j, offset = 0;

	dir = (2 - (dir % 3));  /* triangles drawn CCW but orientation defined CW */
	glBegin(GL_POLYGON);
	if (color < 0 || color > MAX_FACES) {
		(void) printf("color not defined\n");
		return;
	}
	glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, face_material[color]);
	if (tetra)
		setStickerCoord(face, -STICKER_TETRA3, STICKER_TETRA2, STICKER_LENGTH2);
	else
		setStickerCoord(face, CUT, GAP_WIDTH, STICKER_OCTA2);
	for (j = 0; j < 3; j++) {
		center[j] = 0;
		corner[j] = 0;
		mid[j] = 0;
	}
	if (face % 2 == 0)
		offset = 2 + face;
	for (i = 0; i < 6; i++) {
		glVertex3f(stickerCoord[i][0], stickerCoord[i][1], stickerCoord[i][2]);
		for (j = 0; j < 3; j++) {
			center[j] += stickerCoord[i][j];
		}
		if (dir * 2 == i || dir * 2 + 1 == i) {
			for (j = 0; j < 3; j++) {
				corner[j] += stickerCoord[(i + 3 + offset) % 6][j];
			}
		}
		if (dir * 2 == i || dir * 2 + 1 == i) {
			for (j = 0; j < 3; j++) {
				mid[j] += stickerCoord[(i + offset) % 6][j];
			}
		}
	}
	glEnd();
	if (w->pyraminx.orient) {
		for (j = 0; j < 3; j++) {
			center[j] = center[j] / 6;
			if (tetra)
				mid[j] = (center[j] + 2 * mid[j]) / 5;
			else
                                mid[j] = (2 * center[j] + mid[j]) / 4;
			small[j] = mid[j] * width;
		}
		/* GL_LINES does not work well */
		/* Hack because lazy about calculating the real value of small */
		glBegin(GL_POLYGON);
		glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, face_material[BORDER_FACE]);
		glVertex3f(center[0] - small[0], center[1] - small[1], center[2] - small[2]);
		glVertex3f(center[0] + small[0], center[1] + small[1], center[2] + small[2]);
		glVertex3f(mid[0] + small[0], mid[1] + small[1], mid[2] + small[2]);
		glVertex3f(mid[0] - small[0], mid[1] - small[1], mid[2] - small[2]);
		glEnd();
		glBegin(GL_POLYGON);
		glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, face_material[BORDER_FACE]);
		glVertex3f(center[0] - small[0], center[1] - small[1], center[2] + small[2]);
		glVertex3f(center[0] + small[0], center[1] + small[1], center[2] - small[2]);
		glVertex3f(mid[0] + small[0], mid[1] + small[1], mid[2] - small[2]);
		glVertex3f(mid[0] - small[0], mid[1] - small[1], mid[2] + small[2]);
		glEnd();
		glBegin(GL_POLYGON);
		glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, face_material[BORDER_FACE]);
		glVertex3f(center[0] - small[0], center[1] + small[1], center[2] - small[2]);
		glVertex3f(center[0] + small[0], center[1] - small[1], center[2] + small[2]);
		glVertex3f(mid[0] + small[0], mid[1] - small[1], mid[2] + small[2]);
		glVertex3f(mid[0] - small[0], mid[1] + small[1], mid[2] - small[2]);
		glEnd();
		glBegin(GL_POLYGON);
		glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, face_material[BORDER_FACE]);
		glVertex3f(center[0] - small[0], center[1] + small[1], center[2] + small[2]);
		glVertex3f(center[0] + small[0], center[1] - small[1], center[2] - small[2]);
		glVertex3f(mid[0] + small[0], mid[1] - small[1], mid[2] - small[2]);
		glVertex3f(mid[0] - small[0], mid[1] + small[1], mid[2] + small[2]);
		glEnd();
		glBegin(GL_POLYGON);
		glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, face_material[BORDER_FACE]);
		glVertex3f(center[0] + small[0], center[1] - small[1], center[2] - small[2]);
		glVertex3f(center[0] - small[0], center[1] + small[1], center[2] + small[2]);
		glVertex3f(mid[0] - small[0], mid[1] + small[1], mid[2] + small[2]);
		glVertex3f(mid[0] + small[0], mid[1] - small[1], mid[2] - small[2]);
		glEnd();
		glBegin(GL_POLYGON);
		glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, face_material[BORDER_FACE]);
		glVertex3f(center[0] + small[0], center[1] - small[1], center[2] + small[2]);
		glVertex3f(center[0] - small[0], center[1] + small[1], center[2] - small[2]);
		glVertex3f(mid[0] - small[0], mid[1] + small[1], mid[2] - small[2]);
		glVertex3f(mid[0] + small[0], mid[1] - small[1], mid[2] + small[2]);
		glEnd();
		glBegin(GL_POLYGON);
		glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, face_material[BORDER_FACE]);
		glVertex3f(center[0] + small[0], center[1] + small[1], center[2] - small[2]);
		glVertex3f(center[0] - small[0], center[1] - small[1], center[2] + small[2]);
		glVertex3f(mid[0] - small[0], mid[1] - small[1], mid[2] + small[2]);
		glVertex3f(mid[0] + small[0], mid[1] + small[1], mid[2] - small[2]);
		glEnd();
		glBegin(GL_POLYGON);
		glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, face_material[BORDER_FACE]);
		glVertex3f(center[0] + small[0], center[1] + small[1], center[2] + small[2]);
		glVertex3f(center[0] - small[0], center[1] - small[1], center[2] - small[2]);
		glVertex3f(mid[0] - small[0], mid[1] - small[1], mid[2] - small[2]);
		glVertex3f(mid[0] + small[0], mid[1] + small[1], mid[2] + small[2]);
		glEnd();
	}
	if (w->pyraminx.currentFace == face &&
			w->pyraminx.currentPosition == position) {
		glBegin(GL_POLYGON);
		glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, face_material[BORDER_FACE]);
		if (tetra)
			setStickerCoord(face, -STICKER_TETRA3, FACET_TETRA, STICKER_LENGTH2);
		else
			setStickerCoord(face, CUT, GAP_WIDTH2, SELECT_OCTA2);
		for (i = 0; i < 6; i++) {
			glVertex3f(stickerCoord[i][0], stickerCoord[i][1], stickerCoord[i][2]);
		}
		glEnd();
	}
}

static void
drawTetraFacet(PyraminxGLWidget w, int top, int right, int left, int bottom)
{
	drawStickerlessTetra();
	glPushMatrix();
	/* TOP LEFT */
	if (top != NO_POSITION) { /* BLUE */
		glNormal3f(-1.0, 1.0, -1.0);
		drawSticker(w, TOP_FACE, top, True);
	}
	/* TOP RIGHT */
	if (right != NO_POSITION) { /* RED */
		glNormal3f(1.0, 1.0, 1.0);
		drawSticker(w, RIGHT_FACE, right, True);
	}
	/* BOTTOM */
	if (left != NO_POSITION) { /* YELLOW */
		glNormal3f(-1.0, -1.0, 1.0);
		drawSticker(w, LEFT_FACE, left, True);
	}
	/* BACK */
	if (bottom != NO_POSITION) { /* GREEN */
		glNormal3f(1.0, -1.0, -1.0);
		drawSticker(w, BOTTOM_FACE, bottom, True);
	}
	glPopMatrix();
}

/* allows small octahedrons to be cut up */
static void
drawOctaFacet8(PyraminxGLWidget w, int topFace, int rightFace, int bottomFace, int leftFace,
		int topBack, int rightBack, int bottomBack, int leftBack)
{
	glPushMatrix();
	drawStickerlessOcta(topFace, rightFace, bottomFace, leftFace,
		topBack, rightBack, bottomBack, leftBack);
	/* TOP LEFT */
	if (leftFace > NO_POSITION) { /* BLUE */
		glNormal3f(-1.0, 1.0, -1.0);
		drawSticker(w, TOP_FACE, leftFace, False);
	}
	/* TOP RIGHT */
	if (rightFace > NO_POSITION) { /* RED */
		glNormal3f(1.0, 1.0, 1.0);
		drawSticker(w, RIGHT_FACE, rightFace, False);
	}
	/* BOTTOM */
	if (topBack > NO_POSITION) { /* YELLOW */
		glNormal3f(-1.0, -1.0, 1.0);
		drawSticker(w, LEFT_FACE, topBack, False);
	}
	/* BACK */
	if (bottomBack > NO_POSITION) { /* GREEN */
		glNormal3f(1.0, -1.0, -1.0);
		drawSticker(w, BOTTOM_FACE, bottomBack, False);
	}
	glPopMatrix();
}

/* compatible with tetra drawing */
static void
drawOctaFacet(PyraminxGLWidget w, int top, int right, int left, int bottom)
{
	glPushMatrix();
	drawStickerlessOcta(1, 1, 1, 1, 1, 1, 1, 1);
	/* TOP LEFT */
	if (top > NO_POSITION) { /* BLUE */
		glNormal3f(-1.0, 1.0, -1.0);
		drawSticker(w, TOP_FACE, top, False);
	}
	/* TOP RIGHT */
	if (right > NO_POSITION) { /* RED */
		glNormal3f(1.0, 1.0, 1.0);
		drawSticker(w, RIGHT_FACE, right, False);
	}
	/* BOTTOM */
	if (left > NO_POSITION) { /* YELLOW */
		glNormal3f(-1.0, -1.0, 1.0);
		drawSticker(w, LEFT_FACE, left, False);
	}
	/* BACK */
	if (bottom > NO_POSITION) { /* GREEN */
		glNormal3f(1.0, -1.0, -1.0);
		drawSticker(w, BOTTOM_FACE, bottom, False);
	}
	glPopMatrix();
}

static void
drawFacet(PyraminxGLWidget w, int top, int right, int left, int bottom,
		Boolean tetra)
{
	if (tetra)
		drawTetraFacet(w, top, right, left, bottom);
	else
		drawOctaFacet(w, top, right, left, bottom);
}

typedef struct _EdgeDepth {
	 int edge, depth;
} EdgeDepth;

static void
edgeDepth(PyraminxWidget w, int face, int position, int direction, EdgeDepth *ed)
{
	CRD crd;

	toCRD(w, face, position, &crd);
	if (direction == TR || direction == BL) {
		ed->edge = 2;
		ed->depth = 0;
		switch (face) {
		case 0:
			ed->depth = w->pyraminx.size - 1 - crd.diagonal;
			break;
		case 1:
			ed->depth = crd.diagonal - w->pyraminx.size;
			break;
		case 2:
			ed->depth = crd.diagonal;
			break;
		case 3:
			ed->depth = 2 * w->pyraminx.size - 1 - crd.diagonal;
			break;
		}
	} else if (direction == TOP || direction == BOTTOM) {
		ed->edge = 1;
		ed->depth = w->pyraminx.size - 1 - crd.column;
	} else if (direction == LEFT || direction == RIGHT) {
		ed->edge = 0;
		if (face >= 2)
			ed->depth = w->pyraminx.size - 1 - crd.row;
		else
			ed->depth = crd.row;
	} else {
		if (face != -1) {
			(void) printf("edgeDepth error: face %d, position %d, direction %d\n",
				face, position, direction);
		}
		ed->edge = 0;
		ed->depth = 0;
	}
	if (position == -1)
		ed->depth = -1;
}

static void
drawEdges(PyraminxGLWidget w, int face, int position, int direction,
		Boolean use, Boolean all)
{
	int s0, s1, s2;
	int i, j, k, lc, rc, z = -2;
	int pos, col, row, sb, sl, sr, mb, ml, mr;
	int size = w->pyraminx.size, edge, middle;
	Boolean tetra, sticky = w->pyraminx.sticky;
	CRD crd;
	EdgeDepth ed;

	toCRD((PyraminxWidget) w, face, position, &crd);
	edgeDepth((PyraminxWidget) w, face, position, direction, &ed);

	lc = (size - 1) * (size - 1);
	rc = size * size - 1;
	if (size == 1 && (all || use))
		drawTetraFacet(w, 0, 0, 0, 0);
	else if (size == 2) {
		if (all) {
			drawOctaFacet(w, 2, 2, 2, 2);
		} else {
			if  ((use && ed.edge == 0 && ed.depth == 0)
					|| (!use && ed.edge == 0 && ed.depth != 0)) {
				drawOctaFacet8(w, z, -1, -1, 2, -1, -1, 2, z);
			}
			if ((use && ed.edge == 0 && ed.depth == size - 1)
					|| (!use && ed.edge == 0 && ed.depth != size - 1)) {
				drawOctaFacet8(w, -1, 2, z, -1, 2, z, -1, -1);
			}
			if ((use && ed.edge == 1 && ed.depth == 0)
					|| (!use && ed.edge == 1 && ed.depth != 0)) {
				drawOctaFacet8(w, z, 2, -1, -1, -1, z, 2, -1);
			}
			if ((use && ed.edge == 1 && ed.depth == size - 1)
					|| (!use && ed.edge == 1 && ed.depth != size - 1)) {
				drawOctaFacet8(w, -1, -1, z, 2, 2, -1, -1, z);
			}
			if ((use && ed.edge == 2 && ed.depth == 0)
					|| (!use && ed.edge == 2 && ed.depth != 0)) {
				drawOctaFacet8(w, z, 2, z, 2, -1, -1, -1, -1);
			}
			if ((use && ed.edge == 2 && ed.depth == size - 1)
					|| (!use && ed.edge == 2 && ed.depth != size - 1)) {
				drawOctaFacet8(w, -1, -1, -1, -1, 2, z, 2, z);
			}
		}
	}
	if (size > 1) { /* CORNERS */
		s0 = size - 1;
		j = lc;
		k = rc;
		if (all || ((use && ed.edge == 1 && ed.depth == 0)
				|| (use && ed.edge != 1 && ed.depth == size - 1)
				|| (!use && ed.edge == 1 && ed.depth != 0)
				|| (!use && ed.edge != 1 && ed.depth != size - 1))) {
			glPushMatrix();
			glTranslate(s0 * FACET_LENGTH, -s0 * FACET_LENGTH, s0 * FACET_LENGTH);
			drawTetraFacet(w, -1, 0, k, j); /* RIGHT */
			glPopMatrix();
		}
		if (all || ((use && ed.edge == 0 && ed.depth == 0)
				|| (use && ed.edge != 0 && ed.depth == size - 1)
				|| (!use && ed.edge == 0 && ed.depth != 0)
				|| (!use && ed.edge != 0 && ed.depth != size - 1))) {
			glPushMatrix();
			glTranslate(-s0 * FACET_LENGTH, -s0 * FACET_LENGTH, -s0 * FACET_LENGTH);
			drawTetraFacet(w, 0, -1, j, k); /* LEFT */
			glPopMatrix();
		}
		if (all || ((use && ed.depth == 0)
				|| (!use && ed.depth != 0))) {
			glPushMatrix();
			glTranslate(s0 * FACET_LENGTH, s0 * FACET_LENGTH, -s0 * FACET_LENGTH);
			drawTetraFacet(w, k, j, -1, 0); /* TOP */
			glPopMatrix();
		}
		if (all || ((use && ed.edge == 2 && ed.depth == 0)
				|| (use && ed.edge != 2 && ed.depth == size - 1)
				|| (!use && ed.edge == 2 && ed.depth != 0)
				|| (!use && ed.edge != 2 && ed.depth != size - 1))) {
			glPushMatrix();
			glTranslate(-s0 * FACET_LENGTH, s0 * FACET_LENGTH, s0 * FACET_LENGTH);
			drawTetraFacet(w, j, k, 0, -1); /* FRONT */
			glPopMatrix();
		}
	}
	if (size > 2) { /* CORNERS - 1 */
		s0 = size - 2;
		i = 2;
		j = (size - 1) * (size - 1) + 1;
		k = size * size - 2;
		if (all) {
			/* ((use && ed.edge == 1 && ed.depth == 0)
				|| (use && ed.edge != 1 && ed.depth == size - 1)
				|| (!use && ed.edge == 1 && ed.depth != 0)
				|| (!use && ed.edge != 1 && ed.depth != size - 1)) */
			/*drawOctaFacet8(w, z, i, z, z, k, z, j, z);*/ /* RIGHT */
			glPushMatrix();
			glTranslate(s0 * FACET_LENGTH, -s0 * FACET_LENGTH, s0 * FACET_LENGTH);
			drawOctaFacet(w, z, i, k, j); /* RIGHT */
			glPopMatrix();
		} else if ((use && ed.edge == 0 && ed.depth == size - 1)
				|| (!use && ed.edge == 0 && ed.depth != size - 1)) {
			glPushMatrix();
			glTranslate(s0 * FACET_LENGTH, -s0 * FACET_LENGTH, s0 * FACET_LENGTH);
			drawOctaFacet8(w, -1, i, z, -1, k, z, -1, -1);
			glPopMatrix();
		} else if ((use && ed.edge == 1 && ed.depth == 0)
				|| (!use && ed.edge == 1 && ed.depth != 0)) {
			glPushMatrix();
			glTranslate(s0 * FACET_LENGTH, -s0 * FACET_LENGTH, s0 * FACET_LENGTH);
			drawOctaFacet8(w, z, i, -1, -1, -1, z, j, -1);
			glPopMatrix();
		} else if ((use && ed.edge == 2 && ed.depth == size - 1)
				|| (!use && ed.edge == 2 && ed.depth != size - 1)) {
			glPushMatrix();
			glTranslate(s0 * FACET_LENGTH, -s0 * FACET_LENGTH, s0 * FACET_LENGTH);
			drawOctaFacet8(w, -1, -1, -1, -1, k, z, j, z);
			glPopMatrix();
		}
		if (all) {
			/* ((use && ed.edge == 0 && ed.depth == 0)
				|| (use && ed.edge != 0 && ed.depth == size - 1)
				|| (!use && ed.edge == 0 && ed.depth != 0)
				|| (!use && ed.edge != 0 && ed.depth != size - 1)) */
			/*drawOctaFacet8(w, z, z, z, i, j, z, k, z);*/ /* LEFT */
			glPushMatrix();
			glTranslate(-s0 * FACET_LENGTH, -s0 * FACET_LENGTH, -s0 * FACET_LENGTH);
			drawOctaFacet(w, i, z, j, k); /* LEFT */
			glPopMatrix();
		} else if ((use && ed.edge == 0 && ed.depth == 0)
				|| (!use && ed.edge == 0 && ed.depth != 0)) {
			glPushMatrix();
			glTranslate(-s0 * FACET_LENGTH, -s0 * FACET_LENGTH, -s0 * FACET_LENGTH);
			drawOctaFacet8(w, z, -1, -1, i, -1, -1, k, z);
			glPopMatrix();
		} else if ((use && ed.edge == 1 && ed.depth == size - 1)
				|| (!use && ed.edge == 1 && ed.depth != size - 1)) {
			glPushMatrix();
			glTranslate(-s0 * FACET_LENGTH, -s0 * FACET_LENGTH, -s0 * FACET_LENGTH);
			drawOctaFacet8(w, -1, -1, z, i, j, -1, -1, z);
			glPopMatrix();
		} else if ((use && ed.edge == 2 && ed.depth == size - 1)
				|| (!use && ed.edge == 2 && ed.depth != size - 1)) {
			glPushMatrix();
			glTranslate(-s0 * FACET_LENGTH, -s0 * FACET_LENGTH, -s0 * FACET_LENGTH);
			drawOctaFacet8(w, -1, -1, -1, -1, j, z, k, z);
			glPopMatrix();
		}
		if (all) {
			/* ((use && ed.depth == 0)
				|| (!use && ed.depth != 0)) */
			/*drawOctaFacet8(w, z, j, z, k, z, z, i, z);*/ /* TOP */
			glPushMatrix();
			glTranslate(s0 * FACET_LENGTH, s0 * FACET_LENGTH, -s0 * FACET_LENGTH);
			drawOctaFacet(w, k, j, z, i); /* TOP */
			glPopMatrix();
		} else if ((use && ed.edge == 0 && ed.depth == 0)
				|| (!use && ed.edge == 0 && ed.depth != 0)) {
			glPushMatrix();
			glTranslate(s0 * FACET_LENGTH, s0 * FACET_LENGTH, -s0 * FACET_LENGTH);
			drawOctaFacet8(w, z, -1, -1, k, -1, -1, i, z);
			glPopMatrix();
		} else if ((use && ed.edge == 1 && ed.depth == 0)
				|| (!use && ed.edge == 1 && ed.depth != 0)) {
			glPushMatrix();
			glTranslate(s0 * FACET_LENGTH, s0 * FACET_LENGTH, -s0 * FACET_LENGTH);
			drawOctaFacet8(w, z, j, -1, -1, -1, z, i, -1);
			glPopMatrix();
		} else if ((use && ed.edge == 2 && ed.depth == 0)
				|| (!use && ed.edge == 2 && ed.depth != 0)) {
			glPushMatrix();
			glTranslate(s0 * FACET_LENGTH, s0 * FACET_LENGTH, -s0 * FACET_LENGTH);
			drawOctaFacet8(w, z, j, z, k, -1, -1, -1, -1);
			glPopMatrix();
		}
		if (all) {
			/* ((use && ed.edge == 2 && ed.depth == 0)
				|| (use && ed.edge != 2 && ed.depth == size - 1)
				|| (!use && ed.edge == 2 && ed.depth != 0)
				|| (!use && ed.edge != 2 && ed.depth != size - 1))*/
			/*drawOctaFacet8(w, z, k, z, j, i, z, z, z);*/ /* FRONT */
			glPushMatrix();
			glTranslate(-s0 * FACET_LENGTH, s0 * FACET_LENGTH, s0 * FACET_LENGTH);
			drawOctaFacet(w, j, k, i, z); /* FRONT */
			glPopMatrix();
		} else if ((use && ed.edge == 0 && ed.depth == size - 1)
				|| (!use && ed.edge == 0 && ed.depth != size - 1)) {
			glPushMatrix();
			glTranslate(-s0 * FACET_LENGTH, s0 * FACET_LENGTH, s0 * FACET_LENGTH);
			drawOctaFacet8(w, -1, k, z, -1, i, z, -1, -1);
			glPopMatrix();
		} else if ((use && ed.edge == 1 && ed.depth == size - 1)
				|| (!use && ed.edge == 1 && ed.depth != size - 1)) {
			glPushMatrix();
			glTranslate(-s0 * FACET_LENGTH, s0 * FACET_LENGTH, s0 * FACET_LENGTH);
			drawOctaFacet8(w, -1, -1, z, j, i, -1, -1, z);
			glPopMatrix();
		} else if ((use && ed.edge == 2 && ed.depth == 0)
				|| (!use && ed.edge == 2 && ed.depth != 0)) {
			glPushMatrix();
			glTranslate(-s0 * FACET_LENGTH, s0 * FACET_LENGTH, s0 * FACET_LENGTH);
			drawOctaFacet8(w, z, k, z, j, -1, -1, -1, -1);
			glPopMatrix();
		}
	}
	if (size > 2) { /* MIDDLE SIDE */ /* 1, 3, 6; 5, 7, 12; 4, 8, 20 */
	    middle = size / 2;
	    tetra = (size % 2 == 1);
	    s1 = size - 1 - ((tetra) ? 0 : 1);
	    i = (size / 2) * (size / 2) + ((tetra) ? 0: 1);
	    j = (size / 2 + 1) * (size / 2 + 1) - 1 - ((tetra) ? 0 : 1);
	    k = size * size - size;
	    if (all || tetra || sticky) {
		if (all || (!sticky && ((use && ed.edge == 0 && ed.depth == size - 1)
				|| (use && ed.edge != 0 && ed.depth == middle)
				|| (!use && ed.edge == 0 && ed.depth != size - 1)
				|| (!use && ed.edge != 0 && ed.depth != middle)))
			|| (sticky && ((use && ed.edge == 0 && ed.depth == size - 1)
				|| (use && ed.edge != 0 && (ed.depth == 1 || ed.depth == 2))
				|| (!use && ed.edge == 0 && ed.depth != size - 1)
				|| (!use && ed.edge != 0 && ed.depth != 1 && ed.depth != 2)))) {
			glPushMatrix();
			glTranslate(0, 0, s1 * FACET_LENGTH);
			drawFacet(w, -1, j, j, -1, tetra);
			glPopMatrix();
		}
		if (all || (!sticky && ((use && ed.edge == 2 && ed.depth == 0)
				|| (use && ed.edge != 2 && ed.depth == middle)
				|| (!use && ed.edge == 2 && ed.depth != 0)
				|| (!use && ed.edge != 2 && ed.depth != middle)))
			|| (sticky && ((use && ed.edge == 2 && ed.depth == 0)
				|| (use && ed.edge != 2 && (ed.depth == 1 || ed.depth == 2))
				|| (!use && ed.edge == 2 && ed.depth != 0)
				|| (!use && ed.edge != 2 && ed.depth != 1 && ed.depth != 2)))) {
			glPushMatrix();
			glTranslate(0, s1 * FACET_LENGTH, 0);
			drawFacet(w, k, k, -1, -1, tetra);
			glPopMatrix();
		}
		if (all || (!sticky && ((use && ed.edge == 1 && ed.depth == 0)
				|| (use && ed.edge != 1 && ed.depth == middle)
				|| (!use && ed.edge == 1 && ed.depth != 0)
				|| (!use && ed.edge != 1 && ed.depth != middle)))
			|| (sticky && ((use && ed.edge == 1 && ed.depth == 0)
				|| (use && ed.edge != 1 && (ed.depth == 1 || ed.depth == 2))
				|| (!use && ed.edge == 1 && ed.depth != 0)
				|| (!use && ed.edge != 1 && ed.depth != 1 && ed.depth != 2)))) {
			glPushMatrix();
			glTranslate(s1 * FACET_LENGTH, 0, 0);
			drawFacet(w, -1, i, -1, i, tetra);
			glPopMatrix();
		}
		if (all || (!sticky && ((use && ed.edge == 0 && ed.depth == 0)
				|| (use && ed.edge != 0 && ed.depth == middle)
				|| (!use && ed.edge == 0 && ed.depth != 0)
				|| (!use && ed.edge != 0 && ed.depth != middle)))
			|| (sticky && ((use && ed.edge == 0 && ed.depth == 0)
				|| (use && ed.edge != 0 && (ed.depth == 1 || ed.depth == 2))
				|| (!use && ed.edge == 0 && ed.depth != 0)
				|| (!use && ed.edge != 0 && ed.depth != 1 && ed.depth != 2)))) {
			glPushMatrix();
			glTranslate(0, 0, -s1 * FACET_LENGTH);
			drawFacet(w, j, -1, -1, j, tetra);
			glPopMatrix();
		}
		if (all || (!sticky && ((use && ed.edge == 2 && ed.depth == size - 1)
				|| (use && ed.edge != 2 && ed.depth == middle)
				|| (!use && ed.edge == 2 && ed.depth != size - 1)
				|| (!use && ed.edge != 2 && ed.depth != middle)))
			|| (sticky && ((use && ed.edge == 2 && ed.depth == size - 1)
				|| (use && ed.edge != 2 && (ed.depth == 1 || ed.depth == 2))
				|| (!use && ed.edge == 2 && ed.depth != size - 1)
				|| (!use && ed.edge != 2 && ed.depth != 1 && ed.depth != 2)))) {
			glPushMatrix();
			glTranslate(0, -s1 * FACET_LENGTH, 0);
			drawFacet(w, -1, -1, k, k, tetra);
			glPopMatrix();
		}
		if (all || (!sticky && ((use && ed.edge == 1 && ed.depth == size - 1)
				|| (use && ed.edge != 1 && ed.depth == middle)
				|| (!use && ed.edge == 1 && ed.depth != size - 1)
				|| (!use && ed.edge != 1 && ed.depth != middle)))
			|| (sticky && ((use && ed.edge == 1 && ed.depth == size - 1)
				|| (use && ed.edge != 1 && (ed.depth == 1 || ed.depth == 2))
				|| (!use && ed.edge == 1 && ed.depth != size - 1)
				|| (!use && ed.edge != 1 && ed.depth != 1 && ed.depth != 2)))) {
			glPushMatrix();
			glTranslate(-s1 * FACET_LENGTH, 0, 0);
			drawFacet(w, i, -1, i, -1, tetra);
			glPopMatrix();
		}
	    } else {
		/* ((use && ed.edge == 0 && ed.depth == 0)
				|| (use && ed.edge != 0 && ed.depth == middle)
				|| (!use && ed.edge == 0 && ed.depth != 0)
				|| (!use && ed.edge != 0 && ed.depth != middle)) */
			/*drawOctaFacet(w, j, -1, -1, j);*/
			/*drawOctaFacet8(w, z, -1, -1, j, -1, -1, j, z);*/
		if ((use && ed.edge == 0 && ed.depth == 0)
				|| (!use && ed.edge == 0 && ed.depth != 0)) {
			glPushMatrix();
			glTranslate(0, 0, -s1 * FACET_LENGTH);
			drawOctaFacet8(w, z, -1, -1, j, -1, -1, j, z);
			glPopMatrix();
		} else if ((use && ed.edge == 1 && ed.depth == middle - 1)
				|| (!use && ed.edge == 1 && ed.depth != middle - 1)) {
			glPushMatrix();
			glTranslate(0, 0, -s1 * FACET_LENGTH);
			drawOctaFacet8(w, z, z, -1, -1, -1, z, j, -1);
			glPopMatrix();
		} else if ((use && ed.edge == 2 && ed.depth == middle)
				|| (!use && ed.edge == 2 && ed.depth != middle)) {
			glPushMatrix();
			glTranslate(0, 0, -s1 * FACET_LENGTH);
			drawOctaFacet8(w, -1, -1, -1, -1, z, z, j, z);
			glPopMatrix();
		}
		if ((use && ed.edge == 1 && ed.depth == middle)
				|| (!use && ed.edge == 1 && ed.depth != middle)) {
			glPushMatrix();
			glTranslate(0, 0, -s1 * FACET_LENGTH);
			drawOctaFacet8(w, -1, -1, z, j, z, -1, -1, z);
			glPopMatrix();
		} else if ((use && ed.edge == 2 && ed.depth == middle - 1)
				|| (!use && ed.edge == 2 && ed.depth != middle - 1)) {
			glPushMatrix();
			glTranslate(0, 0, -s1 * FACET_LENGTH);
			drawOctaFacet8(w, z, z, z, j, -1, -1, -1, -1);
			glPopMatrix();
		}
		/* ((use && ed.edge == 1 && ed.depth == 0)
				|| (use && ed.edge != 1 && ed.depth == middle)
				|| (!use && ed.edge == 1 && ed.depth != 0)
				|| (!use && ed.edge != 1 && ed.depth != middle)) */
			/*drawOctaFacet(w, -1, i, -1, i);*/
			/*drawOctaFacet8(w, z, i, -1, -1, z, -1, i, -1);*/
		if ((use && ed.edge == 1 && ed.depth == 0)
				|| (!use && ed.edge == 1 && ed.depth != 0)) {
			glPushMatrix();
			glTranslate(s1 * FACET_LENGTH, 0, 0);
			drawOctaFacet8(w, z, i, -1, -1, -1, z, i, -1);
			glPopMatrix();
		} else if ((use && ed.edge == 0 && ed.depth == middle)
				|| (!use && ed.edge == 0 && ed.depth != middle)) {
			glPushMatrix();
			glTranslate(s1 * FACET_LENGTH, 0, 0);
			drawOctaFacet8(w, -1, i, z, -1, z, z, -1, -1);
			glPopMatrix();
		} else if ((use && ed.edge == 2 && ed.depth == middle - 1)
				|| (!use && ed.edge == 2 && ed.depth != middle - 1)) {
			glPushMatrix();
			glTranslate(s1 * FACET_LENGTH, 0, 0);
			drawOctaFacet8(w, z, i, z, z, -1, -1, -1, -1);
			glPopMatrix();
		}
		if ((use && ed.edge == 0 && ed.depth == middle - 1)
				|| (!use && ed.edge == 0 && ed.depth != middle - 1)) {
			glPushMatrix();
			glTranslate(s1 * FACET_LENGTH, 0, 0);
			drawOctaFacet8(w, z, -1, -1, z, -1, -1, i, z);
			glPopMatrix();
		} else if ((use && ed.edge == 2 && ed.depth == middle)
				|| (!use && ed.edge == 2 && ed.depth != middle)) {
			glPushMatrix();
			glTranslate(s1 * FACET_LENGTH, 0, 0);
			drawOctaFacet8(w, -1, -1, -1, -1, z, z, i, z);
			glPopMatrix();
		}
		/* ((use && ed.edge == 2 && ed.depth == 0)
				|| (use && ed.edge != 2 && ed.depth == middle)
				|| (!use && ed.edge == 2 && ed.depth != 0)
				|| (!use && ed.edge != 2 && ed.depth != middle)) */
			/*drawOctaFacet(w, k, k, -1, -1);*/
			/*drawOctaFacet8(w, z, k, z, k, -1, -1, -1, -1);*/
		if ((use && ed.edge == 2 && ed.depth == 0)
				|| (!use && ed.edge == 2 && ed.depth != 0)) {
			glPushMatrix();
			glTranslate(0, s1 * FACET_LENGTH, 0);
			drawOctaFacet8(w, z, k, z, k, -1, -1, -1, -1);
			glPopMatrix();
		} else if ((use && ed.edge == 0 && ed.depth == middle - 1)
				|| (!use && ed.edge == 0 && ed.depth != middle - 1)) {
			glPushMatrix();
			glTranslate(0, s1 * FACET_LENGTH, 0);
			drawOctaFacet8(w, z, -1, -1, k, -1, -1, z, z);
			glPopMatrix();
		} else if ((use && ed.edge == 1 && ed.depth == middle)
				|| (!use && ed.edge == 1 && ed.depth != middle)) {
			glPushMatrix();
			glTranslate(0, s1 * FACET_LENGTH, 0);
			drawOctaFacet8(w, -1, -1, z, k, z, -1, -1, z);
			glPopMatrix();
		}
		if ((use && ed.edge == 0 && ed.depth == middle)
				|| (!use && ed.edge == 0 && ed.depth != middle)) {
			glPushMatrix();
			glTranslate(0, s1 * FACET_LENGTH, 0);
			drawOctaFacet8(w, -1, k, z, -1, z, z, -1, -1);
			glPopMatrix();
		} else if ((use && ed.edge == 1 && ed.depth == middle - 1)
				|| (!use && ed.edge == 1 && ed.depth != middle - 1)) {
			glPushMatrix();
			glTranslate(0, s1 * FACET_LENGTH, 0);
			drawOctaFacet8(w, z, k, -1, -1, -1, z, z, -1);
			glPopMatrix();
		}
		/* ((use && ed.edge == 0 && ed.depth == size - 1)
				|| (use && ed.edge != 0 && ed.depth == middle)
				|| (!use && ed.edge == 0 && ed.depth != size - 1)
				|| (!use && ed.edge != 0 && ed.depth != middle)) */
			/*drawOctaFacet(w, -1, j, j, -1);*/
			/*drawOctaFacet8(w, -1, j, z, -1, j, z, -1, -1);*/
		if ((use && ed.edge == 0 && ed.depth == size - 1)
				|| (!use && ed.edge == 0 && ed.depth != size - 1)) {
			glPushMatrix();
			glTranslate(0, 0, s1 * FACET_LENGTH);
			drawOctaFacet8(w, -1, j, z, -1, j, z, -1, -1);
			glPopMatrix();
		} else if ((use && ed.edge == 1 && ed.depth == middle - 1)
				|| (!use && ed.edge == 1 && ed.depth != middle - 1)) {
			glPushMatrix();
			glTranslate(0, 0, s1 * FACET_LENGTH);
			drawOctaFacet8(w, z, j, -1, -1, -1, z, z, -1);
			glPopMatrix();
		} else if ((use && ed.edge == 2 && ed.depth == middle - 1)
				|| (!use && ed.edge == 2 && ed.depth != middle - 1)) {
			glPushMatrix();
			glTranslate(0, 0, s1 * FACET_LENGTH);
			drawOctaFacet8(w, z, j, z, z, -1, -1, -1, -1);
			glPopMatrix();
		}
		if ((use && ed.edge == 1 && ed.depth == middle)
				|| (!use && ed.edge == 1 && ed.depth != middle)) {
			glPushMatrix();
			glTranslate(0, 0, s1 * FACET_LENGTH);
			drawOctaFacet8(w, -1, -1, z, z, j, -1, -1, z);
			glPopMatrix();
		} else if ((use && ed.edge == 2 && ed.depth == middle)
				|| (!use && ed.edge == 2 && ed.depth != middle)) {
			glPushMatrix();
			glTranslate(0, 0, s1 * FACET_LENGTH);
			drawOctaFacet8(w, -1, -1, -1, -1, j, z, z, z);
			glPopMatrix();
		}
		/* ((use && ed.edge == 1 && ed.depth == size - 1)
				|| (use && ed.edge != 1 && ed.depth == middle)
				|| (!use && ed.edge == 1 && ed.depth != size - 1)
				|| (!use && ed.edge != 1 && ed.depth != middle)) */
			/*drawOctaFacet(w, i, -1, i, -1);*/
			/*drawOctaFacet8(w, -1, -1, z, i, i, -1, -1, z);*/
		if ((use && ed.edge == 1 && ed.depth == size - 1)
				|| (!use && ed.edge == 1 && ed.depth != size - 1)) {
			glPushMatrix();
			glTranslate(-s1 * FACET_LENGTH, 0, 0);
			drawOctaFacet8(w, -1, -1, z, i, i, -1, -1, z);
			glPopMatrix();
		} else if ((use && ed.edge == 0 && ed.depth == middle - 1)
				|| (!use && ed.edge == 0 && ed.depth != middle - 1)) {
			glPushMatrix();
			glTranslate(-s1 * FACET_LENGTH, 0, 0);
			drawOctaFacet8(w, z, -1, -1, i, -1, -1, z, z);
			glPopMatrix();
		} else if ((use && ed.edge == 2 && ed.depth == middle - 1)
				|| (!use && ed.edge == 2 && ed.depth != middle - 1)) {
			glPushMatrix();
			glTranslate(-s1 * FACET_LENGTH, 0, 0);
			drawOctaFacet8(w, z, z, z, i, -1, -1, -1, -1);
			glPopMatrix();
		}
		if ((use && ed.edge == 0 && ed.depth == middle)
				|| (!use && ed.edge == 0 && ed.depth != middle)) {
			glPushMatrix();
			glTranslate(-s1 * FACET_LENGTH, 0, 0);
			drawOctaFacet8(w, -1, z, z, -1, i, z, -1, -1);
			glPopMatrix();
		} else if ((use && ed.edge == 2 && ed.depth == middle)
				|| (!use && ed.edge == 2 && ed.depth != middle)) {
			glPushMatrix();
			glTranslate(-s1 * FACET_LENGTH, 0, 0);
			drawOctaFacet8(w, -1, -1, -1, -1, i, z, z, z);
			glPopMatrix();
		}
		/* ((use && ed.edge == 2 && ed.depth == size - 1)
				|| (use && ed.edge != 2 && ed.depth == middle)
				|| (!use && ed.edge == 2 && ed.depth != size - 1)
				|| (!use && ed.edge != 2 && ed.depth != middle)) */
			/*drawOctaFacet(w, -1, -1, k, k);*/
			/*drawOctaFacet8(w, -1, -1, -1, -1, k, z, k, z);*/
		if ((use && ed.edge == 2 && ed.depth == size - 1)
				|| (!use && ed.edge == 2 && ed.depth != size - 1)) {
			glPushMatrix();
			glTranslate(0, -s1 * FACET_LENGTH, 0);
			drawOctaFacet8(w, -1, -1, -1, -1, k, z, k, z);
			glPopMatrix();
		} else if ((use && ed.edge == 0 && ed.depth == middle)
				|| (!use && ed.edge == 0 && ed.depth != middle)) {
			glPushMatrix();
			glTranslate(0, -s1 * FACET_LENGTH, 0);
			drawOctaFacet8(w, -1, z, z, -1, k, z, -1, -1);
			glPopMatrix();
		} else if ((use && ed.edge == 1 && ed.depth == middle)
				|| (!use && ed.edge == 1 && ed.depth != middle)) {
			glPushMatrix();
			glTranslate(0, -s1 * FACET_LENGTH, 0);
			drawOctaFacet8(w, -1, -1, z, z, k, -1, -1, z);
			glPopMatrix();
		}
		if ((use && ed.edge == 0 && ed.depth == middle - 1)
				|| (!use && ed.edge == 0 && ed.depth != middle - 1)) {
			glPushMatrix();
			glTranslate(0, -s1 * FACET_LENGTH, 0);
			drawOctaFacet8(w, z, -1, -1, z, -1, -1, k, z);
			glPopMatrix();
		} else if ((use && ed.edge == 1 && ed.depth == middle - 1)
				|| (!use && ed.edge == 1 && ed.depth != middle - 1)) {
			glPushMatrix();
			glTranslate(0, -s1 * FACET_LENGTH, 0);
			drawOctaFacet8(w, z, z, -1, -1, -1, z, k, -1);
			glPopMatrix();
		}
	    }
	}
	if (size > 2) { /* rest of CORNER - 1 */
		s0 = size - 2;
		i = 2;
		j = (size - 1) * (size - 1) + 1;
		k = size * size - 2;
		/* ((use && ed.edge == 1 && ed.depth == 0)
			|| (use && ed.edge != 1 && ed.depth == size - 1)
			|| (!use && ed.edge == 1 && ed.depth != 0)
			|| (!use && ed.edge != 1 && ed.depth != size - 1)) */
			/*drawOctaFacet8(w, z, i, z, z, k, z, j, z);*/ /* RIGHT */
		if ((!sticky && ((use && ed.edge == 0 && ed.depth == size - 2)
				|| (!use && ed.edge == 0 && ed.depth != size - 2)))
			|| (sticky && ((use && ed.edge == 0 && (ed.depth == 1 || ed.depth == 2))
				|| (!use && ed.edge == 0 && ed.depth != 1 && ed.depth != 2)))) {
			glPushMatrix();
			glTranslate(s0 * FACET_LENGTH, -s0 * FACET_LENGTH, s0 * FACET_LENGTH);
			drawOctaFacet8(w, z, -1, -1, z, -1, -1, j, z);
			glPopMatrix();
		} else if ((!sticky && ((use && ed.edge == 1 && ed.depth == 1)
				|| (!use && ed.edge == 1 && ed.depth != 1)))
			|| (sticky && ((use && ed.edge == 1 && (ed.depth == 1 || ed.depth == 2))
				|| (!use && ed.edge == 1 && ed.depth != 1 && ed.depth != 2)))) {
			glPushMatrix();
			glTranslate(s0 * FACET_LENGTH, -s0 * FACET_LENGTH, s0 * FACET_LENGTH);
			drawOctaFacet8(w, -1, -1, z, z, k, -1, -1, z);
			glPopMatrix();
		} else if ((!sticky && ((use && ed.edge == 2 && ed.depth == size - 2)
				|| (!use && ed.edge == 2 && ed.depth != size - 2)))
			|| (sticky && ((use && ed.edge == 2 && (ed.depth == 1 || ed.depth == 2))
				|| (!use && ed.edge == 2 && ed.depth != 1 && ed.depth != 2)))) {
			glPushMatrix();
			glTranslate(s0 * FACET_LENGTH, -s0 * FACET_LENGTH, s0 * FACET_LENGTH);
			drawOctaFacet8(w, z, i, z, z, -1, -1, -1, -1);
			glPopMatrix();
		}
		/* ((use && ed.edge == 0 && ed.depth == 0)
			|| (use && ed.edge != 0 && ed.depth == size - 1)
			|| (!use && ed.edge == 0 && ed.depth != 0)
			|| (!use && ed.edge != 0 && ed.depth != size - 1)) */
			/*drawOctaFacet8(w, z, z, z, i, j, z, k, z);*/ /* LEFT */
		if ((!sticky && ((use && ed.edge == 0 && ed.depth == 1)
				|| (!use && ed.edge == 0 && ed.depth != 1)))
			|| (sticky && ((use && ed.edge == 0 && (ed.depth == 1 || ed.depth == 2))
				|| (!use && ed.edge == 0 && ed.depth != 1 && ed.depth != 2)))) {
			glPushMatrix();
			glTranslate(-s0 * FACET_LENGTH, -s0 * FACET_LENGTH, -s0 * FACET_LENGTH);
			drawOctaFacet8(w, -1, z, z, -1, j, z, -1, -1);
			glPopMatrix();
		} else if ((!sticky && ((use && ed.edge == 1 && ed.depth == size - 2)
				|| (!use && ed.edge == 1 && ed.depth != size - 2)))
			|| (sticky && ((use && ed.edge == 1 && (ed.depth == 1 || ed.depth == 2))
				|| (!use && ed.edge == 1 && ed.depth != 1 && ed.depth != 2)))) {
			glPushMatrix();
			glTranslate(-s0 * FACET_LENGTH, -s0 * FACET_LENGTH, -s0 * FACET_LENGTH);
			drawOctaFacet8(w, z, z, -1, -1, -1, z, k, -1);
			glPopMatrix();
		} else if ((!sticky && ((use && ed.edge == 2 && ed.depth == size - 2)
				|| (!use && ed.edge == 2 && ed.depth != size - 2)))
			|| (sticky && ((use && ed.edge == 2 && (ed.depth == 1 || ed.depth == 2))
				|| (!use && ed.edge == 2 && ed.depth != 1 && ed.depth != 2)))) {
			glPushMatrix();
			glTranslate(-s0 * FACET_LENGTH, -s0 * FACET_LENGTH, -s0 * FACET_LENGTH);
			drawOctaFacet8(w, z, z, z, i, -1, -1, -1, -1);
			glPopMatrix();
		}
		/* ((use && ed.depth == 0)
			|| (!use && ed.depth != 0)) */
			/*drawOctaFacet8(w, z, j, z, k, z, z, i, z);*/ /* TOP */
		if ((!sticky && ((use && ed.edge == 0 && ed.depth == 1)
				|| (!use && ed.edge == 0 && ed.depth != 1)))
			|| (sticky && ((use && ed.edge == 0 && (ed.depth == 1 || ed.depth == 2))
				|| (!use && ed.edge == 0 && ed.depth != 1 && ed.depth != 2)))) {
			glPushMatrix();
			glTranslate(s0 * FACET_LENGTH, s0 * FACET_LENGTH, -s0 * FACET_LENGTH);
			drawOctaFacet8(w, -1, j, z, -1, z, z, -1, -1);
			glPopMatrix();
		} else if ((!sticky && ((use && ed.edge == 1 && ed.depth == 1)
				|| (!use && ed.edge == 1 && ed.depth != 1)))
			|| (sticky && ((use && ed.edge == 1 && (ed.depth == 1 || ed.depth == 2))
				|| (!use && ed.edge == 1 && ed.depth != 1 && ed.depth != 2)))) {
			glPushMatrix();
			glTranslate(s0 * FACET_LENGTH, s0 * FACET_LENGTH, -s0 * FACET_LENGTH);
			drawOctaFacet8(w, -1, -1, z, k, z, -1, -1, z);
			glPopMatrix();
		} else if ((!sticky && ((use && ed.edge == 2 && ed.depth == 1)
				|| (!use && ed.edge == 2 && ed.depth != 1)))
			|| (sticky && ((use && ed.edge == 2 && (ed.depth == 1 || ed.depth == 2))
				|| (!use && ed.edge == 2 && ed.depth != 1 && ed.depth != 2)))) {
			glPushMatrix();
			glTranslate(s0 * FACET_LENGTH, s0 * FACET_LENGTH, -s0 * FACET_LENGTH);
			drawOctaFacet8(w, -1, -1, -1, -1, z, z, i, z);
			glPopMatrix();
		}
		/* ((use && ed.edge == 2 && ed.depth == 0)
			|| (use && ed.edge != 2 && ed.depth == size - 1)
			|| (!use && ed.edge == 2 && ed.depth != 0)
			|| (!use && ed.edge != 2 && ed.depth != size - 1))*/
			/*drawOctaFacet8(w, z, k, z, j, i, z, z, z);*/ /* FRONT */
		if ((!sticky && ((use && ed.edge == 0 && ed.depth == size - 2)
				|| (!use && ed.edge == 0 && ed.depth != size - 2)))
			|| (sticky && ((use && ed.edge == 0 && (ed.depth == 1 || ed.depth == 2))
				|| (!use && ed.edge == 0 && ed.depth != 1 && ed.depth != 2)))) {
			glPushMatrix();
			glTranslate(-s0 * FACET_LENGTH, s0 * FACET_LENGTH, s0 * FACET_LENGTH);
			drawOctaFacet8(w, z, -1, -1, j, -1, -1, z, z);
			glPopMatrix();
		} else if ((!sticky && ((use && ed.edge == 1 && ed.depth == size - 2)
				|| (!use && ed.edge == 1 && ed.depth != size - 2)))
			|| (sticky && ((use && ed.edge == 1 && (ed.depth == 1 || ed.depth == 2))
				|| (!use && ed.edge == 1 && ed.depth != 1 && ed.depth != 2)))) {
			glPushMatrix();
			glTranslate(-s0 * FACET_LENGTH, s0 * FACET_LENGTH, s0 * FACET_LENGTH);
			drawOctaFacet8(w, z, k, -1, -1, -1, z, z, -1);
			glPopMatrix();
		} else if ((!sticky && ((use && ed.edge == 2 && ed.depth == 1)
				|| (!use && ed.edge == 2 && ed.depth != 1)))
			|| (sticky && ((use && ed.edge == 2 && (ed.depth == 1 || ed.depth == 2))
				|| (!use && ed.edge == 2 && ed.depth != 1 && ed.depth != 2)))) {
			glPushMatrix();
			glTranslate(-s0 * FACET_LENGTH, s0 * FACET_LENGTH, s0 * FACET_LENGTH);
			drawOctaFacet8(w, -1, -1, -1, -1, i, z, z, z);
			glPopMatrix();
		}
	}
	if (!sticky)
	    for (edge = 1; edge < size / 2; edge++) {
		/*if (size > 3) OUTSIDE EDGE 1 */
		int a, b, c, d, e, f;
		s1 = size - 1;
		s2 = size - 2 * edge - 1;
		a = edge * edge; /* 1, 4 */
		b = (edge + 1) * (edge + 1) - 1; /* 3, 8 */
		c = (size - edge - 1) * (size - edge - 1); /* 4, 9, 16 */
		d = (size - edge) * (size - edge) - 1; /* 8, 15 */
		e = (size - 1) * (size - 1) + 2 * edge; /* 11,  */
		f = size * size - 2 * edge - 1; /* 13, */
		if (all || (use && ed.edge == 0 && ed.depth == edge)
				|| (use && ed.edge == 2 && ed.depth == size - 1 - edge)
				|| (use && ed.edge == 1 && ed.depth == size - 1)
				|| (!use && ed.edge == 0 && ed.depth != edge)
				|| (!use && ed.edge == 2 && ed.depth != size - 1 - edge)
				|| (!use && ed.edge == 1 && ed.depth != size - 1)) {
			glPushMatrix();
			glTranslate(-s1 * FACET_LENGTH, -s2 * FACET_LENGTH, -s2 * FACET_LENGTH);
			drawTetraFacet(w, a, -1, c, -1);
			glPopMatrix();
		}
		if (all || (use && ed.edge == 0 && ed.depth == size - 1 - edge)
				|| (use && ed.edge == 2 && ed.depth == edge)
				|| (use && ed.edge == 1 && ed.depth == size - 1)
				|| (!use && ed.edge == 0 && ed.depth != size - 1 - edge)
				|| (!use && ed.edge == 2 && ed.depth != edge)
				|| (!use && ed.edge == 1 && ed.depth != size - 1)) {
			glPushMatrix();
			glTranslate(-s1 * FACET_LENGTH, s2 * FACET_LENGTH, s2 * FACET_LENGTH);
			drawTetraFacet(w, c, -1, a, -1);
			glPopMatrix();
		}
		if (all || (use && ed.edge != 0 && ed.depth == size - 1 - edge)
				|| (use && ed.edge == 0 && ed.depth == 0)
				|| (!use && ed.edge != 0 && ed.depth != size - 1 - edge)
				|| (!use && ed.edge == 0 && ed.depth != 0)) {
			glPushMatrix();
			glTranslate(-s2 * FACET_LENGTH, -s2 * FACET_LENGTH, -s1 * FACET_LENGTH);
			drawTetraFacet(w, b, -1, -1, d);
			glPopMatrix();
		}
		if (all || (use && ed.edge != 0 && ed.depth == edge)
				|| (use && ed.edge == 0 && ed.depth == 0)
				|| (!use && ed.edge != 0 && ed.depth != edge)
				|| (!use && ed.edge == 0 && ed.depth != 0)) {
			glPushMatrix();
			glTranslate(s2 * FACET_LENGTH, s2 * FACET_LENGTH, -s1 * FACET_LENGTH);
			drawTetraFacet(w, d, -1, -1, b);
			glPopMatrix();
		}
		if (all || (use && ed.edge != 2 && ed.depth == size - 1 - edge)
				|| (use && ed.edge == 2 && ed.depth == 0)
				|| (!use && ed.edge != 2 && ed.depth != size - 1 - edge)
				|| (!use && ed.edge == 2 && ed.depth != 0)) {
			glPushMatrix();
			glTranslate(-s2 * FACET_LENGTH, s1 * FACET_LENGTH, s2 * FACET_LENGTH);
			drawTetraFacet(w, e, f, -1, -1);
			glPopMatrix();
		}
		if (all || (use && ed.edge != 2 && ed.depth == edge)
				|| (use && ed.edge == 2 && ed.depth == 0)
				|| (!use && ed.edge != 2 && ed.depth != edge)
				|| (!use && ed.edge == 2 && ed.depth != 0)) {
			glPushMatrix();
			glTranslate(s2 * FACET_LENGTH, s1 * FACET_LENGTH, -s2 * FACET_LENGTH);
			drawTetraFacet(w, f, e, -1, -1);
			glPopMatrix();
		}
		if (all || (use && ed.edge != 1 && ed.depth == size - 1 - edge)
				|| (use && ed.edge == 1 && ed.depth == 0)
				|| (!use && ed.edge != 1 && ed.depth != size - 1 - edge)
				|| (!use && ed.edge == 1 && ed.depth != 0)) {
			glPushMatrix();
			glTranslate(s1 * FACET_LENGTH, -s2 * FACET_LENGTH, s2 * FACET_LENGTH);
			drawTetraFacet(w, -1, a, -1, c);
			glPopMatrix();
		}
		if (all || (use && ed.edge != 1 && ed.depth == edge)
				|| (use && ed.edge == 1 && ed.depth == 0)
				|| (!use && ed.edge != 1 && ed.depth != edge)
				|| (!use && ed.edge == 1 && ed.depth != 0)) {
			glPushMatrix();
			glTranslate(s1 * FACET_LENGTH, s2 * FACET_LENGTH, -s2 * FACET_LENGTH);
			drawTetraFacet(w, -1, c, -1, a);
			glPopMatrix();
		}
		if (all || (use && ed.edge == 1 && ed.depth == edge)
				|| (use && ed.edge == 2 && ed.depth == size - 1 - edge)
				|| (use && ed.edge == 0 && ed.depth == size - 1)
				|| (!use && ed.edge == 1 && ed.depth != edge)
				|| (!use && ed.edge == 2 && ed.depth != size - 1 - edge)
				|| (!use && ed.edge == 0 && ed.depth != size - 1)) {
			glPushMatrix();
			glTranslate(s2 * FACET_LENGTH, -s2 * FACET_LENGTH, s1 * FACET_LENGTH);
			drawTetraFacet(w, -1, b, d, -1);
			glPopMatrix();
		}
		if (all || (use && ed.edge == 1 && ed.depth == size - 1 - edge)
				|| (use && ed.edge == 2 && ed.depth == edge)
				|| (use && ed.edge == 0 && ed.depth == size - 1)
				|| (!use && ed.edge == 1 && ed.depth != size - 1 - edge)
				|| (!use && ed.edge == 2 && ed.depth != edge)
				|| (!use && ed.edge == 0 && ed.depth != size - 1)) {
			glPushMatrix();
			glTranslate(-s2 * FACET_LENGTH, s2 * FACET_LENGTH, s1 * FACET_LENGTH);
			drawTetraFacet(w, -1, d, b, -1);
			glPopMatrix();
		}
		if (all || (use && ed.edge == 0 && ed.depth == edge)
				|| (use && ed.edge == 1 && ed.depth == size - 1 - edge)
				|| (use && ed.edge == 2 && ed.depth == size - 1)
				|| (!use && ed.edge == 0 && ed.depth != edge)
				|| (!use && ed.edge == 1 && ed.depth != size - 1 - edge)
				|| (!use && ed.edge == 2 && ed.depth != size - 1)) {
			glPushMatrix();
			glTranslate(-s2 * FACET_LENGTH, -s1 * FACET_LENGTH, -s2 * FACET_LENGTH);
			drawTetraFacet(w, -1, -1, e, f);
			glPopMatrix();
		}
		if (all || (use && ed.edge == 0 && ed.depth == size - 1 - edge)
				|| (use && ed.edge == 1 && ed.depth == edge)
				|| (use && ed.edge == 2 && ed.depth == size - 1)
				|| (!use && ed.edge == 0 && ed.depth != size - 1 - edge)
				|| (!use && ed.edge == 1 && ed.depth != edge)
				|| (!use && ed.edge == 2 && ed.depth != size - 1)) {
			glPushMatrix();
			glTranslate(s2 * FACET_LENGTH, -s1 * FACET_LENGTH, s2 * FACET_LENGTH);
			drawTetraFacet(w, -1, -1, f, e);
			glPopMatrix();
		}
	}
	if (sticky) {
	    if (size > 3 && size % 3 != 0) { /* CENTERS */
		k = 6;
		s0 = 1;
		if (all || ((use && ed.edge != 1 && (ed.depth == 1 || ed.depth == 2))
				|| (use && ed.edge == 1 && (ed.depth == 1 || ed.depth == 2))
				|| (!use && ed.edge != 1 && ed.depth != 1 && ed.depth != 2)
				|| (!use && ed.edge == 1 && ed.depth != 1 && ed.depth != 2))) {
			glPushMatrix();
			glTranslate(-s0 * FACET_LENGTH, s0 * FACET_LENGTH, -s0 * FACET_LENGTH);
			drawTetraFacet(w, k, -1, -1, -1);
			glPopMatrix();
		}
		if (all || ((use && ed.edge != 0 && (ed.depth == 1 || ed.depth == 2))
				|| (use && ed.edge == 0 && (ed.depth == 1 || ed.depth == 2))
				|| (!use && ed.edge != 0 && ed.depth != 1 && ed.depth != 2)
				|| (!use && ed.edge == 0 && ed.depth != 1 && ed.depth != 2))) {
			glPushMatrix();
			glTranslate(s0 * FACET_LENGTH, s0 * FACET_LENGTH, s0 * FACET_LENGTH);
			drawTetraFacet(w, -1, k, -1, -1);
			glPopMatrix();
		}
		if (all || ((use && (ed.depth == 1 || ed.depth == 2))
				|| (!use && ed.depth != 1 && ed.depth != 2))) {
			glPushMatrix();
			glTranslate(-s0 * FACET_LENGTH, -s0 * FACET_LENGTH, s0 * FACET_LENGTH);
			drawTetraFacet(w, -1, -1, k, -1);
			glPopMatrix();
		}
		if (all || ((use && ed.edge != 2 && (ed.depth == 1 || ed.depth == 2))
				|| (use && ed.edge == 2 && (ed.depth == 1 || ed.depth == 2))
				|| (!use && ed.edge != 2 && ed.depth != 1 && ed.depth != 2)
				|| (!use && ed.edge == 2 && ed.depth != 1 && ed.depth != 2))) {
			glPushMatrix();
			glTranslate(s0 * FACET_LENGTH, -s0 * FACET_LENGTH, -s0 * FACET_LENGTH);
			drawTetraFacet(w, -1, -1, -1, k);
			glPopMatrix();
		}
	    }
	    { /* OUTSIDE EDGE 1 */
		int a, b, c, d, e, f;
		s1 = size - 1;
		s2 = size - 3;
		edge = 1;
		a = edge * edge; /* 1, 4 */
		b = (edge + 1) * (edge + 1) - 1; /* 3, 8 */
		c = (size - edge - 1) * (size - edge - 1); /* 4, 9, 16 */
		d = (size - edge) * (size - edge) - 1; /* 8, 15 */
		e = (size - 1) * (size - 1) + 2 * edge; /* 11,  */
		f = size * size - 2 * edge - 1; /* 13, */
		if (all	|| (use && (ed.edge == 0 || ed.edge == 2) && (ed.depth == 1 || ed.depth == 2))
				|| (use && ed.edge == 1 && ed.depth == size - 1)
				|| (!use && (ed.edge == 0 || ed.edge == 2) && ed.depth != 1 && ed.depth != 2)
				|| (!use && ed.edge == 1 && ed.depth != size - 1)) {
			glPushMatrix();
			glTranslate(-s1 * FACET_LENGTH, -s2 * FACET_LENGTH, -s2 * FACET_LENGTH);
			drawTetraFacet(w, a, -1, c, -1);
			glPopMatrix();
			glPushMatrix();
			glTranslate(-s1 * FACET_LENGTH, s2 * FACET_LENGTH, s2 * FACET_LENGTH);
			drawTetraFacet(w, c, -1, a, -1);
			glPopMatrix();
		}
		if (all || (use && ed.edge != 0 && (ed.depth == 1 || ed.depth == 2))
				|| (use && ed.edge == 0 && ed.depth == 0)
				|| (!use && ed.edge != 0 && ed.depth != 1 && ed.depth != 2)
				|| (!use && ed.edge == 0 && ed.depth != 0)) {
			glPushMatrix();
			glTranslate(-s2 * FACET_LENGTH, -s2 * FACET_LENGTH, -s1 * FACET_LENGTH);
			drawTetraFacet(w, b, -1, -1, d);
			glPopMatrix();
			glPushMatrix();
			glTranslate(s2 * FACET_LENGTH, s2 * FACET_LENGTH, -s1 * FACET_LENGTH);
			drawTetraFacet(w, d, -1, -1, b);
			glPopMatrix();
		}
		if (all || (use && ed.edge != 2 && (ed.depth == 1 || ed.depth == 2))
				|| (use && ed.edge == 2 && ed.depth == 0)
				|| (!use && ed.edge != 2 && ed.depth != 1 && ed.depth != 2)
				|| (!use && ed.edge == 2 && ed.depth != 0)) {
			glPushMatrix();
			glTranslate(-s2 * FACET_LENGTH, s1 * FACET_LENGTH, s2 * FACET_LENGTH);
			drawTetraFacet(w, e, f, -1, -1);
			glPopMatrix();
			glPushMatrix();
			glTranslate(s2 * FACET_LENGTH, s1 * FACET_LENGTH, -s2 * FACET_LENGTH);
			drawTetraFacet(w, f, e, -1, -1);
			glPopMatrix();
		}
		if (all || (use && ed.edge != 1 && (ed.depth == 1 || ed.depth == 2))
				|| (use && ed.edge == 1 && ed.depth == 0)
				|| (!use && ed.edge != 1 && ed.depth != 1 && ed.depth != 2)
				|| (!use && ed.edge == 1 && ed.depth != 0)) {
			glPushMatrix();
			glTranslate(s1 * FACET_LENGTH, -s2 * FACET_LENGTH, s2 * FACET_LENGTH);
			drawTetraFacet(w, -1, a, -1, c);
			glPopMatrix();
			glPushMatrix();
			glTranslate(s1 * FACET_LENGTH, s2 * FACET_LENGTH, -s2 * FACET_LENGTH);
			drawTetraFacet(w, -1, c, -1, a);
			glPopMatrix();
		}
		if (all || (use && (ed.edge == 1 || ed.edge == 2) && (ed.depth == 1 || ed.depth == 2))
				|| (use && ed.edge == 0 && ed.depth == size - 1)
				|| (!use && (ed.edge == 1 || ed.edge == 2) && ed.depth != 1 && ed.depth != 2)
				|| (!use && ed.edge == 0 && ed.depth != size - 1)) {
			glPushMatrix();
			glTranslate(s2 * FACET_LENGTH, -s2 * FACET_LENGTH, s1 * FACET_LENGTH);
			drawTetraFacet(w, -1, b, d, -1);
			glPopMatrix();
			glPushMatrix();
			glTranslate(-s2 * FACET_LENGTH, s2 * FACET_LENGTH, s1 * FACET_LENGTH);
			drawTetraFacet(w, -1, d, b, -1);
			glPopMatrix();
		}
		if (all || (use && (ed.edge == 0 || ed.edge == 1) && (ed.depth == 1 || ed.depth == 2))
				|| (use && ed.edge == 2 && ed.depth == size - 1)
				|| (!use && (ed.edge == 0 || ed.edge == 1) && ed.depth != 1 && ed.depth != 2)
				|| (!use && ed.edge == 2 && ed.depth != size - 1)) {
			glPushMatrix();
			glTranslate(-s2 * FACET_LENGTH, -s1 * FACET_LENGTH, -s2 * FACET_LENGTH);
			drawTetraFacet(w, -1, -1, e, f);
			glPopMatrix();
			glPushMatrix();
			glTranslate(s2 * FACET_LENGTH, -s1 * FACET_LENGTH, s2 * FACET_LENGTH);
			drawTetraFacet(w, -1, -1, f, e);
			glPopMatrix();
		}
	    }
	}
	for (edge = 1; edge < (size - 1) / 2; edge++) {
		/*if (size > 4) INSIDE EDGE 2 */
		int m1, m2;
		int a, b, c, d, e, f;
		s1 = size - 2;
		s2 = size - 2 * edge - 2;
		a = (edge + 1) * (edge + 1) + 1; /* 5 */
		b = (edge + 2) * (edge + 2) - 2; /* 7 */
		c = (size - edge - 1) * (size - edge - 1) + 1;/* (size - 2) * (size - 2) + 1 */
		d = (size - edge) * (size - edge) - 2; /* (size - 1) * (size - 1) - 2 */
		e = (size - 1) * ( size - 1) + 2 * edge + 1; /* (size - 1) * (size - 1) + 3 */
		f = size * size - 2 * edge - 2; /* size * size - 4 */
		m1 = edge;
		m2 = edge + 1;
		if ((use && ed.edge == 0 && ed.depth == 0)
				|| (!use && ed.edge == 0 && ed.depth != 0)) {
			glPushMatrix();
			glTranslate(-s2 * FACET_LENGTH, -s2 * FACET_LENGTH, -s1 * FACET_LENGTH);
			drawOctaFacet8(w, z, -1, -1, b, -1, -1, d, z);
			glPopMatrix();
		} else if ((use && ed.edge == 1 && ed.depth == size - 1 - m2)
				|| (!use && ed.edge == 1 && ed.depth != size - 1 - m2)) {
			glPushMatrix();
			glTranslate(-s2 * FACET_LENGTH, -s2 * FACET_LENGTH, -s1 * FACET_LENGTH);
			drawOctaFacet8(w, z, z, -1, -1, -1, z, d, -1);
			glPopMatrix();
		} else if ((use && ed.edge == 2 && ed.depth == size - 1 - m1)
				|| (!use && ed.edge == 2 && ed.depth != size - 1 - m1)) {
			glPushMatrix();
			glTranslate(-s2 * FACET_LENGTH, -s2 * FACET_LENGTH, -s1 * FACET_LENGTH);
			drawOctaFacet8(w, -1, -1, -1, -1, z, z, d, z);
			glPopMatrix();
		}
		if ((use && ed.edge == 1 && ed.depth == size - 1 - m1)
				|| (!use && ed.edge == 1 && ed.depth != size - 1 - m1)) {
			glPushMatrix();
			glTranslate(-s2 * FACET_LENGTH, -s2 * FACET_LENGTH, -s1 * FACET_LENGTH);
			drawOctaFacet8(w, -1, -1, z, b, z, -1, -1, z);
			glPopMatrix();
		} else if ((use && ed.edge == 2 && ed.depth == size - 1 - m2)
				|| (!use && ed.edge == 2 && ed.depth != size - 1 - m2)) {
			glPushMatrix();
			glTranslate(-s2 * FACET_LENGTH, -s2 * FACET_LENGTH, -s1 * FACET_LENGTH);
			drawOctaFacet8(w, z, z, z, b, -1, -1, -1, -1);
			glPopMatrix();
		}

		if ((use && ed.edge == 0 && ed.depth == 0)
				|| (!use && ed.edge == 0 && ed.depth != 0)) {
			glPushMatrix();
			glTranslate(s2 * FACET_LENGTH, s2 * FACET_LENGTH, -s1 * FACET_LENGTH);
			drawOctaFacet8(w, z, -1, -1, d, -1, -1, b, z);
			glPopMatrix();
		} else if ((use && ed.edge == 1 && ed.depth == m1)
				|| (!use && ed.edge == 1 && ed.depth != m1)) {
			glPushMatrix();
			glTranslate(s2 * FACET_LENGTH, s2 * FACET_LENGTH, -s1 * FACET_LENGTH);
			drawOctaFacet8(w, z, z, -1, -1, -1, z, b, -1);
			glPopMatrix();
		} else if ((use && ed.edge == 2 && ed.depth == m2)
				|| (!use && ed.edge == 2 && ed.depth != m2)) {
			glPushMatrix();
			glTranslate(s2 * FACET_LENGTH, s2 * FACET_LENGTH, -s1 * FACET_LENGTH);
			drawOctaFacet8(w, -1, -1, -1, -1, z, z, b, z);
			glPopMatrix();
		}
		if ((use && ed.edge == 1 && ed.depth == m2)
				|| (!use && ed.edge == 1 && ed.depth != m2)) {
			glPushMatrix();
			glTranslate(s2 * FACET_LENGTH, s2 * FACET_LENGTH, -s1 * FACET_LENGTH);
			drawOctaFacet8(w, -1, -1, z, d, z, -1, -1, z);
			glPopMatrix();
		} else if ((use && ed.edge == 2 && ed.depth == m1)
				|| (!use && ed.edge == 2 && ed.depth != m1)) {
			glPushMatrix();
			glTranslate(s2 * FACET_LENGTH, s2 * FACET_LENGTH, -s1 * FACET_LENGTH);
			drawOctaFacet8(w, z, z, z, d, -1, -1, -1, -1);
			glPopMatrix();
		}

		if ((use && ed.edge == 1 && ed.depth == 0)
				|| (!use && ed.edge == 1 && ed.depth != 0)) {
			glPushMatrix();
			glTranslate(s1 * FACET_LENGTH, -s2 * FACET_LENGTH, s2 * FACET_LENGTH);
			drawOctaFacet8(w, z, a, -1, -1, -1, z, c, -1);
			glPopMatrix();
		} else if ((use && ed.edge == 0 && ed.depth == size - 1 - m1)
				|| (!use && ed.edge == 0 && ed.depth != size - 1 - m1)) {
			glPushMatrix();
			glTranslate(s1 * FACET_LENGTH, -s2 * FACET_LENGTH, s2 * FACET_LENGTH);
			drawOctaFacet8(w, -1, a, z, -1, z, z, -1, -1);
			glPopMatrix();
		} else if ((use && ed.edge == 2 && ed.depth == size - 1 - m2)
				|| (!use && ed.edge == 2 && ed.depth != size - 1 - m2)) {
			glPushMatrix();
			glTranslate(s1 * FACET_LENGTH, -s2 * FACET_LENGTH, s2 * FACET_LENGTH);
			drawOctaFacet8(w, z, a, z, z, -1, -1, -1, -1);
			glPopMatrix();
		}
		if ((use && ed.edge == 0 && ed.depth == size - 1 - m2)
				|| (!use && ed.edge == 0 && ed.depth != size - 1 - m2)) {
			glPushMatrix();
			glTranslate(s1 * FACET_LENGTH, -s2 * FACET_LENGTH, s2 * FACET_LENGTH);
			drawOctaFacet8(w, z, -1, -1, z, -1, -1, c, z);
			glPopMatrix();
		} else if ((use && ed.edge == 2 && ed.depth == size - 1 - m1)
				|| (!use && ed.edge == 2 && ed.depth != size - 1 - m1)) {
			glPushMatrix();
			glTranslate(s1 * FACET_LENGTH, -s2 * FACET_LENGTH, s2 * FACET_LENGTH);
			drawOctaFacet8(w, -1, -1, -1, -1, z, z, c, z);
			glPopMatrix();
		}

		if ((use && ed.edge == 1 && ed.depth == 0)
				|| (!use && ed.edge == 1 && ed.depth != 0)) {
			glPushMatrix();
			glTranslate(s1 * FACET_LENGTH, s2 * FACET_LENGTH, -s2 * FACET_LENGTH);
			drawOctaFacet8(w, z, c, -1, -1, -1, z, a, -1);
			glPopMatrix();
		} else if ((use && ed.edge == 0 && ed.depth == m2)
				|| (!use && ed.edge == 0 && ed.depth != m2)) {
			glPushMatrix();
			glTranslate(s1 * FACET_LENGTH, s2 * FACET_LENGTH, -s2 * FACET_LENGTH);
			drawOctaFacet8(w, -1, c, z, -1, z, z, -1, -1);
			glPopMatrix();
		} else if ((use && ed.edge == 2 && ed.depth == m1)
				|| (!use && ed.edge == 2 && ed.depth != m1)) {
			glPushMatrix();
			glTranslate(s1 * FACET_LENGTH, s2 * FACET_LENGTH, -s2 * FACET_LENGTH);
			drawOctaFacet8(w, z, c, z, z, -1, -1, -1, -1);
			glPopMatrix();
		}
		if ((use && ed.edge == 0 && ed.depth == m1)
				|| (!use && ed.edge == 0 && ed.depth != m1)) {
			glPushMatrix();
			glTranslate(s1 * FACET_LENGTH, s2 * FACET_LENGTH, -s2 * FACET_LENGTH);
			drawOctaFacet8(w, z, -1, -1, z, -1, -1, a, z);
			glPopMatrix();
		} else if ((use && ed.edge == 2 && ed.depth == m2)
				|| (!use && ed.edge == 2 && ed.depth != m2)) {
			glPushMatrix();
			glTranslate(s1 * FACET_LENGTH, s2 * FACET_LENGTH, -s2 * FACET_LENGTH);
			drawOctaFacet8(w, -1, -1, -1, -1, z, z, a, z);
			glPopMatrix();
		}

		if ((use && ed.edge == 2 && ed.depth == 0)
				|| (!use && ed.edge == 2 && ed.depth != 0)) {
			glPushMatrix();
			glTranslate(-s2 * FACET_LENGTH, s1 * FACET_LENGTH, s2 * FACET_LENGTH);
			drawOctaFacet8(w, z, f, z, e, -1, -1, -1, -1);
			glPopMatrix();
		} else if ((use && ed.edge == 0 && ed.depth == size - 1 - m2)
				|| (!use && ed.edge == 0 && ed.depth != size - 1 - m2)) {
			glPushMatrix();
			glTranslate(-s2 * FACET_LENGTH, s1 * FACET_LENGTH, s2 * FACET_LENGTH);
			drawOctaFacet8(w, z, -1, -1, e, -1, -1, z, z);
			glPopMatrix();
		} else if ((use && ed.edge == 1 && ed.depth == size - 1 - m1)
				|| (!use && ed.edge == 1 && ed.depth != size - 1 - m1)) {
			glPushMatrix();
			glTranslate(-s2 * FACET_LENGTH, s1 * FACET_LENGTH, s2 * FACET_LENGTH);
			drawOctaFacet8(w, -1, -1, z, e, z, -1, -1, z);
			glPopMatrix();
		}
		if ((use && ed.edge == 0 && ed.depth == size - 1 - m1)
				|| (!use && ed.edge == 0 && ed.depth != size - 1 - m1)) {
			glPushMatrix();
			glTranslate(-s2 * FACET_LENGTH, s1 * FACET_LENGTH, s2 * FACET_LENGTH);
			drawOctaFacet8(w, -1, f, z, -1, z, z, -1, -1);
			glPopMatrix();
		} else if ((use && ed.edge == 1 && ed.depth == size - 1 - m2)
				|| (!use && ed.edge == 1 && ed.depth != size - 1 - m2)) {
			glPushMatrix();
			glTranslate(-s2 * FACET_LENGTH, s1 * FACET_LENGTH, s2 * FACET_LENGTH);
			drawOctaFacet8(w, z, f, -1, -1, -1, z, z, -1);
			glPopMatrix();
		}

		if ((use && ed.edge == 2 && ed.depth == 0)
				|| (!use && ed.edge == 2 && ed.depth != 0)) {
			glPushMatrix();
			glTranslate(s2 * FACET_LENGTH, s1 * FACET_LENGTH, -s2 * FACET_LENGTH);
			drawOctaFacet8(w, z, e, z, f, -1, -1, -1, -1);
			glPopMatrix();
		} else if ((use && ed.edge == 0 && ed.depth == m1)
				|| (!use && ed.edge == 0 && ed.depth != m1)) {
			glPushMatrix();
			glTranslate(s2 * FACET_LENGTH, s1 * FACET_LENGTH, -s2 * FACET_LENGTH);
			drawOctaFacet8(w, z, -1, -1, f, -1, -1, z, z);
			glPopMatrix();
		} else if ((use && ed.edge == 1 && ed.depth == m2)
				|| (!use && ed.edge == 1 && ed.depth != m2)) {
			glPushMatrix();
			glTranslate(s2 * FACET_LENGTH, s1 * FACET_LENGTH, -s2 * FACET_LENGTH);
			drawOctaFacet8(w, -1, -1, z, f, z, -1, -1, z);
			glPopMatrix();
		}
		if ((use && ed.edge == 0 && ed.depth == m2)
				|| (!use && ed.edge == 0 && ed.depth != m2)) {
			glPushMatrix();
			glTranslate(s2 * FACET_LENGTH, s1 * FACET_LENGTH, -s2 * FACET_LENGTH);
			drawOctaFacet8(w, -1, e, z, -1, z, z, -1, -1);
			glPopMatrix();
		} else if ((use && ed.edge == 1 && ed.depth == m1)
				|| (!use && ed.edge == 1 && ed.depth != m1)) {
			glPushMatrix();
			glTranslate(s2 * FACET_LENGTH, s1 * FACET_LENGTH, -s2 * FACET_LENGTH);
			drawOctaFacet8(w, z, e, -1, -1, -1, z, z, -1);
			glPopMatrix();
		}

		if ((use && ed.edge == 0 && ed.depth == size - 1)
				|| (!use && ed.edge == 0 && ed.depth != size - 1)) {
			glPushMatrix();
			glTranslate(s2 * FACET_LENGTH, -s2 * FACET_LENGTH, s1 * FACET_LENGTH);
			drawOctaFacet8(w, -1, b, z, -1, d, z, -1, -1);
			glPopMatrix();
		} else if ((use && ed.edge == 1 && ed.depth == m1)
				|| (!use && ed.edge == 1 && ed.depth != m1)) {
			glPushMatrix();
			glTranslate(s2 * FACET_LENGTH, -s2 * FACET_LENGTH, s1 * FACET_LENGTH);
			drawOctaFacet8(w, z, b, -1, -1, -1, z, z, -1);
			glPopMatrix();
		} else if ((use && ed.edge == 2 && ed.depth == size - 1 - m2)
				|| (!use && ed.edge == 2 && ed.depth != size - 1 - m2)) {
			glPushMatrix();
			glTranslate(s2 * FACET_LENGTH, -s2 * FACET_LENGTH, s1 * FACET_LENGTH);
			drawOctaFacet8(w, z, b, z, z, -1, -1, -1, -1);
			glPopMatrix();
		}
		if ((use && ed.edge == 1 && ed.depth == m2)
				|| (!use && ed.edge == 1 && ed.depth != m2)) {
			glPushMatrix();
			glTranslate(s2 * FACET_LENGTH, -s2 * FACET_LENGTH, s1 * FACET_LENGTH);
			drawOctaFacet8(w, -1, -1, z, z, d, -1, -1, z);
			glPopMatrix();
		} else if ((use && ed.edge == 2 && ed.depth == size - 1 - m1)
				|| (!use && ed.edge == 2 && ed.depth != size - 1 - m1)) {
			glPushMatrix();
			glTranslate(s2 * FACET_LENGTH, -s2 * FACET_LENGTH, s1 * FACET_LENGTH);
			drawOctaFacet8(w, -1, -1, -1, -1, d, z, z, z);
			glPopMatrix();
		}

		if ((use && ed.edge == 0 && ed.depth == size - 1)
				|| (!use && ed.edge == 0 && ed.depth != size - 1)) {
			glPushMatrix();
			glTranslate(-s2 * FACET_LENGTH, s2 * FACET_LENGTH, s1 * FACET_LENGTH);
			drawOctaFacet8(w, -1, d, z, -1, b, z, -1, -1);
			glPopMatrix();
		} else if ((use && ed.edge == 1 && ed.depth == size - 1 - m2)
				|| (!use && ed.edge == 1 && ed.depth != size - 1 - m2)) {
			glPushMatrix();
			glTranslate(-s2 * FACET_LENGTH, s2 * FACET_LENGTH, s1 * FACET_LENGTH);
			drawOctaFacet8(w, z, d, -1, -1, -1, z, z, -1);
			glPopMatrix();
		} else if ((use && ed.edge == 2 && ed.depth == m1)
				|| (!use && ed.edge == 2 && ed.depth != m1)) {
			glPushMatrix();
			glTranslate(-s2 * FACET_LENGTH, s2 * FACET_LENGTH, s1 * FACET_LENGTH);
			drawOctaFacet8(w, z, d, z, z, -1, -1, -1, -1);
			glPopMatrix();
		}
		if ((use && ed.edge == 1 && ed.depth == size - 1 - m1)
				|| (!use && ed.edge == 1 && ed.depth != size - 1 - m1)) {
			glPushMatrix();
			glTranslate(-s2 * FACET_LENGTH, s2 * FACET_LENGTH, s1 * FACET_LENGTH);
			drawOctaFacet8(w, -1, -1, z, z, b, -1, -1, z);
			glPopMatrix();
		} else if ((use && ed.edge == 2 && ed.depth == m2)
				|| (!use && ed.edge == 2 && ed.depth != m2)) {
			glPushMatrix();
			glTranslate(-s2 * FACET_LENGTH, s2 * FACET_LENGTH, s1 * FACET_LENGTH);
			drawOctaFacet8(w, -1, -1, -1, -1, b, z, z, z);
			glPopMatrix();
		}

		if ((use && ed.edge == 1 && ed.depth == size - 1)
				|| (!use && ed.edge == 1 && ed.depth != size - 1)) {
			glPushMatrix();
			glTranslate(-s1 * FACET_LENGTH, -s2 * FACET_LENGTH, -s2 * FACET_LENGTH);
			drawOctaFacet8(w, -1, -1, z, a, c, -1, -1, z);
			glPopMatrix();
		} else if ((use && ed.edge == 0 && ed.depth == m1)
				|| (!use && ed.edge == 0 && ed.depth != m1)) {
			glPushMatrix();
			glTranslate(-s1 * FACET_LENGTH, -s2 * FACET_LENGTH, -s2 * FACET_LENGTH);
			drawOctaFacet8(w, z, -1, -1, a, -1, -1, z, z);
			glPopMatrix();
		} else if ((use && ed.edge == 2 && ed.depth == size - 1 - m2)
				|| (!use && ed.edge == 2 && ed.depth != size - 1 - m2)) {
			glPushMatrix();
			glTranslate(-s1 * FACET_LENGTH, -s2 * FACET_LENGTH, -s2 * FACET_LENGTH);
			drawOctaFacet8(w, z, z, z, a, -1, -1, -1, -1);
			glPopMatrix();
		}
		if ((use && ed.edge == 0 && ed.depth == m2)
				|| (!use && ed.edge == 0 && ed.depth != m2)) {
			glPushMatrix();
			glTranslate(-s1 * FACET_LENGTH, -s2 * FACET_LENGTH, -s2 * FACET_LENGTH);
			drawOctaFacet8(w, -1, z, z, -1, c, z, -1, -1);
			glPopMatrix();
		} else if ((use && ed.edge == 2 && ed.depth == size - 1 - m1)
				|| (!use && ed.edge == 2 && ed.depth != size - 1 - m1)) {
			glPushMatrix();
			glTranslate(-s1 * FACET_LENGTH, -s2 * FACET_LENGTH, -s2 * FACET_LENGTH);
			drawOctaFacet8(w, -1, -1, -1, -1, c, z, z, z);
			glPopMatrix();
		}

		if ((use && ed.edge == 1 && ed.depth == size - 1)
				|| (!use && ed.edge == 1 && ed.depth != size - 1)) {
			glPushMatrix();
			glTranslate(-s1 * FACET_LENGTH, s2 * FACET_LENGTH, s2 * FACET_LENGTH);
			drawOctaFacet8(w, -1, -1, z, c, a, -1, -1, z);
			glPopMatrix();
		} else if ((use && ed.edge == 0 && ed.depth == size - 1 - m2)
				|| (!use && ed.edge == 0 && ed.depth != size - 1 - m2)) {
			glPushMatrix();
			glTranslate(-s1 * FACET_LENGTH, s2 * FACET_LENGTH, s2 * FACET_LENGTH);
			drawOctaFacet8(w, z, -1, -1, c, -1, -1, z, z);
			glPopMatrix();
		} else if ((use && ed.edge == 2 && ed.depth == m1)
				|| (!use && ed.edge == 2 && ed.depth != m1)) {
			glPushMatrix();
			glTranslate(-s1 * FACET_LENGTH, s2 * FACET_LENGTH, s2 * FACET_LENGTH);
			drawOctaFacet8(w, z, z, z, c, -1, -1, -1, -1);
			glPopMatrix();
		}
		if ((use && ed.edge == 0 && ed.depth == size - 1 - m1)
				|| (!use && ed.edge == 0 && ed.depth != size - 1 - m1)) {
			glPushMatrix();
			glTranslate(-s1 * FACET_LENGTH, s2 * FACET_LENGTH, s2 * FACET_LENGTH);
			drawOctaFacet8(w, -1, z, z, -1, a, z, -1, -1);
			glPopMatrix();
		} else if ((use && ed.edge == 2 && ed.depth == m2)
				|| (!use && ed.edge == 2 && ed.depth != m2)) {
			glPushMatrix();
			glTranslate(-s1 * FACET_LENGTH, s2 * FACET_LENGTH, s2 * FACET_LENGTH);
			drawOctaFacet8(w, -1, -1, -1, -1, a, z, z, z);
			glPopMatrix();
		}

		if ((use && ed.edge == 2 && ed.depth == size - 1)
				|| (!use && ed.edge == 2 && ed.depth != size - 1)) {
			glPushMatrix();
			glTranslate(-s2 * FACET_LENGTH, -s1 * FACET_LENGTH, -s2 * FACET_LENGTH);
			drawOctaFacet8(w, -1, -1, -1, -1, e, z, f, z);
			glPopMatrix();
		} else if ((use && ed.edge == 0 && ed.depth == m2)
				|| (!use && ed.edge == 0 && ed.depth != m2)) {
			glPushMatrix();
			glTranslate(-s2 * FACET_LENGTH, -s1 * FACET_LENGTH, -s2 * FACET_LENGTH);
			drawOctaFacet8(w, -1, z, z, -1, e, z, -1, -1);
			glPopMatrix();
		} else if ((use && ed.edge == 1 && ed.depth == size - 1 - m1)
				|| (!use && ed.edge == 1 && ed.depth != size - 1 - m1)) {
			glPushMatrix();
			glTranslate(-s2 * FACET_LENGTH, -s1 * FACET_LENGTH, -s2 * FACET_LENGTH);
			drawOctaFacet8(w, -1, -1, z, z, e, -1, -1, z);
			glPopMatrix();
		}
		if ((use && ed.edge == 0 && ed.depth == m1)
				|| (!use && ed.edge == 0 && ed.depth != m1)) {
			glPushMatrix();
			glTranslate(-s2 * FACET_LENGTH, -s1 * FACET_LENGTH, -s2 * FACET_LENGTH);
			drawOctaFacet8(w, z, -1, -1, z, -1, -1, f, z);
			glPopMatrix();
		} else if ((use && ed.edge == 1 && ed.depth == size - 1 - m2)
				|| (!use && ed.edge == 1 && ed.depth != size - 1 - m2)) {
			glPushMatrix();
			glTranslate(-s2 * FACET_LENGTH, -s1 * FACET_LENGTH, -s2 * FACET_LENGTH);
			drawOctaFacet8(w, z, z, -1, -1, -1, z, f, -1);
			glPopMatrix();
		}

		if ((use && ed.edge == 2 && ed.depth == size - 1)
				|| (!use && ed.edge == 2 && ed.depth != size - 1)) {
			glPushMatrix();
			glTranslate(s2 * FACET_LENGTH, -s1 * FACET_LENGTH, s2 * FACET_LENGTH);
			drawOctaFacet8(w, -1, -1, -1, -1, f, z, e, z);
			glPopMatrix();
		} else if ((use && ed.edge == 0 && ed.depth == size - 1 - m1)
				|| (!use && ed.edge == 0 && ed.depth != size - 1 - m1)) {
			glPushMatrix();
			glTranslate(s2 * FACET_LENGTH, -s1 * FACET_LENGTH, s2 * FACET_LENGTH);
			drawOctaFacet8(w, -1, z, z, -1, f, z, -1, -1);
			glPopMatrix();
		} else if ((use && ed.edge == 1 && ed.depth == m2)
				|| (!use && ed.edge == 1 && ed.depth != m2)) {
			glPushMatrix();
			glTranslate(s2 * FACET_LENGTH, -s1 * FACET_LENGTH, s2 * FACET_LENGTH);
			drawOctaFacet8(w, -1, -1, z, z, f, -1, -1, z);
			glPopMatrix();
		}
		if ((use && ed.edge == 0 && ed.depth == size - 1 - m2)
				|| (!use && ed.edge == 0 && ed.depth != size - 1 - m2)) {
			glPushMatrix();
			glTranslate(s2 * FACET_LENGTH, -s1 * FACET_LENGTH, s2 * FACET_LENGTH);
			drawOctaFacet8(w, z, -1, -1, z, -1, -1, e, z);
			glPopMatrix();
		} else if ((use && ed.edge == 1 && ed.depth == m1)
				|| (!use && ed.edge == 1 && ed.depth != m1)) {
			glPushMatrix();
			glTranslate(s2 * FACET_LENGTH, -s1 * FACET_LENGTH, s2 * FACET_LENGTH);
			drawOctaFacet8(w, z, z, -1, -1, -1, z, e, -1);
			glPopMatrix();
		}
	}
	if (!sticky) {
	  pos = 1;
	  for (row = 0; row < size - 3; row++) {
	    pos += 4;
	    for (col = 0; col < 2 * row + 1; col++) {
		pos++;
		tetra = ((col % 2) == 0);
		sb = -size + 5 + 2 * row - ((tetra) ? 0 : 1);
		sl = size - 3 - col;
		sr = size - 4 - 2 * row + 1 + col;
		mb = row + 2;
		ml = size - 2 - col / 2;
		mr = size - row - 2 + (col + 1) / 2;
		if (all || (use && ed.edge == 0 && ed.depth == size - 1 - mr)
				|| (use && ed.edge == 1 && ed.depth == ml)
				|| (use && ed.edge == 2 && ed.depth == size - 1 - mb)
				|| (!use && ed.edge == 0 && ed.depth != size - 1 - mr)
				|| (!use && ed.edge == 1 && ed.depth != ml)
				|| (!use && ed.edge == 2 && ed.depth != size - 1 - mb)) {
			glPushMatrix();
			glTranslate(-sl * FACET_LENGTH, sb * FACET_LENGTH, -sr * FACET_LENGTH);
			drawFacet(w, pos, -1, -1, -1, tetra);
			glPopMatrix();
		}
		if (all || (use && ed.edge == 1 && ed.depth == size - 1 - ml)
				|| (use && ed.edge == 0 && ed.depth == mr)
				|| (use && ed.edge == 2 && ed.depth == size - 1 - mb)
				|| (!use && ed.edge == 1 && ed.depth != size - 1 - ml)
				|| (!use && ed.edge == 0 && ed.depth != mr)
				|| (!use && ed.edge == 2 && ed.depth != size - 1 - mb)) {
			glPushMatrix();
			glTranslate(sl * FACET_LENGTH, sb * FACET_LENGTH, sr * FACET_LENGTH);
			drawFacet(w, -1, pos, -1, -1, tetra);
			glPopMatrix();
		}
		if (all || (use && ed.edge == 0 && ed.depth == mr)
				|| (use && ed.edge == 1 && ed.depth == ml)
				|| (use && ed.edge == 2 && ed.depth == mb)
				|| (!use && ed.edge == 0 && ed.depth != mr)
				|| (!use && ed.edge == 1 && ed.depth != ml)
				|| (!use && ed.edge == 2 && ed.depth != mb)) {
			glPushMatrix();
			glTranslate(-sl * FACET_LENGTH, -sb * FACET_LENGTH, sr * FACET_LENGTH);
			drawFacet(w, -1, -1, pos, -1, tetra);
			glPopMatrix();
		}
		if (all || (use && ed.edge == 1 && ed.depth == size - 1 - ml)
				|| (use && ed.edge == 0 && ed.depth == size - 1 - mr)
				|| (use && ed.edge == 2 && ed.depth == mb)
				|| (!use && ed.edge == 1 && ed.depth != size - 1 - ml)
				|| (!use && ed.edge == 0 && ed.depth != size - 1 - mr)
				|| (!use && ed.edge == 2 && ed.depth != mb)) {
			glPushMatrix();
			glTranslate(sl * FACET_LENGTH, -sb * FACET_LENGTH, -sr * FACET_LENGTH);
			drawFacet(w, -1, -1, -1, pos, tetra);
			glPopMatrix();
		}
	    }
	  }
	}
}

typedef struct _FaceDepth {
	 int face, depth;
} FaceDepth;

static void
faceDepth(PyraminxWidget w, int face, int position, int direction, FaceDepth *fd)
{
	CRD crd;

	toCRD(w, face, position, &crd);
	if (direction == TR || direction == BL) {
		if (face % 2 == 1)
			fd->face = face - 1;
		else
			fd->face = face + 1;
		if (crd.diagonal >= w->pyraminx.size)
			fd->depth = 2 * w->pyraminx.size - 1 - crd.diagonal;
		else
			fd->depth = crd.diagonal;
	} else if (direction == TOP || direction == BOTTOM) {
		fd->face = (face + 2) % 4;
		if (face % 2 == 1)
			fd->depth = crd.column;
		else
			fd->depth = w->pyraminx.size - 1 - crd.column;
	} else if (direction == LEFT || direction == RIGHT) {
		fd->face = MAX_FACES - 1 - face;
		if (face % 2 == 1)
			fd->depth = crd.row;
		else
			fd->depth = w->pyraminx.size - 1 - crd.row;
	} else {
		if (face != -1) {
			(void) printf("faceDepth error: face %d, position %d, direction %d\n",
				face, position, direction);
		}
		fd->face = 0;
		fd->depth = 0;
	}
}

static void
drawFaces(PyraminxGLWidget w, int face, int position, int direction,
		Boolean use, Boolean all)
{
	int s0, s1, s2;
	int i, j, k, lc, rc;
	int pos, col, row, sb, sl, sr, mb, ml, mr;
	int size = w->pyraminx.size, edge, middle;
	Boolean tetra, sticky = w->pyraminx.sticky;
	CRD crd;
	FaceDepth fd;

	toCRD((PyraminxWidget) w, face, position, &crd);
	faceDepth((PyraminxWidget) w, face, position, direction, &fd);

	lc = (size - 1) * (size - 1);
	rc = size * size - 1;
	if (size == 1 && (all || use))
		drawTetraFacet(w, 0, 0, 0, 0);
	else if (size == 2 && (all || (use && fd.depth != 0)
			|| (!use && fd.depth == 0)))
		drawOctaFacet(w, 2, 2, 2, 2);
	if (size > 1) { /* CORNERS */
		s0 = size - 1;
		j = lc;
		k = rc;
		if (all || (!sticky && ((use && fd.face == 0 && fd.depth == 0)
				|| (use && fd.face != 0 && fd.depth == size - 1)
				|| (!use && fd.face != 0 && fd.depth != size - 1)
				|| (!use && fd.face == 0 && fd.depth != 0)))
			|| (sticky && ((use && fd.face == 0 && fd.depth != size - 1)
				|| (use && fd.face != 0 && fd.depth == size - 1)
				|| (!use && fd.face != 0 && fd.depth != size - 1)
				|| (!use && fd.face == 0 && fd.depth == size - 1)))) {
			glPushMatrix();
			glTranslate(s0 * FACET_LENGTH, -s0 * FACET_LENGTH, s0 * FACET_LENGTH);
			drawTetraFacet(w, -1, 0, k, j); /* RIGHT */
			glPopMatrix();
		}
		if (all || (!sticky && ((use && fd.face == 1 && fd.depth == 0)
				|| (use && fd.face != 1 && fd.depth == size - 1)
				|| (!use && fd.face != 1 && fd.depth != size - 1)
				|| (!use && fd.face == 1 && fd.depth != 0)))
			|| (sticky && ((use && fd.face == 1 && fd.depth != size - 1)
				|| (use && fd.face != 1 && fd.depth == size - 1)
				|| (!use && fd.face != 1 && fd.depth != size - 1)
				|| (!use && fd.face == 1 && fd.depth == size - 1)))) {
			glPushMatrix();
			glTranslate(-s0 * FACET_LENGTH, -s0 * FACET_LENGTH, -s0 * FACET_LENGTH);
			drawTetraFacet(w, 0, -1, j, k); /* LEFT */
			glPopMatrix();
		}
		if (all || (!sticky && ((use && fd.face == 2 && fd.depth == 0)
				|| (use && fd.face != 2 && fd.depth == size - 1)
				|| (!use && fd.face != 2 && fd.depth != size - 1)
				|| (!use && fd.face == 2 && fd.depth != 0)))
			|| (sticky && ((use && fd.face == 2 && fd.depth != size - 1)
				|| (use && fd.face != 2 && fd.depth == size - 1)
				|| (!use && fd.face != 2 && fd.depth != size - 1)
				|| (!use && fd.face == 2 && fd.depth == size - 1)))) {
			glPushMatrix();
			glTranslate(s0 * FACET_LENGTH, s0 * FACET_LENGTH, -s0 * FACET_LENGTH);
			drawTetraFacet(w, k, j, -1, 0); /* TOP */
			glPopMatrix();
		}
		if (all || (!sticky && ((use && fd.face == 3 && fd.depth == 0)
				|| (use && fd.face != 3 && fd.depth == size - 1)
				|| (!use && fd.face != 3 && fd.depth != size - 1)
				|| (!use && fd.face == 3 && fd.depth != 0)))
			|| (sticky && ((use && fd.face == 3 && fd.depth != size - 1)
				|| (use && fd.face != 3 && fd.depth == size - 1)
				|| (!use && fd.face != 3 && fd.depth != size - 1)
				|| (!use && fd.face == 3 && fd.depth == size - 1)))) {
			glPushMatrix();
			glTranslate(-s0 * FACET_LENGTH, s0 * FACET_LENGTH, s0 * FACET_LENGTH);
			drawTetraFacet(w, j, k, 0, -1); /* FRONT */
			glPopMatrix();
		}
	}
	if (size > 2) { /* CORNERS - 1 */
		s0 = size - 2;
		j = (size - 1) * (size - 1) + 1;
		k = size * size - 2;
		if (all || (!sticky &&((use && fd.face == 0 && fd.depth == 1)
				|| (use && fd.face != 0 && fd.depth == size - 1)
				|| (!use && fd.face != 0 && fd.depth != size - 1)
				|| (!use && fd.face == 0 && fd.depth != 1)))
			|| (sticky && ((use && fd.face == 0 && fd.depth != size - 1)
				|| (use && fd.face != 0 && fd.depth == size - 1)
				|| (!use && fd.face != 0 && fd.depth != size - 1)
				|| (!use && fd.face == 0 && fd.depth == size - 1)))) {
			glPushMatrix();
			glTranslate(s0 * FACET_LENGTH, -s0 * FACET_LENGTH, s0 * FACET_LENGTH);
			drawOctaFacet(w, -1, 2, k, j); /* RIGHT */
			glPopMatrix();
		}
		if (all || (!sticky && ((use && fd.face == 1 && fd.depth == 1)
				|| (use && fd.face != 1 && fd.depth == size - 1)
				|| (!use && fd.face != 1 && fd.depth != size - 1)
				|| (!use && fd.face == 1 && fd.depth != 1)))
			|| (sticky && ((use && fd.face == 1 && fd.depth != size - 1)
				|| (use && fd.face != 1 && fd.depth == size - 1)
				|| (!use && fd.face != 1 && fd.depth != size - 1)
				|| (!use && fd.face == 1 && fd.depth == size - 1)))) {
			glPushMatrix();
			glTranslate(-s0 * FACET_LENGTH, -s0 * FACET_LENGTH, -s0 * FACET_LENGTH);
			drawOctaFacet(w, 2, -1, j, k); /* LEFT */
			glPopMatrix();
		}
		if (all || (!sticky && ((use && fd.face == 2 && fd.depth == 1)
				|| (use && fd.face != 2 && fd.depth == size - 1)
				|| (!use && fd.face != 2 && fd.depth != size - 1)
				|| (!use && fd.face == 2 && fd.depth != 1)))
			|| (sticky && ((use && fd.face == 2 && fd.depth != size - 1)
				|| (use && fd.face != 2 && fd.depth == size - 1)
				|| (!use && fd.face != 2 && fd.depth != size - 1)
				|| (!use && fd.face == 2 && fd.depth == size - 1)))) {
			glPushMatrix();
			glTranslate(s0 * FACET_LENGTH, s0 * FACET_LENGTH, -s0 * FACET_LENGTH);
			drawOctaFacet(w, k, j, -1, 2); /* TOP */
			glPopMatrix();
		}
		if (all || (!sticky && ((use && fd.face == 3 && fd.depth == 1)
				|| (use && fd.face != 3 && fd.depth == size - 1)
				|| (!use && fd.face != 3 && fd.depth != size - 1)
				|| (!use && fd.face == 3 && fd.depth != 1)))
			|| (sticky && ((use && fd.face == 3 && fd.depth != size - 1)
				|| (use && fd.face != 3 && fd.depth == size - 1)
				|| (!use && fd.face != 3 && fd.depth != size - 1)
				|| (!use && fd.face == 3 && fd.depth == size - 1)))) {
			glPushMatrix();
			glTranslate(-s0 * FACET_LENGTH, s0 * FACET_LENGTH, s0 * FACET_LENGTH);
			drawOctaFacet(w, j, k, 2, -1); /* FRONT */
			glPopMatrix();
		}
	}
	if (size > 2) { /* MIDDLE SIDE */ /* 1, 3, 6; 5, 7, 12; 4, 8, 20*/
		middle = size / 2;
		tetra = (size % 2 == 1);
		s1 = size - 1 - ((tetra) ? 0 : 1);
		i = (size / 2) * (size / 2) + ((tetra) ? 0: 1);
		j = (size / 2 + 1) * (size / 2 + 1) - 1 - ((tetra) ? 0 : 1);
		k = size * size - size;
		if (all || (!sticky && ((use && (fd.face == 0 || fd.face == 3) && fd.depth == middle)
				|| (use && (fd.face == 1 || fd.face == 2) && fd.depth == size - 1)
				|| (!use && (fd.face == 0 || fd.face == 3) && fd.depth != middle)
				|| (!use && (fd.face == 1 || fd.face == 2) && fd.depth != size - 1)))
			|| (sticky && ((use && (fd.face == 0 || fd.face == 3) && fd.depth != size - 1)
				|| (use && (fd.face == 1 || fd.face == 2) && fd.depth == size - 1)
				|| (!use && (fd.face == 0 || fd.face == 3) && fd.depth == size - 1)
				|| (!use && (fd.face == 1 || fd.face == 2) && fd.depth != size - 1)))) {
			glPushMatrix();
			glTranslate(0, 0, s1 * FACET_LENGTH);
			drawFacet(w, -1, j, j, -1, tetra);
			glPopMatrix();
		}
		if (all || (!sticky && ((use && (fd.face == 2 || fd.face == 3) && fd.depth == middle)
				|| (use && (fd.face == 0 || fd.face == 1) && fd.depth == size - 1)
				|| (!use && (fd.face == 2 || fd.face == 3) && fd.depth != middle)
				|| (!use && (fd.face == 0 || fd.face == 1) && fd.depth != size - 1)))
			|| (sticky && ((use && (fd.face == 2 || fd.face == 3) && fd.depth != size - 1)
				|| (use && (fd.face == 0 || fd.face == 1) && fd.depth == size - 1)
				|| (!use && (fd.face == 2 || fd.face == 3) && fd.depth == size - 1)
				|| (!use && (fd.face == 0 || fd.face == 1) && fd.depth != size - 1)))) {
			glPushMatrix();
			glTranslate(0, s1 * FACET_LENGTH, 0);
			drawFacet(w, k, k, -1, -1, tetra);
			glPopMatrix();
		}
		if (all || (!sticky && ((use && (fd.face == 0 || fd.face == 2) && fd.depth == middle)
				|| (use && (fd.face == 1 || fd.face == 3) && fd.depth == size - 1)
				|| (!use && (fd.face == 0 || fd.face == 2) && fd.depth != middle)
				|| (!use && (fd.face == 1 || fd.face == 3) && fd.depth != size - 1)))
			|| (sticky && ((use && (fd.face == 0 || fd.face == 2) && fd.depth != size - 1)
				|| (use && (fd.face == 1 || fd.face == 3) && fd.depth == size - 1)
				|| (!use && (fd.face == 0 || fd.face == 2) && fd.depth == size - 1)
				|| (!use && (fd.face == 1 || fd.face == 3) && fd.depth != size - 1)))) {
			glPushMatrix();
			glTranslate(s1 * FACET_LENGTH, 0, 0);
			drawFacet(w, -1, i, -1, i, tetra);
			glPopMatrix();
		}
		if (all || (!sticky && ((use && (fd.face == 1 || fd.face == 2) && fd.depth == middle)
				|| (use && (fd.face == 0 || fd.face == 3) && fd.depth == size - 1)
				|| (!use && (fd.face == 1 || fd.face == 2) && fd.depth != middle)
				|| (!use && (fd.face == 0 || fd.face == 3) && fd.depth != size - 1)))
			|| (sticky && ((use && (fd.face == 1 || fd.face == 2) && fd.depth != size - 1)
				|| (use && (fd.face == 0 || fd.face == 3) && fd.depth == size - 1)
				|| (!use && (fd.face == 1 || fd.face == 2) && fd.depth == size - 1)
				|| (!use && (fd.face == 0 || fd.face == 3) && fd.depth != size - 1)))) {
			glPushMatrix();
			glTranslate(0, 0, -s1 * FACET_LENGTH);
			drawFacet(w, j, -1, -1, j, tetra);
			glPopMatrix();
		}
		if (all || (!sticky && ((use && (fd.face == 0 || fd.face == 1) && fd.depth == middle)
				|| (use && (fd.face == 2 || fd.face == 3) && fd.depth == size - 1)
				|| (!use && (fd.face == 0 || fd.face == 1) && fd.depth != middle)
				|| (!use && (fd.face == 2 || fd.face == 3) && fd.depth != size - 1)))
			|| (sticky && ((use && (fd.face == 0 || fd.face == 1) && fd.depth != size - 1)
				|| (use && (fd.face == 2 || fd.face == 3) && fd.depth == size - 1)
				|| (!use && (fd.face == 0 || fd.face == 1) && fd.depth == size - 1)
				|| (!use && (fd.face == 2 || fd.face == 3) && fd.depth != size - 1)))) {
			glPushMatrix();
			glTranslate(0, -s1 * FACET_LENGTH, 0);
			drawFacet(w, -1, -1, k, k, tetra);
			glPopMatrix();
		}
		if (all || (!sticky && ((use && (fd.face == 1 || fd.face == 3) && fd.depth == middle)
				|| (use && (fd.face == 0 || fd.face == 2) && fd.depth == size - 1)
				|| (!use && (fd.face == 1 || fd.face == 3) && fd.depth != middle)
				|| (!use && (fd.face == 0 || fd.face == 2) && fd.depth != size - 1)))
			|| (sticky && ((use && (fd.face == 1 || fd.face == 3) && fd.depth != size - 1)
				|| (use && (fd.face == 0 || fd.face == 2) && fd.depth == size - 1)
				|| (!use && (fd.face == 1 || fd.face == 3) && fd.depth == size - 1)
				|| (!use && (fd.face == 0 || fd.face == 2) && fd.depth != size - 1)))) {
			glPushMatrix();
			glTranslate(-s1 * FACET_LENGTH, 0, 0);
			drawFacet(w, i, -1, i, -1, tetra);
			glPopMatrix();
		}
	}
	if (!sticky)
	    for (edge = 1; edge < size / 2; edge++) {
		/*if (size > 3) OUTSIDE EDGE 1 */
		int a, b, c, d, e, f;
		s1 = size - 1;
		s2 = size - 2 * edge - 1;
		a = edge * edge; /* 1, 4 */
		b = (edge + 1) * (edge + 1) - 1; /* 3, 8 */
		c = (size - edge - 1) * (size - edge - 1); /* 4, 9, 16 */
		d = (size - edge) * (size - edge) - 1; /* 8, 15 */
		e = (size - 1) * (size - 1) + 2 * edge; /* 11,  */
		f = size * size - 2 * edge - 1; /* 13, */
		if (all || (use && fd.face == 1 && fd.depth == edge)
				|| (use && fd.face == 3 && fd.depth == size - 1 - edge)
				|| (use && (fd.face == 0 || fd.face == 2) && fd.depth == size - 1)
				|| (!use && fd.face == 1 && fd.depth != edge)
				|| (!use && fd.face == 3 && fd.depth != size - 1 - edge)
				|| (!use && (fd.face == 0 || fd.face == 2) && fd.depth != size - 1)) {
			glPushMatrix();
			glTranslate(-s1 * FACET_LENGTH, -s2 * FACET_LENGTH, -s2 * FACET_LENGTH);
			drawTetraFacet(w, a, -1, c, -1);
			glPopMatrix();
		}
		if (all || (use && fd.face == 1 && fd.depth == size - 1 - edge)
				|| (use && fd.face == 3 && fd.depth == edge)
				|| (use && (fd.face == 0 || fd.face == 2) && fd.depth == size - 1)
				|| (!use && fd.face == 1 && fd.depth != size - 1 - edge)
				|| (!use && fd.face == 3 && fd.depth != edge)
				|| (!use && (fd.face == 0 || fd.face == 2) && fd.depth != size - 1)) {
			glPushMatrix();
			glTranslate(-s1 * FACET_LENGTH, s2 * FACET_LENGTH, s2 * FACET_LENGTH);
			drawTetraFacet(w, c, -1, a, -1);
			glPopMatrix();
		}
		if (all || (use && fd.face == 1 && fd.depth == edge)
				|| (use && fd.face == 2 && fd.depth == size - 1 - edge)
				|| (use && (fd.face == 0 || fd.face == 3) && fd.depth == size - 1)
				|| (!use && fd.face == 1 && fd.depth != edge)
				|| (!use && fd.face == 2 && fd.depth != size - 1 - edge)
				|| (!use && (fd.face == 0 || fd.face == 3) && fd.depth != size - 1)) {
			glPushMatrix();
			glTranslate(-s2 * FACET_LENGTH, -s2 * FACET_LENGTH, -s1 * FACET_LENGTH);
			drawTetraFacet(w, b, -1, -1, d);
			glPopMatrix();
		}
		if (all || (use && fd.face == 1 && fd.depth == size - 1 - edge)
				|| (use && fd.face == 2 && fd.depth == edge)
				|| (use && (fd.face == 0 || fd.face == 3) && fd.depth == size - 1)
				|| (!use && fd.face == 1 && fd.depth != size - 1 - edge)
				|| (!use && fd.face == 2 && fd.depth != edge)
				|| (!use && (fd.face == 0 || fd.face == 3) && fd.depth != size - 1)) {
			glPushMatrix();
			glTranslate(s2 * FACET_LENGTH, s2 * FACET_LENGTH, -s1 * FACET_LENGTH);
			drawTetraFacet(w, d, -1, -1, b);
			glPopMatrix();
		}
		if (all || (use && fd.face == 2 && fd.depth == size - 1 - edge)
				|| (use && fd.face == 3 && fd.depth == edge)
				|| (use && (fd.face == 0 || fd.face == 1) && fd.depth == size - 1)
				|| (!use && fd.face == 2 && fd.depth != size - 1 - edge)
				|| (!use && fd.face == 3 && fd.depth != edge)
				|| (!use && (fd.face == 0 || fd.face == 1) && fd.depth != size - 1)) {
			glPushMatrix();
			glTranslate(-s2 * FACET_LENGTH, s1 * FACET_LENGTH, s2 * FACET_LENGTH);
			drawTetraFacet(w, e, f, -1, -1);
			glPopMatrix();
		}
		if (all || (use && fd.face == 2 && fd.depth == edge)
				|| (use && fd.face == 3 && fd.depth == size - 1 - edge)
				|| (use && (fd.face == 0 || fd.face == 1) && fd.depth == size - 1)
				|| (!use && fd.face == 2 && fd.depth != edge)
				|| (!use && fd.face == 3 && fd.depth != size - 1 - edge)
				|| (!use && (fd.face == 0 || fd.face == 1) && fd.depth != size - 1)) {
			glPushMatrix();
			glTranslate(s2 * FACET_LENGTH, s1 * FACET_LENGTH, -s2 * FACET_LENGTH);
			drawTetraFacet(w, f, e, -1, -1);
			glPopMatrix();
		}
		if (all || (use && fd.face == 0 && fd.depth == edge)
				|| (use && fd.face == 2 && fd.depth == size - 1 - edge)
				|| (use && (fd.face == 1 || fd.face == 3) && fd.depth == size - 1)
				|| (!use && fd.face == 0 && fd.depth != edge)
				|| (!use && fd.face == 2 && fd.depth != size - 1 - edge)
				|| (!use && (fd.face == 1 || fd.face == 3) && fd.depth != size - 1)) {
			glPushMatrix();
			glTranslate(s1 * FACET_LENGTH, -s2 * FACET_LENGTH, s2 * FACET_LENGTH);
			drawTetraFacet(w, -1, a, -1, c);
			glPopMatrix();
		}
		if (all || (use && fd.face == 0 && fd.depth == size - 1 - edge)
				|| (use && fd.face == 2 && fd.depth == edge)
				|| (use && (fd.face == 1 || fd.face == 3) && fd.depth == size - 1)
				|| (!use && fd.face == 0 && fd.depth != size - 1 - edge)
				|| (!use && fd.face == 2 && fd.depth != edge)
				|| (!use && (fd.face == 1 || fd.face == 3) && fd.depth != size - 1)) {
			glPushMatrix();
			glTranslate(s1 * FACET_LENGTH, s2 * FACET_LENGTH, -s2 * FACET_LENGTH);
			drawTetraFacet(w, -1, c, -1, a);
			glPopMatrix();
		}
		if (all || (use && fd.face == 0 && fd.depth == edge)
				|| (use && fd.face == 3 && fd.depth == size - 1 - edge)
				|| (use && (fd.face == 1 || fd.face == 2) && fd.depth == size - 1)
				|| (!use && fd.face == 0 && fd.depth != edge)
				|| (!use && fd.face == 3 && fd.depth != size - 1 - edge)
				|| (!use && (fd.face == 1 || fd.face == 2) && fd.depth != size - 1)) {
			glPushMatrix();
			glTranslate(s2 * FACET_LENGTH, -s2 * FACET_LENGTH, s1 * FACET_LENGTH);
			drawTetraFacet(w, -1, b, d, -1);
			glPopMatrix();
		}
		if (all || (use && fd.face == 0 && fd.depth == size - 1 - edge)
				|| (use && fd.face == 3 && fd.depth == edge)
				|| (use && (fd.face == 1 || fd.face == 2) && fd.depth == size - 1)
				|| (!use && fd.face == 0 && fd.depth != size - 1 - edge)
				|| (!use && fd.face == 3 && fd.depth != edge)
				|| (!use && (fd.face == 1 || fd.face == 2) && fd.depth != size - 1)) {
			glPushMatrix();
			glTranslate(-s2 * FACET_LENGTH, s2 * FACET_LENGTH, s1 * FACET_LENGTH);
			drawTetraFacet(w, -1, d, b, -1);
			glPopMatrix();
		}
		if (all || (use && fd.face == 0 && fd.depth == size - 1 - edge)
				|| (use && fd.face == 1 && fd.depth == edge)
				|| (use && (fd.face == 2 || fd.face == 3) && fd.depth == size - 1)
				|| (!use && fd.face == 0 && fd.depth != size - 1 - edge)
				|| (!use && fd.face == 1 && fd.depth != edge)
				|| (!use && (fd.face == 2 || fd.face == 3) && fd.depth != size - 1)) {
			glPushMatrix();
			glTranslate(-s2 * FACET_LENGTH, -s1 * FACET_LENGTH, -s2 * FACET_LENGTH);
			drawTetraFacet(w, -1, -1, e, f);
			glPopMatrix();
		}
		if (all || (use && fd.face == 0 && fd.depth == edge)
				|| (use && fd.face == 1 && fd.depth == size - 1 - edge)
				|| (use && (fd.face == 2 || fd.face == 3) && fd.depth == size - 1)
				|| (!use && fd.face == 0 && fd.depth != edge)
				|| (!use && fd.face == 1 && fd.depth != size - 1 - edge)
				|| (!use && (fd.face == 2 || fd.face == 3) && fd.depth != size - 1)) {
			glPushMatrix();
			glTranslate(s2 * FACET_LENGTH, -s1 * FACET_LENGTH, s2 * FACET_LENGTH);
			drawTetraFacet(w, -1, -1, f, e);
			glPopMatrix();
		}
	}
	if (sticky) {
	    if (size > 3 && size % 3 != 0) { /* CENTERS */
		s0 = 1;
		k = 6;
		if (all || ((use && fd.face == 0 && fd.depth == size - 1)
				|| (use && fd.face != 0 && fd.depth != size - 1)
				|| (!use && fd.face != 0 && fd.depth == size - 1)
				|| (!use && fd.face == 0 && fd.depth != size - 1))) {
			glPushMatrix();
			glTranslate(-s0 * FACET_LENGTH, s0 * FACET_LENGTH, -s0 * FACET_LENGTH);
			drawTetraFacet(w, k, -1, -1, -1);
			glPopMatrix();
		}
		if (all || ((use && fd.face == 1 && fd.depth == size - 1)
				|| (use && fd.face != 1 && fd.depth != size - 1)
				|| (!use && fd.face != 1 && fd.depth == size - 1)
				|| (!use && fd.face == 1 && fd.depth != size - 1))) {
			glPushMatrix();
			glTranslate(s0 * FACET_LENGTH, s0 * FACET_LENGTH, s0 * FACET_LENGTH);
			drawTetraFacet(w, -1, k, -1, -1);
			glPopMatrix();
		}
		if (all || ((use && fd.face == 2 && fd.depth == size - 1)
				|| (use && fd.face != 2 && fd.depth != size - 1)
				|| (!use && fd.face != 2 && fd.depth == size - 1)
				|| (!use && fd.face == 2 && fd.depth != size - 1))) {
			glPushMatrix();
			glTranslate(-s0 * FACET_LENGTH, -s0 * FACET_LENGTH, s0 * FACET_LENGTH);
			drawTetraFacet(w, -1, -1, k, -1);
			glPopMatrix();
		}
		if (all || ((use && fd.face == 3 && fd.depth == size - 1)
				|| (use && fd.face != 3 && fd.depth != size - 1)
				|| (!use && fd.face != 3 && fd.depth == size - 1)
				|| (!use && fd.face == 3 && fd.depth != size - 1))) {
			glPushMatrix();
			glTranslate(s0 * FACET_LENGTH, -s0 * FACET_LENGTH, -s0 * FACET_LENGTH);
			drawTetraFacet(w, -1, -1, -1, k);
			glPopMatrix();
		}
	  }
	  { /* OUTSIDE EDGE 1 */
		int a, b, c, d, e, f;
		s1 = size - 1;
		s2 = size - 3;
		edge = 1;
		a = edge * edge; /* 1, 4 */
		b = (edge + 1) * (edge + 1) - 1; /* 3, 8 */
		c = (size - edge - 1) * (size - edge - 1); /* 4, 9, 16 */
		d = (size - edge) * (size - edge) - 1; /* 8, 15 */
		e = (size - 1) * (size - 1) + 2 * edge; /* 11,  */
		f = size * size - 2 * edge - 1; /* 13, */
		if (all || (use && (fd.face == 1 || fd.face == 3) && fd.depth != size - 1)
				|| (use && (fd.face == 0 || fd.face == 2) && fd.depth == size - 1)
				|| (!use && (fd.face == 1 || fd.face == 3) && fd.depth == size - 1)
				|| (!use && (fd.face == 0 || fd.face == 2) && fd.depth != size - 1)) {
			glPushMatrix();
			glTranslate(-s1 * FACET_LENGTH, -s2 * FACET_LENGTH, -s2 * FACET_LENGTH);
			drawTetraFacet(w, a, -1, c, -1);
			glPopMatrix();
			glPushMatrix();
			glTranslate(-s1 * FACET_LENGTH, s2 * FACET_LENGTH, s2 * FACET_LENGTH);
			drawTetraFacet(w, c, -1, a, -1);
			glPopMatrix();
		}
		if (all || (use && (fd.face == 1 || fd.face == 2) && fd.depth != size - 1)
				|| (use && (fd.face == 0 || fd.face == 3) && fd.depth == size - 1)
				|| (!use && (fd.face == 1 || fd.face == 2) && fd.depth == size - 1)
				|| (!use && (fd.face == 0 || fd.face == 3) && fd.depth != size - 1)) {
			glPushMatrix();
			glTranslate(-s2 * FACET_LENGTH, -s2 * FACET_LENGTH, -s1 * FACET_LENGTH);
			drawTetraFacet(w, b, -1, -1, d);
			glPopMatrix();
			glPushMatrix();
			glTranslate(s2 * FACET_LENGTH, s2 * FACET_LENGTH, -s1 * FACET_LENGTH);
			drawTetraFacet(w, d, -1, -1, b);
			glPopMatrix();
		}
		if (all || (use && (fd.face == 2 || fd.face == 3) && fd.depth != size - 1)
				|| (use && (fd.face == 0 || fd.face == 1) && fd.depth == size - 1)
				|| (!use && (fd.face == 2 || fd.face == 3) && fd.depth == size - 1)
				|| (!use && (fd.face == 0 || fd.face == 1) && fd.depth != size - 1)) {
			glPushMatrix();
			glTranslate(-s2 * FACET_LENGTH, s1 * FACET_LENGTH, s2 * FACET_LENGTH);
			drawTetraFacet(w, e, f, -1, -1);
			glPopMatrix();
			glPushMatrix();
			glTranslate(s2 * FACET_LENGTH, s1 * FACET_LENGTH, -s2 * FACET_LENGTH);
			drawTetraFacet(w, f, e, -1, -1);
			glPopMatrix();
		}
		if (all || (use && (fd.face == 0 || fd.face == 2) && fd.depth != size - 1)
				|| (use && (fd.face == 1 || fd.face == 3) && fd.depth == size - 1)
				|| (!use && (fd.face == 0 || fd.face == 2) && fd.depth == size - 1)
				|| (!use && (fd.face == 1 || fd.face == 3) && fd.depth != size - 1)) {
			glPushMatrix();
			glTranslate(s1 * FACET_LENGTH, -s2 * FACET_LENGTH, s2 * FACET_LENGTH);
			drawTetraFacet(w, -1, a, -1, c);
			glPopMatrix();
			glPushMatrix();
			glTranslate(s1 * FACET_LENGTH, s2 * FACET_LENGTH, -s2 * FACET_LENGTH);
			drawTetraFacet(w, -1, c, -1, a);
			glPopMatrix();
		}
		if (all || (use && (fd.face == 0 || fd.face == 3) && fd.depth != size - 1)
				|| (use && (fd.face == 1 || fd.face == 2) && fd.depth == size - 1)
				|| (!use && (fd.face == 0 || fd.face == 3) && fd.depth == size - 1)
				|| (!use && (fd.face == 1 || fd.face == 2) && fd.depth != size - 1)) {
			glPushMatrix();
			glTranslate(s2 * FACET_LENGTH, -s2 * FACET_LENGTH, s1 * FACET_LENGTH);
			drawTetraFacet(w, -1, b, d, -1);
			glPopMatrix();
			glPushMatrix();
			glTranslate(-s2 * FACET_LENGTH, s2 * FACET_LENGTH, s1 * FACET_LENGTH);
			drawTetraFacet(w, -1, d, b, -1);
			glPopMatrix();
		}
		if (all || (use && (fd.face == 0 || fd.face == 1) && fd.depth != size - 1)
				|| (use && (fd.face == 2 || fd.face == 3) && fd.depth == size - 1)
				|| (!use && (fd.face == 0 || fd.face == 1) && fd.depth == size - 1)
				|| (!use && (fd.face == 2 || fd.face == 3) && fd.depth != size - 1)) {
			glPushMatrix();
			glTranslate(-s2 * FACET_LENGTH, -s1 * FACET_LENGTH, -s2 * FACET_LENGTH);
			drawTetraFacet(w, -1, -1, e, f);
			glPopMatrix();
			glPushMatrix();
			glTranslate(s2 * FACET_LENGTH, -s1 * FACET_LENGTH, s2 * FACET_LENGTH);
			drawTetraFacet(w, -1, -1, f, e);
			glPopMatrix();
		}
	  }
	}
	for (edge = 1; edge < (size - 1) / 2; edge++) {
		/*if (size > 4) INSIDE EDGE 2 */
		int m1, m2;
		int a, b, c, d, e, f;
		s1 = size - 2;
		s2 = size - 2 * edge - 2;
		a = (edge + 1) * (edge + 1) + 1; /* 5 */
		b = (edge + 2) * (edge + 2) - 2; /* 7 */
		c = (size - edge - 1) * (size - edge - 1) + 1;/* (size - 2) * (size - 2) + 1 */
		d = (size - edge) * (size - edge) - 2; /* (size - 1) * (size - 1) - 2 */
		e = (size - 1) * ( size - 1) + 2 * edge + 1; /* (size - 1) * (size - 1) + 3 */
		f = size * size - 2 * edge - 2; /* size * size - 4 */
		m1 = edge;
		m2 = edge + 1;
		if (all || (use && fd.face == 1 && fd.depth == m2)
				|| (use && fd.face == 3 && fd.depth == size - 1 - m1)
				|| (use && (fd.face == 0 || fd.face == 2) && fd.depth == size - 1)
				|| (!use && fd.face == 1 && fd.depth != m2)
				|| (!use && fd.face == 3 && fd.depth != size - 1 - m1)
				|| (!use && (fd.face == 0 || fd.face == 2) && fd.depth != size - 1)) {
			glPushMatrix();
			glTranslate(-s1 * FACET_LENGTH, -s2 * FACET_LENGTH, -s2 * FACET_LENGTH);
			drawOctaFacet(w, a, -1, c, -1);
			glPopMatrix();
		}
		if (all || (use && fd.face == 1 && fd.depth == size - 1 - m1)
				|| (use && fd.face == 3 && fd.depth == m2)
				|| (use && (fd.face == 0 || fd.face == 2) && fd.depth == size - 1)
				|| (!use && fd.face == 1 && fd.depth != size - 1 - m1)
				|| (!use && fd.face == 3 && fd.depth != m2)
				|| (!use && (fd.face == 0 || fd.face == 2) && fd.depth != size - 1)) {
			glPushMatrix();
			glTranslate(-s1 * FACET_LENGTH, s2 * FACET_LENGTH, s2 * FACET_LENGTH);
			drawOctaFacet(w, c, -1, a, -1);
			glPopMatrix();
		}
		if (all || (use && fd.face == 1 && fd.depth == m2)
				|| (use && fd.face == 2 && fd.depth == size - 1 - m1)
				|| (use && (fd.face == 0 || fd.face == 3) && fd.depth == size - 1)
				|| (!use && fd.face == 1 && fd.depth != m2)
				|| (!use && fd.face == 2 && fd.depth != size - 1 - m1)
				|| (!use && (fd.face == 0 || fd.face == 3) && fd.depth != size - 1)) {
			glPushMatrix();
			glTranslate(-s2 * FACET_LENGTH, -s2 * FACET_LENGTH, -s1 * FACET_LENGTH);
			drawOctaFacet(w, b, -1, -1, d);
			glPopMatrix();
		}
		if (all || (use && fd.face == 1 && fd.depth == size - 1 - m1)
				|| (use && fd.face == 2 && fd.depth == m2)
				|| (use && (fd.face == 0 || fd.face == 3) && fd.depth == size - 1)
				|| (!use && fd.face == 1 && fd.depth != size - 1 - m1)
				|| (!use && fd.face == 2 && fd.depth != m2)
				|| (!use && (fd.face == 0 || fd.face == 3) && fd.depth != size - 1)) {
			glPushMatrix();
			glTranslate(s2 * FACET_LENGTH, s2 * FACET_LENGTH, -s1 * FACET_LENGTH);
			drawOctaFacet(w, d, -1, -1, b);
			glPopMatrix();
		}
		if (all || (use && fd.face == 2 && fd.depth == size - 1 - m1)
				|| (use && fd.face == 3 && fd.depth == m2)
				|| (use && (fd.face == 0 || fd.face == 1) && fd.depth == size - 1)
				|| (!use && fd.face == 2 && fd.depth != size - 1 - m1)
				|| (!use && fd.face == 3 && fd.depth != m2)
				|| (!use && (fd.face == 0 || fd.face == 1) && fd.depth != size - 1)) {
			glPushMatrix();
			glTranslate(-s2 * FACET_LENGTH, s1 * FACET_LENGTH, s2 * FACET_LENGTH);
			drawOctaFacet(w, e, f, -1, -1);
			glPopMatrix();
		}
		if (all || (use && fd.face == 2 && fd.depth == m2)
				|| (use && fd.face == 3 && fd.depth == size - 1 - m1)
				|| (use && (fd.face == 0 || fd.face == 1) && fd.depth == size - 1)
				|| (!use && fd.face == 2 && fd.depth != m2)
				|| (!use && fd.face == 3 && fd.depth != size - 1 - m1)
				|| (!use && (fd.face == 0 || fd.face == 1) && fd.depth != size - 1)) {
			glPushMatrix();
			glTranslate(s2 * FACET_LENGTH, s1 * FACET_LENGTH, -s2 * FACET_LENGTH);
			drawOctaFacet(w, f, e, -1, -1);
			glPopMatrix();
		}
		if (all || (use && fd.face == 0 && fd.depth == m2)
				|| (use && fd.face == 2 && fd.depth == size - 1 - m1)
				|| (use && (fd.face == 1 || fd.face == 3) && fd.depth == size - 1)
				|| (!use && fd.face == 0 && fd.depth != m2)
				|| (!use && fd.face == 2 && fd.depth != size - 1 - m1)
				|| (!use && (fd.face == 1 || fd.face == 3) && fd.depth != size - 1)) {
			glPushMatrix();
			glTranslate(s1 * FACET_LENGTH, -s2 * FACET_LENGTH, s2 * FACET_LENGTH);
			drawOctaFacet(w, -1, a, -1, c);
			glPopMatrix();
		}
		if (all || (use && fd.face == 0 && fd.depth == size - 1 - m1)
				|| (use && fd.face == 2 && fd.depth == m2)
				|| (use && (fd.face == 1 || fd.face == 3) && fd.depth == size - 1)
				|| (!use && fd.face == 0 && fd.depth != size - 1 - m1)
				|| (!use && fd.face == 2 && fd.depth != m2)
				|| (!use && (fd.face == 1 || fd.face == 3) && fd.depth != size - 1)) {
			glPushMatrix();
			glTranslate(s1 * FACET_LENGTH, s2 * FACET_LENGTH, -s2 * FACET_LENGTH);
			drawOctaFacet(w, -1, c, -1, a);
			glPopMatrix();
		}
		if (all || (use && fd.face == 0 && fd.depth == m2)
				|| (use && fd.face == 3 && fd.depth == size - 1 - m1)
				|| (use && (fd.face == 1 || fd.face == 2) && fd.depth == size - 1)
				|| (!use && fd.face == 0 && fd.depth != m2)
				|| (!use && fd.face == 3 && fd.depth != size - 1 - m1)
				|| (!use && (fd.face == 1 || fd.face == 2) && fd.depth != size - 1)) {
			glPushMatrix();
			glTranslate(s2 * FACET_LENGTH, -s2 * FACET_LENGTH, s1 * FACET_LENGTH);
			drawOctaFacet(w, -1, b, d, -1);
			glPopMatrix();
		}
		if (all || (use && fd.face == 0 && fd.depth == size - 1 - m1)
				|| (use && fd.face == 3 && fd.depth == m2)
				|| (use && (fd.face == 1 || fd.face == 2) && fd.depth == size - 1)
				|| (!use && fd.face == 0 && fd.depth != size - 1 - m1)
				|| (!use && fd.face == 3 && fd.depth != m2)
				|| (!use && (fd.face == 1 || fd.face == 2) && fd.depth != size - 1)) {
			glPushMatrix();
			glTranslate(-s2 * FACET_LENGTH, s2 * FACET_LENGTH, s1 * FACET_LENGTH);
			drawOctaFacet(w, -1, d, b, -1);
			glPopMatrix();
		}
		if (all || (use && fd.face == 0 && fd.depth == size - 1 - m1)
				|| (use && fd.face == 1 && fd.depth == m2)
				|| (use && (fd.face == 2 || fd.face == 3) && fd.depth == size - 1)
				|| (!use && fd.face == 0 && fd.depth != size - 1 - m1)
				|| (!use && fd.face == 1 && fd.depth != m2)
				|| (!use && (fd.face == 2 || fd.face == 3) && fd.depth != size - 1)) {
			glPushMatrix();
			glTranslate(-s2 * FACET_LENGTH, -s1 * FACET_LENGTH, -s2 * FACET_LENGTH);
			drawOctaFacet(w, -1, -1, e, f);
			glPopMatrix();
		}
		if (all || (use && fd.face == 0 && fd.depth == m2)
				|| (use && fd.face == 1 && fd.depth == size - 1 - m1)
				|| (use && (fd.face == 2 || fd.face == 3) && fd.depth == size - 1)
				|| (!use && fd.face == 0 && fd.depth != m2)
				|| (!use && fd.face == 1 && fd.depth != size - 1 - m1)
				|| (!use && (fd.face == 2 || fd.face == 3) && fd.depth != size - 1)) {
			glPushMatrix();
			glTranslate(s2 * FACET_LENGTH, -s1 * FACET_LENGTH, s2 * FACET_LENGTH);
			drawOctaFacet(w, -1, -1, f, e);
			glPopMatrix();
		}
	}
	if (!sticky) {
	  pos = 1;
	  for (row = 0; row < size - 3; row++) {
	    pos += 4;
	    for (col = 0; col < 2 * row + 1; col++) {
		pos++;
		tetra = ((col % 2) == 0);
		sb = -size + 5 + 2 * row - ((tetra) ? 0 : 1);
		sl = size - 3 - col;
		sr = size - 4 - 2 * row + 1 + col;
		mb = row + 2;
		ml = size - 2 - col / 2;
		mr = size - row - 2 + (col + 1) / 2;
		if (all || (use && fd.face == 0 && fd.depth == size - 1)
				|| (use && fd.face == 1 && fd.depth == mb)
				|| (use && fd.face == 2 && fd.depth == ml)
				|| (use && fd.face == 3 && fd.depth == mr)
				|| (!use && fd.face == 0 && fd.depth != size - 1)
				|| (!use && fd.face == 1 && fd.depth != mb)
				|| (!use && fd.face == 2 && fd.depth != ml)
				|| (!use && fd.face == 3 && fd.depth != mr)) {
			glPushMatrix();
			glTranslate(-sl * FACET_LENGTH, sb * FACET_LENGTH, -sr * FACET_LENGTH);
			drawFacet(w, pos, -1, -1, -1, tetra);
			glPopMatrix();
		}
		if (all || (use && fd.face == 1 && fd.depth == size - 1)
				|| (use && fd.face == 0 && fd.depth == mb)
				|| (use && fd.face == 2 && fd.depth == mr)
				|| (use && fd.face == 3 && fd.depth == ml)
				|| (!use && fd.face == 1 && fd.depth != size - 1)
				|| (!use && fd.face == 0 && fd.depth != mb)
				|| (!use && fd.face == 2 && fd.depth != mr)
				|| (!use && fd.face == 3 && fd.depth != ml)) {
			glPushMatrix();
			glTranslate(sl * FACET_LENGTH, sb * FACET_LENGTH, sr * FACET_LENGTH);
			drawFacet(w, -1, pos, -1, -1, tetra);
			glPopMatrix();
		}
		if (all || (use && fd.face == 2 && fd.depth == size - 1)
				|| (use && fd.face == 3 && fd.depth == mb)
				|| (use && fd.face == 0 && fd.depth == ml)
				|| (use && fd.face == 1 && fd.depth == mr)
				|| (!use && fd.face == 2 && fd.depth != size - 1)
				|| (!use && fd.face == 3 && fd.depth !=  mb)
				|| (!use && fd.face == 0 && fd.depth != ml)
				|| (!use && fd.face == 1 && fd.depth != mr)) {
			glPushMatrix();
			glTranslate(-sl * FACET_LENGTH, -sb * FACET_LENGTH, sr * FACET_LENGTH);
			drawFacet(w, -1, -1, pos, -1, tetra);
			glPopMatrix();
		}
		if (all || (use && fd.face == 3 && fd.depth == size - 1)
				|| (use && fd.face == 2 && fd.depth == mb)
				|| (use && fd.face == 0 && fd.depth == mr)
				|| (use && fd.face == 1 && fd.depth == ml)
				|| (!use && fd.face == 3 && fd.depth != size - 1)
				|| (!use && fd.face == 2 && fd.depth != mb)
				|| (!use && fd.face == 0 && fd.depth != mr)
				|| (!use && fd.face == 1 && fd.depth != ml)) {
			glPushMatrix();
			glTranslate(sl * FACET_LENGTH, -sb * FACET_LENGTH, -sr * FACET_LENGTH);
			drawFacet(w, -1, -1, -1, pos, tetra);
			glPopMatrix();
		}
	    }
	  }
	}
}

/*-
 * This rotates whole tetrahedron.
 */
static void
rotateAllEdges(PyraminxGLWidget w)
{
	drawEdges(w, 0, -1, 0, False, True);
}

static void
rotateEdgeSlice(PyraminxGLWidget w, GLfloat rotateStep,
		GLfloat x, GLfloat y, GLfloat z,
		int face, int position, int direction)
{
	glPushMatrix();
	glRotatef(rotateStep, x, y, z);
	if (w->pyraminxGL.movement.control)
		drawEdges(w, face, -1, direction, False, True);
	else
		drawEdges(w, face, position, direction, True, False);
	glPopMatrix();
	if (!w->pyraminxGL.movement.control)
		drawEdges(w, face, position, direction, False, False);
}

/*-
 * This rotates whole tetrahedron.
 */
static void
rotateAllFaces(PyraminxGLWidget w)
{
	drawFaces(w, 0, 0, 0, False, True);
}

static void
rotateFaceSlice(PyraminxGLWidget w, GLfloat rotateStep,
		GLfloat x, GLfloat y, GLfloat z,
		int face, int position, int direction)
{
	glPushMatrix();
	glRotatef(rotateStep, x, y, z);
	if (w->pyraminxGL.movement.control)
		drawFaces(w, face, 0, direction, False, True);
	else
		drawFaces(w, face, position, direction, True, False);
	glPopMatrix();
	if (!w->pyraminxGL.movement.control)
		drawFaces(w, face, position, direction, False, False);
}

static void
drawTetra(PyraminxGLWidget w)
{
	GLfloat rotateStep = 0.0;
	int face = w->pyraminxGL.movement.face;
	int position = w->pyraminxGL.movement.position;
	int direction = w->pyraminxGL.movement.direction;
	int style = w->pyraminxGL.movement.style;

	if (face == NO_FACE || direction == IGNORE_DIR) {
		if (style == PERIOD2)
			rotateAllEdges(w);
		else
			rotateAllFaces(w);
/* 6 edge */
/* 0 1 0 rotate front edge CCW <=> 0 BL */
/* 0 -1 0 rotate front edge CW <=> 0 TR */
/* 1 0 0 rotate left edge CW <=> 0 BOTTOM */
/* -1 0 0 rotate left edge CCW <=> 0 TOP */
/* 0 0 1 rotate right edge CCW <=> 0 LEFT */
/* 0 0 -1 rotate right edge CW <=> 0 RIGHT */
	} else if (style == 2) {
		if (face == 1 || face == 2)
			direction = (direction + MAX_ORIENT / 2) % MAX_ORIENT;
		if (face == 0 && (direction == LEFT || direction == RIGHT))
			direction = (direction + MAX_ORIENT / 2) % MAX_ORIENT;
		if ((face == 1 || face == 2) && (direction == TOP || direction == BOTTOM))
			direction = (direction + MAX_ORIENT / 2) % MAX_ORIENT;
		rotateStep = (float) w->pyraminxGL.rotateStep;
		switch (direction) {
		case TOP:
			rotateEdgeSlice(w, rotateStep, -1, 0, 0, face, position, direction);
			break;
		case TR:
			rotateEdgeSlice(w, rotateStep, 0, -1, 0, face, position, direction);
			break;
		case RIGHT:
			rotateEdgeSlice(w, rotateStep, 0, 0, 1, face, position, direction);
			break;
		case BOTTOM:
			rotateEdgeSlice(w, rotateStep, 1, 0, 0, face, position, direction);
			break;
		case BL:
			rotateEdgeSlice(w, rotateStep, 0, 1, 0, face, position, direction);
			break;
		case LEFT:
			rotateEdgeSlice(w, rotateStep, 0, 0, -1, face, position, direction);
			break;
		default:
			rotateAllEdges(w);
		}
	} else /* if (style == 3) */ {
/* 8 face/corner rotate */
/*	  1-1-1  */
/*	111      */
/* 1 -1 -1 rotate front faces CCW <=> 0 LEFT */
/* -1 1 -1 rotate right face CCW <=> 0 BL */
/* -1 -1 1 rotate top face CCW <=> 1 TR */
/* 1 1 1 rotate left face CCW <=> 0 TOP */
/* -1 1 1 rotate front faces CW <=> 0 RIGHT */
/* 1 -1 1 rotate right face CW <=> 0 TR */
/* 1 1 -1 rotate top face CW <=> 1 BL */
/* -1 -1 -1 rotate left face CW <=> BOTTOM */
		rotateStep = (float) w->pyraminxGL.rotateStep;
		switch (w->pyraminxGL.movement.face) {
		case 0:
			switch (direction) {
			case TOP:
				rotateFaceSlice(w, rotateStep, -1, -1, 1, face, position, direction);
				break;
			case TR:
				rotateFaceSlice(w, rotateStep, -1, -1, -1, face, position, direction);
				break;
			case RIGHT:
				rotateFaceSlice(w, rotateStep, 1, -1, -1, face, position, direction);
				break;
			case BOTTOM:
				rotateFaceSlice(w, rotateStep, 1, 1, -1, face, position, direction);
				break;
			case BL:
				rotateFaceSlice(w, rotateStep, 1, 1, 1, face, position, direction);
				break;
			case LEFT:
				rotateFaceSlice(w, rotateStep, -1, 1, 1, face, position, direction);
				break;
			default:
				rotateAllFaces(w);
			}
			break;
		case 1:
			switch (direction) {
			case TOP:
				rotateFaceSlice(w, rotateStep, -1, 1, 1, face, position, direction);
				break;
			case TR:
				rotateFaceSlice(w, rotateStep, -1, 1, -1, face, position, direction);
				break;
			case RIGHT:
				rotateFaceSlice(w, rotateStep, 1, 1, -1, face, position, direction);
				break;
			case BOTTOM:
				rotateFaceSlice(w, rotateStep, 1, -1, -1, face, position, direction);
				break;
			case BL:
				rotateFaceSlice(w, rotateStep, 1, -1, 1, face, position, direction);
				break;
			case LEFT:
				rotateFaceSlice(w, rotateStep, -1, -1, 1, face, position, direction);
				break;
			default:
				rotateAllFaces(w);
			}
			break;
		case 2:
			switch (direction) {
			case TOP:
				rotateFaceSlice(w, rotateStep, -1, 1, -1, face, position, direction);
				break;
			case TR:
				rotateFaceSlice(w, rotateStep, -1, 1, 1, face, position, direction);
				break;
			case RIGHT:
				rotateFaceSlice(w, rotateStep, 1, 1, 1, face, position, direction);
				break;
			case BOTTOM:
				rotateFaceSlice(w, rotateStep, 1, -1, 1, face, position, direction);
				break;
			case BL:
				rotateFaceSlice(w, rotateStep, 1, -1, -1, face, position, direction);
				break;
			case LEFT:
				rotateFaceSlice(w, rotateStep, -1, -1, -1, face, position, direction);
				break;
			default:
				rotateAllFaces(w);
			}
			break;
		case 3:
			switch (direction) {
			case TOP:
				rotateFaceSlice(w, rotateStep, -1, -1, -1, face, position, direction);
				break;
			case TR:
				rotateFaceSlice(w, rotateStep, -1, -1, 1, face, position, direction);
				break;
			case RIGHT:
				rotateFaceSlice(w, rotateStep, 1, -1, 1, face, position, direction);
				break;
			case BOTTOM:
				rotateFaceSlice(w, rotateStep, 1, 1, 1, face, position, direction);
				break;
			case BL:
				rotateFaceSlice(w, rotateStep, 1, 1, -1, face, position, direction);
				break;
			case LEFT:
				rotateFaceSlice(w, rotateStep, -1, 1, -1, face, position, direction);
				break;
			default:
				rotateAllFaces(w);
			}
			break;
		default:
			rotateAllFaces(w);
		}
	}
}

void
movePiecesGL(PyraminxGLWidget w, int face, int position, int direction,
		int style, Boolean control, int fast)
{
	int i, f;

	w->pyraminxGL.movement.face = face;
	w->pyraminxGL.movement.position = position;
	w->pyraminxGL.movement.direction = direction;
	w->pyraminxGL.movement.style = style;
	w->pyraminxGL.movement.control = control;
	w->pyraminxGL.rotateStep = 0.0;
	w->pyraminxGL.angleStep = 0.0;
	f = (w->pyraminxGL.numTurnInc < fast) ? 1 : fast;
	for (i = 0; i < w->pyraminxGL.numTurnInc / f; i++) {
		double increment = (360.0 / style) * f / w->pyraminxGL.numTurnInc;
		w->pyraminxGL.rotateStep += increment;
		/* 6 or 8 possibles */
		/*if (rotateDir[face][direction] == -1)*/
		drawAllPiecesGL(w);
		Sleep((unsigned int) ((w->pyraminx.delay /
			(w->pyraminxGL.numTurnInc * f)) * 40.0));
		w->pyraminx.currentFace = -1;
		w->pyraminx.currentPosition = -1;
	}
	w->pyraminxGL.angleStep = 0.0;
	w->pyraminxGL.movement.face = NO_FACE;
	w->pyraminxGL.movement.position = NO_POSITION;
	w->pyraminxGL.movement.direction = NO_DEPTH;
	w->pyraminxGL.movement.style = NO_STYLE;
	w->pyraminxGL.movement.control = False;
}

void
drawAllPiecesGL(PyraminxGLWidget w)
{
#ifdef WINVER
	wglMakeCurrent(w->core.hDC, hRC);
#else
	if (!glXMakeCurrent(XtDisplay(w), XtWindow(w), *glXContext)) {
		DISPLAY_WARNING("Draw All GL error");
	}
#endif
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glPushMatrix();
	glTranslate(0.0, 0.0, -10.0);
	if (w->core.height > w->core.width)
		glScalef(SCALE, (SCALE * (float) w->core.width / (float) w->core.height), SCALE);
	else
		glScalef((SCALE * (float) w->core.height / (float) w->core.width), SCALE, SCALE);
	switch (w->pyraminx.view) {
	case 0: /* -<  vertical */
		glRotatef((GLfloat) w->pyraminxGL.step / (GLfloat) 2.5, 0, 1, 0);
		glRotatef((GLfloat) w->pyraminxGL.step / (GLfloat) 2.0, 1, 0, 0);
		break;
	case 1: /* ^  !vertical */
		glRotatef((GLfloat) w->pyraminxGL.step / (GLfloat) 2.5, 1, 0, 0);
		glRotatef((GLfloat) w->pyraminxGL.step / (GLfloat) 2.0, 0, 1, 0);
		break;
	case 2: /* -<  vertical */
		glRotatef((GLfloat) (-w->pyraminxGL.step) / (GLfloat) 2.5, 0, 1, 0);
		glRotatef((GLfloat) (-3 * w->pyraminxGL.step) / (GLfloat) 2.0, 1, 0, 0);
		break;
	case 3: /* Y  !vertical */
		glRotatef((GLfloat) (-w->pyraminxGL.step) / (GLfloat) 2.5, 1, 0, 0);
		glRotatef((GLfloat) (-3 * w->pyraminxGL.step) / (GLfloat) 2.0, 0, 1, 0);
		break;
	default:
		break;
	}
	drawTetra(w);
	glPopMatrix();
	glFlush();
#ifdef WINVER
	SwapBuffers(w->core.hDC);
#else
	glXSwapBuffers(XtDisplay(w), XtWindow(w));
#endif
}

/* UNUSED */
/*void
drawFrameGL(PyraminxGLWidget w, Boolean focus)
{
}*/

/* This is fast for small i, a -1 is returned for negative i. */
static int
SQRT(int i)
{
	int j = 0;

	while (j * j <= i)
		j++;
	return (j - 1);
}

static int
normalizePosition(PyraminxGLWidget w, float x, float y, int size)
{
	float xf, yf;
	int xi, yi, di;
	CRD crd;

	xf = (x * (float) size + (float) size) / (float) 2.0;
	yf = (y * (float) size + (float) size) / (float) 2.0;
	xi = (int) xf;
	if (xi < 0)
		xi = 0;
	if (xi >= size)
		xi = size - 1;
	yi = (int) yf;
	if (yi < 0)
		yi = 0;
	if (yi >= size)
		yi = size - 1;
	di = SQRT(xi * xi + yi * yi);
	if (di < 0)
		di = 0;
	if (di >= size)
		di = size - 1;
	crd.column = xi; /* size 3: 0, 1, 2, 4, 5; 3, 6, 7; 8 */
	crd.row = yi; /* size 3: 0, 2, 3, 7, 8; 2, 5, 6; 4 */
	crd.diagonal = di; /* size 3: 0; 1, 2, 3; 4, 5, 6, 7, 8 */
#ifdef DEBUG
	(void) printf("xf %g, yf %g: xi %d, yi %d, di %d\n",
		xf, yf, xi, yi, di);
#endif
	return toPosition((PyraminxWidget) w, crd);
}

Boolean
selectPiecesGL(PyraminxGLWidget w, int x, int y, int *face, int *position)
{
	/* Using gluUnProject By: Luke Benstead */
	/* http://nehe.gamedev.net/data/articles/article.asp?article=13 */

	GLint viewport[4];
	GLdouble modelview[16];
	GLdouble projection[16];
	GLfloat winX, winY, winZ;
	GLdouble posX, posY, posZ;
	Matrix a, b, r;
	Vector t, world, nw;

	glGetDoublev(GL_MODELVIEW_MATRIX, modelview);
	glGetDoublev(GL_PROJECTION_MATRIX, projection);
	glGetIntegerv(GL_VIEWPORT, viewport);

	winX = (float) x;
	winY = (float) viewport[3] - (float) y;
	glReadPixels(x, (int) winY, 1, 1, GL_DEPTH_COMPONENT, GL_FLOAT,
		&winZ);
	gluUnProject(winX, winY, winZ, modelview, projection, viewport,
		&(posX), &(posY), &(posZ));
	t[0] = (float) posX;
	t[1] = (float) posY;
	t[2] = (float) (posZ + 10);
	if (w->core.height > w->core.width)
		t[1] *= (float) w->core.height / (float) w->core.width;
	else
		t[0] *= (float) w->core.width / (float) w->core.height;
	*face = -1;
	if (winZ == 1) {
		return False;
	}
	world[0] = t[0], world[1] = t[1], world[2] = t[2];
	switch (w->pyraminx.view) {
	case 0: /* -<  vertical */
		setMatrix((float) (-w->pyraminxGL.step) / (float) 2.0, 1.0, 0.0, 0.0, a);
		setMatrix((float) (-w->pyraminxGL.step) / (float) 2.5, 0.0, 1.0, 0.0, b);
		matrixMatrix(a, b, r);
		matrixVector(r, t, world);
		nw[0] = -world[0] / (float) 2.0;
		nw[1] = world[1] / (float) 2.0;
		nw[2] = world[2] / (float) 2.0;
		if (nw[0] >= nw[2] && nw[1] >= nw[2]) {
			*face = 0;
			*position = normalizePosition(w, world[0], world[2], w->pyraminx.size);
		} else if (nw[1] >= nw[0] && nw[2] >= nw[0]) {
			*face = 1;
			*position = normalizePosition(w, world[1], -world[2], w->pyraminx.size);
		} else if (nw[0] >= nw[1] && nw[2] >= nw[1]) {
			*face = 2;
			*position = normalizePosition(w, world[0], -world[2], w->pyraminx.size);
		}
		break;
	case 1: /* ^  !vertical */
		setMatrix((float) (3 * w->pyraminxGL.step) / (float) 2.0, 0.0, 1.0, 0.0, a);
		setMatrix((float) w->pyraminxGL.step / (float) 2.65, 1.0, 0.0, 0.0, b);
		matrixMatrix(a, b, r);
		matrixVector(r, t, world);
		nw[0] = -world[0] / (float) 2.0;
		nw[1] = world[1] / (float) 2.0;
		nw[2] = world[2] / (float) 2.0;
		if (nw[0] <= nw[1] && nw[0] <= nw[2]) {
			*face = 0;
			*position = normalizePosition(w, -world[2], -world[1], w->pyraminx.size);
		} else if (nw[1] <= nw[0] && nw[1] <= nw[2]) {
			*face = 2;
			*position = normalizePosition(w, -world[0], world[2], w->pyraminx.size);
		} else if (nw[2] <= nw[0] && nw[2] <= nw[1]) {
			*face = 1;
			*position = normalizePosition(w, world[0], world[1], w->pyraminx.size);
		}
		break;
	case 2: /* <  vertical */
		setMatrix((float) (3 * w->pyraminxGL.step) / (float) 2.0, 1.0, 0.0, 0.0, a);
		setMatrix((float) w->pyraminxGL.step / (float) 2.65, 0.0, 1.0, 0.0, b);
		matrixMatrix(a, b, r);
		matrixVector(r, t, world);
		*face = 3;
		*position = normalizePosition(w, -world[0], world[2], w->pyraminx.size);
		break;
	case 3: /* A  !vertical */
		setMatrix((float) (-w->pyraminxGL.step) / (float) 2.0, 0.0, 1.0, 0.0, a);
		setMatrix((float) (-w->pyraminxGL.step) / (float) 2.5, 1.0, 0.0, 0.0, b);
		matrixMatrix(a, b, r);
		matrixVector(r, t, world);
		*face = 3;
		*position = normalizePosition(w, world[2], -world[0], w->pyraminx.size);
		break;
	default:
		break;
	}
#ifdef DEBUG
	(void) printf("view %d, x %d, y %d\n", w->pyraminx.view, x, y);
	(void) printf(" wx %g, wy %g, wz %g\n", winX, winY, winZ);
	(void) printf("  px %g, py %g, pz %g\n", posX, posY, posZ);
	(void) printf("   tx %g, ty %g, tz %g\n", t[0], t[1], t[2]);
	(void) printf("    rx %g, ry %g, rz %g\n",
		world[0], world[1], world[2]);
	(void) printf("     face %d, pos %d\n", *face, *position);
#endif
	/* FIXME need to determine position */
	return True;
}

void
narrowSelectionGL(PyraminxGLWidget w, int style,
		int *face, int *position, int *direction)
{
	switch (*direction) {
	case BR:
		switch (w->pyraminx.view) {
		case 0:
			switch (*face) {
			case 1:
				*direction = RIGHT;
				break;
			case 2:
				*direction = TR;
				break;
			default:
				*direction = IGNORE_DIR;
				break;
			}
			break;
		case 1:
			switch (*face) {
			case 1:
				*direction = BOTTOM;
				break;
			case 2:
				*direction = RIGHT;
				break;
			default:
				*direction = IGNORE_DIR;
				break;
			}
			break;
		case 2:
			switch (*face) {
			case 3:
				*direction = RIGHT;
				break;
			default:
				*direction = IGNORE_DIR;
				break;
			}
			break;
		case 3:
			switch (*face) {
			case 3:
				*direction = LEFT;
				break;
			default:
				*direction = IGNORE_DIR;
				break;
			}
			break;
		}
		break;
	case TL:
		switch (w->pyraminx.view) {
		case 0:
			switch (*face) {
			case 1:
				*direction = LEFT;
				break;
			case 2:
				*direction = BL;
				break;
			default:
				*direction = IGNORE_DIR;
				break;
			}
			break;
		case 1:
			switch (*face) {
			case 1:
				*direction = TOP;
				break;
			case 2:
				*direction = LEFT;
				break;
			default:
				*direction = IGNORE_DIR;
				break;
			}
			break;
		case 2:
			switch (*face) {
			case 3:
				*direction = LEFT;
				break;
			default:
				*direction = IGNORE_DIR;
				break;
			}
			break;
		case 3:
			switch (*face) {
			case 3:
				*direction = RIGHT;
				break;
			default:
				*direction = IGNORE_DIR;
				break;
			}
			break;
		}
		break;
	case CCW:
		switch (w->pyraminx.view) {
		case 0:
		case 1:
			switch (*face) {
			case 0:
				*direction = LEFT;
				break;
			case 1:
				*direction = TOP;
				break;
			case 2:
				*direction = TR;
				break;
			default:
				*direction = IGNORE_DIR;
				break;
			}
			break;
		case 2:
		case 3:
			*face = 0;
			*direction = RIGHT;
			break;
		}
		break;
	case CW:
		switch (w->pyraminx.view) {
		case 0:
		case 1:
			switch (*face) {
			case 0:
				*direction = RIGHT;
				break;
			case 1:
				*direction = BOTTOM;
				break;
			case 2:
				*direction = BL;
				break;
			default:
				*direction = IGNORE_DIR;
				break;
			}
			break;
		case 2:
		case 3:
			*face = 0;
			*direction = LEFT;
			break;
		}
		break;
	case TOP:
		switch (w->pyraminx.view) {
		case 0:
			switch (*face) {
			case 0:
				*direction = TR;
				break;
			case 1:
				*direction = TOP;
				break;
			case 2:
				*direction = LEFT;
				break;
			default:
				*direction = IGNORE_DIR;
				break;
			}
			break;
		case 1:
			switch (*face) {
			case 0:
			case 1:
				*direction = TR;
				break;
			default:
				*direction = IGNORE_DIR;
				break;
			}
			break;
		case 2:
			switch (*face) {
			case 3:
				*direction = TOP;
				break;
			default:
				*direction = IGNORE_DIR;
				break;
			}
			break;
		case 3:
			*direction = IGNORE_DIR;
			break;
		}
		break;
	case TR:
		switch (w->pyraminx.view) {
		case 0:
			switch (*face) {
			case 0:
				*direction = RIGHT;
				break;
			case 1:
				*direction = TR;
				break;
			default:
				*direction = IGNORE_DIR;
				break;
			}
			break;
		case 1:
			switch (*face) {
			case 0:
				*direction = RIGHT;
				break;
			case 2:
				*direction = TOP;
				break;
			default:
				*direction = IGNORE_DIR;
				break;
			}
			break;
		case 2:
			switch (*face) {
			case 3:
				*direction = TR;
				break;
			default:
				*direction = IGNORE_DIR;
				break;
			}
			break;
		case 3:
			switch (*face) {
			case 3:
				*direction = BOTTOM;
				break;
			default:
				*direction = IGNORE_DIR;
				break;
			}
			break;
		}
		break;
	case RIGHT:
		switch (w->pyraminx.view) {
		case 0:
			switch (*face) {
			case 0:
				*direction = BOTTOM;
				break;
			case 2:
				*direction = TOP;
				break;
			default:
				*direction = IGNORE_DIR;
				break;
			}
			break;
		case 1:
			switch (*face) {
			case 0:
				*direction = BOTTOM;
				break;
			case 1:
				*direction = RIGHT;
				break;
			case 2:
				*direction = TR;
				break;
			default:
				*direction = IGNORE_DIR;
				break;
			}
			break;
		case 2:
			*direction = IGNORE_DIR;
			break;
		case 3:
			switch (*face) {
			case 3:
				*direction = BL;
				break;
			default:
				*direction = IGNORE_DIR;
				break;
			}
			break;
		}
		break;
	case BOTTOM:
		switch (w->pyraminx.view) {
		case 0:
			switch (*face) {
			case 0:
				*direction = BL;
				break;
			case 1:
				*direction = BOTTOM;
				break;
			case 2:
				*direction = RIGHT;
				break;
			default:
				*direction = IGNORE_DIR;
				break;
			}
			break;
		case 1:
			switch (*face) {
			case 0:
			case 1:
				*direction = BL;
				break;
			default:
				*direction = IGNORE_DIR;
				break;
			}
			break;
		case 2:
			switch (*face) {
			case 3:
				*direction = BOTTOM;
				break;
			default:
				*direction = IGNORE_DIR;
				break;
			}
			break;
		case 3:
			switch (*face) {
			case 3:
				*direction = BL;
				break;
			default:
				*direction = IGNORE_DIR;
				break;
			}
			break;
		}
		break;
	case BL:
		switch (w->pyraminx.view) {
		case 0:
			switch (*face) {
			case 0:
				*direction = LEFT;
				break;
			case 1:
				*direction = BL;
				break;
			default:
				*direction = IGNORE_DIR;
				break;
			}
			break;
		case 1:
			switch (*face) {
			case 0:
				*direction = LEFT;
				break;
			case 2:
				*direction = BOTTOM;
				break;
			default:
				*direction = IGNORE_DIR;
				break;
			}
			break;
		case 2:
			switch (*face) {
			case 3:
				*direction = BL;
				break;
			default:
				*direction = IGNORE_DIR;
				break;
			}
			break;
		case 3:
			switch (*face) {
			case 3:
				*direction = TOP;
				break;
			default:
				*direction = IGNORE_DIR;
				break;
			}
			break;
		}
		break;
	case LEFT:
		switch (w->pyraminx.view) {
		case 0:
			switch (*face) {
			case 0:
				*direction = TOP;
				break;
			case 2:
				*direction = BOTTOM;
				break;
			default:
				*direction = IGNORE_DIR;
				break;
			}
			break;
		case 1:
			switch (*face) {
			case 0:
				*direction = TOP;
				break;
			case 1:
				*direction = LEFT;
				break;
			case 2:
				*direction = BL;
				break;
			default:
				*direction = IGNORE_DIR;
				break;
			}
			break;
		case 2:
			*direction = IGNORE_DIR;
			break;
		case 3:
			switch (*face) {
			case 3:
				*direction = TR;
				break;
			default:
				*direction = IGNORE_DIR;
				break;
			}
			break;
		}
		break;
	default:
		break;
	}
}

#ifdef WINVER
static Boolean
setupPixelFormat(PyraminxGLWidget w, BYTE type, DWORD flags)
{
	PIXELFORMATDESCRIPTOR pfd;
	int pixelFormat;

	memset(&pfd, 0, sizeof(PIXELFORMATDESCRIPTOR));
	pfd.nSize = sizeof(PIXELFORMATDESCRIPTOR);
	pfd.nVersion = 1;
	pfd.dwFlags = PFD_DRAW_TO_WINDOW | PFD_SUPPORT_OPENGL | flags;
	pfd.dwLayerMask = PFD_MAIN_PLANE;
	/*pfd.cColorBits = 8;
	pfd.cDepthBits = 16;*/
	pfd.iPixelType = type;
	pfd.cColorBits = 24;
	pfd.cDepthBits = 32;

	if ((pixelFormat = ChoosePixelFormat(w->core.hDC, &pfd)) == 0) {
		DWORD err;
		char *buf1;

		err = GetLastError();
		/* 6 ERROR_INVALID_HANDLE */
		intCat(&buf1, "ChoosePixelFormat failed ", (int) err);
		DISPLAY_WARNING(buf1);
		free(buf1);
		return FALSE;
	}

	if (SetPixelFormat(w->core.hDC, pixelFormat, &pfd) == FALSE) {
		MessageBox(NULL, "SetPixelFormat failed", "Error", MB_OK);
		return FALSE;
	}
	DescribePixelFormat(w->core.hDC, pixelFormat,
		sizeof(PIXELFORMATDESCRIPTOR), &pfd);
	return TRUE;
}

#else

static Boolean
setValuesPuzzleGL(Widget current, Widget request, Widget renew)
{
	PyraminxGLWidget c = (PyraminxGLWidget) current, w = (PyraminxGLWidget) renew;
	Boolean redraw = False;

	if (w->pyraminx.view != c->pyraminx.view) {
		resizePuzzleGL(w);
		redraw = True;
	}
	if (w->pyraminx.size != c->pyraminx.size) {
		resetPieces((PyraminxWidget) w);
		resizePuzzleGL(w);
		redraw = True;
	}
	if (w->pyraminx.delay != c->pyraminx.delay) {
		w->pyraminxGL.numTurnInc = ((w->pyraminx.delay < MAX_TURN_INC) ?
			w->pyraminx.delay + 1 : MAX_TURN_INC);
	}
	return (redraw);
}

static GLXContext *
initGL(PyraminxGLWidget w) {
	XVisualInfo xviIn, *xviOut;
	int numVis;

	/*XGetWindowAttributes(XtDisplay(w), XtWindow(w), &xgwa); */
	xviIn.screen = DefaultScreen(XtDisplay(w));
	xviIn.visualid = XVisualIDFromVisual(DefaultVisual(XtDisplay(w),
		xviIn.screen));
	xviOut = XGetVisualInfo(XtDisplay(w), VisualScreenMask | VisualIDMask,
		&xviIn, &numVis);
	if (!xviOut) {
		XtWarning("Could not get XVisualInfo");
		return (GLXContext *) NULL;
	}
	if (glXContext)
		free(glXContext);
	if ((glXContext = (GLXContext *) malloc(sizeof (GLXContext))) ==
			NULL) {
		DISPLAY_ERROR("Not enough memory for glx info, exiting.");
	}
	/* assertion "glx_dpy" failed on Cygwin */
	if (!glXQueryExtension(XtDisplay(w), NULL, NULL)) {
		XtWarning("Could not find GLX");
		return (GLXContext *) NULL;
	}
	*glXContext = glXCreateContext(XtDisplay(w), xviOut, 0, GL_TRUE);
	(void) XFree((char *) xviOut);
	if (!*glXContext) {
		XtWarning("Could not create rendering context");
		return (GLXContext *) NULL;
	}
	return glXContext;
}
#endif

#ifndef WINVER
static
#endif
void
resizePuzzleGL(PyraminxGLWidget w)
{
#ifdef WINVER
	RECT rect;

	/* Determine size of client area */
	(void) GetClientRect(w->core.hWnd, &rect);
	w->core.width = rect.right;
	w->core.height = rect.bottom;
#endif
	glViewport(0, 0, (GLint) w->core.width, (GLint) w->core.height);
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	glFrustum(-1.0, 1.0, -1.0, 1.0, 5.0, 15.0);
	glMatrixMode(GL_MODELVIEW);
}

static void
makeCurrentGL(PyraminxGLWidget w)
{
	GLboolean rgbaMode;

#ifdef WINVER
	wglMakeCurrent(w->core.hDC, hRC);
#else
	if (!glXMakeCurrent(XtDisplay(w), XtWindow(w), *glXContext)) {
		DISPLAY_WARNING("GL error");
	}
#endif
	madeCurrent = True;
	/* True Color junk */
	glGetBooleanv(GL_RGBA_MODE, &rgbaMode);
	if (!rgbaMode) {
#ifdef WINVER
		glClearIndex(0.0);
#else
		glIndexi(WhitePixel(XtDisplay(w),
			DefaultScreen(XtDisplay(w))));
		glClearIndex((float) BlackPixel(XtDisplay(w),
			DefaultScreen(XtDisplay(w))));
#endif
	}
	resizePuzzleGL(w);

	glDrawBuffer(GL_BACK);
	glClearDepth(1.0);
	glClearColor(face_material[NO_FACE][0], face_material[NO_FACE][1],
		face_material[NO_FACE][2], face_material[NO_FACE][3]);
	glColor3f(1.0, 1.0, 1.0);

	glLightfv(GL_LIGHT0, GL_AMBIENT, ambient);
	glLightfv(GL_LIGHT0, GL_DIFFUSE, diffuse);
	glLightfv(GL_LIGHT0, GL_POSITION, position0);
	glLightfv(GL_LIGHT1, GL_AMBIENT, ambient);
	glLightfv(GL_LIGHT1, GL_DIFFUSE, diffuse);
	glLightfv(GL_LIGHT1, GL_POSITION, position1);
	glLightfv(GL_LIGHT2, GL_AMBIENT, ambient);
	glLightfv(GL_LIGHT2, GL_DIFFUSE, diffuse);
	glLightfv(GL_LIGHT2, GL_POSITION, position2);
	glLightModelfv(GL_LIGHT_MODEL_AMBIENT, lmodel_ambient);
	glLightModelfv(GL_LIGHT_MODEL_TWO_SIDE, lmodel_twoside);
	glEnable(GL_LIGHTING);
	glEnable(GL_LIGHT0);
	glEnable(GL_LIGHT1);
	glEnable(GL_LIGHT2);
	glEnable(GL_DEPTH_TEST);
	glEnable(GL_NORMALIZE);
	glEnable(GL_CULL_FACE);
	glShadeModel(GL_FLAT);
	glMaterialfv(GL_FRONT_AND_BACK, GL_SHININESS, front_shininess);
	glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, front_specular);

	w->pyraminxGL.step = 90;
	w->pyraminxGL.rotateStep = 0.0;
}

#ifndef WINVER
static void
initializePuzzleGL(Widget request, Widget renew)
{
	PyraminxGLWidget w = (PyraminxGLWidget) renew;
	int i;

	setAllColors((PyraminxWidget) w);
	if (initGL(w) == (GLXContext *) NULL) {
		w->pyraminx.dim = 0;
		return;
	}
	w->pyraminx.dim = 4; /* 2 and 3 already taken */
	for (i = 0; i < MAX_FACES + 2; i++)
		pickColor(w, i, w->pyraminx.mono, face_material[i]);
	w->pyraminxGL.numTurnInc = ((w->pyraminx.delay < MAX_TURN_INC) ?
		w->pyraminx.delay + 1 : MAX_TURN_INC);
	resizePuzzleGL(w);
}
#endif

#ifndef WINVER
static
#endif
void
exposePuzzleGL(
#ifdef WINVER
PyraminxGLWidget w
#else
Widget renew, XEvent *event, Region region
#endif
)
{
#ifdef WINVER
	if (hRC == NULL) {
		BYTE type;
		int i;
		DWORD flags = PFD_DOUBLEBUFFER;

#if 1
		type = PFD_TYPE_RGBA;
#else
		type = PFD_TYPE_COLORINDEX;
#endif
		(void) setupPixelFormat(w, type, flags);
		hRC = wglCreateContext(w->core.hDC);
		w->pyraminxGL.movement.face = NO_FACE;
		w->pyraminxGL.movement.position = NO_POSITION;
		w->pyraminxGL.movement.direction = NO_DEPTH;
		w->pyraminxGL.movement.style = NO_STYLE;
		w->pyraminxGL.movement.control = False;
		w->pyraminxGL.rotateStep = 0.0;
		w->pyraminxGL.angleStep = 0.0;
		w->pyraminx.dim = 4; /* 2 and 3 already taken */
		for (i = 0; i < MAX_FACES + 2; i++)
			pickColor(w, i, w->pyraminx.mono, face_material[i]);
		w->pyraminxGL.numTurnInc = ((w->pyraminx.delay < MAX_TURN_INC) ?
			w->pyraminx.delay + 1 : MAX_TURN_INC);
		resizePuzzleGL(w);
	}
#else
	PyraminxGLWidget w = (PyraminxGLWidget) renew;

	if (!w->core.visible)
		return;
	if (w->pyraminx.dim != 4)
		return;
#endif
	if (!madeCurrent) {
		makeCurrentGL(w);
	}
	/*drawFrameGL(w, w->pyraminx.focus);*/
	drawAllPiecesGL(w);
}

#ifndef WINVER
static void
movePuzzleGLTl(PyraminxGLWidget w, XEvent *event, char **args, int nArgs)
{
	movePuzzleInput((PyraminxWidget) w, event->xbutton.x, event->xbutton.y,
		TL,
		(int) (event->xkey.state & (ShiftMask | LockMask)),
		(int) (event->xkey.state & ControlMask));
}

static void
movePuzzleGLTop(PyraminxGLWidget w, XEvent *event, char **args, int nArgs)
{
	movePuzzleInput((PyraminxWidget) w, event->xbutton.x, event->xbutton.y,
		TOP,
		(int) (event->xkey.state & (ShiftMask | LockMask)),
		(int) (event->xkey.state & ControlMask));
}

static void
movePuzzleGLTr(PyraminxGLWidget w, XEvent *event, char **args, int nArgs)
{
	movePuzzleInput((PyraminxWidget) w, event->xbutton.x, event->xbutton.y,
		TR,
		(int) (event->xkey.state & (ShiftMask | LockMask)),
		(int) (event->xkey.state & ControlMask));
}

static void
movePuzzleGLLeft(PyraminxGLWidget w, XEvent *event, char **args, int nArgs)
{
	movePuzzleInput((PyraminxWidget) w, event->xbutton.x, event->xbutton.y,
		LEFT,
		(int) (event->xkey.state & (ShiftMask | LockMask)),
		(int) (event->xkey.state & ControlMask));
}

static void
movePuzzleGLRight(PyraminxGLWidget w, XEvent *event, char **args, int nArgs)
{
	movePuzzleInput((PyraminxWidget) w, event->xbutton.x, event->xbutton.y,
		RIGHT,
		(int) (event->xkey.state & (ShiftMask | LockMask)),
		(int) (event->xkey.state & ControlMask));
}

static void
movePuzzleGLBl(PyraminxGLWidget w, XEvent *event, char **args, int nArgs)
{
	movePuzzleInput((PyraminxWidget) w, event->xbutton.x, event->xbutton.y,
		BL,
		(int) (event->xkey.state & (ShiftMask | LockMask)),
		(int) (event->xkey.state & ControlMask));
}

static void
movePuzzleGLBottom(PyraminxGLWidget w, XEvent *event, char **args, int nArgs)
{
	movePuzzleInput((PyraminxWidget) w, event->xbutton.x, event->xbutton.y,
		BOTTOM,
		(int) (event->xkey.state & (ShiftMask | LockMask)),
		(int) (event->xkey.state & ControlMask));
}

static void
movePuzzleGLBr(PyraminxGLWidget w, XEvent *event, char **args, int nArgs)
{
	movePuzzleInput((PyraminxWidget) w, event->xbutton.x, event->xbutton.y,
		BR,
		(int) (event->xkey.state & (ShiftMask | LockMask)),
		(int) (event->xkey.state & ControlMask));
}
#endif
#else
int x = 0; /* non empty source */
#endif
